AmpPhaseDecomp <- function(xfd, yfd, hfd, rng=xrng)
{
#  Computes the amplitude-phase decomposition for a registration.

#  Arguments:
#  XFD  ...  FD object for unregistered functions
#  YFD  ...  FD object for registered functions
#  HFD  ...  FD object for warping functions

#  Returns:
#  MS.amp ... mean square for amplitude variation 
#  MS.pha ... mean square for amplitude variation 
#  RSQR   ... squared correlation measure of prop. phase variation 
#  C      ... constant C

#  Last modified 6 January 2020 by Jim Ramsay

xbasis  <- xfd$basis
nxbasis <- xbasis$nbasis
nfine   <- max(201,10*nxbasis)
xrng    <- xbasis$rangeval

if (rng[1] < xrng[1] || rng[2] > xrng[2]) stop(
    "RNG is not within the range of the other arguments.")
if (rng[1] >= rng[2]) stop("Elements of rng are not increasing.")
tfine   <- seq(rng[1],rng[2],len=nfine)
delta   <- tfine[2] - tfine[1]

Dhfine  <- eval.fd(tfine, hfd, 1)
xfine   <- eval.fd(tfine, xfd, 0)
yfine   <- eval.fd(tfine, yfd, 0)
mufine  <- apply(xfine, 1, mean)
etafine <- apply(yfine, 1, mean)

N       <- dim(xfine)[2]
rfine   <- yfine - outer(etafine,rep(1,N))

intetasqr <- delta*trapz(etafine^2)
intmusqr  <- delta*trapz(mufine^2)

covDhSy <- rep(0,nfine)
for (i in 1:nfine) {
    Dhi        <- Dhfine[i,]
    Syi        <- yfine[i,]^2
    covDhSy[i] <- cov(Dhi, Syi)
}
intcovDhSy <- delta*trapz(covDhSy)

intysqr <- rep(0,N)
intrsqr <- rep(0,N)
for (i in 1:N) {
    intysqr[i] <- delta*trapz(yfine[,i]^2)
    intrsqr[i] <- delta*trapz(rfine[,i]^2)
}

C      <- 1 + intcovDhSy/mean(intysqr)
MS.amp <- C*mean(intrsqr)
MS.pha <- C*intetasqr - intmusqr
RSQR   <- MS.pha/(MS.amp+MS.pha)

return(list("MS.amp" = MS.amp, "MS.pha" = MS.pha, "RSQR" = RSQR, "C" = C)) 

}

trapz = function(x) {
n = length(x)
intx = sum(x) - 0.5*(x[1]+x[n])
return(intx)
}

argcheck = function(argvals) {

#  check ARGVALS

  if (!is.numeric(argvals)) stop("ARGVALS is not numeric.")

  argvals <- as.vector(argvals)

  if (length(argvals) < 2) stop("ARGVALS does not contain at least two values.")

  return(argvals)

}

as.array3 <- function(x){
#  as.array3 produces a 3-d array from a vector, matrix or array of 
#  up to 3 dimensions, preserving names.

  dimx <- dim(x)
  ndim <- length(dimx)
#   If dimension is already 3
  if (ndim==3) {
    return(x)
  }
#   Otherwise, set up an error message dimension higher than 3
  xName <- substring(deparse(substitute(x)), 1, 22) 
  if (ndim>3)
    stop('length(dim(', xName, ") = ", ndim, ' > 3')
#  If dimension less than 3, ...
  x.      <- as.matrix(x)  #  coerce to matrix
  xNames  <- dimnames(x.)  #  get dimension names if any
  dim(x.) <- c(dim(x.), 1) #  add a unit value for 3rd dimension
#  Assign dimension names
  if(is.list(xNames))
    dimnames(x.) <- list(xNames[[1]], xNames[[2]], NULL)
#  Return result
  x. 
}

as.fd <- function(x, ...) {
  UseMethod('as.fd')
}

as.fd.fdSmooth <- function(x, ...){
  x$fd
}

as.fd.function <- function(x, ...){
# Translate an object of class splinefun to class fd
##
## 1.  check class
##
  objName <- deparse(substitute(x))
  {
    if(length(objName)>1)
      objName <- character(0)
    else
      if(nchar(objName)>33)
        objName <- substring(objName, 1, 33)
  }
  if(!inherits(x, 'function')) 
    stop("'x' (", objName, ") is not of class function")
#
  xenv <- environment(x)
  xz <- get('z', xenv) 
  if(is.null(xz))
    stop("NULL environment of 'x' (", objName,
         ");  therefore, it can NOT have been created by 'splinefun.'")
#  
  if(is.null(xz$method))
    stop("'x' (", objName, ") has a NULL 'method', and therefore",
         " can NOT have been created by 'splinefun.'")
# z$method:  1=periodic, 2=natural, 3=fmm (std B-Splines, I believe)   
#  if(xz$method!=3){
  if(!(xz$method %in% 2:3)){
    msg <- paste("x (", objName, ") ", sep='')
    msg2 <- {
      if(xz$method=="1")
        paste(msg, " uses periodic B-splines, and as.fd ",
              "is programmed\n    to translate only B-splines ",
              "with coincident boundary knots.", sep='')
      else
        paste(msg, "does not use B-splines as required ",
                      "for function 'as.fd'.")
    }
    stop(msg2)
  }
##
## 2.  Create a basis 
##
  Knots <- xz$x
  y.x <- xz$y
  basis <- create.bspline.basis(range(Knots), breaks=Knots)
  fd. <- fdPar(basis, lambda=0)
  nKn <- length(Knots) 
  nobs <- (2*nKn-1)
  x. <- seq(Knots[1], Knots[nKn], length=nobs) 
  smooth.basis(x., x(x.), fd.)$fd
}

as.fd.smooth.spline <- function(x, ...){
# Translate an object of class smooth.spline to class fd
##
## 1.  check class
##
  objName <- deparse(substitute(x))
  {
    if(length(objName)>1)
      objName <- character(0)
    else
      if(nchar(objName)>33)
        objName <- substring(objName, 1, 33)
  }
  if(!inherits(x, 'smooth.spline')) 
    stop("'x' (", objName, ") is not of class smooth.spline")
##
## 2.  Create a basis 
##
  Kn0 <- x$fit$knot
  x0 <- min(x$x)
  x1 <- max(x$x) 
  Knots <- (x0+(x1-x0)*Kn0[4:(length(Kn0)-3)])
# Don't use 'unique' in case 'x' has coincident interior knots.
#  basis <- create.bspline.basis(breaks=Knots)
  basis <- create.bspline.basis(range(Knots), breaks=Knots)
#
  fd(x$fit$coef, basis)
}

as.POSIXct1970 <- function(x, tz="GMT", ...){
#
  if(!is.numeric(x)){
    Px <- try(as.POSIXct(x, tz=tz, ...))
    if(inherits(Px,'try-error')){
      nx <- length(x)
      Px <- rep(as.POSIXct1970(0), nx)
    }
    return(Px)
  }
#
  o1970 <- strptime('1970-01-01', '%Y-%m-%d', tz=tz)
  o1970. <- as.POSIXct(o1970, tz=tz)
#
  as.POSIXct(x, tz=tz, origin=o1970., ...)
}
axisIntervals <- function(side=1, atTick1=fda::monthBegin.5, atTick2=fda::monthEnd.5,
              atLabels=fda::monthMid, labels=month.abb, cex.axis=0.9, ...)
{
#  Here's something trivial ...
# 1.  Interval start
  axis(side, at=atTick1, labels=FALSE, ...)
# 2.  Interval end
  if(any(!is.na(atTick2)))axis(side, at=atTick2, labels=FALSE, ...)
# 3.  Interval labels
  axis(side, at=atLabels, labels=labels, tick=FALSE,
       cex.axis=cex.axis, ...)
}

axesIntervals <- function(side=1:2, atTick1=fda::monthBegin.5,
                          atTick2=fda::monthEnd.5, atLabels=fda::monthMid,
                          labels=month.abb, cex.axis=0.9, las=1, ...)
{
  axisIntervals(side[1], atTick1=atTick1, atTick2=atTick2,
                labels=labels, cex.axis=cex.axis, las=las, ...)
  axis(side[2], las=las, ...)
}

#  Generator function of class basisfd

basisfd <- function(type, rangeval, nbasis, params, dropind=vector("list",0),
                    quadvals=vector("list",0), values=vector("list",0),
                    basisvalues=vector("list",0))
{
  #  BASISFD  generator function of "basisfd" class.
  #  Arguments:
  #  TYPE    ...a string indicating the type of basisobj.
  #             This may be one of:
  #             "Bspline", "bspline", "Bsp", "bsp",
  #             "con", "const", "constant"
  #             "exp", "exponen", "exponential"
  #             "fdVariance"
  #             "FEM"
  #             "Fourier", "fourier", "Fou", "fou",
  #             "mon", "monom", "monomial",
  #             "polyg", "polygon", "polygonal"
  #             "power" "pow"
  #  RANGEVAL...an array of length 2 containing the lower and upper
  #             boundaries for (the rangeval of argument values
  #             If basis is of FEM type, rangeval is not used
  #  NBASIS ... the number of basis functions
  #  PARAMS ... If the basis is "fourier", this is a single number indicating
  #               the period.  That is, the basis functions are periodic on
  #               the interval (0,PARAMS) or any translation of it.
  #             If the basis is "bspline", the values are interior points at
  #               which the piecewise polynomials join.
  #               Note that the number of basis functions NBASIS is equal
  #               to the order of the Bspline functions plus the number of
  #               interior knots, that is the length of PARAMS.
  #             This means that NBASIS must be at least 1 larger than the
  #               length of PARAMS.
  #  DROPIND...A set of indices in 1:NBASIS of basis functions to drop when
  #              basis objects are arguments.  Default is vector("list",0)
  #              Note that argument NBASIS is reduced by the number of
  #              indices, and the derivative matrices in VALUES are also clipped.
  #  QUADVALS...A NQUAD by 2 matrix.  The firs t column contains quadrature
  #              points to be used in a fixed point quadrature.  The second
  #              contains quadrature weights.  For example, for (Simpson"s
  #              rule for (NQUAD = 7, the points are equally spaced and the
  #              weights are delta.*[1, 4, 2, 4, 2, 4, 1]/3.  DELTA is the
  #              spacing between quadrature points.  The default is
  #              matrix("numeric",0,0).
  #  VALUES ...A list, with entries containing the values of
  #              the basis function derivatives starting with 0 and
  #              going up to the highest derivative needed.  The values
  #              correspond to quadrature points in QUADVALS and it is
  #              up to the user to decide whether or not to multiply
  #              the derivative values by the square roots of the
  #              quadrature weights so as to make numerical integration
  #              a simple matrix multiplication.
  #              Values are checked against QUADVALS to ensure the correct
  #              number of rows, and against NBASIS to ensure the correct
  #              number of columns.
  #              The default value of is VALUES is vector("list",0).
  #              VALUES contains values of basis functions and derivatives at
  #              quadrature points weighted by square root of quadrature weights.
  #              These values are only generated as required, and only if slot
  #              QUADVALS is not matrix("numeric",0,0).
  #  BASISVALUES...A vector of lists, allocated by code such as
  #              vector("list",1).
  #              This field is designed to avoid evaluation of a
  #              basis system repeatedly at a set of argument values.
  #              Each list within the vector corresponds to a specific set
  #              of argument values, and must have at least two components,
  #              which may be tagged as you wish.
  #              The first component in an element of the list vector contains the
  #              argument values.
  #              The second component in an element of the list vector
  #              contains a matrix of values of the basis functions evaluated
  #              at the arguments in the first component.
  #              The third and subsequent components, if present, contain
  #              matrices of values their derivatives up to a maximum
  #              derivative order.
  #              Whenever function getbasismatrix is called, it checks
  #              the first list in each row to see, first, if the number of
  #              argument values corresponds to the size of the first dimension,
  #              and if this test succeeds, checks that all of the argument
  #              values match.  This takes time, of course, but is much
  #              faster than re-evaluation of the basis system.  Even this
  #              time can be avoided by direct retrieval of the desired
  #              array.
  #              For example, you might set up a vector of argument values
  #              called "evalargs" along with a matrix of basis function
  #              values for these argument values called "basismat".
  #              You might want too use tags like "args" and "values",
  #              respectively for these.  You would then assign them
  #              to BASISVALUES with code such as
  #                basisobj$basisvalues <- vector("list",1)
  #                basisobj$basisvalues[[1]] <-
  #                             list(args=evalargs, values=basismat)
  #
  #  Returns
  #  BASISOBJ  ... a basisfd object with slots
  #         type
  #         rangeval
  #         nbasis
  #         params
  #         dropind
  #         quadvals
  #         values
  #         basisvalues
  #  Slot VALUES contains values of basis functions and derivatives at
  #   quadrature points weighted by square root of quadrature weights.
  #   These values are only generated as required, and only if slot
  #   quadvals is not empty.
  #
  #  An alternative name for (this function is CREATE.BASIS, but PARAMS argument
  #     must be supplied.
  #  Specific types of bases may be set up more conveniently using functions
  #  CREATE.BSPLINE.BASIS     ...  creates a b-spline basis
  #  CREATE.CONSTANT.BASIS    ...  creates a constant basis
  #  CREATE.EXPONENTIAL.BASIS ...  creates an exponential basis
  #  CREATE.FDVARIANCE.BASIS  ...  creates an fdVariance basis
  #  CREATE.FEM.BASIS         ...  creates an FEM basis  
  #  CREATE.FOURIER.BASIS     ...  creates a fourier basis
  #  CREATE.MONOMIAL.BASIS    ...  creates a monomial basis
  #  CREATE.POLYGON.BASIS     ...  creates a polygonal basis
  #  CREATE.POLYNOMIAL.BASIS  ...  creates a polynomial basis
  #  CREATE.POWER.BASIS       ...  creates a monomial basis
  
  #  Last modified 3 January 2020 by Jim Ramsay
  # value -> values 2012.12.27 by spencer graves
  
  #  Set up default basis if there are no arguments:
  #     order 2 monomial basis over [0,1]
  
  if (nargs()==0) {
    type        <- "bspline"
    rangeval    <- c(0,1)
    nbasis      <- 2
    params      <- vector("list",0)
    dropind     <- vector("list",0)
    quadvals    <- vector("list",0)
    values      <- vector("list",0)
    basisvalues <- vector("list",0)
    
    basisobj  <- list(type=type,     rangeval=rangeval, nbasis=nbasis,
                      params=params, dropind=dropind,   quadvals=quadvals,
                      values=values, basisvalues=basisvalues)
    oldClass(basisobj) <- "basisfd"
    return(basisobj)
  }
  
  #  if first argument is a basis object, return
  
  if (inherits(type,"basisfd")) {
    basisobj <- type
    return(basisobj)
  }
  
  #  check basistype
  
  # type <- moreNames(type)
  
  #  recognize type of basis by use of several variant spellings
  
  if(type == "bspline" ||
     type == "Bspline" ||
     type == "spline"  ||
     type == "Bsp"     ||
     type == "bsp") {
    type = "bspline"
  }
  else if(type == "con"      ||
          type == "const"    ||
          type == "constant") {
    type = "const"
  }
  else if(type == "exp"    ||
          type == "expon"  ||
          type == "exponential") {
    type = "expon"
  }
  else if(type == "fdVariance" ||
          type == "fdVar") {
    type = "fdVariance"
  }
  else if(type == "FEM") {
    type = "FEM"
  }
  else if(type == "Fourier" ||
          type == "fourier" ||
          type == "Fou"     ||
          type == "fou") {
    type = "fourier"
  }
  else if(type == "mon" ||
          type == "monom"  ||
          type == "monomial") {
    type = "monom"
  }
  else if(type == "polyg"    ||
          type == "polygon"  ||
          type == "polygonal") {
    type = "polygonal"
  }
  else if(type == "polynomial"    ||
          type == "polynom") {
    type = "polynomial"
  }
  else if(type == "pow"    ||
          type == "power") {
    type = "power"
  }
  else {
    type = "unknown"
  }
  
  if (type=="unknown"){
    stop("'type' unrecognizable.")
  }
  
  #  check rangeval if the object is not of type FEM
  
  if (!type == "FEM") {
    rangeval = as.vector(rangeval)
    if (!is.numeric(rangeval)) stop("Argument rangeval is not numeric.")
    if (length(rangeval) != 2) stop("Argument rangeval is not of length 2.")
    if (!(rangeval[2] > rangeval[1]))
      stop("Argument rangeval is not strictly increasing.")
  }
  
  #  check nbasis
  
  if (nbasis <= 0)             stop("Argument nbasis is not positive.")
  if (round(nbasis) != nbasis) stop("Argument nbasis is not an integer.")
  
  #  check if QUADVALS is present, and set to default if not
  
  if (missing(quadvals)) quadvals <- vector("list",0)
  else if(!(length(quadvals) == 0 || is.null(quadvals))){
    nquad <- dim(quadvals)[1]
    ncol  <- dim(quadvals)[2]
    if ((nquad == 2) && (ncol > 2)){
      quadvals <- t(quadvals)
      nquad    <- dim(quadvals)[1]
      ncol     <-dim(quadvals)[2]
    }
    if (nquad < 2) stop("Less than two quadrature points are supplied.")
    if (ncol != 2) stop("'quadvals' does not have two columns.")
  }
  
  #  check VALUES is present, and set to a single empty list if not.
  if(!(length(values) == 0 || missing(values) || is.null(values))) {
    n <- dim(values)[1]
    k <- dim(values)[2]
    if (n != nquad)
      stop(paste("Number of rows in 'values' not equal to number of",
                 "quadrature points."))
    if (k != nbasis)
      stop(paste("Number of columns in 'values' not equal to number of",
                 "basis functions."))
  }
  else values <- vector("list",0)
  
  #  check BASISVALUES is present, and set to vector("list",0) if not.
  #  If present, it must be a two-dimensional list created by a command like
  #  listobj <- matrix("list", 2, 3)
  
  if(!(length(basisvalues) == 0 || missing(basisvalues) || !is.null(basisvalues))) {
    if (!is.list(basisvalues)) stop("BASISVALUES is not a list object.")
    sizevec <- dim(basisvalues)
    if (length(sizevec) != 2) stop("BASISVALUES is not 2-dimensional.")
    for (i in 1:sizevec[1]) {
      if (length(basisvalues[[i,1]]) != dim(basisvalues[[i,2]])[1]) stop(
        paste("Number of argument values not equal number",
              "of values."))
    }
  }
  else basisvalues <- vector("list",0)
  
  #  check if DROPIND is present, and set to default if not
  
  if(missing(dropind)) dropind <- vector("list",0)
  
  if (length(dropind) > 0) {
    #  check DROPIND
    ndrop = length(dropind)
    if (ndrop >= nbasis) stop('Too many index values in DROPIND.')
    dropind = sort(dropind)
    if (ndrop > 1 && any(diff(dropind)) == 0)
      stop('Multiple index values in DROPIND.')
    for (i in 1:ndrop) {
      if (dropind[i] < 1 || dropind[i] > nbasis)
        stop('A DROPIND index value is out of range.')
    }
    #  drop columns from VALUES cells if present
    nvalues = length(values)
    if (nvalues > 0 && length(values[[1]] > 0)) {
      for (ivalue in 1:nvalues) {
        derivvals = values[[ivalue]]
        derivvals = derivvals[,-dropind]
        values[[ivalue]] = derivvals
      }
    }
  }
  
  #  select the appropriate type and process
  
  if (type=="fourier"){
    paramvec   <- rangeval[2] - rangeval[1]
    period     <- params[1]
    if (period <= 0)  stop("Period must be positive for (a Fourier basis")
    params <- period
    if ((2*floor(nbasis/2)) == nbasis)  nbasis <- nbasis + 1
  } else if(type=="bspline"){
    if (!missing(params)){
      nparams  <- length(params)
      if(nparams>0){
        if (params[1] <= rangeval[1])
          stop("Smallest value in BREAKS not within RANGEVAL")
        if (params[nparams] >= rangeval[2])
          stop("Largest value in BREAKS not within RANGEVAL")
      }
    }
  } else if(type=="expon") {
    if (length(params) != nbasis)
      stop("No. of parameters not equal to no. of basis fns for (exponential basisobj$")
  }   else if(type=="fdVariance") {
    if (length(params) != 2)
      stop("No. of parameters not equal to 8 for (FEM basisobj$")
  } else if(type=="FEM") {
    # if (length(params) != 8)
    #   stop("No. of parameters not equal to 8 for (FEM basisobj$")
  } else if(type=="polynomial") {
    if (length(params) != nbasis)
      stop("No. of parameters not equal to no. of basis fns for (polynomialal basisobj$")
  } else if(type=="power") {
    if (length(params) != nbasis)
      stop("No. of parameters not equal to no. of basis fns for (power basisobj$")
  } else if(type=="const") {
    params <- 0
  } else if(type=="monom") {
    if (length(params) != nbasis)
      stop("No. of parameters not equal to no. of basis fns for (monomial basisobj$")
  } else if(type=="polygonal") {
    if (length(params) != nbasis)
      stop("No. of parameters not equal to no. of basis fns for polygonal basisobj$")
  } else stop("Unrecognizable basis")
  
  #  Save call
  
  obj.call <- match.call()
  
  #  S4 definition
  
  # basisobj <- new("basisfd", call=obj.call, type=type, rangeval=rangeval,
  #                 nbasis=nbasis,  params=params, dropind=dropind,
  #                 quadvals=quadvals, values=values, basisvalues=basisvalues)
  
  #  S3 definition
  
  basisobj <- list(call=obj.call, type=type, rangeval=rangeval, nbasis=nbasis,
                   params=params, dropind=dropind, quadvals=quadvals,
                   values=values, basisvalues=basisvalues)
  oldClass(basisobj) <- "basisfd"
  
  basisobj
  
}

#  --------------------------------------------------------------------------
#                  print for basisfd class
#  --------------------------------------------------------------------------

print.basisfd <- function(x, ...)
{
  
  #  Last modified 3 January 2008 by Jim Ramsay
  
  basisobj <- x
  cat("\nBasis object:\n")
  if (!inherits(basisobj, "basisfd"))
    stop("Argument not a functional data object")
  
  #  print type
  
  cat(paste("\n  Type:  ", basisobj$type,"\n"))
  
  #  print range
  
  cat(paste("\n  Range: ", basisobj$rangeval[1],
            " to ",        basisobj$rangeval[2],"\n"))
  
  #  return if a constant basis
  
  if (basisobj$type == "const") {
    return()
  }
  
  #  print number of basis functions
  
  cat(paste("\n  Number of basis functions: ",
            basisobj$nbasis,     "\n"))
  
  #  print parameters according to type of basis
  
  if (basisobj$type == "fourier")
    cat(paste("\n  Period: ",basisobj$params,"\n"))
  if (basisobj$type == "bspline") {
    norder <- basisobj$nbasis - length(basisobj$params)
    cat(paste("\n  Order of spline: ", norder, "\n"))
    if (length(basisobj$params) > 0) {
      print("  Interior knots")
      print(basisobj$params)
    } else {
      print("  There are no interior knots.")
    }
  }
  if (basisobj$type == "polyg") {
    print("  Argument values")
    print(basisobj$params)
  }
  if (basisobj$type == "expon") {
    print("  Rate coefficients")
    print(basisobj$params)
  }
  if (basisobj$type == "monom") {
    print("  Exponents")
    print(basisobj$params)
  }
  if (basisobj$type == "power") {
    print("  Exponents")
    print(basisobj$params)
  }
  
  
  #  display indices of basis functions to be dropped
  
  if (length(basisobj$dropind) > 0) {
    print("  Indices of basis functions to be dropped")
    print(basisobj$dropind)
  }
  
}

#  --------------------------------------------------------------------------
#                  summary for basisfd class
#  --------------------------------------------------------------------------

summary.basisfd <- function(object, ...)
{
  basisobj <- object
  cat("\nBasis object:\n")
  if (!inherits(basisobj, "basisfd"))
    stop("Argument not a functional data object")
  cat(paste("\n  Type:  ", basisobj$type,"\n"))
  cat(paste("\n  Range: ", basisobj$rangeval[1],
            " to ",        basisobj$rangeval[2],"\n"))
  if (basisobj$type == "const") {
    return()
  }
  cat(paste("\n  Number of basis functions: ",
            basisobj$nbasis,     "\n"))
  if (basisobj$type == "fourier")
    cat(paste("\n  Period: ",basisobj$params,"\n"))
  if (length(basisobj$dropind) > 0) {
    print(paste(length(basisobj$dropind),
                "indices of basis functions to be dropped"))
  }
}

#  --------------------------------------------------------------------------
#  equality for basisfd class
#  --------------------------------------------------------------------------

"==.basisfd" <- function(basis1, basis2)
{
  
  # EQ assesses whether two bases are equivalent.
  
  #  Last modified 1 January 2007
  
  type1   <- basis1$type
  range1  <- basis1$rangeval
  nbasis1 <- basis1$nbasis
  pars1   <- basis1$params
  drop1   <- basis1$dropind
  
  type2   <- basis2$type
  range2  <- basis2$rangeval
  nbasis2 <- basis2$nbasis
  pars2   <- basis2$params
  drop2   <- basis2$dropind
  
  basisequal <- TRUE
  
  #  check types
  
  if (!(type1 == type2)) {
    basisequal <- FALSE
    return(basisequal)
  }
  
  #  check ranges
  
  if (range1[1] != range2[1] || range1[2] != range2[2]) {
    basisequal <- FALSE
    return(basisequal)
  }
  
  #  check numbers of basis functions
  
  if (nbasis1 != nbasis2) {
    basisequal <- FALSE
    return(basisequal)
  }
  
  #  check parameter vectors
  
  if (!(all(pars1 == pars2))) {
    basisequal <- FALSE
    return(basisequal)
  }
  
  #  check indices of basis function to drop
  
  if (!(all(drop1 == drop2))) {
    basisequal <- FALSE
    return(basisequal)
  }
  
  return(basisequal)
  
}

#  --------------------------------------------------------------------------
#  pointwise multiplication method for basisfd class
#  --------------------------------------------------------------------------

"*.basisfd" <- function (basisobj1, basisobj2)
{
  # TIMES for (two basis objects sets up a basis suitable for (
  #  expanding the pointwise product of two functional data
  #  objects with these respective bases.
  # In the absence of a true product basis system in this code,
  #  the rules followed are inevitably a compromise:
  #  (1) if both bases are B-splines, the norder is the sum of the
  #      two orders - 1, and the breaks are the union of the
  #      two knot sequences, each knot multiplicity being the maximum
  #      of the multiplicities of the value in the two break sequences.
  #      Order, however, is not allowed to exceed 20.
  #      That is, no knot in the product knot sequence will have a
  #      multiplicity greater than the multiplicities of this value
  #      in the two knot sequences.
  #      The rationale this rule is that order of differentiability
  #      of the product at each value will be controlled  by
  #      whichever knot sequence has the greater multiplicity.
  #      In the case where one of the splines is order 1, or a step
  #      function, the problem is dealt with by replacing the
  #      original knot values by multiple values at that location
  #      to give a discontinuous derivative.
  #  (2) if both bases are Fourier bases, AND the periods are the
  #      the same, the product is a Fourier basis with number of
  #      basis functions the sum of the two numbers of basis fns.
  #  (3) if only one of the bases is B-spline, the product basis
  #      is B-spline with the same knot sequence and order two
  #      higher.
  #  (4) in all other cases, the product is a B-spline basis with
  #      number of basis functions equal to the sum of the two
  #      numbers of bases and equally spaced knots.
  
  #  Of course the ranges must also match.
  
  #  Last modified 2022.04.19 by Jim Ramsay
  
  #  check the ranges
  
  range1 <- basisobj1$rangeval
  range2 <- basisobj2$rangeval
  if (range1[1] != range2[1] || range1[2] != range2[2])
    stop("Ranges are not equal.")
  
  #  get the types
  
  type1 <- basisobj1$type
  type2 <- basisobj2$type
  
  #  deal with constant bases
  
  if (type1 == "const" && type2 == "const") {
    prodbasisobj <- create.constant.basis(range1)
    return(prodbasisobj)
  }
  
  if (type1 == "const") {
    prodbasisobj <- basisobj2
    return(prodbasisobj)
  }
  
  if (type2 == "const") {
    prodbasisobj <- basisobj1
    return(prodbasisobj)
  }
  
  #  get the numbers of basis functions
  
  nbasis1 <- basisobj1$nbasis
  nbasis2 <- basisobj2$nbasis
  
  #  work through the cases
  
  if (type1 == "bspline" && type2 == "bspline") {
    #  both are bases B-splines
    #  get orders
    interiorknots1 <- basisobj1$params
    interiorknots2 <- basisobj2$params
    #    uniqueknots    <- sort(union(interiorknots1, interiorknots2))
    interiorknots1.2 <- union(interiorknots1, interiorknots2)
    uniqueknots <- {
      if(is.null(interiorknots1.2)) NULL else sort(interiorknots1.2)
    }
    nunique <- length(uniqueknots)
    multunique <- rep(0,nunique)
    for (i in seq(length=nunique)) {
      mult1 <- {
        if(length(interiorknots1)>0)
          length(interiorknots1[interiorknots1==uniqueknots[i]])
        else 0
      }
      mult2 <- {
        if(length(interiorknots2)>0)
          length(interiorknots2[interiorknots2==uniqueknots[i]])
        else 0
      }
      multunique[i] <- max(mult1,mult2)
    }
    #
    allknots <- rep(0,sum(multunique))
    m2 <- 0
    for (i in seq(length=nunique)) {
      m1 <- m2 + 1
      m2 <- m2 + multunique[i]
      allknots[m1:m2] <- uniqueknots[i]
    }
    norder1 <- nbasis1 - length(interiorknots1)
    norder2 <- nbasis2 - length(interiorknots2)
    #  norder is not allowed to exceed 20
    norder  <- min(c(norder1 + norder2 - 1,20))
    allbreaks  <- c(range1[1], allknots, range1[2])
    nbasis <- length(allbreaks) + norder - 2
    prodbasisobj <-
      create.bspline.basis(range1, nbasis, norder, allbreaks)
    return(prodbasisobj)
  }
  
  if (type1 == "fourier" && type2 == "fourier") {
    #  both bases Fourier
    #  check whether periods match
    #  if they do not, default to the basis below.
    period1 <- basisobj1$params
    period2 <- basisobj2$params
    nbasis  <- nbasis1 + nbasis2-1
    if (period1 == period2) {
      prodbasisobj <- create.fourier.basis(range1, nbasis, period1)
      return(prodbasisobj)
    }
  }
  
  #  Default case when all else fails: the product basis is B-spline
  #  When neither basis is a B-spline basis, the order
  #  is the sum of numbers of bases, but no more than 8.
  #  When one of the bases if B-spline and the other isn"t,
  #  the order is the smaller of 8 or the order of the spline
  #  plus 2.  Under no circumstances can the order exceed 20, however.
  #  See BsplineS where this restriction is tested.
  
  if (type1 == "bspline" || type2 == "bspline") {
    norder <- 8
    if (type1 == "bspline") {
      interiorknots1 <- basisobj1$params
      norder1        <- nbasis1 - length(interiorknots1)
      norder         <- min(c(norder1+2, norder))
    }
    if (type2 == "bspline") {
      interiorknots2 <- basisobj2$params
      norder2        <- nbasis2 - length(interiorknots2)
      norder         <- min(c(norder2+2, norder))
    }
  } else {
    #  neither basis is B-spline
    norder <- min(c(8, nbasis1+nbasis2))
  }
  #  set up the default B-spline product basis
  nbasis <- max(c(nbasis1+nbasis2, norder+1))
  prodbasisobj <- create.bspline.basis(range1, nbasis, norder)
  return(prodbasisobj)
}

#  ---------------------------------------------------------
#        Subscripted reference to a basis object
#  ---------------------------------------------------------

#  Last modified 22 December 2007

"[.basisfd" <- function(basisobj, subs=TRUE)
{
  #  select subsets of basis functions in a basis object
  
  dropind = vector("numeric", 0)
  nbasis <- basisobj$nbasis
  for (i in 1:nbasis) {
    if (!any(subs==i)) dropind = c(dropind, i)
  }
  basisobj$dropind <- dropind
  return(basisobj)
}

bifdPar = function(bifdobj, Lfdobjs=int2Lfd(2), Lfdobjt=int2Lfd(2), 
                   lambdas=0, lambdat=0, estimate=TRUE) {
# Sets up a bivariate functional parameter object
#  Arguments:
#  BIFDOBJ  ... A bivariate functional data object.  The basis for this object 
#               is used to define the bivariate functional parameter.
#               When an initial value is required for iterative 
#               estimation of a bivariate functional parameter, the coefficients
#               will give the initial values for the iteration.
#  LFDOBJS  ... A linear differential operator value or a derivative
#               value for penalizing the roughness of the object
#               with respect to the first argument s.
#               By default, this is 2.
#  LFDOBJT  ... A linear differential operator value or a derivative
#               value for penalizing the roughness of the object
#               with respect to the second argument t.
#               By default, this is 2.
#  LAMBDAS  ... The penalty parameter controlling the smoothness of
#               the estimated parameter with respect to the first argument s.  
#               By default this is 0.
#  LAMBDAT  ... The penalty parameter controlling the smoothness of
#               the estimated parameter with respect to the second argument t.  
#               By default this is 0.
#  ESTIMATE ... If nonzero, the parameter is estimated; if zero, the
#                parameter is held fixed at this value.
#                By default, this is 1.

#  last modified 28 October 2009

#  check BIFDOBJ

if (!inherits(bifdobj, "bifd")) {
    stop("BIFDOBJ is not a bivariate functional data object.")
}

#  check the linear differential operators

Lfdobjs = int2Lfd(Lfdobjs)
Lfdobjt = int2Lfd(Lfdobjt)

if (!is.Lfd(Lfdobjs)) {
    stop("LFDOBJS is not a linear differential operator object.")
}
if (!is.Lfd(Lfdobjt)) {
    stop("LFDOBJT is not a linear differential operator object.")
}

#  check the roughness penalty parameters

if (!is.numeric(lambdas) ) {
    stop("LAMBDAS is not numeric.")
}
if (lambdas < 0) {
    warning("LAMBDAS is negative, and is set to zero.")
    lambdas = 0
}
if (!is.numeric(lambdat)) {
    stop("LAMBDAT is not numeric.")
}
if (lambdat < 0) {
    warning("LAMBDAT is negative, and is set to zero.")
    lambdat = 0
}

if (!is.logical(estimate)) {
    stop("ESTIMATE is not logical.")
}

#  set up the bifdPar object

bifdParobj <- list(bifd=bifdobj, estimate=estimate, 
                   lambdas=lambdas, lambdat=lambdat, Lfds=Lfdobjs, Lfdt=Lfdobjt, 
                   estimate=estimate)

oldClass(bifdParobj) <- "bifdPar"

return(bifdParobj)

}
#  setClass for "bifd"

# setClass("bifd",  representation(coefs     = "array",
#                                  sbasis    = "basisfd",
#                                  tbasis    = "basisfd",
#                                  bifdnames = "list"))

#  Generator function of class bifd

bifd <- function (coef=matrix(0,2,1), sbasisobj=create.bspline.basis(),
                  tbasisobj=create.bspline.basis(), fdnames=defaultnames)
{
  #  This function creates a bivariate functional data object.
  #    A bivariate functional data object consists of two bases for expanding
  #      a bivariate functional observation and a set of coefficients defining
  #      this expansion.
  #    The bases are contained in "basisfd" objects; that is, a realization
  #    of the "basisfd" class.

  #  Arguments:
  #  COEF     ... a two-, three-, or four-dimensional array containing
  #               coefficient values for the expansion of each set of bivariate
  #               function values=terms of a set of basis function values
  #               If COEF is a two-way, it is assumed that there is only
  #                 one variable and only one replication, and then
  #                 the first and second dimensions correspond to
  #                 the basis functions for the first and second argument,
  #                 respectively.
  #               If COEF is a three-way, it is assumed that there is only
  #                 one variable per replication, and then
  #                 the first and second dimensions correspond to
  #                 the basis functions for the first and second argument,
  #                 respectively, and the third dimension corresponds to
  #                 replications.
  #               If COEF is a four-way array, then the fourth dimension
  #                 corresponds to variables
  #  SBASISOBJ ... a functional data basis object for the first  argument s
  #  TBASISOBJ ... a functional data basis object for the second argument t
  #  BIFDNAMES ... A list of length 3 with members containing
  #               1. a single name for the argument domain, such as 'Time'
  #               2. a name for the replications or cases
  #               3. a name for the function.

  #  Returns:
  #  BIFDOBJ ... a bivariate functional data object
#  last modified 2007 May 3 by Spencer Graves
  #  previously modified 20 September 2005

  #  check COEF and get its dimensions

    if(!is.numeric(coef)) stop(
		"coef must be numerical vector or matrix")
    else if (is.vector(coef)) stop(
		"Argument COEF is not at least 2 dimensional.")
    else if (is.matrix(coef)) {
            coefd <- dim(coef)
            ndim  <- length(coefd)
        }
    else if (is.array(coef)) {
            coefd <- dim(coef)
            ndim  <- length(coefd)
        }
    else stop("argument COEF is not correct")

    if (ndim > 4) stop(
        "First argument not of dimension 2, 3 or 4.")

    #  check SBASISOBJ

    if (!inherits(sbasisobj, "basisfd")) stop(
        "Argument SBASISOBJ is not of basis class")

    if (dim(coef)[1] != sbasisobj$nbasis) stop(
        paste("Number of coefficients does not match number of ",
              "basis functions for SBASISOBJ."))

    #  check TBASISOBJ

    if (!inherits(tbasisobj, "basisfd")) stop(
        "Argument TBASISOBJ is not of basis class.")

    if (dim(coef)[2] != tbasisobj$nbasis) stop(
        paste("Number of coefficients does not match number of ",
              "basis functions for TBASISOBJ."))

    #  setup number of replicates and number of variables

    if (ndim > 2) nrep <- coefd[3] else nrep <- 1
    if (ndim > 3) nvar <- coefd[4] else nvar <- 1

    #  set up default fdnames

    if (ndim == 2) defaultnames <- list("time", "reps", "values")
    if (ndim == 2) defaultnames <- list("time",
                                        paste("reps",as.character(1:nrep)),
                                        "values")
    if (ndim == 4) defaultnames <- list("time",
                                        paste("reps",as.character(1:nrep)),
                                        paste("values",as.character(1:nvar)) )

    names(defaultnames) <- c("args", "reps", "funs")

#  S4 definition
#	bifdobj <- new("bifd", coefs=coef, sbasis=sbasisobj, tbasis=tbasisobj,
#	               bifdnames=fdnames)

#  S3 definition

	bifdobj <- list(coefs=coef, sbasis=sbasisobj, tbasis=tbasisobj,
	               bifdnames=fdnames)
	oldClass(bifdobj) <- "bifd"
	
	bifdobj
}

#  "show" method for "bifd"

print.bifd <- function(x, ...)
{
  object <- x	
	cat("bifd:\n\n")
	
	cat("Dimensions of the data:\n")
	  cat(paste("  ",object$fdnames[[1]],"\n"))
	  cat(paste("  ",object$fdnames[[2]],"\n"))
	  cat(paste("  ",object$fdnames[[3]],"\n"))
	  cat("\n")
	
	print(object$sbasis)
	
	print(object$tbasis)
	
}

#  "summary" method for "bifd"

summary.bifd <- function(object, ...)
{
	
	cat("bifd:\n\n")
	
	cat("Dimensions of the data:\n")
	  cat(paste("  ",object$fdnames[[1]],"\n"))
	  cat(paste("  ",object$fdnames[[2]],"\n"))
	  cat(paste("  ",object$fdnames[[3]],"\n"))
	  cat("\n")
	
	print(object$sbasis)
	
	print(object$tbasis)
	
	cat("\nCoefficient matrix:\n\n")
	
	object$coefs
	
}

boxplot.fdSmooth <- function(x, z=NULL, ...){
  boxplot(x$fd, z, ...)
}

boxplot.fdPar <- function(x, z=NULL, ...){
  boxplot(x$fd, z, ...)
}

boxplot.fd <- function(x, z=NULL, ...){
  if(is.numeric(x)){
      fbplot(x, z, ...)
  } else {
      if(is.null(z)){
          rng <- getbasisrange(x$basis)
          z <- seq(rng[1], rng[2], length=101)
      }
      x. <- eval.fd(z, x)
#      x. <- predict(x, z)
      dots <- list(...)
      if(!('xlim' %in% names(dots)))xlim <- range(z)
      if(!('ylim' %in% names(dots)))
          ylim <- c(min(x.)-.5*diff(range(x.)),max(x.)+.5*diff(range(x.)))
      dots$fit <- x.
      dots$x <- z
      dots$xlim <- xlim
      dots$ylim <- ylim
      do.call(fbplot, dots)
#      fbplot(x., z, method=method, depth=depth, plot=plot, prob=prob,
#         color=color, outliercol=outliercol, barcol=barcol,
#         fullout=fullout, factor=factor, ...)
  }
}
bsplinepen <- function(basisobj, Lfdobj=2, rng=basisobj$rangeval,
                       returnMatrix=FALSE)
{

#  Computes the Bspline penalty matrix.
#  Arguments:
#  BASISOBJ    a basis.fd object of type "bspline"
#  LFDOBJ      a linear differential operator object.
#  RNG         a range over which the product is evaluate
#  RETURNMATRIX ... If False, a matrix in sparse storage model can be returned
#               from a call to function BsplineS.  See this function for
#               enabling this option.

#  Returns the penalty matrix.

#  Last modified July 6, 2012 by Spencer Graves
#    for rng, rangeval, and params of class Date and POSIXct
#  Previously modified 9 May by Jim Ramsay

#  check BASISOBJ

  if (!(inherits(basisobj, "basisfd"))) stop(
    "First argument is not a basis.fd object.")

#  check basis type

  type <- basisobj$type
  if (type != "bspline") stop("basisobj not of type bspline")

#  check LFDOBJ

  Lfdobj <- int2Lfd(Lfdobj)

#  get basis information

  nbasis <- basisobj$nbasis
  Params <- basisobj$params

#  if there are no internal knots, use the monomial penalty

  if (length(Params) == 0) {
    basisobj      <- create.monomial.basis(rng, nbasis, 0:(nbasis-1))
    penaltymatrix <- monomialpen(basisobj, Lfdobj, rng)
    return(penaltymatrix)
  }

#  normal case:  PARAMS is not empty

# as.numeric
  Rangeval  <- basisobj$rangeval
  Rng <- rng
  op <- options(warn=-1)
  rng <- as.numeric(Rng)
  rangeval <- as.numeric(Rangeval)
  params <- as.numeric(Params)
  options(op)
# check
  nNA <- sum(is.na(rng))
  if(nNA>0)
    stop('as.numeric(rng) contains ', nNA,
         ' NA', c('', 's')[1+(nNA>1)],
         ';  class(rng) = ', class(Rng))
  nNAr <- sum(is.na(rangeval))
  if(nNAr>0)
    stop('as.numeric(rangeval) contains ', nNAr,
         ' NA', c('', 's')[1+(nNAr>1)],
         ';  class(rangeval) = ', class(Rangeval))
  nNAp <- sum(is.na(params))
  if(nNAp>0)
    stop('as.numeric(params) contains ', nNAp,
         ' NA', c('', 's')[1+(nNAp>1)],
         ';  class(params) = ', class(Params))

  breaks    <- c(rangeval[1],params,rangeval[2])  #  break points
  nbreaks   <- length(breaks)
  ninterval <- nbreaks - 1    #  number of intervals defined by breaks

#  check break values

  if (length(breaks) < 2)
    stop("The length of argument breaks is less than 2.")

#  Find the highest order derivative in LFD

  nderiv <- Lfdobj$nderiv

  norder <- nbasis - length(params)

#  check for order of derivative being equal or greater than
#  order of spline

  if (nderiv >= norder) {
    cat("\n")
    cat(paste(" Derivative of order", nderiv,
              "cannot be taken for B-spline of order", norder,"\n"))
    cat(" Probable cause is a value of the nbasis argument\n")
    cat(" in function create.basis.fd that is too small.\n")
    stop()
}

#  check for order of derivative being equal to order of spline
#  minus one, in which case following code won't work.

  if (nderiv > 0 && nderiv == norder - 1)
    stop(paste("Penalty matrix cannot be evaluated for derivative of order ",
               nderiv, " for B-splines of order ", norder))

#  special case where LFD is D^NDERIV and NDERIV = NORDER - 1

  bwtlist <- Lfdobj$bwtlist
  isintLfd <- TRUE
  if (nderiv > 0) {
    for (ideriv in 1:nderiv) {
      fdj <- bwtlist[[ideriv]]
      if (!is.null(fdj)) {
        if (any(fdj$coefs != 0)) {
            isintLfd <- FALSE
            break
        }
      }
    }
  }

  if (isintLfd && nderiv == norder - 1) {
    #  special case of nderiv = norder - 1
    halfseq    <- (breaks[2:nbreaks] + breaks[1:(nbreaks-1)])/2
    halfmat    <- bsplineS(halfseq, breaks, norder, nderiv, returnMatrix)
    brwidth    <- diff(breaks)
    penaltymat <- (t(halfmat) %*% diag(brwidth) %*% halfmat)
    return(penaltymat)
  }

#  look for knot multiplicities within the range

  intbreaks    <- c(rng[1], params, rng[2])
  index        <- intbreaks >= rng[1] & intbreaks <= rng[2]
  intbreaks    <- intbreaks[index]
  nintbreaks   <- length(intbreaks)
  uniquebreaks <- min(diff(intbreaks)) > 0

#  if LFD is D^NDERIV, and there are no break multiplicities,
#  use exact computation

  if (isintLfd && rng[1] == rangeval[1] &&
    uniquebreaks      && rng[2] == rangeval[2]) {

    #  Set up the knot sequence

    onesv <- matrix(1,1,norder)
    knots <- c(rangeval[1]*onesv, breaks[2:(nbreaks-1)], rangeval[2]*onesv)

    # Construct  the piecewise polynomial representation

    polyorder <- norder - nderiv
    ndegree   <- polyorder - 1
    prodorder <- 2*ndegree + 1   # order of product
    polycoef  <- array(0, c(ninterval, polyorder, norder))
    indxdown  <- seq(norder, (nderiv+1), -1)
    for (i in 1:nbasis) {
        #  compute polynomial representation of B(i,norder,t)(x)
        t       <- knots[i:(i+norder)]
        ppBlist <- ppBspline(t)
        Coeff   <- ppBlist[[1]]
        index   <- ppBlist[[2]]
        nrowcoef <- dim(Coeff)[1]
        # convert the index of the breaks in t to the index in the
        # variable "breaks"
        index <- index + i - norder
        CoeffD <- matrix(Coeff[,1:polyorder],nrowcoef,polyorder)
        if (nderiv > 0) {
            for (ideriv in 1:nderiv) {
                fac    <- indxdown - ideriv
                CoeffD <- outer(rep(1,nrowcoef),fac) * CoeffD
            }
        }
        # add the polynomial representation of B(i,norder,t)(x) to f
        if (i >= norder) k <- norder else k <- i
        if (i <= norder) m <- i      else m <- norder
        for (j in 1:nrowcoef) polycoef[i-k+j,,m-j+1] <- CoeffD[j,]
    }

    # Compute the scalar products

    prodmat <- matrix(0, nbasis, nbasis)
    convmat <- array(0,c(norder, norder, prodorder))
    for (k in 1:ninterval) {
        #  get the coefficients for the polynomials for this interval
        Coeff <- polycoef[k,,]
        #  compute the coefficients for the products
        for (i in 0:(ndegree-1)) {
            ind <- (0:i) + 1
            if (length(ind) == 1) {
                convmat[,,i+1        ] <-
                    outer(Coeff[ind,          ], Coeff[i-ind+2,      ])
                convmat[,,prodorder-i] <-
                    outer(Coeff[ndegree-ind+2,], Coeff[ndegree-i+ind,])
            } else {
                convmat[,,i+1        ] <-
                    crossprod(Coeff[ind,          ], Coeff[i-ind+2,      ])
                convmat[,,prodorder-i] <-
                    crossprod(Coeff[ndegree-ind+2,], Coeff[ndegree-i+ind,])
            }
        }
        ind <- (0:ndegree)+1
        convmat[,,ndegree+1] <-
                crossprod(Coeff[ind,          ], Coeff[ndegree-ind+2,])
        #  compute the coefficients of the integral
        delta    <- breaks[k+1] - breaks[k]
        power    <- delta
        prodmati <- matrix(0, norder, norder)
        for (i in 1:prodorder) {
            prodmati <- prodmati +
                power*convmat[,,prodorder-i+1]/i
            power <- power*delta
        }
        # add the integral to s
        index <- k:(k+norder-1)
        prodmat[index,index] <- prodmat[index,index] + prodmati
    }

     penaltymat <- prodmat

  } else {

    #  set iter

    iter <- 0

    #  LFDOBJ is not D^NDERIV, use approximate integration by calling
    #  function INPROD().

    if (uniquebreaks) {
        #  no knot multiplicities
        prodmat <- inprod(basisobj, basisobj, Lfdobj, Lfdobj, rng)
    } else {
        #  knot multiplicities  find their locations
        rngvec <- rng[1]
        for (i in 2:nbreaks) {
            if (breaks[i] == breaks[i-1]) rngvec <- c(rngvec, breaks[i])
        }
        rngvec <- unique(rngvec)
        nrng   <- length(rngvec)
        if (rngvec[nrng] < rng[2]) {
            rngvec <- c(rngvec,rng[2])
            nrng   <- nrng + 1
        }
        #  sum prodmat over intervals between knot multiplicities
        prodmat <- matrix(0,nbasis,nbasis)
        for (i in 2:nrng) {
            rngi <- c(rngvec[i-1] + 1e-10, rngvec[i] - 1e-10)
            prodmati <- inprod(basisobj, basisobj, Lfdobj, Lfdobj, rngi)
            prodmat <- prodmat + prodmati
        }
    }

    penaltymat <- prodmat
  }

    return( penaltymat )
}
bsplineS <- function (x, breaks, norder=4, nderiv=0, returnMatrix=FALSE)
{
#  This is a wrapper function for the S-PLUS spline.des function.
#  The number of spline functions is equal to the number of
#     discrete break points, length(BREAKVALUES), plus the order, NORDER,
#           minus 2.
#  Arguments are as follows:
#  X      ... array of values at which the spline functions are to
#             evaluated
#  BREAKS ... a STRICTLY INCREASING sequence of break points or knot
#             values.  It must contain all the values of X within its
#             range.
#  NORDER ... order of spline (1 more than degree), so that 1 gives a
#             step function, 2 gives triangle functions,
#             and 4 gives cubic splines
#  NDERIV ... highest order derivative.  0 means only function values
#             are returned.
#  Return is a matrix with length(X) rows and number of columns equal to
#                   number of b-splines

#  last modified 6 May 2012 by Spencer Graves
#  previously modified 2 April 2012 by Jim Ramsay

  x <- as.vector(x)
  n <- length(x)
  tol <- 1e-14
  nbreaks <- length(breaks)
  if (nbreaks < 2) stop('Number of knots less than 2.')
  if (min(diff(breaks)) < 0 ) stop('Knots are not increasing')

  if ( max(x) > max(breaks) + tol ||
       min(x) < min(breaks) - tol )
     stop('Knots do not span the values of X')
  if ( x[n] > breaks[nbreaks]) breaks[nbreaks] <- x[n]
  if ( x[1] < breaks[1]      ) breaks[1]       <- x[1]

  if (norder > 20) stop('NORDER exceeds 20.')
  if (norder <  1) stop('NORDER less than 1.')
  if (nderiv > 19) stop('NDERIV exceeds 19.')
  if (nderiv <  0) stop('NDERIV is negative.')
  if (nderiv >= norder) stop (
         'NDERIV cannot be as large as order of B-spline.')

  knots  <- c(rep(breaks[1      ],norder-1), breaks,
              rep(breaks[nbreaks],norder-1)  )
  derivs <- rep(nderiv,n)
  nbasis <- nbreaks + norder - 2
  if (nbasis >= norder) {
      if (nbasis > 1) {
  	    basismat <- splines::spline.des(knots, x, norder, derivs)$design
      } else {
  	    basismat <- as.matrix(splines::spline.des(knots, x, norder, derivs)$design)
      }
      if((!returnMatrix) && (length(dim(basismat)) == 2)){
          return(as.matrix(basismat))
      }
      return(basismat)
  } else {
      stop("NBASIS is less than NORDER.")
  }
}
cca.fd <- function(fdobj1, fdobj2=fdobj1, ncan = 2,
                   ccafdPar1=fdPar(basisobj1, 2, 1e-10),
                   ccafdPar2=ccafdPar1,
                   centerfns=TRUE)
{
#  Carry out a functional CCA with regularization using two different
#    functional data samples.  These may have different bases, and
#    different penalty functionals.  It is assumed that both are
#    univariate.
#  Arguments:
#  FDOBJ1        ... Functional data object.
#  FDOBJ2        ... Functional data object.
#  NCAN          ... Number of pairs of canonical variates to be found
#  CCAFDPAROBJ1  ... A functional parameter object for the first set of
#                    canonical variables.
#  CCAFDPAROBJ2  ... A functional parameter object for the second set of
#                    canonical variables.
#  CENTERFNS     ... A logical variable indicating whether or not to
#                    center the functions before analysis.  Default is TRUE.
#
#  Returns:  An object of the CCA.FD class containing:
#  CCWTFD1       ... A functional data object for the first set of
#                    canonical variables
#  CCWTFD2       ... A functional data object for the second set of
#                    canonical variables
#  CANCORR       ... A vector of canonical correlations
#  CCAVAR1       ... A matrix of scores on the first canonical variable.
#  CCAVAR2       ... A matrix of scores on the second canonical variable.
#

# Last modified 28 December 2012

#  check functional data objects

if (!(inherits(fdobj1, "fd"))) stop(
		"Argument FDOBJ1 not a functional data object.")
if (!(inherits(fdobj2, "fd"))) stop(
		"Argument FDOBJ2 not a functional data object.")
		
if (centerfns) {
  #  center the data by subtracting the mean function
  fdobj1 <- center.fd(fdobj1)
  fdobj2 <- center.fd(fdobj2)
}

#  extract dimensions for both functional data objects
coef1  <- fdobj1$coefs
coefd1 <- dim(coef1)
ndim1  <- length(coefd1)
nrep1  <- coefd1[2]

coef2  <- fdobj2$coefs
coefd2 <- dim(coef2)
ndim2  <- length(coefd2)
nrep2  <- coefd2[2]

#  check that numbers of replications are equal and greater than 1
if (nrep1 != nrep2) stop("Numbers of replications are not equal.")
if (nrep1 < 2) stop("CCA not possible without replications.")

nrep <- nrep1

#  check that functional data objects are univariate
if (ndim1 > 2 || ndim2 > 2) stop(
		"One or both functions are not univariate.")

#  extract basis information for both objects

basisobj1  <- fdobj1$basis
nbasis1    <- basisobj1$nbasis
dropind1   <- basisobj1$dropind
type1      <- basisobj1$type

basisobj2  <- fdobj2$basis
nbasis2    <- basisobj2$nbasis
dropind2   <- basisobj2$dropind
type2      <- basisobj2$type

#   Set up cross product matrices

Jmat1 <- eval.penalty(basisobj1, 0)
Jmat2 <- eval.penalty(basisobj2, 0)
Jx    <- t(Jmat1 %*% coef1)
Jy    <- t(Jmat2 %*% coef2)
PVxx  <- crossprod(Jx)/nrep
PVyy  <- crossprod(Jy)/nrep
Vxy   <- crossprod(Jx,Jy)/nrep

#  check both fdPar objects

if (inherits(ccafdPar1, "fd") || inherits(ccafdPar1, "basisfd"))
      ccafdPar1 <- fdPar(ccafdPar1)
if (inherits(ccafdPar2, "fd") || inherits(ccafdPar2, "basisfd"))
      ccafdPar2 <- fdPar(ccafdPar2)

if (!inherits(ccafdPar1, "fdPar")) stop(
		"ccafdPar1 is not a fdPar object.")
if (!inherits(ccafdPar2, "fdPar")) stop(
		"ccafdPar2 is not a fdPar object.")

#  get linear differential operators and lambdas
		
Lfdobj1 <- ccafdPar1$Lfd
Lfdobj2 <- ccafdPar2$Lfd
lambda1 <- ccafdPar1$lambda
lambda2 <- ccafdPar2$lambda

#  add roughness penalties if lambdas are positive

if (lambda1 > 0) {
    Kmat1 <- eval.penalty(basisobj1, Lfdobj1)
    PVxx  <- PVxx + lambda1 * Kmat1
}
if (lambda2 > 0) {
    Kmat2 <- eval.penalty(basisobj2, Lfdobj2)
    PVyy  <- PVyy + lambda2 * Kmat2
}

#  do eigenanalysis matrix Vxy with respect to metrics PVxx and PVyy

result <- geigen(Vxy, PVxx, PVyy)

#  set up canonical correlations and coefficients for weight functions

canwtcoef1 <- result$Lmat[,1:ncan]
canwtcoef2 <- result$Mmat[,1:ncan]
corrs      <- result$values

#   Normalize the coefficients for weight functions

for (j in 1:ncan) {
    temp <- canwtcoef1[,j]
    temp <- temp/sqrt(sum(temp^2))
    canwtcoef1[,j] <- temp
    temp <- canwtcoef2[,j]
    temp <- temp/sqrt(sum(temp^2))
    canwtcoef2[,j] <- temp
}

#  set up the canonical weight functions

canwtfdnames      <- fdobj1$fdnames
canwtfdnames[[2]] <- "Canonical Variable"
names(canwtfdnames)[2] <- "Canonical functions"
names(canwtfdnames)[3] <-
            paste("CCA wt. fns. for",names(canwtfdnames)[3])
canwtfd1 <- fd(canwtcoef1, basisobj1, canwtfdnames)
canwtfd2 <- fd(canwtcoef2, basisobj2, canwtfdnames)
canwtfd1$fdnames <- fdobj1$fdnames
canwtfd2$fdnames <- fdobj2$fdnames

#  set up canonical variable values

canvarvalues1 <- Jx %*% canwtcoef1
canvarvalues2 <- Jy %*% canwtcoef2

#  set up return list

ccafd        <- list(canwtfd1, canwtfd2, corrs,
                       canvarvalues1, canvarvalues2)
names(ccafd) <- c("ccawtfd1", "ccawtfd2", "ccacorr",
                    "ccavar1",  "ccavar2")

class(ccafd) <- "cca.fd"

return(ccafd)

}
center.fd <- function(fdobj)
{
#  remove mean function for functional observations

#  Last modified 16 January 2020

if (!(is.fd(fdobj) || is.fdPar(fdobj))) 
  stop("First argument is neither an fd or an fdPar object.")
if (is.fdPar(fdobj)) fdobvj = fdobj$fd

coef     <- as.array(fdobj$coefs)
coefd    <- dim(coef)
ndim     <- length(coefd)
basisobj <- fdobj$basis
nbasis   <- basisobj$nbasis
if (ndim == 2) {
   coefmean <- apply(coef,1,mean)
   coef     <- sweep(coef,1,coefmean)
} else {
   nvar <- coefd[3]
   for (j in 1:nvar) {
       coefmean  <- apply(coef[,,j],1,mean)
       coef[,,j] <- sweep(coef[,,j],1,coefmean)
   }
}
fdnames      <- fdobj$fdnames
fdnames[[3]] <- paste("Centered",fdnames[[3]])
centerfdobj  <- fd(coef, basisobj, fdnames)
return(centerfdobj)	
}
checkDims3 <- function(x, y=NULL, xdim=2:3, ydim=2:3, defaultNames='x',
         subset=c('xiny', 'yinx', 'neither'),
         xName=substring(deparse(substitute(x)), 1, 33),
         yName=substring(deparse(substitute(y)), 1, 33) ){
#  checkDims3 coerces arguments 'x' and 'y' to 3-d arrays, compares their 
#  dimensions, subsets one or the other or throws an error or a warning 
#  as seems appropriate.  It returns a list with components 'x' and 'y' 
#  with appropriate dimnames.

###  
###
### 1.  length(xdim) == length(ydim)?
###
###  
  mDims <- length(xdim)
  if(length(ydim) != mDims)
    stop('length(xdim) = ', mDims, ' != ', length(ydim),
         ' = length(ydim)')
###  
###
### 2.  check defaultnames
###
###  
  nDefault <- length(defaultNames)
  {
    if(nDefault<1){
      dNames0 <- NULL
      dNames  <- rep(list(NULL), mDims)
    }
    else {
      dNames0 <- defaultNames[[nDefault]]      
      dNames  <- rep(as.list(defaultNames), length=mDims)
      dNms    <- names(defaultNames)
      if (!is.null(dNms)) {
        if(is.null(dNames0)) dNames0 <- dNms[nDefault]
        for (i in 1:mDims)
          if(is.null(dNames[[i]])) dNames[[i]] <- dNms[i]
      }
    }
  }
###  
###
### 3.  loop
###
###  
  for(id in 1:mDims){
    dNi  <- c(dNames[id], dNames0)
    xNmi <- paste('subscript', id, 'of', xName)
    yNmi <- paste('subscript', id, 'of', yName) 
    out  <- checkDim3(x, y, xdim[id], ydim[id], dNi, subset,
                     xName=xNmi, yName=yNmi)
    x <- out$x
    y <- out$y
  }
###  
###
### 4.  done
###
###  
  out
}

#  -----------------------------------------------------------------

checkDim3 <- function(x, y=NULL, xdim=1, ydim=1, defaultNames='x',
         subset=c('xiny', 'yinx', 'neither'),
         xName=substring(deparse(substitute(x)), 1, 33),
         yName=substring(deparse(substitute(y)), 1, 33) ){
###
###  
### 1.  Check xdim, ydim 
###
###  
  if (xdim>3) stop('Requested matching xdim = ', xdim,
                   ' > 3 = maximum allowed.')
  if (is.null(y)){ 
    y <- x
    yName <- xName
  }
  if (ydim>3) stop('Requested matching ydim = ', ydim,
                   ' > 3 = maximum allowed.')
###
###  
### 2.  ixperm, iyperm 
###
###  
  ixperm <- list(1:3, c(2, 1, 3), c(3, 2, 1))[[xdim]]
  iyperm <- list(1:3, c(2, 1, 3), c(3, 2, 1))[[ydim]]  
###
###  
### 3.  x3 <- aperm, ... 
###
###  
  x3 <- aperm(as.array3(x), ixperm);
  y3 <- aperm(as.array3(y), iyperm)   
###
###  
### 4.  xNames, yNames 
###
###  
  xNames <- dimnames(x3)
  yNames <- dimnames(y3)
  {
    if(is.null(defaultNames))
      dNames <- NULL
    else { 
      dNames <- defaultNames[[1]]
      if(is.null(dNames)){
        if(is.null(names(defaultNames)))
          dNames <- defaultNames[[2]]
        else
          dNames <- names(defaultNames)[1]
      }
    }
  }    
###
###  
### 5.  Do it:  Subset & dimnames 
###
###  
  sbst <- match.arg(subset) 
##
##  5.1.  'xiny'
##  
  if(sbst == 'xiny'){  
    if(!is.null(dNames)){
      if(length(dNames) < dim(x3)[1])
        dNames <- paste(dNames, 1:dim(x3)[1], sep='')
      else
        dNames <- dNames[1:dim(x3)[1]]
    }
    nx3 <- dim(x3)[1]
    if(is.null(xNames)){
      if(nx3>dim(y3)[1])
        stop('Can NOT subset ', yName, ' because dim(x)[xdim=',
             xdim, '] = ', nx3, ' > ', dim(y3)[1],
             ' = dim(y)[ydim=', ydim, ']') 
      y3 <- y3[1:nx3,,, drop=FALSE]
      {
        if(is.null(yNames)){ 
          if(!is.null(dNames)){
            dNm <- rep(dNames, length=nx3) 
            dimnames(x3) <- list(dNm, NULL, NULL)
            dimnames(y3) <- list(dNm, NULL, NULL)
          }
        }
        else
          if(!is.null(yNames[[1]])){
            dimnames(x3) <- list(yNames[[1]][1:nx3], NULL, NULL)
            if(nx3<dim(y3)[1]) 
              warning(xName, ' is smaller than ', yName,
                      ' but has no dimnames while ', yName,
                      ' does.  Using the first ', nx3,
                      ' elements of dimension ', ydim,
                      ' of ', yName) 
          }
      }
    }
    else {
      if(is.null(xNames[[1]])){
        if(nx3>dim(y3)[1])
          stop('Can NOT subset ', yName, ' because dim(x)[xdim=',
               xdim, '] = ', nx3, ' > ', dim(y3)[1],
               ' = dim(y)[ydim=', ydim, ']')
        y3 <- y3[1:nx3,,, drop=FALSE]
        {
          if(is.null(yNames)){ 
            if(!is.null(dNames)){
              dNm <- rep(dNames, length=nx3) 
              dimnames(x3)[[1]] <- dNm
              dimnames(y3) <- list(dNm, NULL, NULL)
            }
          }
          else {
            if(is.null(yNames[[1]])){
              if(!is.null(dNames)){
                dNm <- rep(dNames, length=nx3) 
                dimnames(x3)[[1]] <- dNm
                dimnames(y3)[[1]] <- dNm
              }
            }
            else {
              dimnames(x3)[[1]] <- dimnames(y3)[[1]]
              if(nx3<dim(y3)[1]) 
                warning(xName, ' is smaller than ', yName,
                        ' but has no dimnames while ', yName,
                        ' does.  Using the first ', nx3,
                        ' elements of dimension ', ydim,
                        ' of ', yName)
            } 
          }
        }
      }
      else {
        if(is.null(yNames)){
          y3 <- y3[1:nx3,,, drop=FALSE]
          dimnames(y3) <- list(xNames[[1]], NULL, NULL)
          if(nx3<dim(y3)[1]) 
            warning(xName, ' is smaller than ', yName,
                    ' but has no dimnames, while ', yName,
                    ' does.  Using the first ', nx3,
                    ' elements of dimension ', ydim,
                    ' of ', yName)
        }
        else {
          if(is.null(yNames[[1]])){
            y3 <- y3[1:nx3,,, drop=FALSE]
            dimnames(y3)[[1]] <- xNames[[1]] 
            if(nx3<dim(y3)[1]) 
              warning(xName, ' is smaller than ', yName,
                      ' but has no dimnames, while ', yName,
                      ' does.  Using the first ', nx3,
                      ' elements of dimension ', ydim,
                      ' of ', yName)
          }
          else {        
            xiny <- is.element(xNames[[1]], yNames[[1]])
            if(any(!xiny))
              stop('Can NOT subset ', yName, ' because some dimnames(',
                   xName, ')[[xdim=', xdim,
                   ']] are not found in dimnames(y)[[ydim=',
                   ydim, ']];  the first one is ',
                   xNames[[1]][!xiny][1]) 
            y3 <- y3[xNames[[1]],,, drop=FALSE]
          }
        }
      }
    }
  }
  else {
##
##  5.2.  'yinx'
##  
    if(sbst == 'yinx'){
      if(!is.null(dNames)){
        if(length(dNames) < dim(y3)[1])
          dNames <- paste(dNames, 1:dim(y3)[1], sep='')
        else
          dNames <- dNames[1:dim(y3)[1]]
      }
      ny3 <- dim(y3)[1] 
      if(is.null(yNames)){
        if(ny3>dim(x3)[1])
          stop('Can NOT subset ', xName, ' because dim(y)[ydim=',
               ydim, '] = ', dim(y3)[1], ' > ', dim(x3)[1],
               ' = dim(x)[xdim=', xdim, ']') 
        x3 <- x3[1:ny3,,, drop=FALSE]
        {
          if(is.null(xNames)){ 
            if(!is.null(dNames)){
              dNm <- rep(dNames, length=nx3) 
              dimnames(x3) <- list(dNm, NULL, NULL)
              dimnames(y3) <- list(dNm, NULL, NULL)
            }
          }
          else
            if(!is.null(xNames[[1]])){ 
              dimnames(y3) <- list(xNames[[1]][1:ny3], NULL, NULL)
              if(ny3<dim(x3)[1]) 
                warning(yName, ' is smaller than ', xName,
                        ' but has no dimnames while ', xName,
                        ' does.  Using the first ', ny3,
                        ' elements of dimension ', xdim,
                        ' of ', xName) 
            }
        }
      }
      else {
        if(is.null(yNames[[1]])){
          if(ny3>dim(x3)[1]) 
            stop('Can NOT subset ', xName, ' because dim(y)[ydim=',
                 ydim, '] = ', ny3, ' > ', dim(x3)[1],
                 ' = dim(x)[xdim=', xdim, ']') 
          x3 <- x3[1:ny3,,, drop=FALSE]
          {
            if(is.null(xNames)){
              if(!is.null(dNames)){
                dNm <- rep(dNames, length=ny3) 
                dimnames(y3)[[1]] <- dNm
                dimnames(x3) <- list(dNm, NULL, NULL)
              }
            }
            else {
              if(is.null(xNames[[1]])){
                if(!is.null(dNames)){
                  dNm <- rep(dNames, length=ny3) 
                  dimnames(y3)[[1]] <- dNm
                  dimnames(x3)[[1]] <- dNm
                }
              }
              else {
                dimnames(y3)[[1]] <- dimnames(x3)[[1]]
                if(ny3<dim(x3)[1]) 
                  warning(yName, ' is smaller than ', xName,
                          ' but has no dimnames while ', xName,
                          ' does.  Using the first ', ny3,
                          ' elements of dimension ', xdim,
                          ' of ', xName)
              } 
            }
          }
        }
        else {
          if(is.null(xNames)){
            x3 <- x3[1:ny3,,, drop=FALSE]
            dimnames(x3) <- list(yNames[[1]], NULL, NULL)
            if(ny3<dim(x3)[1])
              warning(yName, ' is smaller than ', xName,
                      ' but has no dimnames, while ', xName,
                      ' does.  Using the first ', ny3,
                      ' elements of dimension ', xdim,
                      ' of ', xName)
          }
          else{
            if(is.null(xNames[[1]])){
              x3 <- x3[1:ny3,,, drop=FALSE]
              dimnames(x3)[[1]] <- yNames[[1]]
              if(ny3<dim(x3)[1])
                warning(yName, ' is smaller than ', xName,
                        ' but has no dimnames, while ', xName,
                        ' does.  Using the first ', ny3,
                        ' elements of dimension ', xdim,
                        ' of ', xName)
            }
            else {           
              yinx <- is.element(yNames[[1]], xNames[[1]]) 
              if(any(!yinx))
                stop('Can NOT subset ', xName, ' because some dimnames(', 
                     yName, ')[[ydim=', ydim,
                     ']] are not found in dimnames(x)[[xdim=',
                     xdim, ']];  the first one is ',
                     yNames[[1]][!yinx][1])
              x3 <- x3[yNames[[1]],,, drop=FALSE]
            }
          }
        }
      }
    }
    else
##
##  5.3.  'neither'
##      
      if(sbst == 'neither'){
        if(dim(x3)[1] != dim(y3)[1])
          stop('dim(x)[xdim=', xdim, '] = ', dim(x3)[1],
               ' != ', dim(y3)[1], ' = dim(y)[ydim=',
               ydim, ']')
        if(is.null(xNames)){
          if(!is.null(yNames) &&
             !is.null(yNames[[1]]))
            stop('is.null(dimnames(x)) but ',
                 '!is.null(dimnames(y)[[ydim=', ydim, ']]')
        }
        else{
          if(is.null(xNames[[1]])){ 
            if(!is.null(yNames) &&
               !is.null(yNames[[1]]))
              stop('is.null(dimnames(x)[[xdim=', xdim, ']]), but ', 
                   '!is.null(dimnames(y)[[ydim=', ydim, ']]')
          }
          else{
            if(is.null(yNames))
              stop('x has dimnames;  y has none.') 
            xiny <- (xNames[[1]] %in% yNames[[1]])
            if(any(!xiny))
              stop('Some dimnames(x)[[xdim=', xdim,
                   ']] are not in dimnames(y)[[ydim=',
                   ydim, ']]')
            yinx <- (yNames[[1]] %in% xNames[[1]])
            if(any(!yinx))
              stop('Some dimnames(y)[[ydim=', ydim,
                   ']] are not in dimnames(x)[[xdim=',
                   xdim, ']]')
          }
        }
      }
  }
###
###  
### 6.  out = list(x=aperm, ... ) 
###
###  
  list(x=aperm(x3, ixperm), y=aperm(y3, iyperm) )
}
checkLogical <- function(x, length., warnOnly=FALSE) {
##
## 0.  Set up
##
  onExit <- function(...){
    Msg <- paste("In ", callEnv, ':  ', ..., sep='') 
    if(warnOnly){
      warning(Msg, call.=FALSE)
      return(FALSE)
    }
    else
      stop(Msg, call.=FALSE)
  }
##
## 1.  External name of 'x'
##  
  xName <- substring(deparse(substitute(x)), 1, 44)
# name of calling function
  callEnv <- sys.call(-1)[[1]]
  if(is.null(callEnv)) callEnv <- sys.call()[[1]]
##
## 2.  is.null(x)   
##
  good <- TRUE
  if(is.null(x))
    good <- onExit('is.null(', xName, ')')
# good <- stopWarn('is.null(', xName, ')', warnOnly=warnOnly, which.parent=which.parent+1)
##
## 3.  check class(x)
##  
  if(!inherits(x, 'logical'))
    good <- (good & onExit('class(', xName, ') = ', class(x),
                   ";  should be 'logical'")) 
##
## 4.  Check 'length.' 
##
  if(!missing(length.)){
    if(length(x) != length.){
      good <- (good & onExit('length(', xName, ') = ', length(x), ' != ',
             ' required length = ', length.) ) 
    }
  }
##
## 5.  Done
##  
  good
}

checkNumeric <- function(x, lower, upper, length., integer=TRUE,
           unique=TRUE, inclusion=c(TRUE,TRUE), warnOnly=FALSE){
##
## 0.  set up exit / return processing with warnOnly
##
  onExit <- function(...){
    Msg <- paste("In ", callEnv, ':  ', ..., sep='') 
    if(warnOnly){
      warning(Msg, call.=FALSE)
      return(FALSE)
    }
    else
      stop(Msg, call.=FALSE)
  }
##
## 1.  External name of 'x'
##  
  xName <- substring(deparse(substitute(x)), 1, 44)
# name of calling function
  callEnv <- sys.call(-1)[[1]]
  if(is.null(callEnv)) callEnv <- sys.call()[[1]]
##
## 2.  is.null(x)   
##
  if(is.null(x)) {
    return(TRUE)  
  }
##
## 3.  check class(x)
##
  if(!is.numeric(x))
    onExit('class(', xName, ') = ', class(x), ";  should be 'numeric'")
  if(integer){
    x. <- round(x)
    d <- abs(x.-x)
    d.ne0 <- (d != 0)
    if(any(d.ne0)){
      id <- which(d == max(d))[1]
      nne <- sum(d.ne0)
      if(nne>1)
        onExit(sum(d.ne0), ' non-integer values;  ',
               'the most extreme is ', xName, '[', id, '] = ', x[id])
      else
        onExit('One non-integer value:  ',
               xName, '[', id, '] = ', x[id])
    }
  }
##
## 4.  Check limits 
##
  dLow <- (x-lower)
  xLow <- {
    if(inclusion[1]) (dLow<0) else (dLow<=0) 
  }
  if(any(xLow)){
    ilo <- which(dLow == min(dLow))[1]
    nlo <- sum(xLow)
    if(nlo>1)
      onExit(nlo, ' low values;  the most extreme is ',
             xName, '[', ilo, '] = ', x[ilo])
    else
      onExit('One low value:  ', xName, '[', ilo, '] = ', x[ilo])
  }
#
  dHi <- (x-upper)
  inclusion <- rep(inclusion, length=2)
  xHi <- {
    if(inclusion[2]) (dHi>0) else (dHi>=0)
  }
  if(any(xHi)){
    ihi <- which(dHi==max(dHi))[1]
    nhi <- sum(xHi)
    if(nhi>1) 
      onExit(nhi, ' high values;  the most extreme is ',
                   xName, '[', ihi, '] = ', x[ihi])
    else
      onExit('One high value:  ', xName, '[', ihi,
                   '] = ', x[ihi]) 
  }
##
## 5.  Check unique
##
  if(length(x)>1){
    x. <- sort(x)
    dx <- diff(x.)
    if(any(dx==0)){
      x <- unique(x) 
      iun <- which(dx==0)[1]
      nun <- sum(dx==0)
      if(nun>1) 
        onExit(nun, ' repeated values in ', xName,
                     ';  the smallest is ', x.[iun])
      else
        onExit('One repeated value in ', xName, ':  ', x.[iun])
    }
  }
##
## 6.  Check length 
##
  if(!missing(length.)){
    if(length(x) != length.)
      onExit('length(', xName, ') = ', length(x), ' != ',
             ' required length = ', length.)
  }
##
## 7.  Done
##    
  x 
}

checkLogicalInteger <- function(x, length., warnOnly=FALSE){
##
## 0.  set up exit / return processing with warnOnly
##
  onExit <- function(...){
    Msg <- paste("In ", callEnv, ':  ', ..., sep='') 
    if(warnOnly){
      warning(Msg, call.=FALSE)
      return(FALSE)
    }
    else
      stop(Msg, call.=FALSE)
  }
##
## 1.  External name of 'x'
##  
  xName <- substring(deparse(substitute(x)), 1, 44)
# name of calling function
  callEnv <- sys.call(-1)[[1]]
  if(is.null(callEnv)) callEnv <- sys.call()[[1]]
##
## 2.  is.null(x)   
##
  if(is.null(x)) {
    return(rep(TRUE, length=length.))
  }
##
## 3.  check class(x)
##
#  3.1.  is.logical?    
  if(is.logical(x)){
    if(missing(length.)) {
      return(x)
    }
      if(length(x) == length.) {
        return(x)
      } else {
        onExit('length(x) = ', length(x), ';  should be ',length.)
    }
  }
#  3.2.  is.numeric?  
  if(!is.numeric(x))
    onExit('class(', xName, ') = ', class(x),
           ";  should be 'numeric' or 'logical'")
#  3.3.  is.integer?  
  x. <- round(x)
  d <- abs(x.-x)
  d.ne0 <- (d != 0) 
  if(any(d.ne0)){
    id <- which(d == max(d))[1]
    nne <- sum(d.ne0)
    if(nne>1)
      onExit(sum(d.ne0), ' non-integer values;  ',
             'the most extreme is ', xName, '[', id, '] = ', x[id])
    else
      onExit('One non-integer value:  ',
             xName, '[', id, '] = ', x[id])
  }
##
## 4.  Check limits 
##
#  4.1.  x < 1?    
  xLow <- (x<1) 
  if(any(xLow)){
    ilo <- which(x == min(x))[1]
    nlo <- sum(xLow)
    if(nlo>1)
      onExit(nlo, ' low values;  the most extreme is ',
             xName, '[', ilo, '] = ', x[ilo])
    else
      onExit('One low value:  ', xName, '[', ilo, '] = ', x[ilo])
  }
#   4.2.  x > length.?
  if(missing(length.) && warnOnly){
    onExit("argument 'length.' is missing;  setting to max(x)")
    length. <- max(x) 
  }
  xHi <- (x>length.) 
  if(any(xHi)){
    ihi <- which(x==max(x))[1]
    nhi <- sum(xHi)
    if(nhi>1) 
      onExit(nhi, ' high values;  the most extreme is ',
                   xName, '[', ihi, '] = ', x[ihi])
    else
      onExit('One high value:  ', xName, '[', ihi,
                   '] = ', x[ihi]) 
  }
##
## 5.  Check unique
##
  if(length(x)>1){
    x. <- sort(x)
    dx <- diff(x.)
    if(any(dx==0)){
      x <- x[dx != 0] 
      iun <- which(dx==0)[1]
      nun <- sum(dx==0)
      if(nun>1) 
        onExit(nun, ' repeated values in ', xName,
                     ';  the smallest is ', x.[iun])
      else
        onExit('One repeated value in ', xName, ':  ', x.[iun])
    }
  }
##
## 6.  Convert to logical  
##
  X <- rep(FALSE, length.)
  X[x] <- TRUE 
##
## 7.  Done
##    
  X
}
  

coefficients <- function(object, ...)UseMethod('coef')

coef.fd <- function(object, ...) object$coef
coefficients.fd <- function(object, ...) object$coef

coef.fdPar <- function(object, ...) object$fd$coef
coefficients.fdPar <- function(object, ...) object$fd$coef

coef.fdSmooth <- function(object, ...) object$fd$coef
coefficients.fdSmooth <- function(object, ...) object$fd$coef

coef.Taylor <- function(object, ...) object$coef
coefficients.Taylor <- function(object, ...) object$coef
cor.fd <- function(evalarg1, fdobj1, evalarg2=evalarg1, fdobj2=fdobj1)
{
  #  compute correlation matrix / matrices for functional observations

  #  Last modified 16 January 2010 
  
  if (!(is.fd(fdobj1) || is.fdPar(fdobj1))) 
    stop("Second argument is neither an fd object or an fdPar object")
  
  if (!(is.fd(fdobj2) || is.fdPar(fdobj2))) 
    stop("Fourth argument is neither an fd object or an fdPar object")
  
##
## 1.  Compute var1 = bivariate data object for variance(fdobj1)
##
  var1 <- var.fd(fdobj1)
##
## 2.  Evaluate var1 at evalarg1 
##
  evalVar1 <- eval.bifd(evalarg1, evalarg1, var1)
##
## 3.  If missing(fdobj2) convert evalVar1 to correlations 
##
  {
    if(missing(fdobj2)){
      dimV1 <- dim(evalVar1)
      ndV1 <- length(dimV1)
      {
        if(ndV1<3){
          s1 <- sqrt(diag(evalVar1))
          return(evalVar1/outer(s1, s1))
        }
        else{
          if(dimV1[3] != 1)
            stop("Bug in cor.fd:  Programmed only for ",
                 "matrices or 4-d arrays with dim[3]==1.  oops.")
          dim1 <- dim(fdobj1$coefs)
          nVars <- dim1[3]
#         The following identifies the levels of evalVar1
#         containing the covariance matrix of each variable with itself           
          evalV.diag <- choose(2:(nVars+1), 2)
#         Compute the standard deviation vector for each variable 
          nPts <- length(evalarg1)
          s1 <- array(NA, dim=c(nPts, nVars))
          for(i in 1:nVars)
            s1[, i] <- sqrt(diag(evalVar1[,,1,evalV.diag[i]]))
#         Now compute the correlations 
          cor1 <- evalVar1
          m <- 0
          for(i in 1:nVars)for(j in 1:i){
            m <- m+1
            cor1[,,1,m] <- (evalVar1[,,1,m] / outer(s1[, i], s1[, j]))
          }
#         end for i in 1:nVars and j in 1:i
          return(cor1)
        }
#       end if...else nV1>2
      }
    }
    else {
##
## 4.  fdobj2 was also provided 
##
#  4.1  var.df(fdobj2)       
      var2 <- var.fd(fdobj2)
#  4.2.  Evalate at evalarg2
      evalVar2 <- eval.bifd(evalarg2, evalarg2, var2)
#  4.3.  var12 cross covariance
#*** If fdobj1 or fdobj2 are multivariate, var.fd will complain.        
      var12 <- var.fd(fdobj1, fdobj2)
#  4.4.  Evaluate the cross covariances           
      evalVar12 <- eval.bifd(evalarg1, evalarg2, var12)
#  4.5.  Convert evalVar12 to correlations 
      s1 <- sqrt(diag(evalVar1))
      s2 <- sqrt(diag(evalVar2))
      return(evalVar12 / outer(s1, s2))
    }
  }
##
## Done:  Nothing should get this far;
## all should have returned earlier,
## either via univariate or multivariate
## with fdobj1 only 
## or with both fdobj1 and fdobj2
##   
}
covPACE <- function(data,rng , time, meanfd, basis, lambda, Lfdobj){
  # does a bivariate smoothing for estimating the covariance surface for data that has not yet been smoothed
  #
  #   Arguments:
  #   DATA .....  a matrix object or list -- If the set is supplied as a matrix object, 
  #               the rows must correspond to argument values and columns to replications, 
  #               and it will be assumed that there is only one variable per observation.  
  #               If y is a three-dimensional array, the first dimension corresponds to  
  #               argument values, the second to replications, and the third to variables 
  #               within replications. -- If it is a list, each element must be a matrix
  #               object, the rows correspond to argument values per individual. First 
  #               column corresponds to time points and the following columns to argument  
  #               values per variable.
  #   RNG .....   a vector of length 2 defining a restricted range where the data was observed
  #   TIME .....  Array with time points where data was taken. length(time) == dim(data)[1]
  #   MEANFD .... Fd object corresponding to the mean function of the data
  #   BASIS ..... basisfd object for smoothing the covariate function
  #   LAMBDA .... a nonnegative real number specifying the amount of smoothing to be applied to the estimated
  #               functional parameter
  #   Lfdobj .... linear differential operator object for smoothing penalty of the estimated functional parameter
  #
  # Returns a list with the two named entries cov.estimate and meanfd
  
  if(is.list(data)){
    datalist = data 
    s = length(data)
  }else{
    datalist = sparse.list(data,time)
    s = dim(data)[2]
  }
  
  indexes = lapply(datalist, function(x) which(time %in% x[,1])) #sampling points for each subject
  nvar = dim(datalist[[1]])[2] - 1
  
  #centering the data
  mean.point = matrix(eval.fd(time, meanfd),nrow = nrow(eval.fd(time, meanfd)),ncol=nvar)
  data.center = lapply(datalist, function(x){
    y = x[,-1] - mean.point[which(time %in% x[,1]),] 
    return(y)
  })
  data.center = lapply(data.center, as.matrix)

  #create kronecher product matrix for basis evaluation
  phi.eval = eval.basis(time,basis)
  phi.kro = kronecker(phi.eval,phi.eval)
  m = length(time)
  phi.kro.sub = lapply(m*(1:m)-m+1, function(x) return(phi.kro[x:(x+m-1),])) 
  phi.kro.sub = lapply(phi.kro.sub,as.matrix)
  
  #point covariance estimate 
  
  
  if(nvar > 1){
    out = NULL
    for(i in 1:(nvar-1)){
      out = c(out,i:(nvar-1)*nvar+i)
    }
    
    pairs = (expand.grid(1:nvar,1:nvar))[-out,]
    z = list()  
    for(q in 1:nrow(pairs)){
      y = pairs[q,]
      z[[q]] = lapply(data.center, function(x) as.vector(x[,pairs[q,1]] %*% t(x[,pairs[q,2]]))[-(1:nrow(x) + nrow(x) *(0:(nrow(x)-1)))])
    }
  }else{
  z = lapply(data.center, function(x) as.vector(x %*% t(x))[-(1:nrow(x) + nrow(x) *(0:(nrow(x)-1)))])
  }
  
  
  #Calculate t(phi)%*%phi and t(phi)%*%y
  
  phi.norm = 0 # t(phi)%*%phi -- nbasis^2 x nbasis^2 matrix 
  if(nvar == 1) phi.y = 0 #t(phi)%*%y -- nbasis^2 x 1
  if(nvar > 1) phiy.list = as.list(replicate(length(z),0))
  
  for (j in 1:s){
    l = length(indexes[[j]])
    phij = do.call(rbind,lapply(phi.kro.sub[indexes[[j]]], function(x) x[indexes[[j]],]))
    phij = phij[-(1:l + l *(0:(l-1))),] #take out the diagonal 
    phi.norm = phi.norm + t(phij)%*%phij
    
    if(nvar > 1){
      phiy.list = lapply(1:length(z), function(x) phiy.list[[x]] + t(phij)%*%z[[x]][[j]])
    }else{
      phi.y = phi.y + t(phij)%*%z[[j]]
    }
  }
  
  if(nvar > 1) phi.y = do.call(cbind,phiy.list)
  
  
  #Penalty matrix
  Pl = eval.penalty(basis,Lfdobj,rng)
  Po = inprod(basis,basis,rng = rng)
  P = kronecker(t(Pl),Po) + kronecker(Po,Pl)
  
  
  c.estimate = solve(phi.norm+lambda*P) %*% (phi.y)
  
  if(nvar > 1){
    c.list = lapply(1:length(z), function(x) matrix(c.estimate[,x],nrow = sqrt(nrow(c.estimate)), ncol = sqrt(nrow(c.estimate))) )
    cov.estimate = lapply(c.list, function(x) bifd(x,basis,basis) )
  }else{
    c = matrix(c.estimate,nrow = sqrt(length(c.estimate)), ncol = sqrt(length(c.estimate)))
    cov.estimate = bifd(c,basis,basis)
  }
  
  covPACE = list(cov.estimate,meanfd)
  names(covPACE) = c("cov.estimate","meanfd")
  return(covPACE)
}



CRAN <- function(CRAN_pattern, n_R_CHECK4CRAN){
##
## 1.  get environment variables
##
    gete <- Sys.getenv()
    ngete <- names(gete)
    i <- seq(along=gete)
##
## 2.  check CRAN_pattern
##
    if(missing(CRAN_pattern)){
        if('_CRAN_pattern_' %in% ngete){
            CRAN_pattern <- gete['_CRAN_pattern_']
        } else CRAN_pattern <- '^_R_'
    }
##
## 3.  check n_R_CHECK4CRAN
##
    if(missing(n_R_CHECK4CRAN)){
        if('_n_R_CHECK4CRAN_' %in% ngete){
            n_R_CHECK4CRAN <- as.numeric(gete['_n_R_CHECK4CRAN_'])
        } else n_R_CHECK4CRAN <- 5
    }
##
## 4.  Check
##
    for(pati in CRAN_pattern)
        i <- i[grep(pati, ngete[i])]
##
## 5.  Done
##
    cran. <- (length(i) >= n_R_CHECK4CRAN)
    attr(cran., 'Sys.getenv') <- gete
    attr(cran., 'matches') <- i
    cran.
}


create.bspline.basis <- function (rangeval=NULL, nbasis=NULL,
                                  norder=4,      breaks=NULL,
                                  dropind=NULL,  quadvals=NULL,
                                  values=NULL,   basisvalues=NULL,
                                  names="bspl")
{
#  This function creates a bspline functional data basis.
#  Arguments
#  RANGEVAL...an array of length 2 containing the lower and upper
#             boundaries for the rangeval of argument values,
#             or a positive number, in which case command
#             rangeval <- c(0, rangeval) is executed.
#             the default is c(0,1)
#  NBASIS  ...the number of basis functions.  This argument must be
#             supplied, and must be a positive integer.
#  NORDER  ...order of b-splines (one higher than their degree).  The
#             default of 4 gives cubic splines.
#  BREAKS  ...also called knots, these are a non-decreasing sequence
#             of junction points between piecewise polynomial segments.
#             They must satisfy BREAKS[1] = RANGEVAL[1] and
#             BREAKS[NBREAKS] = RANGEVAL[2], where NBREAKS is the total
#             number of BREAKS.  There must be at least 2 BREAKS.
#  There is a potential for inconsistency among arguments NBASIS, NORDER,
#  and BREAKS since
#             NBASIS = NORDER + LENGTH(BREAKS) - 2
#  An error message is issued if this is the case.  Although previous
#  versions of this function attempted to resolve this inconsistency in
#  various ways, this is now considered to be too risky.
#  DROPIND ...A vector of integers specifiying the basis functions to
#             be dropped, if any.  For example, if it is required that
#             a function be zero at the left boundary, this is achieved
#             by dropping the first basis function, the only one that
#             is nonzero at that point.
#  QUADVALS...A NQUAD by 2 matrix.  The firs t column contains quadrature
#             points to be used in a fixed point quadrature.  The second
#             contains quadrature weights.  For example, for (Simpson"s
#             rule for (NQUAD = 7, the points are equally spaced and the
#             weights are delta.*[1, 4, 2, 4, 2, 4, 1]/3.  DELTA is the
#             spacing between quadrature points.  The default is
#             matrix("numeric",0,0).
#  VALUES ... A list, with entries containing the values of
#             the basis function derivatives starting with 0 and
#             going up to the highest derivative needed.  The values
#             correspond to quadrature points in QUADVALS and it is
#             up to the user to decide whether or not to multiply
#             the derivative values by the square roots of the
#             quadrature weights so as to make numerical integration
#             a simple matrix multiplication.
#             Values are checked against QUADVALS to ensure the correct
#             number of rows, and against NBASIS to ensure the correct
#             number of columns.
#             The default value of is VALUES is vector("list",0).
#             VALUES contains values of basis functions and derivatives at
#             quadrature points weighted by square root of quadrature weights.
#             These values are only generated as required, and only if slot
#             QUADVALS is not matrix("numeric",0,0).
#  BASISVALUES...A vector of lists, allocated by code such as
#             vector("list",1).
#             This field is designed to avoid evaluation of a
#             basis system repeatedly at a set of argument values.
#             Each list within the vector corresponds to a specific set
#             of argument values, and must have at least two components,
#             which may be tagged as you wish.
#             The first component in an element of the list vector contains the
#             argument values.
#             The second component in an element of the list vector
#             contains a matrix of values of the basis functions evaluated
#             at the arguments in the first component.
#             The third and subsequent components, if present, contain
#             matrices of values their derivatives up to a maximum
#             derivative order.
#             Whenever function getbasismatrix is called, it checks
#             the first list in each row to see, first, if the number of
#             argument values corresponds to the size of the first dimension,
#             and if this test succeeds, checks that all of the argument
#             values match.  This takes time, of course, but is much
#             faster than re-evaluation of the basis system.  Even this
#             time can be avoided by direct retrieval of the desired
#             array.
#             For example, you might set up a vector of argument values
#             called "evalargs" along with a matrix of basis function
#             values for these argument values called "basismat".
#             You might want too use tags like "args" and "values",
#             respectively for these.  You would then assign them
#             to BASISVALUES with code such as
#               basisobj$basisvalues <- vector("list",1)
#               basisobj$basisvalues[[1]] <-
#                               list(args=evalargs, values=basismat)
#  BASISFNNAMES ... Either a character vector of length NABASIS
#             or a single character string to which NORDER, "." and
#             1:NBASIS are appended by the command
#                paste(names, norder, ".", 1:nbreaks, sep="").
#             For example, if norder = 4, this defaults to
#                     'bspl4.1', 'bspl4.2', ... .
#  Returns
#  BASISFD ...a functional data basis object

#  Last modified  19 November 2021 by Jim Ramsay

#  -------------------------------------------------------------------------
#  Default basis for missing arguments:  A B-spline basis over [0,1] of
#    of specified norder with norder basis functions.
#    norder = 1 = one basis function = constant 1
#    norder = 2 = two basis functions = 2 right triangles,
#      one left, the other right.  They are a basis for straight lines
#      over the unit interval, and are equivalent to a monomial basis
#      with two basis functions.  This B-spline system can be
#      explicitly created with the command
#                create.bspline.basis(c(0,1), 2, 2)
#    norder = 3 = three basis functions:  x^2, x-(x-.5)^2, (x-1)^2
#    norder = 4 = default = 4 basis functions
#      = the simplest cubic spline basis
#  -------------------------------------------------------------------------

  type        <- "bspline"

#  ------------------------------------------------------------------------
#                     Set up non-default basis
#  ------------------------------------------------------------------------

##
## 1.  check RANGEVAL
##
#  1.1.  First check breaks is either NULL
#        or is numeric with positive length
#  Breaks <- breaks
  op <- options(warn=-1)
  Breaks <- as.numeric(breaks)
  options(op)
  if(!is.null(breaks)){
  	if(min(diff(breaks) < 0)) {
  		stop('One or more breaks differences are negative.')
  	}
    if(is.numeric(breaks)){
      if(length(breaks)<1)breaks <- NULL
      if(any(is.na(breaks)))
        stop('breaks contains NAs;  not allowed.')
      if(any(is.infinite(breaks)))
        stop('breaks contains Infs;  not allowed.')
    }
    else {
#     suppress warning if NAs generated
#      op <- options(warn=-1)
#      Breaks <- as.numeric(breaks)
#      options(op)
      nNA <- sum(is.na(Breaks))
      if(nNA>0)
        stop("as.numeric(breaks) contains ", nNA,
             ' NA', c('', 's')[1+(nNA>1)],
             ';  class(breaks) = ', class(breaks))
    }
  }
#
#  Rangeval <- rangeval
  op <- options(warn=-1)
  Rangeval <- as.numeric(rangeval)
  options(op)
  if(length(rangeval)<1) {
    if(is.null(breaks)) {
      rangeval <- 0:1
    } else{
      rangeval <- range(breaks)
      if(diff(rangeval)==0)
        stop('diff(range(breaks))==0;  not allowed.')
    }
  } else {
#    op <- options(warn=-1)
#    rangeval <- as.numeric(rangeval)
#    options(op)
    nNAr <- sum(is.na(Rangeval))
    if(nNAr>0)
      stop('as.numeric(rangeval) contains ', nNAr,
           ' NA', c('', 's')[1+(nNAr>1)],
           ';  class(rangeval) = ', class(rangeval) )
  }
  if(length(rangeval) == 1){
      if(rangeval <= 0)
        stop("'rangeval' a single value that is not positive, is ",
             rangeval)
      rangeval = c(0,rangeval)
  }
# rangeval too long ???
  if(length(rangeval)>2){
    if(!is.null(breaks))
      stop('breaks can not be provided with length(rangeval)>2;  ',
           ' length(rangeval) = ', length(rangeval),
           ' and length(breaks) = ', length(breaks))
    breaks <- rangeval
    rangeval <- range(breaks)
  }
#
  if(rangeval[1]>=rangeval[2])
    stop('rangeval[1] must be less than rangeval[2];  instead ',
         'rangeval[1] = ', rangeval[1], c('==', '>')[diff(rangeval)<0],
         ' rangeval[2] = ', rangeval[2])
##
## 2.  Check norder
##
  if(!is.numeric(norder))
    stop("norder must be numeric;  class(norder) = ",
         class(norder))
#
  if(length(norder)>1)
    stop('norder must be a single number;  length(norder) = ',
         length(norder))
#
  if(norder<=0)stop("norder must be positive, is ", norder)
#
  if((norder%%1) > 0)
    stop("norder must be an integer, = ", norder,
         ', with fractional part = ',norder%%1)
##
## 3.  Check nbasis
##
#  if (is.null(nbasis))     stop("Argument 'nbasis' is not supplied.")
  nbreaks <- length(breaks)
  {
    if(!is.null(nbasis)){
      if(!is.numeric(nbasis))
        stop('nbasis must be numeric, is ', class(nbasis))
      if((lnb <- length(nbasis))>1)
        stop("nbasis must be a single positive integer;  ",
             "length(nbasis) = ", lnb, " > 1;  first 2 elements = ",
             nbasis[1], ", ", nbasis[2])
      if ((nbasis%%1)>0)
        stop("nbasis is not an integer, = ", nbasis,
             ", with fractional part = ", nbasis%%1)
# if (nbasis < 1)          stop("Argument 'nbasis' is not positive.")
      if(nbasis < norder)
        stop('nbasis must be at least norder;  nbasis = ', nbasis,
             ';  norder = ', norder)
##
## 4.  Check breaks
##
#  This argument is optional, and defaults to NULL.
#  if not NULL, it must contain at least two values, the first and last
#  being equal to the corresponding values of RANGEVAL.   The values
#  may not decrease, but there can be sequences of equal values.
#  the number of break values must be consistent with the values
#  of NBASIS and NORDER via the equation
#        NBASIS = NORDER + NBREAKS - 2
      if(!is.null(breaks)){
        if (nbreaks < 2)
          stop("Number of values in argument 'breaks' less than 2.")
        if(breaks[1] != rangeval[1] || breaks[nbreaks] != rangeval[2])
          stop(paste("Range of argument 'breaks' not identical to",
                     "that of argument 'rangeval'."))
        if (min(diff(breaks)) < 0)
          stop("Values in argument 'breaks' are decreasing.")
#  Check for consistency with NBASIS and NORDER
        if (nbasis != norder + nbreaks - 2)
          stop(paste("Relation nbasis = norder + length(breaks) - 2",
                     "does not hold;  nbasis = ", nbasis,
                     "norder = ", norder, "length(breaks) = ",
                     length(breaks)) )
      }
      else{
#  default to nbasis - norder + 2 equally spaced break values
        breaks <- seq(rangeval[1], rangeval[2],
                      length = nbasis - norder + 2)
        nbreaks <- length(breaks)
      }
    }
    else {
#   is.null(nbasis)
      if(is.null(breaks))nbasis <- norder
      else
        nbasis <- length(breaks)+norder-2
    }
  }
##
## 5.  Set up the PARAMS vector, which contains only the interior knots.
##
  if (nbreaks > 2) {
    params <- breaks[2:(nbreaks-1)]
  } else {
    params <- NULL
  }
##
## 6.  set up basis object
##
  basisobj <- basisfd(type=type, rangeval=rangeval, nbasis=nbasis,
                  params=params, dropind=dropind,   quadvals=quadvals,
                  values=values, basisvalues=basisvalues)
##
## 7.  names
##
  {
    ndropind = length(dropind)
    if(length(names) == nbasis)
      basisobj$names <- names
    else {
      if(length(names) > 1)
        stop('length(names) = ', length(names), ';  must be either ',
             '1 or nbasis = ', nbasis)
      basisind = 1:nbasis
      names   = paste(names, norder, ".",as.character(basisind), sep="")
      basisobj$names <- names
    }
  }
##
## 8.  Done
##
##  if(!is.null(axes))basisobj$axes <- axes
  basisobj

}
create.constant.basis <- function(rangeval = c(0,1),
                                  names="const", axes=NULL)
{
#  This function creates a constant basis
#  Argument:
#  RANGEVAL ... an array of length 2 containing the lower and upper
#  Return:
#  BASISOBJ  ... a functional data basis object of type "constant"
#

#  last modified 2008.12.06 by Spencer Graves
#  previously modified 6 January 2008

#  check RANGEVAL

if (length(rangeval) == 1){
    if (rangeval <= 0) stop("RANGEVAL a single value that is not positive.")
    rangeval = c(0,rangeval)
}

if (!rangechk(rangeval)) stop("Argument RANGEVAL is not correct.")

type        <- "const"
nbasis      <- 1
params      <- vector("numeric",0)
dropind     <- vector("numeric",0)
quadvals    <- vector("numeric",0)
values      <- vector("list",0)
basisvalues <- vector("list",0)

basisobj <- basisfd(type=type, rangeval=rangeval, nbasis=nbasis, params=params,
                    dropind=dropind, quadvals=quadvals, values=values,
                    basisvalues=basisvalues)
basisobj$names <- names
  if(!is.null(axes))basisobj$axes <- axes

basisobj

}
create.exponential.basis <- function (rangeval=c(0,1), nbasis=NULL,
                         ratevec=NULL, dropind=NULL, quadvals=NULL,
                         values=NULL, basisvalues=NULL, names='exp',
                         axes=NULL)
{

#  This function creates an exponential functional data basis
#  Arguments
#  RANGEVAL ... An array of length 2 containing the lower and upper
#               boundaries for the rangeval of argument values
#  NBASIS   ... The number of basis functions.  If this conflicts with
#               the length of RATEVEC, the latter is used.
#  RATEVEC  ... The rate parameters defining exp(ratevec[i]*x)
#  DROPIND  ... A vector of integers specifying the basis functions to
#               be dropped, if any.
#  QUADVALS .. A NQUAD by 2 matrix.  The firs t column contains quadrature
#                points to be used in a fixed point quadrature.  The second
#                contains quadrature weights.  For example, for (Simpson"s
#                rule for (NQUAD = 7, the points are equally spaced and the
#                weights are delta.*[1, 4, 2, 4, 2, 4, 1]/3.  DELTA is the
#                spacing between quadrature points.  The default is
#                matrix("numeric",0,0).
#  VALUES  ... A list, with entries containing the values of
#                the basis function derivatives starting with 0 and
#                going up to the highest derivative needed.  The values
#                correspond to quadrature points in QUADVALS and it is
#                up to the user to decide whether or not to multiply
#                the derivative values by the square roots of the
#                quadrature weights so as to make numerical integration
#                a simple matrix multiplication.
#                Values are checked against QUADVALS to ensure the correct
#                number of rows, and against NBASIS to ensure the correct
#                number of columns.
#                The default value of is VALUES is vector("list",0).
#                VALUES contains values of basis functions and derivatives at
#                quadrature points weighted by square root of quadrature weights.
#                These values are only generated as required, and only if slot
#                QUADVALS is not matrix("numeric",0,0).
#  BASISVALUES ... A vector of lists, allocated by code such as
#                vector("list",1).
#                This field is designed to avoid evaluation of a
#                basis system repeatedly at a set of argument values.
#                Each list within the vector corresponds to a specific set
#                of argument values, and must have at least two components,
#                which may be tagged as you wish.
#                The first component in an element of the list vector contains the
#                argument values.
#                The second component in an element of the list vector
#                contains a matrix of values of the basis functions evaluated
#                at the arguments in the first component.
#                The third and subsequent components, if present, contain
#                matrices of values their derivatives up to a maximum
#                derivative order.
#                Whenever function getbasismatrix is called, it checks
#                the first list in each row to see, first, if the number of
#                argument values corresponds to the size of the first dimension,
#                and if this test succeeds, checks that all of the argument
#                values match.  This takes time, of course, but is much
#                faster than re-evaluation of the basis system.  Even this
#                time can be avoided by direct retrieval of the desired
#                array.
#                For example, you might set up a vector of argument values
#                called "evalargs" along with a matrix of basis function
#                values for these argument values called "basismat".
#                You might want too use tags like "args" and "values",
#                respectively for these.  You would then assign them
#                to BASISVALUES with code such as
#                  basisobj$basisvalues <- vector("list",1)
#                  basisobj$basisvalues[[1]] <-
#                               list(args=evalargs, values=basismat)
#  Returns
#  BASISOBJ  ... a functional data basis object of type "expon"

#  Last modified 9 November 2008 by Spencer Graves
#  Last modified 6 January 2008 by Jim Ramsay

#  Default basis for missing arguments
##
## 1.  Check RANGEVAL
##
  if(!is.numeric(rangeval))
    stop('rangaval must be numeric;  class(rangeval) = ',
         class(rangeval) )
  if(length(rangeval)<1)
    stop('rangeval must be a numeric vector of length 2;  ',
         'length(rangeval) = 0.')
  if (length(rangeval) == 1) {
    if( rangeval <= 0)
      stop("rangeval a single value that is not positive:  ",
           rangeval)
    rangeval <- c(0,rangeval)
  }
  if(length(rangeval)>2)
    stop('rangeval must be a vector of length 2;  ',
         'length(rangeval) = ', length(rangeval))
  if(diff(rangeval)<=0)
    stop('rangeval must cover a positive range;  diff(rangeval) = ',
         diff(rangeval) )
##
## 2.  check nbasis and ratevec
##
  {
    if(is.null(nbasis)){
      if(is.null(ratevec)){
        nbasis <- 2
        ratevec <- 0:1
      }
      else {
        nbasis <- length(ratevec)
        if(nbasis<1)
          stop('ratevec must have positive length;  length(ratevec) = 0')
        if(!is.numeric(ratevec))
          stop('ratevec must be numeric;  class(ratevec) = ',
               class(ratevec) )
        if(length(unique(ratevec)) != nbasis)
          stop('ratevec contains duplicates;  not allowed.')
      }
    }
    else {
      if(is.null(ratevec))
        ratevec <- 0:(nbasis-1)
      else{
        if(length(ratevec) != nbasis)
          stop('length(ratevec) must equal nbasis;  length(ratevec) = ',
               length(ratevec), ' != ', 'nbasis = ', nbasis)
        if(length(unique(ratevec)) != nbasis)
          stop('ratevec contains duplicates;  not allowed.')
      }
    }
  }
##
## 3.  check DROPIND
##
  if (length(dropind) > 0){
    if(!is.numeric(dropind))
      stop('dropind must be numeric;  is ', class(dropind))
    doops <- which((dropind%%1)>0)
    if(length(doops)>0)
      stop('dropind must be integer;  element ', doops[1],
           " = ", dropind[doops[1]], '; fractional part = ',
           dropind[doops[1]] %%1)
#
    doops0 <- which(dropind<=0)
    if(length(doops0)>0)
      stop('dropind must be positive integers;  element ',
           doops0[1], ' = ', dropind[doops0[1]], ' is not.')
    doops2 <- which(dropind>nbasis)
    if(length(doops2)>0)
        stop("dropind must not exceed nbasis = ", nbasis,
             ';  dropind[', doops2[1], '] = ', dropind[doops2[1]])
#
    dropind <- sort(dropind)
    if(length(dropind) > 1) {
      if(min(diff(dropind)) == 0)
        stop("Multiple index values in DROPIND.")
    }
  }
##
## 4.  set up the basis object
##
  type        <- "expon"
  params      <- as.vector(ratevec)

  basisobj <- basisfd(type=type,     rangeval=rangeval, nbasis=nbasis,
                    params=params, dropind=dropind,   quadvals=quadvals,
                    values=values, basisvalues=basisvalues)
##
## 5.  names
##
  {
    if(length(names) == nbasis)
      basisobj$names <- names
    else {
      if(length(names)>1)
        stop('length(names) = ', length(names), ';  must be either ',
             '1 or nbasis = ', nbasis)
      basisobj$names <- paste(names, 0:(nbasis-1), sep="")
    }
  }
##
## 6.  Done
##
  if(!is.null(axes))basisobj$axes <- axes

  basisobj

}

#  ------------------------------------------------------------------------

create.fdVar.basis <- function(rangeval, I, J=I) {
#  CREATE_FDVARIANCE_BASIS Creates a functional basis object with type 
#  'fdVariance' thatcan be used to define a covariance surface.  This 
#  object can consist of one or more trapezoidal domains defined over a 
#  common range.  The range is defined in argument RANGEVAL, which is also
#  the RANGEVAL field of the basis object.
#  Each trapezoidal region is defined by the number of vertical
#  intervals in the corresponding element in I, and the number of
#  horizontal elements in the corresponding element in J.  I and J
#  must have the same lengths, and their common length is the number
#  of trapezoidal domains defined.
#  The PARAMS field for the basis object is a struct object with fields
#  I and J.  The TYPE field is 'fdVariance'.  The NBASIS field is the
#  sum over i of (I(i)+1)*(J(i)+1)
#
#  Arguments:

#  Last modified 102 June 2015 by Jim Ramsay

#  check I and J

nI = length(I)
nJ = length(J)

# check I and J

if (nI != nJ)           stop('I and J do not have same lengths.')
if (any(I <= 0))        stop('I has zero or negative entries.')
if (any(J <  0))        stop('I has negative entries.')
if (any(floor(I) != I)) stop('I has non-integer values.')
if (any(floor(J) != J)) stop('J has non-integer values.')

#  check RANGEVAL

if (length(rangeval) == 1) {
    if (rangeval <= 0) stop('RANGEVAL is a single value that is not positive.')
    rangeval = c(0,rangeval)
}
if (rangechk(rangeval) != 1) stop('RANGEVAL is not a legitimate range.')

#  get sum of number of basis functions

nbasis = 0
for (i in 1:nI) nbasis = nbasis + (I[i]+1)*(J[i]+1)

#  construct basis object

type = 'fdVariance'

params = list(I=I, J=J)

basisobj = basisfd(type, rangeval, nbasis, params)

return(basisobj)

}
create.fourier.basis <- function (rangeval=c(0,1), nbasis=3,
         period=diff(rangeval), dropind=NULL, quadvals=NULL,
         values=NULL, basisvalues=NULL, names=NULL, axes=NULL)
{

#  This function creates a fourier functional data basis.
#  Arguments
#  RANGEVAL ... an array of length 2 containing the lower and upper
#               boundaries for the rangeval of argument values
#  NBASIS   ... the number of basis functions.  If the argument value is
#               even, it is increased by one so both sines and cosines are
#               present for each period.  A possible underdetermination of
#               the basis is taken care of in function PROJECT.BASIS.
#  PERIOD   ... The period.  That is, the basis functions are periodic on
#                 the interval [0,PARAMS] or any translation of it.
#  DROPIND  ... A vector of integers specifying the basis functions to
#               be dropped, if any.
#  QUADVALS .. A NQUAD by 2 matrix.  The firs t column contains quadrature
#                points to be used in a fixed point quadrature.  The second
#                contains quadrature weights.  For example, for (Simpson"s
#                rule for (NQUAD = 7, the points are equally spaced and the
#                weights are delta.*[1, 4, 2, 4, 2, 4, 1]/3.  DELTA is the
#                spacing between quadrature points.  The default is
#                matrix("numeric",0,0).
#  VALUES  ... A list, with entries containing the values of
#                the basis function derivatives starting with 0 and
#                going up to the highest derivative needed.  The values
#                correspond to quadrature points in QUADVALS and it is
#                up to the user to decide whether or not to multiply
#                the derivative values by the square roots of the
#                quadrature weights so as to make numerical integration
#                a simple matrix multiplication.
#                Values are checked against QUADVALS to ensure the correct
#                number of rows, and against NBASIS to ensure the correct
#                number of columns.
#                The default value of is VALUES is vector("list",0).
#                VALUES contains values of basis functions and derivatives at
#                quadrature points weighted by square root of quadrature weights.
#                These values are only generated as required, and only if slot
#                QUADVALS is not matrix("numeric",0,0).
#  BASISVALUES ... A vector of lists, allocated by code such as
#                vector("list",1).
#                This field is designed to avoid evaluation of a
#                basis system repeatedly at a set of argument values.
#                Each list within the vector corresponds to a specific set
#                of argument values, and must have at least two components,
#                which may be tagged as you wish.
#                The first component in an element of the list vector contains the
#                argument values.
#                The second component in an element of the list vector
#                contains a matrix of values of the basis functions evaluated
#                at the arguments in the first component.
#                The third and subsequent components, if present, contain
#                matrices of values their derivatives up to a maximum
#                derivative order.
#                Whenever function getbasismatrix is called, it checks
#                the first list in each row to see, first, if the number of
#                argument values corresponds to the size of the first dimension,
#                and if this test succeeds, checks that all of the argument
#                values match.  This takes time, of course, but is much
#                faster than re-evaluation of the basis system.  Even this
#                time can be avoided by direct retrieval of the desired
#                array.
#                For example, you might set up a vector of argument values
#                called "evalargs" along with a matrix of basis function
#                values for these argument values called "basismat".
#                You might want too use tags like "args" and "values",
#                respectively for these.  You would then assign them
#                to BASISVALUES with code such as
#                  basisobj$basisvalues <- vector("list",1)
#                  basisobj$basisvalues[[1]] <-
#                               list(args=evalargs, values=basismat)
#  Returns
#  BASISOBj  ... a functional data basis object of type "fourier"

#  Last modified 20 April 2017 by Jim Ramsay

#  Default basis for missing arguments

  type        <- "fourier"
##
## 1.  check RANGEVAL
##
  if(length(rangeval)<1)
    stop('length(rangeval) = 0;  not allowed.')
  if (length(rangeval)==1) {
    if (rangeval<=0) stop("RANGEVAL a single value that is not positive.")
    rangeval <- c(0,rangeval)
  }
  if (!rangechk(rangeval)) stop("Argument RANGEVAL is not correct.")
##
## 2.  Set up PERIOD
##
#  width <- rangeval[2] - rangeval[1]
  if(!is.numeric(period))
    stop('period must be numeric;  class(period) = ',
         class(period))
  if(length(period)>1)
    stop('period must be a scalar;  length(period) = ',
         length(period))
  if(period <= 0) stop("'period' must be positive, is ", period)
#  if ((period <= 0) || !is.numeric(period))
#    stop ("Period must be positive number for a Fourier basis")
##
## 3.  Check NBASIS
##
  if(!is.numeric(nbasis))
    stop('nbasis must be numeric;  class(nbasis) = ', class(nbasis))
  if(nbasis <= 0)
    stop('nbasis must be positive;  is ', nbasis)
  if((nbasis%%1) > 10*.Machine$double.eps)
    stop ("nBasis must be an integer.")
  nbasis <- ceiling(nbasis)
##
## 4.  check DROPIND
##
  if(is.null(dropind) || (length(dropind)==0)) dropind <- vector("numeric",0)

  if (length(dropind) > 0){
    if(length(dropind) >= nbasis)
      stop('dropind request deleting more basis functions than exist.')
    dropind = sort(dropind)
    if(any( (dropind%%1) > (10*.Machine$double.eps)))
      stop('some dropind are not integers.')
    dropind <- round(dropind)
    if(length(dropind) > 1) {
      if(min(diff(dropind)) == 0)
        stop("dropind requists deleting the same basis function more than once.")
    }
    for(i in 1:length(dropind)) {
      if(dropind[i] < 1 || dropind[i] > nbasis)
        stop("dropind contains an index value out of range:  ",
             dropind[i])
    }
  }
##
## 5.  set up the basis object
##
  params      <- period

  basisobj <- basisfd(type=type,     rangeval=rangeval, nbasis=nbasis,
                      params=params, dropind=dropind, quadvals=quadvals,
                      values=values, basisvalues=basisvalues)
##
## 6.  names?
##
  {
    if(is.null(names)){
      Nms <- 'const'
      if(nbasis>1){
        if(nbasis==3)
          Nms <- c(Nms, 'sin', 'cos')
        else {
          nb2 <- floor(nbasis/2)
          sinCos <- as.vector(outer(c('sin', 'cos'), 1:nb2,
                                    paste, sep=''))
          Nms <- c(Nms, sinCos)
        }
      }
    }
    else{
      if(length(names) != nbasis)
        stop('conflict between nbasis and names:  nbasis = ',
             nbasis, ';  length(names) = ', length(names))
    }
  }
  basisobj$names <- Nms
##
## 7.  done
##
  if(!is.null(axes))basisobj$axes <- axes
  basisobj
}
create.monomial.basis <- function(rangeval=c(0,1), nbasis=NULL,
                exponents=NULL, dropind=NULL, quadvals=NULL,
                values=NULL, basisvalues=NULL, names='monomial',
                axes=NULL)
{
#  CREATE_MONOMIAL_BASIS  Creates a monomial basis:, x^i_1, x^i_2, ...
#  The exponents in this version must be non-negative integers
#  Argument:
#  RANGEVAL ...an array of length 2 containing the lower and upper
#              boundaries for the rangeval of argument values.  If a
#              single value is input, it must be positive and the lower
#              limit of the range is set to 0.
#  NBASIS   ...number of basis functions
#  EXPONENTS...an array of NBASIS nonnegative integer exponents
#              by default this is 0:(NBASIS-1)
#  DROPIND ... A vector of integers specifying the basis functions to
#              be dropped, if any.
#  QUADVALS .. A NQUAD by 2 matrix.  The firs t column contains quadrature
#              points to be used in a fixed point quadrature.  The second
#              contains quadrature weights.  For example, for (Simpson"s
#              rule for (NQUAD = 7, the points are equally spaced and the
#              weights are delta.*[1, 4, 2, 4, 2, 4, 1]/3.  DELTA is the
#              spacing between quadrature points.  The default is
#              matrix("numeric",0,0).
#  VALUES  ... A list, with entries containing the values of
#              the basis function derivatives starting with 0 and
#              going up to the highest derivative needed.  The values
#              correspond to quadrature points in QUADVALS and it is
#              up to the user to decide whether or not to multiply
#              the derivative values by the square roots of the
#              quadrature weights so as to make numerical integration
#              a simple matrix multiplication.
#              Values are checked against QUADVALS to ensure the correct
#              number of rows, and against NBASIS to ensure the correct
#              number of columns.
#              The default value of is VALUES is vector("list",0).
#              VALUES contains values of basis functions and derivatives at
#              quadrature points weighted by square root of quadrature weights.
#              These values are only generated as required, and only if slot
#              QUADVALS is not matrix("numeric",0,0).
# BASISVALUES...A vector of lists, allocated by code such as
#              vector("list",1).
#              This field is designed to avoid evaluation of a
#              basis system repeatedly at a set of argument values.
#              Each list within the vector corresponds to a specific set
#              of argument values, and must have at least two components,
#              which may be tagged as you wish.
#              The first component in an element of the list vector contains the
#              argument values.
#              The second component in an element of the list vector
#              contains a matrix of values of the basis functions evaluated
#              at the arguments in the first component.
#              The third and subsequent components, if present, contain
#              matrices of values their derivatives up to a maximum
#              derivative order.
#              Whenever function getbasismatrix is called, it checks
#              the first list in each row to see, first, if the number of
#              argument values corresponds to the size of the first dimension,
#              and if this test succeeds, checks that all of the argument
#              values match.  This takes time, of course, but is much
#              faster than re-evaluation of the basis system.  Even this
#              time can be avoided by direct retrieval of the desired
#              array.
#              For example, you might set up a vector of argument values
#              called "evalargs" along with a matrix of basis function
#              values for these argument values called "basismat".
#              You might want too use tags like "args" and "values",
#              respectively for these.  You would then assign them
#              to BASISVALUES with code such as
#                basisobj$basisvalues <- vector("list",1)
#                basisobj$basisvalues[[1]] <-
#                             list(args=evalargs, values=basismat)
# Return:
# BASISOBJ  ...a functional data basis object of type "monom"
#
# last modified July 25, 2022
#   for rangeval of class Date and POSIXct
#  Default basis for missing arguments

  type        <- "monom"
##
## 1.  check RANGEVAL
##
  op <- options(warn=-1)
  Rangeval <- as.numeric(rangeval)
  options(op)
  if(length(rangeval)<1)
    stop('rangeval must be a numeric vector of length 2;  ',
         'length(rangeval) = 0.')
  if (length(rangeval) == 1) {
    if( rangeval <= 0)
      stop("rangeval a single value that is not positive:  ",
           rangeval)
    rangeval <- c(0,rangeval)
  }
  if(length(rangeval)>2)
    stop('rangeval must be a vector of length 2;  ',
         'length(rangeval) = ', length(rangeval))
  nNAr <- sum(is.na(Rangeval))
  if(nNAr>0)
    stop('as.numeric(rangeval) contains ', nNAr,
         ' NA', c('', 's')[1+(nNAr>1)],
         ';  class(rangeval) = ', class(rangeval) )
  if(diff(Rangeval)<=0)
    stop('rangeval must cover a positive range;  diff(rangeval) = ',
         diff(Rangeval) )
##
## 2.  check nbasis>0 & whether exponents are nonnegative integers
##
  {
    if(is.null(nbasis)) {
      if(is.null(exponents)){
        nbasis <- 2
        exponents <- 0:1
      }
      else {
        if(is.numeric(exponents)){
          nbasis <- length(exponents)
          if(length(unique(exponents)) != nbasis)
            stop('duplicates found in exponents;  not allowed.')
        }
        else
          stop('exponents must be numeric;  class(exponents) = ',
               class(exponents) )
      }
    }
    else {
      if(is.numeric(nbasis)){
        if(length(nbasis)!=1)
          stop('nbasis must be a scalar;  length(nbasis) = ',
               length(nbasis) )
        if((nbasis %%1) != 0)
          stop('nbasis must be an integer;  nbasis%%1 = ',
               nbasis%%1)
        {
          if(is.null(exponents))
            exponents <- 0:(nbasis-1)
          else {
            if(is.numeric(exponents)){
              if(length(exponents) != nbasis)
                stop('length(exponents) must = nbasis;  ',
                     'length(exponents) = ', length(exponents),
                     ' != nbasis = ', nbasis)
              if(length(unique(exponents)) != nbasis)
                stop('duplicates found in exponents;  not allowed.')
              if(any((exponents %%1) != 0))
                stop('exponents must be integers;  some are not.')
              if(any(exponents<0))
                stop('exponents must be nonnegative;  some are not.')
            }
            else
              stop('exponents must be numeric;  class(exponents) = ',
                   class(exponents) )
          }
        }
      }
      else stop('nbasis must be numeric;  class(nbasis) = ',
                class(nbasis) )
    }
  }
##
## 3.  check DROPIND
##
  if (length(dropind) == 0) dropind <- NULL
#
  if (length(dropind) > 0){
    if(!is.numeric(dropind))
      stop('dropind must be numeric;  is ', class(dropind))
    doops <- which((dropind%%1)>0)
    if(length(doops)>0)
      stop('dropind must be integer;  element ', doops[1],
           " = ", dropind[doops[1]], '; fractional part = ',
           dropind[doops[1]] %%1)
#
    doops0 <- which(dropind<=0)
    if(length(doops0)>0)
      stop('dropind must be positive integers;  element ',
           doops0[1], ' = ', dropind[doops0[1]], ' is not.')
    doops2 <- which(dropind>nbasis)
    if(length(doops2)>0)
        stop("dropind must not exceed nbasis = ", nbasis,
             ';  dropind[', doops2[1], '] = ', dropind[doops2[1]])
#
    dropind <- sort(dropind)
    if(length(dropind) > 1) {
      if(min(diff(dropind)) == 0)
        stop("Multiple index values in DROPIND.")
    }
  }
##
## 4.  set up the basis object
##
  type        <- "monom"
  params      <- exponents
  basisobj <- basisfd(type=type,   rangeval=rangeval, nbasis=nbasis,
                    params=params, dropind=dropind,   quadvals=quadvals,
                    values=values, basisvalues=basisvalues)
##
## 5.  names
##
  {
    if(length(names) == nbasis)
      basisobj$names <- names
    else {
      if(length(names)>1)
        stop('length(names) = ', length(names), ';  must be either ',
             '1 or nbasis = ', nbasis)
      basisobj$names <- paste(names, 0:(nbasis-1), sep="")
    }
  }
##
## 6.  Done
##
  if(!is.null(axes))basisobj$axes <- axes
  basisobj
}
create.polygonal.basis <- function(rangeval=NULL, argvals=NULL,
                      dropind=NULL, quadvals=NULL, values=NULL,
                      basisvalues=NULL, names='polygon', axes=NULL)
{
#  This function creates a polygonal functional data basis.
#  Arguments
#  ARGVALS  ... A strictly increasing vector of argument values at which
#               line segments join to form a polygonal line.
#  DROPIND  ... A vector of integers specificying the basis functions to
#               be dropped, if any.
#  QUADVALS ... A matrix with two columns and a number of rows equal to
#               the number of argument values used to approximate an
#               integral using Simpson's rule.
#               The first column contains these argument values.
#               A minimum of 5 values are required for
#               each inter-knot interval, and that is often enough. These
#               are equally spaced between two adjacent knots.
#               The second column contains the weights used for Simpson's
#               rule.  These are proportional to 1, 4, 2, 4, ..., 2, 4, 1
#   VALUES  ... A list containing the basis functions and their derivatives
#               evaluated at the quadrature points contained in the first
#               column of QUADVALS.
#  Returns
#  BASISOBJ ... a functional data basis object

#  Last modified 5 November 2008 by Spencer Graves
#  Last modified 20 November 2005

  type <- "polygonal"
##
## 1.  Check rangeval & argvals
##
  {
    if(is.null(rangeval)){
      if(is.null(argvals))argvals <- 0:1
      else {
        if(!is.numeric(argvals))
          stop('argvalues must be numeric;  class(argvals) = ',
               class(argvals) )
        if(length(argvals)<2)
          stop('length(argvals) must exceed 1, is ',
               length(argvals) )
        if(any(diff(argvals)<=0))
          stop('argvals must be strictly increasing, but is not.')
      }
      rangeval <- range(argvals)
    }
    else {
      if(!is.numeric(rangeval))
        stop('rangeval must be numeric;  class(rangeval) = ',
             class(rangeval) )
      if(length(rangeval)<2) {
        if(length(rangeval)<1)
          stop('length(rangeval) = 0, must be 2')
        if(rangeval<=0)
          stop('If length(rangeval)=1, it must be positive, is ',
               rangeval)
        rangeval <- c(0, rangeval)
      }
      {
        if(is.null(argvals)) {
          argvals <- rangeval
          rangeval <- range(argvals)
        }
        else {
          if(!is.numeric(argvals))
            stop('argvals must be numeric;  class(argvals) = ',
                 class(argvals) )
          nbasis <- length(argvals)
          if(nbasis<2)
            stop('length(argvals) must exceed 1, is ', nbasis )
          if(any(diff(argvals)<=0))
            stop('argvals must be strictly increasing, but is not.')
#
          if(length(rangeval)>2)
            stop('length(rangeval) must be 2, is ', length(rangeval))
          if(all.equal(argvals[1], rangeval[1])!=TRUE)
            stop('rangeval[1] must equal argvals[1];  rangeval[1] = ',
                 rangeval[1], " != argvals[1] = ", argvals[1])
#
          if(all.equal(rangeval[2], argvals[nbasis]) != TRUE)
            stop('rangeval[2] must equal argvals[nbasis];  ',
                 'rangeval[1] = ', rangeval[1],
                 ' != argvals[nbasis] = ', argvals[nbasis])
        }
      }
    }
  }
  nbasis <- length(argvals)
##
## 2.  check DROPIND
##
  if (length(dropind)<1) dropind <- NULL
#
  if (length(dropind) > 0) {
    if(!is.numeric(dropind))
      stop('dropind must be numeric;  is ', class(dropind))
    doops <- which((dropind%%1)>0)
    if(length(doops)>0)
      stop('dropind must be integer;  element ', doops[1],
           " = ", dropind[doops[1]], '; fractional part = ',
           dropind[doops[1]] %%1)
#
    doops0 <- which(dropind<=0)
    if(length(doops0)>0)
      stop('dropind must be positive integers;  element ',
           doops0[1], ' = ', dropind[doops0[1]], ' is not.')
    doops2 <- which(dropind>nbasis)
    if(length(doops2)>0)
        stop("dropind must not exceed nbasis = ", nbasis,
             ';  dropind[', doops2[1], '] = ', dropind[doops2[1]])
#
    dropind <- sort(dropind)
    if(length(dropind) > 1) {
      if(min(diff(dropind)) == 0)
        stop("Multiple index values in DROPIND.")
    }
  }
##
## 3.  set up the basis object
##
  basisobj <- basisfd(type=type, rangeval=rangeval, nbasis=nbasis,
                      params=argvals, dropind=dropind,
                      quadvals=quadvals, values=values)
##
## 4.  names
##
  {
    if(length(names) == nbasis)
      basisobj$names <- names
    else {
      if(length(names)>1)
        stop('length(names) = ', length(names), ';  must be either ',
             '1 or nbasis = ', nbasis)
      basisobj$names <- basisobj$names <- paste(names, 1:nbasis, sep="")
    }
  }
##
## 5.  Done
##
  if(!is.null(axes))basisobj$axes <- axes

  basisobj
}
create.polynomial.basis <- function (rangeval=c(0,1), nbasis=2, ctr=0,
                   dropind=NULL, quadvals=NULL, values=NULL,
                   basisvalues=NULL, names='polynom', axes=NULL)
{

#  This function creates a polynomial functional data basis, for
#    polynomials of the form  (x - c)^j
#  Arguments
#  RANGEVAL ... an array of length 2 containing the lower and upper
#               boundaries for the rangeval of argument values
#  NBASIS   ... the number of basis functions
#  CTR      ... The centering constant C.  By default, this is 0.
#  DROPIND  ... A vector of integers specificying the basis functions to
#               be dropped, if any.
#  Returns
#  BASISOBJ ... a functional data basis object of type "polynomial"

#  Last modified 9 November 2008 by Spencer Graves
#  Last modified 3 January 2008 by Jim Ramsay

##
## 1.  check RANGEVAL
##
  if(!is.numeric(rangeval))
    stop('rangaval must be numeric;  class(rangeval) = ',
         class(rangeval) )
  if(length(rangeval)<1)
    stop('rangeval must be a numeric vector of length 2;  ',
         'length(rangeval) = 0.')
  if (length(rangeval) == 1) {
    if( rangeval <= 0)
      stop("rangeval a single value that is not positive:  ",
           rangeval)
    rangeval <- c(0,rangeval)
  }
  if(length(rangeval)>2)
    stop('rangeval must be a vector of length 2;  ',
         'length(rangeval) = ', length(rangeval))
  if(diff(rangeval)<=0)
    stop('rangeval must cover a positive range;  diff(rangeval) = ',
         diff(rangeval) )
##
## 2.  check nbasis>0
##
#  check NBASIS
  if(!is.numeric(nbasis))
    stop('nbasis must be numeric;  class(nbasis) = ',
         class(nbasis) )
  if(length(nbasis) != 1)
    stop('nbasis must be a scalar;  length(nbasis) = ',
         length(nbasis) )
  if((nbasis%%1) != 0)
    stop('nbasis must be an integer;  nbasis%%1 = ',
         nbasis%%1)
  nbasis <- as.integer(nbasis)
  if (nbasis <= 0) stop ("NBASIS is not positive.")
##
## 3.  check DROPIND
##
  if (length(dropind) == 0) dropind <- NULL
#
  if (length(dropind) > 0){
    if(!is.numeric(dropind))
      stop('dropind must be numeric;  is ', class(dropind))
    doops <- which((dropind%%1)>0)
    if(length(doops)>0)
      stop('dropind must be integer;  element ', doops[1],
           " = ", dropind[doops[1]], '; fractional part = ',
           dropind[doops[1]] %%1)
#
    doops0 <- which(dropind<=0)
    if(length(doops0)>0)
      stop('dropind must be positive integers;  element ',
           doops0[1], ' = ', dropind[doops0[1]], ' is not.')
    doops2 <- which(dropind>nbasis)
    if(length(doops2)>0)
        stop("dropind must not exceed nbasis = ", nbasis,
             ';  dropind[', doops2[1], '] = ', dropind[doops2[1]])
#
    dropind <- sort(dropind)
    if(length(dropind) > 1) {
      if(min(diff(dropind)) == 0)
        stop("Multiple index values in DROPIND.")
    }
  }
##
## 5.  Check ctr
##
  if(!is.numeric(ctr))
    stop('ctr must be numeric;  class(ctr) = ',
         class(ctr) )
  if(length(ctr) != 1)
    stop('ctr must be a scalar;  length(ctr) = ',
         length(ctr) )
##
## 6.  set up the basis object
##
  type        <- "polynom"
  params      <- as.vector(ctr)
#
  basisobj <- basisfd(type=type,   rangeval=rangeval, nbasis=nbasis,
                    params=params, dropind=dropind,   quadvals=quadvals,
                    values=values, basisvalues=basisvalues)
##
## 7.  names
##
  {
    if(length(names) == nbasis)
      basisobj$names <- names
    else {
      if(length(names)>1)
        stop('length(names) = ', length(names), ';  must be either ',
             '1 or nbasis = ', nbasis)
      basisobj$names <- paste(names, 0:(nbasis-1), sep="")
    }
  }
##
## 8.  Done
##
  if(!is.null(axes))basisobj$axes <- axes
  basisobj

}
create.power.basis <- function(rangeval  = c(0,1), nbasis = NULL,
                               exponents = NULL, dropind=NULL,
                               quadvals=NULL, values=NULL,
                               basisvalues=NULL, names='power',
                               axes=NULL)
{
#  This function creates an power functional data basis
#  Arguments
#  RANGEVAL ... An array of length 2 containing the lower and upper
#               boundaries for the rangeval of argument values
#  NBASIS   ... The number of basis functions.  If this conflicts with
#               the length of exponent, the latter is used.
#  EXPONENT ... The rate parameters defining x^exponent[i]
#  DROPIND  ... A vector of integers specifying the basis functions to
#               be dropped, if any.
#  QUADVALS .. A NQUAD by 2 matrix.  The firs t column contains quadrature
#                points to be used in a fixed point quadrature.  The second
#                contains quadrature weights.  For example, for (Simpson"s
#                rule for (NQUAD = 7, the points are equally spaced and the
#                weights are delta.*[1, 4, 2, 4, 2, 4, 1]/3.  DELTA is the
#                spacing between quadrature points.  The default is
#                matrix("numeric",0,0).
#  VALUES  ... A list, with entries containing the values of
#                the basis function derivatives starting with 0 and
#                going up to the highest derivative needed.  The values
#                correspond to quadrature points in QUADVALS and it is
#                up to the user to decide whether or not to multiply
#                the derivative values by the square roots of the
#                quadrature weights so as to make numerical integration
#                a simple matrix multiplication.
#                Values are checked against QUADVALS to ensure the correct
#                number of rows, and against NBASIS to ensure the correct
#                number of columns.
#                The default value of is VALUES is vector("list",0).
#                VALUES contains values of basis functions and derivatives at
#                quadrature points weighted by square root of quadrature weights.
#                These values are only generated as required, and only if slot
#                QUADVALS is not matrix("numeric",0,0).
#  BASISVALUES ... A vector of lists, allocated by code such as
#                vector("list",1).
#                This field is designed to avoid evaluation of a
#                basis system repeatedly at a set of argument values.
#                Each list within the vector corresponds to a specific set
#                of argument values, and must have at least two components,
#                which may be tagged as you wish.
#                The first component in an element of the list vector contains the
#                argument values.
#                The second component in an element of the list vector
#                contains a matrix of values of the basis functions evaluated
#                at the arguments in the first component.
#                The third and subsequent components, if present, contain
#                matrices of values their derivatives up to a maximum
#                derivative order.
#                Whenever function getbasismatrix is called, it checks
#                the first list in each row to see, first, if the number of
#                argument values corresponds to the size of the first dimension,
#                and if this test succeeds, checks that all of the argument
#                values match.  This takes time, of course, but is much
#                faster than re-evaluation of the basis system.  Even this
#                time can be avoided by direct retrieval of the desired
#                array.
#                For example, you might set up a vector of argument values
#                called "evalargs" along with a matrix of basis function
#                values for these argument values called "basismat".
#                You might want too use tags like "args" and "values",
#                respectively for these.  You would then assign them
#                to BASISVALUES with code such as
#                  basisobj$basisvalues <- vector("list",1)
#                  basisobj$basisvalues[[1]] <-
#                               list(args=evalargs, values=basismat)

#  Returns
#  BASISOBJ  ... a functional data basis object of type "power"

#  Last modified 9 November 2008 by Spencer Graves
#  Previously modified 6 January 2008 by Jim Ramsay

#  Default basis for missing arguments

##
## 1.  check RANGEVAL
##
  if(!is.numeric(rangeval))
    stop('rangaval must be numeric;  class(rangeval) = ',
         class(rangeval) )
  if(length(rangeval)<1)
    stop('rangeval must be a numeric vector of length 2;  ',
         'length(rangeval) = 0.')
  if (length(rangeval) == 1) {
    if( rangeval <= 0)
      stop("rangeval a single value that is not positive:  ",
           rangeval)
    rangeval <- c(0,rangeval)
  }
  if(length(rangeval)>2)
    stop('rangeval must be a vector of length 2;  ',
         'length(rangeval) = ', length(rangeval))
  if(diff(rangeval)<=0)
    stop('rangeval must cover a positive range;  diff(rangeval) = ',
         diff(rangeval) )
##
## 2.  check 0 < nbasis == length(exponents)
##
  {
    if(is.null(nbasis)){
      if(is.null(exponents)){
        nbasis <- 2
        exponents <- 0:1
      }
      else {
        if(is.numeric(exponents)){
          nbasis <- length(exponents)
          if(length(unique(exponents)) != nbasis)
            stop('duplicates found in exponents;  not allowed.')
        }
        else
          stop('exponents must be numeric;  class(exponents) = ',
               class(exponents) )
      }
    }
    else {
      if(is.numeric(nbasis)){
        if(length(nbasis)!=1)
          stop('nbasis must be a scalar;  length(nbasis) = ',
               length(nbasis) )
        if((nbasis %%1) != 0)
          stop('nbasis just be an integer;  nbasis%%1 = ',
               nbasis%%1)
        {
          if(is.null(exponents))
            exponents <- 0:(nbasis-1)
          else {
            if(is.numeric(exponents)){
              if(length(exponents) != nbasis)
                stop('length(exponents) must = nbasis;  ',
                     'length(exponents) = ', length(exponents),
                     ' != nbasis = ', nbasis)
              if(length(unique(exponents)) != nbasis)
                stop('duplicates found in exponents;  not allowed.')
            }
            else
              stop('exponents must be numeric;  class(exponents) = ',
                   class(exponents) )
          }
        }
      }
      else stop('nbasis must be numeric;  class(nbasis) = ',
                class(nbasis) )
    }
  }
#  check whether exponents are negative,
#  and if so whether the range includes nonpostive values

  if (any(exponents < 0) && rangeval[1] <= 0)
    stop("An exponent is negative and range contains",
         " 0 or negative values.")
##
## 3.  check DROPIND
##
  if (length(dropind) == 0) dropind <- NULL
#
  if (length(dropind) > 0){
    if(!is.numeric(dropind))
      stop('dropind must be numeric;  is ', class(dropind))
    doops <- which((dropind%%1)>0)
    if(length(doops)>0)
      stop('dropind must be integer;  element ', doops[1],
           " = ", dropind[doops[1]], '; fractional part = ',
           dropind[doops[1]] %%1)
#
    doops0 <- which(dropind<=0)
    if(length(doops0)>0)
      stop('dropind must be positive integers;  element ',
           doops0[1], ' = ', dropind[doops0[1]], ' is not.')
    doops2 <- which(dropind>nbasis)
    if(length(doops2)>0)
        stop("dropind must not exceed nbasis = ", nbasis,
             ';  dropind[', doops2[1], '] = ', dropind[doops2[1]])
#
    dropind <- sort(dropind)
    if(length(dropind) > 1) {
      if(min(diff(dropind)) == 0)
        stop("Multiple index values in DROPIND.")
    }
  }
##
## 4.  set up the basis object
##
  type        <- "power"
  params      <- sort(as.vector(exponents))

  basisobj <- basisfd(type=type,     rangeval=rangeval, nbasis=nbasis,
                    params=params, dropind=dropind,   quadvals=quadvals,
                    values=values, basisvalues=basisvalues)
##
## 5.  names
##
  {
    if(length(names) == nbasis)
      basisobj$names <- names
    else {
      if(length(names)>1)
        stop('length(names) = ', length(names), ';  must be either ',
             '1 or nbasis = ', nbasis)
      basisobj$names <- paste(names, 0:(nbasis-1), sep="")
    }
  }
##
## 6.  Done
##
  if(!is.null(axes))basisobj$axes <- axes
  basisobj

}
CSTR2in <- function(Time, condition = 
   c('all.cool.step', 'all.hot.step', 'all.hot.ramp', 'all.cool.ramp',
     'Tc.hot.exponential', 'Tc.cool.exponential', 'Tc.hot.ramp',
     'Tc.cool.ramp', 'Tc.hot.step', 'Tc.cool.step'),
   tau=1){
##
## Simulated Input Vectors for
## Continuously Stirred Temperature Reactor:
##
## Returns data.frame(Fvec, CA0vec, T0vec, Tcinvec, Fcvec)

##
## condition = c( 'all_cool_step', 'all_hot_step',
##                'all_hot_ramp', 'all_cool_ramp',
##     'Tc_hot_exponential', 'Tc_cool_exponential',
##                 'Tc_hot_ramp', 'Tc_cool_ramp'
##                 'Tc_hot_step', 'Tc_cool_step'

#  Last modified 20 April 2007 by Spencer Graves
#%  Matlab version last modified 2 May 2005
  rtnMat <- function(Fvec=Fvec, CA0vec=CA0vec,
        T0vec=T0vec, Tcinvec=Tcinvec, Fcvec=Fcvec){
    x <- cbind(F.=Fvec, CA0=CA0vec, T0=T0vec,
               Tcin=Tcinvec, Fc=Fcvec)
#    Fnames <- dimnames(Fvec)[[1]]
#    dimnames(x) <- list(Fnames,
#        c("Fvec", "CA0vec", "T0vec", "Tcinvec", "Fcvec"))
    x
  }
#
  n       = length(Time);
# defaults used for condition == Tc* 
  CA0vec  =   rep(2, n) 
  T0vec   = rep(323, n) 
  Fcvec   =  rep(15, n);
  
#switch condition
  if(condition[1] == 'all.cool.step'){
#        %  compute F
        
    Fvec = rep(1.0, n);
    Fvec[(4 <= Time) & (Time <  8)] = 1.5;
    Fvec[(8 <= Time) & (Time < 12)] = 0.5;
        
#        %  compute C_{A0}
        
    CA0vec = rep(2.0, n);
    CA0vec[(16 <= Time) & (Time < 20)] = 2.2;
    CA0vec[(20 <= Time) & (Time < 24)] = 1.8;
        
#        %  compute T0
        
    T0vec = rep(323,n);
    T0vec[(28 <= Time) & (Time < 32)] = 343;
    T0vec[(32 <= Time) & (Time < 36)] = 303;
        
#        %  compute Tcin
        
    Tcinbase = 335;
    Tcinvec  = rep(Tcinbase,n);
    Tcinvec[(40 <= Time) & (Time < 44)] = Tcinbase+5;
    Tcinvec[(44 <= Time) & (Time < 48)] = Tcinbase-5;
        
#        %  compute Fc
        
    Fcvec = rep(15,n);
    Fcvec[(52 <= Time) & (Time < 56)] = 20;
    Fcvec[(56 <= Time) & (Time < 60)] = 10;

    return(rtnMat(Fvec, CA0vec, T0vec, Tcinvec, Fcvec))
  }
#
  if(condition[1] == 'all.hot.step'){
        
#        %  compute F
        
    Fvec = rep(1.0,n);
    Fvec[(4 <= Time) & (Time <  8)] = 1.5;
    Fvec[(8 <= Time) & (Time < 12)] = 0.5;
        
#        %  compute C_{A0}
        
    CA0vec = rep(2.0, n);
    CA0vec[(16 <= Time) & (Time < 20)] = 2.2;
    CA0vec[(20 <= Time) & (Time < 24)] = 1.8;
        
#        %  compute T0
        
    T0vec = rep(323, n);
    T0vec[(28 <= Time) & (Time < 32)] = 343;
    T0vec[(32 <= Time) & (Time < 36)] = 303;
        
#        %  compute Tcin
        
    Tcinbase = 365;
    Tcinvec  = rep(Tcinbase, n);
    Tcinvec[(40 <= Time) & (Time < 44)] = Tcinbase+5;
    Tcinvec[(44 <= Time) & (Time < 48)] = Tcinbase-5;
        
#        %  compute Fc
        
    Fcvec = rep(15, n);
    Fcvec[(52 <= Time) & (Time < 56)] = 20;
    Fcvec[(56 <= Time) & (Time < 60)] = 10;

    return(rtnMat(Fvec, CA0vec, T0vec, Tcinvec, Fcvec))

  }

  if(condition[1] == 'all.hot.ramp'){

    Fvec    = rep(1.0, n);
    Tcinvec = rep(365, n);
        
    index = ((2 <= Time) & (Time < 10));
    Fvec[index] = 1.5;
    index = ((Time >= 10) & (Time < 14));
    Fvec[index] = -0.2*(Time[index]-10) + 1.5;
    index = ((Time >= 14) & (Time < 18));
    Fvec[index] = -0.2*4 + 1.5;
    
    index = ((Time >= 26) & (Time < 34));
    CA0vec[index] = 2.5;
    index = ((Time >= 34) & (Time < 38));
    CA0vec[index] = -0.2*(Time[index]-34) + 2.5;
    index = ((Time >= 38) & (Time < 42));
    CA0vec[index] = -0.2*4 + 2.5;
    
    index = ((Time >= 50) & (Time < 58));
    T0vec[index] = 353;
    index = ((Time >= 58) & (Time < 62));
    T0vec[index] = -20*(Time[index]-58) + 353;
    index = ((Time >= 62) & (Time < 66));
    T0vec[index] = -20*4 + 353;
        
    index = ((Time >= 74) & (Time < 82));
    Tctop = 390;
    Tcinvec[index] = Tctop;
    index = ((Time >= 82) & (Time < 86));
    Tcinvec[index] = -10*(Time[index]-82) + Tctop;
    index = ((Time >= 86) & (Time < 90));
    Tcinvec[index] = -10*4 + Tctop;
        
    index = ((Time >=  98) & (Time < 106));
    Fcvec[index] = 25;
    index = ((Time >= 106) & (Time < 110));
    Fcvec[index] = -5*(Time[index]-106) + 25;
    index = ((Time >= 110) & (Time < 114));
    Fcvec[index] = -5*4 + 25;

    return(rtnMat(Fvec, CA0vec, T0vec, Tcinvec, Fcvec))

  }

  if(condition[1] == 'all.cool.ramp'){

    Fvec    = rep(0.05, n);
    Tcinvec = rep(330, n);
        
    index = (( 2 <= Time) & (Time < 10));
    Fvec[index] = 1.5;
    index = ((Time >= 10) & (Time < 14));
    Fvec[index] = -0.2*(Time[index]-10) + 1.5;
    index = ((Time >= 14) & (Time < 18));
    Fvec[index] = -0.2*4 + 1.5;
    
    index = ((Time >= 26) & (Time < 34));
    CA0vec[index] = 2.5;
    index = ((Time >= 34) & (Time < 38));
    CA0vec[index] = -0.2*(Time[index]-34) + 2.5;
    index = ((Time >= 38) & (Time < 42));
    CA0vec[index] = -0.2*4 + 2.5;
    
    index = ((Time >= 50) & (Time < 58));
    T0vec[index] = 353;
    index = ((Time >= 58) & (Time < 62));
    T0vec[index] = -20*(Time[index]-58) + 353;
    index = ((Time >= 62) & (Time < 66));
    T0vec[index] = -20*4 + 353;
    
    index = ((Time >= 74) & (Time < 82));
    Tctop = 355;
    Tcinvec[index] = Tctop;
    index = ((Time >= 82) & (Time < 86));
    Tcinvec[index] = -10*(Time[index]-82) + Tctop;
    index = ((Time >= 86) & (Time < 90));
    Tcinvec[index] = -10*4 + Tctop;
    
    index = ((Time >=  98) & (Time < 106));
    Fcvec[index] = 25;
    index = ((Time >= 106) & (Time < 110));
    Fcvec[index] = -5*(Time[index]-106) + 25;
    index = ((Time >= 110) & (Time < 114));
    Fcvec[index] = -5*4 + 25;

    return(rtnMat(Fvec, CA0vec, T0vec, Tcinvec, Fcvec))

  }

  if(condition[1] == 'Tc.hot.exponential'){
        
    Fvec    = rep(1.0, n);
    Tcinvec = rep(365, n);
    
    index = (Time < 10); 
    Tcinvec[index] = 400 - (400 - 365)*exp(-(Time[index])/tau);
    index = ((10 <= Time) & (Time < 20)); 
    Tcinvec[index] = 344 - (344 - 400)*exp(-(Time[index] - 10)/tau);
    index = (20 <= Time); 
    Tcinvec[index] = 365 - (365 - 344)*exp(-(Time[index] - 20)/tau);

    return(rtnMat(Fvec, CA0vec, T0vec, Tcinvec, Fcvec))

  }

  if(condition[1] == 'Tc.cool.exponential'){
        
    Fvec    = rep(0.05, n);
    Tcinvec =  rep(330, n);
    
    index = (Time/5 < 10); 
    Tcinvec[index] = 400 - (400 - 365)*exp(-(Time[index] )/tau);
    index = ((10 <= Time/5) & (Time/5 < 20)); 
    Tcinvec[index] = 344 - (344 - 400)*exp(-(Time[index] - 10)/tau);
    index = (20 <= Time/5); 
    Tcinvec[index] = 365 - (365 - 344)*exp(-(Time[index] - 20)/tau);

    return(rtnMat(Fvec, CA0vec, T0vec, Tcinvec, Fcvec))
    
  }

  if(condition[1] == 'Tc.hot.ramp'){

    Fvec    = rep(1.0, n);
    Tcinvec = rep(365, n);
        
    index = ((2 <= Time) & (Time < 10)); 
    Tcinvec[index] = 400;
    index = ((10 <= Time) & (Time < 14)); 
    Tcinvec[index] = -14*(Time[index]-10) + 400;
    index = ((14 <= Time) & (Time < 18)); 
    Tcinvec[index] = -14*4 + 400;

    return(rtnMat(Fvec, CA0vec, T0vec, Tcinvec, Fcvec))
    
  }

  if(condition[1] == 'Tc.cool.ramp'){

    Fvec    = rep(0.05, n);
    Tcinvec =  rep(330, n);
        
    index = ((2 <= Time/5) & (Time/5 < 10)); 
    Tcinvec[index] = 340;
    index = ((10 <= Time/5) & (Time/5 < 14)); 
    Tcinvec[index] = -0.8*(Time[index]-5*10) + 340;
    index = ((14 <= Time/5) & (Time/5 < 22)); 
    Tcinvec[index] = -5*0.8*4 + 340;

    return(rtnMat(Fvec, CA0vec, T0vec, Tcinvec, Fcvec))
    
  }

  if(condition[1] == 'Tc.hot.step'){

    Fvec    = rep(1.0, n);
    Tcinvec = rep(365, n);
    
    index = ((2 <= Time & Time < 12)); 
    Tcinvec[index] = 350;

    return(rtnMat(Fvec, CA0vec, T0vec, Tcinvec, Fcvec))
    
  }

  if(condition[1] == 'Tc.cool.step'){

    Fvec    = rep(0.05, n);
    Tcinvec =  rep(335, n);
    
    index = ((2 <= Time) & (Time < 12)); 
    Tcinvec[index] = 320;
        
    return(rtnMat(Fvec, CA0vec, T0vec, Tcinvec, Fcvec))
    
  }
}

CSTR2 <- function(Time, y, parms){
#function [Dy, DDy] = CSTR2(Time, y, parms)
#%  CSTR2 is called by the ODE solving function and evaluates the
#%  right hand side of the equation.
#
# 'lsoda' requires this function to return a list
# whose first component is a vector giving the derivatives of 'y'
#      with respect to 'Time' and 
# whose second component (if any) "is a vector ... of global values
#      that are required at each point in 'Time'".
# CSTR2 provides no second component.    

#%  R port 20 April 2007 by Spencer Graves
#%  Matlab version last modified 5 May 2005

  tau = 1;

#[F,CA0,T0,Tcin,Fc] = CSTR2in(Time, condition, tau);
  CSTR.in <- CSTR2in(Time, parms$condition, tau) 

  fitstruct <- parms$fitstruct
  kref   = fitstruct$kref;
  EoverR = fitstruct$EoverR;
  a      = fitstruct$a;
  b      = fitstruct$b;

  V      = fitstruct$V;
  FoverV = CSTR.in[, "F."]/V;

  aFc2bp = a*CSTR.in[, "Fc"]^b;
  const1 = V*fitstruct$rho*fitstruct$Cp;
  const2 = 2*fitstruct$rhoc*fitstruct$Cpc;

  betaTTcin = (CSTR.in[, "Fc"]*aFc2bp/(    
    const1*(CSTR.in[, "Fc"]+aFc2bp/const2)) )
  betaTT0   = FoverV;

#%  set up current values of variables
  {
    if( length(y) == 2 ){
      Ci = y[1];
      Ti = y[2];
    }
    else{
      Ci = y[,1];
      Ti = y[,2];
    }
  }
# end if(length(y)==2) 

  Tdif = 1/as.vector(Ti) - 1/fitstruct$Tref;
  betaCC = kref*exp(-1e4*EoverR*Tdif);
  betaTC = (-fitstruct$delH/const1)*betaCC;
  betaTT = FoverV + betaTTcin;

  DC = (-(FoverV + betaCC)*as.vector(Ci)  +
    (CSTR.in[, "F."]/V)*CSTR.in[, "CA0"]); 
  DT = (-betaTT*Ti + betaTC*Ci + (CSTR.in[, "F."]/V) * 
        CSTR.in[, "T0"] + betaTTcin*CSTR.in[, "Tcin"]) ;
#
  x <- cbind(Conc=DC, Temp=DT)
# 
  list(x)
}
CSTRfitLS <- function(coef, datstruct, fitstruct, 
                                 lambda, gradwrd=FALSE){
#
#function [res, Dres] = CSTRfitLS(coef, datstruct, fitstruct, ...
#                                 lambda, gradwrd)

#  Last modified 2007.05.10 by Spencer Graves
#% previously modified 9 May 2005
##
## 1.  Set up   
##
  max.log.betaCC <- (log(.Machine$double.xmax)/3)
# For certain values of 'coef',
# naive computation of betaCC will return +/-Inf,
# which generates NAs in Dres.
# Avoid this by clipping betaCC
#
# log(.Machine$double.xmax)/2 is too big,
# because a multiple of it is squared in CSTRfn ... 
#  
  fit = fitstruct$fit;

  basismat = datstruct$basismat;
# basisMat <- read.csv('CSTRbasismat.csv', header=FALSE)
#[N, nbasis] = size(datstruct.basismat);
  N <- dim(basismat)[1]
  nbasis <- dim(basismat)[2]
#           
  ind1    = 1:nbasis
  ind2    = ((nbasis+1):(2*nbasis))
# onesb is redefined later before it is used  
#  onesb <- rep(1, nbasis)
  zeromat <- array(0, dim=c(N, nbasis), 
            dimnames=dimnames(basismat) )
#
  Ccoef  = coef[ind1];
  Tcoef  = coef[ind2];
  
  Cwt  = as.vector(datstruct$Cwt)
  Twt  = as.vector(datstruct$Twt)
##
## 2.  Sres = matrix of residuals = 
##     datstruct$y - predicted 
##
# Sres = []
  yNames <- c("Conc", "Temp")
  fitNames <- yNames[as.logical(fit)]
  fit12 <- length(fitNames)

  yobs = datstruct$y;
  basisNames <- dimnames(basismat)
  Sres <- array(NA, dim=c(N, fit12), dimnames=
                list(basisNames[[1]], fitNames))

  if( fit[1]){
    resC = yobs[,1] - basismat%*%Ccoef;
    Sres[,1] <- resC/sqrt(Cwt) 
  }

  if( fit[2]){
#    resT = yobs[,fit12] - basismat%*%Tcoef;
    resT = yobs[,2] - basismat%*%Tcoef;
    Sres[,fit12] <- resT/sqrt(Twt)
  }
##
## 3.  Compute (Conc, Temp) and d/dt at quadrature points 
##
#  3.1.  Get basic model coefficients  
  kref   = fitstruct$kref;
  EoverR = fitstruct$EoverR;
  a      = fitstruct$a;
  b      = fitstruct$b;
    
#%  3.2.  basis function values at quadrature points

  quadmat  = datstruct$quadbasismat;
  Dquadmat = datstruct$Dquadbasismat;
# [nquad, nbasis] = size(quadmat);
  nquad <- dim(quadmat)[1]
  onesb = rep(1,nbasis)
  names(onesb) <- dimnames(quadmat)[[2]]
#  onesq = rep(nquad, 1);
  onesq <- rep(1, nquad)
  names(onesq) <- dimnames(quadmat)[[1]] 

#%  3.3.  set up the values of C and T at quad. pts.

  Chatquad = as.vector(quadmat%*%Ccoef)
  Thatquad = as.vector(quadmat%*%Tcoef)

  DC = Dquadmat%*%Ccoef;
  DT = Dquadmat%*%Tcoef;
##
## 4.  Right hand side of differential equation
##  

#% 4.1.  set up some constants that are required

  V      = fitstruct$V;
  rho    = fitstruct$rho;
  rhoc   = fitstruct$rhoc;
  delH   = fitstruct$delH;
  Cp     = fitstruct$Cp;
  Cpc    = fitstruct$Cpc;
  Tref   = fitstruct$Tref;
  
#%  these constants can vary.
#%  see function CSTR2in for other conditions

  Fc  = datstruct$Fc
  F.   = datstruct$F.
  CA0 = datstruct$CA0;
  T0  = datstruct$T0;
  Tc  = datstruct$Tcin;

#% 4.2.  compute multipliers of outputs

  Tdif    = 1./Thatquad - 1./Tref;
#
#  betaCC  = kref*exp(-1e4*EoverR*Tdif);
  log.betaCC <- (log(abs(kref))-1e4*EoverR*Tdif)
  oops <- (log.betaCC > max.log.betaCC)
  if(any(oops)){
    warning(sum(oops), " of ", length(log.betaCC),
            " values of log(abs(betaCC)) exceed the max = ",
            max.log.betaCC, ";  thresholding.")
    log.betaCC[oops] <- max.log.betaCC 
  }
  betaCC <- sign(kref)*exp(log.betaCC)
#                     
  TCfac   = -delH/(rho*Cp);
  betaTC  = TCfac*betaCC;
  aFc2b   = a*Fc^b;
  K1      = V*rho*Cp;
  K2      = 1./(2.*rhoc*Cpc);
  betaTT  = Fc*aFc2b/(K1*(Fc + K2*aFc2b));
  betaTT0 = F./V;

#% 4.3.  compute right sides of equations 

  DChat = (-(betaTT0 + betaCC)*Chatquad + betaTT0*CA0)
  DThat = (-(betaTT0 + betaTT)*Thatquad + betaTC*Chatquad
              + betaTT0*T0 + betaTT*Tc)

#  4.4.  Deviation between left and right hand sides   
  LC = DC - DChat
  LT = DT - DThat

#  4.5.  Quadrature weights   
  quadwts = datstruct$quadwts;
  rootwts = sqrt(quadwts);

#  lambdaC = lambda[1];
#  lambdaT = lambda[2];
  lambdaC.5 = sqrt(lambda[1]/Cwt) 
  lambdaT.5 = sqrt(lambda[2]/Twt)
##
## 5.  Lres = scaled deviations of Dy from predicted 
##    
#  Lres <- rbind(LC*rootwts*sqrt(lambdaC/Cwt),
#                LT*rootwts*sqrt(lambdaT/Twt) )
  Lres <- cbind(LConc = as.vector(LC)*rootwts*lambdaC.5,
                LTemp = as.vector(LT)*rootwts*lambdaT.5)

##
## 6.  Combine Sres and Lres
##  
  res  = list(Sres=Sres, Lres=Lres);
  out <- list(res=res) 
##
##% 7. compute gradient if required
##
  if(gradwrd){

#  7.1.  Derivatives of fit residuals
    
#    DSres = [];
#     if(fit[1])DSres = cbind( -basismat./sqrt(Cwt),zeromat)
#    if( fit[2])DSres = c(DSres, zeromat, -basismat./sqrt(Twt));
    DSres <- NULL
    if(fit[1])
      DSres <- cbind(-basismat/sqrt(Cwt), zeromat)
    if(fit[2])
      DSres <- rbind(DSres, cbind(zeromat, -basismat/sqrt(Twt)))
#  DSresMat <- read.csv('CSTR-DSres.csv', header=FALSE)
# d.DSres <- DSres - as.matrix(DSresMat)
# quantile(d.DSres)    
#           0%           25%           50%           75%          100% 
#-3.971928e-06  0.000000e+00  0.000000e+00  0.000000e+00  1.573097e-05
# sqrt(mean(DSres^2)) =0.218
# Reasonable accuracy.      
    
#  7.2.  Derivatives of weight functions
    
    DtbetaCC = (1e4*EoverR/Thatquad^2)*betaCC;
    DtbetaTC = TCfac*DtbetaCC;
    
#  7.3.  Derivatives of RHS of operators
    
    DcDChat  = -(betaCC + betaTT0);
    DtDChat  = -DtbetaCC*Chatquad;
    DcDThat  =  betaTC;
    DtDThat  = -(betaTT+betaTT0) + DtbetaTC*Chatquad;
    
#  7.4.  Operator derivatives
    
#    DcLC   = Dquadmat - (DcDChat*onesb).*quadmat;
#    DtLC   =          - (DtDChat*onesb).*quadmat;
#    DcLT   =          - (DcDThat*onesb).*quadmat;
#    DtLT   = Dquadmat - (DtDThat*onesb).*quadmat;
    DcLC   = Dquadmat - outer(DcDChat, onesb)*quadmat;
#    DcLC   = Dquadmat - tcrossprod(DcDChat, onesb)*quadmat;
    DtLC   =          - outer(DtDChat, onesb)*quadmat;
    DcLT   =          - outer(DcDThat, onesb)*quadmat;
    DtLT   = Dquadmat - outer(DtDThat, onesb)*quadmat;
    
#  7.5.  Multiply operator derivatives by root of
#    %  quadrature weights over root of SSE weights
    
#    wtmat   = rootwts*onesb;
#    DcLCmat = DcLC.*wtmat.*sqrt(lambdaC./Cwt);
#    DtLCmat = DtLC.*wtmat.*sqrt(lambdaC./Cwt);
#    DcLTmat = DcLT.*wtmat.*sqrt(lambdaT./Twt);
#    DtLTmat = DtLT.*wtmat.*sqrt(lambdaT./Twt);
    wtmat   = outer(rootwts, onesb) 
    DcLCmat = DcLC*wtmat*lambdaC.5
    DtLCmat = DtLC*wtmat*lambdaC.5
    DcLTmat = DcLT*wtmat*lambdaT.5
    DtLTmat = DtLT*wtmat*lambdaT.5

#  7.6.  Matrices of derivative of operator residuals
    
#    DLres  = [[DcLCmat, DtLCmat]; [DcLTmat, DtLTmat]]
    DLres <- rbind(cbind(DcLCmat, DtLCmat),
                   cbind(DcLTmat, DtLTmat))
#  DLresMat <- read.csv('CSTR-DLres.csv', header=FALSE)
# d.DLres <- DLres - as.matrix(DLresMat)
# quantile(d.DLres)    
#           0%           25%           50%           75%          100% 
#-0.0006900152  0.0000000000  0.0000000000  0.0000000000  0.0006814540 
# sqrt(mean(DLres^2))
# 0.812
# Not great but acceptable.      
    
#  7.7.  Combine with derivative of fit residuals
    
    out$Dres   = list(DSres=DSres, DLres=DLres)
    
  }
  
#  else
#     Dres = vector("list", 0)

  return(out)
}
CSTRfn <- function(parvec, datstruct, fitstruct,
                   CSTRbasis, lambda, gradwrd=TRUE){
#function [res, Dres, fitstruct, df, gcv] =
#    CSTRfn(parvec, datstruct, fitstruct, CSTRbasis, lambda, gradwrd)

# Last modified with R port 2007.07.21 by Spencer Graves
#%  Previously modified 29 July 2005

  max.log.betaCC <- (log(.Machine$double.xmax)/3)
# For certain values of 'coef',
# naive computation of betaCC will return +/-Inf,
# which generates NAs in Dres.
# Avoid this by clipping betaCC
#
# log(.Machine$double.xmax)/2 is too big,
# because a multiple of it is squared in CSTRfn ...
#

#if nargin < 6, gradwrd = 1;  end
##
## 1.  Modify fitstruct for use with CSTRfitLS
##
#  cat("CSTRfn: parvec =", parvec, "\n")
#
  eps <- .Machine$double.eps
  eps1 <- (1+2*eps)
  fit <- fitstruct$fit
  if(is.null(fit))
    stop("fitstruct has no 'fit' component")

#%  load parameters
#[kref, EoverR, a, b] = par2vals(parvec, fitstruct)
#%  set up fitstruct
#fitstruct.kref   = kref
#fitstruct.EoverR = EoverR
#fitstruct.a      = a
#fitstruct.b      = b

  estimate <- fitstruct$estimate
  if(is.null(estimate))
    stop("fitstruct has no 'estimate' component")
#  Compare estimate with parvec
  if(sum(estimate) != length(parvec)){
    cat("ERROR in CSTRfn:  sum(estimate) != length(parvec)\n")
    cat("parvec = ", parvec, "; \n")
    stop("fitstruct$estimate = ",
         paste(estimate, collapse=" "))
  }
  m <- 0
  {
#  1.1.  Estimate kref starting from parvec or use fitstruct?
    if(estimate[1]){
      m <- m+1
      kref <- parvec[m]
      fitstruct$kref <- kref
    }
    else
      kref <- fitstruct$kref
  }
  {
#  1.2.  Estimate EoverR starting from parvec or use fitstruct?
    if(estimate[2]){
      m <- m+1
      EoverR <- parvec[m]
      fitstruct$EoverR <- EoverR
    }
    else
      EoverR<- fitstruct$EoverR
  }
  {
#  1.3.  Estimate 'a' starting from parvec or use fitstruct?
    if(estimate[3]){
      m <- m+1
      a <- parvec[m]
      fitstruct$a <- a
    }
    else
      a <- fitstruct$a
  }
  {
#  1.4.  Estimate b starting from parvec or use fitstruct?
    if(estimate[4]){
      m <- m+1
      b <- parvec[m]
      fitstruct$b <- b
    }
    else
      b<- fitstruct$b
  }
##
## 2.  Set up inner optimization:  optimize fit with respect to coef
##
#  2.1.  Set up
  tolval <- 1e-10
  itermax <- 10
  coef0 <- as.matrix(fitstruct$coef0)
  dim.coef0 <- dim(coef0)
  if(length(dim.coef0)>1){
    coef0 <- as.vector(coef0)
    if(length(dim.coef0)==2 && (dim.coef0[2]==2)){
      coefNames <- outer(c("Conc", "Temp"), 1:dim.coef0[1], paste,
                         sep="")
      names(coef0) <- t(coefNames)
    }
  }
  ncoef <- length(coef0)
#  2.2.  initial value
#[res0, Dres] = CSTRfitLS(coef0, datstruct, fitstruct, lambda, 1)
  CSTR0 <- CSTRfitLS(coef0, datstruct, fitstruct, lambda, 1)

#  2.3.  Reshape for easier manipulation
  res0 <- with(CSTR0$res, c(Sres, Lres))
#
  N <- dim(CSTR0$res$Sres)[1]
  k12 <- dim(CSTR0$res$Sres)[2]
  nquad <- dim(CSTR0$res$Lres)[1]
  n.Sres <- N*k12
  n.Lres <- nquad*2
  Nval <- n.Sres + n.Lres
#
  nbasis <- (dim(CSTR0$Dres$DSres)[2]/2)
  Dres0 <- with(CSTR0$Dres, rbind(DSres, DLres))
# res0.mat <- readMat("CSTRfitLSres0.mat")# OK
# d.res0 <- (res0-res0.mat$res0)
# quantile(d.res0)
#      0%      25%      50%      75%     100%
#-7.1e-06 -1.4e-07  1.3e-08  2.1e-07  6.0e-06
# res0 good.

# Dres.mat <- read.csv("CSTRfitLSDres.csv", header=FALSE)
# d.Dres <- (Dres0-as.matrix(Dres.mat))
# quantile(d.Dres)
#           0%           25%           50%           75%          100%
#-0.0006900152  0.0000000000  0.0000000000  0.0000000000  0.0006814540
# sqrt(mean(Dres0^2)) # 0.747
# sqrt(mean(d.Dres^2)) # 9.8e-6
# Good.

# F0 = mean(res0.^2)
  F0 <- mean(res0^2)
  F1 <- 0
  iter <- 0
  fundif <- 1
# gradnorm0 = mean(res0'*Dres)
  r.D <- crossprod(res0, Dres0)
  gradnorm0 <- mean(r.D)
  if(is.na(gradnorm0))
    stop("Initial call to CSTRfitLS returned NAs with parvec = ",
         paste(parvec, collapse=", "), ";  sum(is.na(res0)) = ",
         sum(is.na(res0)), "; sum(is.na(Dres0)) = ", sum(is.na(Dres0)))
#
  gradnorm1 <- 1
##
## 3.  Gauss-Newton optimization loop:
##     optimize fit with respect to coef
##
  while(( gradnorm1 > tolval) | (fundif > tolval)){
#
    iter <- iter + 1
    if( iter > itermax) break
#
#    Dcoef = Dres\res0
    nNA.res0 <- sum(is.na(res0))
    nNA.Dres0 <- sum(is.na(Dres0))
    if(nNA.res0 | nNA.Dres0) {
      dump("parvec", "parvecError.R")
      cat("Error: ", nNA.res0, "and", nNA.Dres0,
          "NAs found in res0 and Dres0 with parvec =",
          parvec, "; parvec dumped to parvecError.R\n")
    }
#
#    Dcoef <- (lm.fit(Dres0, res0)$coefficients)
#
    Dres.svd <- svd(Dres0)
    ikeep <- with(Dres.svd, which(d > eps*max(d)))
    Dres.rank <- length(ikeep)
    if(Dres.rank < min(dim(Dres0)))
      warning("Dres0 has rank ", Dres.rank, " < dim(Dres0) = ",
              paste(dim(Dres0), collapse=", "),
              " in iteration ", iter,
              ".  Ignoring singular subspace.  ",
              "First (rank+1) singular values = ",
              paste(Dres.svd$d[1:(Dres.rank+1)], collapse=", "))
    Dcoef <- with(Dres.svd, v %*% ((t(u) %*% res0) / d))
#
#    %  initial step:  alpha = 1
    coef1 <- coef0 - Dcoef
#
#[res1, Dres] = CSTRfitLS(coef1, datstruct, fitstruct, lambda, 1)
    CSTR1 <- CSTRfitLS(coef1, datstruct, fitstruct, lambda, 1)
    res1 <- with(CSTR1$res, c(Sres, Lres))
    Dres1 <- with(CSTR1$Dres, rbind(DSres, DLres))
#
    F1 <- mean(res1^2)
    alpha <- 1
    fundif <- abs(F0-F1)/abs(F0)

#%     %  smaller steps as required, halving alpha each time
#%     while F1 >= F0*(1+2*eps)
    while(is.na(F1) || F1>= (eps1*F0) || any(is.na(Dres1))){
      alpha <- alpha/2
      if(is.na(F1)){
        n.na <- sum(is.na(res1))
        attr(coef1, "n.na.in.res1") <- n.na
#        ..CSTRfn.coef1.gen.NA <<- list(parvec=parvec, coef1=coef1)
        warning(n.na, " NAs in res1.")
#        warning(n.na, " NAs in res1;  coef1 saved in ",
#                "'..CSTRfn.coef1.gen.NA'")
      }
      if(alpha< 1e-4){
        pv <- paste(signif(parvec, 3), collapse=", ")
#        ..CSTRfn.coef1.gen.5 <<- list(parvec=parvec, coef1=coef1)
#        warning("Stepsize reduced below the minimum with parvec = ",
#             pv, " on iteration ", iter,
#             " in trying to optimize ", length(coef0),
#             " coefficients;  using suboptimal coefficients;  ",
#                "saved in '..CSTRfn.coef1.gen.5'")
        warning("Stepsize reduced below the minimum with parvec = ",
             pv, " on iteration ", iter,
             " in trying to optimize ", length(coef0),
             " coefficients;  using suboptimal coefficients. ")
        break
      }
#
      coef1 <- coef0 - alpha*Dcoef
#%    [res1, Dres] = CSTRfitLS(coef1, datstruct, fitstruct, lambda, 1)
      CSTR1 <- CSTRfitLS(coef1, datstruct, fitstruct, lambda, 1)
      res1 <- with(CSTR1$res, c(Sres, Lres))
      Dres1 <- with(CSTR1$Dres, rbind(DSres, DLres))
      F1 <- mean(res1^2)
      fundif <- abs(F0-F1)/abs(F0)
    }
#    gradnorm1 = mean(res1'*Dres)
    gradnorm1 <- mean(crossprod(res1, Dres1))
#%     disp(num2str([iter, F1, fundif, gradnorm1]))
    coef0 <- coef1
    res0 <- res1
    Dres0 <- Dres1
    F0 <- F1
    gradnorm0 <- gradnorm1
# end while(( gradnorm1 > tolval) | (fundif > tolval)){
  }
##
##% 4.  update coef
##
  coef <- coef0
  fitstruct$coef0 <- coef
#
# DresMat <- read.csv("CSTRfnDres.csv", header=FALSE)
# d.Dres <- Dres - as.matrix(DresMat)
# quantile(d.Dres)
#           0%           25%           50%           75%          100%
#-0.0006207360  0.0000000000  0.0000000000  0.0000000000  0.0006317204
# sqrt(mean(Dres^2)) # 0.747
# sqrt(mean(d.Dres^2))# 1.0e-5
#
##
##% 5.  compute df and gcv
##
  Zmat <- Dres0[1:ncoef,]
#  Nval = length(res1)
  Rfac <- Dres0[(ncoef+1):Nval,]
#  Smat = Zmat*inv(Zmat'*Zmat + Rfac'*Rfac)*Zmat'
# Use singular value decomposition so we never have to worry about
# ill conditioning.
  Zsvd <- svd(Zmat)
# Zmat = with(Zsvd, u %*% diag(d) %*% t(v))
# so Z'Z+R'R = v d^2 v' + R'R
#          = v %*% (d^2 + (R%*%v)'(R%*%v))
  Rv <- (Rfac %*% Zsvd$v)
  d2.vR.Rv <- (diag(Zsvd$d^2)+crossprod(Rv))
  d2.eig <- eigen(d2.vR.Rv, symmetric=TRUE)
# Check for ill conditioning
  d2.ev <- d2.eig$values
  ZR.rank <- sum(d2.ev>0)
  {
    if(ZR.rank<1){
      warning("Z'Z+R'R = 0")
      Smat <- array(0, dim=c(ncoef, ncoef))
    }
    else{
      if(ZR.rank<ncoef)
        warning("Z'Z+R'R is not positive definite.  rank = ",
                ZR.rank, ";  dim = ", ncoef, "; increasing the ",
                ncoef-ZR.rank, " smallest eigenvalues ",
                "to improve numeric stability.")
      d2.evMin <- eps*d2.ev[1]
      ZR.rank1 <- sum(d2.ev >= d2.evMin)
      if(ZR.rank1 < ZR.rank)
        warning("Z'Z+R'R is ill conditioned.  Increasing the ",
                ncoef-ZR.rank1, " smallest eigenvalues ",
                "to improve numeric stability.")
#  Z'(inv(Z'Z+R'R)Z
#     = (u%*%d%*%w) solve(LAM) (udw)'
      udw <- ((Zsvd$u * rep(Zsvd$d, each=ncoef)) %*% d2.eig$vectors)
# or
#udw1<-((Zsvd$u[,1:ZR.rank1, drop=FALSE]*rep(Zsvd$d[1:ZR.rank1],e=ncoef))%*%d2.eig$vectors[1:ZR.rank1,,drop=FALSE])
      d2.ev2 <- pmax(d2.ev, d2.evMin)
      Smat <- ((udw / rep(d2.ev2, each=ncoef)) %*% t(udw))
# or
#Smat1<-((udw1/rep(d2.ev2, each=ncoef)) %*% t(udw1))
    }
  }
  df. <- sum(diag(Smat))
  dfe <- ncoef - df.
  gcv <- (ncoef/dfe)*sum(res1[1:ncoef]^2)/dfe
##
##  6.  compute fits and residuals
##
# [N, nbasis] = size(datstruct.basismat)
  ind1 <- 1:nbasis
  ind2 <- (nbasis+1):(2*nbasis)
  Ccoef <- coef[ind1]
  Tcoef <- coef[ind2]
#
  phimat <- datstruct$basismat
#
  Chat0 <- phimat %*% Ccoef
  That0 <- phimat %*% Tcoef
#
  yobs <- datstruct$y
#
  Cwt <- as.vector(datstruct$Cwt)
  Twt <- as.vector(datstruct$Twt)
#
#  res = []
#  if fit(1) res = [res; (yobs(:,1) - Chat0)./sqrt(Cwt)]
#  if fit(2)res = [res; (yobs(:,2) - That0)./sqrt(Twt)]
#
  yNames <- c("Conc", "Temp")
  fitNames <- yNames[as.logical(fit)]
  fit12 <- length(fitNames)
  basisNames <- dimnames(phimat)
  res <- array(NA, dim=c(N, fit12), dimnames=
                list(basisNames[[1]], fitNames))
  if(fit[1])res[, 1] <- ((yobs[,1] - Chat0)/sqrt(Cwt))
#  if(fit[2])res[, fit12] <- ((yobs[,fit12] - That0)/sqrt(Twt))
  if(fit[2])res[, fit12] <- ((yobs[,2] - That0)/sqrt(Twt))
#  res[1:5] matches Matlab 2007.05.29
##
## 7.  Derivatives?
##
  Dres <- Dres0
  if( gradwrd){
#
#  7.1.  set up basis and basis derivatve matrices
#
    quadmat <- datstruct$quadbasismat
    Dquadmat <- datstruct$Dquadbasismat
#
#    [nquad, nbasis] = size(quadmat)
#    onesb = ones(1,nbasis)
#    onesq = ones(nquad, 1)
    onesb <- rep(1, nbasis)
    onesq <- rep(1, nquad)
#
#  7.2.  set up some constants that are required
#
    V      <- fitstruct$V
    rho    <- fitstruct$rho
    rhoc   <- fitstruct$rhoc
    delH   <- fitstruct$delH
    Cp     <- fitstruct$Cp
    Cpc    <- fitstruct$Cpc
    Tref   <- fitstruct$Tref
#
#  7.3.  Set up input arrays
#
    F.  <- datstruct$F.
    CA0 <- datstruct$CA0
    T0  <- datstruct$T0
    Tc  <- datstruct$Tcin
    Fc  <- datstruct$Fc
#
#  7.4.  C and T values at fine grid
#
    Chat  <- as.vector(quadmat%*%Ccoef)
    That  <- as.vector(quadmat%*%Tcoef)
    DChat <- as.vector(Dquadmat%*%Ccoef)
    DThat <- as.vector(Dquadmat%*%Tcoef)
#
#  7.5.  betaCC and betaTC depend on kref and Eover R
    Tdif   <- 1/That - 1/Tref
#    temp   = exp(-1e4*EoverR*Tdif)
    log.temp <- (-1e4*EoverR*Tdif)
    oops <- (log.temp > max.log.betaCC)
    if(any(oops)){
      warning(sum(oops), " of ", length(log.temp),
              " values of (-1e4*EoverR*Tdif) exceed the max = ",
              max.log.betaCC, ";  thresholding.")
      log.temp[oops] <- max.log.betaCC
    }
    temp <- exp(log.temp)
#
    betaCC <- kref*temp
    TCfac  <- (-delH/(rho*Cp))
    betaTC <- TCfac*betaCC
#
#  7.6.  betaTT depends on a and b
    Fc2b    <- Fc^b
    aFc2b   <- a*Fc2b
    K1      <- V*rho*Cp
    K2      <- 1./(2.*rhoc*Cpc)
    betaTT  <- Fc*aFc2b/(K1*(Fc + K2*aFc2b))
    betaTT0 <- F./V
#
#  7.7.  compute derivatives of residuals
#
#    %  L values
#
    LC <- DChat + (betaTT0 + betaCC)*Chat - betaTT0*CA0*onesq
    LT <- DThat + ((betaTT0 + betaTT)*That - betaTC*Chat -
                 (betaTT0*T0 + betaTT*Tc))
#
#    %  first order derivatives of L values
#    %  derivatives of L values with respect to
#    %  coefficient vectors c and t
#
    DtbetaCC <- (1e4*EoverR/That^2)*betaCC
    DtbetaTC <- TCfac*DtbetaCC
#
    DcDChat  <- (-(betaCC + betaTT0))
    DtDChat  <- (-DtbetaCC*Chat)
    DcDThat  <-  betaTC
    DtDThat  <- (-(betaTT+betaTT0) + DtbetaTC*Chat)
#
    DcLC   <- (Dquadmat - outer(DcDChat, onesb)*quadmat)
    DtLC   <-          - outer(DtDChat, onesb)*quadmat
    DcLT   <-          - outer(DcDThat, onesb)*quadmat
    DtLT   <- (Dquadmat - outer(DtDThat, onesb)*quadmat)
#
    quadwts    <- datstruct$quadwts
    rootwts    <- sqrt(quadwts)
    quadwtsmat <- outer(quadwts, onesb)
#
#    %  k derivatives
#
    lamC <- lambda[1]
    lamT <- lambda[2]

#  7.8.  assemble the Jacobian matrix

#    DLC <- sqrt(lamC/Cwt).*[DcLC, DtLC]
#    DLT <- sqrt(lamT/Twt).*[DcLT, DtLT]
    DLC <- sqrt(lamC/Cwt)*cbind(DcLC, DtLC)
    DLT <- sqrt(lamT/Twt)*cbind(DcLT, DtLT)
#
#    Jacobian <- [DLC; DLT]
    Jacobian <- rbind(DLC, DLT)
#
#  7.9.  compute derivatives with respect to parameters
#
#    %  set up right hand side of equation D2GDc
#
#    D2GDc <- []
    D2GDc. <- vector('list', 4)
    names(D2GDc.) <- c("kref", "EoverR", "a", "b")
#
#    %  kref
#
    if( estimate[1]) {
#
#        %  first derivative of L values
#
      DkbetaCC <- temp
      DkbetaTC <- TCfac*DkbetaCC
#
      DkLC <-  DkbetaCC*Chat
      DkLT <- -DkbetaTC*Chat
#
#        %  second derivative of L values
#
      DktbetaCC <- (1e4*EoverR/That^2)*temp
      DktbetaTC <- TCfac*DktbetaCC
#
      DkcLC <- outer(  DkbetaCC,   onesb)*quadmat
      DkcLT <- outer( -DkbetaTC, onesb)*quadmat
      DktLC <- outer( DktbetaCC*Chat, onesb)*quadmat
      DktLT <- outer(-DktbetaTC*Chat, onesb)*quadmat
#
#      D2GDck <- zeros(2*nbasis,1)
#
      D2GDck <- rep(0, 2*nbasis)
#        D2GDck(ind1,1) <- (lamC/Cwt).* ...
#            (DcLC'*(DkLC.*quadwts) + DkcLC'*(LC.*quadwts)) + ...
#            (lamT/Twt).* ...
#            (DcLT'*(DkLT.*quadwts) + DkcLT'*(LT.*quadwts))
      D2GDck[ind1] <- ((lamC/Cwt)*
            (t(DcLC)%*%(DkLC*quadwts) + t(DkcLC)%*%(LC*quadwts)) +
            (lamT/Twt)*
            (t(DcLT)%*%(DkLT*quadwts) + t(DkcLT)%*%(LT*quadwts)) )
#        D2GDck(ind2,1) <- (lamC/Cwt).* ...
#            (DtLC'*(DkLC.*quadwts) + DktLC'*(LC.*quadwts)) + ...
#            (lamT/Twt).* ...
#            (DtLT'*(DkLT.*quadwts) + DktLT'*(LT.*quadwts))
      D2GDck[ind2] <- ((lamC/Cwt)*
            (t(DtLC)%*%(DkLC*quadwts) + t(DktLC)%*%(LC*quadwts)) +
            (lamT/Twt)*
            (t(DtLT)%*%(DkLT*quadwts) + t(DktLT)%*%(LT*quadwts)) )
#
#        D2GDc <- [D2GDc, D2GDck]
      D2GDc.$kref <- D2GDck
#
#    end kref
    }

#    %  EoverR
    if(estimate[2]){
#
#        %  first derivative of L values
#
      Dtemp    <- (-1e4*kref*Tdif*temp)
      DEbetaCC <- Dtemp
      DEbetaTC <- TCfac*DEbetaCC
#
      DELC  <-  DEbetaCC*Chat
      DELT  <- (-DEbetaTC*Chat)
#
#      DEtbetaCC <- (1e4.*kref  ./That.^2).* ...
#            (1 - 1e4.*EoverR.*Tdif).*temp
      DEtbetaCC <- ((1e4*kref/That^2)*
            (1 - 1e4*EoverR*Tdif)*temp)
      DEtbetaTC <- TCfac*DEtbetaCC
#
#        DEcLC <- (  DEbetaCC        *onesb).*quadmat
#        DEcLT <- ( -DEbetaTC        *onesb).*quadmat
#        DEtLC <- (( DEtbetaCC.*Chat)*onesb).*quadmat
#        DEtLT <- ((-DEtbetaTC.*Chat)*onesb).*quadmat
      DEcLC <- outer(  DEbetaCC, onesb)*quadmat
      DEcLT <- outer( -DEbetaTC, onesb)*quadmat
      DEtLC <- outer(( DEtbetaCC*Chat), onesb)*quadmat
      DEtLT <- outer((-DEtbetaTC*Chat), onesb)*quadmat
#
#        D2GDcE <- zeros(2*nbasis,1)
      D2GDcE <- rep(0, 2*nbasis)
#        D2GDcE(ind1,1) <- (lamC/Cwt).* ...
#            (DcLC'*(DELC.*quadwts) + DEcLC'*(LC.*quadwts)) + ...
#            (lamT/Twt).* ...
#            (DcLT'*(DELT.*quadwts) + DEcLT'*(LT.*quadwts))
      DcLC.. <- (crossprod(DcLC, DELC*quadwts) +
                  crossprod(DEcLC, LC*quadwts))
      DcLT.. <- (crossprod(DcLT, DELT*quadwts) +
                  crossprod(DEcLT, LT*quadwts))
      D2GDcE[ind1] <- ((lamC/Cwt)* DcLC.. + (lamT/Twt)* DcLT..)
#        D2GDcE(ind2,1) <- (lamC./Cwt).* ...
#            (DtLC'*(DELC.*quadwts) + DEtLC'*(LC.*quadwts)) + ...
#            (lamT./Twt).* ...
#            (DtLT'*(DELT.*quadwts) + DEtLT'*(LT.*quadwts))
      DtLC.. <- (crossprod(DtLC, DELC*quadwts) +
                  crossprod(DEtLC, LC*quadwts))
      DtLT.. <- (crossprod(DtLT, DELT*quadwts) +
                  crossprod(DEtLT, LT*quadwts))
      D2GDcE[ind2] <- ((lamC/Cwt)* DtLC.. + (lamT/Twt)* DtLT..)
#        D2GDc <- [D2GDc, D2GDcE]
      D2GDc.$EoverR <- D2GDcE
#
#    end EoverR
    }

#    %  a
    if(estimate[3]){
#
#        %  first derivative of L values
#
#        DhbetaTT <- (betaTT./aFc2b).*(1 - K1.*K2.*betaTT./Fc)
#        DabetaTT <- DhbetaTT.*Fc2b
      DhbetaTT <- ((betaTT/aFc2b)*(1 - K1*K2*betaTT/Fc))
      DabetaTT <- DhbetaTT*Fc2b
#
      DaLT <- (DabetaTT*(That - Tc))
#
      DatLT <- outer(DabetaTT, onesb)*quadmat
#
#        D2GDca <- zeros(2*nbasis,1)
      D2GDca <- rep(0, 2*nbasis)
#
#        D2GDca(ind1,1) <- (lamT/Twt).*(DcLT'*(DaLT.*quadwts))
#        D2GDca(ind2,1) <- (lamT./Twt).* ...
#            (DtLT'*(DaLT.*quadwts) + DatLT'*(LT.*quadwts))
      D2GDca[ind1] <- ((lamT/Twt)*(t(DcLT) %*%(DaLT*quadwts)))
      D2GDca[ind2] <- ((lamT/Twt)*
            (t(DtLT)%*%(DaLT*quadwts) + t(DatLT)%*%(LT*quadwts)) )
#
#        D2GDc <- [D2GDc, D2GDca]
      D2GDc.$a <- D2GDca
#
#    end 'a'
    }

    if( estimate[4]){
#
#        %  b derivative of L values
#
#        DhbetaTT <- (betaTT./aFc2b).*(1 - K1.*K2.*betaTT./Fc)
#        DbbetaTT <- DhbetaTT.*b.*aFc2b./Fc
      DhbetaTT <- (betaTT/aFc2b)*(1 - K1*K2*betaTT/Fc)
      DbbetaTT <- DhbetaTT*b*aFc2b/Fc
#
      DbLT <- DbbetaTT*(That - Tc)
#
#        DbtLT <- (DbbetaTT*onesb).*quadmat
      DbtLT <- outer(DbbetaTT, onesb)*quadmat
#
#        D2GDcb <- zeros(2*nbasis,1)
#        D2GDcb(ind1,1) <- (lamT/Twt).*(DcLT'*(DbLT.*quadwts))
#        D2GDcb(ind2,1) <- (lamT./Twt).* ...
#            (DtLT'*(DbLT.*quadwts) + DbtLT'*(LT.*quadwts))
#
      D2GDcb <- rep(0, 2*nbasis)
      D2GDcb[ind1] <- ((lamT/Twt)*(t(DcLT)%*%(DbLT*quadwts)) )
      D2GDcb[ind2] <- ((lamT/Twt)*
            (t(DtLT)%*%(DbLT*quadwts) + t(DbtLT)%*%(LT*quadwts)) )
#
#        D2GDc <- [D2GDc, D2GDcb]
      D2GDc.$b <- D2GDcb
#
#    end 'b'
    }
#   Convert from a list to a matrix,
#   dropping columns not in estimate
    D2GDc <- do.call(cbind, D2GDc.)
##
## 8.  Construct D2GDc2
##
#  8.1.  First part
#    Wmat <- [quadwtsmat, quadwtsmat; quadwtsmat, quadwtsmat]
    W.5 <- cbind(quadwtsmat, quadwtsmat)
    Wmat <- rbind(W.5, W.5)
#
#    D2GDc2  <- (Jacobian.*Wmat)'*Jacobian
    D2GDc2  <- crossprod(Jacobian*Wmat, Jacobian)
#
#    ZtZmat <- phimat'*phimat
    ZtZmat <- crossprod(phimat)

    if( fit[1])
        D2GDc2[ind1,ind1] <- D2GDc2[ind1,ind1] + ZtZmat/Cwt
#    end
    if( fit[2])
        D2GDc2[ind2,ind2] <- D2GDc2[ind2,ind2] + ZtZmat/Twt
#    end

#  8.2.  Add second derivative information

    DttbetaCC <- ((1e4*kref*EoverR/That^2)*
                 (1e4*EoverR/That^2 - 2/That)*temp)
    DttbetaTC <- TCfac*DttbetaCC
#
#    DctLC <- sparse(zeros(nbasis,nbasis))
#    DttLC <- sparse(zeros(nbasis,nbasis))
#    DctLT <- sparse(zeros(nbasis,nbasis))
#    DttLT <- sparse(zeros(nbasis,nbasis))
    DctLC <- array(0, dim=c(nbasis, nbasis))
    DttLT <- DctLT <- DttLC <- DctLC

#    norder <- nbasis - length(getbasispar(CSTRbasis))
## *** 'getbasispar' <- interior knots
    norder <- nbasis - length(CSTRbasis$params)
#
    for( i in 1:nbasis){
      jstart <- max(c(1,i-norder+1))
      for( j in jstart:i) {
        qijvec <- quadmat[,i]*quadmat[,j]*quadwts
        DctLC[i,j] <- sum(qijvec*LC*DtbetaCC)
        DttLC[i,j] <- sum(qijvec*LC*DttbetaCC*Chat)
        DctLT[i,j] <- sum(qijvec*LT*DtbetaTC)
        DttLT[i,j] <- sum(qijvec*LT*DttbetaTC*Chat)
        if( i != j){
          DctLC[j,i] <- DctLC[i,j]
          DttLC[j,i] <- DttLC[i,j]
          DctLT[j,i] <- DctLT[i,j]
          DttLT[j,i] <- DttLT[i,j]
#       end if(i!=j)
        }
#    end for(j in jstart:i)
      }
#   end for(i in 1:nbasis)
    }
#    DctL <- lamC.*DctLC./Cwt + lamT.*DctLT./Twt
    DctL <- lamC*DctLC/Cwt + lamT*DctLT/Twt
#      DttL <- lamC.*DttLC./Cwt + lamT.*DttLT./Twt
    DttL <- lamC*DttLC/Cwt + lamT*DttLT/Twt

#  8.3.  modify D2GDc2

    D2GDc2[ind1,ind2] <- D2GDc2[ind1,ind2] + DctL
    D2GDc2[ind2,ind1] <- D2GDc2[ind2,ind1] + t(DctL)
    D2GDc2[ind2,ind2] <- D2GDc2[ind2,ind2] + DttL

#  8.4.  compute (D2GDc2)^{-1} D2GDc

#    DcDtheta <- D2GDc2\D2GDc
    DcDtheta <- try(solve(D2GDc2, D2GDc))
    if(inherits(DcDtheta,"try-error")) {
#
      D2GDc2.eig <- eigen(D2GDc2, symmetric=TRUE)
      Dc.ev <- D2GDc2.eig$values
      Dc.rank <- sum(Dc.ev>0)
#
      if(Dc.rank<ncoef)
        warning("D2GDc2 has reduced rank ", Dc.rank, "; using ginverse.")
      Dc.rank1 <- sum(Dc.ev > (eps*Dc.ev[1]))
      if(Dc.rank1 < Dc.rank)
        warning("D2GDc2 is ill conditioned.  Reducing rank to ",
                Dc.rank1, " from ", Dc.rank)
      jrank <- 1:Dc.rank1
      DcDtheta <- with(D2GDc2.eig, (vectors[, jrank] /
          rep(Dc.ev[jrank], each=ncoef)) %*% crossprod(vectors[, jrank], D2GDc))
    }
#
#  8.5.  set up Dres
#
#    Dres <- []
    Dres <- NULL
#
    if( fit[1])
      Dres <- phimat%*%DcDtheta[ind1,]/sqrt(Cwt)
#       end
    if( fit[2])
      Dres <- rbind(Dres, phimat%*%DcDtheta[ind2,]/sqrt(Twt))
  }
##
## 9.  Done
##
#  list(res=res, Dres=Dres, fitstruct=fitstruct, df=df, lambda=lambda, gradwrd=gradwrd)
  list(res=res, Dres=Dres, fitstruct=fitstruct, df=df., gcv=gcv)
}

CSTRsse <- function(par, datstruct, fitstruct, CSTRbasis, lambda){
#%  2007.07.06 by Spencer Graves
#
  estimate = fitstruct$estimate;
  if(is.null(estimate))
    stop("fitstruct has no 'estimate' component")
#  Compare estimate with par
  if(sum(estimate) != length(par)){
    cat("ERROR in CSTRfn:  sum(estimate) != length(par)\n")
    cat("par = ", par, "; \n")
    stop("fitstruct$estimate = ",
         paste(estimate, collapse=" "))
  }
  m <- 0
  {
#  1.1.  Estimate kref starting from par or use fitstruct?
    if(estimate[1]){
      m <- m+1
      kref <- par[m]
      fitstruct$kref <- kref
    }
    else
      kref <- fitstruct$kref
  }
  {
#  1.2.  Estimate EoverR starting from parvec or use fitstruct?
    if(estimate[2]){
      m <- m+1
      EoverR <- par[m]
      fitstruct$EoverR <- EoverR
    }
    else
      EoverR<- fitstruct$EoverR
  }
  {
#  1.3.  Estimate 'a' starting from parvec or use fitstruct?
    if(estimate[3]){
      m <- m+1
      a <- par[m]
      fitstruct$a <- a
    }
    else
      a <- fitstruct$a
  }
  {
#  1.4.  Estimate b starting from parvec or use fitstruct?
    if(estimate[4]){
      m <- m+1
      b <- par[m]
      fitstruct$b <- b
    }
    else
      b<- fitstruct$b
  }
#
  res <- CSTRres(kref=kref, EoverR=EoverR, a=a, b=b,
               datstruct=datstruct, fitstruct=fitstruct,
               CSTRbasis=CSTRbasis, lambda=lambda,
               gradwrd=FALSE)
  sse <- sum(res^2)
  if(is.na(sse)){
    parch <- paste(par, collapse=", ")
    warning('Missing values returned by CSTRres with par = ',
        parch, '; ', sum(is.na(res)), ' NAs out of ', length(res))
  }
  sse
}

CSTRres <- function(kref=NULL, EoverR=NULL, a=NULL, b=NULL,
               datstruct, fitstruct, CSTRbasis, lambda,
               gradwrd=FALSE){
#%  2007.06.02 by Spencer Graves
##
## 1.  Construct 'parvec'
##
#  parvec <- vector("list", 4)
#  names(parvec) <- c("kref", "EoverR", "a", "b")
#  parvec$kref  <- kref
#  parvec$Eover <- EoverR
#  parvec$a     <- a
#  parvec$b     <- b
  parvec <- list(kref=kref, EoverR=EoverR, a=a, b=b)
  pv     <- unlist(parvec)
##
## 2.  Call CSTRfn
##
  cstr. <- CSTRfn(parvec=pv, datstruct=datstruct, fitstruct=fitstruct,
                  CSTRbasis=CSTRbasis, lambda=lambda, gradwrd=gradwrd)
  Res <- as.vector(cstr.$res)
  if(length(d.r <- dim(Res))>1){
    ys <- dimnames(Res)[[2]]
    if(!is.null(ys)){
      resNames <- t(outer(ys, 1:d.r[1], paste, sep=""))
      Res      <- as.vector(Res)
      names(Res) <- resNames
    }
  }
  if(gradwrd)
    attr(Res, "gradient") <- cstr.$Dres
#
  Res
}
cumfd <- function(xrnd, xrng, nbreaks=7, nfine=101) {
  #  Compute cdf_fd over a closed interval using smooth.morph.  
  #  Only the values of x within the interior of xrng are used 
  #  in order to avoid distortion due to boundary inflation.
  #  Arguments:
  #  xrnd    ... A vector of variable values (unsorted)
  #  xrng    ... A vector of length 2 containing the boundary values.
  #  Wnbasis ... Number of basis functions used by smooth.morph.
  #  Snbasis ... Number of basis functions used by smooth.basis.
  #  nfine   ... Length of vector of equally spaced values spanning xrng.
  
  #  Last modified 25 March 2022 by Jim Ramsay
  
  #  check that values of x are within xrng
  
  if (min(xrnd) < xrng[1] || max(xrnd) > xrng[2]) 
    stop("Values of x outside of xrng.")
  
  #  sort the data and set up probability values
  
  xsort  <- sort(xrnd[xrnd > xrng[1] & xrnd < xrng[2]])
  N      <- length(xsort)
  prbvec <- (1:N)/(N+1)
  
  #  add boundary values
  
  pmesh <- c(0, prbvec, 1)
  xmesh <- c(xrng[1], xsort,  xrng[2])
  
  #  set up fdPar object for smooth.morph
  
  index = c(1, round(N*2:(nbreaks-1)/nbreaks), N+2)
  
  Wnorder <- 4
  Wnbasis <- nbreaks + Wnorder - 2
  Wbreaks <- xmesh[index]
  Wbasis  <- create.bspline.basis(xrng, Wnbasis, Wnorder, Wbreaks)
  WfdPar  <- fdPar(fd(matrix(0,Wnbasis,1), Wbasis))
  
  #  use smooth.morph to map sorted data into the interior of [0,1]
  
  result  <- smooth.morph(xmesh, pmesh, c(0,1), WfdPar)
  xfine   <- seq(0,1,len=101)
  Wfdobj  <- result$Wfdobj
  
  cdffine <- result$hfine
  cdffine[1]               <- 0
  cdffine[length(cdffine)] <- 1
  
  # plot(xfine, cdffine, type="l")
  # points(xmesh, pmesh)
  
  # plot(Wfdobj)
  
  return(list(Wfdobj=Wfdobj, cdffine=cdffine))
  
}

cycleplot.fd <- function(fdobj, matplt = TRUE, nx = 201, ...)
{

#  Performs a cycle plot of a functional data object FDOBJ,
#   assuming that FD is a bivariate function...the first component
#   of which is the x-coordinate and the second the y-coordinate
#
#  If MATPLT is TRUE, matplot is used to plot all curves in
#     a single plot.
#  Otherwise, each curve is plotted separately, and the
#     next curve is plotted when the mouse is clicked.
#  NX is the number of sampling points to use (default 128)


 #  Last modified 20 November 2005

  if (!inherits(fdobj, "fd")) stop(
     "First argument is not a functional data object.")

  coef  <- fdobj$coefs
  coefd <- dim(coef)
  ndim  <- length(coefd)
  if(ndim < 3) stop("Univariate functions cannot be cycle plotted")
  nbasis <- coefd[1]
  nrep   <- coefd[2]
  nvar   <- coefd[3]
  if(nvar > 2) warning("Only first two functions used")
  basisobj <- fdobj$basis
  crvnames <- dimnames(coef)[[2]]
  varnames <- dimnames(coef)[[3]][1:2]
  rangex   <- basisobj$rangeval
  x        <- seq(rangex[1], rangex[2], length = nx)
  fdmat    <- eval.fd(x, fdobj)
  fdnames  <- fdobj$fdnames
  crvnames <- fdnames[[2]]
  varnames <- fdnames[[3]]
  if(matplt) {
    matplot(fdmat[,  , 1], fdmat[,  , 2], type = "l", lty = 1,
            xlab=varnames[1], ylab=varnames[2], ...)
  }
  if(!matplt) {
    for (irep in 1:nrep) {
      plot(fdmat[, irep, 1], fdmat[, irep, 2], type = "l",
         lty = 1, xlab=varnames[1], ylab=varnames[2],
         main = paste("Curve", irep, crvnames[irep]), ...)
      mtext("Click to advance to next plot", side = 3,
                  line = -3, outer = TRUE)
      text(locator(1), "")
    }
  }
  invisible()
}
Data2fd <- function(argvals=NULL, y=NULL, basisobj=NULL, nderiv=NULL,
                    lambda=3e-8/diff(as.numeric(range(argvals))),
                    fdnames=NULL, covariates=NULL, method="chol",
                    dfscale=1)
{
  
  argChk <- argvalsySwap(argvals, y, basisobj)
  
  # Change proposed by Spencer Graves 2010.12.08
  # if(is.null(lambda))
  #   lambda <- 1e-9*sd(argChk$y)/diff(range(argChk$argvals))
  #
  # 2020-01-16: 
  # Error in smooth.basis ... argvals is not numeric
  # in R CMD check, cannot replicate line by line.  
  if(!is.numeric(AV <- argChk$argvals)){
    if(is.null(AV))
      stop('is.null(argChk$argvals); should be numeric')
    cat('argChk$argvals is not numeric.\n')
    cat('class(argChk$argvals) = ', class(AV), '\n')
    print(AV)
  }  
  
  smBasis <- smooth.basisPar(argChk$argvals, argChk$y,
                            fdobj=basisobj, Lfdobj=nderiv, lambda=lambda,
                            fdnames=fdnames,
                            covariates=covariates, method="chol", dfscale=dfscale)
  
  # smBasis <- with(argChk, smooth.basisPar(argvals=argvals, y=y,
  #                                         fdobj=basisobj, Lfdobj=nderiv, lambda=lambda,
  #                                         fdnames=fdnames,
  #                                         covariates=covariates, method="chol", dfscale=dfscale) )
  #
  smBasis$fd
}

#  -------------------------------------------------------------------------

## 2020-01-16:  Spencer Graves makes argvalsySwap 
## an internal function
argvalsySwap = function(argvals=NULL, y=NULL, basisobj=NULL)
{
  ##
  ## 1.  if(is.null(y)) ...
  ##
  if(is.null(y)){
    if(is.null(argvals)) stop("'y' is missing with no default")
    #   Store argvals as y
    cat("'y' is missing, using 'argvals'\n") 
    y <- argvals
    argvals <- NULL 
  }
  ##
  ## 2.  if(is.null(argvals))argvals <- seq(basisobj$rangeval, dim(y)[1])
  ##
  dimy <- dim(as.array(y))
  if(is.null(argvals)){
    {
      if(is.null(basisobj)){
        basisobj <- create.bspline.basis(basisobj)
      } else {
        if(is.numeric(basisobj)) {
          if(length(basisobj)>1){
            basisobj <- create.bspline.basis(basisobj)
          } else 
            basisobj <- create.bspline.basis(norder=basisobj)
        }
        else {
          if(inherits(basisobj, 'fd')){
            basisobj <- basisobj$basis
          } else 
            if(inherits(basisobj, 'fdPar'))
              basisobj <- basisobj$fd$basis
        }
      }
    }
    a01 <- basisobj$rangeval
    if(is.null(a01))
      stop('basisobj does not have a required ',
           'rangeval component.')
    #    
    n <- dimy[1]
    cat(paste("'argvals' is missing;  using seq(", a01[1],
              ", ", a01[2], ", length=", n, ")\n"))       
    argvals <- seq(a01[1], a01[2], length=n)
    return(list(argvals=argvals, y=y, basisobj=basisobj))
  }
  ##
  ## 3.  if(length(dim(argvals)) == length(dim(y))) ... 
  ##
  dima <- dim(as.array(argvals))
  {
    if(length(dimy) == length(dima)){
      if(any(dimy != dima))
        stop("dimensions of 'argvals' and 'y' must be compatible;\n",
             "  dim(argvals) = ", paste(dima, collapse=' x '),
             ";  dim(y) = ", paste(dimy, collapse=' x ') )
      #     Check basisobj
      {
        if(inherits(basisobj, 'fd'))basisobj <- basisobj$basis
        else {
          if(inherits(basisobj, 'fdPar'))
            basisobj <- basisobj$fd$basis
          else {
            if(inherits(basisobj, 'array')){
              fd. <- fd(basisobj)
              basisobj <- fd.$basis
            }
            else { 
              if(inherits(basisobj, 'integer'))
                basisobj <- create.bspline.basis(argvals, norder=basisobj)
              else {
                if(is.null(basisobj))
                  basisobj <- create.bspline.basis(argvals)
                else
                  if(!inherits(basisobj, 'basisfd'))
                    stop("'basisobj' is NOT a functional basis",
                         " object (class 'basisfd');  class = ",
                         class(basisobj)[1])
              }
            }
          }
        }
        }
      a01 <- basisobj$rangeval
      arng <- range(argvals)
      if ((a01[1]<=arng[1]) && (arng[2]<=a01[2])) {
        return(list(argvals=argvals, y=y, basisobj=basisobj))
      }
      #
      yrng <- range(y)
      if((a01[1]<=yrng[1]) && (yrng[2]<=a01[2])) {
        cat(paste("'argvals' is NOT contained in basisobj$rangeval",
                  ", but 'y' is;  swapping 'argvals' and 'y'.\n"))
        return(list(argvals=y, y=argvals, basisobj=basisobj)) 
      }
      #      
      stop("Neither 'argvals' nor 'y' are contained in ",
           "basisobj$rangeval")
    }
  }        
  ##
  ## 4.  If(length(dimy) < length(dima)) swap ...
  ##
  if(length(dimy)<length(dima)){
    cat(paste("Swapping 'y' and 'argvals', because 'y' is ",
              "simpler,\n  and 'argvals' should be;  now ",
              "dim(argvals) = ", paste(dimy, collapse=" x "),
              ";  dim(y) = ", paste(dima, collapse=" x "),"\n" )) 
    y. <- argvals
    argvals <- y
    y <- y.
    #
    d. <- dima
    dima <- dimy
    dimy <- d.
  }   
  #
  if(any(dima != dimy[1:length(dima)]))
    stop("A dimension of 'argvals' does not match 'y':\n",
         "  dim(argvals) = ", paste(dima, collapse=" x "),
         ";  dim(y) = ", paste(dimy, collapse=" x ") )      
  ##        
  ## 5.  Check compatibility of argvals with basisobj
  ##        
  {
    if(inherits(basisobj, 'fd'))basisobj <- basisobj$basis
    else {
      if(inherits(basisobj, 'fdPar'))
        basisobj <- basisobj$fd$basis
      else {
        if(inherits(basisobj, 'array')){
          fd. <- fd(basisobj)
          basisobj <- fd.$basis
        }
        else { 
          if(inherits(basisobj, 'integer'))
            basisobj <- create.bspline.basis(argvals, norder=basisobj)
          else {
            if(is.null(basisobj))
              basisobj <- create.bspline.basis(argvals)
            else
              if(!inherits(basisobj, 'basisfd'))
                stop("'basisobj' is NOT a functional basis",
                     " object (class 'basisfd');  class = ",
                     class(basisobj)[1])
          }
        }
      }
    }
    }
  a01 <- basisobj$rangeval
  arng <- range(argvals)
  if((a01[1]<=arng[1]) && (arng[2]<=a01[2])) {
    return(list(argvals=argvals, y=y, basisobj=basisobj))
  }
  #
  stop("'argvals' are not contained in basisobj$rangeval")
}

density <- function(x, ...)UseMethod('density')

density.fd <- function(x, WfdParobj, conv=0.0001, iterlim=20,
                      active=1:nbasis, dbglev=0, ...) {
# DENSITYFD estimates the density of a sample of scalar observations.

#  These observations may be one of two forms:
#   1.  a vector of observatons x_i
#   2.  a two-column matrix, with the observations x_i in the
#       first column, and frequencies f_i in the second.
#   Option 1. corresponds to all f_i = 1.

#  Arguments are:
#  X         ... data value array, either a vector or a two-column
#                matrix.
#  WFDPAROBJ ... functional parameter object specifying the initial log
#              density, the linear differential operator used to smooth
#              smooth it, and the smoothing parameter.
#  CONV      ... convergence criterion
#  ITERLIM   ... iteration limit for scoring iterations
#  ACTIVE    ... indices among 1:NBASIS of parameters to optimize
#  DBGLEV    ... level of output of computation history

#  Returns:
#  A list containing
#  WFDOBJ ...   functional data basis object defining final density
#  C      ...   normalizing constant for density p = exp(WFDOBJ)/C
#  FLIST  ...   Struct object containing
#               FSTR$f     final log likelihood
#               FSTR$norm  final norm of gradient
#  ITERNUM   Number of iterations
#  ITERHIST  History of iterations

#  To plot the density function or to evaluate it, evaluate WFDOBJ,
#  exponentiate the resulting vector, and then divide by the normalizing
#  constant C.

# last modified 3 July 2020 by Jim Ramsay

#  check WfdParobj

if (!inherits(WfdParobj, "fdPar")) {
	if (inherits(WfdParobj, "fd") || inherits(WfdParobj, "basisfd")) {
	    WfdParobj <- fdPar(WfdParobj)
	} else {
          stop("WFDPAROBJ is not a fdPar object")
      }
}

#  set up WFDOBJ

Wfdobj   <- WfdParobj$fd

#  set up LFDOBJ

Lfdobj <- WfdParobj$Lfd
Lfdobj <- int2Lfd(Lfdobj)

#  set up BASIS

basisobj <- Wfdobj$basis
nbasis   <- basisobj$nbasis
rangex   <- basisobj$rangeval

x    <- as.matrix(x)
xdim <- dim(x)
N    <- xdim[1]
m    <- xdim[2]

if (m > 2 && N > 2)
    	stop("Argument X must have either one or two columns.")

if ((N == 1 | N == 2) & m > 1) {
    x <- t(x)
    n <- N
    N <- m
    m <- n
}

if (m == 1) {
    f <- rep(1,N)
} else {
    f    <- x[,2]
    fsum <- sum(f)
    f    <- f/fsum
    x    <- x[,1]
}
f = as.matrix(f)

#  check for values outside of the range of WFD0

inrng <- (1:N)[x >= rangex[1] & x <= rangex[2]]
if (length(inrng) != N) {
    print(c(length(inrng), N))
    print(c(rangex[1], rangex[2], min(x), max(x)))
    warning("Some values in X out of range and not used.")
}
x     <- x[inrng]
f     <- f[inrng]
nobs  <- length(x)

#  set up some arrays

climit <- c(rep(-50,nbasis),rep(400,nbasis))
cvec0  <- Wfdobj$coefs
dbgwrd <- dbglev > 1

zeromat <- zerobasis(nbasis)

#  initialize matrix Kmat defining penalty term

lambda <- WfdParobj$lambda
if (lambda > 0) Kmat <- lambda*getbasispenalty(basisobj, Lfdobj)

#  evaluate log likelihood
#    and its derivatives with respect to these coefficients

result <- loglfnden(x, f, basisobj, cvec0)
logl   <- result[[1]]
Dlogl  <- result[[2]]

#  compute initial badness of fit measures

fun  <- -logl
gvec <- -Dlogl
if (lambda > 0) {
   gvec <- gvec + 2*(Kmat %*% cvec0)
   fun    <- fun + t(cvec0) %*% Kmat %*% cvec0
}
Foldstr <- list(f = fun, norm = sqrt(mean(gvec^2)))
gvec0 <- t(zeromat) %*% as.matrix(gvec)

#  compute the initial expected Hessian

hmat <- Varfnden(x, basisobj, cvec0)
if (lambda > 0) hmat <- hmat + 2*Kmat
hmat0 = t(zeromat) %*% hmat %*% zeromat

#  evaluate the initial update vector for correcting the initial bmat

deltac0  <- -solve(hmat0,gvec0)
deltac   <- zeromat %*% as.matrix(deltac0)
cosangle <- -sum(gvec0*deltac0)/sqrt(sum(gvec0^2)*sum(deltac0^2))

#  initialize iteration status arrays

iternum <- 0
status <- c(iternum, Foldstr$f, -logl, Foldstr$norm)
if (dbglev > 0) {
  cat("Iteration  Criterion  Neg. Log L  Grad. Norm\n")
  cat("      ")
  cat(format(iternum))
  cat("    ")
  cat(format(status[2:4]))
  cat("\n")
}
iterhist <- matrix(0,iterlim+1,length(status))
iterhist[1,]  <- status

#  quit if ITERLIM == 0

if (iterlim == 0) {
    Flist     <- Foldstr
    iterhist <- iterhist[1,]
    C        <- normden.phi(basisobj, cvec0)
    return( list(Wfdobj=Wfdobj, C=C, Flist=Flist, iternum=iternum,
                   iterhist=iterhist) )
}

#  -------  Begin iterations  -----------

STEPMAX <- 5
MAXSTEP <- 400
trial   <- 1
cvec    <- cvec0
linemat <- matrix(0,3,5)

for (iter in 1:iterlim) {
   	iternum <- iternum + 1
  	#  take optimal stepsize
	  dblwrd <- rep(FALSE,2)
	  limwrd <- rep(FALSE,2)
  	stpwrd <- 0
	  ind    <- 0
	  #  compute slope
      Flist <- Foldstr
      linemat[2,1] <- sum(deltac*gvec)
      #  normalize search direction vector
      sdg     <- sqrt(sum(deltac^2))
      deltac  <- deltac/sdg
      dgsum   <- sum(deltac)
      linemat[2,1] <- linemat[2,1]/sdg
      #  return with stop condition if (initial slope is nonnegative
      if (linemat[2,1] >= 0) {
        print("Initial slope nonnegative.")
        ind <- 3
        iterhist <- iterhist[1:(iternum+1),]
        break
      }
      #  return successfully if (initial slope is very small
      if (linemat[2,1] >= -1e-5) {
        if (dbglev>1) print("Initial slope too small")
        iterhist <- iterhist[1:(iternum+1),]
        break
      }
    	#  load up initial search matrix
      linemat[1,1:4] <- 0
      linemat[2,1:4] <- linemat[2,1]
      linemat[3,1:4] <- Foldstr$f
     	#  output initial results for stepsize 0
      stepiter  <- 0
      if (dbglev > 1) {
	      cat("              ")
	      cat(format(stepiter))
	      cat(format(linemat[,1]))
	      cat("\n")
	    }
      ips <- 0
      #  first step set to trial
      linemat[1,5]  <- trial
      #  Main iteration loop for linesrch
      for (stepiter in 1:STEPMAX) {
        #  ensure that step does not go beyond limits on parameters
        limflg  <- 0
        #  check the step size
        result <- stepchk(linemat[1,5], cvec, deltac, limwrd, ind,
                            climit, active, dbgwrd)
	      linemat[1,5] <- result[[1]]
	      ind          <- result[[2]]
	      limwrd       <- result[[3]]
        if (linemat[1,5] <= 1e-9) {
          	#  Current step size too small  terminate
          	Flist   <- Foldstr
          	cvecnew <- cvec
          	gvecnew <- gvec
          	if (dbglev > 1) print(paste("Stepsize too small:", linemat[1,5]))
          	if (limflg) ind <- 1 else ind <- 4
          	break
        }
        cvecnew <- cvec + linemat[1,5]*deltac
        #  compute new function value and gradient
	      result  <- loglfnden(x, f, basisobj, cvecnew)
	      logl    <- result[[1]]
	      Dlogl   <- result[[2]]
        Flist$f <- -logl
        gvecnew <- -as.matrix(Dlogl)
        if (lambda > 0) {
            gvecnew <- gvecnew + 2*Kmat %*% cvecnew
            Flist$f <- Flist$f + t(cvecnew) %*% Kmat %*% cvecnew
        }
        gvecnew0 <- t(zeromat) %*% gvecnew
        Flist$norm <- sqrt(mean(gvecnew0^2))
        #  compute new directional derivative
        linemat[2,5] <- sum(deltac*gvecnew)
        linemat[3,5] <- Flist$f
        if (dbglev > 1) {
           cat("              ")
           cat(format(stepiter))
	         cat(format(linemat[,5]))
	         cat("\n")
	      }
        #  compute next step
	      result  <- stepit(linemat, ips, dblwrd, MAXSTEP)
	      linemat <- result$linemat
	      ips     <- result$ips
	      ind     <- result$ind
	      dblwrd  <- result$dblwrd
        trial   <- linemat[1,5]
        #  ind == 0 implies convergence
        if (ind == 0 | ind == 5) break
        #  end of line search loop
     	}

    	#  update current parameter vectors

     	cvec  <- cvecnew
     	gvec  <- gvecnew
      gvec0 <- t(zeromat) %*% as.matrix(gvec)
	    Wfdobj$coefs <- cvec
     	status <- c(iternum, Flist$f, -logl, Flist$norm)
     	iterhist[iter+1,] <- status
     	if (dbglev > 0) {
	      cat("      ")
	      cat(format(iternum))
	      cat("    ")
	      cat(format(status[2:4]))
	      cat("\n")
     	}

     	#  test for convergence

     	if (abs(Flist$f-Foldstr$f) < conv) {
          iterhist <- iterhist[1:(iternum+1),]
  	      C <- normden.phi(basisobj, cvec)
	        denslist <- list("Wfdobj" = Wfdobj, "C" = C, "Flist" = Flist,
			                     "iternum" = iternum, "iterhist" = iterhist)
	        return( denslist )
     	}
     	if (Flist$f >= Foldstr$f) break
     	#  compute the Hessian
     	hmat <- Varfnden(x, basisobj, cvec)
     	if (lambda > 0) hmat <- hmat + 2*Kmat
      hmat0 <- t(zeromat) %*% hmat %*% zeromat
     	#  evaluate the update vector
     	deltac0 <- -solve(hmat0,gvec0)
     	cosangle  <- -sum(gvec0*deltac0)/sqrt(sum(gvec0^2)*sum(deltac0^2))
     	if (cosangle < 0) {
       	if (dbglev > 1) print("cos(angle) negative")
       	deltac0 <- -gvec0
     	}
      deltac <- zeromat %*% as.matrix(deltac0)
     	Foldstr <- Flist
		#  end of iterations
  	}
      #  compute final normalizing constant
 	C <- normden.phi(basisobj, cvec)
	denslist <- list("Wfdobj" = Wfdobj, "C" = C, "Flist" = Flist,
			             "iternum" = iternum, "iterhist" = iterhist)
 	return( denslist )
}

#  -----------------------------------------------------------------------------

loglfnden <- function(x, f, basisobj, cvec=FALSE) {
	#  Computes the log likelihood and its derivative with
	#    respect to the coefficients in CVEC
   	N       <- length(x)
   	nbasis  <- basisobj$nbasis
   	fmat    <- outer(f, rep(1,nbasis))
   	fsum    <- sum(f)
   	nobs    <- length(x)
   	phimat  <- getbasismatrix(x, basisobj)
   	Cval    <- normden.phi(basisobj, cvec, )
   	logl    <- sum((phimat %*% cvec) * f - fsum*log(Cval)/N)
    EDw     <- expectden.phi(basisobj, cvec, Cval)
   	Dlogl   <- apply((phimat - outer(rep(1,nobs),EDw))*fmat,2,sum)
	return( list(logl, Dlogl) )
}

#  -----------------------------------------------------------------------------

Varfnden <- function(x, basisobj, cvec=FALSE) {
	#  Computes the expected Hessian
   	nbasis  <- basisobj$nbasis
   	nobs    <- length(x)
   	Cval    <- normden.phi(basisobj, cvec)
   	EDw     <- expectden.phi(basisobj, cvec, Cval)
   	EDwDwt  <- expectden.phiphit(basisobj, cvec, Cval)
   	Varphi  <- nobs*(EDwDwt - outer(EDw,EDw))
	return(Varphi)
}

#  -----------------------------------------------------------------------------

normden.phi <- function(basisobj, cvec, JMAX=15, EPS=1e-7) {

#  Computes integrals of
#      p(x) = exp phi"(x) %*% cvec
#  by numerical integration using Romberg integration

  	#  check arguments, and convert basis objects to functional data objects

  	if (!inherits(basisobj, "basisfd") )
    	stop("First argument must be a basis function object.")

	  nbasis <- basisobj$nbasis
  	oneb   <- matrix(1,1,nbasis)
  	rng    <- basisobj$rangeval

  	#  set up first iteration

  	width <- rng[2] - rng[1]
  	JMAXP <- JMAX + 1
  	h <- matrix(1,JMAXP,1)
  	h[2] <- 0.25
  	#  matrix SMAT contains history of discrete approximations to the integral
  	smat <- matrix(0,JMAXP,1)
  	#  the first iteration uses just the }points
  	x  <- rng
  	nx <- length(x)
  	ox <- matrix(1,nx,1)
  	fx <- getbasismatrix(x, basisobj)
  	wx <- fx %*% cvec
  	wx[wx < -50] <- -50
  	px <- exp(wx)
  	smat <- matrix(0,JMAXP,1)
  	smat[1]  <- width*sum(px)/2
  	tnm <- 0.5
  	j   <- 1

  	#  now iterate to convergence
  	for (j in 2:JMAX) {
    	tnm  <- tnm*2
    	del  <- width/tnm
    	if (j == 2) {
      		x <- (rng[1] + rng[2])/2
    	} else {
      		x <- seq(rng[1]+del/2, rng[2], del)
    	}
    	fx <- getbasismatrix(x, basisobj)
    	wx <- fx %*% cvec
    	wx[wx < -50] <- -50
    	px <- exp(wx)
    	smat[j] <- (smat[j-1] + width*sum(px)/tnm)/2
    	if (j >= 5) {
      		ind <- (j-4):j
			result <- polintarray(h[ind],smat[ind],0)
			ss  <- result[[1]]
			dss <- result[[2]]
      		if (!any(abs(dss) >= EPS*max(abs(ss)))) {
        		#  successful convergence
        		return(ss)
      		}
    	}
    	smat[j+1] <- smat[j]
    	h[j+1]    <- 0.25*h[j]
 	}
  	warning(paste("No convergence after ",JMAX," steps in NORMDEN.PHI"))
	return(ss)
}

#  -----------------------------------------------------------------------------

expectden.phi <- function(basisobj, cvec, Cval=1, nderiv=0,
                     JMAX=15, EPS=1e-7) {
    #  Computes expectations of basis functions with respect to density
    #      p(x) <- Cval^{-1} exp t(c)*phi(x)
    #  by numerical integration using Romberg integration

    #  check arguments, and convert basis objects to functional data objects

    if (!inherits(basisobj, "basisfd"))
    	stop("First argument must be a basis function object.")

    nbasis <- basisobj$nbasis
    rng    <- basisobj$rangeval
    oneb   <- matrix(1,1,nbasis)

    #  set up first iteration

    width <- rng[2] - rng[1]
    JMAXP <- JMAX + 1
    h <- matrix(1,JMAXP,1)
    h[2] <- 0.25
    #  matrix SMAT contains the history of discrete approximations to the integral
    sumj <- matrix(0,1,nbasis)
    #  the first iteration uses just the }points
    x  <- rng
    nx <- length(x)
    ox <- matrix(1,nx,nx)
    fx <- getbasismatrix(x, basisobj, 0)
    wx <- fx %*% cvec
    wx[wx < -50] <- -50
    px <- exp(wx)/Cval
    if (nderiv == 0) {
    	Dfx <- fx
    } else {
    	Dfx <- getbasismatrix(x, basisobj, 1)
    }
    sumj <- t(Dfx) %*% px
    smat <- matrix(0,JMAXP,nbasis)
    smat[1,]  <- width*as.vector(sumj)/2
    tnm <- 0.5
    j   <- 1

    #  now iterate to convergence

    for (j in 2:JMAX) {
      tnm  <- tnm*2
    	del  <- width/tnm
    	if (j == 2) {
        x <- (rng[1] + rng[2])/2
    	} else {
        x <- seq(rng[1]+del/2, rng[2], del)
    	}
    	nx <- length(x)
    	fx <- getbasismatrix(x, basisobj, 0)
    	wx <- fx %*% cvec
    	wx[wx < -50] <- -50
    	px <- exp(wx)/Cval
    	if (nderiv == 0) {
        Dfx <- fx
    	} else {
        Dfx <- getbasismatrix(x, basisobj, 1)
    	}
    	sumj <- t(Dfx) %*% px
    	smat[j,] <- (smat[j-1,] + width*as.vector(sumj)/tnm)/2
    	if (j >= 5) {
        ind <- (j-4):j
        temp <- smat[ind,]
	      result <- polintarray(h[ind],temp,0)
	      ss  <- result[[1]]
	      dss <- result[[2]]
        if (!any(abs(dss) > EPS*max(abs(ss)))) {
          #  successful convergence
          return(ss)
        }
    	}
    	smat[j+1,] <- smat[j,]
    	h[j+1]     <- 0.25*h[j]
    }
    warning(paste("No convergence after ",JMAX," steps in EXPECTDEN.PHI"))
    return(ss)
}

#  -----------------------------------------------------------------------------

expectden.phiphit <- function(basisobj, cvec, Cval=1, nderiv1=0, nderiv2=0,
                              JMAX=15, EPS=1e-7) {

#  Computes expectations of cross product of basis functions with
#  respect to density
#      p(x) = Cval^{-1} int [exp t(c) %*% phi(x)] phi(x) t(phi(x)) dx
#  by numerical integration using Romberg integration

  	#  check arguments, and convert basis objects to functional data objects

  	if (!inherits(basisobj, "basisfd"))
    	stop("First argument must be a basis function object.")

  	nbasis <- basisobj$nbasis
  	rng    <- basisobj$rangeval
  	oneb   <- matrix(1,1,nbasis)

  	#  set up first iteration

  	width <- rng[2] - rng[1]
  	JMAXP <- JMAX + 1
  	h <- matrix(1,JMAXP,1)
  	h[2] <- 0.25
  	#  matrix SMAT contains history of discrete approximations to the integral
  	#  the first iteration uses just the }points
  	x  <- rng
  	nx <- length(x)
  	fx <- getbasismatrix(x, basisobj, 0)
  	wx <- fx %*% cvec
  	wx[wx < -50] <- -50
  	px <- exp(wx)/Cval
  	if (nderiv1 == 0) {
    	Dfx1 <- fx
  	} else {
    	Dfx1 <- getbasismatrix(x, basisobj, 1)
  	}
  	if (nderiv2 == 0) {
    	Dfx2 <- fx
  	} else {
    	Dfx2 <- getbasismatrix(x, basisobj, 2)
  	}
  	oneb <- matrix(1,1,nbasis)
  	sumj <- t(Dfx1) %*% ((px %*% oneb) * Dfx2)
  	smat <- array(0,c(JMAXP,nbasis,nbasis))
  	smat[1,,]  <- width*as.matrix(sumj)/2
  	tnm <- 0.5
  	j   <- 1

  	#  now iterate to convergence
  	for (j in 2:JMAX) {
    	tnm  <- tnm*2
    	del  <- width/tnm
    	if (j == 2) {
      		x <- (rng[1] + rng[2])/2
    	} else {
      		x <- seq(rng[1]+del/2, rng[2], del)
    	}
    	nx <- length(x)
    	fx <- getbasismatrix(x, basisobj, 0)
    	wx <- fx %*% cvec
    	wx[wx < -50] <- -50
    	px <- exp(wx)/Cval
    	if (nderiv1 == 0) {
      		Dfx1 <- fx
    	} else {
      		Dfx1 <- getbasismatrix(x, basisobj, 1)
    	}
    	if (nderiv2 == 0) {
      		Dfx2 <- fx
    	} else {
      		Dfx2 <- getbasismatrix(x, basisobj, 2)
    	}
    	sumj <- t(Dfx1) %*% ((px %*% oneb) * Dfx2)
    	smat[j,,] <- (smat[j-1,,] + width*as.matrix(sumj)/tnm)/2
    	if (j >= 5) {
      		ind <- (j-4):j
      		temp <- smat[ind,,]
	   		result <- polintarray(h[ind],temp,0)
	   		ss  <- result[[1]]
	   		dss <- result[[2]]
      		if (!any(abs(dss) > EPS*max(max(abs(ss))))) {
        		#  successful convergence
        		return(ss)
      		}
    	}
    	smat[j+1,,] <- smat[j,,]
    	h[j+1] <- 0.25*h[j]
  	}
  	warning(paste("No convergence after ",JMAX," steps in EXPECTDEN.PHIPHIT"))
	return(ss)
}

#  -----------------------------------------------------------------------------

polintarray <- function(xa, ya, x0) {
  	#  YA is an array with up to 4 dimensions
  	#     with 1st dim the same length same as the vector XA
  	n     <- length(xa)
  	yadim <- dim(ya)
  	if (is.null(yadim)) {
		yadim <- n
		nydim <- 1
  	} else {
    	nydim <- length(yadim)
  	}
  	if (yadim[1] != n) stop("First dimension of YA must match XA")
  	difx <- xa - x0
  	absxmxa <- abs(difx)
  	ns <- min((1:n)[absxmxa == min(absxmxa)])
  	cs <- ya
  	ds <- ya
  	if (nydim == 1) y <- ya[ns]
  	if (nydim == 2) y <- ya[ns,]
  	if (nydim == 3) y <- ya[ns,,]
  	if (nydim == 4) y <- ya[ns,,,]
  	ns <- ns - 1
  	for (m in 1:(n-1)) {
    	if (nydim == 1) {
      		for (i in 1:(n-m)) {
        		ho    <- difx[i]
        		hp    <- difx[i+m]
        		w     <- (cs[i+1] - ds[i])/(ho - hp)
        		ds[i] <- hp*w
        		cs[i] <- ho*w
      		}
      		if (2*ns < n-m) {
        		dy <- cs[ns+1]
      		} else {
        		dy <- ds[ns]
        		ns <- ns - 1
      		}
  		}
  		if (nydim == 2) {
      		for (i in 1:(n-m)) {
        		ho     <- difx[i]
        		hp     <- difx[i+m]
        		w      <- (cs[i+1,] - ds[i,])/(ho - hp)
        		ds[i,] <- hp*w
        		cs[i,] <- ho*w
      		}
      		if (2*ns < n-m) {
        		dy <- cs[ns+1,]
      		} else {
        		dy <- ds[ns,]
        		ns <- ns - 1
      		}
  		}
   		if (nydim == 3) {
      		for (i in 1:(n-m)) {
        		ho       <- difx[i]
        		hp       <- difx[i+m]
        		w        <- (cs[i+1,,] - ds[i,,])/(ho - hp)
        		ds[i,,] <- hp*w
        		cs[i,,] <- ho*w
      		}
      		if (2*ns < n-m) {
        		dy <- cs[ns+1,,]
      		} else {
        		dy <- ds[ns,,]
        		ns <- ns - 1
      		}
  		}
   		if (nydim == 4) {
      		for (i in 1:(n-m)) {
        		ho      <- difx[i]
        		hp      <- difx[i+m]
        		w       <- (cs[i+1,,,] - ds[i,,,])/(ho - hp)
        		ds[i,,,] <- hp*w
        		cs[i,,,] <- ho*w
      		}
      		if (2*ns < n-m) {
        		dy <- cs[ns+1,,,]

      		} else {
        		dy <- ds[ns,,,]
        		ns <- ns - 1
      		}
  		}
   		y <- y + dy
	}
   	return( list(y, dy) )
}
derivchk <- function (x, y, Dy) {

  #  checks that DY is the derivative of Y by comparing it
  #  with Y's central difference estimate.
  #  The value of |DYHAT-DY|/|DY| is returned.

  n <- length(x)
  if (n < 3) stop("X does not have enough elements")
  if (n != length(y) | n != length(Dy)) stop(
        "Lengh of Y or DY not consistent with length of X")
  indup <- 3:n
  inddn <- 1:(n-2)
  indct <- 2:(n-1)
  xdiff <- x[indup]-x[inddn]
  if (min(xdiff) <= 0) stop("X not strictly increasing")
  Dyhat <- (y[indup]-y[inddn])/xdiff
  ratio <- sqrt(mean((Dyhat-Dy[indct])^2))/sqrt(mean(Dy[indct]^2))
  return(ratio)
}
deriv <- function(expr, ...) UseMethod('deriv')

deriv.fd <- function(expr, Lfdobj=int2Lfd(1), ...)
{
  #  Applies linear differential operator LFD to functional data object FD
  #    and returns the result as functional data object DERIVFD.

  #  Last modified 6 January 2020 by Jim Ramsay

  fdobj <- expr
  if (!inherits(fdobj, "fd")) stop(
		"Argument  FD not a functional data object.")

  Lfdobj   <- int2Lfd(Lfdobj)

  basisobj <- fdobj$basis
  nbasis   <- basisobj$nbasis
  rangeval <- basisobj$rangeval

  evalarg  <- seq(rangeval[1], rangeval[2], len=10*nbasis+1)
  Lfdmat   <- eval.fd(evalarg, fdobj, Lfdobj)

  Lfdcoef  <- project.basis(Lfdmat, evalarg, basisobj)

  Dfdnames <- fdobj$fdnames
  Dfdnames[[3]] <- paste("D",Dfdnames[[3]])

  Dfdobj <- fd(Lfdcoef, basisobj, Dfdnames)

  return(Dfdobj)
}
derivs <- function(tnow, y, bwtlist) {
	#  Sets up a linear differential equation of order m
	#  as a system of m first order equations
	#  Arguments:
	#  TNOW    ... A vector of values of the independent variable t
	#  Y       ... A matrix of m values of Y corresponding to TNOW
	#  BWTLIST ... A functional data object containing coefficient functions
	#  Returns:
	#  DY      ... A matrix of derivative values corresponding to Y
	
	#  Last modified:  26 October 2005
	
  m  <- length(bwtlist);
  wmat <- matrix(0, m, m)
  wmat[1:(m-1),2:m] <- diag(rep(1,m-1))
  for (j in 1:m) {
	   bfdParj <- bwtlist[[j]]
	   wj      <- eval.fd(tnow, bfdParj$fd)
	   wmat[m,j] <- -wj
  }
  dy <- wmat %*% y
  return(dy)
}
df2lambda <- function(argvals, basisobj, wtvec=rep(1,n), Lfdobj=0, df=nbasis)
{
#  Convert a degree of freedom DF for a smooth to the equivalent value
#    of the smoothing parameter lambda.

#  Last modified 26 October 2005

n <- length(argvals)
nbasis <- basisobj$nbasis
if (df >= nbasis) {
   lambda <- 0
   return(lambda)
}

TOL    <- 1e-3
GOLD   <- 1.0
GLIMIT <- 2.0
TINY   <- 1.0e-20

#  find machine precision
eps <- 1
tol1 <- 1 + eps
while (tol1 > 1) {
   eps  <- eps/2
   tol1 <- 1 + eps
}
eps <- sqrt(eps)

#  ------  initialization of lambda by finding bracketing values ------------
#             a < b < c such that  fb < fa  and  fb < fc
#  first use input value for lambda unless it is zero, in which case -1
bx <- -4.0
#  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
lambda <- 10^(bx)
fb <- (lambda2df(argvals, basisobj, wtvec, Lfdobj, lambda) - df)^2
#  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
#  now try bracketing the minimum by using a large value and a small
#  value.  If (this doesn't work, revert to the iterative method
#  at statement 5
if (bx >= -10 &&  bx <= 5) {
#  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   cx <- 5  #  the upper limit
#  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   lambda <- 10^(cx)
   fc <- (lambda2df(argvals, basisobj, wtvec, Lfdobj, lambda) - df)^2
#  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   ax <- -8  #  the lower limit
#  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   lambda <- 10^(ax)
   fa <- (lambda2df(argvals, basisobj, wtvec, Lfdobj, lambda) - df)^2
}
#  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
#  check to see if minimum bracketed
#print(c(ax,bx,cx,fa,fb,fc, lambda))
if (fb >= fa || fb >= fc) {
  #  Failure to bracket minimum, proceed with iterative search for
  #    bracketing values.
  #  First, as an alternative value for ax, use the input value plus 0.1
  ax <- bx + 1
  #  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  lambda <- 10^(ax)
  fa <- (lambda2df(argvals, basisobj, wtvec, Lfdobj, lambda) - df)^2
  #  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  #  now the bracketing process begins
  if (fb > fa) {
     #  exchange ax and bx
     dum <- ax
     ax  <- bx
     bx  <- dum
     dum <- fb
     fb  <- fa
     fa  <- dum
  }
  #  first guess at cx
  cx <- bx + GOLD*(bx - ax)
  #  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  lambda <- 10^(cx)
  fc <- (lambda2df(argvals, basisobj, wtvec, Lfdobj, lambda) - df)^2
  #  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  #  check if (three values bracket minimum
  #print(c(ax,bx,cx,fa,fb,fc, lambda))
  while (fb >= fc) {
     r <- (bx - ax)*(fb - fc)
     q <- (bx - cx)*(fb - fa)
     u <- bx -
       ((bx - cx)*q - (bx - ax)*r)/(2.0*sign(max(c(abs(q-r),TINY)))*(q-r))
     ulim <- bx + GLIMIT*(cx - bx)
     if ((bx-u)*(u-cx) > 0.0) {
        #  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        lambda <- 10^(u)
        fu <- (lambda2df(argvals, basisobj, wtvec, Lfdobj, lambda) - df)^2
        #  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        if (fu < fc) {
           #  success
           ax <- bx
           bx <- u
           fa <- fb
           fb <- fu
           break
        }
        if (fu > fb) {
           #  also success
           cx <- u
           fc <- fu
           break
        }
        #  failure:  fu >= fb
        u <- cx + GOLD*(cx - bx)
        #  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        lambda <- 10^(u)
        fu <- (lambda2df(argvals, basisobj, wtvec, Lfdobj, lambda) - df)^2
        #  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     }
     if ((cx - u)*(u - ulim) > 0.0) {
        #  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        lambda <- 10^(u)
        fu <- (lambda2df(argvals, basisobj, wtvec, Lfdobj, lambda) - df)^2
        #  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        if (fu < fc) {
           bx <- cx
           cx <-  u
           u  <- cx + GOLD*(cx - bx)
           fb <- fc
           fc <- fu
           #  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
           lambda <- 10^(u)
           fu <- (lambda2df(argvals, basisobj, wtvec, Lfdobj, lambda) - df)^2
           #  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        }
     }
     if ((u-ulim)*(ulim-cx) >= 0.0) {
        u <- ulim
        #  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        lambda <- 10^(u)
        fu <- (lambda2df(argvals, basisobj, wtvec, Lfdobj, lambda) - df)^2
        #  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     } else {
        u <- cx + GOLD*(cx - bx)
        #  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        lambda <- 10^(u)
        fu <- (lambda2df(argvals, basisobj, wtvec, Lfdobj, lambda) - df)^2
        #  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     }
     ax <- bx
     bx <- cx
     cx <- u
     fa <- fb
     fb <- fc
     fc <- fu
     #print(c(ax,bx,cx,fa,fb,fc, lambda))
  }  #  end of while loop
}
#  ---------------------------------------------------------------------
#  --------------------  bracketing successful  ------------------------
#  ---------------------------------------------------------------------
a  <- min(c(ax,cx))
b  <- max(c(ax,cx))
v  <- bx
w  <- v
x  <- v
e  <- 0.0
fx <- fb
fv <- fx
fw <- fx
#  ---------------------------------------------------------------------
#  --------------------  main loop starts here -------------------------
#  ---------------------------------------------------------------------
xm   <- 0.5*(a + b)
tol1 <- eps*abs(x) + TOL/3
tol2 <- 2*tol1
crit <- abs(x - xm) - (tol2 - 0.5*(b - a))
#print(c(crit, lambda))
while (crit > 0) {
   #  is golden-section necessary?
   if (abs(e) > tol1) {
      #  fit parabola
      r <- (x - w)*(fx - fv)
      q <- (x - v)*(fx - fw)
      p <- (x - v)*q - (x - w)*r
      q <- 2.0*(q - r)
      if (q > 0.0)   p <- -p
      q <- abs(q)
      s <- e
      e <- d
      #  is parabola acceptable?
      if (abs(p) < abs(0.5*q*s) & p > q*(a - x) & p < q*(b - x)) {
         #  a parabolic interpolation step
         d <- p/q
         u <- x + d
         #  f must not be evaluated too close to a or b
         if ((u - a) < tol2 ||  b - u < tol2) {
            if (xm - x >= 0.0) d <- tol1 else d <- -tol1
         }
      } else {
         #  a golden-section step
         if (x >= xm) e <- a - x else e <- b - x
         d <- 0.382*e
      }
   } else {
      #  a golden-section step
      if (x >= xm) e <- a - x else e <- b - x
      d <- 0.382*e
   }
#  f must not be evaluated too close to x
   if (abs(d) >=  tol1) {
      u <- x + d
   } else {
      if (d >= 0.0) u <- x + tol1 else u <- x - tol1
  }
  #  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  lambda <- 10^u
  fu <- (lambda2df(argvals, basisobj, wtvec, Lfdobj, lambda) - df)^2
  #  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  #  update  a, b, v, w, and x
  if (fu <= fx) {
     if (u  >= x) a <- x else b <- x
     v  <- w
     w  <- x
     x  <- u
     fv <- fw
     fw <- fx
     fx <- fu
  } else {
     if (u  < x) a <- u else b <- u
     if (fu <= fw || w == x) {
        v  <- w
        w  <- u
        fv <- fw
        fw <- fu
     }
     if (fu <= fv || v == x || v == w) {
        v  <- u
        fv <- fu
     }
  }
  xm   <- 0.5*(a + b)
  tol1 <- eps*abs(x) + TOL/3
  tol2 <- 2*tol1
  crit <- abs(x - xm) - (tol2 - 0.5*(b - a))
  #print(c(crit, lambda))
#  -------------------  } of main loop  ------------------------------
}
return(lambda)
}
df.residual.fRegress <- function(object, ...){
  dfm <-   object$df
  if(is.null(dfm))
    stop("'object' does not have a 'df' component.")
#
  nobs <- length(object$wt)
  dfr <- nobs-dfm
  attr(dfr, 'nobs') <- nobs
  attr(dfr, 'df.model') <- dfm
  dfr
}
dirs <- function(path='.', pattern=NULL, exclude=NULL, all.files=FALSE,
                 full.names=FALSE, recursive=FALSE, ignore.case=FALSE){
##
## 1.  mainDir <- dir(...)
##
  mD.all <- dir(path=path, pattern=pattern, all.files=all.files,
                 full.names=full.names, ignore.case=ignore.case)
  {
    if(is.null(exclude)) mainDir <- mD.all
    else {
      mD.excl <- dir(path=path, pattern=exclude, all.files=all.files,
                     full.names=full.names, ignore.case=ignore.case)
      mainDir <- mD.all[!(mD.all %in% mD.excl)]
    }
  }
  if(length(mainDir)<1)return(character(0))
##
## 2.  file.info(...)$isdir
##
  mD <- {
    if(full.names) mainDir
    else file.path(path, mainDir)
  }
  fi <- file.info(mD)
# 
  mainDirs <- mainDir[fi$isdir]
  nDirs <- length(mainDirs)
  if(nDirs<1) return(character(0))
##
## 3.  if(recursive)...
##
  if(!recursive)return(mainDirs)
# 
  mD.full <- mD[fi$isdir]
  Dirs <- vector('list', nDirs)
  names(Dirs) <- mainDirs
#
  for(i in 1:nDirs){
    id <- dirs(path=mD.full[i], pattern=pattern, exclude=exclude, 
               all.files=all.files, full.names=full.names,
               ignore.case=ignore.case)
    if(!full.names)
      id <- file.path(mainDirs[[i]], id)
    Dirs[[i]] <- c(mainDirs[i], id) 
  }
#
  unlist(Dirs, use.names=FALSE)
}
eigchk <- function(Cmat) {
    
  #  Last modified 25 August 2020 by Jim Ramsay
  
  #  Cmat for NA's
  
  if (any(is.na(Cmat))) stop("Cmat has NA values.")
  
  #  check Cmat for Cmatmetry
  
  if (max(abs(Cmat-t(Cmat)))/max(abs(Cmat)) > 1e-10) {
    stop('CMAT is not symmetric.')
  } else {
    Cmat <- (Cmat + t(Cmat))/2
  }
  
  #  check Cmat for singularity
  
  eigval <- eigen(Cmat)$values
  ncoef  <- length(eigval)
  if (eigval[ncoef] < 0) {
    neig <- min(length(eigval),10)
    cat("\nSmallest eigenvalues:\n")
    print(eigval[(ncoef-neig+1):ncoef])
    cat("\nLargest  eigenvalues:\n")
    print(eigval[1:neig])
    stop("Negative eigenvalue of coefficient matrix.")
  }
  if (eigval[ncoef] == 0) stop("Zero eigenvalue of coefficient matrix.")
  logcondition <- log10(eigval[1]) - log10(eigval[ncoef])
  if (logcondition > 12) {
    warning("Near singularity in coefficient matrix.")
    cat(paste("\nLog10 Eigenvalues range from\n",
              log10(eigval[ncoef])," to ",log10(eigval[1]),"\n"))
  }
}

eigen.pda = function(pdaList,plotresult=TRUE,npts=501,...)
{
  rangval = pdaList$resfdlist[[1]]$basis$rangeval
  
  m = length(pdaList$resfdlist)  
  tfine = seq(rangval[1],rangval[2],length.out=npts)

  bwtlist = pdaList$bwtlist
  awtlist = pdaList$awtlist
  ufdlist = pdaList$ufdlist

  if(m == 1){
    d = length(bwtlist)
    xlabstr = names(bwtlist[[1]]$fd$fdnames)[[1]]

    betamat = array(0,c(npts,d,d))
      
    for(i in 1:d){
      betamat[,1,d-i+1] = -eval.fd(tfine,bwtlist[[i]]$fd)
      if(i < d) betamat[,i+1,i] = 1
    }
    
    if(!is.null(awtlist)){
      umat = matrix(0,npts,d)
      for(i in 1:length(awtlist)){
        umat[,1] = umat[,1] + 
          eval.fd(tfine,awtlist[[i]]$fd)*eval.fd(tfine,ufdlist[[i]])
      }
    
    }
    
  }
  else{
    d = length(bwtlist[[1]][[1]])
    xlabstr = names(bwtlist[[1]][[1]][[1]]$fd$fdnames)[[1]]
#    betamat = array(0,c(npts,m,m,d))

    betamat = array(0,c(npts,m*d,m*d))
    
    for(k in 1:d){
      for(j in 1:m){
        for(i in 1:m){
#                betamat[,i,j,k] = eval.fd(tfine,bwtlist[[i]][[j]][[k]]$fd)
          if(!is.null(bwtlist[[i]][[j]][[k]]))
          betamat[,j,m*(d-k)+i] = -eval.fd(tfine,bwtlist[[i]][[j]][[k]]$fd)
        }
        if(k < d){
            betamat[,m*k+j,m*(k-1)+j] = 1
        }
      }
    }
    
    if(!is.null(awtlist)){
      umat = matrix(0,npts,m*d)
      for(k in 1:d){
        for(i in 1:length(awtlist[[k]])){
          if(!is.null(awtlist[[k]][[i]]))
            umat[,k] = umat[,k] + 
               eval.fd(tfine,awtlist[[k]][[i]]$fd)*eval.fd(tfine,ufdlist[[k]][[i]])
        }
      }
    }
    
  } 
  
  eigvals = matrix(0,npts,m*d)
  limvals = matrix(0,npts,m*d)
              
  for(i in 1:npts){
      eigvals[i,] = eigen(betamat[i,,])$values
      if(!is.null(awtlist)) limvals[i,] = solve(betamat[i,,],umat[i,])
  }
  
  if(plotresult){
     if(!is.null(awtlist)) par(mfrow=c(3,1))
     else par(mfrow=c(2,1))
     matplot(tfine,Re(eigvals),type='l',xlab=xlabstr,ylab='Real',main='Eigenvalues',...)
     abline(h = 0)
     matplot(tfine,Im(eigvals),type='l',xlab=xlabstr,ylab='Imaginary',...)
     abline(h = 0)
    
     if(!is.null(awtlist))
     matplot(tfine,limvals[,1:d],type='l',xlab=xlabstr,ylab='Fixed Point',main='Instantaneous Limits',...)

  }
  
  return(list(argvals=tfine,eigvals=eigvals,limvals=limvals))  
} 
  
  Eigen <- function(x, symmetric, only.values = FALSE, valuenames ){
##
## 1.  symmetric?  
##
  N <- nrow(x)
  if (missing(symmetric)) 
    symmetric <- isSymmetric.matrix(x)  
##
## 2.  eigen
##
  ev <- eigen(x, symmetric, only.values = FALSE)
##
## 3.  rNames
##
  rNames <- rownames(x)
  if(is.null(rNames))
    rNames <- {
      if(symmetric) paste('x', 1:N, sep='')
      else paste('xrow', 1:N, sep='')
    }
##
## 4.  parse valuenames
##   
  {
    if(missing(valuenames)){
      cNames <- colnames(x)
      if(is.null(cNames))
        cNames <- {
          if(symmetric) paste('x', 1:N, sep='')
          else paste('xcol', 1:N, sep='')
        }
      if(symmetric){
        valuenames <- {
          if(all(rNames==cNames))paste('ev', 1:N, sep='')
          else cNames
        }
      }
      else
        valuenames <- cNames
    }
    else{
      if(length(valuenames)<N)
        valuenames <- paste(valuenames, 1:N, sep='')
      else {
        if(length(valuenames)>N)
          warning('length(valuenames) = ', length(valuenames),
                  ' > nrow(x) = ', N,
                  '; using only the first ', N)
        valuenames <- valuenames[1:N]
      }
    }
  }  
##
## 5.  rNames
##
  names(ev$values) <- valuenames
  if(!only.values)
    dimnames(ev$vectors) <- list(rNames, valuenames)
##
## 6.  Done
##
  ev
}
predict.basisfd <- function(object, newdata=NULL, Lfdobj=0,
                            returnMatrix=FALSE, ...){
##
## 1.  newdata?
##
  if(is.null(newdata)){
    type <- object$type
    if(length(type) != 1)
      stop('length(object$type) must be 1;  is ',
           length(type) )
    newdata <- {
      if(type=='bspline') {
        unique(knots(object, interior=FALSE))
      } else object$rangeval
    }
  }
##
## 2.  eval.basis
##
  eval.basis(newdata, object, Lfdobj, returnMatrix)
}

eval.basis <- function(evalarg, basisobj, Lfdobj=0, returnMatrix=FALSE) {
#  Computes the basis matrix evaluated at arguments in EVALARG associated
#    with basis.fd object BASISOBJ.  The basis matrix contains the values
#    at argument value vector EVALARG of applying the nonhomogeneous
#    linear differential operator LFD to the basis functions.  By default
#    LFD is 0, and the basis functions are simply evaluated at argument
#    values in EVALARG.
#
#  If LFD is a functional data object with m + 1 functions c_1, ... c_{m+1},
#   then it is assumed to define the order m HOMOGENEOUS linear differential
#   operator
#
#            Lx(t) = c_1(t) + c_2(t)x(t) + c_3(t)Dx(t) + ... +
#                    c_{m+1}D^{m-1}x(t) + D^m x(t).
#
#  If the basis type is either polygonal or constant, LFD is ignored.
#
#  Arguments:
#  EVALARG ... Either a vector of values at which all functions are evaluated,
#              or a matrix of values, with number of columns corresponding to
#              number of functions in argument FD.  If the number of evaluation
#              values varies from curve to curve, pad out unwanted positions in
#              each column with NA.  The number of rows is equal to the maximum
#              of number of evaluation points.
#  BASISOBJ ... A basis object
#  LFDOBJ   ... A linear differential operator object
#               applied to the basis functions before they are to be evaluated.
#  RETURNMATRIX ... If False, a matrix in sparse storage model can be returned
#               from a call to function BsplineS.  See this function for
#               enabling this option.

#  Note that the first two arguments may be interchanged.

#  Last modified 24 December 2012
##
## 1.  check
##
#   Exchange the first two arguments if the first is a BASIS.FD object
#     and the second numeric

  if (is.numeric(basisobj) && inherits(evalarg, "basisfd")) {
    temp     <- basisobj
    basisobj <- evalarg
    evalarg  <- temp
  }

#  check EVALARG

#  if (!(is.numeric(evalarg))){# stop("Argument EVALARG is not numeric.")
# turn off warnings in checking if argvals can be converted to numeric.
  if(is.numeric(evalarg)){
    if(!is.vector(evalarg))stop("Argument 'evalarg' is not a vector.")
    Evalarg <- evalarg
  } else {
    op <- options(warn=-1)
    Evalarg <- as.numeric(evalarg)
    options(op)
    nNA <- sum(is.na(Evalarg))
    if(nNA>0)
      stop('as.numeric(evalarg) contains ', nNA,
           ' NA', c('', 's')[1+(nNA>1)],
           ';  class(evalarg) = ', class(evalarg))
#  if(!is.vector(Evalarg))
#      stop("Argument EVALARG is not a vector.")
  }

#  check basisobj

  if (!(inherits(basisobj, "basisfd"))) stop(
    "Second argument is not a basis object.")

#  check LFDOBJ

  Lfdobj <- int2Lfd(Lfdobj)
##
## 2.  set up
##
#  determine the highest order of derivative NDERIV required

  nderiv <- Lfdobj$nderiv

#  get weight coefficient functions

  bwtlist <- Lfdobj$bwtlist
##
## 3.  Do
##
#  get highest order of basis matrix

  basismat <- getbasismatrix(evalarg, basisobj, nderiv, returnMatrix)

#  Compute the weighted combination of derivatives is
#  evaluated here if the operator is not defined by an
#  integer and the order of derivative is positive.

  if (nderiv > 0) {
    nbasis    <- dim(basismat)[2]
    oneb      <- matrix(1,1,nbasis)
    nonintwrd <- FALSE
    for (j in 1:nderiv) {
        bfd    <- bwtlist[[j]]
        bbasis <- bfd$basis
        if (bbasis$type != "constant" || bfd$coefs != 0) nonintwrd <- TRUE
    }
    if (nonintwrd) {
      for (j in 1:nderiv) {
        bfd   <- bwtlist[[j]]
        if (!all(c(bfd$coefs) == 0.0)) {
            wjarray   <- eval.fd(evalarg, bfd, 0, returnMatrix)
            Dbasismat <- getbasismatrix(evalarg, basisobj, j-1,
                                        returnMatrix)
            basismat  <- basismat + (wjarray %*% oneb)*Dbasismat
        }
      }
    }
  }

  if((!returnMatrix) && (length(dim(basismat)) == 2)){
      return(as.matrix(basismat))
  } else {
      return(basismat)
  }

}

eval.bifd <- function(sevalarg, tevalarg, bifd, sLfdobj = 0, tLfdobj = 0) {

  #  Evaluates a bi-functional data object BIFD at argument values in arrays
  #  SEVALARG and TEVALARG.  Differential operators SLFD and TLFD are
  #     applied to BIFD if present.

  #  Last modified 6 January 2020 by Jim Ramsay

  if (!is.vector(sevalarg)) stop(
     "First argument is not a vector.")
  if (!is.vector(tevalarg)) stop(
     "Second argument is not a vector.")

  ns   <- length(sevalarg)
  nt   <- length(tevalarg)

  if (!(inherits(bifd, "bifd"))) stop("Third argument is not a bifd object")

  sbasisobj <- bifd$sbasis
  snbasis   <- sbasisobj$nbasis
  rangeval  <- sbasisobj$rangeval
  if (min(sevalarg) < rangeval[1] || max(sevalarg) > rangeval[2]) stop(
    "Values of the first argument are outside of permitted range.")

  tbasisobj <- bifd$tbasis
  tnbasis   <- tbasisobj$nbasis
  rangeval  <- tbasisobj$rangeval
  if (min(tevalarg) < rangeval[1] || max(tevalarg) > rangeval[2]) stop(
    "Values of the second argument are outside of permitted range.")

  coef  <- bifd$coefs
  coefd <- dim(coef)
  ndim  <- length(coefd)

  sLfdobj <- int2Lfd(sLfdobj)
  tLfdobj <- int2Lfd(tLfdobj)

  snderiv <- sLfdobj$nderiv
  tnderiv <- tLfdobj$nderiv

  sbasismat <- eval.basis(sevalarg,sbasisobj,sLfdobj)

  tbasismat <- eval.basis(tevalarg,tbasisobj,tLfdobj)

  if (ndim == 2) {
    evalbifd <- sbasismat %*% coef %*% t(tbasismat)
  }
  if (ndim == 3) {
    nrep  <- coefd[3]
    evalbifd <- array(0,c(ns,nt,nrep))
    for (i in 1:nrep) {
      evalbifd[,,i] <- sbasismat %*% coef[,,i] %*% t(tbasismat)
    }
    dimnames(evalbifd) <- list(NULL,NULL,dimnames(coef)[[3]])
  }
  if (ndim > 3) {
    nrep  <- coefd[3]
    nvar  <- coefd[4]
    evalbifd <- array(0,c(ns,nt,nrep,nvar))
    for (i in 1:nrep) for (j in 1:nvar) {
      evalbifd[,,i,j] <-
        sbasismat %*% coef[,,i,j] %*% t(tbasismat)
    }
    dimnames(evalbifd) <-
        list(NULL,NULL,dimnames(coef)[[3]],dimnames(coef)[[4]])
  }
  return(evalbifd)
}
evaldiag.bifd <- function(evalarg, bifdobj, sLfd=int2Lfd(0), tLfd=int2Lfd(0))
{
#  EVALDIAG_BIFD  evaluates a bi-functional data object BIFD
#  with both argument values in array EVALARG.
#  SLfd and TLfd are either integers giving the order of derivative,
#  or linear differential operators to be applied before evaluation.
#  Their defaults are 0, meaning that the function itself is evaluated.

#  last modified 2008(?) replacing Matlab subscripts with R style
# in lines 70, 77, 78;  previously modified 26 October 2005

#  Last modified 6 January 2020 by Jim Ramsay

#  exchange order if BIFD is the first argument
if (inherits(evalarg, "bifd")) {
    temp    <- bifdobj
    bifdobj <- evalarg
    evalarg <- temp
}

#  check EVALARG

evalarg <- as.vector(evalarg)

if (!inherits(bifdobj, "bifd")) stop(
    "Argument BIFD is not a bivariate functional data object.")

n <- length(evalarg)

#  extract the two bases

sbasisobj <- bifdobj$sbasis
tbasisobj <- bifdobj$tbasis
snbasis   <- sbasisobj$nbasis
tnbasis   <- tbasisobj$nbasis
ranges    <- sbasisobj$rangeval
ranget    <- tbasisobj$rangeval

#  check that the bases have the same range

if (any(ranges != ranget)) stop(
    "The ranges are not identical.")

#  check the differential operators

sLfd <- int2Lfd(sLfd)
tLfd <- int2Lfd(tLfd)

#  compute the basis matrix for SBASISOBJ

snderiv   <- sLfd$nderiv
sbasismat <- eval.basis(evalarg, sbasisobj, sLfd)

#  compute the basis matrix for tBASISOBJ

tnderiv   <- tLfd$nderiv
tbasismat <- eval.basis(evalarg, tbasisobj, tLfd)

#  Extract the coefficient matrix

coef  <- bifdobj$coefs
coefd <- dim(coef)
ndim  <- length(coefd)

if        (ndim == 2) {
        evalarray <- diag(sbasismat %*% coef %*% t(tbasismat))
} else if (ndim == 3) {
        ncurves   <- coefd[3]
        evalarray <- matrix(0,n,ncurves)
        for (i in 1:ncurves)
            evalarray[,i] <- diag(sbasismat %*% coef[,,i] %*% t(tbasismat))
} else if (ndim == 4) {
        ncurves  <- coefd[3]
        nvar     <- coefd[4]
        evalarray <- array(0,c(n,ncurves,nvar))
        for (i in 1:ncurves) {
            for (j in 1:nvar) {
                evalarray[,i,j] <-
                    diag(sbasismat %*% coef[,,i,j] %*% t(tbasismat))
            }
        }
} else {
       stop("The coefficient array has improper dimension.")
}

return(evalarray)
}
predict.fdSmooth <- function(object, newdata=NULL, Lfdobj=0,
                             returnMatrix=FALSE, ...){
  if(is.null(newdata)){
    newdata <- object$argvals
  }
  eval.fd(newdata, object$fd, Lfdobj, returnMatrix=returnMatrix)
}

fitted.fdSmooth <- function(object, returnMatrix=FALSE, ...){
  newdata <- object$argvals
  eval.fd(newdata, object$fd, 0, returnMatrix=returnMatrix)
}

residuals.fdSmooth <- function(object, returnMatrix=FALSE, ...){
  newdata <- object$argvals
  pred <- eval.fd(newdata, object$fd, 0, returnMatrix=returnMatrix)
  object$y-pred
}

predict.fdPar <- function(object, newdata=NULL, Lfdobj=0,
                          returnMatrix=FALSE, ...){
  predict.fd(object$fd, newdata, Lfdobj,
             returnMatrix=returnMatrix, ...)
}

predict.fd <- function(object, newdata=NULL, Lfdobj=0,
                       returnMatrix=FALSE, ...){
  if(is.null(newdata)){
    basis <- object$basis
    type <- basis$type
    if(length(type) != 1)
      stop('length(object$type) must be 1;  is ',
           length(type) )
    newdata <- {
      if(type=='bspline')
        unique(knots(basis, interior=FALSE))
      else basis$rangeval
    }
  }
  eval.fd(newdata, object, Lfdobj, returnMatrix=returnMatrix)
}

#  ----------------------------------------------------------------------------

eval.fd <- function(evalarg, fdobj, Lfdobj=0, returnMatrix=FALSE) {

#  EVAL_FD evaluates a functional data observation at argument
#  values EVALARG.
#
#  LFDOBJ is a functional data object defining the order m
#  HOMOGENEOUS linear differential operator of the form
#  Lx(t) = w_0(t) x(t) + ... + w_{m-1}(t) D^{m-1}x(t) +
#          \exp[w_m(t)] D^m x(t)
#
#  Arguments:
#  EVALARG ... A vector of values at which all functions are to
#              evaluated.
#  FDOBJ   ... Functional data object
#  LFDOBJ  ... A linear differential operator object
#              applied to the functions before they are evaluated.
#  RETURNMATRIX ... If False, a matrix in sparse storage model can be returned
#               from a call to function BsplineS.  See this function for
#               enabling this option.

#  Note that the first two arguments may be interchanged.

#  Returns:  An array of function values corresponding to the evaluation
#              arguments in EVALARG

#  Last modified Oct 19, 2012 by Spencer Graves

#  Check LFDOBJ

  Lfdobj <- int2Lfd(Lfdobj)

#  Exchange the first two arguments if the first is an FD object
#    and the second numeric

  if (inherits(fdobj, "numeric") && inherits(evalarg, "fd")) {
      temp    <- fdobj
      fdobj   <- evalarg
      evalarg <- temp
  }

#  check EVALARG

#  if (!(is.numeric(evalarg))) stop("Argument EVALARG is not numeric.")
  Evalarg <- evalarg
  if(!is.numeric(evalarg)){
    op <- options(warn=-1)
    evalarg <- as.numeric(Evalarg)
    options(op)
    nNA <- sum(is.na(evalarg))
    if(nNA>0)
      stop('as.numeric(evalarg) contains ', nNA,
           ' NA', c('', 's')[1+(nNA>1)],
           ';  class(evalarg) = ', class(Evalarg))
  }

  evaldim <- dim(evalarg)
  if (!(length(evaldim) < 3))
      stop("Argument 'evalarg' is not a vector or a matrix.")

#  check FDOBJ

  if (!(inherits(fdobj, "fd")))
      stop("Argument FD is not a functional data object.")

#  Extract information about the basis

  basisobj <- fdobj$basis
  nbasis   <- basisobj$nbasis
  rangeval <- basisobj$rangeval
  onerow   <- rep(1,nbasis)

  temp <- c(evalarg)
  temp <- temp[!(is.na(temp))]
  EPS  <- 5*.Machine$double.eps
  if (min(temp) < rangeval[1]-EPS || max(temp) > rangeval[2]+EPS) {
    warning(paste("Values in argument 'evalarg' are outside ",
                  "of permitted range and will be ignored."))
    print(c(rangeval[1]-min(temp), max(temp) - rangeval[2]))
  }

#  get maximum number of evaluation values

  if (is.vector(evalarg)) {
      n <- length(evalarg)
  } else {
      n <- evaldim[1]
  }

#  Set up coefficient array for FD

  coef  <- fdobj$coefs
  coefd <- dim(coef)
  ndim  <- length(coefd)
  if (ndim <= 1) nrep <- 1 else nrep <- coefd[2]
  if (ndim <= 2) nvar <- 1 else nvar <- coefd[3]

# check coef is conformable with evalarg

  if(length(evaldim)>1){
    if(evaldim[2]==1){
      evalarg <- c(evalarg)
    } else {
      if(evaldim[2] != coefd[2]){
        stop('evalarg has ', evaldim[2], ' columns;  does not match ',
             ndim[2], ' = number of columns of ffdobj$coefs')
      }
    }
  }

#  Set up array for function values

  if (ndim <= 2) {
      evalarray <- matrix(0,n,nrep)
  } else evalarray <- array(0,c(n,nrep,nvar))
  if (ndim == 2) dimnames(evalarray) <- list(NULL,dimnames(coef)[[2]])
  if (ndim == 3)
      dimnames(evalarray) <- list(NULL,dimnames(coef)[[2]],
                                  dimnames(coef)[[3]])

#  Case where EVALARG is a vector of values to be used for all curves

  if (is.vector(evalarg)) {

    evalarg[evalarg < rangeval[1]-1e-10] <- NA
    evalarg[evalarg > rangeval[2]+1e-10] <- NA
    basismat <- eval.basis(evalarg, basisobj, Lfdobj, returnMatrix)

    #  evaluate the functions at arguments in EVALARG

    if (ndim <= 2) {
      evalarray <- basismat %*% coef
#     needed because dimnames may malfunction with Matrix basismat
      dimnames(evalarray) <- list(rownames(basismat), colnames(coef))
    } else {
       evalarray <- array(0,c(n,nrep,nvar))
       for (ivar in 1:nvar) evalarray[,,ivar] <- basismat %*% coef[,,ivar]
    }

  } else {

  #  case of evaluation values varying from curve to curve

      for (i in 1:nrep) {
        evalargi <- evalarg[,i]
        if (all(is.na(evalargi))) stop(
            paste("All values are NA for replication",i))

        index    <- !(is.na(evalargi) | evalargi < rangeval[1] |
                                    evalargi > rangeval[2])
        evalargi <- evalargi[index]
        basismat <- eval.basis(evalargi, basisobj, Lfdobj, returnMatrix)

       #  evaluate the functions at arguments in EVALARG

        if (ndim == 2) {
            evalarray[  index, i] <- as.vector(basismat %*% coef[,i])
            evalarray[!(index),i] <- NA
        }
        if (ndim == 3) {
            for (ivar in 1:nvar) {
                evalarray[   index,i,ivar] <-
                    as.vector(basismat %*% coef[,i,ivar])
                evalarray[!(index),i,ivar] <- NA
            }
        }
    }
  }

  if((length(dim(evalarray))==2) && !returnMatrix) {
      return(as.matrix(evalarray))
  } else return(evalarray)
}

predict.monfd <- function(object, newdata=NULL, Lfdobj=0,
                            returnMatrix=FALSE, ...) {

  if(is.null(newdata))newdata <- object$argvals
##
## 1.  eval.monfd
##
  evalMon <- eval.monfd(newdata, object$Wfdobj, Lfdobj, returnMatrix)
##
## 2.  beta
##
  beta <- object$beta
  {
    if(length(dim(beta))<2){
      if(length(dim(evalMon))<2){
        be <- beta[2]*evalMon
        if(Lfdobj<1)
          be <- beta[1]+be
        return(be)
      }
      else
        stop('beta does not match eval.monfd(...)')
    }
    else {
      nem <- dim(evalMon)
      if(length(dim(beta)<3)) {
        if(length(nem)==2){
          be <- (evalMon*rep(beta[2,], each=nem[1]))
          if(Lfdobj<1)
            be <- (be+rep(beta[1,], each=nem[1]))
          return(be)
        }
        else
          stop('beta does not match eval.monfd(...)')
      }
      else {
        if(length(nem)==3){
          be <- (evalMon*rep(beta[2,,], each=nem[1]))
          if(Lfdobj<1)
            be <- (be+rep(beta[1,,], each=nem[1]))
          return(be)
        }
        else
          stop('beta does not match eval.monfd(...)')
      }
    }
  }

}

fitted.monfd <- function(object, ...){
  predict(object)
}

residuals.monfd <- function(object, ...){
  pred <- predict(object)
  object$y-pred
}

eval.monfd <- function(evalarg, Wfdobj, Lfdobj=int2Lfd(0), returnMatrix=FALSE) {
  #  Evaluates a monotone functional data observation, or the value of a linear
  #  differential operator LFD applied to the object,
  #  at argument values in an array EVALARGS.
  #  Functional data object LFD, if an integer, defines NDERIV, the
  #  order of derivative to be evaluated.
  #  Functional data object LFD, if a fd object, defines weight
  #  functions for computing the value of a linear differential operator
  #  applied to the functions that are evaluated.

  #  A monotone functional data object h  is in the form

  #           h(x) = [D^{-1} exp Wfdobj](x)

  #  where  D^{-1} means taking the indefinite integral.
  #  The interval over which the integration takes places is defined in
  #  the basisfd object in WFD.

  #  RETURNMATRIX ... If False, a matrix in sparse storage model can be returned
  #               from a call to function BsplineS.  See this function for
  #               enabling this option.

  #  Last modified 9 May 2012  by Jim Ramsay
  #  check Wfdobj

  if (!inherits(Wfdobj, "fd")) stop("Wfdobj is not a fd object.")

  #  extract number of variables and curves from coefficient matrix for Wfdobj

  coef  <- Wfdobj$coefs
  if (is.vector(coef)) coef <- as.matrix(coef)
  coefd <- dim(coef)
  ndim  <- length(coefd)
  if (ndim == 2) {
    ncurve <- coefd[2]
    nvar   <- 1
  } else {
    ncurve <- coefd[2]
    nvar   <- coefd[3]
  }

  #  determine if LFDOBJ is an integer

  Lfdobj <- int2Lfd(Lfdobj)

  if (!is.integerLfd(Lfdobj)) stop(
		"LFDOBJ is not an integer operator.")

  nderiv <- Lfdobj$nderiv

  n  <- length(evalarg)

  hmat <- array(0,c(n,ncurve,nvar))

  if (nderiv >= 2) Dwmat  <- getbasismatrix(evalarg, Wfdobj$basis, 1,
                                            returnMatrix)
  if (nderiv == 3) D2wmat <- getbasismatrix(evalarg, Wfdobj$basis, 2,
                                            returnMatrix)
  basislist = vector("list", 15)
  
  for (ivar in 1:nvar) {
    for (icurve in 1:ncurve) {

  	if (nderiv == 0) {
    	  if (ndim == 2) {
          if (ncurve == 1) {
            hmat[,icurve,ivar] <- monfn(evalarg, Wfdobj, basislist, 
                                        returnMatrix)
          } else {
            hmat[,icurve,ivar] <- monfn(evalarg, Wfdobj[icurve], basislist, 
                                        returnMatrix)
          }
        } else {
            hmat[,icurve,ivar] <- monfn(evalarg, Wfdobj[icurve,ivar], basislist,
                                        returnMatrix)
        }
  	}

  	if (nderiv == 1) {
    	  if (ndim == 2) {
            hmat[,icurve,ivar] <- exp(eval.fd(evalarg, Wfdobj[icurve], 0,
                                            returnMatrix))
        } else {
            hmat[,icurve,ivar] <- exp(eval.fd(evalarg, Wfdobj[icurve,ivar], 0,
                                            returnMatrix))
        }
  	}

  	if (nderiv == 2) {
        if (ndim == 2) {
          temp = (Dwmat %*% coef[,icurve])*
                                 exp(eval.fd(evalarg, Wfdobj[icurve],  0,
                                            returnMatrix))
    	    hmat[,icurve,ivar] <- as.vector(temp)
        } else {
          temp = (Dwmat %*% coef[,icurve])*
                                 exp(eval.fd(evalarg, Wfdobj[icurve,ivar], 0,
                                            returnMatrix))
    	    hmat[,icurve,ivar] <- as.vector(temp)
        }
  	}

  	if (nderiv == 3) {
        if (ndim == 2) {
    	    hmat[,icurve,ivar] <- as.vector(((D2wmat %*% coef[,icurve]) +
                                 (Dwmat  %*% coef[,icurve])^2)*
                                  exp(eval.fd(evalarg, Wfdobj[icurve], 0,
                                            returnMatrix)))
        } else {
    	    hmat[,icurve,ivar] <- as.vector(((D2wmat %*% coef[,icurve,ivar]) +
                                 (Dwmat  %*% coef[,icurve,ivar])^2)*
                                  exp(eval.fd(evalarg, Wfdobj[icurve,ivar],
                                            0, returnMatrix)))
        }
  	}

  	if (nderiv > 3) stop ("Derivatives higher than 3 are not implemented.")

    }
  }

  if (nvar == 1) hmat <- as.matrix(hmat[,,1])

  if((!returnMatrix) && (length(dim(hmat)) == 2)){
    return(as.matrix(hmat))
  }
  return(hmat)

}
eval.penalty  <- function(basisobj, Lfdobj=int2Lfd(0),
                          rng=rangeval)
{
#  EVAL_PENALTY evaluates the inner products of a linear
#  differential operator L defined by LFDOBJ applied to a set of
#  basis functions defined by BASISOBJ.
#
#  LFDOBJ is a functional data object defining the order m
#  NONHOMOGENEOUS linear differential operator of the form
#  Lx(t) = w_0(t) x(t) + ... + w_{m-1}(t) D^{m-1}x(t) +
#          \exp[w_m(t)] D^m x(t).
#  This is a change from previous usage where LFDOBJ was assumed to
#  define a HOMOGONEOUS differential operator.  See function
#  $Lfd/Lfd() for details.
#
#  Arguments:
#  BASISOBJ ... Either a basis object or an fd object or
#               an fdPar object.  If an fdPar object,
#               and if no LFDOBJ is supplied, the LFDOBJ
#               in the fdPar object is used.
#  LFDOBJ   ... A linear differential operator object
#               applied to the functions that are evaluated.
#  RNG      ... A range over which the product is evaluated
#
#  Returns:
#  PENALTYMAT ... Symmetric matrix containing inner products.
#                 This matrix should be non-negative definite
#                 With NDERIV zero eigenvalues, where NDERIV
#                 is the highest order derivative in LFDOBJ.
#                 However, rounding error will likely cause
#                 NDERIV smallest eigenvalues to be nonzero,
#                 so be careful about calling CHOL or otherwise
#                 assuming the range is N - NDERIV.

#  last modified 21 December 2012

#  check BASISOBJ

if (inherits(basisobj, "fd")) basisobj <- basisobj$basis
	
if (inherits(basisobj, "fdPar")) {
    fdobj       <- basisobj$fd
    basisobj    <- fdobj$basis
}

if (!inherits(basisobj, "basisfd"))  stop(
	"Argument BASISOBJ is not a functional basis object.")

#  set up default values

rangeval <- basisobj$rangeval

#  deal with the case where LFDOBJ is an integer

Lfdobj <- int2Lfd(Lfdobj)

#  determine basis type

type <- basisobj$type

#  choose appropriate penalty matrix function

if (type=="bspline") penaltymat <- bsplinepen(basisobj, Lfdobj, rng)
else if(type=="const") {
        rangeval   <- getbasisrange(basisobj)
        penaltymat <- rangeval[2] - rangeval[1]
      }
else if(type=="expon")     penaltymat <- exponpen(basisobj,   Lfdobj)
else if(type=="fourier")   penaltymat <- fourierpen(basisobj, Lfdobj)
else if(type=="monom")     penaltymat <- monomialpen(basisobj,   Lfdobj)
else if(type=="polygonal") penaltymat <- polygpen(basisobj,   Lfdobj)
else if(type=="power")     penaltymat <- powerpen(basisobj,   Lfdobj)
else stop("Basis type not recognizable, can not find penalty matrix")

#  If drop indices are provided, drop rows and cols
#  associated with these indices

dropind <- basisobj$dropind
nbasis  <- basisobj$nbasis

if (length(dropind) > 0) {
    index <- 1:nbasis
    index <- index[-dropind]
    penaltymat <- penaltymat[index,index]
}

#  Make matrix symmetric since small rounding errors can
#  sometimes results in small asymmetries

penaltymat <- (penaltymat + t(penaltymat))/2

return(penaltymat)

}
predict.posfd <- function(object, newdata=NULL, Lfdobj=0, ...){
#  Last modified  7 May 2012 by Jim Ramsay
  if (is.null(newdata)) newdata <- object$argvals
  evalPos <- eval.posfd(newdata, object$Wfdobj, Lfdobj)
#
  evalPos
}

fitted.posfd <- function(object, ...){
  predict(object)
}

residuals.posfd <- function(object, ...){
  pred <- predict(object)
  object$y-pred
}

eval.posfd <- function(evalarg, Wfdobj, Lfdobj=int2Lfd(0))
{
#  Evaluates a value or a derivative of a positive functional
#  data object.
#  A positive functional data object h  is = the form
#           h(x) = (exp Wfdobj)(x)
#  Note that the linear differential operator object LFDOBJ
#  MUST be an integer in the range 0 to 1.
#  Note that the first two arguments may be interchanged.
#
#  Arguments:
#  EVALARG ... A vector of values at which all functions are to
#              evaluated.
#  WFDOBJ  ... Functional data object.  It must define a single
#              functional data observation.
#  LFDOBJ  ... A linear differential operator object
#              applied to the functions that are evaluated.
#              Default is INT2LFD(0).
#
#  Returns:  An array of function values corresponding to the
#              argument values in EVALARG

#  Exchange the first two arguments if the first is an FD object
#    and the second numeric

if (is.numeric(Wfdobj) & inherits(class(evalarg), "fd")) {
    temp    <- Wfdobj
    Wfdobj  <- evalarg
    evalarg <- temp
}

#  Check the arguments

if (!is.numeric(evalarg)) stop(
	"Argument EVALARG is not numeric.")

evalarg <- as.vector(evalarg)

#  check WFDOBJ

if (!inherits(Wfdobj, "fd")) stop(
    "Argument WFDOBJ is not a functional data object.")

#  check LFDOBJ

Lfdobj = int2Lfd(Lfdobj)
if (!inherits(Lfdobj, "Lfd")) stop(
    "LFDOBJ is not linear differential operator object.")

nderiv = Lfdobj$nderiv

#  Extract information about the basis

basisobj <- Wfdobj$basis
nbasis   <- basisobj$nbasis
rangeval <- basisobj$rangeval
onerow   <- rep(1,nbasis)

#  Set up coefficient array for FD

coef  <- Wfdobj$coefs

#  Evaluate function values

index <- evalarg < rangeval[1]-1e-10
if (length(evalarg[index]) > 0) evalarg <- evalarg[!index]
index <- evalarg > rangeval[2]+1e-10
if (length(evalarg[index]) > 0) evalarg <- evalarg[!index]

basismat <- getbasismatrix(evalarg, basisobj, 0)
fdvec    <- exp(basismat %*% coef)

#  If a differential operator has been defined in LFDOBJ, compute
#  the derivative values

if (nderiv > 0) {
     Lbasismat <- eval.basis(evalarg, basisobj, Lfdobj)
     evalarray <- fdvec*(Lbasismat %*% coef)
} else evalarray <- fdvec

return(evalarray)

}

eval.surp <- function(evalarg, Wfdobj, nderiv=0) {
  #  Evaluates a value of a surprisal coordinate functional data object. 
  # #  Evaluates a value or a derivative of a surprisal coordinate functional  
  # #  data object. 
  #  A positive functional data object h  is <- the form
  #           h(x) <- (exp Wfdobj)(x)
  #  Note that the first two arguments may be interchanged.
  #  
  #
  #  Arguments:
  #  EVALARG ... A vector of values at which all functions are to 
  #              evaluated.
  #  WFDOBJ  ... Functional data object.  It must define a single
  #              functional data observation.
  #  NDERIV  ... The order of the derivative.  Must be 0, 1 or 2.
  #  Returns:  An array of function values corresponding to the 
  #              argument values in EVALARG
  
  #  Last modified 25 April 2025 by Jim Ramsay
  
  #  check arguments
  
  if (floor(nderiv) != nderiv) {
    stop('Third argument nderiv is not an integer.')
  }
  
  if (nderiv < 0 || nderiv > 2) {
    stop('Third argument nderiv is not 0, 1 or 2.')
  }
  
  #  Exchange the first two arguments if the first is an FD object
  #    and the second numeric
  
  if (is.numeric(Wfdobj) && fda::is.fd(evalarg)) {
    temp    <- Wfdobj
    Wfdobj  <- evalarg
    evalarg <- temp
  }
  
  #  Check the arguments
  
  if (!is.numeric(evalarg)) {
    stop('Argument EVALARG is not numeric.')
  }
  
  #  transpose EVALARG if necessary to make it a column vector
  
  evaldim <- dim(as.matrix(evalarg))
  if (evaldim[1] == 1 && evaldim[2] > 1) {  
    evalarg <- t(evalarg)  
  }
  
  #  check EVALARG
  
  if (evaldim[1] > 1 && evaldim[2] > 1) 
    stop('Argument EVALARG is not a vector.')
    
  evalarg  <- as.vector(evalarg)
  basisfd  <- Wfdobj$basis
  rangeval <- basisfd$rangeval
  evalarg[evalarg < rangeval[1]-1e-10] <- NA
  evalarg[evalarg > rangeval[2]+1e-10] <- NA
  
  #  check FDOBJ
    
  if (!fda::is.fd(Wfdobj)) 
        stop('Argument FD is not a functional data object.')
  
  #  compute Zmat, a M by M-1 orthonormal matrix
  
  Bmat <- Wfdobj$coefs
  M    <- dim(Bmat)[2] + 1
  if (M == 2) {
    root2 <- sqrt(2)
    Zmat <- matrix(1/c(root2,-root2),2,1)
  } else {
    Zmat <- zerobasis(M)
  }
  
  #  Set up coefficient array for FD
  
  BZmat    <- Bmat %*% t(Zmat)
  Xmat     <- fda::eval.basis(evalarg, basisfd) %*% BZmat
  MXmat    <- M^Xmat
  sum0     <- rowSums(MXmat)
  SumMXmat <- matrix(rep(sum0,each=M), ncol=M, byrow=TRUE)
  
  #  Case where EVALARG is a vector of values to be used for all curves
  #  NB:  Resist the temptation of use /log(M) anywhere else.  
  #  This code sets up Smat as defined with log basis M, and no further
  #  definition of log basis is required.
  
  if (nderiv == 0) {
    Smat   <- -Xmat + log(SumMXmat)/log(M)
    return(Smat)
  }
  
  # Note:  derivative values computed using Pmat rather than Bmat
  # This needs correcting
  
  #  First derivative:
  
  if (nderiv == 1) {
    Pmat    <- MXmat/SumMXmat
    DXmat   <- fda::eval.basis(evalarg, basisfd, 1) %*% BZmat
    matSum  <- rowSums(Pmat*DXmat)
    Rmat    <- matrix(rep(matSum,each=M), ncol=M, byrow=TRUE)
    DSmat   <- -DXmat + Rmat
    return(DSmat)
  }
  
  #  Second derivative
  
  if (nderiv == 2) {
    Pmat    <- MXmat/SumMXmat
    DXmat   <- fda::eval.basis(evalarg, basisfd, 1) %*% BZmat
    D2Xmat  <- fda::eval.basis(evalarg, basisfd, 2) %*% BZmat
    matSum  <- rowSums(Pmat*DXmat)
    Rmat    <- matrix(rep(matSum,each=M), ncol=M, byrow=TRUE)
    matSum2 <- rowSums((Pmat*(D2Xmat + DXmat^2) - Rmat*DXmat))
    D2Smat  <- -D2Xmat + 
      matrix(rep(matSum2,each=M), ncol=M, byrow=TRUE)
    return(D2Smat)
  }
}

#  -------------------------------------------------------------------
#  power method for "fd"
#  -------------------------------------------------------------------

"^.fd" <- function(e1, e2){
#  A positive integer pointwise power of a functional data object with
#  a B-splinebasis.  powerfd = fdobj^a.
#  Generic arguments e1 = fdobj and e2 = a.
#
#  The basis is tested for being a B-spline basis.  The function then
#  sets up a new spline basis with the same knots but with an order
#  that is M-1 higher than the basis for FDOBJ, where M = ceiling(a),
#  so that the order of differentiability for the new basis is
#  appropriate in the event that a is a positive integer, and also
#  to accommodate the additional curvature arising from taking a power.
#  The power of the values of the function over a fine mesh are computed,
#  and these are fit using the new basis.
#
#  Powers should be requested with caution, however, and especially if
#  a < 1, because, if there is strong local curvature in FDOBJ,
#  and if its basis is just barely adequate to capture this curvature,
#  then the power of the function may have considerable error
#  over this local area.  fdobj^a where a is close to zero is just
#  such a situation.
#
#  If a power of a functional data object is required for which the
#  basis is not a spline, it is better to either re-represent the
#  function in a spline basis, or, perhaps even better, to do the
#  math required to get the right basis and interpolate function
#  values over a suitable mesh.  This is especially true for fourier
#  bases.

#  Last modified 2012.07.01 by Spencer Graves
#  previously modified 3 November 2009

#  check first two arguments

  fdobj = e1
  a     = e2
  tol   = 1e-4

  if ((!(inherits(fdobj, "fd"))))
        stop("First argument for ^ is not a functional data object.")
  if ((!(is.numeric(a))))
        stop("Second argument for ^ is not numeric.")

#  extract basis

  basisobj = fdobj$basis

#  test the basis for being of B-spline type

  if (basisobj$type != "bspline"){
    e12 <- exponentiate.fd(e1, e2,
       tolint=.Machine$double.eps^0.75, basisobj=e1$basis,
       tolfd=sqrt(.Machine$double.eps)*
          sqrt(sum(e1$coefs^2)+.Machine$double.eps)^abs(e2),
       maxbasis=NULL, npoints=NULL)
    return(e12)
#    stop("FDOBJ does not have a spline basis.")
#    a1 <- round(a)
#    if(abs(a-a1)>.Machine$double.eps^.75)
#        stop('Fractional power not allowed of an fd object ',
#             'without a spline basis.')
#    if(a1<0)
#        stop('Negative powers not allowed of an fd object ',
#             'without a spline basis.')
#    if(a1==0){
#      if(a==0){
#          rng <- basisobj$rangeval
#          fdNames <- list(fdobj$fdnames$args, NULL, NULL)
#          fdout <- fd(1, const, fdNames)
#          return(fdout)
#      } else {
#          stop('very small nonzero powers not allowed of an fd object ',
#               'without a spline basis;  requested power = ', a)
#      }
#    }
#    fdout <- fdobj
#    for(i in seq(length=a1-1)) fdout <- fdout*fdobj
#    return(fdout)
  }

  nbasis        = basisobj$nbasis
  rangeval      = basisobj$rangeval
  interiorknots = basisobj$params
  norder        = nbasis - length(interiorknots)

#  Number of points at which to evaluate the power.  Even low
#  order bases can generate steep slopes and sharp curvatures,
#  especially if powers less than 1 are involved.

  nmesh = max(10*nbasis+1,501)

#  determine number curves and variables

  coefmat = fdobj$coef
  coefd   = dim(coefmat)
  ncurve  = coefd[2]
  if (length(coefd) == 2) {
    nvar = 1
  } else {
    nvar = coefd[3]
  }

#  evaluate function over this mesh

  tval = seq(rangeval[1],rangeval[2],len=nmesh)
  fmat = eval.fd(tval, fdobj)
  fmatNeg <- (fmat<0)
# eliminate negatives from roundoff only
  if(any(fmatNeg) && all(fmat[fmatNeg]>(-.Machine$double.eps)))
      fmat[fmatNeg] <- 0

#  find the minimum value over this mesh.  If the power is less than
#  one, return an error message.

  fmin = min(c(fmat))

#  a == 0:  set up a constant basis and return the unit function(s)

  if (a == 0) {
    newbasis = create.constant.basis(rangeval)
    if (nvar == 1) {
      powerfd = fd(matrix(1,1,ncurve), newbasis)
    } else {
      powerfd = fd(array(1,c(1,ncurve,nvar)), newbasis)
    }
    return(powerfd)
  }

#  a == 1:  return the function

  if (a == 1) {
    powerfd = fdobj
    return(powerfd)
  }

#  Otherwise:

  m = ceiling(a)

#  Check the size of the power.  If greater than one, estimating the
#  functional data object is relatively safe since the curvatures
#  involved are mild.  If not, then taking the power is a dangerous
#  business.

  if (m == a && m > 1) {

    #  a is an integer greater than one

    newnorder = (norder-1)*m + 1
    if (length(interiorknots) < 9) {
        newbreaks = seq(rangeval[1], rangeval[2], len=11)
    } else {
        newbreaks = c(rangeval[1], interiorknots, rangeval[2])
    }
    nbreaks   = length(newbreaks)
    newnbasis = newnorder + nbreaks - 2
    newbasis  = create.bspline.basis(rangeval, newnbasis, newnorder,
                                     newbreaks)
    ymat    = fmat^a
    ytol    = max(abs(c(ymat)))*tol
    powerfd = smooth.basis(tval, ymat, newbasis)$fd
    ymathat = eval.fd(tval,powerfd)
    ymatres = ymat - ymathat
    maxerr  = max(abs(c(ymatres)))
    while  (maxerr > ytol && nbreaks < nmesh) {
        newnbasis = newnorder + nbreaks - 2
        newbasis  = create.bspline.basis(rangeval, newnbasis, newnorder,
                                         newbreaks)
        newfdPar  = fdPar(newbasis, 2, 1e-20)
        powerfd   = smooth.basis(tval, ymat, newfdPar)$fd
        ymathat   = eval.fd(tval,powerfd)
        ymatres   = ymat - ymathat
        maxerr    = max(abs(c(ymatres)))
        if (nbreaks*2 <= nmesh) {
        newbreaks = sort(c(newbreaks,
            (newbreaks[1:(nbreaks-1)]+newbreaks[2:nbreaks])/2))
        } else {
            newbreaks = tval
        }
        nbreaks = length(newbreaks)
    }

    if (maxerr > ytol)
        warning("The maximum error exceeds the tolerance level.")

    return(powerfd)

  } else {

    #  a is fractional or negative

    #  check for negative values and a fractional power

    if (a > 0 && fmin < 0) stop(
         paste("There are negative values",
               "and the power is a positive fraction."))

    #  check for zero or negative values and a negative power

    if (a < 0 && fmin <= 0) stop(
        paste("There are zero or negative values",
               "and the power is negative."))

    if (length(interiorknots) < 9) {
        newbreaks = seq(rangeval[1], rangeval[2], n=11)
    } else {
        newbreaks = c(rangeval[1], interiorknots, rangeval[2])
    }
    nbreaks   = length(newbreaks)
    newnorder = max(4, norder+m-1)
    newnbasis = newnorder + nbreaks - 2
    newbasis  = create.bspline.basis(rangeval, newnbasis, newnorder,
                                     newbreaks)
    nmesh     = max(10*nbasis+1,101)
    tval      = seq(rangeval[1],rangeval[2],len=nmesh)
    fmat      = eval.fd(tval, fdobj)
    fmatNeg <- (fmat<0)
    if(any(fmatNeg) && all(fmat[fmatNeg]>(-.Machine$double.eps))){
        fmat[fmatNeg] <- 0
    }
#
    ymat      = fmat^a
    ytol      = max(abs(c(ymat)))*tol
    newfdPar  = fdPar(newbasis, 2, 1e-20)
    powerfd   = smooth.basis(tval, ymat, newfdPar)$fd
    ymathat   = eval.fd(tval,powerfd)
    ymatres   = ymat - ymathat
    maxerr    = max(abs(c(ymatres)))
    while (maxerr > ytol && nbreaks < nmesh) {
        newnbasis = newnorder + nbreaks - 2
        newbasis  = create.bspline.basis(rangeval, newnbasis,
                                         newnorder, newbreaks)
        newfdPar  = fdPar(newbasis, 2, 1e-20)
        powerfd   = smooth.basis(tval, ymat, newfdPar)$fd
        ymathat   = eval.fd(tval,powerfd)
        ymatres   = ymat - ymathat
        maxerr    = max(abs(ymatres))*tol
        if (nbreaks*2 <= nmesh) {
            newbreaks = sort(c(newbreaks,
            (newbreaks[1:(nbreaks-1)]+newbreaks[2:nbreaks])/2))
        } else {
            newbreaks = tval
        }
        nbreaks = length(newbreaks)
    }

    if (maxerr > ytol) {
        warning("The maximum error exceeds the tolerance level.")
    }

    return(powerfd)

  }

}

#  -------------------------------------------------------------------
#  sqrt method for "fd"
#  -------------------------------------------------------------------

sqrt.fd <- function(x)
{
#  Arguments:
#  x ...  A functional data object
#  Returns:
#  FDAROOT  ...  A functional data object that is the square root of x
#  Last modified:  27 october 2009

    if ((!(inherits(x, "fd")))) stop(
      "Argument for ^ is not a functional data object.")

    fdaroot <- x^0.5

    return(fdaroot)
}

exponentiate.fd <- function(e1, e2, tolint = .Machine$double.eps^0.75, basisobj = e1$basis, 
	tolfd = sqrt(.Machine$double.eps) * sqrt(sum(e1$coefs^2) 
	           + .Machine$double.eps)^abs(e2), 
	maxbasis = NULL, npoints = NULL) {
	##\n
	## e2=0?\n
    ##\n
    e1basis <- e1$basis
	rng <- e1basis$rangeval
	coefmat <- e1$coef
	coefd <- dim(coefmat)
	ncurve <- coefd[2]
	if (length(coefd) == 2) {
		nvar <- 1
	} else nvar <- coefd[3]
	#\n
	if (e2 == 0) {
		const <- create.constant.basis(rng)
		fdNames <- list(e1$fdnames$args, NULL, NULL)
		if (nvar == 1) {
			fdout <- fd(matrix(1, 1, ncurve), const, fdNames)
		} else {
			fdout <- fd(array(1, c(1, ncurve, nvar)), const, fdNames)
		}
		return(fdout)
	}
	##\n
	## e2<0?\n
    ##\n
    if (e2 < 0) {
		if (e1$type == "bspline") 
			return(e1^e2)
		    stop("Negative powers not allowed of fd objects without ", 
		         "a spline basis; power = ", e2)
	}
	##\n
	## Fourier basis\n
    ##\n
    if (e1basis$type == "fourier") {
		e2. <- floor(e2)
		if (e2. < 1) {
			const <- create.constant.basis(rng)
			fdNames <- list(e1$fdnames$args, NULL, NULL)
			if (nvar == 1) {
				e120 <- fd(matrix(1, 1, ncurve), const, fdNames)
			} else {
				e120 <- fd(array(1, c(1, ncurve, nvar)), const, fdNames)
			}
			outbasis <- e1basis
		} else {
			e120 <- e1
			for (i in seq(length(e2. - 1))) e120 <- e120 * e1
			outbasis <- e120$basis
		}
		if ((e2 - e2.) == 0) 
			return(e120)
		#\n
		rng <- outbasis$rangeval
		if (is.null(maxbasis)) 
			maxbasis <- 2 * outbasis$nbasis + 1
		if (is.null(npoints)) 
			npoints <- max(10 * maxbasis + 1, 501)
		#\n
		Time <- seq(rng[1], rng[2], length = npoints)
		e1. <- predict(e1, Time)
		e1.2 <- e1.^e2
		n.na <- sum(is.na(e1.2))
		if (n.na > 0) {
			stop("NAs generated in computing e1^e2 at ", n.na, " of ", npoints, 
				" sample points.")
		}
		#    fd1.2 <- Data2fd(Time, e1.2, outbasis)\n
		fd1.2 <- smooth.basis(Time, e1.2, outbasis)$fd
		d1.2 <- (e1.2 - predict(fd1.2, Time))
		if (all(abs(d1.2) < tolfd)) 
			return(fd1.2$fd)
		#\n
		morebases <- (maxbasis - outbasis$nbasis + 1)
		for (i in 1:morebases) {
			nbasisi <- outbasis$nbasis + 2
			if (nbasisi == (nbasisi/2) * 2) {
				nbasisi = nbasisi + 1
			}
			outbasis <- create.fourier.basis(rng, nbasis = nbasisi, 
			                                 period = diff(rng))
			#      fd1.2 <- Data2fd(Time, e1.2, npoints)\n
			fd1.2 <- smooth.basis(Time, e1.2, outbasis)$fd
			d1.2 <- (e1.2 - predict(fd1.2, Time))
			maxd1.2 <- max(abs(d1.2))
			if (maxd1.2 < tolfd) 
				return(fd1.2$fd)
		}
		e1name <- deparse(substitute(e1))
		e2name <- deparse(substitute(e2))
		warning("Lack of precision in ", e1name, "^", e2name, ": max error at ", 
			npoints, " sample points = ", maxd1.2)
		return(fd1.2$fd)
	}
	##\n
	## positive integer\n
    ##\n
    if (abs(e2%%1) <= tolint) {
		e2. <- round(e2)
		if (e2. == 0) 
			stop("powers near zero of fd objects without ", 
			     "a spline basis not allowed;  power = ", e2)
		fdout <- e1
		for (i in seq(length = e2. - 1)) fdout <- fdout * e1
		return(fdout)
	}
}
exponpen <- function(basisobj, Lfdobj=int2Lfd(2))
{

#  Computes the Exponential penalty matrix.
#  Argument:
#  BASISOBJ ... a basis.fd object of type "expon"
#  LFDOBJ   ... either the order of derivative or a
#                linear differential operator to be penalized.
#  Returns the penalty matrix.

#  Last modified 9 February 2007

#  Check BASISOBJ

if (!(inherits(basisobj, "basisfd"))) stop(
    "First argument is not a basis object.")

type <- basisobj$type
if (type != "expon") stop ("Wrong basis type")

#  Check LFDOBJ

Lfdobj <- int2Lfd(Lfdobj)

#  Compute penalty matrix

if (is.integerLfd(Lfdobj)) {
    nderiv  <- Lfdobj$nderiv
    ratevec <- basisobj$params
    nrate   <- length(ratevec)
    penaltymatrix <- matrix(0,nrate,nrate)
    tl <- basisobj$rangeval[1]
    tu <- basisobj$rangeval[2]
    for (irate in 1:nrate) {
      	ratei <- ratevec[irate]
      	for (jrate in 1:irate) {
        	ratej <- ratevec[jrate]
        	ratesum <- ratei + ratej
        	if (ratesum != 0) {
          		penaltymatrix[irate,jrate] <- (ratei*ratej)^nderiv *
              	(exp(ratesum*tu) - exp(ratesum*tl)) / ratesum
        	} else {
          		if (nderiv == 0) penaltymatrix[irate,jrate] <- tu - tl
        	}
        	penaltymatrix[jrate,irate] <- penaltymatrix[irate,jrate]
      	}
	}
} else {
    penaltymatrix <- inprod(basisobj, basisobj, Lfdobj, Lfdobj)
}

penaltymatrix
}
expon <- function (x, ratevec=1, nderiv=0)
{
#  This computes values of the exponentials, or their derivatives.
#  RATEVEC is a vector containing the rate constants, or mulipliers of X
#    in the exponent of e.
#  The default is the exponential function.
#  Arguments are as follows:
#  X       ... array of values at which the polynomials are to
#             evaluated
#  RATEVEC ... a vector containing the rate constants, or mulipliers of X
#              in the exponent of e.
#  NDERIV  ... order of derivative.  0 means only function values
#             are returned.
#  Return is a matrix with length(X) rows and NRATE columns containing
#  the values of the exponential functions or their derivatives.

#  last modified 5 December 2001

  x <- as.vector(x)
  n <- length(x)
  nrate <- length(as.vector(ratevec))
  expval <- matrix(0,n,nrate)
  for (irate in 1:nrate) {
    rate <- ratevec[irate]
    expval[,irate] <- rate^nderiv * exp(rate*x)
  }
  return (expval)

}
#combination
combinat=function(n,p){
        if (n<p){combinat=0}
        else {combinat=exp(lfactorial(n)-(lfactorial(p)+lfactorial(n-p)))}
}



#BD2
fBD2=function(data){
	p=dim(data)[1]
	n=dim(data)[2]
	rmat=apply(data,1,rank)
	down=apply(rmat,1,min)-1
	up=n-apply(rmat,1,max)
	(up*down+n-1)/combinat(n,2)
	
}

#MBD
fMBD=function(data){
	p=dim(data)[1]
	n=dim(data)[2]
	rmat=apply(data,1,rank)
	down=rmat-1
	up=n-rmat
	(rowSums(up*down)/p+n-1)/combinat(n,2)
}
#function boxplot
#fit: p by n functional data matrix, n is the number of curves
#method: BD2, MBD
fbplot=function(fit,x=NULL,method='MBD',depth=NULL,plot=TRUE,prob=0.5,color=6,outliercol=2,
				barcol=4,fullout=FALSE, factor=1.5,xlim=c(1,nrow(fit)),ylim=c(min(fit)-.5*diff(range(fit)),max(fit)+.5*diff(range(fit))),...){
				
  #if(is.fdSmooth(fit) | is.fdPar(fit)){ fit = fit$fd }  
	#if(is.fd(fit)){
    #if(length(x)==0){
    #  x = seq(fit$basis$rangeval[1],fit$basis$rangeval[2],len=101)
    #}
    #fit = eval.fd(x,fit)
  #}				
				
	tp=dim(fit)[1]
	n=dim(fit)[2]
	if (length(x)==0) {x=1:tp}
  #compute band depth	
  if (length(depth)==0){
	if (method=='BD2') {depth=fBD2(fit)}
	else if (method=='MBD') {depth=fMBD(fit)}
	else if (method=='Both') {depth=round(fBD2(fit),4)*10000+fMBD(fit)}
  }

	dp_s=sort(depth,decreasing=TRUE)
	index=order(depth,decreasing=TRUE)
	med=depth==max(depth)
	medavg=matrix(fit[,med],ncol=sum(med),nrow=tp)
	y=apply(medavg,1,mean)
	
	if (plot) {
	plot(x,y,lty=1,lwd=2,col=1,type='l',xlim,ylim,...)
	}
	for (pp in 1:length(prob)){
		m=ceiling(n*prob[pp])#at least 50%
		center=fit[,index[1:m]]
		out=fit[,index[(m+1):n]]
		inf=apply(center,1,min)
		sup=apply(center,1,max)
		
		if (prob[pp]==0.5){ #check outliers
			dist=factor*(sup-inf)
			upper=sup+dist
			lower=inf-dist
			outly=(fit<=lower)+(fit>=upper)
			outcol=colSums(outly)
			remove=(outcol>0)
			#outlier column
			colum=1:n
			outpoint=colum[remove==1]
			out=fit[,remove]
			woout=fit
			good=woout[,(remove==0),drop=FALSE]
			maxcurve=apply(good,1,max)
			mincurve=apply(good,1,min)
			if (sum(outly)>0){
				if (plot) {
				matlines(x,out,lty=2,col=outliercol,type='l',...)
				}
			}
			barval=(x[1]+x[tp])/2
			bar=which(sort(c(x,barval))==barval)[1]
			if (plot) {
			lines(c(x[bar],x[bar]),c(maxcurve[bar],sup[bar]),col=barcol,lwd=2)
		    lines(c(x[bar],x[bar]),c(mincurve[bar],inf[bar]),col=barcol,lwd=2)
			}
		}
		xx=c(x,x[order(x,decreasing=TRUE)])
		supinv=sup[order(x,decreasing=TRUE)]
		yy=c(inf,supinv)
		if (plot) {
		if (prob[pp]==0.5) {polygon(xx,yy,col=color[pp],border=barcol,lwd=2)}
		else {polygon(xx,yy,col=color[pp],border=NA)}
		}
	}
	if (plot) {
	lines(x,fit[,index[1]],lty=1,lwd=2,col=1,type='l')
	lines(x,maxcurve,col=barcol,lwd=2)
	lines(x,mincurve,col=barcol,lwd=2)
	if (fullout) {
		if (sum(outly)>0){
				if (plot) {
				matlines(x,out,lty=2,col=outliercol,type='l',...)
				}
			}
		}
	}
	return(list(depth=depth,outpoint=outpoint,medcurve=which(med)))
}



fd2list <- function(fdobj)
{
#  FD2LIST converts a univariate functional data object to a list
#  object, mainly for purposes of defining a linear differential
#  operator object.

#  For example, this code sets up a harmonic acceleration Lfd object
#    over the interval [0,365] for the daily weather data.
#  Lbasis  = create.constant.basis(c(0,365));  #  create a constant basis
#  Lcoef   = matrix(c(0,(2*pi/365)^2,0),1,3)   #  set up three coefficients
#  wfdobj  = fd(Lcoef,Lbasis)      # define an FD object for weight functions
#  wfdlist = fd2list(wfdobj)       # convert the FD object to a cell object
#  harmaccelLfd = Lfd(3, wfdlist)  #  define the operator object

#  Last modified 26 October 2005

#  get the coefficient matrix and the basis

    coef     <- fdobj$coefs
    coefsize <- dim(coef)
    nrep     <- coefsize[2]

#  check whether FDOBJ is univariate

    if (length(coefsize) > 2)
    	stop("FDOBJ is not univariate.")

    fdlist <- vector("list",0)
    for (i in 1:nrep) fdlist[[i]] <- fdobj[i]
    return(fdlist)
}
fdlabels <- function(fdnames, nrep, nvar) {

#  Extract plot labels and, if available, names for each replicate and
#  each variable

#  check fdnames, which must be a list object of length 3

  if (!inherits(fdnames, "list"))
    stop("Argument fdnames is not a list object.")

  if (length(fdnames) != 3)
    stop("Argument fdnames is not of length 3.")

#  xlabel is fdnames[[1]] if it has length 1 and is not null
#  otherwise xlabel is names(fdnames)[1]

  xlabel = fdnames[[1]]
  if (length(xlabel) > 1 || is.null(xlabel)) xlabel = names(fdnames)[1]
  if (!is.character(xlabel)) xlabel = ""

#  ylabel is fdnames[[3]] if it has length not equal to nvar and is not null
#  otherwise ylabel is names(fdnames)[3]

  ylabel = fdnames[[3]]
  if ( (nvar > 1 && length(ylabel) == nvar) || 
      is.null(ylabel)) ylabel = names(fdnames)[3]
  if (length(ylabel) > 1) {
    if     (inherits(ylabel, "character")) ylabel = ylabel[1]
    else {
      if (inherits(ylabel, "list"))        ylabel = ylabel[[1]]
      else                                 ylabel = ""
    }
  }
  if (!is.character(ylabel)) ylabel = ""

#  set up casenames
 
  if (length(fdnames[[2]]) == nrep) {
    casenames = as.character(fdnames[[2]])
  } else {                             
    casenames = NULL
  }

#  set up varnames

  if (length(fdnames[[3]]) == nvar) {
    varnames  = as.character(fdnames[[3]])
  } else {                             
    varnames  = NULL
  }

  return(list(xlabel=xlabel, ylabel=ylabel, 
              casenames=casenames, varnames=varnames)) 
}

fdParcheck = function(fdParobj, ncurve=NULL) {
  #  Last modified 16 November 2021 by Jim Ramsay
  if (inherits(fdParobj, "basisfd") && is.null(ncurve)) 
    stop("First argument is basisfd object and second argument is missing.")
  if (!inherits(fdParobj, "fdPar")) {
    if (inherits(fdParobj, "fd")) {
        fdParobj <- fdPar(fdParobj)
    }
    if (inherits(fdParobj, "basisfd")) {
      nbasis   <- fdParobj$nbasis
      fdParobj <- fdPar(fd(matrix(0,nbasis,ncurve),fdParobj))
    } else {
        stop(paste("'fdParobj' is not a functional parameter object,",
               "not a functional data object, and",
               "not a basis object."))
    }
  }
  return(fdParobj)
  
}

#  Generator function of class fdPar

fdPar <- function(fdobj=NULL, Lfdobj=NULL, lambda=0, estimate=TRUE,
                  penmat=NULL){

# Sets up a functional parameter object
#  Arguments:
#  FDOBJ    ... A functional data object.
#               The basis for this object is used to define
#               the functional parameter, or functional
#               parameters of FDOBJ has replications.
#               When an initial value is required for iterative
#               estimation of a functional parameter, the coefficients
#               will give the initial values for the iteration.
#  LFDOBJ   ... A linear differential operator value or a derivative
#               value for penalizing the roughness of the object.
#               By default, this is 0.
#  LAMBDA   ... The penalty parameter controlling the smoothness of
#               the estimated parameter.  By default this is 0.
#  ESTIMATE ... If nonzero, the parameter is estimated; if zero, the
#               parameter is held fixed at this value.
#               By default, this is 1.
#  PENMAT   ... The penalty matrix.
#               In repeated calls to SMOOTH_BASIS, if this is
#               saved, then the penalty does not need evaluating
#               repeatedly.  Don't use, though, if LFDOBJ or LAMBDA
#               are changed in the calculation.
#
#  An alternative argument list:
#  The first argument can also be a basis object.  In this case, an
#  FD object is set up with an empty coefficient matrix.
#  For many purposes, the coefficient array is either not needed, or
#  supplied later.
#
#  Return:
#  FDPAROBJ ... A functional parameter object

#  Last modified 16 April 2021 by Jim Ramsay

#  ----------------------------------------------------------------------
#                            Default fdPar objects
#  ----------------------------------------------------------------------

  if(!inherits(fdobj, 'fd')) {
    #  the first argument is not an fd object
    if (is.null(fdobj)) {
    #  fdPar called without arguments
      fdobj = fd()
    }  else {
      if (inherits(fdobj, "basisfd")) {
        #  if the first argument is a basis object, convert it to
        #  a default FD object with an zero square coefficient matrix.
        nbasis  <- fdobj$nbasis
        dropind <- fdobj$dropind
        nbasis  <- nbasis - length(dropind)
        coefs   <- matrix(0,nbasis,nbasis)
        fdnames <- list('time', 'reps 1', 'values')
        if(!is.null(fdobj$names)){
          basisnames <- {
            if(length(dropind)>0)
              fdobj$names[-dropind]
            else
              fdobj$names
          }
          dimnames(coefs) <- list(basisnames, NULL)
          fdnames[[1]] <- basisnames
        }
        fdobj <- fd(coefs, fdobj, fdnames)
      }
      else if(is.numeric(fdobj))fdobj <- fd(fdobj)

      else stop("First argument is neither a functional data object ",
                "nor a basis object.")
    }
  } else {
    # the first object is an fd object, and we need nbasis later
    nbasis <- fdobj$basis$nbasis
  }

#  ----------------------------------------------------------------------
#                            Check parameters
#  ----------------------------------------------------------------------

#  check Lfdobj

  {
    if (is.null(Lfdobj)) {
      if(fdobj$basis$type=='fourier'){
        rng <- fdobj$basis$rangeval
        Lfdobj <- vec2Lfd(c(0,(2*pi/diff(rng))^2,0), rng)
#        warning("Provding default Lfdobj = harmonic acceleration ",
#                "operator on c(", rng[1], ', ', rng[2],
#                ') = vec2Lfd(c(0,(2*pi/diff(rng))^2,0), rng);',
#                '  [default prior to fda 2.1.0:  int2Lfd(0)].')
      } else {
        norder <- {
          if (fdobj$basis$type=='bspline') norder.bspline(fdobj$basis)
          else 2
        }
        Lfdobj <- int2Lfd(max(0, norder-2))
      }
    }
    else
      Lfdobj <- int2Lfd(Lfdobj)
  }

  if (!inherits(Lfdobj, "Lfd"))
    stop("'Lfdobj' is not a linear differential operator object.")

#  check lambda

if (!is.numeric(lambda)) stop("Class of LAMBDA is not numeric.")
if (lambda < 0) stop("LAMBDA is negative.")

#  check estimate

if (!is.logical(estimate)) stop("Class of ESTIMATE is not logical.")

#  check penmat

if (!is.null(penmat)) {
    if (!is.numeric(penmat)) stop("PENMAT is not numeric.")
#    penmatsize <- size(penmat)
    penmatsize <- dim(penmat)
    if (any(penmatsize != nbasis)) stop("Dimensions of PENMAT are not correct.")
}

#  ----------------------------------------------------------------------
#                    set up the fdPar object
#  ----------------------------------------------------------------------

#  S4 definition
# fdParobj <- new("fdPar", fd=fdobj, Lfd=Lfdobj, lambda=lambda, estimate=estimate,
#                  penmat=penmat)

#  S3 definition

fdParobj <- list(fd=fdobj, Lfd=Lfdobj, lambda=lambda, estimate=estimate,
                 penmat=penmat)

oldClass(fdParobj) <- "fdPar"

fdParobj

}

#  ----------------------------------------------------------------------

#  "print" method for "fdPar"

print.fdPar <- function(x, ...)
{
  object <- x
  cat("Functional parameter object:\n\n")
      print("Functional data object:")
  print.fd(object$fd)
      print("Linear differential operator object:")
  print.Lfd(object$Lfd)
  cat(paste("\nSmoothing parameter =",object$lambda,"\n"))
  cat(paste("\nEstimation status =",object$estimate,"\n"))
      if (!is.null(object$penmat)) {
          print("Penalty matrix:")
          print(object$penmat)
      }
}

#  ----------------------------------------------------------------------

#  "summary" method for "fdPar"

summary.fdPar <- function(object, ...)
{
  cat("Functional parameter object:\n\n")
      print("Functional data object:")
  summary.fd(object$fd)
      print("Linear differential operator object:")
  summary.Lfd(object$Lfd)
  cat(paste("\nSmoothing parameter =",object$lambda,"\n"))
  cat(paste("\nEstimation status =",object$estimate,"\n"))
      if (!is.null(object$penmat))
          print(paste("Penalty matrix dimensions:",dim(object$penmat)))
}


#  setClass for "fd"

# setClass("fd",    representation(coef     = "array",
#                                  basisobj = "basisfd",
#                                  fdnames  = "list"))

#  Generator function of class fd

fd <- function (coef=NULL, basisobj=NULL, fdnames=NULL)
{
# This function creates a functional data object.
#    A functional data object consists of a basis for expanding a functional
#    observation and a set of coefficients defining this expansion.
#    The basis is contained in a "basisfd" object that is, a realization
#    of the "basisfd" class.

#  Arguments
#  COEF ... An array containing coefficient values for the expansion of each
#             set of function values in terms of a set of basis functions.
#           If COEF is a three-way array, then the first dimension
#             corresponds to basis functions, the second to replications,
#             and the third to variables.
#           If COEF is a matrix, it is assumed that there is only
#             one variable per replication, and then
#                 rows    correspond to basis functions
#                 columns correspond to replications
#           If COEF is a vector, it is assumed that there is only one
#             replication and one variable.
#  BASISOBJ ... a functional data basis object
#  FDNAMES  ... The analogue of the dimnames attribute of an array, this is
#               a list of length 3 with members containing:
#               1. a character vector of names for the argument values
#               2. a character vector of names for the replications or cases
#               3. a character vector of names for the functions
#               Each of these vectors can have a name referring to the modality
#                 of the data.  An example would be "time", "reps", "values"

#  Returns:
#  FD ... a functional data object

#  Last modified 24 December 2012 by Jim Ramsay

##
## 1.  check coef and get its dimensions
##

  if(is.null(coef) && is.null(basisobj)) basisobj <- basisfd()

  if(is.null(coef))coef <- rep(0, basisobj[['nbasis']])

  type <- basisobj$type

  {
    if (!is.numeric(coef)) stop("'coef' is not numeric.")
    else if (is.vector(coef)) {
      coef  <- as.matrix(coef)
      if (identical(type, "constant")) coef <- t(coef)
      coefd <- dim(coef)
      ndim  <- length(coefd)
    }
    else if (is.matrix(coef)) {
      coefd <- dim(coef)
      ndim  <- length(coefd)
    }
    else if (is.array(coef)) {
      coefd <- dim(coef)
      ndim  <- length(coefd)
    }
    else stop("Type of 'coef' is not correct")
  }

  if (ndim > 3)
    stop("'coef' not of dimension 1, 2 or 3")
##
## 2.  Check basisobj
##
  {
    if(is.null(basisobj)){
      rc <- range(coef)
      if(diff(rc)==0) rc <- rc+0:1
      dimC <- dim(coef)
      nb <- {
        if(is.null(dimC)) length(coef)
        else dimC[1]
      }
      basisobj <- create.bspline.basis(rc, nbasis=max(4, nb))
      type <- basisobj$type
    }
    else
      if (!(inherits(basisobj, "basisfd")))
        stop("Argument basis must be of basis class")
  }

  nbasis   <- basisobj$nbasis
  dropind  <- basisobj$dropind
  ndropind <- length(basisobj$dropind)
  if (coefd[1] != nbasis - ndropind)
    stop("First dim. of 'coef' not equal to 'nbasis - ndropind'.")


#  setup number of replicates and number of variables

  if (ndim > 1) nrep <- coefd[2] else nrep <- 1
  if (ndim > 2) nvar <- coefd[3] else nvar <- 1
##
## 3.  fdnames & dimnames(coef)
##
    #  set up default fdnames

  if(is.null(fdnames)){
    if (ndim == 1) fdnames <- list("time", "reps", "values")
    if (ndim == 2) fdnames <- list("time",
            paste("reps",as.character(1:nrep)), "values")
    if (ndim == 3) fdnames <- list("time",
            paste("reps",as.character(1:nrep)),
            paste("values",as.character(1:nvar)) )

    names(fdnames) <- c("args", "reps", "funs")
  }
  if(is.null(dimnames(coef))){
    dimc <- dim(coef)
    ndim <- length(dimc)
    dnms <- vector('list', ndim)
    if(dimc[1] == length(fdnames[[1]]))
      dnms[[1]] <- fdnames[[1]]
    if((ndim>1) && (dimc[2]==length(fdnames[[2]])))
      dnms[[2]] <- fdnames[[2]]
    if((ndim>2) && (dimc[3]==length(fdnames[[3]])))
      dnms[[3]] <- fdnames[[3]]
    if(!all(sapply(dnms, is.null)))
      dimnames(coef) <- dnms
  }

#  S4 definition
#   fdobj <- new("fd", coefs=coef, basis=basisobj, fdnames=fdnames)

#  S3 definition

  fdobj <- list(coefs=coef, basis=basisobj, fdnames=fdnames)
    oldClass(fdobj) <- "fd"
    fdobj
}

#  ------------------------------------------------------------------
#  "print" method for "fd"
#  ------------------------------------------------------------------

print.fd <- function(x, ... )
{
  object <- x
    cat("Functional data object:\n\n")

    cat(" Dimensions of the data:\n")
    cat(paste("   ",names(object$fdnames),"\n"))

    print.basisfd(object$basis)

}

#  ------------------------------------------------------------------
#  "summary" method for "fd"
#  ------------------------------------------------------------------

summary.fd <- function(object,...)
{
    cat("Functional data object:\n\n")
    cat(" Dimensions of the data:\n")
    cat(paste("   ",names(object$fdnames),"\n"))
    print.basisfd(object$basis)
    cat("\nCoefficient matrix:\n\n")
    object$coefs
}

#  -----------------------------------------------------------------
#  plus method for "fd"
#  -----------------------------------------------------------------

"+.fd" <- function(e1, e2){
  plus.fd(e1, e2)
}

plus.fd <- function(e1, e2, basisobj=NULL)
{
#  PLUS: Pointwise sum of two functional data objects,
#    the sum of a scalar and a functional data object,
#    or the sum of a vector and a functional data obect
#       where the length of the vector is the same as the
#       number of replications of the object.
#  When both arguments are functional data objects,
#  they need not have the same bases,
#  but they must either (1)  have the same number of replicates, or
#  [2] one function must have a single replicate and other multiple
#  replicates.  In the second case, the singleton function is
#  replicated to match the number of replicates of the other function.
#  In either case, they must have the same number of functions.
#  When both arguments are functional data objects, and the
#  bases are not the same,
#  the basis used for the sum is constructed to be of higher
#  dimension than the basis for either factor according to rules
#  described in function TIMES for two basis objects.
#  Finally, in the simple case where both arguments are
#  functional data objects, the bases are the same, and the
#  coefficient matrices are the same dims, the coefficient
#  matrices are simply added.

# Last modified 2010.06.21 by Giles Hooker
# Previously modified 2008.12.26 by Spencer Graves
# Previously modified 2008.09.30 by Giles Hooker

  if (!(inherits(e1, "fd") || inherits(e2, "fd")))
      stop("Neither argument for + is a functional data object.")

  if (inherits(e1, "fd") && inherits(e2, "fd")) {
    #  both arguments are functional data objects
    #  check to see of the two bases are identical
    #  and if (the coefficient matrices are conformable.
    basisobj1 <- e1$basis
    basisobj2 <- e2$basis
    type1     <- basisobj1$type
    type2     <- basisobj2$type
    nbasis1   <- basisobj1$nbasis
    nbasis2   <- basisobj2$nbasis
    range1    <- basisobj1$rangeval
    range2    <- basisobj2$rangeval
    params1   <- basisobj1$params
    params2   <- basisobj2$params
    coef1     <- e1$coefs
    coef2     <- e2$coefs
    coefd1    <- dim(coef1)
    coefd2    <- dim(coef2)
    #  test to see if the two objects match completely
    if (basisobj1 == basisobj2) {
        #  the two coefficient matrices can be simply added
      fdnames <- e1$fdnames
      plusfd  <- fd(coef1 + coef2, basisobj1, fdnames)
      return(plusfd)
    }
    basisobj <- basisobj1 * basisobj2
    #  check to see if (the number of dimensions match
    ndim1  <- length(coefd1)
    ndim2  <- length(coefd2)
    if (ndim1 != ndim2)
      stop("Dimensions of coefficient matrices not compatible.")
    #  allow for one function being a single replicate,
    #  and if (so, copy it as many times as there are replicates
    #  in the other function.
    if (coefd1[2] == 1 && coefd2[2] > 1) {
      if      (ndim1 == 2) {
        coef1 <- outer(as.vector(coef1),rep(1,coefd2[2]))
      } else if (ndim1 == 3) {
          temp <- array(0,coefd2)
          for (j in 1:coefd1[3]) {
            temp[,,j] <- outer(as.vector(coef1[,1,j]),rep(1,coefd2[2]))
          }
          coef1 <- temp
      } else {
        stop("Dimensions of coefficient matrices not compatible.")
      }
      coefd1   <- dim(coef1)
      e1$coefs <- coef1
    }
    if (coefd1[2] >  1 && coefd2[2] == 1) {
      if      (ndim2 == 2) {
        coef2 <- outer(as.vector(coef2),rep(1,coefd1[2]))
      } else if (ndim1 == 3) {
        temp <- array(0, coefd1)
        for (j in 1:coefd2[3]) {
          temp[,,j] <- outer(as.vector(coef2[,1,j]),rep(1, coefd1[2]))
        }
        coef2 <- temp
      } else {
        stop("Dimensions of coefficient matrices not compatible.")
      }
      coefd2   <- dim(coef2)
      e2$coefs <- coef2
    }
    #  check for equality of dimensions of coefficient matrices
    if (coefd1[2] != coefd2[2])
      stop("Number of replications are not equal.")
    #  check for equality of numbers of functions
    if (ndim1 > 2 && ndim2 > 2 && ndim1 != ndim2)
      stop(paste("Both arguments multivariate, ",
                 "but involve different numbers ",
                 "of functions."))
    basisobj1 <- e1$basis
    basisobj2 <- e2$basis
    #  check for equality of two bases
    if (basisobj1 == basisobj2) {
        #  if equal, just add coefficient matrices
      fdnames <- e1$fdnames
      plusfd <- fd(coef1 + coef2, basisobj1, fdnames)
      return(plusfd)
    } else {
      nbasis1   <- basisobj1$nbasis
      nbasis2   <- basisobj2$nbasis
      rangeval1 <- basisobj1$rangeval
      rangeval2 <- basisobj2$rangeval
      if (any(rangeval1 != rangeval2))
        stop("The ranges of the arguments are not equal.")
      neval     <- max(10*max(nbasis1+nbasis2) + 1, 201)
      evalarg   <- seq(rangeval1[1], rangeval2[2], len=neval)
      fdarray1  <- eval.fd(evalarg, e1)
      fdarray2  <- eval.fd(evalarg, e2)
      if ((ndim1 <= 2 && ndim2 <= 2) ||
          (ndim1 >  2 && ndim2 >  2))
        fdarray <- fdarray1 + fdarray2
      if (ndim1 == 2 && ndim2 > 2) {
        fdarray <- array(0,coefd2)
        for (ivar in 1:coefd2[3])
          fdarray[,,ivar] <- fdarray1 + fdarray2[,,ivar]
      }
      if (ndim1 > 2 && ndim2 == 2) {
        fdarray <- array(0,coefd1)
        for (ivar  in  1:coefd1[3])
          fdarray[,,ivar] <- fdarray1[,,ivar] + fdarray2
      }
      #  set up basis for sum
      coefsum  <- project.basis(fdarray, evalarg, basisobj, 1)
      fdnames1 <- e1$fdnames
      fdnames2 <- e2$fdnames
      fdnames  <- fdnames1
      fdnames[[3]] <- paste(fdnames1[[3]],"+",fdnames2[[3]])
    }
  } else {
    #  one argument is numeric and the other is functional
    if (!(is.numeric(e1) || is.numeric(e2)))
      stop("Neither argument for + is numeric.")
    if (is.numeric(e1) && is.fd(e2)) {
      fac   <- e1
      fdobj <- e2
    } else if (is.fd(e1) && is.numeric(e2)) {
      fac   <- e2
      fdobj <- e1
    } else
    stop("One of the arguments for + is of the wrong class.")
    coef     <- fdobj$coefs
    coefd    <- dim(coef)
    basisobj <- fdobj$basis
    nbasis   <- basisobj$nbasis
    rangeval <- basisobj$rangeval
    neval    <- max(10*nbasis + 1,501)
#    neval    <- min(neval,501)
    evalarg  <- seq(rangeval[1],rangeval[2], len=neval)
    fdmat    <- eval.fd(evalarg, fdobj)
    #  If one of the objects has length 1 and the other
    #  is longer, expand the scalar object into a vector

    if( length(fac) > 1){
      if (length(fac) > 1 && coefd[2] == 1) {
        fdmat <- outer(fdmat,rep(1,length(fac)))
        fac   <- t(outer(rep(neval,1),fac))
      }
      if (length(fac) == coefd[2]){
        fac = t(outer(rep(neval,1),fac))}
      if( coefd[2]>1 && length(fac) !=coefd[2] ){
        stop(paste("Dimensions of numerical factor and functional",
                   "factor cannot be reconciled."))
      }
    }

    fdarray <- fac + fdmat
    coefsum <- project.basis(fdarray, evalarg, basisobj)
    fdnames <- fdobj$fdnames
    if (length(fac) == 1)
      fdnames[[3]] <- paste(fac," + ",fdnames[[3]])
  }

  plusfd <- fd(coefsum, basisobj, fdnames)
  return(plusfd)

}

#  ---------------------------------------------------------------
#  minus method for "fd"
#  ---------------------------------------------------------------

"-.fd" <- function(e1, e2){
  minus.fd(e1, e2)
}

minus.fd <- function(e1, e2, basisobj=NULL)
{
#  MINUS: Pointwise difference two functional data objects,
#    the between a scalar and a functional data object,
#    or the difference between a vector and a functional data obect
#       where the length of the vector is the same as the
#       number of replications of the object.
#  When both arguments are functional data objects,
#  they need not have the same bases,
#  but they must either (1)  have the same number of replicates, or
#  [2] one function must have a single replicate and other multiple
#  replicates.  In the second case, the singleton function is
#  replicated to match the number of replicates of the other function.
#  In either case, they must have the same number of functions.
#  When both arguments are functional data objects, and the
#  bases are not the same,
#  the basis used for the sum is constructed to be of higher
#  dimension than the basis for either factor according to rules
#  described in function TIMES for two basis objects.
#  Finally, in the simple case where both arguments are
#  functional data objects, the bases are the same, and the
#  coefficient matrices are the same dims, the coefficient
#  matrices are simply added.

# Last modified 2010.06.21 by Giles Hooker
# Previously modified 2008.12.27 by Spencer Graves
# Previously modified 2008.09.30 by Giles Hooker

  if(missing(e2)){
    if(!inherits(e1, 'fd'))
      stop('e1 is not a functional data object;  class(e1) = ',
           class(e1) )
#
    e1$coefs <- (-coef(e1))
    return(e1)
  }
#
  if(!(inherits(e1, "fd") || inherits(e2, "fd")))
      stop("Neither argument for - is a functional data object.")

  if(inherits(e1, "fd") && inherits(e2, "fd")) {
    #  both arguments are functional data objects
    #  check to see of the two bases are identical
    #  and if (the coefficient matrices are conformable.
    basisobj1 <- e1$basis
    basisobj2 <- e2$basis
    type1     <- basisobj1$type
    type2     <- basisobj2$type
    nbasis1   <- basisobj1$nbasis
    nbasis2   <- basisobj2$nbasis
    range1    <- basisobj1$rangeval
    range2    <- basisobj2$rangeval
    params1   <- basisobj1$params
    params2   <- basisobj2$params
    coef1     <- e1$coefs
    coef2     <- e2$coefs
    coefd1    <- dim(coef1)
    coefd2    <- dim(coef2)
    #  test to see if the two objects match completely
    if (basisobj1 == basisobj2) {
        #  the two coefficient matrices can be simply added
        fdnames <- e1$fdnames
        minusfd  <- fd(coef1 - coef2, basisobj1, fdnames)
        return(minusfd)
    }
    basisobj <-  basisobj1 * basisobj2
    #  check to see if (the number of dimensions match
    ndim1  <- length(coefd1)
    ndim2  <- length(coefd2)
    if (ndim1 != ndim2)
        stop("Dimensions of coefficient matrices not compatible.")
    #  allow for one function being a single replicate,
    #  and if (so, copy it as many times as there are replicates
    #  in the other function.
    if (coefd1[2] == 1 && coefd2[2] > 1) {
      if      (ndim1 == 2) {
        coef1 <- outer(as.vector(coef1),rep(1,coefd2[2]))
      } else if (ndim1 == 3) {
          temp <- array(0,coefd2)
          for (j in 1:coefd1[3]) {
            temp[,,j] <- outer(as.vector(coef1[,1,j]),rep(1,coefd2[2]))
          }
          coef1 <- temp
      } else {
        stop("Dimensions of coefficient matrices not compatible.")
      }
      coefd1   <- dim(coef1)
      e1$coefs <- coef1
    }
    if (coefd1[2] >  1 && coefd2[2] == 1) {
      if      (ndim2 == 2) {
        coef2 <- outer(as.vector(coef2),rep(1,coefd1[2]))
      } else if (ndim1 == 3) {
        temp <- array(0, coefd1)
        for (j in 1:coefd2[3]) {
          temp[,,j] <- outer(as.vector(coef2[,1,j]),rep(1, coefd1[2]))
        }
        coef2 <- temp
      } else {
        stop("Dimensions of coefficient matrices not compatible.")
      }
      coefd2   <- dim(coef2)
      e2$coefs <- coef2
    }
    #  check for equality of dimensions of coefficient matrices
    if (coefd1[2] != coefd2[2])
        stop("Number of replications are not equal.")
    #  check for equality of numbers of functions
    if (ndim1 > 2 && ndim2 > 2 && ndim1 != ndim2)
        stop(paste("Both arguments multivariate, ",
                   "but involve different numbers ",
                   "of functions."))
    basisobj1 <- e1$basis
    basisobj2 <- e2$basis
    #  check for equality of two bases
    if (basisobj1 == basisobj2) {
        #  if equal, just difference coefficient matrices
        fdnames <- e1$fdnames
        minusfd <- fd(coef1 - coef2, basisobj1, fdnames)
        return(minusfd)
    } else {
        nbasis1   <- basisobj1$nbasis
        nbasis2   <- basisobj2$nbasis
        rangeval1 <- basisobj1$rangeval
        rangeval2 <- basisobj2$rangeval
        if (any(rangeval1 != rangeval2))
            stop("The ranges of the arguments are not equal.")
        neval     <- max(10*max(nbasis1+nbasis2) + 1, 201)
        evalarg   <- seq(rangeval1[1], rangeval2[2], len=neval)
        fdarray1  <- eval.fd(e1, evalarg)
        fdarray2  <- eval.fd(e2, evalarg)
        if ((ndim1 <= 2 && ndim2 <= 2) ||
            (ndim1 >  2 && ndim2 >  2))
            fdarray <- fdarray1 - fdarray2
        if (ndim1 == 2 && ndim2 > 2) {
            fdarray <- array(0,coefd2)
            for (ivar in 1:coefd2[3])
                fdarray[,,ivar] <- fdarray1 - fdarray2[,,ivar]
        }
        if (ndim1 > 2 && ndim2 == 2) {
            fdarray <- array(0,coefd1)
            for (ivar  in  1:coefd1[3])
                fdarray[,,ivar] <- fdarray1[,,ivar] - fdarray2
        }
        #  set up basis for sum
        coefsum  <- project.basis(fdarray, evalarg, basisobj, 1)
        fdnames1 <- e1$fdnames
        fdnames2 <- e2$fdnames
        fdnames  <- fdnames1
        fdnames[[3]] <- paste(fdnames1[[3]], "-", fdnames2[[3]])
    }
 } else {
    #  one argument is numeric and the other is functional
    if (!(is.numeric(e1) || is.numeric(e2)))
        stop("Neither argument for - is numeric.")
    if (is.numeric(e1) && is.fd(e2)) {
        fac   <- e1
        fdobj <- e2
    } else if (is.fd(e1) && is.numeric(e2)) {
        fac   <- -e2
        fdobj <- -e1
    } else
        stop("One of the arguments for - is of the wrong class.")
    coef     <- fdobj$coefs
    coefd    <- dim(coef)
    basisobj <- fdobj$basis
    nbasis   <- basisobj$nbasis
    rangeval <- basisobj$rangeval
    neval    <- max(10*nbasis + 1,501)
#    neval    <- min(neval,201)
    evalarg  <- seq(rangeval[1],rangeval[2], len=neval)
    fdmat    <- eval.fd(evalarg, fdobj)
    #  If one of the objects has length 1 and the other
    #  is longer, expand the scalar object into a vector

    if( length(fac) > 1){
    	 if (length(fac) > 1 && coefd[2] == 1) {
           fdmat <- outer(fdmat,rep(1,length(fac)))
           fac   <- t(outer(rep(neval,1),fac))
     	  }
     	  if (length(fac) == coefd[2]){
	  	fac = t(outer(rep(neval,1),fac))}
	  if( coefd[2]>1 && length(fac) !=coefd[2] ){
		stop(paste("Dimensions of numerical factor and functional",
                       "factor cannot be reconciled."))
	  }
     }


    fdarray <- fac - fdmat
    coefsum <- project.basis(fdarray, evalarg, basisobj)
    fdnames <- fdobj$fdnames
    if (length(fac) == 1)
        fdnames[[3]] <- paste(fac," - ",fdnames[[3]])
}

minusfd <- fd(coefsum, basisobj, fdnames)
return(minusfd)

}

#  -----------------------------------------------------------------
#  point-wise product method for "fd"
#  -----------------------------------------------------------------

"*.fd" <- function(e1, e2){
  times.fd(e1, e2)
}

times.fd <- function(e1, e2, basisobj=NULL)
{
#  TIMES: Pointwise product of two functional data objects,
#    the product of a scalar and a functional data object,
#    or the product of a vector and a functional data obect
#       where the length of the vector is the same as the
#       number of replications of the object.
#  When both arguments are functional data objects,
#  they need not have the same bases,
#  but they must either (1)  have the same number of replicates, or
#  (2) one function must have a single replicate and other multiple
#  replicates.  In the second case, each function in the multiple
#  replicate object is multiplied by the singleton function in the
#  other objects.
#  In either case, they must have the same number of functions.

#  When both arguments are functional data objects, the
#  basis used for the product is constructed to be of higher
#  dimension than the basis for either factor according to rules
#  described in function TIMES for two basis objects.

#  Arguments:
#  e1   ... Either a functional data object or a number
#  e2   ... Either a functional data object or a number
#  BASISOBJ ... An optional basis for the product.
#  At least one of e1 and e2 must be a functional data object.
#  Returns:
#  FDAPROD  ...  A functional data object that is e1 times e2

#  Last modified 2008.12.27 by Spencer Graves
#  previously modified:  3 January 2007

# Check if at least one argument is a functional data object

  if ((!(inherits(e1, "fd") | inherits(e2, "fd"))))
    stop("Neither argument for * is a functional data object.")

#  Determine which of two cases hold:
#   1.  both variables are functional
#   2.  only one of them is functional

  if ( inherits(e1, "fd") & inherits(e2, "fd") ) {

    #  --------------------------------------------------------
    #       both arguments are functional data objects
    #  --------------------------------------------------------


    #  get the dimensions of the two objects

    coef1  <- e1$coefs
    coef2  <- e2$coefs
    coefd1 <- dim(coef1)
    coefd2 <- dim(coef2)
    ndim1  <- length(coefd1)
    ndim2  <- length(coefd2)

    #  check that the two coefficient arrays have the same
    #  number of dimensions

    if (length(coefd1) != length(coefd2))
      stop("Number of dimensions of coefficient arrays do not match.")

    #  allow for one function having a single replicate,
    #  and if so, copy it as many times as there are replicates
    #  in the other function.

    #  e1 is single,  e2 has replications

    if (coefd1[2] == 1 && coefd2[2] > 1) {
      if (ndim1 == 2) {
        coef1 <- matrix(coef1,coefd1[1],coefd2[2])
      } else if (ndim1 == 3) {
        temp <- array(0,coefd2)
        for (j in 1:coefd1[3])
          temp[,,j] <- outer(coef1[,1,j],rep(1,coefd2[2]))
        coef1 <- temp
      } else {
        stop("Dimensions of coefficient matrices not compatible.")
      }
      coefd1       <- dim(coef1)
      e1$coefs <- coef1
    }

    #  e2 is single,  e1 has replications

    if (coefd1[2] >  1 && coefd2[2] == 1) {

      if      (ndim2 == 2) {
        coef2 <- matrix(coef2,coefd2[1],coefd1[2])
      } else if (ndim1 == 3) {
        temp <- array(0,coefd1)
        for (j in 1:coefd2[3])
          temp[,,j] <- outer(coef2[,1,j],rep(1,coefd1[2]))
        coef2 <- temp
      } else {
        stop("Dimensions of coefficient matrices not compatible.")
      }
      coefd2       <- dim(coef2)
      e2$coefs <- coef2
    }

    #  check that numbers of replications are equal

    if (coefd1[2] != coefd2[2])
      stop("Number of replications are not equal.")

    #  check for matching in the multivariate case

    if (ndim1 > 2 && ndim2 > 2 && ndim1 != ndim2)
      stop(paste("Both arguments multivariate, ",
                 "but involve different numbers ",
                 "of functions."))

    #  extract the two bases

    basisobj1 <- e1$basis
    basisobj2 <- e2$basis
    nbasis1   <- basisobj1$nbasis
    nbasis2   <- basisobj2$nbasis

    #  check that the ranges match if a range not supplied

    rangeval1 <- basisobj1$rangeval
    rangeval2 <- basisobj2$rangeval
    if (any(rangeval1 != rangeval2))
      stop("The ranges of the arguments are not equal.")

    #  set default basis object

    if(is.null(basisobj)) basisobj <- basisobj1*basisobj2

    #  set up a fine mesh for evaluating the product

    neval   <- max(10*max(nbasis1,nbasis2) + 1, 201)
    evalarg <- seq(rangeval1[1],rangeval2[2], length=neval)

    #  set up arrays of function values

    fdarray1  <- eval.fd(evalarg, e1)
    fdarray2  <- eval.fd(evalarg, e2)

    #  compute product arrays

    if ((ndim1 <= 2 && ndim2 <= 2) || (ndim1 > 2 && ndim2 > 2)) {
        #  product array where the number of dimensions match
      fdarray = fdarray1*fdarray2
    } else {
        #  product array where the number of dimensions don't match
      if (ndim1 == 2 && ndim2 > 2) {
        fdarray = array(0,coefd2)
        for (ivar in 1:coefd2[3])
          fdarray[,,ivar] <- fdarray1*fdarray2[,,ivar]
      }
      if (ndim1 > 2 && ndim2 == 2) {
        fdarray = array(0,coefd1)
        for (ivar in 1:coefd1[3])
          fdarray[,,ivar] <- fdarray1[,,ivar]*fdarray2
      }
    }

    #  set up the coefficient by projecting on to the
    #  product basis

    coefprod = project.basis(fdarray, evalarg, basisobj)

    #  set up the names

    fdnames1 <- e1$fdnames
    fdnames2 <- e2$fdnames
    fdnames  <- fdnames1
    fdnames[[3]] <- paste(fdnames1[[3]],"*",fdnames2[[3]])

  } else {

    #  --------------------------------------------------------
    #    one argument is numeric and the other is functional
    #  --------------------------------------------------------

    if ((!(is.numeric(e1) || is.numeric(e2))))
      stop("Neither argument for * is numeric.")
    #  make the numerical factor the first objec
    if (is.numeric(e1) && inherits(e2, "fd")) {
      fac   <- e1
      fdobj <- e2
    } else if (is.numeric(e2) && inherits(e1, "fd")) {
      fac   <- e2
      fdobj <- e1
    } else stop("One of the arguments for * is of the wrong class.")
    coef     <- fdobj$coefs
    coefd    <- dim(coef)
    fac <- as.vector(fac)
    #  check the length of the factor
    if (!(length(fac) == coefd[2] || length(fac) == 1)) stop(
        "The length of the numerical factor is incorrect.")
    #  compute the coefficients for the product
    coefprod <- fac*coef
    basisobj <- fdobj$basis

    #  set up the names

    fdnames  <- fdobj$fdnames
    fdnames[[3]] <- paste(as.character(fac),"*",fdnames[[3]])

  }

#  set up the functional data object

  fdprod   <- fd(coefprod, basisobj, fdnames)

  return(fdprod)

}

#  -------------------------------------------------------------------
#  power & sqrt methods for "fd"
#  -------------------------------------------------------------------

# see exponentiate.fd

# http://r.789695.n4.nabble.com/warning-creating-an-as-array-method-in-a-package-td3080309.html
#
# need the following to eliminate a goofy warning in R CMD check
#    Warning:  found an S4 version of 'mean'
#    so it has not been imported correctly
mean <- function(x, ...)UseMethod('mean')

#  ----------------------------------------------------------------
#       mean for fd class
#  ----------------------------------------------------------------

mean.fd <- function(x, ...)
{
  if(!inherits(x, 'fd'))
    stop("'x' is not of class 'fd'")
#
  coef      <- x$coefs
  coefd     <- dim(coef)
  ndim      <- length(coefd)
  basisobj  <- x$basis
  nbasis    <- basisobj$nbasis
  dropind   <- basisobj$dropind
  ndropind  <- length(dropind)
  if (ndim == 2) {
    coefmean  <- matrix(apply(coef,1,mean),nbasis-ndropind,1)
    coefnames <- list(dimnames(coef)[[1]],"Mean")
  } else {
    nvar <- coefd[3]
    coefmean  <- array(0,c(coefd[1],1,nvar))
    for (j in 1:nvar) coefmean[,1,j] <- apply(coef[,,j],1,mean)
    coefnames <- list(dimnames(coef)[[1]], "Mean", dimnames(coef)[[3]])
  }
  fdnames <- x$fdnames
  fdnames[[2]] <- "mean"
  fdnames[[3]] <- paste("mean",fdnames[[3]])
  meanfd <- fd(coefmean, basisobj, fdnames)
#
  meanfd
}

#  ----------------------------------------------------------------
#       sum for fd class
#  ----------------------------------------------------------------

sum.fd <- function(..., na.rm=FALSE)
{
  #  Compute sum function for functional observations

  #  Last modified 15 January 2013

  fd <- list(...)[[1]]

  if (!(inherits(fd, "fd")))
    stop("Argument FD not a functional data object.")

  coef   <- fd$coefs
  coefd  <- dim(coef)
  ndim   <- length(coefd)
  basis  <- fd$basis
  nbasis <- basis$nbasis
  dropind   <- basis$dropind
  ndropind  <- length(dropind)
  if (ndim == 2) {
    coefsum   <- matrix(apply(coef,1,sum),nbasis-ndropind,1)
    coefnames <- list(dimnames(coef)[[1]],"Sum")
  } else {
    nvar <- coefd[3]
    coefsum  <- array(0,c(coefd[1],1,nvar))
    for (j in 1:nvar) coefsum[,1,j] <- apply(coef[,,j],1,sum)
    coefnames <- list(dimnames(coef)[[1]], "Sum", dimnames(coef)[[3]])
  }
  fdnames <- fd$fdnames
  fdnames[[2]] <- "1"
  names(fdnames)[2] <- "Sum"
  names(fdnames)[3] <- paste("Sum",names(fdnames)[3])
  sumfd <- fd(coefsum, basis, fdnames)

  sumfd
}

#  ----------------------------------------------------------------
#       c for fd class
#  ----------------------------------------------------------------

"c.fd"<- function(...)
{
#
#   concatenates a number of .fd objects.  It is assumed that all the
#   objects have the same basisfd objects, and that all the coef arrays
#   have the same number of dimensions
#

#  Last modified 17 September 2005

  fdlist <- list(...)
  n      <- length(fdlist)
  fd1    <- fdlist[[1]]
  if (n == 1) return(fd1)
  coef    <- fd1$coefs
  coefd   <- dim(coef)
  ndim    <- length(coefd)
  basisfd <- fd1$basis
  fdnames <- fd1$fdnames
  	#  check that the fd objects are consistent with each other
  if(!inherits(fd1, "fd")) stop("Objects must be of class fd")
  for(j in (2:n)) {
    fdj <- fdlist[[j]]
    if(!inherits(fdj, "fd")) stop("Objects must be of class fd")
    if(any(unlist(fdj$basis) != unlist(basisfd)))
      stop("Objects must all have the same basis")
    if(length(dim(fdj$coefs)) != ndim)
      stop("Objects must all have the same number of multiple functions")
  }
  	#  concatenate by concatenate coefficient matrices
  if (ndim == 2) {
    for (j in 2:n) {
      nameslist <- dimnames(coef)
      fdj       <- fdlist[[j]]
      coefj     <- fdj$coefs
      coef      <- cbind(coef, coefj)
      nameslist[[2]] <- c(nameslist[[2]], dimnames(coefj)[[2]])
    }
  } else {
    for(j in (2:n)) {
      nameslist <- dimnames(coef)
      fdj       <- fdlist[[j]]
      coefj     <- fdj$coefs
      coef      <- c(coef, aperm(coefj, c(1, 3, 2)))
      nameslist[[2]] <- c(nameslist[[2]], dimnames(coefj)[[2]])
    }
    dim(coef) <- c(coefd[1], coefd[3],
                   length(coef)/(coefd[1] * coefd[3]))
    coef <- aperm(coef, c(1, 3, 2))
  }
  dimnames(coef) <- nameslist
  concatfd <- fd(coef, basisfd, fdnames)
  return(concatfd)
}
fourierpen <- function(basisobj, Lfdobj=int2Lfd(2))
{

  #  Computes the Fourier penalty matrix.
  #  Arguments:
  #  BASISOBJ ... a basis object of type "fourier"
  #  LFDOBJ   ... either the order of derivative or a
  #                linear differential operator to be penalized.
  #  Returns  the penalty matrix.

  #  Note:  The number of basis functions is always odd.  If BASISOBJ
  #  specifies an even number of basis functions, then the number of basis
  #  functions is increased by 1, and this function returns a matrix of
  #  order one larger.

  #  Last modified 9 February 2007

  if (!(inherits(basisobj, "basisfd"))) stop(
		"First argument is not a basis object.")

  nbasis <- basisobj$nbasis
  if (2*(nbasis %/% 2) == nbasis) basisobj$nbasis <- nbasis + 1

  type <- basisobj$type
  if (type != "fourier") stop ("Wrong basis type")

  Lfdobj=int2Lfd(Lfdobj)

  width  <- basisobj$rangeval[2] - basisobj$rangeval[1]
  period <- basisobj$params[1]
  ratio  <- round(width/period)
  nderiv <- Lfdobj$nderiv

  if ((width/period) == ratio && is.integerLfd(Lfdobj)) {

    #  Compute penalty matrix for penalizing integral over one period.

    penaltymatrix <- diag(pendiagfn(basisobj, nderiv))

  } else {

    #  Compute penalty matrix by numerical integration

    penaltymatrix <- inprod(basisobj, basisobj, Lfdobj, Lfdobj)

  }

  return( penaltymatrix )
}

#  ------------------------------------------------------------------

pendiagfn <- function(basisobj, nderiv) {

    nbasis  <- basisobj$nbasis
    period  <- basisobj$params[1]
    rangev  <- basisobj$rangeval
    omega   <- 2*pi/period
    halfper <- period/2
    twonde  <- 2*nderiv
    pendiag <- rep(0,nbasis)
    if (nderiv == 0) pendiag[1] <- period/2.0 else pendiag[1] <- 0
    j   <- seq(2,nbasis-1,2)
    fac <- halfper*(j*omega/2)^twonde
    pendiag[j]   <- fac
    pendiag[j+1] <- fac
    pendiag <- 2*pendiag/period
    return(pendiag)
}
fourier <- function(x, nbasis = n, period = span, nderiv = 0)
{
  #  Computes the NDERIV derivative of the Fourier series basis
  #    for NBASIS functions with period PERIOD, these being evaluated
  #    at values in vector X
  #  Returns an N by NBASIS matrix of function values
  #  Note:  The number of basis functions always odd.  If the argument
  #   NBASIS is even, it is increased by one.

  #  last modified 23 February 2007 by Spencer Graves
  #  Previously modified 15 December 2005

  #  check x and set up range

  xNames <- names(x)
#
  x      <- as.vector(x)
  n      <- length(x)
  onen   <- rep(1,n)
  xrange <- range(x)
  span   <- xrange[2] - xrange[1]

  #  check period and set up omega

  if (period <= 0) stop("PERIOD not positive.")
  omega  <- 2*pi/period
  omegax <- omega*x

  #  check nbasis

  if (nbasis <= 0) stop("NBASIS not positive")

  #  check nderiv

  if (nderiv <  0) stop("NDERIV is negative.")

  #  if nbasis is even, add one

  if ((nbasis %/% 2)*2 == nbasis) nbasis <- nbasis + 1

  #  compute basis matrix

  basismat <- matrix(0,n,nbasis)
  if (nderiv == 0) {
    #  The fourier series itself is required.
    basismat[,1] <- 1/sqrt(2)
    if(nbasis>1){
      j    <- seq(2,nbasis-1,2)
      k    <- j/2
      args <- outer(omegax,k)
      basismat[,j]   <- sin(args)
      basismat[,j+1] <- cos(args)
    }
  } else {
    #  The derivative of order nderiv is required.
    basismat[,1] <- 0.0
    if(nbasis>1){
      if (nderiv == (nderiv %/% 2)*2) {
        mval  <- nderiv/2
        ncase <- 1
      } else {
        mval <- (nderiv-1)/2
        ncase <- 2
      }
      j    <- seq(2,nbasis-1,2)
      k    <- j/2
      fac  <- outer(onen,((-1)^mval)*(k*omega)^nderiv)
      args <- outer(omegax,k)
      if (ncase == 1) {
        basismat[,j]   <-  fac * sin(args)
        basismat[,j+1] <-  fac * cos(args)
      } else {
        basismat[,j]   <-  fac * cos(args)
        basismat[,j+1] <- -fac * sin(args)
      }
    }
  }
  basismat <- basismat/sqrt(period/2)
#
  fNames <- "const"
  n2 <- (nbasis %/%2)
  if(n2>0){
    SC <- outer(c("sin", "cos"), 1:n2, paste, sep="")
    fNames <- c(fNames, as.vector(SC))
  }
#
  dimnames(basismat) <- list(xNames, fNames)
#
  return(basismat)
}
Fperm.fd <- function(yfdPar, xfdlist, betalist, wt=NULL,
            nperm=200,argvals=NULL,q=0.05,plotres=TRUE,...)
{
# yfdpar, xfdList, betalist, wt = standard inputs to fRegress
# nperm = number of permutations,
# argvals = where to evaluate functional responses,
# q =  quantile to compare
# plotres:  Do we plot the results?

  Fnull     = rep(0,nperm)
  Fnullvals = c()

  q = 1-q

  begin <- proc.time()
  fRegressList <- fRegress(yfdPar, xfdlist, betalist)
  elapsed.time <- max(proc.time()-begin,na.rm=TRUE)

  if( elapsed.time > 30/nperm ){
    print(paste('Estimated Computing time =',
                round(nperm*elapsed.time),'seconds.'))
  }

  yhat <- fRegressList$yhatfdobj
  if(is.null(yhat)){ yhat = fRegressList$yhat }
  if(is.list(yhat) && ('fd' %in% names(yhat))) yhat <- yhat$fd

  tFstat <- Fstat.fd(yfdPar,yhat,argvals)

  Fvals <- tFstat$F
  Fobs = max(Fvals)

  argvals = tFstat$argvals

  if(is.vector(yfdPar)){ 
      n = length(yfdPar) 
  }  else { 
      n = ncol(yfdPar$coefs) 
  }

  for(i in 1:nperm){

    tyfdPar = yfdPar[sample(n)]

    fRegressList <- fRegress(tyfdPar, xfdlist, betalist)

    if(is.fd(yhat)){
		yhat <- fRegressList$yhatfdobj
		if(is.list(yhat) && ('fd' %in% names(yhat))) yhat <- yhat$fd
	} else{ yhat = fRegressList$yhat }

    tFstat = Fstat.fd(tyfdPar,yhat,argvals)

    Fnullvals <- cbind(Fnullvals,tFstat$F)

    Fnull[i] = max(Fnullvals[,i])
  }

    pval = mean( Fobs < Fnull )
    qval = quantile(Fnull,q)

    pvals.pts = apply(Fvals<Fnullvals,1,mean)
    qvals.pts = apply(Fnullvals,1,quantile,q)

    if(plotres){
        if(is.fd(yfdPar)){
            ylims = c(min(c(Fvals,qval,qvals.pts)),max(c(Fobs,qval)))

		if( is.null(names(yhat$fdnames)) ){ xlab = 'argvals' }
		else{ xlab = names(yhat$fdnames)[1] }

            plot(argvals,Fvals,type="l",ylim=ylims,col=2,lwd=2,
			xlab=xlab,ylab='F-statistic',main='Permutation F-Test',...)
            lines(argvals,qvals.pts,lty=3,col=4,lwd=2)
            abline(h=qval,lty=2,col=4,lwd=2)

	      legendstr = c('Observed Statistic',
		    paste('pointwise',1-q,'critical value'),
		    paste('maximum',1-q,'critical value'))

		legend(argvals[1],ylims[2],legend=legendstr,col=c(2,4,4),
			lty=c(1,3,2),lwd=c(2,2,2))
        }
        else{
            xlims = c(min(c(Fnull,Fobs)),max(c(Fnull,Fobs)))
            hstat = hist(Fnull,xlim=xlims,lwd=2,xlab='F-value',
			       main = 'Permutation F-Test')
            abline(v = Fobs,col=2,lwd=2)
            abline(v = qval,col=4,lty=2,lwd=2)

	      legendstr = c('Observed Statistic',
		    paste('Permutation',1-q,'critical value'))

		legend(xlims[1],max(hstat$counts),legend=legendstr,col=c(2,4),
			lty=c(1,2),lwd=c(2,2))
        }
    }

    return(list(pval=pval,qval=qval,Fobs=Fobs,Fnull=Fnull,
        Fvals=Fvals,Fnullvals=Fnullvals,pvals.pts=pvals.pts,qvals.pts=qvals.pts,
        fRegressList=fRegressList,argvals=argvals))
}
fRegressArgCheck <- function(yfd, xfdlist, betalist, wt=NULL) 
{
  #  FREGRESS_ARGCHECK checks the first four arguments for the functions
  #  for function regression, including FREGRESS.
  
  #  Last modified 16 December 2020 by Jim Ramsay
  
  #  --------------------  Check classes of arguments  --------------------
  
  #  check that YFD is of class either 'fd' or 'numeric' and compute sample size N
  
  if (!(is.fdPar(yfd) || is.fd(yfd) || is.numeric(yfd) || is.matrix(yfd))) stop(
    "First argument is not of class 'fdPar', 'fd', 'numeric' or 'matrix'.")
  
  #  As of 2020, if yfd is an fdPar object, it is converted to an fd object.
  #  The added structure of the fdPar class is not used in any of the fRegress codes.
  # The older versions of fda package used yfdPar as the name for the first member.
  
  if (is.fdPar(yfd)) yfd <- yfd$fd
  
  if (inherits(yfd, "fd")) {
    ycoef <- yfd$coefs
    N     <- dim(ycoef)[2]
  } else {
    N <- length(yfd)
  } 
  
  #  check that xfdlist is a list object and compute number of covariates p
  
  #  check XFDLIST
  
  if (inherits(xfdlist, "fd") || inherits(xfdlist, "numeric")) 
    xfdlist <- list(xfdlist)
  
  if (!inherits(xfdlist, "list")) stop(
    "Argument XFDLIST is not a list object.")
  
  #  get number of independent variables p
  
  p <- length(xfdlist)
  
  #  check BETALIST
  
  if (inherits(betalist, "fd")) betalist <- list(betalist)
  
  if (!inherits(betalist, "list")) stop(
    "Argument BETALIST is not a list object.")
  
  if (length(betalist) != p)  {
    cat(paste("\nNumber of regression coefficients does not match\n",
              "number of independent variables."))
    stop("")
  }
    
  #  extract the range if YFD is functional
  
  if (inherits(yfd, "fd")) {
    rangeval <- yfd$basis$rangeval
  } else {
    rangeval = c(0,1)
  #   allscalar <- TRUE
  #   for (j in 1:p) {
  #     if (inherits(xfdlist[[j]], "fd")) {
  #       rangeval <- xfdlist[[j]]$basis$rangeval            
  #       allscalar <- FALSE
  #       break
  #     }
  #   }
    # if (allscalar) stop(
    #   paste("The dependent variable and all the independent",   
    #         "variables are scalar."))
  }
  
  #  --------------------  check contents of XFDLIST  -------------------
  
  #  If the object is a vector of length N,
  #  it is converted to a functional data object with a
  #  constant basis
  
  onebasis <- create.constant.basis(rangeval)
  onesfd   <- fd(1,onebasis)
  
  xerror <- FALSE
  for (j in 1:p) {
    xfdj <- xfdlist[[j]]
    if (inherits(xfdj, "fd")) {
      xcoef <- xfdj$coefs
      if (length(dim(xcoef)) > 2) stop(
        paste("Covariate",j,"is not univariate."))
      #  check size of coefficient array
      Nj <- dim(xcoef)[2]
      if (Nj != N) {
        print(
          paste("Incorrect number of replications in XFDLIST",
                "for covariate",j))
        xerror = TRUE
      }
    } 
    if (inherits(xfdj, "numeric")) {
      if (!is.matrix(xfdj)) xfdj = as.matrix(xfdj)
      Zdimj <- dim(xfdj)
      if (Zdimj[1] != N && Zdimj != 1) {
        print(paste("Vector in XFDLIST[[",j,"]] has wrong length."))
        xerror = TRUE 
      } 
      if (Zdimj[2] != 1) {
        print(paste("Matrix in XFDLIST[[",j,"]] has more than one column."))
        xerror = TRUE 
      } 
      xfdlist[[j]] <- fd(matrix(xfdj,1,N), onebasis)
    } 
    if (!(inherits(xfdlist[[j]], "fd"     ) || 
          inherits(xfdlist[[j]], "numeric") ||
          inherits(xfdlist[[j]], "matrix" ))) {
      print(paste("XFDLIST[[",j,"]] is not an FD or numeric or matrix object."))
      xerror = TRUE
    }
  }
  
  #  --------------------  check contents of BETALIST  -------------------
  
  berror <- FALSE
  for (j in 1:p) {
    betafdParj <- betalist[[j]]
    if (inherits(betafdParj, "fd") || inherits(betafdParj, "basisfd")) {
      betafdParj    <- fdPar(betafdParj)
      betalist[[j]] <- betafdParj
    }
    if (!inherits(betafdParj, "fdPar")) {
      print(paste("BETALIST[[",j,"]] is not a FDPAR object."))
      berror <- TRUE
    }
  }
  
  if (xerror || berror) stop(
    "An error has been found in either XFDLIST or BETALIST.")
  
  #  --------------------  check contents of WEIGHTS  -------------------
  
  if (is.null(wt)) wt = rep(1,N)
  if (length(wt) != N) stop("Number of weights not equal to N.")
  if (any(wt < 0))     stop("Negative weights found.")
  
  #  ---------------------  return the argument list  --------------------
  
  # The older versions of fda package used yfdPar as the name for the first member.
  
  return(list(yfd=yfd, xfdlist=xfdlist, betalist=betalist, wt=wt))
  
}

fRegress.CV <- function(y, xfdlist, betalist, wt=NULL, CVobs=1:N,
                        returnMatrix=FALSE, ...)
{

# FREGRESS.CV computes cross-validated error sum of squares
# for scalar or functional responses. NOTE: ordinary and
# generalized cross validation scores are now returned by fRegress
# when scalar responses are used.

# last modified 16 December 2020 by Jim Ramsay

#  check the arguments

argList  <- fRegressArgCheck(y, xfdlist, betalist, wt)

yfdobj   <- argList$yfd
xfdlist  <- argList$xfdlist
betalist <- argList$betalist
wt       <- argList$wt

# extract dimensions of the data and analysis

p <- length(xfdlist)
N <- dim(xfdlist[[1]]$coef)[2]
M <- length(CVobs)

#  branch to either scalar or functional dependent variable

if (inherits(yfdobj, "numeric"))  {

    #  scalar dependent variable case

    yvec   <- yfdobj
    SSE.CV <- 0
    errfd  <- c()
    for (m in 1:M) {
      i        <- CVobs[m]
      #  eliminate case i from the weights
      wti <- wt[-i]
      xfdlisti <- vector("list",p)
      for (j in 1:p) {
        xfdj          <- xfdlist[[j]]
        if (inherits(xfdj, "numeric")) {
          betafdParj <- betalist[[j]]
          betafdj    <- betafdParj$fd
          basisj     <- betafdj$basis
          betarangej <- basisj$rangeval
          conbasisj  <- create.constant.basis(betarangej)
          xfdj       <- fd(matrix(xfdj,1,N), conbasisj)
        }
        basisj <- xfdj$basis
        coefj  <- xfdj$coefs
        if (dim(coefj)[1] == 1) coefj <- matrix(coefj[-i],1,N-1)
        else                    coefj <- as.matrix(coefj[,-i])
        xfdlisti[[j]] <- fd(coefj,basisj)
      }
      yveci         <- yvec[-i]
      fRegressListi <- fRegress(yveci, xfdlisti, betalist, wti)
      betaestlisti  <- fRegressListi$betaestlist
      yhati <- 0
      for (j in 1:p) {
        betafdParj <- betaestlisti[[j]]
        betafdj    <- betafdParj$fd
        xfdj       <- xfdlist[[j]]
        bbasisj    <- betafdj$basis
        rangej     <- bbasisj$rangeval
        nfine      <- max(501, bbasisj$nbasis*10+1)
        tfine      <- seq(rangej[1], rangej[2], len=nfine)
        delta      <- tfine[2]-tfine[1]
        betavec    <- eval.fd(tfine, betafdj, 0, returnMatrix)
        xveci      <- eval.fd(tfine, xfdj[i], 0, returnMatrix)
        yhati      <- yhati + delta*(sum(xveci*betavec) -
                                    0.5*( xveci[1]    *betavec[1] +
                                          xveci[nfine]*betavec[nfine] ))
      }
      errfd[i] = yvec[i] - yhati;
      SSE.CV <- SSE.CV + errfd[i]^2
    }
 } else {

    #  functional dependent variable case

    yfd      <- yfdobj
    SSE.CV   <- 0
    errcoefs <- c()
    for(m in 1:length(CVobs)){
      # index of case to eliminate
      i <-  CVobs[m]
      # eliminate case i from the weights
      wti <- wt[-i]
      # eliminate case i from covariates
      txfdlist <- xfdlist
      for(k in 1:p){
        txfdlist[[k]] <- xfdlist[[k]][-i]
      }
      # eliminate case i from dependent variable
      yfdi <- yfd[-i]
      # carry out the functional regression analysis
      tres <- fRegress(yfdi,txfdlist,betalist,wti)
      #  extract the regression coefficient functions
      betaestlisti <- tres$betaestlist
      #  compute the fit to the data for case i
      yhatfdi <- 0
      for(k in 1:p){
        betafdPark = betaestlisti[[k]]
        betafdk    = betafdPark$fd
        xfdk       = xfdlist[[k]]
        xfdik      = xfdk[i]
        tempfd     = xfdik*betafdk
        yhatfdi <- yhatfdi + tempfd
      }
      #  compute the residual function
      errfdi   <- yfd[i] - yhatfdi
      #  increment the error sum of squares by the integral of the
      #  square of the residual function
      SSE.CV   <- SSE.CV + inprod(errfdi,errfdi)
      #  add the coefficients for the residual function
      errcoefs <- cbind(errcoefs,errfdi$coefs)
    }
    #  set up the functional data object for the residual fns
    errfd <- fd(errcoefs,errfdi$basis)
    names(errfd$fdnames)[[3]] <- "Xval Errors"
}
return(list(SSE.CV=SSE.CV, errfd.cv=errfd))
}


fRegress.double <- function(y, xfdlist, betalist, wt=NULL,
                             y2cMap=NULL, SigmaE=NULL, returnMatrix=FALSE, ...)
{
  
  #  FREGRESS.DOUBLE  Fits a scalar dependent variable using the concurrent
  #                    functional regression model using inner products
  #                    of functional covariates and functional regression
  #                    functions.
  #
  #  Arguments:
  #  Y        ... an object for the dependent variable,
  #               which  be a numeric vector
  #  XFDLIST  ... a list object of length p with each list
  #               containing an object for an independent variable.
  #               the object may be:
  #                   a functional data object or
  #                   a vector
  #               if XFDLIST is a functional data object or a vector,
  #               it is converted to a list of length 1.
  #  BETALIST ... a list object of length p with each list
  #               containing a functional parameter object for
  #               the corresponding regression function.  If any of
  #               these objects is a functional data object, it is
  #               converted to the default functional parameter object.
  #               if BETALIST is a functional parameter object
  #               it is converted to a list of length 1.
  #  WT       ... a vector of nonnegative weights for observations
  #  Y2CMAP   ... the matrix mapping from the vector of observed values
  #               to the coefficients for the dependent variable.
  #               This is output by function SMOOTH_BASIS.  If this is
  #               supplied, confidence limits are computed, otherwise not.
  #  SIGMAE   ... Estimate of the covariances among the residuals.  This
  #               can only be estimated after a preliminary analysis
  #               with .
  #  RETURNMATRIX ... If False, a matrix in sparse storage model can be returned
  #               from a call to function BsplineS.  See this function for
  #               enabling this option.
  #
  #  Returns LIST  ... A list containing seven members with names:
  #    yfdobj      ... first  argument of 
  #    xfdlist     ... second argument of 
  #    betalist    ... third  argument of 
  #    betaestlist ... estimated regression functions
  #    yhatfdobj   ... functional data object containing fitted functions
  #    Cmatinv     ... inverse of the coefficient matrix, needed for
  #                    function .STDERR that computes standard errors
  #    wt          ... weights for observations
  #    df          ... degrees of freedom for fit
  #  This list object is converted to a class with the name ""
  #  function predict. is an example of a method that can be called simply
  #  as predict(List).  In this call List can be any object of the
  #  "".
  
  # Last modified 16 December 2020 by Jim Ramsay
  
  #  check Y and compute sample size N
  
  if (!inherits(y, "numeric")) stop("Y is not a numeric vector.")
    
  #  ----------------------------------------------------------------
  #                   yfdobj is scalar or multivariate
  #  ----------------------------------------------------------------
    
  #  As of 2020, if yfd is an fdPar object, it is converted to an fd object.
  #  The added structure of the fdPar class is not used in any of the fRegress codes.
  #  The older versions of fda package used yfdPar as the name for the first member.

  arglist <- fRegressArgCheck(y, xfdlist, betalist, wt)
  
  yfdobj   <- arglist$yfd
  xfdlist  <- arglist$xfdlist
  betalist <- arglist$betalist
  wt       <- arglist$wt
  
  ymat <- as.matrix(y)
  N    <- dim(ymat)[1]
  p    <- length(xfdlist)
    
  Zmat  <- NULL
  Rmat  <- NULL
  pjvec <- rep(0,p)
  ncoef <- 0
  for (j in 1:p) {
    xfdj       <- xfdlist[[j]]
    if (!inherits(xfdj, "fd")) {
      stop(paste("Independent variable",j,"is not of class fd."))
    }
    xcoef      <- xfdj$coefs
    xbasis     <- xfdj$basis
    betafdParj <- betalist[[j]]
    bbasis     <- betafdParj$fd$basis
    bnbasis    <- bbasis$nbasis
    pjvec[j]   <- bnbasis
    Jpsithetaj <- inprod(xbasis,bbasis)
    Zmat       <- cbind(Zmat,crossprod(xcoef,Jpsithetaj))
    if (betafdParj$estimate) {
      lambdaj    <- betafdParj$lambda
      if (lambdaj > 0) {
        Lfdj  <- betafdParj$Lfd
        Rmatj <- lambdaj*eval.penalty(bbasis,Lfdj)
      } else {
        Rmatj <- matrix(0,bnbasis,bnbasis)
      }
      if (ncoef > 0) {
        zeromat <- matrix(0,ncoef,bnbasis)
        Rmat    <- rbind(cbind(Rmat,       zeromat),
                         cbind(t(zeromat), Rmatj))
		ncoef <- ncoef + bnbasis
      } else {
        Rmat  <- Rmatj
        ncoef <- ncoef + bnbasis
      }
    }
  }
  
  #  -----------------------------------------------------------
  #          set up the linear equations for the solution
  #  -----------------------------------------------------------
  
  #  solve for coefficients defining BETA
  
  if (any(wt != 1)) {
    rtwt   <- sqrt(wt)
    Zmatwt <- Zmat*rtwt
    ymatwt <- ymat*rtwt
    Cmat   <- t(Zmatwt) %*% Zmatwt + Rmat
    Dmat   <- t(Zmatwt) %*% ymatwt
  } else {
    Cmat <- t(Zmat) %*% Zmat + Rmat
    Dmat <- t(Zmat) %*% ymat
  }
  
  eigchk(Cmat)
  
  Cmatinv  <- solve(Cmat)
  
  betacoef <- Cmatinv %*% Dmat
  
#    df <- sum(diag(Zmat %*% Cmatinv %*% t(Zmat)))

    hatvals = diag(Zmat %*% Cmatinv %*% t(Zmat))
    df <- sum(hatvals)
  
  #  set up fdPar object for BETAESTFDPAR
  
  betaestlist <- betalist
  mj2 <- 0
  for (j in 1:p) {
    betafdParj <- betalist[[j]]
    betafdj    <- betafdParj$fd
    ncoefj     <- betafdj$basis$nbasis
    mj1        <- mj2 + 1
    mj2        <- mj2 + ncoefj
    indexj     <- mj1:mj2
    betacoefj        <- betacoef[indexj]
    betaestfdj       <- betafdj
    betaestfdj$coefs <- as.matrix(betacoefj)
    betaestfdParj    <- betafdParj
    betaestfdParj$fd <- betaestfdj
    betaestlist[[j]] <- betaestfdParj
  }
  
  #  set up fd object for predicted values
  
  yhatmat <- matrix(0,N,1)
  for (j in 1:p) {
    xfdj <- xfdlist[[j]]
    if (inherits(xfdj, "fd")) {
      xbasis     <- xfdj$basis
      xnbasis    <- xbasis$nbasis
      xrng       <- xbasis$rangeval
      nfine      <- max(501,10*xnbasis+1)
      tfine      <- seq(xrng[1], xrng[2], len=nfine)
      deltat     <- tfine[2]-tfine[1]
      xmat       <- eval.fd(tfine, xfdj, 0, returnMatrix)
      betafdParj <- betaestlist[[j]]
      betafdj    <- betafdParj$fd
      betamat    <- eval.fd(tfine, betafdj, 0, returnMatrix)
      fitj       <- deltat*(crossprod(xmat,betamat) -
                              0.5*(outer(xmat[1,    ],betamat[1,    ]) +
                                     outer(xmat[nfine,],betamat[nfine,])))
      yhatmat    <- yhatmat + fitj
    } else{
      betaestfdParj <- betaestlist[[j]]
      betavecj      <- betaestfdParj$fd$coefs
      yhatmat       <- yhatmat + xfdj %*% t(betavecj)
    }
  }
  yhatfdobj <- yhatmat
  
    # Calculate OCV and GCV scores

    OCV = sum( (ymat-yhatmat)^2/(1-hatvals)^2 )
    GCV = sum( (ymat-yhatmat)^2 )/( (sum(1-hatvals))^2 )
  
  #  -----------------------------------------------------------------------
  #        Compute pointwise standard errors of regression coefficients
  #               if both y2cMap and SigmaE are supplied.
  #  -----------------------------------------------------------------------
  
  
  if (!(is.null(y2cMap) || is.null(SigmaE))) {
    
    #  check dimensions of y2cMap and SigmaE
    
    y2cdim <- dim(y2cMap)
    if (y2cdim[2] != dim(SigmaE)[1])  stop(
      "Dimensions of Y2CMAP not correct.")
    
    
    #  compute linear mapping c2bMap takinging coefficients for
    #  response into coefficients for regression functions
    
    c2bMap <- Cmatinv %*% t(Zmat)
    y2bmap <- c2bMap
    bvar   <- y2bmap %*% as.matrix(SigmaE) %*% t(y2bmap)
    betastderrlist <- vector("list",p)
    mj2 <- 0
    for (j in 1:p) {
      betafdParj <- betalist[[j]]
      betabasisj <- betafdParj$fd$basis
      ncoefj     <- betabasisj$nbasis
      mj1        <- mj2 + 1
      mj2        <- mj2 + ncoefj
      indexj     <- mj1:mj2
      bvarj      <- bvar[indexj,indexj]
      betarng    <- betabasisj$rangeval
      nfine      <- max(c(501,10*ncoefj+1))
      tfine      <- seq(betarng[1], betarng[2], len=nfine)
      bbasismat  <- eval.basis(tfine, betabasisj, 0, returnMatrix)
      bstderrj   <- sqrt(diag(bbasismat %*% bvarj %*% t(bbasismat)))
      bstderrfdj <- smooth.basis(tfine, bstderrj, betabasisj)$fd
      betastderrlist[[j]] <- bstderrfdj
    }
  } else {
    betastderrlist = NULL
    bvar           = NULL
    c2bMap         = NULL
  }
  
  #  -----------------------------------------------------------------------
  #                  Set up output list object
  #  -----------------------------------------------------------------------
  
  fRegressList <-
    list(yfdobj         = y,
         xfdlist        = xfdlist,
         betalist       = betalist,
         betaestlist    = betaestlist,
         yhatfdobj      = yhatfdobj,
         Cmat           = Cmat,
         Dmat           = Dmat,
         Cmatinv        = Cmatinv,
         wt             = wt,
         df             = df,
		 GCV			= GCV,
		 OCV			= OCV,
         y2cMap         = y2cMap,
         SigmaE         = SigmaE,
         betastderrlist = betastderrlist,
         bvar           = bvar,
         c2bMap         = c2bMap)
  
  class(fRegressList) <- "fRegress"
  
  return(fRegressList)
  
}

fRegress.character <- function(y, data=NULL, betalist=NULL,
                               wt=NULL, y2cMap=NULL, SigmaE=NULL,
                               method=c('fRegress', 'model'),
                               sep='.', ...) {
  fRegress.formula(y=y, data=data, betalist=betalist,
                   wt=wt, y2cMap=y2cMap, SigmaE=SigmaE,
                   method=method, sep=sep, ...)
}

fRegress.formula <- function(y, data=NULL, betalist=NULL,
                             wt=NULL, y2cMap=NULL, SigmaE=NULL,
                             method=c('fRegress', 'model'),
                             sep='.', ...) {
  
  #  Last modified 1 November 2020 by Jim Ramsay
  
  ##
  ## 1.  get y = left hand side of the formula
  ##
  Formula <- y                    #  character vector
  yName   <- Formula[[2]]         #  name of dependent variable object
  yNm     <- as.character(yName)  #  character strong for name
  #  check name of dependent variable
  if(!inherits(yName, 'name') || (length(yNm) > 1))
    stop('The left hand side of formula must be a simple object; ',
         ' instead, LHS = ', as.character(Formula)[2],
         ', which has class = ', class(yName))
  #
  
  ##
  ## 2.  check the data argument
  ##
  
  dataNames <- names(data)
  #  extract name of dependent variable
  y <- {
    if (yNm %in% dataNames) data[[yNm]]
    else get(yNm)  # Search by name for object yNm
  }
  #  get the range of the  dependent variable  and 
  #  obtain the dimensions of the coefficient matrix if y is of class 'fd'
  #  obtain the dimensions of the y if y is of class 'numeric'
  trng <- NULL
  {
    if(inherits(y, 'fd')){
      ydim <- dim(y$coefs)
      if(is.null(ydim) || (length(ydim)<2)) {
        y$coefs <- as.matrix(y$coefs)
        ydim   <- dim(y$coefs)
      }
      ny   <- ydim[2]
      trng <- y$basis$rangeval
    } else{
      if(inherits(y, 'numeric')){
        ydim <- dim(y)
        if(is.null(ydim))
          ny <- length(y)
        else
          ny <- ydim[1]
      }
      else
        stop('The left hand side of formula must have class ',
             'numeric or fd;  instead is ', class(y))
    }
  }
  
  ##
  ## 3.  check the formula for excessive complexity
  ##
  
  allVars  <- all.vars(Formula)
  xNms     <- allVars[allVars != yNm]
  Terms    <- terms(Formula)
  termLbls <- attr(Terms, 'term.labels')
  oops     <- which(!(termLbls %in% xNms))
  if(length(oops) > 0)
    stop('formula contains terms that fRegress can not handle; ',
         ' the first one = ', termLbls[oops[1]])
  #
  k1 <- length(allVars)
  type <- rep(NA,k1)
  names(type) <- allVars
  nbasis      <- type
  if(inherits(y, 'fd')){
    type[1] <- y$basis$type
    nb      <- y$basis$nbasis
    if(!is.null(nb)) nbasis[1] <- nb
  }
  
  ##
  ## 4.  Inventory the right hand side
  ##
  
  k0       <- length(xNms)
  xfdlist0 <- vector('list', k0)
  names(xfdlist0) <- xNms
  xNames          <- xfdlist0
  nVars           <- rep(NA, k0)
  names(nVars)    <- xNms
  oops <- FALSE
  for(i in 1:k0) {
    xNm <- xNms[i]
    xi <- {
      if(xNm %in% dataNames) data[[xNm]]
      else get(xNm)
    }
    {
      if(class(xi) %in% c('fd', 'fdPar')) {
        xj <- {
          if(inherits(xi, 'fd')) xi
          else xi$fd
        }
        xrng <- xj$basis$rangeval
        {
          if(is.null(trng))
            trng <- xrng
          else
            if(any(xrng != trng)){
              oops <- TRUE
              cat('incompatible rangeval found in ', xNm,
                  '$rangeval = ', paste(xrng, collapse=', '),
                  ' != previous = ', paste(trng, collapse=', '),
                  sep='')
            }
        }
        xdim <- dim(xj$coefs)
        {
          if(is.null(xdim) || (length(xdim)<2)){
            xj$coefs <- as.matrix(xj$coefs)
            xdim <- dim(xj$coefs)
            nxi <- xdim[2]
            nVars[i] <- 1
            xNames[[i]] <- xNm
          }
          else {
            if(length(xdim)<3){
              nxi <- xdim[2]
              nVars[i] <- 1
              xNames[[i]] <- xNm
            }
            else {
              nxi <- xdim[2]
              if(length(xdim)<4){
                nVars[i] <- xdim[3]
                xNmsi <- dimnames(xj$coefs)[[3]]
                {
                  if(is.null(xNmsi))
                    xNames[[i]] <- paste(xNm, 1:xdim[3], sep=sep)
                  else
                    xNames[[i]] <- paste(xNm, xNmsi, sep=sep)
                }
              }
              else {
                oops <- TRUE
                cat(xNm, 'has too many levels:  dim(x$coefs) =',
                    paste(xdim, collapse=', '))
              }
            }
          }
        }
        type[i+1] <- xj$basis$type
        nb <- xj$basis$nbasis
        if(!is.null(nb))nbasis[i+1] <- nb
        xfdlist0[[i]] <- xi
      }
      else {
        if(is.numeric(xi)){
          xdim <- dim(xi)
          {
            if(is.null(xdim) || (length(xdim)<2)){
              nxi <- length(xi)
              nVars[i] <- 1
              xNames[[i]] <- xNm
            }
            else {
              nxi <- xdim[1]
              {
                if(length(xdim)<3){
                  nVars[i] <- xdim[2]
                  xNmsi <- dimnames(xi)[[2]]
                  {
                    if(is.null(xNmsi))
                      xNames[[i]] <- paste(xNm, 1:xdim[2], sep=sep)
                    else
                      xNames[[i]] <- paste(xNm, xNmsi, sep=sep)
                  }
                }
                else{
                  oops <- TRUE
                  cat(xNm, 'has too many levels:  dim(x) =',
                      paste(xdim, collapse=', '))
                }
              }
            }
          }
          xfdlist0[[i]] <- xi
        }
        else {
          if(inherits(xi, 'character'))
            xi <- factor(xi)
          {
            if(inherits(xi, 'factor')) {
              f.i <- formula(paste('~', xNm))
              Xi.df <- data.frame(xi)
              names(Xi.df) <- xNm
              Xi <- (model.matrix(f.i, Xi.df)[, -1, drop=FALSE])
              nxi <- dim(Xi)[1]
              xiNms <- dimnames(Xi)[[2]]
              nVars[i] <- length(xiNms)
              xNmLen <- nchar(xNm)
              xiLvls <- substring(xiNms, xNmLen+1)
              xNames[[i]] <- paste(xNm, xiLvls, sep=sep)
              xfdlist0[[i]] <- Xi
            }
            else{
              oops <- TRUE
              cat('ERROR:  variable', xNm, 'must be of class',
                  'fd, numeric, character or factor;  is', class(xi))
              nxi <- length(xi)
            }
            }
        }
      }
    }
    if(nxi != ny){
      cat('ERROR:  variable', xNm, 'has only',
          nxi, 'observations !=', ny,
          '= the number of observations of y.')
      oops <- TRUE
    }
  }
  if(oops)stop('illegal variable on the right hand side.')
  # If no functions found:
  if(is.null(trng)){
    warning("No functions found;  setting rangeval to 0:1")
    trng <- 0:1
  }
  
  ##
  ## 5.  Create xfdlist
  ##
  
  xL.L0   <- rep(1:k0, nVars)
  xNames. <- c('const', unlist(xNames))
  k <- 1+sum(nVars)
  xfdlist <- vector('list', k)
  names(xfdlist) <- xNames.
  #  create constfd for the intercept
  #  xfdlist[[1]] <- create.constant.basis(trng)
  xfdlist[[1]] <- rep(1, ny)
  i1 <- 1
  for(ix in 1:k0) {
    i0  <- i1+1
    xNm <- xNms[ix]
    xi  <- xfdlist0[[ix]]
    {
      if(inherits(xi, 'fd')) {
        if(nVars[ix]<2) {
          i1            <- i0
          xfdlist[[i0]] <- xi
        }
        else {
          #          i1 <- (i1+nVars[ix])
          for(i in 1:nVars[ix]){
            i1  <- i1+1
            xii <- xi
            xii$coefs <- xi$coefs[,,i, drop=FALSE]
            xfdlist[[i1]] <- xii
          }
        }
      }
      else {
        if(is.numeric(xi)) {
          if(nVars[ix]<2) {
            i1 <- i0
            xfdlist[[i0]] <- xi
          }
          else{
            for(i in 1:nVars[ix]) {
              i1 <- i1+1
              xfdlist[[i1]] <- xi[, i]
            }
          }
        }
      }
    }
  }
  
  ##
  ## 6.  check betalist or set up betalist
  ##
  
  {
    if(inherits(betalist, 'list')) {
      #  betalist is an argument
      if(length(betalist) != k)
        stop('length(betalist) = ', length(betalist),
             ';  must be ', k, ' to match length(xfdlist).')
      betaclass <- sapply(betalist, class)
      oops      <- which(betaclass != 'fdPar')
      if(length(oops)>0)
        stop('If betalist is a list, all components must have class ',
             'fdPar;  component ', oops[1], ' has class ',
             betaclass[oops[1]])
    }
    else {
      # betalist must be set up
      betalist <- vector('list', k)
      names(betalist) <- xNames.
      for(i in 1:k) {
        if(is.numeric(xfdlist[[i]])) {
          #  if xfdlist[[i]] is numeric, basis is set up using that  of dependent variable y
          if(is.numeric(y)) {
            bbasis        <- create.constant.basis(c(0,1))
            bfd           <- fd(basisobj=bbasis)
            betalist[[i]] <- fdPar(bfd)
          } else {
            #  if 'fd' use the basis of dependent variable
            if(inherits(y, 'fd')) {
              bfd           <- with(y, fd(basisobj=basis, fdnames=fdnames))
              betalist[[i]] <- fdPar(bfd)
            }
            else {
              bfd           <- with(y, fd(basisobj=basis, fdnames=fdnames))
              betalist[[i]] <- with(y, fdPar(bfd, Lfd, lambda, estimate, penmat))
            }
          }
        }
        else {
          #  use basis for the independent variable
          xfdi <- {
            if(i>1) xfdlist0[[xL.L0[i-1]]]
            else    xfdlist[[1]]
          }
          if(inherits(xfdi, 'fd')){
            bfd           <- with(xfdi, fd(basisobj=basis, fdnames=fdnames))
            betalist[[i]] <- fdPar(bfd)
          }
          else{
            bfd           <- with(xfdi$fd, fd(basisobj=basis, fdnames=fdnames))
            betalist[[i]] <- with(xfdi, fdPar(bfd, Lfd, lambda,
                                              estimate, penmat))
          }
        }
      }
    }
  }
  
  ##
  ## 7.  extract or set up weight
  ##
  
  {
    if(is.null(wt))
      wt <- rep(1, ny)
    else {
      if(length(wt) != ny)
        stop('length(wt) must match y;  length(wt) = ',
             length(wt), ' != number of y observations = ', ny)
      if(any(wt<0))
        stop('Negative weights found;  not allowed.')
    }
  }
  xiEnd   <- cumsum(nVars)
  xiStart <- c(1, xiEnd[-1])
  fRegressList <- list(y=y, xfdlist=xfdlist, betalist=betalist, wt=wt)
  
  ##
  ## 8.  either output argument list for fRegress() or invoke itcs
  ##
  
  method <- match.arg(method)
  if(method=='model') {
    return(fRegressList)
  } else {
    if(inherits(y, 'fd')) {
      do.call('fRegress.fd',    fRegressList)
    } else {
      do.call('fRegress.double', fRegressList)
    }
  }
}

fRegress <- function(y, ...) {
  UseMethod("fRegress")
}

fRegress.fd <- function(y, xfdlist, betalist, wt=NULL,
                        y2cMap=NULL, SigmaE=NULL, returnMatrix=FALSE,
                        method=c('fRegress', 'model'),
                        sep='.', ...) {
  
  #  FREGRESS  Fits a functional linear model using multiple
  #  functional independent variables with the dependency being
  #  pointwise or concurrent.
  #  The case of a scalar independent variable is included by treating
  #  it as a functional independent variable with a constant basis
  #  and a unit coefficient.
  #  
  #  Arguments:
  #  Y        ... an object for the dependent variable,
  #               which may be:
  #                   a functional data object or a numerical vector
  #  XFDLIST  ... a list object of length p with each list
  #               containing an object for an independent variable.
  #               the object may be:
  #                   a functional data object or
  #                   a vector
  #               if XFDLIST is a functional data object or a vector,
  #               it is converted to a list of length 1.
  #  BETALIST ... a list object of length p with each list
  #               containing a functional parameter object for
  #               the corresponding regression function.  If any of
  #               these objects is a functional data object, it is
  #               converted to the default functional parameter object.
  #               if BETALIST is a functional parameter object
  #               it is converted to a list of length 1.
  #  WT       ... a vector of nonnegative weights for observations
  #  Y2CMAP   ... the matrix mapping from the vector of observed values
  #               to the coefficients for the dependent variable.
  #               This is output by function SMOOTH_BASIS.  If this is
  #               supplied, confidence limits are computed, otherwise not.
  #  SIGMAE   ... Estimate of the covariances among the residuals.  This
  #               can only be estimated after a preliminary analysis
  #               with FREGRESS.
  #  RETURNMATRIX ... If False, a matrix in sparse storage model can be returned
  #               from a call to function BsplineS.  See this function for
  #               enabling this option.
  #
  #  Returns FREGRESSLIST  ... A list containing seven members with names:
  #    yfdobj      ... first  argument of FREGRESS
  #    xfdlist     ... second argument of FREGRESS
  #    betalist    ... third  argument of FREGRESS
  #    betaestlist ... estimated regression functions
  #    yhatfdobj   ... functional data object containing fitted functions
  #    Cmat        ... coefficient matrix for the linear system defining
  #                    the regression coefficient basis vector
  #    Dmat        ... right side vector for the linear system defining
  #                    the regression coefficient basis vector
  #    Cmatinv     ... inverse of the coefficient matrix, needed for
  #                    function FREGRESS.STDERR that computes standard errors
  #    wt          ... weights for observations
  #    df          ... degrees of freedom for fit
  #  This list object is converted to a class with the name "fRegress"
  #  function predict.fRegress is an example of a method that can be called simply
  #  as predict(fRegressList).  In this call fRegressList can be any object of the
  #  "fRegress".
  
  #  Last modified 5 November 2020 by Jim Ramsay
  
  if (is.fdPar(y)) y <- y$fd
  
  #  As of 2020, if yfd is an fdPar object, it is converted to an fd object.
  #  The added structure of the fdPar class is not used in any of the fRegress codes.
  #  The older versions of fda package used yfdPar as the name for the first member.
  
  arglist <- fRegressArgCheck(y, xfdlist, betalist, wt)
  
  yfdobj   <- arglist$yfd  # the older version used yfdPar as the name.
  xfdlist  <- arglist$xfdlist
  betalist <- arglist$betalist
  wt       <- arglist$wt
  
  p <- length(xfdlist)
  
  wtconst <- var(wt) == 0
  
  #  --------------------------------------------------------------------------
  #  branch depending on whether the dependent variable is functional or scalar
  #  --------------------------------------------------------------------------
  
  #  ----------------------------------------------------------------
  #           YFDOBJ is a functional data object
  #  ----------------------------------------------------------------
  
  #  extract dependent variable information
  ycoef     <- yfdobj$coefs
  ycoefdim  <- dim(ycoef)
  N         <- ycoefdim[2]
  ybasisobj <- yfdobj$basis
  rangeval  <- ybasisobj$rangeval
  ynbasis   <- ybasisobj$nbasis
  onesbasis <- create.constant.basis(rangeval)
  onesfd    <- fd(1,onesbasis)
  
  if (length(ycoefdim) > 2) stop("YFDOBJ from YFD is not univariate.")
  
  #  --------  set up the linear equations for the solution  -----------
  
  #  compute the total number of coefficients to be estimated
  
  ncoef <- 0
  for (j in 1:p) {
    betafdParj <- betalist[[j]]
    if (betafdParj$estimate) {
      ncoefj     <- betafdParj$fd$basis$nbasis
      ncoef      <- ncoef + ncoefj
    }
  }
  
  Cmat <- matrix(0,ncoef,ncoef)
  Dmat <- rep(0,ncoef)
  
  #  ------------------------------------------------------------------------
  #  Compute the symmetric positive definite matrix CMAT and
  #  the column matrix DMAT.  CMAT contains weighted inner products of
  #  bases for each pair of terms plus, for lambda > 0, a roughness penalty
  #  matrix to ensure that the estimated coefficient functions will be smooth
  #  The weight vector is the point-wise product of the associated functional
  #  covariates.  
  #  Dmat contains for each covariate the weighted integral of the basis 
  #  functions, with the weight function being the covariate function
  #  pointwise multiplied the dependent variate yobj.
  #  The estimated coefficients functions are defined by the solution
  #  CMAT %*% COEF = DMAT.
  #  ------------------------------------------------------------------------
  
  #  loop through rows of CMAT
  
  mj2 <- 0
  for (j in 1:p) {
    betafdParj <- betalist[[j]]
    if (betafdParj$estimate) {
      #  get jth beta basis
      betafdj    <- betafdParj$fd
      betabasisj <- betafdj$basis
      ncoefj     <- betabasisj$nbasis
      #  row indices of CMAT and DMAT to fill
      mj1    <- mj2 + 1
      mj2    <- mj2 + ncoefj
      indexj <- mj1:mj2
      #  compute right side of equation DMAT
      #  compute weight function for DMAT
      xfdj <- xfdlist[[j]]
      if (wtconst) {
        xyfdj <- xfdj*yfdobj
      } else {
        xyfdj <- (xfdj*wt)*yfdobj
      }
      wtfdj <- sum(xyfdj)
      #  Compute jth component of DMAT
      Dmatj <- inprod(betabasisj,onesfd,0,0,rangeval,wtfdj)
      Dmat[indexj] <- Dmatj
      #  loop through columns of CMAT
      mk2 <- 0
      for (k in 1:j) {
        betafdPark <- betalist[[k]]
        if (betafdPark$estimate) {
          #  get the kth basis
          betafdk    <- betafdPark$fd
          betabasisk <- betafdk$basis
          ncoefk     <- betabasisk$nbasis
          #  column indices of CMAT to fill
          mk1 <- mk2 + 1
          mk2 <- mk2 + ncoefk
          indexk <- mk1:mk2
          #  set up weight function for CMAT component
          xfdk <- xfdlist[[k]]
          if (wtconst) {
            xxfdjk <- xfdj*xfdk
          } else {
            xxfdjk <- (xfdj*wt)*xfdk
          }
          wtfdjk <- sum(xxfdjk)
          #  compute the inner product
          Cmatjk <- inprod(betabasisj, betabasisk, 0, 0,
                           rangeval, wtfdjk)
          Cmat[indexj,indexk] <- Cmatjk
          Cmat[indexk,indexj] <- t(Cmatjk)
        }
      }
      #  attach penalty term to diagonal block if required
      lambdaj <- betafdParj$lambda
      if (lambdaj > 0) {
        Rmatj <- betafdParj$penmat
        if (is.null(Rmatj)) {
          Lfdj  <- betafdParj$Lfd
          Rmatj <- eval.penalty(betabasisj, Lfdj)
        }
        Cmat[indexj,indexj] <- Cmat[indexj,indexj] +
          lambdaj*Rmatj
      }
    }
  }
  
  #  ensure symmetry
  
  Cmat <- (Cmat+t(Cmat))/2
  
  #  check Cmat for singularity
  
  eigchk(Cmat)
  
  #  solve for coefficients defining BETA
  
  Lmat    <- chol(Cmat)
  Lmatinv <- solve(Lmat)
  Cmatinv <- Lmatinv %*% t(Lmatinv)
  
  betacoef <- Cmatinv %*% Dmat
  
  #  set up fdPar objects for reg. fns. in BETAESTLIST
  
  betaestlist <- betalist
  mj2 <- 0
  for (j in 1:p) {
    betafdParj <- betalist[[j]]
    if (betafdParj$estimate) {
      betafdj <- betafdParj$fd
      ncoefj  <- betafdj$basis$nbasis
      mj1     <- mj2 + 1
      mj2     <- mj2 + ncoefj
      indexj  <- mj1:mj2
      coefj   <- betacoef[indexj]
      betafdj$coefs <- as.matrix(coefj)
      betafdParj$fd <- betafdj
    }
    betaestlist[[j]] <- betafdParj
  }
  
  #  set up fd objects for predicted values in YHATFDOBJ
  
  nfine   <- max(501,10*ynbasis+1)
  tfine   <- seq(rangeval[1], rangeval[2], len=nfine)
  yhatmat <- matrix(0,nfine,N)
  for (j in 1:p) {
    xfdj       <- xfdlist[[j]]
    xmatj      <- eval.fd(tfine, xfdj, 0, returnMatrix)
    betafdParj <- betaestlist[[j]]
    betafdj    <- betafdParj$fd
    betavecj   <- eval.fd(tfine, betafdj, 0, returnMatrix)
    yhatmat    <- yhatmat + xmatj*as.vector(betavecj)
  }
  yhatfdobj <- smooth.basis(tfine, yhatmat, ybasisobj)$fd
  
  df <- NA
  
  #  -----------------------------------------------------------------------
  #        Compute pointwise standard errors of regression coefficients
  #               if both y2cMap and SigmaE are supplied.
  #        y2cMap is supplied by the smoothing of the data that defined
  #        the dependent variable.
  #        SigmaE has to be computed from a previous analysis of the data.
  #  -----------------------------------------------------------------------
  
  if (!(is.null(y2cMap) || is.null(SigmaE))) {
    
    #  check dimensions of y2cMap and SigmaE
    
    y2cdim = dim(y2cMap)
    if (y2cdim[1] != ynbasis || y2cdim[2] != dim(SigmaE)[1]) {
      stop("Dimensions of Y2CMAP not correct.")
    }
    
    ybasismat = eval.basis(tfine, ybasisobj, 0, returnMatrix)
    
    deltat    = tfine[2] - tfine[1]
    
    #  compute BASISPRODMAT
    
    basisprodmat = matrix(0,ncoef,ynbasis*N)
    
    mj2 = 0
    for (j in 1:p) {
      betafdParj = betalist[[j]]
      betabasisj = betafdParj$fd$basis
      ncoefj     = betabasisj$nbasis
      bbasismatj = eval.basis(tfine, betabasisj, 0, returnMatrix)
      xfdj       = xfdlist[[j]]
      tempj      = eval.fd(tfine, xfdj, 0, returnMatrix)
      #  row indices of BASISPRODMAT to fill
      mj1    = mj2 + 1
      mj2    = mj2 + ncoefj
      indexj = mj1:mj2
      #  inner products of beta basis and response basis
      #    weighted by covariate basis functions
      mk2 = 0
      for (k in 1:ynbasis) {
        #  row indices of BASISPRODMAT to fill
        mk1    = mk2 + 1
        mk2    = mk2 + N
        indexk = mk1:mk2
        tempk  = bbasismatj*ybasismat[,k]
        basisprodmat[indexj,indexk] =
          deltat*crossprod(tempk,tempj)
      }
    }
    
    #  compute variances of regression coefficient function values
    
    c2bMap    = solve(Cmat,basisprodmat)
    VarCoef   = y2cMap %*% SigmaE %*% t(y2cMap)
    CVariance = kronecker(VarCoef,diag(rep(1,N)))
    bvar      = c2bMap %*% CVariance %*% t(c2bMap)
    betastderrlist = vector("list", p)
    mj2 = 0
    for (j in 1:p) {
      betafdParj = betalist[[j]]
      betabasisj = betafdParj$fd$basis
      ncoefj     = betabasisj$nbasis
      mj1 	     = mj2 + 1
      mj2 	     = mj2 + ncoefj
      indexj     = mj1:mj2
      bbasismat  = eval.basis(tfine, betabasisj, 0, returnMatrix)
      bvarj      = bvar[indexj,indexj]
      bstderrj   = sqrt(diag(bbasismat %*% bvarj %*% t(bbasismat)))
      bstderrfdj = smooth.basis(tfine, bstderrj, betabasisj)$fd
      betastderrlist[[j]] = bstderrfdj
    }
  } else {
    betastderrlist = NULL
    bvar           = NULL
    c2bMap         = NULL
  }
  
  #  -------------------------------------------------------------------
  #                       Set up output list object
  #  -------------------------------------------------------------------
  
  fRegressList <-
    list(yfdobj         = yfdobj,
         xfdlist        = xfdlist,
         betalist       = betalist,
         betaestlist    = betaestlist,
         yhatfdobj      = yhatfdobj,
         Cmat           = Cmat,
         Dmat           = Dmat,
         Cmatinv        = Cmatinv,
         wt             = wt,
         df             = df,
         y2cMap         = y2cMap,
         SigmaE         = SigmaE,
         betastderrlist = betastderrlist,
         bvar           = bvar,
         c2bMap         = c2bMap)
  
  
  return(fRegressList)
  
}

#  -------------------------------------------------------------------------------------

eigchk <- function(Cmat) {
  
  #  Last modified 25 August 2020 by Jim Ramsay
  
  #  Cmat for NA's
  
  if (any(is.na(Cmat))) stop("Cmat has NA values.")
  
  #  check Cmat for Cmatmetry
  
  if (max(abs(Cmat-t(Cmat)))/max(abs(Cmat)) > 1e-10) {
    stop('CMAT is not symmetric.')
  } else {
    Cmat <- (Cmat + t(Cmat))/2
  }
  
  #  check Cmat for singularity
  
  eigval <- eigen(Cmat)$values
  ncoef  <- length(eigval)
  if (eigval[ncoef] < 0) {
    neig <- min(length(eigval),10)
    cat("\nSmallest eigenvalues:\n")
    print(eigval[(ncoef-neig+1):ncoef])
    cat("\nLargest  eigenvalues:\n")
    print(eigval[1:neig])
    stop("Negative eigenvalue of coefficient matrix.")
  }
  if (eigval[ncoef] == 0) stop("Zero eigenvalue of coefficient matrix.")
  logcondition <- log10(eigval[1]) - log10(eigval[ncoef])
  if (logcondition > 12) {
    warning("Near singularity in coefficient matrix.")
    cat(paste("\nLog10 Eigenvalues range from\n",
              log10(eigval[ncoef])," to ",log10(eigval[1]),"\n"))
  }
}
fRegress.stderr <- function (y, y2cMap, SigmaE, returnMatrix = FALSE, ...) 
{
  
  #  FREGRESS.STDERR  computes standard error estimates for regression
  #       coefficient functions estimated by function FREGRESS.
  #
  #  Arguments:
  #
  #  Y            ... a list object produced by function FREGRESS with class name
  #                   'fRegress".  
  #                   This is indicated by Y in the arguments since R syntax
  #                   requires all of tghe fRegress family of functions to
  #                   use this notation.
  #  Y2CMAP       ... the matrix mapping from the vector of observed values
  #                   to the coefficients for the dependent variable.
  #                   This is output by function SMOOTH_BASIS.  If this is
  #                   supplied, confidence limits are computed, otherwise not.
  #  SIGMAE       ... Estimate of the covariances among the residuals.  This
  #                   can only be estimated after a preliminary analysis
  #                   with FREGRESS.
  #  RETURNMATRIX ... If False, a matrix in sparse storage model can be returned
  #               from a call to function BsplineS.  See this function for
  #               enabling this option.
  #
  #  Returns:
  #
  #  BETASTDERRLIST ... a list object, each list containing a fdPar object
  #                     for the standard error of a regression function.
  #  BVAR           ... the symmetric matrix of sampling variances and
  #                     covariances for the matrix of regression coefficients
  #                     for the regression functions.  These are stored
  #                     column-wise in defining BVARIANCE.
  #  C2BMAP         ... the matrix mapping from response variable coefficients
  #                     to coefficients for regression coefficients
  
  #  Last modified 16 December 2020
  
  #  retrieve objects from y
  
  yfdobj         <- y$yfdobj
  xfdlist        <- y$xfdlist
  betalist       <- y$betalist
  betaestlist    <- y$betaestlist
  yhatfdobj      <- y$yhatfdobj
  Cmat           <- y$Cmat
  Dmat           <- y$Dmat
  Cmatinv        <- y$Cmatinv
  wt             <- y$wt
  df             <- y$df
  betastderrlist <- y$betastderrlist
  YhatStderr     <- y$YhatStderr
  Bvar           <- y$Bvar
  c2bMap         <- y$c2bMap
  
  #  get number of independent variables
  
  p <- length(xfdlist)

  #  compute number of coefficients
  
  ncoef <- 0
  for (j in 1:p) {
    betaParfdj <- betalist[[j]]
    ncoefj <- betaParfdj$fd$basis$nbasis
    ncoef <- ncoef + ncoefj
  }
  
  if (inherits(yfdobj, "fdPar") || inherits(yfdobj, "fd")) {
    
    #  ----------------------------------------------------------------
    #           yfdobj is functional data object
    #  ----------------------------------------------------------------
    
    #  As of 2020, if yfdobj is an fdPar object, it is converted to an fd object.
    #  The added structure of the fdPar class is not used in any of the fRegress codes.
    #  The older versions of fda package used yfdPar as the name for the first member.
    
    if (inherits(yfdobj, "fdPar")) yfdobj <- yfdobj$fd
    
    #  get number of replications and basis information for YFDPAR
    
    N         <- dim(yfdobj$coefs)[2]
    ybasisobj <- yfdobj$basis
    rangeval  <- ybasisobj$rangeval
    ynbasis   <- ybasisobj$nbasis
    ninteg    <- max(501, 10 * ynbasis + 1)
    tinteg    <- seq(rangeval[1], rangeval[2], len = ninteg)
    deltat    <- tinteg[2] - tinteg[1]
    
    ybasismat <- eval.basis(tinteg, ybasisobj, 0, returnMatrix)
    
    #  compute BASISPRODMAT
    
    basisprodmat <- matrix(0, ncoef, ynbasis * N)
    
    mj2 <- 0
    for (j in 1:p) {
      betafdParj <- betalist[[j]]
      betabasisj <- betafdParj$fd$basis
      ncoefj <- betabasisj$nbasis
      bbasismatj <- eval.basis(tinteg, betabasisj, 0, returnMatrix)
      xfdj <- xfdlist[[j]]
      tempj <- eval.fd(tinteg, xfdj, 0, returnMatrix)
      mj1 <- mj2 + 1
      mj2 <- mj2 + ncoefj
      indexj <- mj1:mj2
      mk2 <- 0
      for (k in 1:ynbasis) {
        mk1 <- mk2 + 1
        mk2 <- mk2 + N
        indexk <- mk1:mk2
        tempk <- bbasismatj * ybasismat[, k]
        basisprodmat[indexj, indexk] <- deltat * crossprod(tempk, 
                                                           tempj)
      }
    }
    
    #  check dimensions of Y2CMAP
    
    y2cdim <- dim(y2cMap)
    if (y2cdim[1] != ynbasis || y2cdim[2] != dim(SigmaE)[1]) 
      stop("Dimensions of Y2CMAP not correct.")
    
    #  compute variances of regression coefficient function values
    
    Varc <- y2cMap %*% SigmaE %*% t(y2cMap)
    CVar <- kronecker(Varc, diag(rep(1, N)))
    C2BMap <- Cmatinv %*% basisprodmat
    Bvar <- C2BMap %*% CVar %*% t(C2BMap)
    nplot <- max(51, 10 * ynbasis + 1)
    tplot <- seq(rangeval[1], rangeval[2], len = nplot)
    betastderrlist <- vector("list", p)
    PsiMatlist <- vector("list", p)
    mj2 <- 0
    for (j in 1:p) {
      betafdParj <- betalist[[j]]
      betabasisj <- betafdParj$fd$basis
      ncoefj <- betabasisj$nbasis
      mj1 <- mj2 + 1
      mj2 <- mj2 + ncoefj
      indexj <- mj1:mj2
      bbasismat <- eval.basis(tplot, betabasisj, 0, returnMatrix)
      PsiMatlist <- bbasismat
      bvarj <- Bvar[indexj, indexj]
      bstderrj <- sqrt(diag(bbasismat %*% bvarj %*% t(bbasismat)))
      bstderrfdj <- smooth.basis(tplot, bstderrj, betabasisj)$fd
      betastderrlist[[j]] <- bstderrfdj
    }
    
    #  compute estimated variance-covariance matrix over plotting grid
    #  of fitted values
    
    YhatStderr <- matrix(0, nplot, N)
    B2YhatList <- vector("list", p)
    for (iplot in 1:nplot) {
      YhatVari <- matrix(0, N, N)
      tval <- tplot[iplot]
      for (j in 1:p) {
        Zmat <- eval.fd(tval, xfdlist[[j]])
        betabasisj <- betalist[[j]]$fd$basis
        PsiMatj <- eval.basis(tval, betabasisj)
        B2YhatMapij <- t(Zmat) %*% PsiMatj
        B2YhatList[[j]] <- B2YhatMapij
      }
      m2j <- 0
      for (j in 1:p) {
        m1j <- m2j + 1
        m2j <- m2j + betalist[[j]]$fd$basis$nbasis
        B2YhatMapij <- B2YhatList[[j]]
        m2k <- 0
        for (k in 1:p) {
          m1k <- m2k + 1
          m2k <- m2k + betalist[[k]]$fd$basis$nbasis
          B2YhatMapik <- B2YhatList[[k]]
          YhatVari <- YhatVari + 
            B2YhatMapij %*% Bvar[m1j:m2j,m1k:m2k] %*% t(B2YhatMapik)
        }
      }
      YhatStderr[iplot, ] <- matrix(sqrt(diag(YhatVari)), 1, N)
    }
  }
  
  else {
    
    #  ----------------------------------------------------------------
    #                   yfdobj is scalar or multivariate
    #  ----------------------------------------------------------------
    
    ymat <- as.matrix(yfdobj)
    N <- dim(ymat)[1]
    Zmat <- NULL
    for (j in 1:p) {
      xfdj <- xfdlist[[j]]
      if (inherits(xfdj, "fd")) {
        xcoef <- xfdj$coefs
        xbasis <- xfdj$basis
        betafdParj <- betalist[[j]]
        bbasis <- betafdParj$fd$basis
        Jpsithetaj <- inprod(xbasis, bbasis)
        Zmat <- cbind(Zmat, t(xcoef) %*% Jpsithetaj)
      }
      else if (inherits(xfdj, "numeric")) {
        Zmatj <- xfdj
        Zmat <- cbind(Zmat, Zmatj)
      }
    }
    
    #  compute linear mapping c2bMap takinging coefficients for
    #  response into coefficients for regression functions
    
    c2bMap <- Cmatinv %*% t(Zmat)
    y2bmap <- c2bMap
    Bvar <- y2bmap %*% as.matrix(SigmaE) %*% t(y2bmap)
    betastderrlist <- vector("list", p)
    mj2 <- 0
    for (j in 1:p) {
      betafdParj <- betalist[[j]]
      betabasisj <- betafdParj$fd$basis
      ncoefj <- betabasisj$nbasis
      mj1 <- mj2 + 1
      mj2 <- mj2 + ncoefj
      indexj <- mj1:mj2
      bvarj <- Bvar[indexj, indexj]
      xfdj <- xfdlist[[j]]
      if (inherits(xfdj, "fd")) {
        betarng <- betabasisj$rangeval
        ninteg <- max(c(501, 10 * ncoefj + 1))
        tinteg <- seq(betarng[1], betarng[2], len = ninteg)
        bbasismat <- eval.basis(tinteg, betabasisj, 0, 
                                returnMatrix)
        bstderrj <- sqrt(diag(bbasismat %*% bvarj %*% 
                                t(bbasismat)))
        bstderrfdj <- smooth.basis(tinteg, bstderrj, 
                                   betabasisj)$fd
      }
      else {
        bsterrj <- sqrt(diag(bvarj))
        onebasis <- create.constant.basis(betabasisj$rangeval)
        bstderrfdj <- fd(t(bstderrj), onebasis)
      }
      betastderrlist[[j]] <- bstderrfdj
    }
    
    #  compute estimated variance-covariance matrix for fitted values
    
    B2YhatList <- vector("list", p)
    YhatVari <- matrix(0, N, N)
    for (j in 1:p) {
      betabasisj <- betalist[[j]]$fd$basis
      Xfdj <- xfdlist[[j]]
      B2YhatMapij <- inprod(Xfdj, betabasisj)
      B2YhatList[[j]] <- B2YhatMapij
    }
    m2j <- 0
    for (j in 1:p) {
      m1j <- m2j + 1
      m2j <- m2j + betalist[[j]]$fd$basis$nbasis
      B2YhatMapij <- B2YhatList[[j]]
      m2k <- 0
      for (k in 1:p) {
        m1k <- m2k + 1
        m2k <- m2k + betalist[[k]]$fd$basis$nbasis
        B2YhatMapik <- B2YhatList[[k]]
        YhatVari <- YhatVari + B2YhatMapij %*% Bvar[m1j:m2j, 
                                                    m1k:m2k] %*% t(B2YhatMapik)
      }
    }
    YhatStderr <- matrix(sqrt(diag(YhatVari)), N, 1)
  }
  
  #  return output object of class fRegress
  
  fRegressList <- list(yfdobj = y$yfdobj, xfdlist = y$xfdlist, 
                       betalist = y$betalist, betaestlist = y$betaestlist, 
                       yhatfdobj = y$yhatfdobj, Cmat = y$Cmat, Dmat = y$Dmat, 
                       Cmatinv = y$Cmatinv, wt = y$wt, 
                       df = y$df, y2cMap = y2cMap, SigmaE = SigmaE, 
                       betastderrlist = betastderrlist, YhatStderr = YhatStderr, 
                       Bvar = Bvar, c2bMap = c2bMap)
  
  class(fRegressList) = "fRegress"
  return(fRegressList)
}

Fstat.fd <- function(y,yhat,argvals=NULL) {    

# observed, predicted and where to evaluate

    if( is.numeric(yhat) ){ yhat = as.vector(yhat) }

    if( (is.vector(y) & !is.vector(yhat)) | (is.fd(y) &!is.fd(yhat)) ){
        stop("y and yhat must both be either scalars or functional data objects.")
    }


    if( is.fd(y) ){
        rangeobs = y$basis$range
        rangehat = yhat$basis$range

        if( !prod(rangeobs == rangehat) ){
            stop("y and yhat do not have the same range")
        }


        if(is.null(argvals)){
            argvals = seq(rangeobs[1],rangeobs[2],length.out=101)
        }

        yvec = eval.fd(argvals,y)
        yhatvec = eval.fd(argvals,yhat)

        F = apply(yhatvec,1,var)/apply( (yvec-yhatvec)^2,1,mean)

    }
    else{
        yvec = y
        yhatvec = yhat
        F = var(yhatvec)/mean( (yvec-yhatvec)^2 )
    }

    return( list(F=F,argvals=argvals) )
}
funcint <- function(func,cvec, basisobj, nderiv=0, JMAX=15, EPS=1e-7)
{
  
  #  computes the definite integral of a function defined in terms of 
  #  a functional data object with basis BASISOBJ and coefficient vector CVEC.
  
  #  FUNC     ... the name of a function
  #  CVEC     ..  a vector of coefficients defining the functional data
  #               object required to compute the value of the function.
  #  BASISOBJ ... a functional data basis object
  #  NDERIV   ... a non-negative integer defining a derivative to be used.
  #  JMAX   maximum number of allowable iterations
  #  EPS    convergence criterion for relative stop
  
  #  Return: the value of the function
  
  #  Last modified 15 June 2020 by Jim Ramsay
  
  #  set iter
  
  iter <- 0
  
  # The default case, no multiplicities.
  
  rng    <- basisobj$rangeval
  
  nbasis <- basisobj$nbasis
  
  nrep   <- dim(basisobj$coef)[2]
  
  inprodvec <- matrix(0,nrep,1)
  
  #  set up first iteration
  
  iter  <- 1
  width <- rng[2] - rng[1]
  JMAXP <- JMAX + 1
  h     <- rep(1,JMAXP)
  h[2]  <- 0.25
  s <- matrix(0,JMAXP,nrep)
  sdim <- length(dim(s))
  #  the first iteration uses just the endpoints
  fdobj <- fd(cvec, basisobj)
  fx    <- func(eval.fd(rng, fdobj, nderiv))
  #  multiply by values of weight function if necessary
  s[1,] <- width*apply(fx,2,sum)/2
  tnm  <- 0.5
  
  #  now iterate to convergence
  
  for (iter in 2:JMAX) {
    tnm <- tnm*2
    if (iter == 2) {
      x <- mean(rng)
    } else {
      del <- width/tnm
      x   <- seq(rng[1]+del/2, rng[2]-del/2, del)
    }
    fx <- func(eval.fd(x, fdobj, nderiv))
    chs <- width*apply(fx,2,sum)/tnm
    s[iter,] <- (s[iter-1,] + chs)/2
    if (iter >= 5) {
      ind <- (iter-4):iter
      ya <- s[ind,,]
      ya <- matrix(ya,5,nrep)
      xa <- h[ind]
      absxa <- abs(xa)
      absxamin <- min(absxa)
      ns <- min((1:length(absxa))[absxa == absxamin])
      cs <- ya
      ds <- ya
      y  <- ya[ns,,]
      ns <- ns - 1
      for (m in 1:4) {
        for (i in 1:(5-m)) {
          ho      <- xa[i]
          hp      <- xa[i+m]
          w       <- (cs[i+1,] - ds[i,])/(ho - hp)
          ds[i,] <- hp*w
          cs[i,] <- ho*w
        }
        if (2*ns < 5-m) {
          dy <- cs[ns+1,]
        } else {
          dy <- ds[ns,]
          ns <- ns - 1
        }
        y <- y + dy
      }
      ss     <- y
      errval <- max(abs(dy))
      ssqval <- max(abs(ss))
      if (all(ssqval > 0)) {
        crit <- errval/ssqval
      } else {
        crit <- errval
      }
      if (crit < EPS && iter >= 5) break
    }
    s[iter+1,] <- s[iter,]
    h[iter+1]   <- 0.25*h[iter]
    if (iter == JMAX) warning("Failure to converge.")
  }
  inprodvec <- inprodvec + ss
  
}


geigen <- function(Amat, Bmat, Cmat)
{
  #  solve the generalized eigenanalysis problem
  #
  #    max {tr L'AM / sqrt[tr L'BL tr M'CM] w.r.t. L and M
  #
  #  Arguments:
  #  AMAT ... p by q matrix
  #  BMAT ... order p symmetric positive definite matrix
  #  CMAT ... order q symmetric positive definite matrix
  #  Returns:
  #  VALUES ... vector of length s = min(p,q) of eigenvalues
  #  LMAT   ... p by s matrix L
  #  MMAT   ... q by s matrix M

  #  last modified 9 November 2020 to use svd

  Bdim <- dim(Bmat)
  Cdim <- dim(Cmat)
  if (Bdim[1] != Bdim[2]) stop('BMAT is not square')
  if (Cdim[1] != Cdim[2]) stop('CMAT is not square')
  p <- Bdim[1]
  q <- Cdim[1]
  s <- min(c(p,q))
  if (max(abs(Bmat - t(Bmat)))/max(abs(Bmat)) > 1e-10) stop(
    'BMAT not symmetric.')
  if (max(abs(Cmat - t(Cmat)))/max(abs(Cmat)) > 1e-10) stop(
    'CMAT not symmetric.')
  Bmat  <- (Bmat + t(Bmat))/2
  Cmat  <- (Cmat + t(Cmat))/2
  Bfac  <- chol(Bmat)
  Cfac  <- chol(Cmat)
  Bfacinv <- solve(Bfac)
  Cfacinv <- solve(Cfac)
  Dmat <- t(Bfacinv) %*% Amat %*% Cfacinv
  if (p >= q) {
    result <- svd(Dmat)
    values <- result$d
    Lmat <- Bfacinv %*% result$u
    Mmat <- Cfacinv %*% result$v
  } else {
    result <- svd(t(Dmat))
    values <- result$d
    Lmat <- Bfacinv %*% result$v
    Mmat <- Cfacinv %*% result$u
  }
  geigenlist <- list (values, Lmat, Mmat)
  names(geigenlist) <- c('values', 'Lmat', 'Mmat')
  return(geigenlist)
}
getbasismatrix <- function(evalarg, basisobj, nderiv=0, returnMatrix=FALSE) {
#  Computes the basis matrix evaluated at arguments in EVALARG associated
#    with basis.fd object BASISOBJ.  The basis matrix contains the values
#    at argument value vector EVALARG of applying the nonhomogeneous
#    linear differential operator LFD to the basis functions.  By default
#    LFD is 0, and the basis functions are simply evaluated at argument
#    values in EVALARG.
#
#  If LFD is a functional data object with m + 1 functions c_1, ... c_{m+1},
#    then it is assumed to define the order m HOMOGENEOUS linear
#    differential operator
#  Lx(t) = c_1(t) + c_2(t)x(t) + c_3(t)Dx(t) + ... +
#                             c_{m+1}D^{m-1}x(t) + D^m x(t).
#
#  If the basis type is either polygonal or constant, LFD is ignored.
#
#  Arguments:
#  EVALARG...Either a vector of values at which all functions are to evaluated,
#            or a matrix of values, with number of columns corresponding to
#            number of functions in argument FD.  If the number of evaluation
#            values varies from curve to curve, pad out unwanted positions in
#            each column with NA.  The number of rows is equal to the maximum
#            of number of evaluation points.
#  BASISOBJ...A basis object
#  NDERIV ... A nonnegative integer indicating a derivative to be evaluated.

#
#  Note that the first two arguments may be interchanged.
#

#  Last modified 6 January 2020

##
##  Exchange the first two arguments if the first is an BASIS.FD object
##    and the second numeric
##
  if (is.numeric(basisobj) && inherits(evalarg, "basisfd")) {
    temp     <- basisobj
    basisobj <- evalarg
    evalarg  <- temp
  }
##
##  check EVALARG
##
#  if (!(is.numeric(evalarg)))  stop("Argument EVALARG is not numeric.")
  if(is.null(evalarg)) stop('evalarg required;  is NULL.')
  Evalarg <- evalarg
# turn off warnings in checking if argvals can be converted to numeric
  op <- options(warn=-1)
  evalarg <- as.numeric(Evalarg)
  options(op)
  nNA <- sum(is.na(evalarg))
  if(nNA>0)
    stop('as.numeric(evalarg) contains ', nNA,
         ' NA', c('', 's')[1+(nNA>1)],
         ';  class(evalarg) = ', class(Evalarg))
##
##  check BASISOBJ
##
  if (!(inherits(basisobj, "basisfd")))
      stop("Second argument is not a basis object.")
##
##  search for stored basis matrix and return it if found
##
  if (!(length(basisobj$basisvalues) == 0 || is.null(basisobj$basisvalues))) {
    #  one or more stored basis matrices found,
    #  check that requested derivative is available
    if (!is.vector(basisobj$basisvalues)) stop("BASISVALUES is not a vector.")
    basisvalues <- basisobj$basisvalues
    nvalues     <- length(basisvalues)
    #  search for argvals match
    N  <- length(evalarg)
    OK <- FALSE
    for (ivalues in 1:nvalues) {
      basisvaluesi <- basisvalues[ivalues]
      if (!is.list(basisvaluesi)) stop("BASISVALUES does not contain lists.")
      argvals <- basisvaluesi[[1]]
      if (!length(basisvaluesi) < nderiv+2) {
          if (N == length(argvals)) {
              if (all(argvals == evalarg)) {
                  basismat <- basisvaluesi[[nderiv+2]]
                  OK <- TRUE
              }
          }
      }
    }
    #   dimnames
    dimnames(basismat) <- list(NULL, basisobj$names)

    if (OK){
        if(length(dim(basismat)) == 2){
            return(as.matrix(basismat))
        }
        return(basismat)
    }
  }
##
##  compute the basis matrix and return it
##
#  Extract information about the basis
  type     <- basisobj$type
  nbasis   <- basisobj$nbasis
  params   <- basisobj$params
  rangeval <- basisobj$rangeval
  dropind  <- basisobj$dropind

##
##  Select basis and evaluate it at EVALARG values
##

#  -----------------------------  B-spline basis  -------------------

  if (type == "bspline") {
      if (length(params) == 0) {
          breaks   <- c(rangeval[1], rangeval[2])
      } else {
   	    breaks   <- c(rangeval[1], params, rangeval[2])
      }
   	norder   <- nbasis - length(breaks) + 2
   	basismat <- bsplineS(evalarg, breaks, norder, nderiv)
   	# The following lines call spline.des in the base R system.  
   	# This is slightly slower than the above call to bsplineS
    # nbreaks  <- length(breaks)
    # knots    <- c(rep(rangeval[1],norder), breaks(2:(nbreaks-1)),
    #               rep(rangeval[2],norder))
    # basismat <- spline.des(knots, evalarg, norder, nderiv, sparse=TRUE)
#  -----------------------------  Constant basis  --------------------

  } else if (type == "const") {
   	basismat  <- matrix(1,length(evalarg),1)

#  -----------------------------  Exponential basis  -------------------

  } else if (type == "expon") {
   	basismat  <- expon(evalarg, params, nderiv)

#  -------------------------------  Fourier basis  -------------------

  } else if (type == "fourier") {
   	period   <- params[1]
   	basismat <- fourier(evalarg, nbasis, period, nderiv)

#  -----------------------------  Monomial basis  -------------------

  } else if (type == "monom") {
   	basismat  <- monomial(evalarg, params, nderiv)

#  -----------------------------  Polygonal basis  -------------------

  } else if (type == "polygonal") {
    basismat  <- polyg(evalarg, params)

#  -----------------------------  Power basis  -------------------

  } else if (type == "power") {
    basismat  <- powerbasis(evalarg, params, nderiv)

#  -----------------------  Unrecognizable basis  --------------------

  } else {
   	stop("Basis type not recognizable")
  }
#  dimnames
  dimnames(basismat) <- list(NULL, basisobj$names)

#  remove columns for bases to be dropped

  if (length(dropind) > 0) basismat <- basismat[,-dropind]
  if (length(evalarg) == 1) {
    basismat = matrix(basismat,1,length(basismat))
  }
    

  if (length(dim(basismat)) == 2){
    #  coerce basismat to be nonsparse
      return(as.matrix(basismat))
  } else {
    #  allow basismat to be sparse if it already is
      return(as.matrix(basismat))
  }

}
getbasispenalty <- function(basisobj, Lfdobj=NULL)
{
#  Computes the penaltymat matrix  associated with basis object basisobj.
#    This is defined in terms of a linear differential operator LFDOBJ.
#    The default for LFDOBJ depends on the nature of the basis.

#  Last modified 19 March 2014

#  check BASISOBJ

if (!(inherits(basisobj, "basisfd"))) stop(
    "First argument is not a basis object.")

type   <- basisobj$type
nbasis <- basisobj$nbasis

if        (type == "fourier") {
    if (is.null(Lfdobj)) Lfdobj <- 2
    penaltymat <- fourierpen(basisobj, Lfdobj)
} else if (type == "bspline") {
    norder <- basisobj$nbasis - length( basisobj$params )
    if (is.null(Lfdobj)) Lfdobj <- 2
    penaltymat <- bsplinepen(basisobj, Lfdobj)
} else if (type == "expon")   {
    if (is.null(Lfdobj)) Lfdobj <- 2
    penaltymat <- exponpen(basisobj, Lfdobj)
} else if (type == "polyg" | type == "polygonal")   {
    if (is.null(Lfdobj)) Lfdobj <- 1
    penaltymat <- polygpen(basisobj, Lfdobj)
} else if (type == "power")   {
    if (is.null(Lfdobj)) Lfdobj <- 2
    penaltymat <- powerpen(basisobj, Lfdobj)
} else if (type == "const")   {
    if (is.null(Lfdobj)) Lfdobj <- 0
    if (Lfdobj == 0) {
      penaltymat <- basisobj$rangeval[2] - basisobj$rangeval[1]
    } else {
      penaltymat <- 0
    }
} else {
    stop("Basis type not recognizable")
}

return(penaltymat)
}
getbasisrange <- function(basisobj){
#  GETBASISRANGE   Extracts the range from basis object BASISOBJ.
#  R port 2007.11.28 by Spencer Graves  
#  previously modified 30 June 1998

  if(!is.basis(basisobj))
    stop("'basisobj' is not a functional basis object.")
#
  rangeval <- basisobj$rangeval
  rangeval
}
glm.fda <- function(basismat, y, family, lamRmat, wtvec=NULL, 
                    bvec0=NULL, addterm=NULL) {
  #GLM.FDA Fits a generalized linear model with regularization. 
  #  This function is called by function smooth.GLM
  #  Arguments
  #
  #  BASISMAT An N by NBASIS matrix of values of basis functions 
  #  Y        May be
  #                a N by NCURVE matrix of data to be fitted
  #                or, in the binomial case with local sample sizes M.i,
  #                a list array of length 2, the first of which cantains
  #                the matrix above containing observed frequencies,
  #                and the second of which contains the corresponding 
  #                sample sizes.  Note that in the binary or Bernoulli case,
  #                Y may be a matrix of 1"s and 0"s and the M"s are
  #                taken to be 1"s.
  #  FAMILY   A string indicating which of the five GLM family members
  #              is assumed
  #              "normal" or "gaussian" or "Gaussian"
  #              "binomial" or "binary" or "Bernoulli"
  #              "poisson"
  #              "gamma"
  #              "inverse gaussian" or "inverse Gaussian"
  #              or a list array of length(N) with each list containing
  #              a specification of the GLM family of a single observation.
  #  LAMRMAT  a \lambda R, that is, a roughness penalty matrix R of 
  #              order equal to the number of basis functions used or number
  #              of columns of basismat multiplied by a scalar roughness 
  #              penalty parameter \lambda
  #  wtvec    a vector of prior weights, such as the inverses of the
  #              relative variance of each observation.
  #  BVEC0    starting values for regresscion coefficients
  #  ADDTERM  a addterm with a coefficient fixed at 1.0.
  #
  #  Returns
  #  BVEC      Final estimate of coefficients
  #  DEVIANCE  Deviance values
  #
  #   Last modified 17 May 2018 by Jim Ramsay
  
  #--------------------------------------------------------------------------
  #                    Check arguments
  #--------------------------------------------------------------------------
  
  #  dimensions of basismat
  
  basismatDim <- dim(basismat)
  n       <- basismatDim[1]
  nbasis  <- basismatDim[2]
  if (is.list(y)) {
    yDim <- dim(as.matrix(y[[1]]))
    ntmp    <- yDim[1]
    ncurve  <- yDim[2]
  } else { 
    y    <- as.matrix(y)
    yDim <- dim(y)
    ntmp    <- yDim[1]
    ncurve  <- yDim[2]
  }
  if (n != ntmp) {
    stop("basismat and y do not have the same number of rows.")
  }
  
  #  define default weight vector wtvec and check for positivity
  
  if (is.null(wtvec)) {
    wtvec <- matrix(1,n,1)
  }
  
  if (any(wtvec <= 0)) {
    stop("Non-positive values of wtvec found.")
  }
  
  #--------------------------------------------------------------------------
  #  Process y and define anonymous functions according to the 
  #  distribution of y
  #     devFn    the deviance or loss function, 
  #                 called after convergence is achieved
  #     stdFn    the scale factor multiplying D eta
  #                 called second inside loop        
  #     linkFn   link function, eta <- linkFn(mu),
  #                 called prior to loop, maps data space into real line
  #     DlinkFn  derivative of the link function wrt to mu
  #                 called first inside loop
  #     IlinkFn  the inverse of the link function IlinkFn[eta] <- mu,
  #                 called last inside loop, maps eta into data space
  # Then set a starting value for the mean mu, avoiding boundary values.
  #--------------------------------------------------------------------------
  
  M <- NULL
  if (is.character(family)) {
    #  --------------------------------------------------------------------
    #    All observations are in the same family, family is a string
    #  --------------------------------------------------------------------
    if (!(family == "normal"   ||
          family == "binomial" ||
          family == "poisson"  ||
          family == "gamma"    ||
          family == "inverse gaussian")) {
      stop("The distribution is not valid.")
    }
    if (family == "normal") {
      #  Note  y can be any real number, no restrictions
      devFn   <- function(mu,y) (y - mu)^2
      stdFn   <- function(mu)  matrix(1,dim(mu))
      linkFn  <- function(mu)  mu
      DlinkFn <- function(mu)  matrix(1,dim(mu))
      IlinkFn <- function(eta) eta
      mu      <- y
    } 
    #  --------------------------------------------------------------------
    if (family == "binomial") {
      if (is.numeric(y)) {
        #  If y a matrix, M is taken to be 1 (set below)
        #  and it must be a binary matrix containing only 0"s and 1"s
        if (any(y < 0 | y > 1)) {
          stop(c("For binomial case, y a single column but ", 
                 " contains values other than 0 or 1."))
        }
        M <- matrix(1,n,ncurve)
      } else {
        if (is.list(y) && length(y) == 2) {
          #  If y is a list array of length 2, then first list 
          #  contains a matrix containing the number of successes and 
          #  the second list either contains a matrix of the same 
          #  size as the matrix in y{1} or a single positive 
          #  integer.  
          #  These values or this value is the number of trials M
          #  for a binomial or bernoulli distribution.
          #  M must be a positive integer.
          Freq <- y[[1]]
          M    <- y[[2]]
          if (length(M) == 1) {
            M <- M*matrix(1,n,ncurve)
          }
          if (!all(dim(M) == dim(Freq))) {
            stop(c("FAMILY is binomial and matrix M is not the same ", 
                   "size as matrix FREQ"))
          }
          if (any(M < 0)) {
            stop(c("FAMILY is binomial and one or more values in M ", 
                   "have nonpositive values"))
          }
          if (any(any(floor(M) != M))) {
            stop(c("FAMILY is binomial and one or more values in M ", 
                   "have noninteger values."))
          }
          #  Redefine y is the proportion of sucesses
          y <- Freq/M
        } else {
          stop(c("FAMILY is binomial and y has incorrect dimensions ", 
                 " or is of wrong type."))
        }
        devFn   <- function(mu,y) 2*M*(y*log((y+(y==0))/mu) + 
                                            (1-y)*log((1-y+(y==1))/(1-mu)))
        stdFn   <- function(mu)  sqrt(mu*(1-mu)/M)
        linkFn  <- function(mu)   log(mu/(1-mu))
        DlinkFn <- function(mu)    1/(mu*(1-mu))
        loBnd   <- -16
        upBnd   <- -loBnd
        IlinkFn <- function(eta) 1/(1 + exp(-constrain(eta,loBnd,upBnd)))
        mu      <- (M*y + 0.5)/(M + 1)
      }
    }
    #  --------------------------------------------------------------------
    if (family == "poisson") {
      #  Note y must not contain negative numbers
      if (any(y < 0)) {
        stop("FAMILY is poisson and y contains negative values")
      }
      devFn   <- function(mu,y) 2*(y*(log((y+(y==0))/mu)) - 
                                     (y - mu))
      stdFn   <- function(mu)  sqrt(mu)
      linkFn  <- function(mu)   log(mu)
      DlinkFn <- function(mu)     1/mu
      loBnd   <- -16
      upBnd   <- -loBnd
      IlinkFn <- function(eta) exp(constrain(eta,loBnd,upBnd))
      mu      <- y + 0.25
    }
    #  --------------------------------------------------------------------
    if (family == "gamma") {
      #  Note  y must contain only positive numbers
      if (any(y <= 0)) {
        stop("FAMILY is gamma and Y contains nonpositive values")
      }
      devFn   <- function(mu,y) 2*(-log(y/mu) + (y - mu)/mu)
      stdFn   <- function(mu)    mu
      linkFn  <- function(mu)  1/mu
      DlinkFn <- function(mu) -1/mu^2
      loBnd   <- -16
      upBnd   <- 1/loBnd
      IlinkFn <- function(eta) 1/constrain(eta,loBnd,upBnd)
      mu      <- max(y, eps)
    }
    #  --------------------------------------------------------------------
    if (family == "inverse gaussian") {
      #  Note  y must contain only positive numbers
      if (any(y <= 0)) {
        stop(c("FAMILY is inverse gaussian and Y contains ", 
               "nonpositive values"))
      }
      devFn   <- function(mu,y) ((y - mu)/mu)^2/ y
      stdFn   <- function(mu)  mu^(3/2)
      loBnd   <- -8
      upBnd   <- 1/loBnd
      linkFn  <- function(mu)  constrain(mu,loBnd,upBnd)^(-2)
      DlinkFn <- function(mu)  -2*mu^(-3)
      IlinkFn <- function(eta) constrain(eta,loBnd,upBnd)^(-1/2)
      mu      <- y
    }
  }
  # } else {
  #   if (is.list(family) && length(family) == n) {
  #     #  --------------------------------------------------------------------
  #     #    Observations can be in different families, family is a list array.
  #     #  --------------------------------------------------------------------
  #     mu      <- matrix(0,n,1)
  #     loBnd   <- matrix(0,n,1)
  #     upBnd   <- matrix(0,n,1)
  #     devFn   <- vector("list",n)
  #     stdFn   <- vector("list",n)
  #     linkFn  <- vector("list",n)
  #     DlinkFn <- vector("list",n)
  #     IlinkFn <- vector("list",n)
  #     #  Dealing with the presence of some binomial observations y has
  #     #  to be a list with n rows and 2 columns for all data.  Ugh!
  #     binomwrd <- is.list(y) && all(dim(y) == c(n,2))
  #   }
  #   for (i in 1:n) {
  #     familyi <- family[[i]]
  #     if (!is.character(familyi)) {
  #       stop("A distribution specification is not a string.")
  #     }
  #     if (family == "normal") {
  #       #  Note  y can be any real number, no restrictions
  #       devFn[[i]]   <- function(mu,y) (y - mu)^2
  #       stdFn[[i]]   <- function(mu)  matrix(1,dim(mu))
  #       linkFn[[i]]  <- function(mu)  mu
  #       DlinkFn[[i]] <- function(mu)  matrix(1,dim(mu))
  #       IlinkFn[[i]] <- function(eta) eta
  #       mu[i,] <- y[i,]
  #     }
  #     if (family == "binomial") {
  #       if (all(isnumeric(y[i,]))) {
  #         #  If y a matrix, M is taken to be 1 (set below)
  #         #  and it must be a binary matrix containing only 
  #         #0"s and 1"s
  #         if (any(y[i,] < 0 | y[i,] > 1)) {
  #           stop(c("For binomial case, y a single column but ", 
  #                  " contains values other than 0 or 1."))
  #         }
  #       } else {
  #         if (binomwrd) {
  #           Freqi <- y[[i,1]]
  #           Mi    <- y[[i,2]]
  #           if (length(Mi) == 1) {
  #             Mi <- Mi*matrix(1,1,ncurve)
  #           }
  #           if (!all(dim(Mi) == dim(Freqi))) {
  #             stop(paste("FAMILY is binomial and matrix M is not the same ", 
  #                        "dim as matrix FREQ"))
  #           }
  #           if (any(any(Mi < 0))) {
  #             stop(c("FAMILY is binomial and one or more values in M ", 
  #                    "have nonpositive values"))
  #           }
  #           if (any(any(floor(Mi) != Mi))) {
  #             stop(paste("FAMILY is binomial and one or more values in M ", 
  #                        "have noninteger values."))
  #           }
  #           #  Redefine y is the proportion of sucesses
  #           y[i,] <- (Freqi/Mi)
  #         } else {
  #           stop(paste("FAMILY is binomial and y has incorrect dimensions ", 
  #                      " or is of wrong type."))
  #         }
  #         devFn[[i]]   <- function(mu,y) 2*M*(y*log((y+(y==0))/mu) + 
  #                                               (1-y)*log((1-y+(y==1))/(1-mu)))
  #         stdFn[[i]]   <- function(mu)  sqrt(mu*(1-mu)/M)
  #         linkFn[[i]]  <- function(mu)   log(mu/(1-mu))
  #         DlinkFn[[i]] <- function(mu)    1/(mu*(1-mu))
  #         loBnd[i]   <- log(eps)
  #         upBnd[i]   <- -loBnd[i]
  #         IlinkFn[[i]] <- function(eta) 1/(1 + exp(-constrain(eta,loBnd,upBnd)))
  #         mu[i]      <- (M[i]*y[i] + 0.5)/(M[i] + 1)
  #       }
  #       if (family == "gamma") {
  #         #  Note  y must contain only positive numbers
  #         if (any(y[i] <= 0)) {
  #           stop("FAMILY is gamma and Y contains nonpositive values")
  #         }
  #         devFn[[i]]   <- function(mu,y) 2*(-log(y/mu) + (y - mu)/mu)
  #         stdFn[[i]]   <- function(mu)    mu
  #         linkFn[[i]]  <- function(mu)  1/mu
  #         DlinkFn[[i]] <- function(mu) -1/mu^2
  #         loBnd[i]   <- eps
  #         upBnd[i]   <- 1/loBnd[i]
  #         IlinkFn[[i]] <- function(eta) 1/constrain(eta,loBnd,upBnd)
  #         mu[i,]    <- max(y[i,], eps)
  #       }
  #       if (family == "inverse gaussian") {
  #         #  Note  y must contain only positive numbers
  #         if (any(y[i,] <= 0)) {
  #           stop(c("FAMILY is inverse gaussian and Y contains ", 
  #                  "nonpositive values"))
  #         }
  #         devFn[[i]]   <- function(mu,y) ((y - mu)/mu)^2/ y
  #         stdFn[[i]]   <- function(mu)  mu^(3/2)
  #         loBnd[i]   <- eps^(1/2)
  #         upBnd[i]   <- 1/loBnd[i]
  #         linkFn[[i]]  <- function(mu)  constrain(mu,loBnd,upBnd)^(-2)
  #         DlinkFn[[i]] <- function(mu)  -2*mu^(-3)
  #         IlinkFn[[i]] <- function(eta) constrain(eta,loBnd,upBnd)^(-1/2)
  #         mu[i,]    <- y[i,]
  #       }
  #     }
  #   }
  
  #--------------------------------------------------------------------------
  #                   Initialize mu and eta from y.
  #--------------------------------------------------------------------------
  
  # compute eta <- E(y) from mu
  
  if (is.character(family)) {
    eta <- linkFn(mu)
  # } else {
  #   eta  <- matrix(0,n,nurve)
  #   Deta <- matrix(0,n,nurve)
  #   stdm <- matrix(0,n,nurve)
  #   for (i in 1:n) {
  #     linkFni  <- linkFn[[i]]
  #     eta[i,] <- linkFni(mu[i,])
  #   }
  }
  
  #--------------------------------------------------------------------------
  #                        Set up for iterations
  #--------------------------------------------------------------------------
  
  iter     <- 0
  iterLim  <- 100
  seps     <- sqrt(eps)
  convcrit <- 1e-6
  sqrtwt   <- sqrt(wtvec)
  
  #  set up starting value bvec0 if required
  
  if (is.null(bvec0)) {
    bvec0 <- matrix(0,nbasis,ncurve)
  }
  bvec <- bvec0
  
  # Enforce limits on mu to guard against an inverse linkFn that doesn"t map 
  # into the support of the distribution.
  
  if (family == "binomial") {
    # mu is a probability, so order one is the natural scale, and eps is a
    # reasonable lower limit on that scale (plus it"s symmetric).
    eps <- 1e-16
    muLims <- c(eps, 1-eps)
  }
  if (family == "poisson" || family == "gamma" || family == "inverse gaussian") {
    # Here we don"t know the natural scale for mu, so make the lower limit
    # small.  This choice keeps mu^4 from underflowing.  No upper limit.
    muLims <- 1e-4
  }
  
  #--------------------------------------------------------------------------
  #                       Start of GLM iteration loop
  #--------------------------------------------------------------------------
  
  while (iter <= iterLim) {
    iter <- iter+1
    
    # Compute adjusted dep}ent variable for least squares fit
    
    if (is.character(family)) {
      Deta <- DlinkFn(mu)
      stdm <- stdFn(mu)
    # } else {
    #   for (i in 1:n) {
    #     DlinkFni  <- DlinkFn[[i]]
    #     stdFni    <- stdFn[[i]]
    #     mui       <- mu[i,]
    #     Deta[i,]  <- DlinkFni(mui)
    #     stdm[i,]  <- stdFni(mui)
    #   }
    }
    Zvec <- eta + (y - mu)*Deta
    
    # Compute IRLS weights the inverse of the variance function
    
    sqrtw <- (sqrtwt %*% matrix(1,1,ncurve))/(abs(Deta)*stdm)
    
    # Compute coefficient estimates for this iteration - the IRLS step
    
    bvec.old   <- bvec
    if (!is.null(addterm)) {
      ytmp <- Zvec - addterm
    } else {
      ytmp <- Zvec
    }
    yw   <- ytmp*sqrtw
    basismatw   <- basismat*(sqrtwt %*% matrix(1,1,nbasis))
    if (is.null(lamRmat)) {
      Mmat <- crossprod(basismatw)
    } else {
      Mmat <- crossprod(basismatw) + lamRmat
    }
    bvec    <- solve(Mmat,t(basismatw)) %*% yw
    if (!is.null(addterm)) {
      eta <- basismat %*% bvec + addterm
    } else {
      eta <- basismat %*% bvec
    }
    if (is.character(family)) {
      mu <- IlinkFn(eta)
    # } else {
    #   for (i in 1:n) {
    #     IlinkFni <- IlinkFn[[i]]
    #     mu[i,]   <- IlinkFni(eta[i,])
    #   }
    }
    
    # Force mean in bounds, in case the linkFn function is faulty
    
    if (is.character(family)) {
      if (family == "binomial") {
        if (any(mu < muLims[1] | muLims[2] < mu)) {
          for (j in 1:n) {
            mu[,j] <- max(min(mu[,j],muLims[2]),muLims[1])
          }
        }
      }
      if (family == "poisson" || 
          family == "gamma" || 
          family == "inverse gaussian") {
        if (any(mu < muLims[1])) {
          for (j in 1:n) {
            mu[j] <- max(mu[j],muLims[1])
          }
        }
      }
    # } else {
    #   for (i in 1:n) {
    #     familyi <- family[[i]]
    #     if (family == "binomial") {
    #       if (any(mu[i,] < muLims[1] | muLims(2) < mu[i,])) {
    #         for (j in 1:m) {
    #           mu[i,j] <- max(min(mu[i,j],muLims[2]),muLims[1])
    #         }
    #       }
    #     }
    #     if (family == "poisson" || family == "gamma" || 
    #         family == "inverse gaussian") {
    #       if (any(mu[i,] < muLims[1])) {
    #         for (j in 1:m) {
    #           mu[i,j]q() <- max(mu[i,j],muLims[1])
    #         }
    #       }
    #     }
    #   }
    }
    
    # Check stopping conditions
    
    print(max(abs(bvec-bvec.old)))
    if (max(abs(bvec-bvec.old)) < 
        convcrit*max(max(abs(bvec.old))) ) {
      break 
    }
    
  }
  
  #--------------------------------------------------------------------------
  #                    end of GLM iteration loop
  #--------------------------------------------------------------------------
  
  if (iter > iterLim) {
    warning("Iteration limit reached.")
  }
  
  # Sum components of deviance to get the total deviance.
  
  if (is.character(family)) {
    di       <- devFn(mu,y)
    Deviance <- sum((wtvec %*% matrix(1,1,ncurve))*di)
  # } else {
  #   Deviance <- matrix(0,n,ncurve)
  #   for (i in 1:n) {
  #     devFni <- devFn[[i]]
  #     di     <- devFni(mu[i,],y[,i])
  #     Deviance[i,] <- sum((wtvec[i]*matrix(1,1,ncurve))*di)
  #   }
  }
  
  return(list(bvec=bvec, Deviance=Deviance))
  
}

constrain <- function(eta, loBnd, upBnd) {
  eta[eta<loBnd] <- loBnd
  eta[eta>upBnd] <- upBnd
  return(eta)
}

hex <- function(ctr=c(0,0), rad=1, sig=0, meshlevel=1) {
  #  HEX sets up six boundary points around a hexagon,
  #  the PET architecture, and data at 19 locations
  angle <- seq(0,2*pi,len=7)
  x <- round(rad*cos(angle) + ctr[1],7)
  y <- round(rad*sin(angle) + ctr[2],7)
  edg <- matrix(0,7,2)
  edg[1:6,] <- cbind(x[1:6],y[1:6])
  edg[  7,] <- edg[1,]
  #  define the mesh
  if (meshlevel == 1) {
    pts  <- matrix(0,7,2)
    tri  <- matrix(0,6,3)
    loc  <- matrix(0,19,2)
    pts[1:6,] <- edg[1:6,]
    tri[,1] <- 7
    tri[1:6,2]  <- 1:6
    tri[   ,3]  <- c(2:6,1)
    loc[ 1: 6,] <- pts[1:6,]
    loc[ 7:12,] <- (pts[1:6,1] + pts[1:6,2])/2
    loc[13:19,] <- pts/2
    edg <- edg[1:6,]
  } else if (meshlevel == 2) {
    pts <- matrix(0,19,2)
    #  set up points matrix
    mpts2 <- 0
    for (i in 1:6) {
      mpts1 <- mpts2 + 1
      mpts2 <- mpts2 + 3
      pts[mpts1,  ] <-  edg[i,]
      pts[mpts1+1,] <-  edg[i,]/2
      pts[mpts1+2,] <- (edg[i,] + edg[i+1,])/2
    }
    pts[   19,] <- c(0,0)
    #  set up triangle matrix
    tri <- matrix(0,24,3)
    mpts2 <- 0
    mtri2 <- 0
    for (i in 1:6) {
      mpts1 <- mpts2 + 1
      mpts2 <- mpts2 + 3
      mtri1 <- mtri2 + 1
      mtri2 <- mtri2 + 4
      
      tri[mtri1  ,1] <- mpts1
      tri[mtri1  ,2] <- mpts1+2
      tri[mtri1  ,3] <- mpts1+1
      
      tri[mtri1+1,1] <- mpts1+1
      tri[mtri1+1,2] <- mpts1+2
      tri[mtri1+1,3] <- (mpts1+4) %% 18
      
      tri[mtri1+2,1] <- mpts1+2
      tri[mtri1+2,2] <- (mpts1+3) %% 18
      tri[mtri1+2,3] <- (mpts1+4) %% 18
      
      tri[mtri1+3,1] <- mpts1+1
      tri[mtri1+3,2] <- (mpts1+4) %% 18
      tri[mtri1+3,3] <- 19        
    }
    pts <- rbind(pts[1:18,], matrix(0,1,2))
    edg <- edg[1:6,]
    loc <- pts
  } else {
    stop('Argument meshlevel is neither 1 or 2.')
  }
  dat <- 1 - loc[,1]^2 - loc[,2]^2
  dat <- dat + rnorm(19,1)*sig
  return(list(p=pts, e=edg, t=tri, loc=loc, dat=dat))
}

inprod.bspline <- function(fdobj1, fdobj2=fdobj1, nderiv1=0, nderiv2=0) {
#INPROD_BSPLINE  computes matrix of inner products of the derivatives
#  of order DERIV1 and DERIV2 of two functional data objects
#  FD1 and FD2, respectively.
#  These must both have Bspline bases, and these bases must have
#  a common break point sequence.   However, the orders can differ.
#  If only the first argument is present, the inner products of
#  FD1 with itself is taken.  If argument DERIV is not supplied,
#  it is taken to be 0.
#

#  Last modified 26 October 2005

if(!inherits(fdobj1,"fd")) stop("FD1 is not a functional data object.")
if(!inherits(fdobj2,"fd")) stop("FD2 is not a functional data object.")

basis1 <- fdobj1$basis
type1  <- basis1$type
if(type1!="bspline")   stop("FDOBJ1 does not have a B-spline basis.")

range1  <- basis1$rangeval
breaks1 <- c(range1[1], basis1$params, range1[2])
nbasis1 <- basis1$nbasis
norder1 <- nbasis1 - length(breaks1) + 2

basis2 <- fdobj2$basis
type2  <- basis2$type
if(type2!="bspline")    stop("FDOBJ2 does not have a B-spline basis.")

range2  <- basis2$rangeval
breaks2 <- c(range2[1], basis2$params, range2[2])
nbasis2 <- basis2$nbasis
norder2 <- nbasis2 - length(breaks2) + 2

if(any((range1 - range2) != 0)) stop(
	"The argument ranges for FDOBJ1 and FDOBJ2 are not identical.")

#  check that break values are equal and set up common array

if(length(breaks1) != length(breaks2)) stop(
	"The numbers of knots for FDOBJ1 and FDOBJ2 are not identical")

if(any((breaks1 - breaks2) != 0)) stop(
	"The knots for FDOBJ1 and FDOBJ2 are not identical.")
else    breaks <- breaks1

if(length(breaks) < 2) stop(
	"The length of argument BREAKS is less than 2.")

breakdiff <- diff(breaks)
if(min(breakdiff) <= 0) stop(
	"Argument BREAKS is not strictly increasing.")

#  set up the two coefficient matrices

coef1 <- as.matrix(fdobj1$coefs)
coef2 <- as.matrix(fdobj2$coefs)
if(length(dim(coef1)) > 2) stop("FDOBJ1 is not univariate.")
if(length(dim(coef2)) > 2) stop("FDOBJ2 is not univariate.")

nbreaks   <- length(breaks)
ninterval <- nbreaks - 1
nbasis1   <- ninterval + norder1 - 1
nbasis2   <- ninterval + norder2 - 1
if (dim(coef1)[1] != nbasis1 || dim(coef2)[1] != nbasis2)
    stop(paste("Error: coef1 should have length no. breaks1+norder1-2",
           "and coef2 no. breaks2+norder2-2."))

breaks1 <- breaks[1]
breaksn <- breaks[nbreaks]

# The knot sequences are built so that there are no continuity conditions
# at the first and last breaks.  There are k-1 continuity conditions at
# the other breaks.

temp   <- breaks[2:(nbreaks-1)]
knots1 <- c(breaks1*rep(1,norder1),temp,breaksn*rep(1,norder1))
knots2 <- c(breaks1*rep(1,norder2),temp,breaksn*rep(1,norder2))

# Construct  the piecewise polynomial representation of
#    f^(DERIV1) and g^(DERIV2)

nrep1 <- dim(coef1)[2]
polycoef1 <- array(0,c(ninterval,norder1-nderiv1,nrep1))
for (i in 1:nbasis1) {
    #  compute polynomial representation of B(i,norder1,knots1)(x)
    ppBlist <- ppBspline(knots1[i:(i+norder1)])
    Coeff <- ppBlist[[1]]
    index <- ppBlist[[2]]
    # convert the index of the breaks in knots1 to the index in the
    # variable "breaks"
    index <- index + i - norder1
    CoeffD <- ppderiv(Coeff,nderiv1)  # differentiate B(i,norder1,knots1)(x)
    # add the polynomial representation of B(i,norder1,knots1)(x) to f
    if (nrep1 == 1)
       polycoef1[index,,1] <- coef1[i]*CoeffD +
                                        polycoef1[index,,1]
    else {
        for (j in 1:length(index)){
	         temp <- outer(CoeffD[j,],coef1[i,])
	         polycoef1[index[j],,] <-  temp + polycoef1[index[j],,]
        }
    }
}

nrep2 <- dim(coef2)[2]
polycoef2 <- array(0,c(ninterval,norder2-nderiv2,nrep2))
for(i in 1:nbasis2){
    #  compute polynomial representation of B(i,norder2,knots2)(x)
    ppBlist <- ppBspline(knots2[i:(i+norder2)])
    Coeff <- ppBlist[[1]]
    index <- ppBlist[[2]]
    # convert the index of the breaks in knots2 to the index in the
    # variable "breaks"
    index <- index + i - norder2
    CoeffD <- ppderiv(Coeff, nderiv2)  # differentiate B(i,norder2,knots2)(x)
    # add the polynomial representation of B(i,norder2,knots2)(x) to g
    if (nrep2 == 1) polycoef2[index,,1] <- coef2[i]*CoeffD +
                                           polycoef2[index,,1]
    else{
        for(j in 1:length(index)){
	         polycoef2[index[j],,] <- outer(CoeffD[j,],coef2[i,]) +
	                                    polycoef2[index[j],,]
        }
    }
}

# Compute the scalar product between f and g

prodmat <- matrix(0,nrep1,nrep2)
for (j in 1:ninterval){
    # multiply f(i1) and g(i2) piecewise and integrate
    c1 <- as.matrix(polycoef1[j,,])
    c2 <- as.matrix(polycoef2[j,,])
    polyprodmat <- polyprod(c1,c2)
    # compute the coefficients of the anti-derivative
    N <- dim(polyprodmat)[3]
    delta <- breaks[j+1] - breaks[j]
    power <- delta
    prodmati <- matrix(0,nrep1,nrep2)
    for (i in 1:N){
        prodmati <- prodmati + power*polyprodmat[,,N-i+1]/i
        power    <- power*delta
    }
    # add the integral to s
    prodmat <- prodmat + prodmati
}

prodmat

}
inprod <- function(fdobj1, fdobj2=NULL, Lfdobj1=int2Lfd(0), Lfdobj2=int2Lfd(0),
                   rng = range1, wtfd = 0)
{

#  computes matrix of inner products of functions by numerical
#    integration using Romberg integration

#  Arguments:
#  FDOBJ1 and FDOBJ2    These may be either functional data or basis
#               function objects.  In the latter case, a functional
#               data object is created from a basis function object
#               by using the identity matrix as the coefficient matrix.
#               Both functional data objects must be univariate.
#               If inner products for multivariate objects are needed,
#               use a loop and call inprod(FDOBJ1[i],FDOBJ2[i]).
#     If FDOBJ2 is not provided or is NULL, it defaults to a function
#     having a constant basis and coefficient 1 for all replications.
#     This permits the evaluation of simple integrals of functional data
#     objects.
#  LFDOBJ1 and LFDOBJ2  order of derivatives for inner product for
#               FDOBJ1 and FDOBJ2, respectively, or functional data
#               objects defining linear differential operators
#  RNG    Limits of integration
#  WTFD   A functional data object defining a weight
#  JMAX   maximum number of allowable iterations
#  EPS    convergence criterion for relative stop

#  Return:
#  A matrix of NREP1 by NREP2 of inner products for each possible pair
#  of functions.

#  Last modified 6 January 2020 by Jim Ramsay

#  Check FDOBJ1 and get no. replications and basis object

result1   <- fdchk(fdobj1)
nrep1     <- result1[[1]]
fdobj1    <- result1[[2]]
coef1     <- fdobj1$coefs
basisobj1 <- fdobj1$basis
type1     <- basisobj1$type
range1    <- basisobj1$rangeval

#  Default FDOBJ2 to a constant function, using a basis that matches
#  that of FDOBJ1 if possible.

if (is.null(fdobj2)) {
    tempfd    <- fdobj1
    tempbasis <- tempfd$basis
    temptype  <- tempbasis$type
    temprng   <- tempbasis$rangeval
    if (temptype == "bspline") {
        basis2 <- create.bspline.basis(temprng, 1, 1)
    } else {
        if (temptype == "fourier") basis2 <- create.fourier.basis(temprng, 1)
        else                       basis2 <- create.constant.basis(temprng)
    }
    fdobj2 <- fd(1,basis2)
}

#  Check FDOBJ2 and get no. replications and basis object

result2   <- fdchk(fdobj2)
nrep2     <- result2[[1]]
fdobj2    <- result2[[2]]
coef2     <- fdobj2$coefs
basisobj2 <- fdobj2$basis
type2     <- basisobj2$type
range2    <- basisobj2$rangeval

# check ranges

if (rng[1] < range1[1] || rng[2] > range1[2]) stop(
	 "Limits of integration are inadmissible.")

#  Call B-spline version if
#  [1] both functional data objects are univariate
#  [2] both bases are B-splines
#  (3) the two bases are identical
#  (4) both differential operators are integers
#  (5) there is no weight function
#  (6) RNG is equal to the range of the two bases.

if (is.fd(fdobj1)                    && 
    is.fd(fdobj2)                    &&
    type1 == "bspline"               && 
    type2 == "bspline"               &&
    is.eqbasis(basisobj1, basisobj2) &&
    is.integer(Lfdobj1)              && 
    is.integer(Lfdobj2)              &&
    length(basisobj1$dropind) == 0   &&
    length(basisobj1$dropind) == 0   &&
    wtfd == 0                        && all(rng == range1)) {

    inprodmat <- inprod.bspline(fdobj1, fdobj2,
                     Lfdobj1$nderiv, Lfdobj2$nderiv)
    return(inprodmat)
}

#  check LFDOBJ1 and LFDOBJ2

Lfdobj1 <- int2Lfd(Lfdobj1)
Lfdobj2 <- int2Lfd(Lfdobj2)

#  Else proceed with the use of the Romberg integration.

#  ------------------------------------------------------------
#  Now determine the number of subintervals within which the
#  numerical integration takes.  This is important if either
#  basis is a B-spline basis and has multiple knots at a
#  break point.
#  ------------------------------------------------------------

#  set iter

iter <- 0

# The default case, no multiplicities.

rngvec <- rng

#  check for any knot multiplicities in either argument

knotmult <- numeric(0)
if (type1 == "bspline") knotmult <- knotmultchk(basisobj1, knotmult)
if (type2 == "bspline") knotmult <- knotmultchk(basisobj2, knotmult)

#  Modify RNGVEC defining subinvervals if there are any
#  knot multiplicities.

if (length(knotmult) > 0) {
    knotmult <- sort(unique(knotmult))
    knotmult <- knotmult[knotmult > rng[1] && knotmult < rng[2]]
    rngvec   <- c(rng[1], knotmult, rng[2])
}

#  check for either coefficient array being zero

if ((all(c(coef1) == 0) || all(c(coef2) == 0)))
	return(matrix(0,nrep1,nrep2))

#  -----------------------------------------------------------------
#                   loop through sub-intervals
#  -----------------------------------------------------------------

#  Set constants controlling convergence tests

JMAX <- 15
JMIN <-  5
EPS  <- 1e-4

inprodmat <- matrix(0,nrep1,nrep2)

nrng <- length(rngvec)
for (irng  in  2:nrng) {
    rngi <- c(rngvec[irng-1],rngvec[irng])
    #  change range so as to avoid being exactly on
    #  multiple knot values
    if (irng > 2   ) rngi[1] <- rngi[1] + 1e-10
    if (irng < nrng) rngi[2] <- rngi[2] - 1e-10

    #  set up first iteration

    iter  <- 1
    width <- rngi[2] - rngi[1]
    JMAXP <- JMAX + 1
    h <- rep(1,JMAXP)
    h[2] <- 0.25
    s <- array(0,c(JMAXP,nrep1,nrep2))
    sdim <- length(dim(s))
    #  the first iteration uses just the endpoints
    fx1 <- eval.fd(rngi, fdobj1, Lfdobj1)
    fx2 <- eval.fd(rngi, fdobj2, Lfdobj2)
    #  multiply by values of weight function if necessary
    if (!is.numeric(wtfd)) {
        wtd <- eval.fd(rngi, wtfd, 0)
        fx2 <- matrix(wtd,dim(wtd)[1],dim(fx2)[2]) * fx2
    }
    s[1,,] <- width*matrix(crossprod(fx1,fx2),nrep1,nrep2)/2
    tnm  <- 0.5

    #  now iterate to convergence

    for (iter in 2:JMAX) {
        tnm <- tnm*2
        if (iter == 2) {
            x <- mean(rngi)
        } else {
            del <- width/tnm
            x   <- seq(rngi[1]+del/2, rngi[2]-del/2, del)
        }
        fx1 <- eval.fd(x, fdobj1, Lfdobj1)
        fx2 <- eval.fd(x, fdobj2, Lfdobj2)
        if (!is.numeric(wtfd)) {
            wtd <- eval.fd(wtfd, x, 0)
            fx2 <- matrix(wtd,dim(wtd)[1],dim(fx2)[2]) * fx2
        }
        chs <- width*matrix(crossprod(fx1,fx2),nrep1,nrep2)/tnm
        s[iter,,] <- (s[iter-1,,] + chs)/2
        if (iter >= 5) {
            ind <- (iter-4):iter
            ya <- s[ind,,]
            ya <- array(ya,c(5,nrep1,nrep2))
            xa <- h[ind]
            absxa <- abs(xa)
            absxamin <- min(absxa)
            ns <- min((1:length(absxa))[absxa == absxamin])
            cs <- ya
            ds <- ya
            y  <- ya[ns,,]
            ns <- ns - 1
            for (m in 1:4) {
                for (i in 1:(5-m)) {
                    ho      <- xa[i]
                    hp      <- xa[i+m]
                    w       <- (cs[i+1,,] - ds[i,,])/(ho - hp)
                    ds[i,,] <- hp*w
                    cs[i,,] <- ho*w
                }
                if (2*ns < 5-m) {
                    dy <- cs[ns+1,,]
                } else {
                    dy <- ds[ns,,]
                    ns <- ns - 1
                }
                y <- y + dy
            }
            ss     <- y
            errval <- max(abs(dy))
            ssqval <- max(abs(ss))
            if (all(ssqval > 0)) {
                crit <- errval/ssqval
            } else {
                crit <- errval
            }
            if (crit < EPS && iter >= JMIN) break
        }
        s[iter+1,,] <- s[iter,,]
        h[iter+1]   <- 0.25*h[iter]
        if (iter == JMAX) warning("Failure to converge.")
    }
    inprodmat <- inprodmat + ss

}

if(length(dim(inprodmat) == 2)) {
    #  coerce inprodmat to be nonsparse
    return(as.matrix(inprodmat))
} else {
    #  allow inprodmat to be sparse if it already is
    return(inprodmat)
}

}

#  -------------------------------------------------------------------------------

fdchk <- function(fdobj) {
  
  #  check the class of FDOBJ and extract coefficient matrix
  
  if (inherits(fdobj, "fd")) {
    coef  <- fdobj$coefs
  } else {
    if (inherits(fdobj, "basisfd")) {
      coef  <- diag(rep(1,fdobj$nbasis - length(fdobj$dropind)))
      fdobj <- fd(coef, fdobj)
    } else { 
      stop("FDOBJ is not an FD object.")
    }
  }
  
  #  extract the number of replications and basis object
  
  coefd <- dim(as.matrix(coef))
  if (length(coefd) > 2) stop("Functional data object must be univariate")
  nrep     <- coefd[2]
  basisobj <- fdobj$basis
  
  return(list(nrep, fdobj))
  
}

#  -------------------------------------------------------------------------------

knotmultchk <- function(basisobj, knotmult) {
    type <- basisobj$type
    if (type == "bspline") {
        # Look for knot multiplicities in first basis
        params  <- basisobj$params
        nparams <- length(params)
        norder  <- basisobj$nbasis - nparams
        if (norder == 1) {
        	knotmult <- c(knotmult, params)
        } else {
          if (nparams > 1) {
            for (i in 2:nparams) 
                if (params[i] == params[i-1]) knotmult <- c(knotmult, params[i])
          }
        }
    }
    return(knotmult)
}


int2Lfd <- function(m=0)
{
#INT2LFD converts a nonnegative integer to a linear differential
#  operator object that is equivalent to D^m.  The range of the
#  functional data object in any cell is set to [0,1], and is
#  not actually used when a linear differential operator object
#  of this nature is applied.  
#  In the event that m is already a linear differential operator
#  object, it returns the object immediately.  Thus, INT2LFD can
#  be used to screen whether an object is an integer or not.

#  Last modified 17 September 2005

#  check M

if (inherits(m, "Lfd")) {
    Lfdobj <- m
    return(Lfdobj)
}

if (!is.numeric(m)) 
    stop("Argument not numeric and not a linear differential operator.")
 

if (length(m) != 1) stop("Argument is not a scalar.")

if (round(m) != m)  stop("Argument is not an integer.")

if (m < 0)   stop("Argument is negative.")

#  all the checks passed, set up a functional data object
#  The range will not be used in this case, and can be set
#  to [0, 1]

#  set up the list object for the homogeneous part

if (m==0) {
    #  if derivative is zero, BWTLIST is empty
    bwtlist <- NULL
} else {
    basisobj <- create.constant.basis(c(0,1))
    bwtlist  <- vector("list", m)
    for (j in 1:m) bwtlist[[j]] <- fd(0, basisobj)
}

#  define the Lfd object

Lfdobj <- Lfd(m, bwtlist)

return(Lfdobj)

}
intensity.fd <- function(x, WfdParobj, conv=0.0001, iterlim=20, dbglev=1, 
                            returnMatrix=FALSE) {
# INTENSITYFD estimates the intensity function \lambda(x) of a
#  nonhomogeneous Poisson process from a sample of event times.

#  Arguments are:
#  X         ... data value array.
#  WFDPAROBJ ... functional parameter object specifying the initial log
#              density, the linear differential operator used to smooth
#              smooth it, and the smoothing parameter.
#  CONV      ... convergence criterion
#  ITERLIM   ... iteration limit for scoring iterations
#  DBGLEV    ... level of output of computation history

#  Returns:
#  A list containing
#  WFDOBJ ...   functional data basis object defining final log intensity
#  FLIST  ...   Struct object containing
#               FSTR$f     final log likelihood
#               FSTR$norm  final norm of gradient
#  ITERNUM   Number of iterations
#  ITERHIST  History of iterations
#  RETURNMATRIX ... If False, a matrix in sparse storage model can be returned
#               from a call to function BsplineS.  See this function for
#               enabling this option.

#  last modified 10 May 2012 by Jim Ramsay

	#  check WfdParobj
	
	if (!inherits(WfdParobj, "fdPar"))
		if (inherits(WfdParobj, "fd") || inherits(WfdParobj, "basisfd"))
			WfdParobj <- fdPar(WfdParobj)
		else stop("WFDPAROBJ is not a fdPar object")
					
	#  set up WFDOBJ

	Wfdobj   <- WfdParobj$fd

	#  set up LFDOBJ
	
	Lfdobj <- WfdParobj$Lfd
	Lfdobj <- int2Lfd(Lfdobj)

	#  set up BASIS

	basisobj <- Wfdobj$basis
	nbasis   <- basisobj$nbasis
	rangex   <- basisobj$rangeval
	active   <- 1:nbasis
	
	x    <- as.vector(x)
	N    <- length(x)

	#  check for values outside of the range of WFD0

	inrng <- (1:N)[x >= rangex[1] & x <= rangex[2]]
	if (length(inrng) != N) {
		print(c(length(inrng), N))
		print(c(rangex[1], rangex[2], min(x), max(x)))
    	warning("Some values in X out of range and not used.")
	}

	x     <- x[inrng]
	nobs  <- length(x)

	#  set up some arrays

	climit    <- c(rep(-50,nbasis),rep(400,nbasis))
	cvec0     <- Wfdobj$coefs
	hmat      <- matrix(0,nbasis,nbasis)
	dbgwrd    <- dbglev > 1

	#  initialize matrix Kmat defining penalty term

	lambda <- WfdParobj$lambda
	if (lambda > 0) Kmat <- lambda*getbasispenalty(basisobj, Lfdobj)
  
	#  evaluate log likelihood
	#    and its derivatives with respect to these coefficients

	result <- loglfninten(x, basisobj, cvec0, returnMatrix)
	logl   <- result[[1]]
	Dlogl  <- result[[2]]

	#  compute initial badness of fit measures

	f0    <- -logl
	gvec0 <- -Dlogl
	if (lambda > 0) {
   		gvec0 <- gvec0 +           2*(Kmat %*% cvec0)
   		f0    <- f0    + t(cvec0) %*% Kmat %*% cvec0
	}
	Foldstr <- list(f = f0, norm = sqrt(mean(gvec0^2)))

	#  compute the initial expected Hessian

	hmat0 <- Varfninten(basisobj, cvec0, returnMatrix)
	if (lambda > 0) hmat0 <- hmat0 + 2*Kmat

	#  evaluate the initial update vector for correcting the initial bmat

	deltac   <- -solve(hmat0,gvec0)
	cosangle <- -sum(gvec0*deltac)/sqrt(sum(gvec0^2)*sum(deltac^2))

	#  initialize iteration status arrays

	iternum <- 0
	status <- c(iternum, Foldstr$f, -logl, Foldstr$norm)
	if (dbglev > 0) {
		cat("Iteration  Criterion  Neg. Log L  Grad. Norm\n")
		cat("      ")
		cat(format(iternum))
		cat("    ")
		cat(format(status[2:4]))
		cat("\n")
	}
	iterhist <- matrix(0,iterlim+1,length(status))
	iterhist[1,]  <- status
	
	#  quit if ITERLIM == 0
	
	if (iterlim == 0) {
    	Flist     <- Foldstr
    	iterhist <- iterhist[1,]
    	return( list(Wfdobj=Wfdobj, Flist=Flist, iternum=iternum, iterhist=iterhist) )
	} else {
		gvec <- gvec0
		hmat <- hmat0
	}

	#  -------  Begin iterations  -----------

	STEPMAX <- 5
	MAXSTEP <- 400
	trial   <- 1
	cvec    <- cvec0
	linemat <- matrix(0,3,5)

	for (iter in 1:iterlim) {
   		iternum <- iternum + 1
	   	#  take optimal stepsize
   		dblwrd <- c(0,0)
		  limwrd <- c(0,0)
		  stpwrd <- 0
		  ind    <- 0
	   	#  compute slope
      	Flist <- Foldstr
      	linemat[2,1] <- sum(deltac*gvec)
      	#  normalize search direction vector
      	sdg     <- sqrt(sum(deltac^2))
      	deltac  <- deltac/sdg
      	dgsum   <- sum(deltac)
      	linemat[2,1] <- linemat[2,1]/sdg
      	#  return with stop condition if (initial slope is nonnegative
      	if (linemat[2,1] >= 0) {
        	print("Initial slope nonnegative.")
        	ind <- 3
        	iterhist <- iterhist[1:(iternum+1),]
        	break
      	}
      	#  return successfully if (initial slope is very small
      	if (linemat[2,1] >= -1e-5) {
        	if (dbglev>1) print("Initial slope too small")
        	iterhist <- iterhist[1:(iternum+1),]
        	break
      	}
    	#  load up initial search matrix
      	linemat[1,1:4] <- 0
      	linemat[2,1:4] <- linemat[2,1]
      	linemat[3,1:4] <- Foldstr$f
     	#  output initial results for stepsize 0
     	stepiter  <- 0
      	if (dbglev > 1) {
			cat("              ")
			cat(format(stepiter))
			cat(format(linemat[,1]))
			cat("\n")
		}
      	ips <- 0
      	#  first step set to trial
      	linemat[1,5]  <- trial
      	#  Main iteration loop for linesrch
      	for (stepiter in 1:STEPMAX) {
        	#  ensure that step does not go beyond limits on parameters
        	limflg  <- 0
        	#  check the step size
        	result <- stepchk(linemat[1,5], cvec, deltac, limwrd, ind,
                            climit, active, dbgwrd)
			linemat[1,5] <- result[[1]]
			ind          <- result[[2]]
			limwrd       <- result[[3]]
       	if (linemat[1,5] <= 1e-9) {
          		#  Current step size too small  terminate
          		Flist   <- Foldstr
          		cvecnew <- cvec
          		gvecnew <- gvec
          		if (dbglev > 1) print(paste("Stepsize too small:", linemat[1,5]))
          		if (limflg) ind <- 1 else ind <- 4
          		break
        	}
        	cvecnew <- cvec + linemat[1,5]*deltac
        	#  compute new function value and gradient
			    result  <- loglfninten(x, basisobj, cvecnew, returnMatrix)
			    logl    <- result[[1]]
			    Dlogl   <- result[[2]]
        	Flist$f <- -logl
        	gvecnew <- -Dlogl
        	if (lambda > 0) {
            	gvecnew <- gvecnew + 2*Kmat %*% cvecnew
            	Flist$f <- Flist$f + t(cvecnew) %*% Kmat %*% cvecnew
        	}
        	Flist$norm <- sqrt(mean(gvecnew^2))
        	linemat[3,5] <- Flist$f
        	#  compute new directional derivative
        	linemat[2,5] <- sum(deltac*gvecnew)
      		if (dbglev > 1) {
				cat("              ")
				cat(format(stepiter))
				cat(format(linemat[,1]))
				cat("\n")
			}
        	#  compute next step
			result  <- stepit(linemat, ips, dblwrd, MAXSTEP)
			linemat <- result[[1]]
			ips     <- result[[2]]
			ind     <- result[[3]]
			dblwrd  <- result[[4]]
        	trial   <- linemat[1,5]
        	#  ind == 0 implies convergence
        	if (ind == 0 | ind == 5) break
        	#  end of line search loop
     	}

    	#  update current parameter vectors

    	cvec <- cvecnew
     	gvec <- gvecnew
	  	Wfdobj$coefs <- cvec
     	status <- c(iternum, Flist$f, -logl, Flist$norm)
     	iterhist[iter+1,] <- status
		cat("      ")
		cat(format(iternum))
		cat("    ")
		cat(format(status[2:4]))
		cat("\n")

     	#  test for convergence

     	if (abs(Flist$f-Foldstr$f) < conv) {
       	iterhist <- iterhist[1:(iternum+1),]
			denslist <- list("Wfdobj" = Wfdobj, "Flist" = Flist,
			          			"iternum" = iternum, "iterhist" = iterhist)
			return( denslist )
     	}
     	if (Flist$f >= Foldstr$f) break
     	#  compute the Hessian
     	hmat <- Varfninten(basisobj, cvec, returnMatrix)
     	if (lambda > 0) hmat <- hmat + 2*Kmat
     	#  evaluate the update vector
     	deltac <- -solve(hmat,gvec)
     	cosangle  <- -sum(gvec*deltac)/sqrt(sum(gvec^2)*sum(deltac^2))
     	if (cosangle < 0) {
       	if (dbglev > 1) print("cos(angle) negative")
       	deltac <- -gvec
     	}
     	Foldstr <- Flist
		#  end of iterations
  	}
	#  return final results
	intenslist <- list("Wfdobj" = Wfdobj, "Flist" = Flist,
			          "iternum" = iternum, "iterhist" = iterhist)
 	return( intenslist )
}

#  ---------------------------------------------------------------

loglfninten <- function(x, basisobj, cvec, returnMatrix=FALSE) {
	#  Computes the log likelihood and its derivative with
	#    respect to the coefficients in CVEC
   	nobs    <- length(x)
   	cval    <- normint.phi(basisobj, cvec, returnMatrix=returnMatrix)
   	phimat  <- getbasismatrix(x, basisobj, 0, returnMatrix)
   	logl    <- sum(phimat %*% cvec) - cval
	  EDW     <- expect.phi(basisobj, cvec, returnMatrix=returnMatrix)
   	Dlogl   <- apply(phimat,2,sum) - EDW
	return( list(logl, Dlogl) )
}

#  ---------------------------------------------------------------

Varfninten <- function(basisobj, cvec, returnMatrix=FALSE) {
	#  Computes the expected Hessian
   	Varphi  <- expect.phiphit(basisobj, cvec, returnMatrix=returnMatrix)
	return(Varphi)
}
	
#  ---------------------------------------------------------------

normint.phi <- function(basisobj, cvec, JMAX=15, EPS=1e-7, returnMatrix=FALSE) 
{

#  Computes integrals of
#      p(x) = exp phi'(x) %*% cvec
#  by numerical integration using Romberg integration

  	#  check arguments, and convert basis objects to functional data objects

  	if (!inherits(basisobj, "basisfd") )
    	stop("First argument must be a basis function object.")

	  nbasis <- basisobj$nbasis
	  rng    <- basisobj$rangeval
  	oneb   <- matrix(1,1,nbasis)

  	#  set up first iteration

  	width <- rng[2] - rng[1]
  	JMAXP <- JMAX + 1
  	h <- matrix(1,JMAXP,1)
  	h[2] <- 0.25
  	#  matrix SMAT contains the history of discrete approximations to the integral
  	smat <- matrix(0,JMAXP,1)
  	#  the first iteration uses just the }points
  	x  <- rng
  	nx <- length(x)
  	ox <- matrix(1,nx,1)
  	fx <- getbasismatrix(x, basisobj, 0, returnMatrix)
  	wx <- fx %*% cvec
  	wx[wx < -50] <- -50
  	px <- exp(wx)
  	smat[1]  <- width*sum(px)/2
  	tnm <- 0.5
  	j   <- 1

  	#  now iterate to convergence
  	for (j in 2:JMAX) {
    	tnm  <- tnm*2
    	del  <- width/tnm
    	if (j == 2) {
      		x <- (rng[1] + rng[2])/2
    	} else {
      		x <- seq(rng[1]+del/2, rng[2], del)
    	}
    	fx <- getbasismatrix(x, basisobj, 0, returnMatrix)
    	wx <- fx %*% cvec
    	wx[wx < -50] <- -50
    	px <- exp(wx)
    	smat[j] <- (smat[j-1] + width*sum(px)/tnm)/2
    	if (j >= 5) {
      		ind <- (j-4):j
			result <- polintarray(h[ind],smat[ind],0)
			ss  <- result[[1]]
			dss <- result[[2]]
      		if (!any(abs(dss) >= EPS*max(abs(ss)))) {
        		#  successful convergence
        		return(ss)
      		}
    	}
    	smat[j+1] <- smat[j]
    	h[j+1]    <- 0.25*h[j]
 	}
  	warning(paste("No convergence after ",JMAX," steps in NORMALIZE.PHI"))
	return(ss)
}

#  ---------------------------------------------------------------

expect.phi <- function(basisobj, cvec, nderiv=0, JMAX=15, EPS=1e-7, 
                       returnMatrix=FALSE) {
#  Computes expectations of basis functions with respect to intensity
#      p(x) <- exp t(c)*phi(x)
#  by numerical integration using Romberg integration

  	#  check arguments, and convert basis objects to functional data objects

  	if (!inherits(basisobj, "basisfd"))
    	stop("First argument must be a basis function object.")

  	nbasis <- basisobj$nbasis
  	rng    <- basisobj$rangeval
  	oneb   <- matrix(1,1,nbasis)

  	#  set up first iteration

  	width <- rng[2] - rng[1]
  	JMAXP <- JMAX + 1
  	h <- matrix(1,JMAXP,1)
  	h[2] <- 0.25
  	#  matrix SMAT contains the history of discrete approximations to the integral
  	smat <- matrix(0,JMAXP,nbasis)
  	sumj <- matrix(0,1,nbasis)
  	#  the first iteration uses just the }points
  	x  <- rng
  	nx <- length(x)
  	ox <- matrix(1,nx,nx)
  	fx <- as.matrix(getbasismatrix(x, basisobj, 0, returnMatrix))
  	wx <- fx %*% cvec
  	wx[wx < -50] <- -50
  	px <- exp(wx)
  	if (nderiv == 0) {
    	  Dfx <- fx
  	} else {
    	  Dfx <- as.matrix(getbasismatrix(x, basisobj, 1, returnMatrix))
  	}
  	sumj <- t(Dfx) %*% px
  	smat[1,]  <- width*sumj/2
  	tnm <- 0.5
  	j   <- 1

  	#  now iterate to convergence

  	for (j in 2:JMAX) {
    	tnm  <- tnm*2
    	del  <- width/tnm
    	if (j == 2) {
        x <- (rng[1] + rng[2])/2
    	} else {
        x <- seq(rng[1]+del/2, rng[2], del)
    	}
    	nx <- length(x)
    	fx <- as.matrix(getbasismatrix(x, basisobj, 0, returnMatrix))
    	wx <- fx %*% cvec
    	wx[wx < -50] <- -50
    	px <- exp(wx)
    	if (nderiv == 0) {
        Dfx <- fx
    	} else {
        Dfx <- as.matrix(getbasismatrix(x, basisobj, 1, returnMatrix))
    	}
    	sumj <- t(Dfx) %*% px
    	smat[j,] <- (smat[j-1,] + width*sumj/tnm)/2
    	if (j >= 5) {
      		ind <- (j-4):j
      		temp <- smat[ind,]
			result <- polintarray(h[ind],temp,0)
			ss  <- result[[1]]
			dss <- result[[2]]
      		if (!any(abs(dss) > EPS*max(abs(ss)))) {
        		#  successful convergence
        		return(ss)
      		}
    	}
    	smat[j+1,] <- smat[j,]
    	h[j+1] <- 0.25*h[j]
  	}
  	warning(paste("No convergence after ",JMAX," steps in EXPECT.PHI"))
	return(ss)
}

#  ---------------------------------------------------------------

expect.phiphit <- function(basisobj, cvec, nderiv1=0, nderiv2=0,
                           JMAX=15, EPS=1e-7, returnMatrix=FALSE) {

#  Computes expectations of cross product of basis functions with
#  respect to intensity
#      p(x) = exp t(c) %*% phi(x)
#  by numerical integration using Romberg integration

  	#  check arguments, and convert basis objects to functional data objects

  	if (!inherits(basisobj, "basisfd"))
    	stop("First argument must be a basis function object.")

  	nbasis <- basisobj$nbasis
  	rng    <- basisobj$rangeval
  	oneb   <- matrix(1,1,nbasis)

  	#  set up first iteration

  	width <- rng[2] - rng[1]
  	JMAXP <- JMAX + 1
  	h <- matrix(1,JMAXP,1)
  	h[2] <- 0.25
  	#  matrix SMAT contains the history of discrete approximations to the integral
  	smat <- array(0,c(JMAXP,nbasis,nbasis))
  	#  the first iteration uses just the }points
  	x  <- rng
  	nx <- length(x)
  	fx <- as.matrix(getbasismatrix(x, basisobj, 0, returnMatrix))
  	wx <- fx %*% cvec
  	wx[wx < -50] <- -50
  	px <- exp(wx)
  	if (nderiv1 == 0) {
    	  Dfx1 <- fx
  	} else {
    	  Dfx1 <- as.matrix(getbasismatrix(x, basisobj, 1, returnMatrix))
  	}
  	if (nderiv2 == 0) {
    	  Dfx2 <- fx
  	} else {
    	  Dfx2 <- as.matrix(getbasismatrix(x, basisobj, 1, returnMatrix))
  	}
  	oneb <- matrix(1,1,nbasis)
  	sumj <- t(Dfx1) %*% ((px %*% oneb) * Dfx2)
  	smat[1,,]  <- width*sumj/2
  	tnm <- 0.5
  	j   <- 1

  	#  now iterate to convergence
  	for (j in 2:JMAX) {
    	tnm  <- tnm*2
    	del  <- width/tnm
    	if (j == 2) {
        x <- (rng[1] + rng[2])/2
    	} else {
        x <- seq(rng[1]+del/2, rng[2], del)
    	}
    	nx <- length(x)
    	fx <- as.matrix(getbasismatrix(x, basisobj, 0, returnMatrix))
    	wx <- fx %*% cvec
    	wx[wx < -50] <- -50
    	px <- exp(wx)
    	if (nderiv1 == 0) {
        Dfx1 <- fx
    	} else {
        Dfx1 <- as.matrix(getbasismatrix(x, basisobj, 1, returnMatrix))
    	}
    	if (nderiv2 == 0) {
        Dfx2 <- fx
    	} else {
        Dfx2 <- as.matrix(getbasismatrix(x, basisobj, 2, returnMatrix))
    	}
    	sumj <- t(Dfx1) %*% ((px %*% oneb) * Dfx2)
    	smat[j,,] <- (smat[j-1,,] + width*sumj/tnm)/2
    	if (j >= 5) {
      		ind <- (j-4):j
      		temp <- smat[ind,,]
	   		result <- polintarray(h[ind],temp,0)
	   		ss  <- result[[1]]
	   		dss <- result[[2]]
      		if (!any(abs(dss) > EPS*max(max(abs(ss))))) {
        		#  successful convergence
        		return(ss)
      		}
    	}
    	smat[j+1,,] <- smat[j,,]
    	h[j+1] <- 0.25*h[j]
  	}
  	warning(paste("No convergence after ",JMAX," steps in EXPECT.PHIPHIT"))
	return(ss)
}
#  ---------------------------------------------------------------

polintarray <- function(xa, ya, x0) {
  	#  YA is an array with up to 4 dimensions
  	#     with 1st dim the same length same as the vector XA
  	n     <- length(xa)
  	yadim <- dim(ya)
  	if (is.null(yadim)) {
		yadim <- n
		nydim <- 1
  	} else {
    	nydim <- length(yadim)
  	}
  	if (yadim[1] != n) stop("First dimension of YA must match XA")
  	difx <- xa - x0
  	absxmxa <- abs(difx)
  	ns <- min((1:n)[absxmxa == min(absxmxa)])
  	cs <- ya
  	ds <- ya
  	if (nydim == 1) y <- ya[ns]
  	if (nydim == 2) y <- ya[ns,]
  	if (nydim == 3) y <- ya[ns,,]
  	if (nydim == 4) y <- ya[ns,,,]
  	ns <- ns - 1
  	for (m in 1:(n-1)) {
    	if (nydim == 1) {
      		for (i in 1:(n-m)) {
        		ho    <- difx[i]
        		hp    <- difx[i+m]
        		w     <- (cs[i+1] - ds[i])/(ho - hp)
        		ds[i] <- hp*w
        		cs[i] <- ho*w
      		}
      		if (2*ns < n-m) {
        		dy <- cs[ns+1]
      		} else {
        		dy <- ds[ns]
        		ns <- ns - 1
      		}
  		}
  		if (nydim == 2) {
      		for (i in 1:(n-m)) {
        		ho     <- difx[i]
        		hp     <- difx[i+m]
        		w      <- (cs[i+1,] - ds[i,])/(ho - hp)
        		ds[i,] <- hp*w
        		cs[i,] <- ho*w
      		}
      		if (2*ns < n-m) {
        		dy <- cs[ns+1,]
      		} else {
        		dy <- ds[ns,]
        		ns <- ns - 1
      		}
  		}
   		if (nydim == 3) {
      		for (i in 1:(n-m)) {
        		ho       <- difx[i]
        		hp       <- difx[i+m]
        		w        <- (cs[i+1,,] - ds[i,,])/(ho - hp)
        		ds[i,,] <- hp*w
        		cs[i,,] <- ho*w
      		}
      		if (2*ns < n-m) {
        		dy <- cs[ns+1,,]
      		} else {
        		dy <- ds[ns,,]
        		ns <- ns - 1
      		}
  		}
   		if (nydim == 4) {
      		for (i in 1:(n-m)) {
        		ho      <- difx[i]
        		hp      <- difx[i+m]
        		w       <- (cs[i+1,,,] - ds[i,,,])/(ho - hp)
        		ds[i,,,] <- hp*w
        		cs[i,,,] <- ho*w
      		}
      		if (2*ns < n-m) {
        		dy <- cs[ns+1,,,]

      		} else {
        		dy <- ds[ns,,,]
        		ns <- ns - 1
      		}
  		}
   		y <- y + dy
	}
   	return( list(y, dy) )
}
is.basis <- function(basisobj) {
#  check whether BASISOBJ is a functional data basis object

#  Last modified 20 November 2005

  if (inherits(basisobj, "basisfd")) return(TRUE) else return(FALSE)
}
is.diag <- function (c, EPS=1e-12) {
  #  tests a matrix for being diagonal
  #  C   ... matrix to be tested
  #  EPS ... testing criterion: max off-diagonal element over min diagonal
  #          element must be less than EPS
  if (!is.matrix(c)) return(FALSE)
  cd <- dim(c)
  if (cd[1] != cd[2]) return(FALSE)
  mindg <- min(abs(diag(c)))
  maxodg <- max(abs(c - diag(diag(c))))
  if (maxodg/mindg < EPS) return(TRUE) else return(FALSE)
}
is.eqbasis <- function(basisobj1, basisobj2) {
	
	#  tests to see of two basis objects are identical
	
	eqwrd <- TRUE
	
	#  test type
	
	if (basisobj1$type != basisobj2$type) {
		eqwrd <- FALSE
		return(eqwrd)
	}
	
	#  test range
	
	if (any(basisobj1$rangeval != basisobj2$rangeval)) {
		eqwrd <- FALSE
		return(eqwrd)
	}
	
	#  test nbasis
	
	if (basisobj1$nbasis != basisobj2$nbasis) {
		eqwrd <- FALSE
		return(eqwrd)
	}
	
	#  test params
	
	if (any(basisobj1$params != basisobj2$params)) {
		eqwrd <- FALSE
		return(eqwrd)
	}

   #  test dropind
	
	if (any(basisobj1$dropind != basisobj2$dropind)) {
		eqwrd <- FALSE
		return(eqwrd)
	}
	
	return(eqwrd)
	
}

	
	
is.fdPar <- function(fdParobj) {
#  check whether FDPAROBJ is a functional data object

#  Last modified 20 November 2005

  if (inherits(fdParobj, "fdPar")) return(TRUE) else return(FALSE)
}
is.fd <- function(fdobj) {
#  check whether FDOBJ is a functional data object

#  Last modified 20 November 2005

  if (inherits(fdobj, "fd")) return(TRUE) else return(FALSE)
}
is.fdSmooth <- function(fdSmoothobj) {
#  check whether FDPAROBJ is a functional data object

#  Last modified 20 November 2005

  if (inherits(fdSmoothobj, "fdSmooth")) return(TRUE) else return(FALSE)
}

is.integerLfd <- function(Lfdobj)
{
  #  check whether Lfd object is a simple differential operator

  #  Last modified 9 February 2007

  nderiv  <- Lfdobj$nderiv
  bintwrd <- TRUE
  if (nderiv > 0) {
    bwtlist <- Lfdobj$bwtlist
      if (!is.null(bwtlist)) {
    	  nderiv <- Lfdobj$nderiv
    	  for (j in 1:nderiv) {
          bfdj <- bwtlist[[j]]
          if (any(bfdj$coefs != 0.0)) bintwrd <- FALSE
    	  }
	}
    }
  bintwrd
}
is.Lfd <- function(Lfdobj) {
#  check whether LFDOBJ is a linear differential operator
  if (inherits(Lfdobj, "Lfd")) return(TRUE) else return(FALSE)
}
isotone <- function(y) {
	#  Compute an isotonic regression line from data in Y
	#  This is the piecewise linear line that is monotonic and 
	#  most closely approximates Y in the least squares sense.
  n    <- length(y)
  mony <- y
  eb   <- 0
  indx <- 1:(n-1)
  while (eb < n) {
    negind <- (diff(mony) < 0)
    ib <- min(indx[diff(mony) < 0])
    if (is.na(ib)) {
      bb <- eb <- n
    } else {
      bb <- eb <- ib
    } 
    while (eb < n && mony[bb] == mony[eb+1]) eb <- eb + 1
    poolflg <- -1
    while (poolflg != 0) {
      if (eb >=  n || mony[eb] <= mony[eb+1]) poolflg <- 1
      if (poolflg == -1) {
        br <- er <- eb+1
        while (er < n && mony[er+1] == mony[br]) er <- er + 1
        pmn <- (mony[bb]*(eb-bb+1) + mony[br]*(er-br+1))/(er-bb+1)
        eb <- er
        mony[bb:eb] <- pmn
        poolflg <- 1
      }
      if (poolflg == 1) {
        if (bb <= 1 || mony[bb-1] <= mony[bb]) {
          poolflg <- 0
        } else {
          bl <- el <- bb-1
          while (bl > 1 && mony[bl-1] == mony[el]) bl <- bl - 1
          pmn <- (mony[bb]*(eb-bb+1) + mony[bl]*(el-bl+1))/(eb-bl+1)
          bb <- bl
          mony[bb:eb] <- pmn
          poolflg <- -1
        }
      }
    }
  }
  return(mony)
}
knots.basisfd <- function(Fn, interior = TRUE, ...) {
##
## 1.  object$type = 'bspline'?
##
  type <- Fn$type
  oName <- substring(deparse(substitute(Fn)), 1, 33)
  if(is.null(type))
    stop('is.null((', oName, ')$type);  must be "bspline"') 
  if(type != 'bspline')
    stop('(', oName, ')$type) = ', type[1], ';  must be "bspline"')
##
## 2.  knots
##
  int <- Fn$params
  if(interior) return(int)
#
  nord <- norder(Fn)
  rng <- Fn$rangeval 
  allKnots <- c(rep(rng[1], nord), int, rep(rng[2], nord))
  return(allKnots) 
}

knots.fd <- function(Fn, interior=TRUE, ...){
  knots(Fn$basis, interior=interior, ...)
}
knots.fdSmooth <- function(Fn, interior=TRUE, ...){
  knots(Fn$fd, interior=interior, ...)
} 
lambda2df <- function (argvals, basisobj, wtvec=rep(1,n), Lfdobj=NULL, lambda=0)
{
  #  Computes the the degrees of freedom associated with a regularized
  #    basis smooth by calculating the trace of the smoothing matrix.

  #  Arguments for this function:
  #
  #  ARGVALS  ... A set of argument values.
  #  BASISOBJ ... A basis.fd object created by function create.basis.fd.
  #  WTVEC    ... A vector of N weights, set to one by default, that can
  #               be used to differentially weight observations in the
  #               smoothing phase
  #  LFDOBJ   ... The order of derivative or a linear differential
  #               operator to be penalized in the smoothing phase.
  #               By default Lfdobj is set in function GETBASISPENALTY
  #  LAMBDA   ... The smoothing parameter determining the weight to be
  #               placed on the size of the derivative in smoothing.  This
  #               is 0 by default.
  #  Returns:
  #  DF    ...  a degrees of freedom measure

  #  Last modified:  6 January 2020

  n        <- length(argvals)
  nbasis   <- basisobj$nbasis
  if (lambda == 0) {
    df <- nbasis
    return( df )
  }
  if (length(wtvec) != n) stop("WTVEC of wrong length")
  if (min(wtvec) <= 0)    stop("All values of WTVEC must be positive.")
  basismat <- getbasismatrix(argvals, basisobj, 0)
  basisw   <- basismat*outer(wtvec,rep(1,nbasis))
  Bmat     <- crossprod(basisw,basismat)
  penmat   <- getbasispenalty(basisobj, Lfdobj)
  Bnorm    <- sqrt(sum(Bmat^2))
  pennorm  <- sqrt(sum(penmat^2))
  condno   <- pennorm/Bnorm
  if (lambda*condno > 1e12) {
    lambda <- 1e12/condno
    warning(paste("lambda reduced to",lambda,"to prevent overflow"))
  }
  Cmat     <- Bmat + lambda*penmat
  Cmat     <- (Cmat + t(Cmat))/2
  if (is.diag(Cmat)) {
      Cmatinv <- diag(1/diag(Cmat))
  } else {
      Lmat    <- chol(Cmat)
      Lmatinv <- solve(Lmat)
      Cmatinv <- crossprod(t(Lmatinv))
  }
  hatmat <- Cmatinv %*% Bmat
  df <- sum(diag(hatmat))
  return( df )
}
lambda2gcv <- function(log10lambda, argvals, y, fdParobj, 
                          wtvec=rep(1,length(argvals)))
{
#  LAMBDA2GCV smooths data using smooth_basis using 
#  LAMBDA = 10^LOG10LAMBDA, and returns the GCV value that result. 

#  Return:
#  GCV ... a vector or matrix of GCV values, depending on whether Y is a
#          matrix of array, respectively. 

#  Last modified 21 October 2008 by Jim Ramsay

  fdParobj$lambda <- 10^log10lambda
  gcv <- smooth.basis(argvals, y, fdParobj, wtvec)$gcv

  return(gcv)
}

landmarkreg <- function(unregfd, ximarks, x0marks, x0lim=NULL, 
                        WfdPar=NULL, WfdPar0=NULL, ylambda=1e-10) {
  #  This version of landmarkreg does not assume that the target marks
  #  x0marks are within the same interval as that for the curves to be
  #  registered.  Consequently, it needs a required extra argument X0LIM 
  #  defining the target interval and optional fdPar argument for 
  #  representing the inverse warping function.
  
  #  Arguments:
  #  UNREGFD ... functional data object for curves to be registered
  #  XIMARKS ... N by NL array of times of interior landmarks for
  #                 each observed curve
  #  XOMARKS ... vector of length NL of times of interior landmarks for
  #                 target curve
  #  X0LIM   ... vector of length 2 containing the lower and upper boundary
  #              of the target interval containing x0marks.
  #  WFDPAR  ... a functional parameter object defining a warping function
  #  MONWRD  ... If TRUE, warping functions are estimated by monotone smoothing,
  #                 otherwise by regular smoothing.  The latter is faster, but
  #                 not guaranteed to produce a strictly monotone warping
  #                 function.  If MONWRD is 0 and an error message results
  #                 indicating nonmonotonicity, rerun with MONWRD = 1.
  #                 Default:  TRUE
  #  YLAMBDA ... smoothing parameter to be used in computing the registered
  #                 functions.  For high dimensional bases, local wiggles may be
  #                 found in the registered functions or its derivatives that are
  #                 not seen in the unregistered functions.  In this event, this
  #                 parameter should be increased.
  #  Returns:
  #  FDREG   ... a functional data object for the registered curves
  #  WARPFD  ... a functional data object for the warping functions
  #  WFD     ... a functional data object for the W functions defining the
  #              warping functions
  
  # Warning:  As of March 2022, landmark registration cannot be done using
  # function smooth.basis instead of function smooth.morph.  The 
  # warping function must be strictly monotonic, and we have found that using 
  # smooth.basis too often violates this contraint.  Function 
  # smooth.morph ensures monotonicity.
  
  #  Last modified 2 June 2022  by Jim Ramsay
  
  #  check unregfd containing the curves to be registered
  
  if (!(inherits(unregfd,  "fd"))) stop(
    "Argument unregfd  not a functional data object.")
  
  Ybasis   <- unregfd$basis
  nbasis   <- Ybasis$nbasis
  rangeval <- Ybasis$rangeval
  
  if (is.null(x0lim)) x0lim = rangeval
  
  #   ---------------------------------------------------------------------
  #                  check ximarks and x0marks
  #   ---------------------------------------------------------------------
  
  #  check ximarks being matrix with ncurve rows and nmarks columns
  
  if (is.numeric(ximarks)) {
    nximarks <- length(ximarks)
    # if ximarks is a vector, coerce it to a single row matrix
    if (is.vector(ximarks))     ximarks <- matrix(ximarks,1,nximarks)
    # if ximarks is a data.frame, coerce it to a matrix
    if (is.data.frame(ximarks)) ximarks <- as.matrix(ximarks)
  } else {
    stop("Argument ximarks is not numeric.")
  }
  
  #  check x0marks and coerce it to be a one-row matrix
  
  if (is.numeric(x0marks)) {
    nx0marks <- length(x0marks)
    if (is.vector(x0marks)) x0marks <- matrix(x0marks,1,nx0marks)
  } else {
    stop("Argument x0marks is not numeric.")
  }
  
  #  check that ximarks and x0marks have same number of columns
  
  if (ncol(ximarks) != length(x0marks)) 
    stop("The number of columns in ximarks is not equal to length of x0marks.")
  
  # check that ximarks are within range of unregfd
  
  if (any(ximarks <= rangeval[1]) || any(ximarks >= rangeval[2]))
    stop("Argument ximarks has values outside of range of unregfd.")
  
  # check that x0marks are within range of target interval
  
  if (any(x0marks <= x0lim[1]) || any(x0marks >= x0lim[2]))
    stop("Argument x0marks has values outside of range of target interval.")
  
  #  determine the number of curves to be registered
  
  ncurve   <- dim(ximarks)[1]
  
  #   ---------------------------------------------------------------------
  #                        check WFDPAR
  #   ---------------------------------------------------------------------
  
  #  set up default WfdPar for warping function
  
  if (is.null(WfdPar)) {
    Wnbasis   <- length(x0marks) + 2
    Wbasis    <- create.bspline.basis(rangeval, Wnbasis)
    Wfd       <- fd(matrix(0,Wnbasis,1), Wbasis)
    WfdPar    <- fdPar(Wfd, 2, 1e-10)
  } else {
    WfdPar  <- fdParcheck(WfdPar,  1)
    Wfd     <- WfdPar$fd
    Wbasis  <- Wfd$basis
    Wnbasis <- Wbasis$nbasis
  }

  #  set up default WfdPar0 for inverse warping function
  
  if (is.null(WfdPar0)) {
    Wnbasis0  <- length(x0marks) + 2
    Wbasis0   <- create.bspline.basis(x0lim, Wnbasis0)
    Wfd0      <- fd(matrix(0,Wnbasis0,1), Wbasis0)
    WfdPar0   <- fdPar(Wfd0, 2, 1e-10)
  } else {
    WfdPar0  <- fdParcheck(WfdPar0, 1)
    Wfd0     <- WfdPar0$fd
    Wbasis0  <- Wfd0$basis
    Wnbasis0 <- Wbasis0$nbasis
  }
  
  #   ---------------------------------------------------------------------
  #                        set up analysis
  #   ---------------------------------------------------------------------
  
  nfine   <- min(c(101,10*nbasis))
  xfine   <- seq(rangeval[1], rangeval[2], length=nfine)
  xfine0  <- seq(   x0lim[1],    x0lim[2], length=nfine)
  yfine   <- eval.fd(xfine, unregfd)
  yregmat <- yfine
  hfunmat <- matrix(0,nfine,ncurve)
  hinvmat <- matrix(0,nfine,ncurve)
  
  xval    <- matrix(c(x0lim[1],x0marks,x0lim[2]),nx0marks+2,1)
  Wcoef   <- matrix(0,Wnbasis,ncurve)
  nval    <- length(xval)
  
  #  --------------------------------------------------------------------
  #                  Iterate through curves to register
  #  --------------------------------------------------------------------
  
  if (ncurve > 1) cat("Progress:  Each dot is a curve\n")
  
  for (icurve in 1:ncurve) {
    if (ncurve > 1) cat(".")
    #  set up landmark times for this curve
    yval   <- matrix(c(rangeval[1],ximarks[icurve,],rangeval[2]),nx0marks+2,1)
    #  smooth relation between this curve"s values and target"s values
    #  use monotone smoother
    
    Wfd  <- smooth.morph(xval, yval, rangeval, WfdPar)$Wfdobj
    
    hfun <- monfn(xfine, Wfd)
    b    <- (rangeval[2]-rangeval[1])/(hfun[nfine]-hfun[1])
    a    <- rangeval[1] - b*hfun[1]
    hfun <- a + b*hfun
    hfun[c(1,nfine)] <- rangeval
    Wcoefi           <- Wfd$coef
    Wcoef[,icurve]   <- Wcoefi
    hfunmat[,icurve] <- hfun
    
    #  compute h-inverse  in order to register curves
    
    Wcoefi       <- Wfd$coefs
    Wfdinv       <- smooth.morph(hfun, xfine, x0lim, WfdPar0)$Wfdobj
    hinv         <- monfn(xfine, Wfdinv)
    b            <- (x0lim[2]-x0lim[1])/(hinv[nfine]-hinv[1])
    a            <- x0lim[1] - b*hinv[1]
    hinv         <- a + b*hinv
    hinv[c(1,nfine)] <- rangeval
    hinvmat[,icurve] <- hinv
    
    #  compute registered curves
    
    yregfd <- smooth.basis(hinv, yfine[,icurve], Ybasis)$fd
    yregmat[,icurve] <- eval.fd(xfine, yregfd, 0)
  }
  
  if (ncurve > 1) cat("\n")
  
  #  create functional data objects for the registered curves
  
  regfdPar <- fdPar(Ybasis, 2, ylambda)
  regfd    <- smooth.basis(xfine, yregmat, regfdPar)$fd
  regnames <- unregfd$fdnames
  names(regnames)[3] <- paste("Registered",names(regnames)[3])
  regfd$fdnames <- regnames
  
  #  create functional data objects for the warping functions
  
  warpfd                <- smooth.basis(xfine, hfunmat, Ybasis)$fd
  warpfdnames           <- unregfd$fdnames
  names(warpfdnames)[3] <- paste("Warped",names(regnames)[1])
  warpfd$fdnames        <- warpfdnames

  #  create functional data objects for the inverse warping functions
  
  Ybasis0               <- create.bspline.basis(x0lim, nbasis)
  warpinvfd             <- smooth.basis(xfine0, hinvmat, Ybasis0)$fd
  warpfdnames           <- unregfd$fdnames
  names(warpfdnames)[3] <- paste("Warped",names(regnames)[1])
  warpinvfd$fdnames     <- warpfdnames
  
  #  The core function defining the strictly monotone warping
  
  Wfd <- fd(Wcoef, Wbasis)
  
  return( list(regfd=regfd, warpfd=warpfd, warpinvfd=warpinvfd, Wfd=Wfd) )
}
#  setClass statement for "Lfd" class

# setClass("Lfd",
# 	representation(call="call", nderiv="integer", bwtlist="list"))

#  Generator function for class Lfd

Lfd = function(nderiv=0, bwtlist=vector("list",0))
{

#  LFD creates a linear differential operator object of the form
#
#  Lx(t) = b_0(t) x(t) +  + b_{m-1}(t) D^{m-1}x(t) + D^m x(t)
#  or
#  Lx(t) = b_0(t) x(t) +  + b_{m-1}(t) D^{m-1}x(t) + D^m x(t)
#          \exp[b_m(t) D^m x(t).
#
#  Function x(t) is operated on by this operator L, and the operator
#  computes a linear combination of the function and its first m
#  derivatives.   The function x(t) must be scalar.
#  The operator L is called the HOMOGENEOUS because it does
#  not involve input or forcing functions.
#
#  The linear combination of derivatives is defined by the weight
#  or coefficient functions b_j(t), and these are assumed to vary
#  over t, although of course they may also be constant as a
#  special case.  Each of these must be scalar function.
#
#  The weight coefficient for D^m is special in that it must
#  be positive to properly identify the operator.  This is why
#  it is exponentiated.  In most situations, it will be 0,
#  implying a weight of one, and this is the default.
#
#  Some important functions also have the capability of allowing
#  the argument that is an LFD object be an integer. They convert
#  the integer internally to an LFD object by INT2LFD().  These are:
#     EVAL.FD()
#     EVAL.MON()
#     EVAL.POS()
#     EVAL.BASIS()
#     EVAL.PENALTY()
#
#  Arguments:
#
#  NDERIV  the order of the operator, that is,
#          the highest order of derivative.  This corresponds
#          to m in the above equation.
#  BWTLIST  A list object of length either NDERIV or NDERIV + 1.
#          Each list contains an FD object, an FDPAR object or
#          a numerical scalar constant.
#          If there are NDERIV functions, then the coefficient of D^m
#          is set to 1 otherwise, function NDERIV+1 contains a function
#          that is exponentiated to define the actual coefficient.
#          bwtlist may also be a vector of length NDERIV or NDERIVE + 1
#          containing constants.  If this is the case, the constants
#          are used as coefficients for a constant basis function.
#          The default is a row vector of NDERIV zeros.
#
#  Returns:
#
#  LFDOBJ  a functional data object

# last modified 2007 May 3 by Spencer Graves
#  Previously modified 9 October 2005

#  check nderiv

if (!is.numeric(nderiv))
    stop("Order of operator is not numeric.")
if (nderiv != round(nderiv))
    stop("Order of operator is not an integer.")
if (nderiv < 0)
    stop("Order of operator is negative.")

#  check that bwtlist is either a list or a fd object

if (!inherits(bwtlist, "list") && !inherits(bwtlist, "fd") &&
    !is.null(bwtlist) && !missing(bwtlist))
	stop("BWTLIST is neither a LIST or a FD object")

#  if bwtlist is missing or NULL, convert it to a constant basis FD object

if (is.null(bwtlist)) {
   bwtlist <- vector("list", nderiv)
	if (nderiv > 0) {
        conbasis <- create.constant.basis()
	    for (j in 1:nderiv) bwtlist[[j]]  <- fd(0, conbasis)
    }
}

#  if BWTLIST is a fd object, convert to a list object.

if (inherits(bwtlist, "fd")) bwtlist <- fd2list(bwtlist)

#  check size of bwtlist

nbwt <- length(bwtlist)

if (nbwt != nderiv & nbwt != nderiv + 1)
    stop("The size of bwtlist inconsistent with NDERIV.")

#  check individual list entries for class
#  and find a default range

if (nderiv > 0) {
    rangevec <- c(0,1)
    for (j in 1:nbwt) {
        bfdj <- bwtlist[[j]]
        if (inherits(bfdj, "fdPar")) {
			  bfdj <- bfdj$fd
			  bwtlist[[j]] <- bfdj
		 }
        if (!inherits(bfdj, "fd") && !inherits(bfdj, "integer"))
            stop(paste("An element of BWTLIST contains something other ",
               " than an fd object or an integer"))
        if (inherits(bfdj, "fd")) {
	        bbasis   <- bfdj$basis
	        rangevec <- bbasis$rangeval
        } else {
            if (length(bfdj) == 1) {
                bwtfd <- fd(bfdj, conbasis)
                bwtlist[[j]] <- bwtfd
            }
            else stop("An element of BWTLIST contains a more than one integer.")
        }
    }

    #  check that the ranges are compatible

    for (j in 1:nbwt) {
        bfdj    <- bwtlist[[j]]
        if (inherits(bfdj, "fdPar")) bfdj <- bfdj$fd
        bbasis <- bfdj$basis
        btype  <- bbasis$type
        #  constant basis can have any range
        if (!btype == "const") {
            brange = bbasis$rangeval
            if (any(rangevec != brange)) stop(
                "Ranges are not compatible.")
        }
    }
}

#  Save call

Lfd.call <- match.call()

#  set up the Lfd object

#  S4 definition
# Lfdobj <- new("Lfd", call=Lfd.call, nderiv=nderiv, bwtlist=bwtlist)

#  S3 definition

Lfdobj <- list(call=Lfd.call, nderiv=nderiv, bwtlist=bwtlist)
oldClass(Lfdobj) <- "Lfd"

Lfdobj

}

#  "print" method for "Lfd"

print.Lfd <- function(x, ...)
{
  object <- x
	nderiv  <- object$nderiv
	bwtlist <- object$bwtlist

	cat("Lfd:\n")

	cat(paste("nderiv =",nderiv,"\n"))

	if (nderiv > 0) {
		cat("\nbwtlist:\n")
		for (ideriv in 1:nderiv) {
			cat(paste("\nWeight function:",ideriv-1,"\n\n"))
			print(object$bwtlist[[ideriv]])
		}
	}
}

#  "summary" method for "Lfd"

summary.Lfd <- function(object, ...)
{
	print(object)
}

#  "plot" method for "Lfd"

plot.Lfd <- function(x, axes=NULL, ...)
{
	nderiv <- x$nderiv
	oldPar <- par(mfrow=c(nderiv,1))
	on.exit(par(oldPar))
	for (ideriv in 1:nderiv) {
		plot(x$bwtlist[[ideriv]], axes=axes, ...)
	}
	invisible(NULL)
}




lines.fdSmooth <- function(x, Lfdobj=int2Lfd(0), nx=201, ...){
  lines(x$fd, Lfdobj=Lfdobj, nx=nx, ...)
}

lines.fd <- function(x, Lfdobj=int2Lfd(0), nx=201, ...)
{
  #  Plot a functional data object FD using lines in a pre-existing plot.
  #  If there are multiple variables, each curve will appear in the same plot.
  #  The remaining optional arguments are the same as those available
  #     in the regular "lines" function.
  
  # Last modified 16 January 2020
  
  fdobj <- x
  
  if (!(is.fd(fdobj) || is.fdPar(fdobj)))  stop(
		"First argument is neither a functional data or a functional parameter object.")
  if (is.fdPar(fdobj)) fdobj <- fdobj$fd
  
  if (!inherits(Lfdobj, "Lfd")) stop(
      "Second argument is not a linear differential operator.")

  coef   <- fdobj$coefs
  coefd  <- dim(coef)
  ndim   <- length(coefd)
  nbasis <- coefd[1]
  nrep   <- coefd[2]
  if (ndim > 3) nvar <- coefd[3] else nvar <- 1
  crvnames <- fdobj$fdnames[[2]]
  varnames <- fdobj$fdnames[[3]]

  basisobj <- fdobj$basis
#
  xlim <- par('usr')[1:2]
  if(par('xlog')) xlim <- 10^xlim
# 
  rngx <- basisobj$rangeval 
  xmin <- max(rngx[1], xlim[1])
  xmax <- min(rngx[2], xlim[2])
  x.        <- seq(xmin, xmax, length=nx)
  fdmat    <- eval.fd(x.,fdobj,Lfdobj)

  if (length(dim(coef)) < 2) {
    lines (x.,fdmat,...)
  }
  if (length(dim(coef)) ==2 ) {
    matlines (x.,fdmat,...)
  }
  if (length(dim(coef)) == 3) {
    for (ivar in 1:nvar) {
      matlines (x.,fdmat[,,ivar],type="l",lty=1,
                main=varnames[ivar],...)
    }
  }
  invisible()
}
linmod = function(xfdobj, yfdobj, betaList, wtvec=NULL)  {
#  LINMOD  Fits an unrestricted or full functional linear model of the form
#       y(t) = \alpha(t) + \int x(s) \beta(s,t) ds + \epsilon(t),
#  where
#       \beta(s,t) = \phi'(s) B \psi(t)
#  
#  Arguments:
#  XFD       a functional data object for the independent variable 
#  YFD       a functional data object for the   dependent variable 
#  BETACELL  a List array of length 2 containing a functional
#               parameter object for \alpha and a bifdPar object for
#               \beta as a function of s and t, respectively.
#  WTVEC    a vector of weights
#  Returns:  a list object linmodList with fields
#  BETA0ESTFD    a functional parameter object for \alpha
#  BETA1ESTBIFD  a bivariate functional parameter object for \beta
#  YHATFDOBJ     a functional data object for the approximation to y

#  Last modified 11 June 2010

#  check xfdobj and yfdobj

if (!is.fd(xfdobj)) {
    stop("XFD is not a functional data object.")
}

if (!is.fd(yfdobj)) {
    stop("YFD is not a functional data object.")
}

ybasis  = yfdobj$basis
ynbasis = ybasis$nbasis
ranget  = ybasis$rangeval

xbasis  = xfdobj$basis
ranges  = xbasis$rangeval

nfine = max(c(201,10*ynbasis+1))
tfine = seq(ranget[1],ranget[2],len=nfine)

#  get dimensions of data

coefy   = yfdobj$coef
coefx   = xfdobj$coef
coefdx  = dim(coefx)
coefdy  = dim(coefy)
ncurves = coefdx[2]
if (coefdy[2] != ncurves) {
    stop ("Numbers of observations in first two arguments do not match.")
}

#  set up or check weight vector

if (!is.null(wtvec)) wtvec = wtcheck(ncurves, wtvec)

#  get basis parameter objects

if (!inherits(betaList, "list")) stop("betaList is not a list object.")

if (length(betaList) != 2) stop("betaList not of length 2.")

alphafdPar  = betaList[[1]]
betabifdPar = betaList[[2]]

if (!inherits(alphafdPar, "fdPar")) {
    stop("BETACELL[[1]] is not a fdPar object.")
}
if (!inherits(betabifdPar, "bifdPar")) {
    stop("BETACELL[[2]] is not a bifdPar object.")
}

#  get Lfd objects

alphaLfd = alphafdPar$Lfd
betasLfd = betabifdPar$Lfds
betatLfd = betabifdPar$Lfdt

#  get smoothing parameters

alphalambda = alphafdPar$lambda
betaslambda = betabifdPar$lambdas
betatlambda = betabifdPar$lambdat

#  get basis objects

alphafd    = alphafdPar$fd

alphabasis = alphafd$basis
alpharange = alphabasis$rangeval
if (alpharange[1] != ranget[1] || alpharange[2] != ranget[2]) {
    stop("Range of ALPHAFD coefficient and YFD not compatible.")
}

betabifd = betabifdPar$bifd

betasbasis = betabifd$sbasis
betasrange = betasbasis$rangeval
if (betasrange[1] != ranges[1] || betasrange[2] != ranges[2]) {
    stop("Range of BETASFD coefficient and XFD not compatible.")
}

betatbasis = betabifd$tbasis
betatrange = betatbasis$rangeval
if (betatrange[1] != ranget[1] || betatrange[2] != ranget[2]) {
    stop("Range of BETATFD coefficient and YFD not compatible.")
}

#  get numbers of basis functions

alphanbasis = alphabasis$nbasis
betasnbasis = betasbasis$nbasis
betatnbasis = betatbasis$nbasis

#  get inner products of basis functions and data functions

Finprod = inprod(ybasis, alphabasis)
Ginprod = inprod(ybasis, betatbasis)
Hinprod = inprod(xbasis, betasbasis)

ycoef = yfdobj$coef
xcoef = xfdobj$coef
Fmat = t(ycoef) %*% Finprod
Gmat = t(ycoef) %*% Ginprod
Hmat = t(xcoef) %*% Hinprod

if (is.null(wtvec)) {
    HHCP = t(Hmat) %*% Hmat
    HGCP = t(Hmat) %*% Gmat
    H1CP = as.matrix(apply(Hmat,2,sum))
    F1CP = as.matrix(apply(Fmat,2,sum))
} else {
    HHCP = t(Hmat) %*% (outer(wtvec,rep(betasnbasis))*Hmat)
    HGCP = t(Hmat) %*% (outer(wtvec,rep(betatnbasis))*Gmat)
    H1CP = t(Hmat) %*% wtvec
    F1CP = t(Fmat) %*% wtvec
}

#  get inner products of basis functions

alphattmat = inprod(alphabasis, alphabasis)
betalttmat = inprod(betatbasis, alphabasis)
betassmat  = inprod(betasbasis, betasbasis)
betattmat  = inprod(betatbasis, betatbasis)

#  get penalty matrices

if (alphalambda > 0) {
    alphapenmat = eval.penalty(alphabasis, alphaLfd)
} else {
    alphapenmat = NULL
}
if (betaslambda > 0) {
    betaspenmat = eval.penalty(betasbasis, betasLfd)
} else {
    betaspenmat = NULL
}
if (betatlambda > 0) {
    betatpenmat = eval.penalty(betatbasis, betatLfd)
} else {
    betatpenmat = NULL
}

#  set up coefficient matrix and right side for stationary equations

betan = betasnbasis*betatnbasis
ncoef = alphanbasis + betan
Cmat  = matrix(0,ncoef,ncoef)
Dmat  = matrix(0,ncoef,1)

#  rows for alpha

ind1 = 1:alphanbasis
ind2 = ind1
Cmat[ind1,ind2] = ncurves*alphattmat
if (alphalambda > 0) {
    Cmat[ind1,ind2] = Cmat[ind1,ind2] + alphalambda*alphapenmat
}
ind2 = alphanbasis + (1:betan)
Cmat[ind1,ind2] = t(kronecker(H1CP,betalttmat))

Dmat[ind1] = F1CP

#  rows for beta

ind1 = alphanbasis + (1:betan)
ind2 = 1:alphanbasis
Cmat[ind1,ind2] = t(Cmat[ind2,ind1])
ind2 = ind1
Cmat[ind1,ind2] = kronecker(HHCP,betattmat)
if (betaslambda > 0) {
    Cmat[ind1,ind2] = Cmat[ind1,ind2] + 
                      betaslambda*kronecker(betaspenmat,betattmat)
}
if (betatlambda > 0) {
    Cmat[ind1,ind2] = Cmat[ind1,ind2] + 
                      betatlambda*kronecker(betassmat,betatpenmat)
}

Dmat[ind1] = matrix(t(HGCP),betan,1)

#  solve the equations

coefvec = symsolve(Cmat, Dmat)

#  set up the coefficient function estimates

#  functional structure for the alpha function

ind1 = 1:alphanbasis
alphacoef = coefvec[ind1]

alphafdnames = yfdobj$fdnames
alphafdnames[[3]] = "Intercept"
alphafd = fd(alphacoef, alphabasis, alphafdnames)

#  bi-functional structure for the beta function

ind1 = alphanbasis + (1:betan)
betacoef    = matrix(coefvec[ind1],betatnbasis,betasnbasis)
betafdnames = xfdobj$fdnames
betafdnames[[3]] = "Reg. Coefficient"
betafd = bifd(t(betacoef), betasbasis, betatbasis, betafdnames)

#  functional data structure for the yhat functions

xbetacoef = betacoef %*% t(Hmat)
xbetafd   = fd(xbetacoef, betatbasis)
yhatmat   = eval.fd(tfine, alphafd) %*% matrix(1,1,ncurves) + 
            eval.fd(tfine, xbetafd)
yhatfd    = smooth.basis(tfine, yhatmat, ybasis)$fd 

linmodList = list(beta0estfd=alphafd, beta1estbifd=betafd, yhatfdobj=yhatfd)

return(linmodList)

}


lnsrch <- function (xold, fold, g, p, func, dataList, stpmax, 
                    itermax=20, TOLX=1e-10, dbglev=0) {
    n     <- length(xold)
    check <- FALSE
    f2    <- 0
    alam2 <- 0
    ALF   <- 1e-4
    psum  <- sqrt(sum(p^2))
    #  scale if attempted step is too big
    if (psum > stpmax) {
        p <- p*(stpmax/psum)
    }
    slope <- sum(g*p)
    if (dbglev > 1) {
      cat("\n")
      cat("      ")
      cat(0)
      cat("      ")
      cat(round(slope,5))
      cat("      ")
      cat(round(fold,5))
    } 
    if (slope >= 0) {
        stop('Initial slope not negative.')
    }
    # compute lambdamin
    test <- 0
    for (i in 1:n) {
        temp <- abs(p[i])/max(abs(xold[i]),1)
        if (temp > test) {
            test <- temp
        }
    }
    alamin <- TOLX/test
    #  always try full Newton step first
    alam   <- 1
    #  start of iteration loop
    iter <- 0
    while (iter <= itermax) {
        iter <- iter + 1
        x <- xold + alam*p
        #  -------------  function evaluation  -----------
        result <- func(x, dataList)
        f <- result$PENSSE
        g <- result$DPENSSE
        if (dbglev > 1) {
          cat("\n")
          cat("      ")
          cat(iter)
          cat("      ")
          cat(round(slope,5))
              cat("      ")
              cat(round(fold,5))
        }
        #  -----------------------------------------------
        #  convergence on x change.
        if (alam < alamin) {
            x     <- xold
            check <- TRUE
            return(list(x=x, check=check))
        } else {
            #  sufficient function decrease
            if (f <= fold + ALF*alam*slope) {
                return(list(x=x, check=check))
            }
            #  backtrack
            if (alam == 1) {
                #  first time
                tmplam <- -slope/(2*(f-fold-slope))
            } else {
                #  subsequent backtracks
                rhs1 <- f  - fold - alam *slope
                rhs2 <- f2 - fold - alam2*slope
                a <- (rhs1/alam^2 - rhs2/alam2^2)/(alam-alam2)
                b <- (-alam2*rhs1/alam^2 + alam*rhs2/(alam*alam2))/(alam-alam2)
                if (a == 0) {
                    tmplam <- -slope/(2*b)
                } else {
                    disc <- b^2 - 3*a*slope
                    if (disc < 0) {
                        tmplam <- 0.5*alam
                    } else {
                        if (b <= 0) {
                            tmplam <- (-b+sqrt(disc))/(3*a)
                        } else {
                            tmplam <- -slope/(b+sqrt(disc))
                        }
                    }
                    if (tmplam > 0.5*alam) {
                        tmplam <- 0.5*alam
                    }
                }
            }
            alam2 <- alam
            f2    <- f
            #  lambda > 0.1 lambda1
            alam <- max(tmplam, 0.1*alam)
        }
        #  try again
    }
    
    return(list(x=x, check=check))
    
}

matplot <- function(x, ...) {
    UseMethod('matplot')
}

#  --------------------------------------------------------------------

matplot.matrix <- function(x, ...){
    fda::matplot.default(x, ...)
}

#  --------------------------------------------------------------------

matplot.numeric <- function(x, ...){
    fda::matplot.default(x, ...)
}

#  --------------------------------------------------------------------

matplot.default <- function(x, y, type = "p", lty = 1:5, lwd = 1,
    lend = par("lend"), pch = NULL, col = 1:6, cex = NULL, bg = NA,
    xlab = NULL, ylab = NULL, xlim = NULL, ylim = NULL, ..., add = FALSE,
    verbose = getOption("verbose")){
#
  if(is.null(xlab))
      xlab <- deparse(substitute(x))
  if(is.null(ylab))
      ylab <- deparse(substitute(y))
#
  graphics::matplot(x, y, type = type, lty = lty, lwd = lwd,
    lend = lend, pch = pch, col = col, cex = cex, bg = bg,
    xlab = xlab, ylab = ylab, xlim = xlim, ylim = ylim, ..., add = add,
    verbose = getOption("verbose"))
#
}

#  --------------------------------------------------------------------

matplot.Date <- function(x, y, type = "p", lty = 1:5, lwd = 1,
    lend = par("lend"), pch = NULL, col = 1:6, cex = NULL, bg = NA,
    xlab = NULL, ylab = NULL, xlim = NULL, ylim = NULL, ..., add = FALSE,
    verbose = getOption("verbose")){
#
  if(is.null(xlab))
      xlab <- deparse(substitute(x))
  if(is.null(ylab))
      ylab <- deparse(substitute(y))
#
  matplot.POSIXct(x, y, type = type, lty = lty, lwd = lwd,
    lend = lend, pch = pch, col = col, cex = cex, bg = bg,
    xlab = xlab, ylab = ylab, xlim = xlim, ylim = ylim, ..., add = add,
    verbose = getOption("verbose"))
}

#  --------------------------------------------------------------------

matplot.POSIXct <- function(x, y, type = "p", lty = 1:5, lwd = 1,
    lend = par("lend"), pch = NULL, col = 1:6, cex = NULL, bg = NA,
    xlab = NULL, ylab = NULL, xlim = NULL, ylim = NULL, ..., add = FALSE,
    verbose = getOption("verbose")){
#
  if(is.null(xlab))
      xlab <- deparse(substitute(x))
  if(is.null(ylab))
      ylab <- deparse(substitute(y))
  if(is.null(xlim))
      xlim <- range(x, na.rm=TRUE)
  if(is.null(ylim))
      ylim <- range(y, na.rm=TRUE)
#
  if(!add){
    plot(range(x), range(y), type='n', cex=cex, bg=bg, xlab=xlab, ylab=ylab,
         xlim=xlim, ylim=ylim, ...)
    out <- matlines(x, y, type=type, cex=cex, ..., verbose=verbose)
    return(out)
  }
  graphics::matplot(x, y, type = type, lty = lty, lwd = lwd,
    lend = lend, pch = pch, col = col, cex = cex, bg = bg,
    xlab = xlab, ylab = ylab, xlim = xlim, ylim = ylim, ..., add = add,
    verbose = getOption("verbose"))
}
#  --------------------------------------------------------------

monfn <- function(argvals, Wfdobj, basislist=vector("list",JMAX), 
                  returnMatrix=FALSE) {
#  evaluates a monotone function of the form
#            h(x) = [D^{-1} exp Wfdobj](x)
#  where  D^{-1} means taking the indefinite integral.
#  The interval over which the integration takes places is defined in
#  the basis object in Wfdobj.
#  Arguments:
#  ARGVALS   ... argument values at which function and derivatives are evaluated
#  WFDOBJ    ... a functional data object
#  BASISLIST ... a list containing values of basis functions
#  Returns:
#  HVAL   ... matrix or array containing values of h.
#  RETURNMATRIX ... If False, a matrix in sparse storage model can be returned
#               from a call to function BsplineS.  See this function for
#               enabling this option.

#  Last modified 8 May 2012

  #  check Wfdobj
  
  if (!inherits(Wfdobj, "fd")) stop("Wfdobj is not a fd object.")


  JMAX <- 15
  JMIN <- 11
  EPS  <- 1E-5

  coef  <- Wfdobj$coefs
  coefd <- dim(coef)
  ndim  <- length(coefd)
  if (ndim > 1 && coefd[2] != 1) stop("WFDOBJ is not a single function")

  basisobj <- Wfdobj$basis
  rangeval <- basisobj$rangeval

  #  set up first iteration

  width <- rangeval[2] - rangeval[1]
  JMAXP <- JMAX + 1
  h <- rep(1,JMAXP)
  h[2] <- 0.25
  #  matrix SMAT contains the history of discrete approximations to the
  #    integral
  smat <- matrix(0,JMAXP)
  #  array TVAL contains the argument values used in the approximation
  #  array FVAL contains the integral values at these argument values,
  #     rows corresponding to argument values
  #  the first iteration uses just the endpoints
  tval <- rangeval
  j   <- 1
  if (is.null(basislist[[j]])) {
      bmat <- getbasismatrix(tval, basisobj, 0, returnMatrix)
      basislist[[j]] <- bmat
  } else {
      bmat <- basislist[[j]]
  }
  fx   <- as.matrix(exp(bmat %*% coef))
  fval <- fx
  smat[1,]  <- width*apply(fx,2,sum)/2
  tnm <- 0.5
  for (j in 2:JMAX) {
    tnm  <- tnm*2
    del  <- width/tnm
    flag <- ifelse(rangeval[1]+del/2 >= rangeval[2]-del/2, -1, 1)
    tj   <- seq(rangeval[1]+del/2, rangeval[2]-del/2, by=flag*abs(del))
    tval <- c(tval, tj)
    if (is.null(basislist[[j]])) {
        bmat <- getbasismatrix(tj, basisobj, 0, returnMatrix)
        basislist[[j]] <- bmat
    } else {
        bmat <- basislist[[j]]
    }
    fx   <- as.matrix(exp(bmat %*% coef))
    fval <- c(fval,fx)
    smat[j] <- (smat[j-1] + width*apply(fx,2,sum)/tnm)/2
    if (j >= JMIN) {
      ind <- (j-4):j
      result <- polintmat(h[ind],smat[ind],0)
      ss  <- result[[1]]
      dss <- result[[2]]
      if (all(abs(dss) < EPS*max(abs(ss)))) {
        # successful convergence
        # sort argument values and corresponding function values
        ordind <- order(tval)
        tval   <- tval[ordind]
        fval   <- fval[ordind]
        nx     <- length(tval)
        del    <- tval[2] - tval[1]
        fval   <- del*(cumsum(fval) - 0.5*(fval[1] + fval))
        hval   <- approx(tval, fval, argvals)$y
        return(hval)
      }
    }
    smat[j+1] <- smat[j]
    h[j+1]    <- 0.25*h[j]
  }
  stop(paste("No convergence after",JMAX," steps in MONFN"))
}
#  --------------------------------------------------------------------------

mongrad <- function(x, Wfdobj, basislist=vector("list",JMAX), 
                    returnMatrix=FALSE) {
#  Evaluates the gradient with respect to the coefficients in Wfdobj
#     of a monotone function of the form
#            h(x) = [D^{-1} exp Wfdobj](x)
#  where  D^{-1} means taking the indefinite integral.
#  The interval over which the integration takes places is defined in
#  the basisfd object in Wfdobj.
#  Arguments:
#  X      ... argument values at which function and derivatives are evaluated
#  WFDOBJ ... a functional data object
#  BASISLIST ... a list containing values of basis functions
#  Returns:
#  GVAL   ... value of gradient at input values in X.
#  RETURNMATRIX ... If False, a matrix in sparse storage model can be returned
#               from a call to function BsplineS.  See this function for
#               enabling this option.

#  Last modified 9 May 2012 by Jim Ramsay

  JMAX <- 15
  JMIN <- 11
  EPS  <- 1E-5

  coef  <- Wfdobj$coefs
  coefd <- dim(coef)
  ndim  <- length(coefd)
  if (ndim > 1 && coefd[2] != 1) stop("Wfdobj is not a single function")

  basisfd  <- Wfdobj$basis
  rangeval <- basisfd$rangeval
  nbasis   <- basisfd$nbasis
  onebas   <- rep(1,nbasis)
  width    <- rangeval[2] - rangeval[1]

  #  set up first iteration

  JMAXP <- JMAX + 1
  h <- rep(1,JMAXP)
  h[2] <- 0.25
  #  matrix SMAT contains the history of discrete approximations to the
  #    integral
  smat <- matrix(0,JMAXP,nbasis)
  #  array TVAL contains the argument values used in the approximation
  #  array FVAL contains the integral values at these argument values,
  #     rows corresponding to argument values
  #  the first iteration uses just the endpoints
  j   <- 1
  tval <- rangeval
  if (is.null(basislist[[j]])) {
      bmat <- getbasismatrix(tval, basisfd, 0, returnMatrix)
      basislist[[j]] <- bmat
  } else {
      bmat <- basislist[[j]]
  }
  fx   <- as.matrix(exp(bmat %*% coef))
  fval <- as.matrix(outer(c(fx),onebas)*bmat)
  smat[1,]  <- width*apply(fval,2,sum)/2
  tnm <- 0.5

  #  now iterate to convergence
  for (iter in 2:JMAX) {
    tnm  <- tnm*2
    del  <- width/tnm  
    flag <- ifelse(rangeval[1]+del/2 >= rangeval[2]-del/2, -1, 1)
    tj   <- seq(rangeval[1]+del/2, rangeval[2]-del/2, by=flag*abs(del))
    tval <- c(tval, tj)
    if (is.null(basislist[[iter]])) {
        bmat <- getbasismatrix(tj, basisfd, 0, returnMatrix)
        basislist[[iter]] <- bmat
    } else {
        bmat <- basislist[[iter]]
    }
    fx   <- as.matrix(exp(bmat %*% coef))
    gval <- as.matrix(outer(c(fx),onebas)*bmat)
    fval <- rbind(fval,gval)
    smat[iter,] <- (smat[iter-1,] + width*apply(fval,2,sum)/tnm)/2
    if (iter >= max(c(5,JMIN))) {
      ind <- (iter-4):iter
      result <- polintmat(h[ind],smat[ind,],0)
      ss  <- result[[1]]
      dss <- result[[2]]
      if (all(abs(dss) < EPS*max(abs(ss))) || iter >= JMAX) {
        # successful convergence
        # sort argument values and corresponding function values
        ordind <- order(tval)
        tval   <- tval[ordind]
        fval   <- as.matrix(fval[ordind,])
        # set up partial integral values
        lval   <- outer(rep(1,length(tval)),fval[1,])
        del    <- tval[2] - tval[1]
        fval   <- del*(apply(fval,2,cumsum) - 0.5*(lval + fval))
        gval   <- matrix(0,length(x),nbasis)
        for (i in 1:nbasis) gval[,i] <- approx(tval, fval[,i], x)$y
        return(gval)
      }
    }
    smat[iter+1,] <- smat[iter,]
    h[iter+1]     <- 0.25*h[iter]
  }
}
monhess <- function(x, Wfd, basislist)
{
#  MONHESS evaluates the second derivative of monotone fn. wrt coefficients
#  The function is of the form h[x] <- (D^{-1} exp Wfd)(x)
#  where  D^{-1} means taking the indefinite integral.
#  The interval over which the integration takes places is defined in
#       the basis object <- WFD.
#  The derivatives with respect to the coefficients in WFD up to order
#       NDERIV are also computed, max(NDERIV) <- 2.
#  Arguments:
#  X       argument values at which function and derivatives are evaluated
#             x[1] must be at lower limit, and x(n) at upper limit.
#  WFD     a functional data object
#  Returns:
#  D2H   values of D2 h wrt c
#  TVAL  Arguments used for trapezoidal approximation to integral

#  Last modified 6 January 2020 by Jim Ramsay

#  set some constants

EPS    <- 1e-4
JMIN   <-  7
JMAX   <- 15

#  get coefficient matrix and check it

coef  <- Wfd$coefs
coefd <- dim(coef)
ndim  <- length(coefd)
if (ndim > 1 & coefd[2] != 1) stop("WFD is not a single function")

#  get the basis

basis    <- Wfd$basis
rangeval <- basis$rangeval
nbasis   <- basis$nbasis
nbaspr   <- nbasis*(nbasis+1)/2
onebaspr <- matrix(1,1,nbaspr)

#  set up first iteration

width <- rangeval[2] - rangeval[1]
JMAXP <- JMAX + 1
h     <- matrix(1,JMAXP,1)
h[2]  <- 0.25
#  matrix SMAT contains the history of discrete approximations to the
#    integral
smatD2h <- matrix(0,JMAXP,nbaspr)
#  array TVAL contains the argument values used <- the approximation
#  array FVAL contains the integral values at these argument values,
#     rows corresponding to argument values
#  the first iteration uses just the endpoints
iter  <- 1
xiter <- rangeval
tval  <- xiter
if (is.null(basislist[[iter]])) {
    bmat <- getbasismatrix(tval, basis, 0)
    basislist[[iter]] <- bmat
} else {
    bmat <- basislist[[iter]]
}
fx   <- as.matrix(exp(bmat %*% coef))
D2fx <- matrix(0,2,nbaspr)
m <- 0
for (ib in 1:nbasis) {
   for (jb in 1:ib) {
      m <- m + 1
      D2fxij   <- as.matrix(fx*bmat[,ib]*bmat[,jb])
      D2fx[,m] <- D2fxij
   }
}
D2fval <- D2fx
smatD2h[1,] <- width*sum(D2fx)/2
tnm <- 0.5

#  now iterate to convergence

for (iter in 2:JMAX) {
   tnm   <- tnm*2
   del   <- width/tnm
   hdel  <- del/2
   xiter <- seq(rangeval[1]+del/2, rangeval[2]-del/2, del)
   tval  <- c(tval, xiter)
   if (is.null(basislist[[iter]])) {
      bmat <- getbasismatrix(xiter, basis, 0)
      basislist[[iter]] <- bmat
   } else {
      bmat <- basislist[[iter]]
   }
   fx   <- as.matrix(exp(bmat%*%coef))
   D2fx <- matrix(0,length(xiter),nbaspr)
   m <- 0
   for (ib in 1:nbasis) {
      for (jb in 1:ib) {
         m <- m + 1
         D2fxij   <- as.matrix(fx*bmat[,ib]*bmat[,jb])
         D2fx[,m] <- D2fxij
      }
   }
   D2fval <- rbind(D2fval, D2fx)
   smatD2h[iter,] <- (smatD2h[iter-1,] + del*sum(D2fx))/2
   if (iter >= max(c(JMIN,5))) {
      ind <- (iter-4):iter
      result <- polintmat(h[ind],smatD2h[ind,],0)
      D2ss   <- result[[1]]
      D2dss  <- result[[2]]
      if (all(abs(D2dss) < EPS*max(abs(D2ss))) || iter >= JMAX) {
         # successful convergence
         # sort argument values and corresponding function values
         ordind  <- order(tval)
         tval    <- tval[ordind]
         D2fval  <- as.matrix(D2fval[ordind,])
         # set up partial integral values
         lval    <- outer(rep(1,length(tval)),D2fval[1,])
         del     <- tval[2] - tval[1]
         D2ifval <- del*(apply(D2fval,2,cumsum) - 0.5*(lval + D2fval))
         D2h     <- matrix(0,length(x),nbaspr)
         for (i in 1:nbaspr) D2h[,i] <- approx(tval, D2ifval[,i], x)$y
         return(D2h)
      }
    }
    h[iter+1] <- 0.25*h[iter]
  }
}

monomialpen <- function(basisobj, Lfdobj=int2Lfd(2),
                        rng=basisobj$rangeval)
{
  #  MONOMIALPEN  Computes a monomial basis penalty matrix.
  #
  #  Warning:  This version is incomplete in the sense that
  #    it only works with LFDOBJ = D^m
  #
  #  Arguments:
  #  BASISOBJ ... a monomial basis object
  #  LFDOBJ   ... either the order of derivative or a
  #               linear differential operator to be penalized.
  #  RNG      ... a range over which the penalty matrix is computed.
  #  Returns a list the first element of which is the basis matrix
  #   and the second element of which is the diagonal of the penalty matrix.
  
  #  Last modified:  26 October 2005
  
  #  check BASISOBJ
  
  if (!inherits(basisobj, "basisfd")) stop(
    "First argument is not a basis.fd object.")
  
  type <- basisobj$type
  if (!(type == "monom")) stop("BASISOBJ not of type monom")
  
  #  check LFDOBJ
  
  Lfdobj <- int2Lfd(Lfdobj)
  
  #  check whether LFDOBJ is of form D^m
  
  nderiv <- Lfdobj$nderiv
  
  #  get basis information
  
  nbasis    <- basisobj$nbasis
  exponents <- basisobj$params
  
  #  check exponents
  
  for (ibasis in 1:nbasis) {
    ideg <- exponents[ibasis]
    if (ideg-floor(ideg) != 0) stop(
      "An exponent is not an integer.")
  }
  
  #  compute the penalty matrix
  
  penaltymat <- matrix(0,nbasis,nbasis)
  for (ibasis in 1:nbasis) {
    ideg <- exponents[ibasis]
    ifac <- 1
    if (nderiv > 0) {
      ifac <- ideg
      if (nderiv > 1) for (k in 2:nderiv) ifac <- ifac*(ideg - k + 1)
    }
    for (jbasis in 1:ibasis) {
      jdeg <- exponents[jbasis]
      jfac <- 1
      if (nderiv > 0) {
        jfac <- jdeg
        if (nderiv > 1) for (k in 2:nderiv) jfac <- jfac*(jdeg - k + 1)
      }
      if ((ideg >= nderiv) & (jdeg >= nderiv)) {
        ipow <- ideg+jdeg-2*nderiv+1
        penaltymat[ibasis,jbasis] <-
          (rng[2]^ipow - rng[1]^ipow)*ifac*jfac/ipow
        penaltymat[jbasis,ibasis] <- penaltymat[ibasis,jbasis]
      }
    }
  }
  penaltymat
}
monomial <- function(evalarg, exponents=1, nderiv=0, argtrans=c(0,1))
{
#  MONOMIAL Values of monomials, or their derivatives.
#  The powers of EVALARG are the NBASIS nonnegative integers in EXPONENTS.
#  The default is 1, meaning EVALARG itself.
#  Arguments are as follows:
#  EVALARG   ... array of values at which the polynomials are to
#                evaluated
#  EXPONENTS ... array of nonnegative integer exponents of EVALARG
#  NDERIV    ... order of derivative to be returned.
  #  ARGTRANS  ... A vector of two constants for shifting and scaling the 
  #                argument range.  In function MONOMIAL and MONOM, 
  #                the EVALARG is transformed to 
  #                [evalarg-argtrans(1))/evalarg(2);
  #                Defaults to [0,1]
  #  Return is:
#  A matrix with length(EVALARG) rows and NBASIS columns containing
#    the values of the monomials or their derivatives

#  last modified 9 January 2020 by Jim Ramsay

	evalarg <- as.vector(evalarg)
	evalarg <- (evalarg - argtrans[1])/argtrans[2]
  n       <- length(evalarg)    

	nbasis <- length(exponents)

	#  check whether exponents are nonnegative integers

	for (ibasis in 1:nbasis) {
    	if (exponents[ibasis] - round(exponents[ibasis]) != 0) {
        	stop("An exponent is not an integer.")
    	}
    	if (exponents[ibasis] < 0) {
        	stop("An exponent is negative.")
    	}
	}

	# check if there are duplicate exponents

  if((length(exponents)>1) && (min(diff(sort(exponents))) == 0))
          stop("There are duplicate exponents.")

	monommat <- matrix(0,n,nbasis)

	if (nderiv == 0) {
    	#  use the recursion formula to compute monomnomial values
    	for (ibasis in 1:nbasis) {
			monommat[,ibasis] <- evalarg^exponents[ibasis]
		}
	} else {
    	for (ibasis in 1:nbasis) {
    	  print(ibasis)
        	degree <- exponents[ibasis]
        	if (nderiv <= degree) {
            	fac <- degree
            	if (nderiv >= 2) {
            	    for (ideriv in 2:nderiv) {
                	    fac <- fac*(degree-ideriv)
           	      }
           	  }
            	print(fac)
            	print(degree-nderiv)
            	monommat[,ibasis] <- fac*evalarg^(degree-nderiv)
        	}
    	}
	}

	return(monommat)
}

norder         <- function(x, ...) UseMethod("norder")

norder.fd      <- function(x, ...) norder.bspline(x$basis)

norder.basisfd <- function(x, ...) norder.bspline(x)

norder.default <- function(x, ...){
#
  xB. <- sapply(x, function(x){
    inherits(x, 'basisfd') || inherits(x, 'fd')
  } )
  xB <- which(xB.)
#
  {
    if(length(xB)<1)
      stop("input is not a 'basisfd' object and does not have ",
           "a 'basisfd' component.")
    else
      return(norder(x[[xB[1]]]))
  }
}

norder.bspline <- function(x, ...){
  if(!('type' %in% names(x))){
    xName <- substring(deparse(substitute(x)), 1, 33)
    stop('object ', xName, " does NOT have a 'type' component, ",
         "and therefore can NOT be a functional data object")
  }
  if(x$type != 'bspline'){
    xName <- substring(deparse(substitute(x)), 1, 33)
    stop('object ', xName, " is of type = ", x$type,
         ";  'norder' is only defined for type = 'bsline'")
  }
  with(x, nbasis - length(params))
}
objAndNames <- function(object, preferred, default)
{
  dimo <- dim(object)
  ndo <- length(dimo)
##
## 1.  NO 'dim' attribute
##
  if(ndo<1){
    n.o <- length(object)
    {
      if(is.list(preferred)){
        if((length(preferred)>0) && 
           (length(preferred[[1]])==n.o)){
          names(object) <- preferred[[1]]
          return(object)
        }
      }
      else if(length(preferred)==n.o){
        names(object) <- preferred
        return(object)
      }
    }
    {
      if(is.list(default)){
        if((length(default)>0) && 
           (length(default[[1]])==n.o) ){
          names(object) <- default[[1]]
          return(object)
        }
      }
      else if(length(default)==n.o){
        names(default) <- default
        return(object)
      }
    }
    return(object)
  }
##
## 2.  'dim' atribute
##
  dimn.o <- vector("list", ndo)
  for(id in 1:ndo){
    {
      if(is.list(preferred)){
        if((length(preferred)>=id) && 
           (length(preferred[[id]])==dimo[id])) {
          dimn.o[[id]] <- preferred[[id]]
          next
        }
      }
      else if((id==1) && (length(preferred)==dimo[id])){
        dimn.o[[id]] <- preferred
        next 
      }
    }
    {
      if(is.list(default)){
        if((length(default)>=id) && 
           (length(default[[id]])==dimo[id]) ){
          dimn.o[[id]] <- default[[id]]
          next 
        }
      }
      else if((id==1) && (length(default)==dimo[id]) ){
        dimn.o[[id]] <- default
        next 
      }
    }
  }
  dimnames(object) <- dimn.o
  object 
}


odesolv <- function(bwtlist, ystart=diag(rep(1,norder)),
                    h0=width/100, hmin=width*1e-10, hmax=width*0.5,
                    EPS=1e-4, MAXSTP=1000)
{
#  Solve L u = 0,
#  L being an order M homogeneous linear differential operator,
#     (Lu)(t) = w_1(t) u(t) + w_2(t) Du(t) + ...
#                  w_m(t) D^{m-1} u(t) + D^m u(t) = 0
#  for function u and its derivatives up to order m - 1.
#  Each such solution is determined by the values of u and its
#  m - 1 derivatives at time t = 0.  These initial conditions are
#  contained in the columns of the matrix YSTART, which has exactly
#  m rows.  The number of solutions computed by ODESOLV is equal to
#  the number of columnsof YSTART.  In order for the solutions to be
#  linearly independent functions, the columns of YSTART must be
#  linearly independent.  This means that the maximum number of
#  linearly independent solutions is m.
#  The solution for each value of t is a matrix, y(t).
#  Any column of y(t) contains u, Du, ... , D^{m-1} u at argument
#  value t, for the corresponding set of starting values for these
#  first m derivatives. It is the job of this function to estimate
#  these values, and ODESOLV will choose a set of values of TNOW at
#  which these can be estimated with satisfactory accuracy.
#  ODESOLV uses the Runge-Kutta method, which is a good general
#  purpose numerical method.  But it does not work well for stiff
#  systems, and it can fail for poor choices of initial conditions
#  as well as other problems.

#  Arguments:
#  BWTLIST ... a list containing m = 1 weightfunctions.
#             The weight functions w_1, ... , w_m are the functions with
#             indices 2, 3, ..., m+1.
#  YSTART ... initial values for Y.  This is a matrix with M rows,
#             were M is the order of the operator L.  Any column of M
#             specifies intial values for derivatives 0, 1, ... M-1.  Each
#             column must specify a unique set of initial conditions.  A
#             frequent choice is the identity matrix of order M, and this is
#             the default.
#  H0     ... initial trial step size
#  HMIN   ... minimum step size
#  HMAX   ... maximum step size
#  EPS    ... error tolerance
#  MAXSTP ... maximum number of Runge Kutta steps permitted. If the equation
#             is difficult to solve, this may have to be increased.

#  Returns:
#  TP     ... vector of TRUE values used
#  YP     ... m by m by length(TP) array of Y-values generated for
#             values in TP.

#  Note that ODESOLV calls function DERIVS in order to evaluate the
#  differential operator.  Also, it works by redefining the order m
#  linear differential equation as a linear system of m first order
#  differential equations.  DERIVS evaluates the right side of this
#  linear system.

#  Last modified 30 October 2005

  	MAXWARN <- 10

#  determine the order of the system m
  	
   norder <- length(bwtlist)

#  determine the range of values over which the equation is solved

   bfdPar1  <- bwtlist[[1]]
   wbasis   <- bfdPar1$fd$basis
  	rangeval <- wbasis$rangeval
  	tbeg     <- rangeval[1]
  	tend     <- rangeval[2]
  	width    <- tend - tbeg
  	tnow     <- tbeg

  	h        <- min(c(h0, hmax))
  	tp       <- tnow

#  set up the starting values

  	ystartd  <- dim(ystart)
  	if (ystartd[1] != norder) stop("YSTART has incorrect dimensions")
  	n        <- ystartd[2]
  	yp       <- c(ystart)
  	index    <- abs(yp) > 1e-10
  	y        <- ystart

#  Since ODESOLVE is slow, progress is displayed

  	cat("Progress:  each dot is 10% of interval\n")

#  initialize the solution

  	tdel  <- (tend - tbeg)/10
  	tdot  <- tdel
  	iwarn <- 0

#  solve the equation using a maximum of MAXSTP steps

  	for (nstp in 1:MAXSTP) {
		#  evaluate the right side at the current value
    	dydt  <- derivs(tnow, y, bwtlist)
    	yscal <- c(abs(y) + abs(h*dydt) + 1e-30)[index]
    	if (nstp > 1) {
      		tp <- c(tp,tnow)
      		yp <- c(yp,c(y))
    	}
    	if (tnow >= tdot) {
      		cat(".")
      		tdot <- tdot + tdel
    	}
    	if ((tnow+h-tend)*(tnow+h-tbeg) > 0) h <- tend-tnow
		#  take a Runge-Kutta step to the next value
    	result <-  rkqs(y, dydt, tnow, h, bwtlist, yscal, index, EPS)
    	tnow   <- result[[1]]
    	y      <- result[[2]]
    	h      <- result[[3]]
    	hnext  <- result[[4]]
		#  test to see if interval has been covered, and return
		#  if it has.
    	if ((tnow-tend)*(tend-tbeg) >= 0) {
      		cat(".")
      		tp <- c(tp,tnow)
      		yp <- c(yp,c(y))
      		yp <- array(yp,c(norder,n,length(tp)))
      		return(list(tp, yp))
    	}
		#  test if the step is too small.
    	if (abs(hnext) < hmin) {
			warning("Stepsize smaller than minimum")
			hnext <- hmin
			iwarn <- iwarn + 1
			if (iwarn >= MAXWARN) stop("Too many warnings.")
		}
		#  test if the step is too large.
    	h <- min(c(hnext, hmax))
  	}
  	warning("Too many steps.")
}

#  ---------------------------------------------------------------

rkqs <- function(y, dydt, tnow, htry, bwtlist, yscal, index, EPS)
{
#  Take a single step using the Runge-Kutta procedure to
#  control for accuracy.  The function returns the new
#  value of t, the value of the solution at t, the step
#  size, and a proposal for the next step size.
	h <- htry
	#  check the accuracy of the step
   	result <- rkck(y, dydt, tnow, h, bwtlist)
   	ytemp  <- result[[1]]
   	yerr   <- c(result[[2]])[index]
   	errmax <- max(abs(yerr/yscal))/EPS
	#  modify the step size if ERRMAX is too large
   	while (errmax > 1) {
        	htemp <- 0.9*h*(errmax^(-0.25))
        	h     <- max(c(abs(htemp),0.1*abs(h)))
        	tnew  <- tnow + h
        	if (tnew == tnow) stop("stepsize underflow in rkqs")
        	result <- rkck(y, dydt, tnow, h, bwtlist)
        	ytemp  <- result[[1]]
        	yerr   <- result[[2]]
        	errmax <- max(abs(yerr/yscal))/EPS
 	}
	# set up the proposed next step
   	if (errmax > 1.89e-4) {
        	hnext <- 0.9*h*(errmax^(-0.2))
   	} else {
        	hnext <- 5.*h
  	}
   	tnow <- tnow + h
   	y <- ytemp
  	return( list(tnow, y, h, hnext) )
}

#  ------------------------------------------------------------------

rkck <- function(y, dydt, tnow, h, bwtlist)
{
#  Take a single Runge-Kutta step.
#  Return the solution, and an estimate of its error.
      C1  <-   37/378
      C3  <-  250/621
      C4  <-  125/594
      C6  <-  512/1771
      DC5 <- -277/14336
      DC1 <-  C1 - 2825/27648
      DC3 <-  C3 - 18575/48384
      DC4 <-  C4 - 13525/55296
      DC6 <-  C6 - 0.25

      ytemp <- y + h*0.2*dydt
      ak2   <- derivs(tnow+0.2*h,   ytemp, bwtlist)
      ytemp <- y + h*(0.075*dydt  + 0.225*ak2)
      ak3   <- derivs(tnow+0.3*h,   ytemp, bwtlist)
      ytemp <- y + h*(0.3*dydt    - 0.9*ak2 + 1.2*ak3)
      ak4   <- derivs(tnow+0.6*h,   ytemp, bwtlist)
      ytemp <- y + h*(-11*dydt/54 + 2.5*ak2 + (-70*ak3+35*ak4)/27)
      ak5   <- derivs(tnow+h,       ytemp, bwtlist)
      ytemp <- y + h*(1631*dydt/55296  + 575*ak2/512 + 575*ak3/13824 +
                      44275*ak4/110592 + 253*ak5/4096)
      ak6   <- derivs(tnow+0.875*h, ytemp, bwtlist)
      yout  <- y + h*(C1*dydt + C3*ak3 + C4*ak4 + C6*ak6)
      yerr  <- h*(DC1*dydt    + DC3*ak3 + DC4*ak4 + DC5*ak5 + DC6*ak6)
      return (list( yout, yerr ) )
}
pca.fd <- function(fdobj, nharm = 2, harmfdPar=fdPar(fdobj),
                   centerfns = TRUE)
{
  #  Carry out a functional PCA with regularization
  #  Arguments:
  #  FDOBJ      ... Functional data object
  #  NHARM     ... Number of principal components or harmonics to be kept
  #  HARMFDPAR ... Functional parameter object for the harmonics
  #  CENTERFNS ... If TRUE, the mean function is first subtracted from each 
  #                function.
  #
  #  Returns:  An object PCAFD of class "pca.fd" with these named entries:
  #  harmonics  ... A functional data object for the harmonics or eigenfunctions
  #  values     ... The complete set of eigenvalues
  #  scores     ... A matrix or array of scores on the principal components or 
  #                 harmonics
  #  varprop    ... A vector giving the proportion of variance explained
  #                 by each eigenfunction
  #  meanfd     ... A functional data object giving the mean function
  #
  
  #  Last modified:  13 March 2020 
  
  #  Check FDOBJ
  
  if (!(is.fd(fdobj) || is.fdPar(fdobj)))  stop(
    "First argument is neither a functional data or a functional parameter object.")
  if (is.fdPar(fdobj)) fdobj <- fdobj$fd
  
  #  compute mean function and center if required
  
  meanfd <- mean.fd(fdobj)
  if (centerfns) {
    fdobj <- center.fd(fdobj)
  }
  
  #  get coefficient matrix and its dimensions
  
  coef  <- fdobj$coefs
  coefd <- dim(coef)
  ndim  <- length(coefd)
  nrep  <- coefd[2]
  coefnames <- dimnames(coef)
  if (nrep < 2) stop("PCA not possible without replications.")
  
  basisobj <- fdobj$basis
  nbasis   <- basisobj$nbasis
  type     <- basisobj$type
  
  #  set up HARMBASIS
  
  harmbasis <- harmfdPar$fd$basis
  nhbasis   <- harmbasis$nbasis
  
  #  set up LFDOBJ and LAMBDA
  
  Lfdobj <- harmfdPar$Lfd
  lambda <- harmfdPar$lambda
  
  #  compute CTEMP whose cross product is needed
  
  if (ndim == 3) {
    nvar <- coefd[3]
    ctemp <- matrix(0, nvar * nbasis, nrep)
    for(j in 1:nvar) {
      index <- 1:nbasis + (j - 1) * nbasis
      ctemp[index,  ] <- coef[,  , j]
    }
  } else {
    nvar  <- 1
    ctemp <- coef
  }
  
  #  set up cross product Lmat for harmonic basis,
  #  roughness penalty matrix Rmat, and
  #  penalized cross product matrix Lmat
  
  Lmat <- eval.penalty(harmbasis, 0)
  if (lambda > 0) {
    Rmat <- eval.penalty(harmbasis, Lfdobj)
    Lmat <- Lmat + lambda * Rmat
  }
  Lmat <- (Lmat + t(Lmat))/2
  
  #  compute the Choleski factor Mmat of Lmat
  
  Mmat    <- chol(Lmat)
  Mmatinv <- solve(Mmat)
  
  #  set up cross product and penalty matrices
  
  Wmat <- crossprod(t(ctemp))/nrep
  
  Jmat = inprod(harmbasis, basisobj)
  MIJW = crossprod(Mmatinv,Jmat)
  
  #  set up matrix for eigenanalysis
  
  if(nvar == 1) {
    Cmat = MIJW %*% Wmat %*% t(MIJW)
  } else {
    Cmat = matrix(0,nvar*nhbasis,nvar*nhbasis)
    for (i in 1:nvar) {
      indexi <- 1:nbasis + (i - 1) * nbasis
      for (j in 1:nvar) {
        indexj <- 1:nbasis + (j - 1) * nbasis
        Cmat[indexi, indexj] <- MIJW %*% Wmat[indexi,indexj] %*% t(MIJW)
      }
    }
  }
  
  #  eigenalysis
  
  Cmat    <- (Cmat + t(Cmat))/2
  result  <- eigen(Cmat)
  eigvalc <- result$values
  eigvecc <- as.matrix(result$vectors[, 1:nharm])
  sumvecc <- apply(eigvecc, 2, sum)
  eigvecc[,sumvecc < 0] <-  - eigvecc[, sumvecc < 0]
  
  varprop <- eigvalc[1:nharm]/sum(eigvalc)
  
  #  set up harmfd
  
  if (nvar == 1) {
    harmcoef <- Mmatinv %*% eigvecc
  } else {
    harmcoef <- array(0, c(nbasis, nharm, nvar))
    for (j in 1:nvar) {
      index <- 1:nbasis + (j - 1) * nbasis
      temp <- eigvecc[index,  ]
      harmcoef[,  , j] <- Mmatinv %*% temp
    }
  }
  harmnames <- rep("", nharm)
  for(i in 1:nharm)
    harmnames[i] <- paste("PC", i, sep = "")
  if(length(coefd) == 2)
    harmnames <- list(coefnames[[1]], harmnames,"values")
  if(length(coefd) == 3)
    harmnames <- list(coefnames[[1]], harmnames, coefnames[[3]])
  harmfd   <- fd(harmcoef, harmbasis, harmnames)
  
  #  set up harmscr
  
  if (nvar == 1) {
    harmscr  <- inprod(fdobj, harmfd)
  } else {
    harmscr  <- array(0, c(nrep,   nharm, nvar))
    coefarray <- fdobj$coefs
    harmcoefarray <- harmfd$coefs
    for (j in 1:nvar) {
      fdobjj  <- fd(as.matrix(    coefarray[,,j]), basisobj)
      harmfdj <- fd(as.matrix(harmcoefarray[,,j]), basisobj)
      harmscr[,,j] <- inprod(fdobjj, harmfdj)
    }
  }
  
  #  set up the object pcafd of the pca.fd class containing the results
  
  pcafd        <- list(harmfd, eigvalc, harmscr, varprop, meanfd)
  class(pcafd) <- "pca.fd"
  names(pcafd) <- c("harmonics", "values", "scores", "varprop", "meanfd")
  
  return(pcafd)
}
pcaPACE <- function(covestimate, nharm = 2, harmfdPar=NULL, cross = TRUE)
{
  #  Carry out a functional PCA with regularization from the estimate of the covariance surface
  #  Arguments:
  #  COVESTIMATE  ... list of length 2 with named entries cov.estimate and meanfd
  #  NHARM     ... Number of principal components or harmonics to be kept
  #  HARMFDPAR ... Functional parameter object for the harmonics
  #  CROSS     ... If TRUE, ........................
  #
  #  Returns:  An object PCAFD of class "pca.PACE.fd" with these named entries:
  #  harmonics  ... A functional data object for the harmonics or eigenfunctions
  #  values     ... The complete set of eigenvalues
  #  scores     ... TO DO ===================================
  #  varprop    ... A vector giving the proportion of variance explained
  #                 by each eigenfunction
  #
  
  #  set up HARMBASIS
  
  harmbasis <- harmfdPar$fd$basis
  nhbasis   <- harmbasis$nbasis
  
  #  set up LFDOBJ and LAMBDA
  
  Lfdobj <- harmfdPar$Lfd
  lambda <- harmfdPar$lambda
  
  
  #  set up cross product Lmat for harmonic basis,
  #  roughness penalty matrix Rmat, and
  #  penalized cross product matrix Lmat
  
  Lmat <- eval.penalty(harmbasis, 0)
  if (lambda > 0) {
    Rmat <- eval.penalty(harmbasis, Lfdobj)
    Lmat <- Lmat + lambda * Rmat
  }
  Lmat <- (Lmat + t(Lmat))/2
  
  #  compute the Choleski factor Mmat of Lmat
  
  Mmat    <- chol(Lmat)
  Mmatinv <- solve(Mmat)
  
  #  set up cross product and penalty matrices
  
  if(inherits(covestimate$cov.estimate, "bifd")) {
    Wmat = covestimate$cov.estimate$coefs
    nvar = 1
    basisobj = covestimate$cov.estimate$sbasis
  }else{
    k=0
    l = length(covestimate$cov.estimate)
    while(l>0){
      k = k + 1
      l = l - k
    }
    nvar= k
    basisobj = covestimate$cov.estimate[[1]]$sbasis
    nbasis = basisobj$nbasis
    
    if(!cross){
      diag = cumsum(c(1,k - 0:(k-2)))
      Wmat = lapply(covestimate$cov.estimate[diag], function(x) x$coefs)
    }else{
      Wmat = matrix(0,nvar*nbasis,nvar*nbasis)
      r=0
      for(i in 1:nvar){
        indexi <- 1:nbasis + (i - 1) * nbasis
        for(j in 1:nvar){
          indexj <- 1:nbasis + (j - 1) * nbasis
          if(j>=i){
            r=r+1
            Wmat[indexi,indexj] = covestimate$cov.estimate[[r]]$coefs
            Wmat[indexj,indexi] = covestimate$cov.estimate[[r]]$coefs
          }
        }
      }
    }
    
    
    
  }
  
  Jmat = inprod(harmbasis, basisobj)
  MIJW = crossprod(Mmatinv,Jmat)
  
  #  set up matrix for eigenanalysis
  nbasis = basisobj$nbasis
  if(nvar == 1) {
    Cmat = MIJW %*% Wmat %*% t(MIJW)
  } else {
    if(!cross){
      Cmat = lapply(Wmat, function(x) MIJW %*% x %*% t(MIJW) )
      Cmat = lapply(Cmat, function(x) (x + t(x))/2)
    }else{
      Cmat = matrix(0,nvar*nhbasis,nvar*nhbasis)
      for(i in 1:nvar){
        indexi <- 1:nbasis + (i - 1) * nbasis
        for(j in 1:nvar){
          indexj <- 1:nbasis + (j - 1) * nbasis
          Cmat[indexi,indexj] = MIJW %*% Wmat[indexi,indexj] %*% t(MIJW)
          Cmat[indexj,indexi] = MIJW %*% Wmat[indexj,indexi] %*% t(MIJW)
        }
      }
    }
    
    
  }

  
  #  eigenalysis
  
  if (nvar == 1 | cross){
    Cmat    <- (Cmat + t(Cmat))/2
    result  <- eigen(Cmat)
    eigvalc <- result$values[1:nharm]
    if(nvar>1) eigvalc <- t(replicate(nvar, eigvalc))
    eigvecc <- as.matrix(result$vectors[, 1:nharm])
    sumvecc <- apply(eigvecc, 2, sum)
    eigvecc[,sumvecc < 0] <-  - eigvecc[, sumvecc < 0]
    varprop <- eigvalc[1:nharm]/sum(eigvalc) 
  } else {
    Cmat  = lapply(Cmat, function(x) (x + t(x))/2) 
    result = lapply(Cmat,eigen)
    eigval = lapply(result, function(x) x$values[1:nharm])
    eigvalc = do.call(rbind, eigval)
    eigvecc = lapply(result, function(x) as.matrix(x$vectors[, 1:nharm]))
    
    sumvecc = lapply(eigvecc, function(x) apply(x,2,sum))
    for(r in 1:length(eigvecc)){
      eigvecc[[r]][,sumvecc[[r]]<0] <- - eigvecc[[r]][,sumvecc[[r]]<0]
    }
    varprop <- eigvalc[1:nharm]/sum(eigvalc)
  }
  
  #  set up harmfd
  
  if (nvar == 1) {
    harmcoef <- Mmatinv %*% eigvecc
  } else {
    harmcoef <- array(0, c(nbasis, nharm, nvar))
    if(!cross){
      for (j in 1:nvar) {
        harmcoef[,  , j] <- Mmatinv %*% eigvecc[[j]]
      }
    }else{
      for (j in 1:nvar) {
        index <- 1:nbasis + (j - 1) * nbasis
        harmcoef[,  , j] <- Mmatinv %*% eigvecc[index,  ]
      }
    }
    
    
  }
  harmnames <- rep("", nharm)
  for(i in 1:nharm)
    harmnames[i] <- paste("PC", i, sep = "")
  harmfd   <- fd(harmcoef, harmbasis)
  
  #  set up the object pcafd of the pca.fd class containing the results
  
  scores = NULL
  pcafd        <- list(harmfd, eigvalc, scores,varprop, covestimate$meanfd)
  class(pcafd) <- "pca.fd"
  names(pcafd) <- c("harmonics", "values","scores" ,"varprop", "meanfd")
  
  return(pcafd)
}

pda.fd  <-  function(xfdlist, bwtlist=NULL, awtlist=NULL, ufdlist=NULL,
                     nfine=501)
{
  #PDA computes the basis function expansions of the
  #  estimates of the coefficient functions a_k(t) and b_j(t)
  #  in the possibly nonhomogeneous linear differential operator
  #
  #    Lx(t) =
  #       b_0(t)x(t) + b_1(t)Dx(t) + ... + b_{M-1}D^{M-1}x(t) + D^M x(t)
  #       - a_1(t)u_1(t) - ... - a_k(t)u_K(t)
  #
  #  of order M = DIFEORDER that minimizes in a least squares sense the residual
  #  functions f(t) = Lx(t).
  #
  #  The J equations may be of different orders and have different numbers
  #     of forcing functions.  In the rather complicated description of the
  #     arguments below, we use M_j to stand for the order of the jth equation.
  #
  #  If (DIFEORDER = 0, PDALIST fits the varying coefficient or pointwise
  #  linear model using the functions x(t) as dependent variables and
  #  the forcing functions u(t) as indep}ent variables.  In this case,
  #  there must be at least one forcing function.
  #
  #  The functions x(t) are in functional data object XFDOBJ.
  #  The forcing functions u_k(t) are in functional data object UFDOBJ.
  #  The coefficient functions for u_k(t) and x(t) are expanded in terms of the
  #  basis functions specified in AWTLIST and BWTLIST, respectively.
  #
  #
  #  Arguments:
  #  XFDLIST     list vector of length J of functional data objects for the
  #                 J functions whose derivatives define the DIFE.
  #  UFDLIST     list of length J whose components are themselves lists.
  #                  The jth component of this list contains a list vector of
  #                  length K_j of functional data objects for the
  #                 independent variables or u-variables forcing the equation
  #                  for the jth variable.
  #              In the special univariate case where there is only a single
  #                  DIFE, UFDLIST can have components which are the
  #                  functional data objects for the forcing functions, rather
  #                  than being a list with a single component list containing
  #                  these functions.
  #  BWTLIST     list vector of length J, the jth component of which is a list
  #                   vector of length J, each component which is itself a list
  #                   vector of length M_k equal to the order of the kth
  #                   differential equation in the containing list.
  #                   Each component of these lists within lists within a list
  #                   is a functional parameter object defining a weighting
  #                   coefficient function.
  #              that is, BWTLIST is three levels or layers of lists, the top
  #                   level a single list of length J, the second level a
  #                   set of J lists, each corresponding to an equation, and the
  #                   third level containing lists of length M_k containing
  #                   coefficient functions defining the contribution of the
  #                   m_kth derivative of variable k to the equation for
  #                   variable j.
  #              that is, if the orders of all the equations were the same
  #                   and R supported list arrays, their dimensions would be
  #                   J, J and M.
  #              However, in the special case of J = 1, where only M
  #                   coefficients are required, BWTLIST may be a simple
  #                   list vector of length M.
  #  AWTLIST     list of length J whose components are themselves lists.
  #                 The jth component of this list contains a list vector of
  #                  length K_j of functional parameter objects for the
  #                  coefficient functions a_{jk}(t) multipling the
  #                  corresponding forcing function in UFDLIST.
  #              In the special univariate case where there is only a single
  #                  DIFE, AWTLIST can have components which are the
  #                  functional parameter objects for the forcing functions,
  #                  rather than being a list with a single component list
  #                  containing these functional parameter objects.
  #  NFINE       number of sampling points for numerical integration, set by
  #                  default to 501, but adjusted as required to define a mesh
  #                  that is sufficient fine as to give a satisfactory
  #                  approximation to an integral.
  
  #  The value in each component of lists (within lists) XFDLIST and UFDLIST is a
  #      scalar FD object.
  #  The value in each component of lists (within lists) AWTLIST AND BWTLIST is a
  #      scalar FDPAR object.
  
  #  Returns:
  #  BWTLIST     list structure identical to that of the argument BWTLIST
  #  RESFDLIST   FD object for residual functions.
  #  AWTLIST     list structure identical to that of the argument AWTLIST
  
  #  Last modified 10 January 2020 
  
  #  check dimensions of the lists
  
  # check XFDLIST
  
  if (inherits(xfdlist, "fd")) xfdlist = list(xfdlist)
  
  if (!inherits(xfdlist, "list")) stop(
    "XFDLIST is neither a list or a FD object")
  
  nvar <- length(xfdlist)
  
  #  ----------------------------------------------------------------
  #     For efficiency, there are two versions of this code:
  #     one for a single variable, and another for multiple variables.
  #  ----------------------------------------------------------------
  
  if (nvar == 1) {
    
    #  ----------------------------------------------------------------
    #                   Single variable case
    #  ----------------------------------------------------------------
    
    difeorder <- length(bwtlist)
    difeordp1 <- difeorder + 1
    
    xfdobj <- xfdlist[[1]]
    xbasis <- xfdobj$basis
    xcoef  <- xfdobj$coefs
    xrange <- xbasis$rangeval
    
    #  check the dimensions of UFDLIST and AWTLIST and get number of forcing
    #    functions NFORCE
    
    if (is.null(ufdlist) | is.null(awtlist)) {
      nforce  <- 0
    } else {
      if (inherits(ufdlist[[1]], "list")) {
        #  UFDLIST is a list with a single component that is a list.
        #  convert to a list of length NFORCE.
        nforce <- length(ufdlist[[1]])
        temp <- vector("list", nforce)
        for (iu in 1:nforce) temp[[iu]] <- ufdlist[[1]][[iu]]
        ufdlist <- temp
      } else {
        nforce <- length(ufdlist)
      }
      if (inherits(awtlist[[1]], "list")) {
        #  AWTLIST is a list with a single component that is a list.
        #  convert to a list of length NFORCE.
        if (length(awtlist[[1]]) != nforce)
          stop("The length of AWTLIST is incorrect.")
        temp <- vector("list", nforce)
        for (iu in 1:nforce) temp[[iu]] <- awtlist[[1]][[iu]]
        awtlist <- temp
      } else {
        if (length(awtlist) != nforce)
          stop("The length of AWTLIST is incorrect.")
      }
    }
    
    #  check to see if there is anything to estimate
    
    if (difeorder == 0 && nforce == 0)
      stop("There are no coefficient functions to estimate.")
    
    ncurve     <- dim(xcoef)[2]
    
    nbasmax <- xbasis$nbasis
    
    #  check UFDLIST and AWTLIST
    
    if (nforce > 0) {
      errorwrd <- FALSE
      for (iu in 1:nforce) {
        if (!inherits(ufdlist[[iu]], "fd")) {
          print(paste("UFDLIST[[",iu,
                      "]] is not a functional data object.",sep=""))
          errorwrd <- TRUE
        } else {
          ufdi   <- ufdlist[[iu]]
          urange <- ufdi$basis$rangeval
          #  check that urange is equal to xrange
          if (any(urange != xrange)) {
            print(paste(
              "XRANGE and URANGE are not identical for UFDLIST[[",
              iu,"]].",sep=""))
            errorwrd <- TRUE
          }
        }
        afdPari <- awtlist[[iu]]
        afdi    <- afdPari$fd
        if (!inherits(afdi, "fd")) {
          print(paste(
            "AFDI is not a functional data object for AWTLIST[[",
            iu,"]].",sep=""))
          errorwrd <- TRUE
        } else {
          basisi <- afdi$basis
          if (any(basisi$rangeval != urange)) {
            print(paste("Ranges are incompatible for AWTLIST[[",
                        iu,"]].",sep=""))
            errorwrd <- TRUE
          }
          nbasmax <- max(c(nbasmax,basisi$nbasis))
        }
      }
      if (errorwrd) stop("")
    }
    
    #  check BWTLIST
    
    #  convert to a single-layer list of necessary
    if (inherits(bwtlist[[1]],"list")) {
      temp <- vector("list",difeorder)
      for (j in 1:nvar) {
        if (inherits(bwtlist[[1]][[j]], "list")) {
          bwtlist[[1]][[j]] <- bwtlist[[1]][[j]][[1]]
        }
        temp[[j]] <- bwtlist[[1]][[j]]
      }
      bwtlist <- temp
    }
    #  check the components
    errorwrd <- FALSE
    for (j in 1:difeorder) {
      if (!is.null(bwtlist[[j]])) {
        bfdParj <- bwtlist[[j]]
        if (!inherits(bfdParj,"fdPar")) {
          print(paste(
            "BWTLIST[[",j,"]] is not a functional parameter object.",sep=""))
          errorwrd <- TRUE
        }  else {
          bfdj <- bfdParj$fd
          if (!inherits(bfdj, "fd")) {
            print(paste(
              "BFDJ in BWTLIST[[",j,"]] is not a functional data object.",
              sep=""))
            errorwrd <- TRUE
          } else {
            basisj <- bfdj$basis
            if (any(basisj$rangeval != xrange)) print(paste(
              "Ranges are incompatible for BWTLIST[[",j,"]].",sep=""))
          }
        }
        nbasmax <- max(c(nbasmax,basisj$nbasis))
      }
    }
    if (errorwrd) stop("")
    
    #  Set up sampling values to be used in numerical integration
    #    and set up matrix of basis values.  The number of sampling
    #  NFINE is here set to a usually workable value if too small.
    
    if (nfine < 5*nbasmax) nfine <- 5*nbasmax
    
    deltax <- (xrange[2]-xrange[1])/(nfine-1)
    tx     <- seq(xrange[1],xrange[2],deltax)
    
    #  set up  YARRAY to hold values of x functions and their derivatives
    
    yarray <- array(0,c(nfine,ncurve,difeordp1))
    for (j in 1:difeordp1) yarray[,,j] <- eval.fd(tx, xfdobj, j-1)
    
    #  set up  UARRAY to hold values of u functions
    
    if (nforce > 0) {
      uarray <- array(0,c(nfine,ncurve,nforce))
      for (iu in 1:nforce)
        uarray[,,iu] <- eval.fd(tx, ufdlist[[iu]])
    }
    
    #  set up array YPROD to hold mean of products of values in YARRAY
    
    yprod <- array(0,c(nfine,difeordp1,difeordp1))
    for (j1 in 1:difeordp1) for (j2 in 1:j1) {
      if (ncurve == 1) yprodval <- yarray[,1,j1]*yarray[,1,j2]
      else             yprodval <- apply(yarray[,,j1]*yarray[,,j2],1,mean)
      yprod[,j1,j2] <- yprodval
      yprod[,j2,j1] <- yprodval
    }
    
    #  set up array YUPROD to hold mean of u-variables u times
    #    x functions and their derivatives
    
    if (nforce > 0) {
      yuprod <- array(0,c(nfine, nforce, difeordp1))
      for (iu in 1:nforce) {
        for (j1 in 1:difeordp1) {
          if (ncurve == 1) {
            yuprodval <- yarray[,1,j1]*uarray[,1,iu]
          } else {
            yuprodval <- apply(yarray[,,j1]*uarray[,,iu],1,mean)
          }
          yuprod[,iu,j1] <- yuprodval
        }
      }
    }
    
    #  set up array UPROD to hold mean of products of u-variables u
    
    if (nforce > 0) {
      uprod <- array(0,c(nfine, nforce, nforce))
      for (iu in 1:nforce) for (ju in 1:iu) {
        if (ncurve == 1) uprodval <- uarray[,1,iu]*uarray[,1,ju]
        else             uprodval <- apply(uarray[,,iu]*uarray[,,ju],1,mean)
        uprod[,iu,ju] <- uprodval
        uprod[,ju,iu] <- uprodval
      }
    }
    
    #  set up an index array and some arrays of 1's
    
    onesn <- rep(1,nfine)
    
    #  set up array to hold coefficients for basis expansions
    
    if (nforce > 0) {
      aarray <- matrix(0,nfine,nforce)
    } else {           
      aarray <- NULL
    }
    
    barray <- matrix(0,nfine,difeorder)
    
    #  --------------  beginning of loop through variables  -------------------
    
    #  get number of coefficients to be estimated for this equation
    
    # loop through u-variables
    
    neqns  <- 0
    
    if (nforce > 0) {
      for (iu in 1:nforce) {
        if (!is.null(awtlist[[iu]])) {
          afdPari <- awtlist[[iu]]
          if (afdPari$estimate)
            neqns <- neqns + afdPari$fd$basis$nbasis
        }
      }
    }
    
    # loop through x functions and their derivatives
    
    for (j1 in 1:difeorder) {
      if (!is.null(bwtlist[[j1]])) {
        bfdParj <- bwtlist[[j1]]
        if (bfdParj$estimate)
          neqns <- neqns + bfdParj$fd$basis$nbasis
      }
    }
    
    if (neqns < 1) stop(
      "Number of equations to solve is not positive.")
    
    #  set up coefficient array and right side array for linear equation
    
    cmat   <- matrix(0,neqns, neqns)
    dmat   <- matrix(0,neqns, 1)
    
    #  evaluate default weight functions for this variable
    
    if (nforce > 0) {
      for (iu in 1:nforce) {
        if (!is.null(awtlist[[iu]])) {
          afdPari     <- awtlist[[iu]]
          aarray[,iu] <- eval.fd(tx, afdPari$fd)
        }
      }
    }
    
    for (j1 in 1:difeorder) {
      if (!is.null(bwtlist[[j1]])) {
        bfdParj     <- bwtlist[[j1]]
        bvecj       <- eval.fd(tx, bfdParj$fd)
        barray[,j1] <- bvecj
      }
    }
    
    #  loop through equations,
    #    corresponding to rows for CMAT and DMAT
    
    #  loop through equations for u-variables
    
    mi12 <- 0
    if (nforce > 0) {
      for (iu1 in 1:nforce) {
        if (!is.null(awtlist[[iu1]])) {
          afdPari1   <- awtlist[[iu1]]
          if (afdPari1$estimate) {
            abasisi1    <- afdPari1$fd$basis
            abasismati1 <- getbasismatrix(tx, abasisi1)
            mi11 <- mi12 + 1
            mi12 <- mi12 + abasisi1$nbasis
            indexi1 <- mi11:mi12
            #  DMAT entry for u-variable
            weighti1 <- -yuprod[,iu1,difeordp1]
            dmat[indexi1] <-
              trapzmat(abasismati1,onesn,deltax,weighti1)
            # add terms corresponding to x-derivate weights
            # that are not estimated
            for (j1 in 1:difeorder) {
              bfdParij <- bwtlist[[j1]]
              if (!bfdParij$estimate) {
                weightij <- -yuprod[,iu1,j1]
                dmat[indexi1] <- dmat[indexi1] +
                  trapzmat(abasismati1, barray[,j1],
                           deltax, weightij)
              }
            }
            #  loop through weight functions to be estimated,
            #    corresponding to columns for CMAT
            #  begin with u-variables
            mi22 <- 0
            for (iu2 in 1:nforce) {
              if (!is.null(awtlist[[iu2]])) {
                afdPari2   <- awtlist[[iu2]]
                if (afdPari2$estimate) {
                  abasisi2    <- afdPari2$fd$basis
                  abasismati2 <- getbasismatrix(tx, abasisi2)
                  weighti2    <- uprod[,iu1,iu2]
                  Cprod  <- trapzmat(abasismati1, abasismati2,
                                     deltax, weighti2)
                  mi21 <- mi22 + 1
                  mi22 <- mi22 + abasisi2$nbasis
                  indexi2 <- mi21:mi22
                  #  coefficient matrix CMAT entry
                  cmat[indexi1,indexi2] <- Cprod
                }
              }
            }
            #  remaining columns:
            #    loop through u-variable -- x-derivative pairs
            mij22 <- mi22
            for (j2 in 1:difeorder) {
              if (!is.null(bwtlist[[j2]])) {
                bfdParj2     <- bwtlist[[j2]]
                if (bfdParj2$estimate) {
                  bbasisij2    <- bfdParj2$fd$basis
                  bbasismatij2 <- getbasismatrix(tx, bbasisij2)
                  weightij12   <- -yuprod[,iu1,j2]
                  Cprod <- trapzmat(abasismati1,bbasismatij2,
                                    deltax,weightij12)
                  mij21 <- mij22 + 1
                  mij22 <- mij22 + bbasisij2$nbasis
                  indexij2  <- mij21:mij22
                  cmat[indexi1,indexij2] <- Cprod
                }
              }
            }
            #  add roughness penalty matrix to diagonal entries
            lambdai1 <- afdPari1$lambda
            if (lambdai1 > 0) {
              Lfdobj <- afdPari1$Lfd
              penmat <- lambdai1*eval.penalty(abasisi1, Lfdobj)
              cmat[indexi1,indexi1] <- cmat[indexi1,indexi1] + penmat
            }
          }
        }
      }
    }
    
    #  loop through equations for x-derivatives
    
    mij12 <- mi12
    for (j1 in 1:difeorder) {
      if (!is.null(bwtlist[[j1]])) {
        bfdParj1 <- bwtlist[[j1]]
        if (bfdParj1$estimate) {
          bbasisij1    <- bfdParj1$fd$basis
          bbasismatij1 <- getbasismatrix(tx,bbasisij1)
          mij11 <- mij12 + 1
          mij12 <- mij12 + bbasisij1$nbasis
          indexij1 <- mij11:mij12
          #  DMAT entry for u-variable -- x-derivative pair
          weightij1 <- yprod[,j1,difeordp1]
          dmat[indexij1] <-
            trapzmat(bbasismatij1,onesn,deltax,weightij1)
          #  add terms corresponding to forcing functions
          #  with unestimated coefficients
          if (nforce > 0) {
            for (iu in 1:nforce) {
              if (!is.null(awtlist[[iu]])) {
                afdPari <- awtlist[[iu]]
                if (!afdPari$estimate) {
                  weightijk <- -yuprod[,iu,j1]
                  dmat[indexij1] <- dmat[indexij1] +
                    trapzmat(bbasisij1, aarray[,iu],deltax, weightijk)
                }
              }
            }
          }
        }
      }
      #  first columns of CMAT: u-variable entries
      mi22 <- 0
      if (nforce > 0) {
        for (iu2 in 1:nforce) {
          if (!is.null(awtlist[[iu2]])) {
            afdPari2 <- awtlist[[iu2]]
            if (afdPari2$estimate) {
              abasisi2    <- afdPari2$fd$basis
              abasismati2 <- getbasismatrix(tx, abasisi2)
              weighti2    <- -yuprod[,iu2,j1]
              Cprod <- trapzmat(bbasismatij1,abasismati2,deltax,weighti2)
              mi21 <- mi22 + 1
              mi22 <- mi22 + abasisi2$nbasis
              indexi2 <- mi21:mi22
              cmat[indexij1,indexi2] <- Cprod
            }
          }
        }
      }
      #  remaining columns: x-derivative pairs
      mij22 <- mi22
      for (j2 in 1:difeorder) {
        if (!is.null(bwtlist[[j2]])) {
          bfdParj2  <- bwtlist[[j2]]
          bbasisij2 <- bfdParj2$fd$basis
          if (bfdParj2$estimate) {
            bbasismatij2 <- getbasismatrix(tx, bbasisij2)
            weightij22   <- yprod[,j1,j2]
            Cprod <- trapzmat(bbasismatij1,bbasismatij2,deltax,weightij22)
            mij21 <- mij22 + 1
            mij22 <- mij22 + bbasisij2$nbasis
            indexij2 <- mij21:mij22
            cmat[indexij1,indexij2] <- Cprod
          }
        }
      }
      # add roughness penalty matrix to diagonal entries
      lambdaj1 <- bfdParj1$lambda
      if (lambdaj1 > 0) {
        Lfdobj <- bfdParj1$Lfd
        penmat <- lambdaj1*eval.penalty(bbasisij1, Lfdobj)
        cmat[indexij1,indexij1] <- cmat[indexij1,indexij1] + penmat
      }
    }
    
    #  --------------  end of loop through variables  -------------------
    
    # solve for coefficients of basis expansions
    
    dvec <- -symsolve(cmat,dmat)
    
    #  set up u-function weight functions
    
    mi2 <- 0
    if (nforce > 0) {
      for (iu in 1:nforce) {
        if (!is.null(awtlist[[iu]])) {
          afdPari <- awtlist[[iu]]
          if (afdPari$estimate) {
            mi1 <- mi2 + 1
            mi2 <- mi2 + afdPari$fd$basis$nbasis
            indexi <- mi1:mi2
            afdPari$fd$coefs <- as.matrix(dvec[indexi])
            awtlist[[iu]] <- afdPari
          }
        }
      }
    }
    
    #  set up X-function derivative weight functions
    
    mij2 <- mi2
    for (j in 1:difeorder) {
      if (!is.null(bwtlist[[j]])) {
        bfdParj <- bwtlist[[j]]
        if (bfdParj$estimate) {
          mij1 <- mij2 + 1
          mij2 <- mij2 + bfdParj$fd$basis$nbasis
          indexij <- mij1:mij2
          bfdParj$fd$coefs <- as.matrix(dvec[indexij])
          bwtlist[[j]] <- bfdParj
        }
      }
    }
    
    #  set up residual list RESFDLIST
    
    #  initialize with highest order derivative for this variable
    resmat  <- eval.fd(tx, xfdobj, difeorder)
    #  add contributions from weighted u-functions
    if (nforce > 0) {
      onesncurve <- rep(1,ncurve)
      for (iu in 1:nforce) {
        if (!is.null(awtlist[[iu]])) {
          afdPari  <- awtlist[[iu]]
          aveci    <- as.vector(eval.fd(tx, afdPari$fd))
          umati    <- eval.fd(tx, ufdlist[[iu]])
          aumati   <- outer(aveci,onesncurve)*umati
          resmat   <- resmat - aumati
        }
      }
    }
    #  add contributions from weighted x-function derivatives
    for (j in 1:difeorder) {
      if (!is.null(bwtlist[[j]])) {
        bfdParj <- bwtlist[[j]]
        bmatij  <- as.vector(eval.fd(tx, bfdParj$fd))
        xmatij  <- eval.fd(tx, xfdobj, j-1)
        resmat  <- resmat + bmatij*xmatij
      }
    }
    #  set up the functional data object
    resbasis <- xbasis
    resfd    <- smooth.basis(tx, resmat, resbasis)$fd
    resfdnames      <- xfdobj$fdnames
    resfdnames[[2]] <- "Residual function"
    resfdnames[[3]] <- "Residual function value"
    resfd$fdnames   <- resfdnames
    resfdlist       <- list(resfd)
    
    #  ----------------------------------------------------------------
    #                   End of single variable case
    #  ----------------------------------------------------------------
    
  } else {
    
    #  ----------------------------------------------------------------
    #                   Multiple variable case
    #  ----------------------------------------------------------------
    
    #  check the dimensions of UFDLIST and AWTLIST
    
    if (is.null(ufdlist) || is.null(awtlist)) {
      awtlist <- NULL
    } else {
      if (length(ufdlist) != nvar)
        stop(paste("The length of UFDLIST",
                   " does not match that of XFDLIST."))
      errorwrd = FALSE
      for (j in 1:nvar) {
        if (!is.null(ufdlist[[j]])) {
          nforce <- length(ufdlist[[j]])
          if (length(awtlist[[j]]) != nforce) {
            print(paste("The length of AWTLIST[[",j,
                        "]] is incorrect.",sep=""))
            errorwrd = TRUE
          }
        }
      }
      if (errorwrd) stop("")
    }
    
    #  check the dimensions of BWTLIST
    
    if (length(bwtlist) != nvar) stop("Length of BWTLIST is incorrect.")
    errorwrd = FALSE
    for (ivar in 1:nvar) {
      if (length(bwtlist[[ivar]]) != nvar) {
        print(paste("The length of BWTLIST[[",ivar,
                    "]] is incorrect.",sep=""))
        errorwrd = TRUE
      }
    }
    if (errorwrd) stop("")
    
    #  check XFDLIST and extract NCURVE and XRANGE
    
    xfd1       <- xfdlist[[1]]
    xcoef1     <- xfd1$coefs
    xbasis1    <- xfd1$basis
    xrange1    <- xbasis1$rangeval
    ncurve     <- dim(xcoef1)[2]
    resfdnames <- xfd1$fdnames
    
    errorwrd = FALSE
    for (ivar in 1:nvar) {
      xfdi    <- xfdlist[[ivar]]
      xcoefi  <- xfdi$coefs
      xbasisi <- xfdi$basis
      xrangei <- xbasisi$rangeval
      ncurvei <- dim(xcoefi)[2]
      if (!inherits(xfdi, "fd")) {
        print(paste("XFDLIST[[",ivar,
                    "]] is not a functional data object.",sep=""))
        errorwrd = TRUE
      } else {
        if (any(xrangei != xrange1)) {
          print("Ranges are incompatible for XFDLIST.")
          errorwrd = TRUE
        }
        if (ncurvei != ncurve) {
          print("Number of curves is incompatible for XFDLIST.")
          errorwrd = TRUE
        }
      }
    }
    if (errorwrd) stop("")
    
    nbasmax <- xbasis1$nbasis
    
    #  This will be the maximum number of basis functions
    
    #  check compatibility of UFDLIST and AWTLIST
    
    if (!(is.null(ufdlist) || is.null(awtlist))) {
      urange <- ufdlist[[1]]$basis$rangeval
      errorwrd <- FALSE
      for (ivar in 1:nvar) {
        if (!is.null(ufdlist[[ivar]])) {
          for (iu in 1:length(ufdlist[[ivar]])) {
            ufdiviu <- ufdlist[[ivar]][[iu]]
            if (!inherits(ufdiviu, "fd")) {
              print(paste("UFDLIST[[",ivar,",",iu,
                          "]] is not a functional data object.",
                          sep=""))
              errorwrd <- TRUE
            }
            if (any(ufdiviu$basis$rangeval != urange)) {
              print("Ranges are incompatible for UFDLIST.")
              errorwrd <- TRUE
            }
            awtfdPari <- awtlist[[ivar]][[iu]]
            if (!inherits(awtfdPari, "fdPar")) {
              print(paste("AWTFDPAR[[",ivar,"]][[",iu,
                          "]] is not a functional parameter object.",sep=""))
              errorwrd <- TRUE
            }
            afdi   <- awtfdPari$fd
            basisi <- afdi$basis
            if (any(basisi$rangeval != urange)) {
              print("Ranges are incompatible for AWTLIST.")
              errorwrd <- TRUE
            }
            nbasmax <- max(c(nbasmax,basisi$nbasis))
          }
          if (errorwrd) stop("")
        }
      }
    }
    
    #  check BWTLIST
    
    errorwrd <- FALSE
    for (ivar1 in 1:nvar) {
      for (ivar2 in 1:nvar) {
        difeorder <- length(bwtlist[[ivar1]][[ivar2]])
        for (j in 1:difeorder) {
          if (!is.null(bwtlist[[ivar1]][[ivar2]][[j]])) {
            bfdPari1i2j <- bwtlist[[ivar1]][[ivar2]][[j]]
            if (!inherits(bfdPari1i2j, "fdPar")) {
              print(paste("BWTLIST[[",ivar1, ",",ivar2, ",",j,
                          "]] is not a functional parameter object.",sep=""))
              errorwrd = TRUE
            }
            basisi1i2j <- bfdPari1i2j$fd$basis
            if (any(basisi1i2j$rangeval != xrange1)) {
              print(paste("Ranges are incompatible for BWTLIST[[",
                          ivar1,"]][[",ivar2,"]][[",
                          j,"]]",sep=""))
              errorwrd <- TRUE
            }
            nbasmax <- max(c(nbasmax,basisi1i2j$nbasis))
          }
        }
      }
    }
    if (errorwrd) stop("")
    
    #  set up sampling values to be used in numerical integration
    #    and set up matrix of basis values.  The number of sampling
    #  NFINE is here set to a usually workable value if too small.
    
    if (nfine < 5*nbasmax) nfine <- 5*nbasmax
    
    deltax <- (xrange1[2]-xrange1[1])/(nfine-1)
    tx     <- seq(xrange1[1],xrange1[2],deltax)
    
    #  set up  YARRAY to hold values of x functions and their derivatives
    
    yarray <- vector("list", 0)
    for (ivar in 1:nvar) {
      difeorder <- length(bwtlist[[ivar]][[ivar]])
      difeordp1 <- difeorder + 1
      yarray[[ivar]] <- array(0,c(nfine,ncurve,difeordp1))
      for (j in 1:difeordp1){
        yj <- eval.fd(tx, xfdlist[[ivar]], j-1)
        yarray[[ivar]][,,j] <- as.matrix(yj)
      }
    }
    
    #  set up  UARRAY to hold values of u functions
    
    if (!is.null(ufdlist)) {
      uarray <- vector("list", nvar)
      for (ivar in 1:nvar) {
        if (is.null(ufdlist[[ivar]])) {
          uarray[[ivar]] <- NULL
        } else {
          nforce <- length(ufdlist[[ivar]])
          uarray[[ivar]] <- vector("list", nforce)
          for (iu in 1:nforce)
            uarray[[ivar]][[iu]] <- matrix(0,nfine,ncurve)
        }
      }
      for (ivar in 1:nvar) {
        if (!is.null(ufdlist[[ivar]])) {
          nforce <- length(ufdlist[[ivar]])
          for (iu in 1:nforce)
            uarray[[ivar]][[iu]] <- eval.fd(tx, ufdlist[[ivar]][[iu]])
        }
      }
    }
    
    #  set up array YPROD to hold mean of products of values in YARRAY
    
    yprod <- vector("list", nvar)
    for (i1 in 1:nvar) yprod[[i1]] <- vector("list", nvar)
    for (i1 in 1:nvar) {
      difeord1p1 <- length(bwtlist[[i1]][[i1]]) + 1
      for (i2 in 1:nvar) {
        difeord2p1 <- length(bwtlist[[i2]][[i2]]) + 1
        yprod[[i1]][[i2]] <- array(0,c(nfine,difeord2p1,difeord2p1))
      }
    }
    
    for (i1 in 1:nvar) {
      difeord1p1 <- length(bwtlist[[i1]][[i1]]) + 1
      for (j1 in 1:difeordp1) {
        for (i2 in 1:nvar) {
          difeord2p1 <- length(bwtlist[[i2]][[i2]]) + 1
          for (j2 in 1:difeord2p1) {
            if (ncurve == 1) {
              yprodval <-       yarray[[i1]][,1,j1]*yarray[[i2]][,1,j2]
            } else {
              yprodval <- apply(yarray[[i1]][,,j1]*yarray[[i2]][,,j2],1,mean)
            }
            yprod[[i1]][[i2]][,j1,j2] <- yprodval
          }
        }
      }
    }
    
    #  set up array YUPROD to hold mean of u-variables u times
    #    x functions and their derivatives
    
    if (!is.null(ufdlist)) {
      yuprod <- vector("list", nvar)
      for (i1 in 1:nvar) {
        if (!is.null(ufdlist[[i1]])) {
          nforce <- length(ufdlist[[i1]])
          if (nforce > 0) {
            yuprod[[i1]] <- vector("list", nforce)
            for (iu in 1:nforce) {
              difeordp1 <- length(bwtlist[[i1]][[i1]]) + 1
              yuprod[[i1]][[iu]] <- matrix(0,nfine,difeordp1)
            }
          }
        }
      }
      onesncurve <- rep(1,ncurve)
      for (i1 in 1:nvar) {
        if (!is.null(ufdlist[[i1]])) {
          nforce <- length(ufdlist[[i1]])
          if (nforce > 0) {
            difeordp1 <- length(bwtlist[[i1]][[i1]]) + 1
            for (iu in 1:nforce) {
              for (j1 in 1:difeordp1) {
                if (ncurve == 1) {
                  yuprodval <- yarray[[i1]][,1,j1]*uarray[[i1]][[iu]]
                } else {
                  yuprodval <- apply(yarray[[i1]][,,j1]*
                                       outer(uarray[[i1]][[iu]],onesncurve),1,mean)
                }
                yuprod[[i1]][[iu]][,j1] <- yuprodval
              }
            }
          }
        }
      }
    }
    
    #  set up array UPROD to hold mean of products of u-variables u
    
    if (!is.null(ufdlist)) {
      uprod <- vector("list", nvar)
      for (ivar in 1:nvar) {
        nforce <- length(ufdlist[[ivar]])
        if (nforce > 0) {
          uprod[[ivar]] <- array(0,c(nfine, nforce, nforce))
          for (iu in 1:nforce) for (ju in 1:iu) {
            uprodval <- uarray[[ivar]][[iu]]*uarray[[ivar]][[ju]]
            uprod[[ivar]][,iu,ju] <- uprodval
            uprod[[ivar]][,ju,iu] <- uprodval
          }
        }
      }
    }
    
    #  set up an index array and some arrays of 1"s
    
    onesn <- rep(1,nfine)
    
    #  set up array to hold coefficients for basis expansions
    
    #  --------------  beginning of loop through variables  -------------------
    
    for (ivar in 1:nvar) {
      #  get number of coefficients to be estimated for this equation
      
      neqns  <- 0
      
      # loop through u-variables  if required
      
      if (is.null(ufdlist) || is.null(ufdlist[[ivar]])) nforce <- 0
      else  nforce <- length(ufdlist[[ivar]])
      if (nforce > 0) {
        for (iu in 1:nforce) {
          afdPari <- awtlist[[ivar]][[iu]]
          if (afdPari$estimate) {
            nbasisiu <- afdPari$fd$basis$nbasis
            neqns <- neqns + nbasisiu
          }
        }
      }
      
      # loop through x functions and their derivatives
      
      for (i2 in 1:nvar) {
        difeorder <- length(bwtlist[[ivar]][[i2]])
        for (j2 in 1:difeorder) {
          if (!is.null(bwtlist[[ivar]][[i2]][[j2]])) {
            bfdParij <- bwtlist[[ivar]][[i2]][[j2]]
            nbasisi2j2 = bfdParij$fd$basis$nbasis
            if (bfdParij$estimate) neqns <- neqns + nbasisi2j2
          }
        }
      }
      if (neqns < 1)  stop("Number of equations to solve is not positive.")
      
      #  set up coefficient array and right side array for linear equation
      
      cmat   <- matrix(0,neqns, neqns)
      dmat   <- matrix(0,neqns, 1)
      
      #  evaluate default weight functions for this variable
      
      if (nforce > 0) {
        aarray <- matrix(0,nfine,nforce)
        for (iu in 1:nforce) {
          if (!is.null(awtlist[[ivar]][[iu]])) {
            afdPari <- awtlist[[ivar]][[iu]]
            aarray[,iu] <- eval.fd(tx, afdPari$fd)
          }
        }
      }
      barray <- vector("list", nvar)
      for (i in 1:nvar) {
        difeorder <- length(bwtlist[[ivar]][[i]])
        barray[[i]] <- matrix(0,nfine,difeorder)
        for (j in 1:difeorder) {
          if (!is.null(bwtlist[[ivar]][[i]][[j]])) {
            bfdParij     <- bwtlist[[ivar]][[i]][[j]]
            barray[[i]][,j] <- as.matrix(eval.fd(tx, bfdParij$fd))
          }
        }
      }
      
      #  loop through equations,
      #    corresponding to rows for CMAT and DMAT
      
      #  loop through equations for u-variables
      
      mi12 <- 0
      if (nforce > 0) {
        for (iu1 in 1:nforce) {
          if (!is.null(awtlist[[ivar]][[iu1]])) {
            afdPari1   <- awtlist[[ivar]][[iu1]]
            if (afdPari1$estimate) {
              abasisi1    <- afdPari1$fd$basis
              abasismati1 <- getbasismatrix(tx, abasisi1)
              mi11 <- mi12 + 1
              mi12 <- mi12 + abasisi1$nbasis
              indexi1 <- mi11:mi12
              #  DMAT entry for u-variable
              weighti1 <- -yuprod[[ivar]][[iu1]][,difeordp1]
              dmat[indexi1] <- trapzmat(abasismati1,onesn,deltax,weighti1)
              #  add terms corresponding to x-derivative weights
              #  that are not estimated
              for (i in 1:nvar) {
                difeorder <- length(bwtlist[[ivar]][[i]])
                for (j in 1:difeorder) {
                  bfdParij <- bwtlist[[ivar]][[i]][[j]]
                  if (!is.null(bwtlist[[ivar]][[i]][[j]])) {
                    if (!bfdParij$estimate) {
                      weightij <- -yuprod[[ivar]][[iu1]][,j]
                      dmat[indexi1] <- dmat[indexi1] +
                        trapzmat(abasismati1, barray[[ivar]][,j],
                                 deltax, weightij)
                    }
                  }
                }
              }
              #  loop through weight functions to be estimated,
              #    corresponding to columns for CMAT
              #  begin with u-variables
              mi22 <- 0
              for (iu2 in 1:nforce) {
                if (!is.null(awtlist[[ivar]][[iu2]])) {
                  afdPari2   <- awtlist[[ivar]][[iu2]]
                  if (afdPari2$estimate) {
                    abasisi2    <- afdPari2$fd$basis
                    abasismati2 <- getbasismatrix(tx, abasisi2)
                    weighti2    <- uprod[[ivar]][,iu1,iu2]
                    Cprod       <- trapzmat(abasismati1, abasismati2,
                                            deltax, weighti2)
                    mi21 <- mi22 + 1
                    mi22 <- mi22 + abasisi2$nbasis
                    indexi2 <- mi21:mi22
                    #  coefficient matrix CMAT entry
                    cmat[indexi1,indexi2] <- Cprod
                  }
                }
              }
              #  remaining columns:
              #    loop through u-variable -- x-derivative pairs
              mij22 <- mi22
              for (i2 in 1:nvar) {
                if (!is.null(bwtlist[[ivar]][[i2]])) {
                  difeorder <- length(bwtlist[[ivar]][[i2]])
                  for (j2 in 1:difeorder) {
                    bfdParij2   <- bwtlist[[ivar]][[i2]][[j2]]
                    if (bfdParij2$estimate) {
                      bbasisij2    <- bfdParij2$fd$basis
                      bbasismatij2 <- getbasismatrix(tx, bbasisij2)
                      weightij12   <- -yuprod[[i2]][[iu1]][,j2]
                      Cprod        <- trapzmat(abasismati1,bbasismatij2,
                                               deltax,weightij12)
                      mij21 <- mij22 + 1
                      mij22 <- mij22 + bbasisij2$nbasis
                      indexij2  <- mij21:mij22
                      cmat[indexi1,indexij2] <- Cprod
                    }
                  }
                }
              }
              #  add roughness penalty matrix to diagonal entries
              lambdai1 <- afdPari1$lambda
              if (lambdai1 > 0) {
                Lfdobj <- afdPari1$Lfd
                penmat <- lambdai1*eval.penalty(abasisi1,Lfdobj)
                cmat[indexi1,indexi1] <- cmat[indexi1,indexi1] + penmat
              }
            }
          }
        }
      }
      
      #  loop through equations for x-derivatives
      
      mij12 <- mi12
      for (i1 in 1:nvar) {
        difeorder1 <- length(bwtlist[[ivar]][[i1]])
        difeordp1  <- difeorder1 + 1
        for (j1 in 1:difeorder1) {
          if (!is.null(bwtlist[[ivar]][[i1]][[j1]])) {
            bfdParij1 <- bwtlist[[ivar]][[i1]][[j1]]
            if (bfdParij1$estimate) {
              bbasisij1    <- bfdParij1$fd$basis
              bbasismatij1 <- getbasismatrix(tx, bbasisij1)
              mij11 <- mij12 + 1
              mij12 <- mij12 + bbasisij1$nbasis
              indexij1 <- mij11:mij12
              #  DMAT entry for u-variable -- x-derivative pair
              weightij1 <- yprod[[i1]][[ivar]][,j1,difeordp1]
              trapzij1  <- trapzmat(bbasismatij1,onesn,deltax,weightij1)
              dmat[indexij1] <- trapzij1
              #  add terms corresponding to forcing functions
              #  with unestimated coefficients
              if (nforce > 0) {
                for (iu in 1:nforce) {
                  if (!is.null(awtlist[[ivar]][[iu]])) {
                    afdPari <- awtlist[[ivar]][[iu]]
                    if (!afdPari$estimate) {
                      weightijk <- yprod[,ivar,iu,j1]
                      trapzijk  <-trapzmat(bbasismatij1,aarray[,iu],
                                           deltax,weightijk)
                      dmat[indexij1] <- dmat[indexij1] + trapzijk
                    }
                  }
                }
              }
              #  first columns of CMAT: u-variable entries
              mi22 <- 0
              if (nforce > 0) {
                for (iu2 in 1:nforce) {
                  if (!is.null(awtlist[[ivar]][[iu2]])) {
                    afdPari2  <- awtlist[[ivar]][[iu2]]
                    if (afdPari2$estimate) {
                      abasisi2    <- afdPari2$fd$basis
                      abasismati2 <- getbasismatrix(tx, abasisi2)
                      weighti2    <- -yuprod[[i1]][[iu2]][,j1]
                      mi21 <- mi22 + 1
                      mi22 <- mi22 + abasisi2$nbasis
                      indexi2 <- mi21:mi22
                      Cprod <- trapzmat(bbasismatij1,abasismati2,deltax,weighti2)
                      cmat[indexij1,indexi2] <- cmat[indexij1,indexi2] + Cprod
                    }
                  }
                }
              }
              #  remaining columns: x-derivative pairs
              mij22 <- mi22
              for (i2 in 1:nvar) {
                difeorder2 <- length(bwtlist[[ivar]][[i2]])
                for (j2 in 1:difeorder2) {
                  if (!is.null(bwtlist[[ivar]][[i2]][[j2]])) {
                    bfdParij2 <- bwtlist[[ivar]][[i2]][[j2]]
                    bbasisij2    <- bfdParij2$fd$basis
                    bbasismatij2 <- getbasismatrix(tx, bbasisij2)
                    weightij22   <- yprod[[i1]][[i2]][,j1,j2]
                    Cprod <- trapzmat(bbasismatij1,bbasismatij2,deltax,weightij22)
                    if (bfdParij2$estimate) {
                      mij21 <- mij22 + 1
                      mij22 <- mij22 + bbasisij2$nbasis
                      indexij2 <- mij21:mij22
                      cmat[indexij1,indexij2] <- cmat[indexij1,indexij2] + Cprod
                    }
                  }
                }
              }
              #  add roughness penalty terms to diagonal entries
              lambdaij1 <- bfdParij1$lambda
              if (lambdaij1 > 0) {
                Lfdobj <- bfdParij1$Lfd
                penmat <- lambdaij1*eval.penalty(bbasisij1,Lfdobj)
                cmat[indexij1,indexij1] <- cmat[indexij1,indexij1] +
                  penmat
              }
            }
          }
        }
      }
      dvec <- -solve(cmat,dmat)
      
      #  set up u-function weight functions
      
      mi2 <- 0
      if (nforce > 0) {
        for (iu in 1:nforce) {
          if (!is.null(awtlist[[ivar]][[iu]])) {
            afdPari <- awtlist[[ivar]][[iu]]
            if (afdPari$estimate) {
              mi1 <- mi2 + 1
              mi2 <- mi2 + afdPari$fd$basis$nbasis
              indexi <- mi1:mi2
              afdPari$fd$coefs <- as.matrix(dvec[indexi])
              awtlist[[ivar]][[iu]] <- afdPari
            }
          }
        }
      }
      
      #  set up X-function derivative weight functions
      
      mij2 <- mi2
      for (i1 in 1:nvar) {
        difeorder <- length(bwtlist[[ivar]][[i1]])
        for (j1 in 1:difeorder) {
          if (!is.null(bwtlist[[ivar]][[i1]][[j1]])) {
            bfdParij <- bwtlist[[ivar]][[i1]][[j1]]
            if (bfdParij$estimate) {
              mij1 <- mij2 + 1
              mij2 <- mij2 + bfdParij$fd$basis$nbasis
              indexij <- mij1:mij2
              bfdParij$fd$coefs <- as.matrix(dvec[indexij])
              bwtlist[[ivar]][[i1]][[j1]] <- bfdParij
            }
          }
        }
      }
    }
    
    #  --------------  end of loop through variables  -------------------
    
    #  set up residual list RESFDLIST
    
    resfdlist <- vector("list", nvar)
    
    for (ivar in 1:nvar) {
      difeorder <- length(bwtlist[[ivar]][[ivar]])
      xfdi      <- xfdlist[[ivar]]
      resbasis  <- xfdi$basis
      #  initialize with highest order derivative for this variable
      resmat    <- eval.fd(tx, xfdi, difeorder)
      #  add contributions from weighted u-functions
      onesncurve <- rep(1,ncurve)
      if (!is.null(ufdlist)) {
        nforce <- length(ufdlist[[ivar]])
        if (nforce > 0) {
          for (iu in 1:nforce) {
            if (!is.null(awtlist[[ivar]][[iu]])) {
              afdPari  <- awtlist[[ivar]][[iu]]
              amati    <- as.vector(eval.fd(tx, afdPari$fd))
              umati    <- eval.fd(tx, ufdlist[[ivar]][[iu]])
              if (ncurve == 1) aumati <- amati*umati
              else             aumati <- outer(amati,onesncurve)*umati
              resmat   <- resmat - aumati
            }
          }
        }
      }
      #  add contributions from weighted x-function derivatives
      for (i1 in 1:nvar) {
        difeorder <- length(bwtlist[[ivar]][[i1]])
        for (j1 in 1:difeorder) {
          if (!is.null(bwtlist[[ivar]][[i1]][[j1]])) {
            bfdParij <- bwtlist[[ivar]][[i1]][[j1]]
            bfdij    <- bfdParij$fd
            bvecij   <- as.vector(eval.fd(tx, bfdij))
            if (ncurve == 1) {
              bmatij <- bvecij
            }  else  {
              bmatij <- outer(bvecij,onesncurve)
            }
            xmatij <- eval.fd(tx, xfdlist[[i1]], j1-1)
            resmat <- resmat + bmatij*xmatij
          }
        }
      }
      #  set up the functional data object
      resfdi            <- smooth.basis(tx, resmat, resbasis)$fd
      resfdnames        <- xfdi$fdnames
      resfdnames[[2]]   <- "Residual function"
      resfdnames[[3]]   <- "Residual function value"
      resfdlist[[ivar]] <- resfdi
    }
    
    #  ----------------------------------------------------------------
    #                   End of multiple variable case
    #  ----------------------------------------------------------------
    
  }
  
  pdaList <- list(bwtlist=bwtlist, resfdlist=resfdlist, awtlist=awtlist)
  class(pdaList) <- 'pda.fd'
  pdaList
}
phaseplanePlot <- function(evalarg, fdobj, Lfdobj1=1, Lfdobj2=2,
      lty=c("longdash", "solid"),
      labels=list(evalarg=seq(evalarg[1], max(evalarg), length=13),
             labels=fda::monthLetters),
      abline=list(h=0, v=0, lty=2),
      xlab="Velocity", ylab="Acceleration",
                       returnMatrix=FALSE, ... ){
##
## 1.  Check 'evalarg'
##
  if(missing(evalarg))
    evalarg <- fdobj$basis$rangeval
  if(length(evalarg)<3){
    if(length(evalarg)<2)evalarg[2] <- evalarg+1
    evalarg <- seq(evalarg[1], evalarg[2], length=181)
  }
##
## 2.  Compute points to plot
##
  Eval <- sort(unique(c(evalarg, labels$evalarg)))
  D1 <- eval.fd(Eval, fdobj, Lfdobj1, returnMatrix)
  D2 <- eval.fd(Eval, fdobj, Lfdobj2, returnMatrix)
#
  nT <- length(Eval)
  n2 <- ceiling(nT/2)
##
## 3.  Set up the plot
##
  plot(range(D1), range(D2), xlab=xlab, ylab=ylab,
       type="n", ...)
  if(!is.null(abline))do.call("abline", abline)
##
## 4.  Plot the lines
##
  lines(D1[1:n2], D2[1:n2], lty=lty[1])
  lines(D1[n2:nT], D2[n2:nT], lty=lty[2])
##
## 5. Label midmonths
##
  D1. <- eval.fd(labels$evalarg, fdobj, Lfdobj1, returnMatrix)
  D2. <- eval.fd(labels$evalarg, fdobj, Lfdobj2, returnMatrix)
  text(D1., D2., labels$labels)
##
## 6.  Done
##
  out <- cbind(D1, D2)
  fd.name <- deparse(substitute(names))
  D1.name <- {
    if(is.numeric(Lfdobj1) && Lfdobj1==1)
      "Velocity"
    else
      paste(fd.name, deparse(substitute(Lfdobj1)), sep=".")
  }
  D2.name <- {
    if(is.numeric(Lfdobj2) && Lfdobj1==2)
      "Acceleration"
    else
      paste(fd.name, deparse(substitute(Lfdobj1)), sep=".")
  }
  dimnames(out) <- list(names(evalarg), c(D1.name, D2.name))
#
  invisible(cbind(D1, D2))
}
plot.basisfd <- function(x, knots=TRUE, axes=NULL, ...) {
  basisobj <- x
#  plot a basis object

# last modified 26 May 2012 by Jim Ramsay

#  check BASISOBJ

  if (!inherits(basisobj, "basisfd"))
    stop("argument x is not a basis object.")
#
  dot.args <- list(...)
  {
    if(is.null(axes)){
      if(is.null(x$axes)){
        dot.args$axes <- TRUE
        axFun <- FALSE
      }
      else {
        if(!inherits(x$axes, 'list'))
          stop('x$axes must be a list;  class(x$axes) = ',
               class(x$axes))
        if(!(inherits(x$axes[[1]], 'character') ||
             inherits(x$axes[[1]], 'function') ) )
          stop('x$axes[[1]] must be either a function or the ',
               'name of a function;  class(x$axes[[1]]) = ',
               class(x$axes[[1]]) )
        axList <- c(x$axes, ...)
        dot.args$axes <- FALSE
        axFun <- TRUE
      }
    }
    else{
      if(is.logical(axes)){
        dot.args$axes <- axes
        axFun <- FALSE
      }
      else{
        if(!inherits(axes, 'list'))
          stop('axes must be a logical or a list;  class(axes) = ',
               class(axes))
        if(!(inherits(axes[[1]], 'character') ||
             inherits(axes[[1]], 'function') ) )
          stop('axes[[1]] must be either a function or the ',
               'name of a function;  class(axes[[1]]) = ',
               class(axes[[1]]) )
        axList <- c(axes, ...)
        dot.args$axes <- FALSE
        axFun <- TRUE
      }
    }
  }

  if(is.null(dot.args$xlab))dot.args$xlab <- ''
  if(is.null(dot.args$ylab))dot.args$ylab <- ''

  nbasis   <- basisobj$nbasis

  if(is.null(dot.args$type))dot.args$type <- 'l'

  if(is.null(dot.args$lty))
    dot.args$lty <- rep(1:3, max(1, nbasis/3))

  nx       <- max(501,10*nbasis)

  {
    if(is.null(dot.args$xlim))rangex <- basisobj$rangeval
    else {
      rangex <- dot.args$xlim
      rangex[1] <- max(basisobj$rangeval[1], dot.args$xlim[1])
      rangex[2] <- min(rangex[2], dot.args$xlim[2])
    }
  }

  argvals  <- seq(rangex[1],rangex[2],len=nx)
  basismat <- eval.basis(argvals, basisobj)

#  minval   <- min(basismat)
#  maxval   <- max(basismat)
#  if (minval == maxval) {
#    if (abs(minval) < 1e-1) {
#      minval <- minval - 0.05
#      maxval <- maxval + 0.05
#    } else {
#      minval <- minval - 0.05*minval
#      maxval <- maxval + 0.05*minval
#    }
#  }

#  matplot (argvals, basismat, type="l", lty=ltype,
#           xlab=xlabel, ylab=ylabel, cex=cexval,
#           xlim=c(argvals[1],argvals[nx]),
#           ylim=, ...)

  dot.args$x <- argvals
  dot.args$y <- basismat

  do.call('matplot', dot.args)

# knots?
  if(knots && (x$type=='bspline'))
    abline(v=knots(x), lty='dotted', col='red')
# axes?
  if(axFun)
    do.call(axList[[1]], axList[-1])
}
plotbeta = function(betaestlist, betastderrlist=NULL, argvals=NULL,
                    xlab="", ...)
{
#  PLOTBETA plots a functional parameter along with confidence
#  limits
#  Arguments
#  BETAESTLIST    ... A list object containing one or more functional
#                     parameter objects or functional data objects.
#  BETASTDERRLIST ... A list object containing functional data objects
#                     for the standard error of the objects in
#                     BETAESTLIST.

#  Last modified 6 January 2020

#  check BETAESTLIST

if (inherits(betaestlist, "fdPar") || inherits(betaestlist, "fd")) {
    betaestlist = list(betaestlist)
}

if (!inherits(betaestlist, "list")) {
    stop("BETAESTLIST is not a list, fd, or fdpar object.")
}

#  check BETASTDERRLIST

  if (!is.null(betastderrlist)){
    if (inherits(betastderrlist, "fd")) {
      betastderrlist = list(betastderrlist)
    }
    if (!inherits(betastderrlist, "list")) {
      stop("BETASTDERRLIST is not a list, or fd object.")
    }
  }
  
#  get range

if (is.fdPar(betaestlist[[1]])) {
    rangeval = betaestlist[[1]]$fd$basis$rangeval
} else {
    if (is.fd(betaestlist[[1]])) {
        rangeval = betaestlist[[1]]$basis$rangeval
    } else {
        stop(paste("A list does not contain either a functional parameter ",
           "or a functional data object."))
    }
}

if (is.null(argvals)) {
    argvals = seq(rangeval[1],rangeval[2],len=51)
}
n = length(argvals)
p = length(betaestlist)

par(ask=T)
for (j in 1:p) {
    if (is.fdPar(betaestlist[[j]])) {
        betavec = eval.fd(argvals, betaestlist[[j]]$fd)
    } else {
        if (is.fd(betaestlist[[j]])) {
            betavec = eval.fd(argvals, betaestlist[[j]])
        } else {
            stop(
        "BETAESTLIST does not contain a functional parameter or data object.")
        }
    }
  zeroval  = c(0,0)
  if (is.null(betastderrlist)) {
      plot(argvals, betavec, type="l", xlab=xlab, ylab="",
           xlim=rangeval, ylim=c(min(betavec),max(betavec)), ...)
      lines(rangeval, zeroval, col=1, lty=3)
    } else {
      betastderr = eval.fd(argvals, betastderrlist[[j]])
      betavecp   = betavec + 2*betastderr
      betavecm   = betavec - 2*betastderr
      plot(argvals, betavec, type="l", xlab=xlab, ylab="",
           xlim=rangeval, ylim=c(min(betavecm),max(betavecp)), ...)
      lines(rangeval, zeroval,lty=3, col=2)
      lines(argvals, betavecp, col=1, lwd=1)
      lines(argvals, betavecm, col=1, lwd=1)
    }
    title(paste("Regression function ",j))
}

}
plot.cca.fd <- function(x, cexval = 1, ...)
{
#  Plot a functional canonical correlation analysis object CCAFD
#
#  Other arguments are passed to plot.fd
#
# last modified 2007 May 3 by Spencer Graves
#  Previously modified 20 March 2006

ccafd <- x

if (!(inherits(ccafd, "cca.fd"))) stop("First argument not of CCA.FD class.")

ccafd1    <- ccafd[[1]]
ccacoef1  <- ccafd1$coefs
ccabasis1 <- ccafd1$basis
ccafd2    <- ccafd[[2]]
ccacoef2  <- ccafd1$coefs
ccabasis2 <- ccafd2$basis

rangeval <- ccabasis1$rangeval

argvals <- seq(rangeval[1],rangeval[2],len=201)
ccamat1 <- eval.fd(argvals, ccafd1)
ccamat2 <- eval.fd(argvals, ccafd2)

ncan <- dim(ccacoef1)[2]
par(mfrow=c(2,1), pty="s")
if (ncan > 1) par(ask=TRUE) else par(ask=FALSE)
for (j in (1:ncan)) {
    plot(argvals, ccamat1[,j], type="l", cex=cexval,
         ylab="First  canonical weight", main=paste("Harmonic",j))
    plot(argvals, ccamat2[,j], type="l", cex=cexval,
         ylab="Second canonical weight", main="")
}
par(ask=FALSE)

invisible(NULL)
}
plot.fdSmooth <- function(x, y, Lfdobj=0, href=TRUE, titles=NULL,
                          xlim=NULL, ylim=NULL, xlab=NULL,
                          ylab=NULL, ask=FALSE, nx=NULL, axes=NULL,
                          ...){
  plot(x$fd, y, Lfdobj=Lfdobj, href=href, titles=titles,
       xlim=xlim, ylim=ylim, xlab=xlab,
       ylab=ylab, ask=ask, nx=nx, axes=axes, ...)
}

plot.fdPar <- function(x, y, Lfdobj=0, href=TRUE, titles=NULL,
                    xlim=NULL, ylim=NULL, xlab=NULL,
                    ylab=NULL, ask=FALSE, nx=NULL, axes=NULL, ...){
  plot(x$fd, y, Lfdobj=Lfdobj, href=href, titles=titles,
       xlim=xlim, ylim=ylim, xlab=xlab,
       ylab=ylab, ask=ask, nx=nx, axes=axes, ...)
}

plot.fd <- function(x, y, Lfdobj=0, href=TRUE, titles=NULL,
                    xlim=NULL, ylim=NULL, xlab=NULL,
                    ylab=NULL, ask=FALSE, nx=NULL, axes=NULL, 
                    ...) {
  
  #  -----------------------------------------------------------------------
  #       plot for fd class
  #  -----------------------------------------------------------------------
  
  #  Plot a functional data object fdobj.
  #  Arguments:
  #  fdobj     ... a functional data object
  #  Lfdobj    ... linear differental operator to be applied to fdobj before
  #             plotting
  #  HREF   ... If TRUE, a horizontal dotted line through 0 is plotted.
  #  The argument ASK, if TRUE, causes the curves to be displayed one at a time.
  #  NX     ... The number of sampling points to use for
  #             plotting.  (default 101)
  
  #  The remaining optional arguments are the same as those available
  #     in the regular "plot" function.
  
  #  Note that for multivariate fdobj, a suitable matrix of plots
  #    must be set up before calling plot by using something such as
  #    par(mfrow=c(1,nvar),pty="s")
  
  # last modified 2022-02-18 by Spencer Graves
  
  ##
  ## 1.  Basic checks
  ##
  fdobj <- x
  if (!(is.fd(fdobj) || is.fdPar(fdobj)))  stop(paste(
    "First argument is neither a functional data or a ",
    "functional parameter object."))
  if (is.fdPar(fdobj)) fdobj <- fdobj$fd
  
  # process axes
  
  if(is.null(axes)) {
    if(is.null(fdobj$basis$axes)) {
      Axes <- TRUE
      axFun <- FALSE
    } else {
      if(!inherits(fdobj$basis$axes, 'list'))
        stop('fdobj$basis$axes must be a list;  ',
             'class(fdobj$basis$axes) = ', class(fdobj$basis$axes))
      if(!(inherits(fdobj$basis$axes[[1]], 'character') ||
           inherits(fdobj$basis$axes[[1]], 'function') ) )
        stop('fdobj$basis$axes[[1]] must be either a function or the ',
             'name of a function;  class(fdobj$basis$axes[[1]]) = ',
             class(fdobj$basis$axes[[1]]) )
      Axes <- FALSE
      axFun <- TRUE
      axList <- c(fdobj$basis$axes, ...)
    }  
  } else {
    if(is.logical(axes)){
      Axes <- axes
      axFun <- FALSE
    } else {
      if(!inherits(axes, 'list'))
        stop('axes must be a logical or a list;  class(axes) = ',
             class(axes))
      if(!(inherits(axes[[1]], 'character') ||
           inherits(axes[[1]], 'function') ) )
        stop('axes[[1]] must be either a function or the ',
             'name of a function;  class(axes[[1]]) = ',
             class(axes[[1]]) )
      Axes <- FALSE
      axFun <- TRUE
      axList <- c(axes, ...)
    }
  }
  
  #  check Lfdobj
  
  Lfdobj <- int2Lfd(Lfdobj)
  if (!inherits(Lfdobj, "Lfd")) stop(
    "Second argument is not a linear differential operator.")
  
  #  extract dimension information
  
  coef   <- fdobj$coefs
  coefd  <- dim(coef)
  ndim   <- length(coefd)
  # Number of basis functions
  nbasis    <- coefd[1]
  if(is.null(nx)) nx <- max(c(501,10*nbasis + 1))
  # Number of functional observations
  nrep   <- coefd[2]
  if (ndim > 2) nvar <- coefd[3] else nvar <- 1
  
  #  get basis information
  
  basisobj <- fdobj$basis
  rangex   <- basisobj$rangeval
  #  set up a set of argument values for the plot
  
  if (missing(y)) {
    y <- nx
  } else {
    if(is.numeric(y)) y <- as.vector(y)
  }
  
  Y <- y
  if (length(y) == 1) {
    if (y >= 1) {
      y <- seq(rangex[1],rangex[2],len=round(y))
    } else {
      stop("'y' a single number less than one.")
    }
  }
  if (min(y) < rangex[1] || max(y) > rangex[2])
    stop("Values in Y are outside the basis range.")
  if (is.null(xlim)){
    xlim <- rangex
  } else {
    rangex[1] <- max(rangex[1], xlim[1])
    rangex[2] <- min(rangex[2], xlim[2])
    if(length(Y)==1)
      y <- seq(rangex[1],rangex[2],len=round(Y))
  }
  
  #  evaluate LFDOBJ(FDOBJ) at the argument values
  
  fdmat    <- eval.fd(y, fdobj, Lfdobj)
  rangey   <- range(fdmat)
  if (is.null(ylim)) ylim <- rangey
  
  #  set up axis labels and,
  #  optionally, caselabels and variable labels
  
  fdnames      = fdobj$fdnames
  fdlabelslist = fdlabels(fdnames, nrep, nvar)
  
  # Ramsay 2008.08.26
  xlabel    = fdlabelslist$xlabel
  ylabel    = fdlabelslist$ylabel
  casenames = fdlabelslist$casenames
  varnames  = fdlabelslist$varnames
  
  #  check xlab and ylab
  if (is.null(xlab)) xlab <- xlabel
  if (is.null(ylab)) ylab <- ylabel
  #  if (missing(xlab)) xlab <- xlabel
  #  if (missing(ylab)) ylab <- ylabel
  #  crvnames <- fdobj$fdnames[[2]]
  #  varnames <- fdobj$fdnames[[3]]
  # Don't ask for the first plot, but do for later plots if(ask)
  #  op <- par(ask=FALSE)
  # Don't ask for the first plot,
  # but if ask==TRUE, do ask for succeeding plots
  #  on.exit(par(op))
  # A single line?
  
  # A single line?
  if (ndim < 2) {
    plot (y, fdmat, type="l", xlim=xlim, ylim=ylim,
          xlab=xlab, ylab=ylab, axes=Axes, ...)
    if(axFun)
      do.call(axList[[1]], axList[-1])
    #   Ramsay 2008.08.26
    if (zerofind(fdmat) && href) abline(h=0,lty=2)
    #   Graves 2008.07.04
    #    if (zerofind(ylim) && href) abline(h=0,lty=2)
  }
  # Several copies of one function?
  if (ndim ==2 ) {
    if (!ask) {
      matplot(y, fdmat, type="l",
              xlim=xlim,   ylim=ylim,
              xlab=xlab, ylab=ylab, axes=Axes, 
              ...)
      if(axFun)
        do.call(axList[[1]], axList[-1])
      #   Ramsay 2008.08.26
      if (zerofind(fdmat) && href) abline(h=0,lty=2)
      #   Graves 2008.07.04
      #    if (zerofind(ylim) && href) abline(h=0,lty=2)
    } else  {
      #   Graves 2008.07.04:  par, cat absent from Ramsay 2008.08.26
      op <- par(ask=FALSE)
      # Don't ask for the first plot,
      # but if ask==TRUE, do ask for succeeding plots
      on.exit(par(op))
      cat('Multiple plots:  Click in the plot to advance to the next')
      #      op <- par(ask = TRUE)
      #      on.exit(par(op))
      for (irep in 1:nrep) {
        plot (y, fdmat[,irep], type="l",
              xlim=xlim, ylim=ylim,
              xlab=xlab, ylab=ylab, axes=Axes, ...)
        if(axFun)
          do.call(axList[[1]], axList[-1])
        if(irep<2) par(ask=ask)
        
        #        if (zerofind(ylim) && href) abline(h=0,lty=2)
        #        if (!is.null(titles)) title(titles[irep])
        #        else title(paste(crvnames[irep]))
        
        if (!is.null(casenames)) title(casenames[irep])
        else                     title(paste("Case",irep))
        #        if (zerofind(fdmat[,irep]) && href) abline(h=0,lty=2)
        if (zerofind(ylim) && href) abline(h=0,lty=2)
        
        #        mtext("Click in graph to see next plot", side=3, outer=FALSE)
        #        text("",locator(1))
      }
    }
  }
  # Possibly multiple copies of different functions
  if (ndim == 3) {
    if (!ask) {
      for (ivar in 1:nvar) {
        matplot (y, fdmat[,,ivar], type="l",
                 xlim=xlim, ylim=ylim,
                 xlab=xlab, ylab=ylab, ask=FALSE, axes=Axes, 
                 ...)
        if(axFun)
          do.call(axList[[1]], axList[-1])
        if (!is.null(varnames)) title(varnames[ivar])
        else                    title(paste("Variable",ivar))
        #        if (zerofind(fdmat[,,ivar]) && href) abline(h=0,lty=2)
        if (zerofind(ylim) && href) abline(h=0,lty=2)
      }
    } else {
      op <- par(ask=FALSE)
      # Don't ask for the first plot,
      # but if ask==TRUE, do ask for succeeding plots
      on.exit(par(op))
      cat('Multiple plots:  Click in the plot to advance to the next')
      
      for (irep in 1:nrep) {
        for (ivar in 1:nvar) {
          plot(y,fdmat[,irep,ivar],type="l",
               xlim=xlim, ylim=ylim,
               xlab=xlab, ylab=ylab, axes=Axes, ...)
          if(axFun)
            do.call(axList[[1]], axList[-1])
          if (!is.null(casenames)) titlestr = casenames[irep]
          else                  titlestr = paste("Case",irep)
          if (!is.null(varnames)) {
            titlestr = paste(titlestr,"  ",varnames[ivar])
          } else {
            titlestr = paste(titlestr,"  ","Variable",ivar)
          }
          title(titlestr)
          #          if (zerofind(fdmat[,irep,ivar]) && href) abline(h=0,lty=2)
          if (zerofind(ylim) && href) abline(h=0,lty=2)
          #          if (!is.null(titles)) title(titles[irep])
          #          else title(paste("Curve", irep, varnames[ivar]))
          
          #          mtext("Click in graph to see next plot", side=3, outer=FALSE)
          #          text("",locator(1))
        }
      }
    }
  }
  #  invisible(NULL)
  # This used to return 'invisible(NULL)'.
  # However, with R 2.7.0 under XEmacs with ESS,
  # it sometimes failed to plot.  I changed the return value,
  # and the problem disappeared.
  'done'
}

#  --------------------------------------------------------------------

zerofind <- function(fmat)
{
  frng <- range(fmat)
  if (frng[1] <= 0 && frng[2] >= 0) zeroin <- TRUE else zeroin <- FALSE
  return(zeroin)
}



#plotfit <- function (x, ...){
#  UseMethod("plotfit")
#}

plotfit.fdSmooth <- function(y, argvals, fdSm, rng = NULL,
                       index = NULL, nfine = 101, residual = FALSE,
                       sortwrd = FALSE, titles=NULL,  ylim=NULL,
                       ask=TRUE, type=c("p", "l")[1+residual],
                       xlab=NULL, ylab=NULL, sub=NULL, col=1:9,
                       lty=1:9, lwd=1, cex.pch=1, axes=NULL, ...) {
  plotfit.fd(y, argvals, fdSm$fd, rng = rng, index = index,
             nfine = nfine, residual = residual,
             sortwrd = sortwrd, titles=titles,  ylim=ylim,
             ask=ask, type=c("p", "l")[1+residual],
             xlab=xlab, ylab=ylab, sub=sub, col=1:9, lty=1:9,
             lwd=1, cex.pch=1, axes=axes, ...)
}

plotfit.fd <- function(y, argvals, fdobj, rng = NULL,
                       index = NULL, nfine = 101, residual = FALSE,
                       sortwrd = FALSE, titles=NULL,  ylim=NULL,
                       ask=TRUE, type=c("p", "l")[1+residual],
                       xlab=NULL, ylab=NULL, sub=NULL, col=1:9, lty=1:9,
                       lwd=1, cex.pch=1, axes=NULL, ...)
{
  #PLOTFIT plots discrete data along with a functional data object for
  #  fitting the data.  It is designed to be used after DATA2FD,
  #  SMOOTH.FD or SMOOTH.BASIS to check the fit of the data offered by
  #  the FD object.
  
  #  Arguments:
  #  Y        ... the data used to generate the fit
  #  ARGVALS  ... discrete argument values associated with data
  #  fdobj       ... a functional data object for fitting the data
  #  RNG      ... a range of argument values to be plotted
  #  INDEX    ... an index for plotting subsets of the curves
  #       (either sorted or not)
  #  NFINE    ... number of points to use for plotting curves
  #  RESIDUAL ... if TRUE, the residuals are plotted instead of
  #         the data plus curve
  #  SORTWRD  ... sort plots by mean square error
  #  TITLES   ... vector of title strings for curves
  
  # Last modified 18 January 2020 by Jim Ramsay
  
  ##
  ## 1.  Basic checks
  ##
  #  check third argument, FDOBJ
  
  if (!(is.fd(fdobj) || is.fdPar(fdobj)))  stop(
    "Third argument is neither a functional data nor a functional parameter object.")
  if (is.fdPar(fdobj)) fdobj <- fdobj$fd
  
  #  extract basis object from FDOBJ
  basisobj <- fdobj$basis
  #  set range of arguments
  if (is.null(rng)) rng <- basisobj$rangeval
  #  extract FDNAMES from FDOBJ
  fdnames <- fdobj$fdnames
  yName   <- substring(deparse(substitute(y)), 1, 33)
  fdName  <- paste(substring(deparse(substitute(fdobj)), 1, 22),
                   "$coef", sep='')
  ##
  ## 2.  Use 'checkDims' to reconcile y and fdoj$coef
  ##
  #  The default fdnames may not work well
  defaultNms <- c(fdnames[2], fdnames[3], x='x')
  if((length(defaultNms[[2]])<2) && !is.null(names(defaultNms))
     && !is.na(names(defaultNms)[2]))
    defaultNms[[2]] <- names(defaultNms)[2]
  subset <- checkDims3(y, fdobj$coef, defaultNames = defaultNms,
                       xName=yName, yName=fdName)
  y <- subset$x
  fdobj$coef <- subset$y
  #  number of argument values
  n <- dim(y)[1]
  #  number of replications
  nrep <- dim(y)[2]
  #  number  of variables
  nvar <- dim(y)[3]
  curveno <- 1:nrep
  #  names for argument values
  argname <- names(fdnames)[[1]]
  if (is.null(argname)) {
    if (is.null(fdnames[[1]])) argname <- "Argument Value"
    else                       argname <- fdnames[[1]]
  }
  if (is.null(xlab)) xlab <- argname
  #  names for replicates
  casenames <- dimnames(y)[[2]]
  #  names for variables
  varnames  <- dimnames(y)[[3]]
  if (is.null(axes)) {
    if (is.null(fdobj$basis$axes)) {
      Axes  <- TRUE
      axFun <- FALSE
    } else {
      if (!inherits(fdobj$basis$axes, "list"))
        stop("fdobj$basis$axes must be a list;  ",
             "class(fdobj$basis$axes) = ", class(fdobj$basis$axes))
      if (!(inherits(fdobj$basis$axes[[1]], "character") ||
            inherits(fdobj$basis$axes[[1]], "function")))
        stop("fdobj$basis$axes[[1]] must be either a function or the ",
             "name of a function;  class(fdobj$basis$axes[[1]]) = ",
             class(fdobj$basis$axes[[1]]))
      Axes   <- FALSE
      axFun  <- TRUE
      axList <- c(fdobj$basis$axes, ...)
    }
  } else {
    if (is.logical(axes)) {
      Axes  <- axes
      axFun <- FALSE
    }
    else {
      if (!inherits(axes, "list"))
        stop("axes must be a logical or a list;  class(axes) = ",
             class(axes))
      if (!(inherits(axes[[1]], "character") || inherits(axes[[1]],
                                                         "function")))
        stop("axes[[1]] must be either a function or the ",
             "name of a function;  class(axes[[1]]) = ",
             class(axes[[1]]))
      Axes <- FALSE
      axFun <- TRUE
      axList <- c(axes, ...)
    }
  }
  dots <- list(...)
  if (is.null(titles) && ("main" %in% names(dots)))
    titles <- dots$main
  ##
  ## 3.  Computed fitted values for argvals and a fine mesh
  ##
  yhat.  <- eval.fd(argvals, fdobj)
  yhat   <- as.array3(yhat.)
  res    <- y - yhat
  MSE    <- apply(res^2,c(2,3),mean)
  dimnames(MSE) <- list(casenames, varnames)
  MSEsum <- apply(MSE,1,sum)
  #  compute fitted values for fine mesh of values
  xfine  <- seq(rng[1], rng[2], len=nfine)
  yfine  <- array(eval.fd(xfine, fdobj),c(nfine,nrep,nvar))
  #  sort cases by MSE if desired?????
  if (sortwrd && nrep > 1) {
    MSEind <- order(MSEsum)
    y      <- y    [,MSEind,, drop=FALSE]
    yhat   <- yhat [,MSEind,, drop=FALSE]
    yfine  <- yfine[,MSEind,, drop=FALSE]
    res    <- res  [,MSEind,, drop=FALSE]
    MSE    <- MSE  [ MSEind,, drop=FALSE]
    casenames  <- casenames[MSEind]
    titles     <- titles[MSEind]
  }
  
  #  set up fit and data as 3D arrays, selecting curves in INDEX
  
  if (is.null(index)) index <- 1:nrep
  #
  nrepi <- length(index)
  y     <- y    [,index,, drop=FALSE]
  yhat  <- yhat [,index,, drop=FALSE]
  res   <- res  [,index,, drop=FALSE]
  yfine <- yfine[,index,, drop=FALSE]
  MSE   <- MSE  [ index,, drop=FALSE]
  casenames <- casenames[index]
  titles    <- titles[index]
  
  # How many plots on a page?
  nOnOne <- 1   #  only one plot is default
  # types of plots
  if (ask & ((nvar*nrepi/nOnOne) > 1))
    cat('Multiple plots:  Click in the plot to advance to the next plot\n')
  col <- rep(col, length=nOnOne)
  lty <- rep(lty, length=nOnOne)
  lwd <- rep(lwd, length=nOnOne)
  cex.pch <- rep(cex.pch, length=nOnOne)
  
  #  select values in ARGVALS, Y, and YHAT within RNG
  
  argind    <- ((argvals >= rng[1]) & (argvals <= rng[2]))
  argvals   <- argvals[argind]
  casenames <- casenames[argind]
  y         <- y   [argind,,, drop=FALSE]
  yhat      <- yhat[argind,,, drop=FALSE]
  res       <- res [argind,,, drop=FALSE]
  n         <- length(argvals)
  
  xfiind <- ((xfine >= rng[1]) & (xfine <= rng[2]))
  xfine  <- xfine[xfiind]
  yfine  <- yfine[xfiind,,, drop=FALSE]
  nfine  <- length(xfine)
  ##
  ## 4.  Plot the results
  ##
  ndigit = abs(floor(log10(min(c(MSE)))) - 1)
  if (is.null(sub))
    sub <- paste("  RMS residual =", round(sqrt(MSE),ndigit))
  if (length(sub) != length(MSE)){
    warning('length(sub) = ', length(sub), ' != ',
            length(MSE), ' = ', length(MSE), '; forcing equality')
    sub <- rep(sub, length=length(MSE))
  }
  if (is.null(dim(sub))){
    dim(sub) <- dim(MSE)
  }
  if (!all(dim(sub)==dim(MSE))) {
    warning('dim(sub) = ', dim(sub), " != dim(MSE) = ",
            paste(dim(MSE), collapse=', '), ';  forcing equality.')
    dim(sub) <- dim(MSE)
  }
  # 'ask' is controlled by 'nOnOne' ...
  op <- par(ask=FALSE)
  # Don't ask for the first plot,
  # but if ask==TRUE, do ask for succeeding plots
  on.exit(par(op))
  # Reset 'ask' on.exit (normal or otherwise)
  iOnOne <- 0
  if (residual) {
    #  plot the residuals
    if(is.null(ylim))ylim <- range(res)
    if(is.null(ylab))ylab=paste('Residuals for', varnames)
    for (j in 1:nvar) {
      for (i in 1:nrepi){
        if(iOnOne %in% c(0, nOnOne)[1:(1+ask)]){
          if(is.null(xlab))xlab <- argname
          plot(rng, ylim, type="n", xlab=xlab,
               ylab=ylab[j], axes=Axes, ...)
          if(axFun)
            do.call(axList[[1]], axList[-1])
          par(ask=ask)
          abline(h=0, lty=4, lwd=2)
          if(nOnOne==1){
            {
              if (is.null(titles))
                title(main=casenames[i],sub=sub[i, j])
              else title(main=titles[i], sub=sub[i, j])
            }
          }
          iOnOne <- 0
        }
        iOnOne <- iOnOne+1
        lines(argvals, res[,i,j], type=type,
              col=col[iOnOne], lty=lty[iOnOne],
              lwd=lwd[iOnOne], cex=cex.pch[iOnOne])
      }
    }
  } else {
    #  plot the data and fit
    if(is.null(ylim))ylim <- range(c(c(y),c(yfine)))
    varname <- names(fdnames)[[3]]
    if (is.null(varname)) {
      if (is.null(fdnames[[3]])) rep("Function Value", nvar)
      else                       varname <- fdnames[[3]]
    }
    if(is.null(ylab))ylab <- varname
    if (nvar == 1) {
      for (i in 1:nrepi) {
        if(iOnOne %in% c(0, nOnOne)[1:(1+ask)]){
          plot(rng, ylim, type="n", xlab=xlab,
               ylab=ylab, axes=Axes, ...)
          if(axFun)
            do.call(axList[[1]], axList[-1])
          par(ask=ask)
          if(nOnOne==1){
            if(is.null(titles)) title(main=casenames[i],
                                      sub=sub[i, 1])
            else title(main=titles[i], sub=sub[i, 1])
          }
          iOnOne <- 0
        }
        iOnOne <- iOnOne+1
        points(argvals, y[,i,1], type=type,
               xlab=xlab, ylab=varnames, col=col[iOnOne],
               lty=lty[iOnOne], lwd=lwd[iOnOne],
               cex=cex.pch[iOnOne])
        lines(xfine, yfine[,i,1], col=col[iOnOne],
              lty=lty[iOnOne], lwd=lwd[iOnOne])
      }
    } else {
      if (ask) {
        aski = FALSE
        for (i in 1:nrepi) {
          par(mfrow=c(nvar,1),ask=aski)
          for (j in 1:nvar) {
            plot(rng, ylim, type="n", xlab=xlab,ylab=varnames[j], axes=Axes)
            if (axFun) do.call(axList[[1]], axList[-1])
            if (j == 1) {
              if (is.null(titles)) {
                title(paste("Record",i))
              } else {
                title(main=titles[i], sub=sub[i, j])
              }
            }
            points(argvals, y[,i,j], xlab=xlab, ylab=varnames[j])
            lines(xfine, yfine[,i,j])
          }
          aski = TRUE
        }
      } else {
        askj <- FALSE
        for (j in 1:nvar) {
          par(mfrow=c(1,1),ask=askj)
          matplot(argvals, y[,,j], type="p", pch="o", ylim=ylim, xlab=xlab,
                  ylab=varnames[j])
          if (axFun) do.call(axList[[1]], axList[-1])
          matlines(xfine, yfine[,,j])
          askj <- TRUE
        }
      }
    }
  }
  invisible(NULL)
}
plot.pca.fd <- function(x, nx = 128, pointplot = TRUE, harm = 0,
                        expand = 0, cycle = FALSE, ...)
{
#
#	Plots the harmonics produced by PCA.FD.
#
#  If pointplot=TRUE, then the harmonics are plotted as + and -
#  otherwise lines are used.	Another thing that needs doing is an
#		 arrowplot option.
#
# If harm = 0 (the default) then all the computed harmonics are plotted.
#	 Otherwise those in jharm are plotted.
# If expand =0 then effect of +/- 2 standard deviations of each pc are given
#	 otherwise the factor expand is used.
# If cycle=TRUE and there are 2 variables then a cycle plot will be drawn
#	If the number of variables is anything else, cycle will be ignored.
#

# last modified 2007 May 3 by Spencer Graves	
#	previously modified 20 March 2006

  pcafd <- x
  if (!(inherits(pcafd, "pca.fd"))) 
    stop("Argument 'x' is not a pca.fd object.")

  harmfd	<- pcafd[[1]]
  basisfd <- harmfd$basis
  rangex	<- basisfd$rangeval
  {
    if(length(nx)>1){
      x <- nx
      nx <- length(x)
    }
    else    
      x <- seq(rangex[1], rangex[2], length = nx)
  }
  fdmat	 <- eval.fd(x, harmfd)
  meanmat <- eval.fd(x, pcafd$meanfd)
  dimfd	 <- dim(fdmat)
  nharm	 <- dimfd[2]
#
# check number of panels
  plotsPerPg <- sum(par("mfrow"))
#   
  harm	<- as.vector(harm)
  if(harm[1] == 0) harm <- (1:nharm)
#  
  if(length(dimfd) == 2) {
    for(jharm in 1:length(harm)) {
#    for(iharm in harm) {
      if(jharm==2){
        op <- par(ask=TRUE)
        on.exit(par(op)) 
      }
#        
      iharm <- harm[jharm] 
      if(expand == 0) {
        fac <- 2*sqrt(pcafd$values[iharm])
      } else {
        fac <- expand
      }
#      
      vecharm <- fdmat[, iharm]
      pcmat <- cbind(meanmat + fac * vecharm, meanmat - fac * vecharm)
      if (pointplot) plottype <- "p" else plottype <- "l"
      percentvar <- round(100 * pcafd$varprop[iharm], 1)
      plot(x, meanmat, type = "l", ylim=c(min(pcmat),max(pcmat)),
           ylab=paste("Harmonic", iharm),
           main=paste("PCA function", iharm,
             "(Percentage of variability", percentvar, ")"),
           ...)
      if (pointplot) {
        points(x, pcmat[,1], pch="+")
        points(x, pcmat[,2], pch="-")
      } else {
        lines(x, pcmat[,1], lty=2)
        lines(x, pcmat[,2], lty=3)
      }
    }
  } else {
    if(cycle && dimfd[3] == 2) {
      meanmat <- drop(meanmat)
      for(jharm in 1:length(harm)) {
#      for(iharm in harm) {
        if(jharm==2){
          op <- par(ask=TRUE)
          on.exit(par(op)) 
        }
        iharm <- harm[jharm]
#
        {
          if(expand == 0) fac <- 2 * sqrt(pcafd$values[iharm])
          else fac <- expand
        }
        matharm <- fdmat[, iharm,	]
        mat1 <- meanmat + fac * matharm
        mat2 <- meanmat - fac * matharm
        if (pointplot) plottype <- "p" else plottype <- "l"
        percentvar <- round(100 * pcafd$varprop[iharm],1)
        plot(meanmat[,1], meanmat[,2], type=plottype,
             xlim=c(min(c(mat1[,1],mat2[,1])),max(c(mat1[,1],mat2[,1]))),
             ylim=c(min(c(mat1[,2],mat2[,2])),max(c(mat1[,2],mat2[,2]))),
             main=paste("PCA function", iharm,
               "(Percentage of variability", percentvar, ")"),
             ...)
        if (pointplot) {
          points(mat1[, 1], mat1[, 2], pch="+")
          points(mat2[, 1], mat2[, 2], pch="-")
        }
        else {
          lines (mat1[, 1], mat1[, 2], lty=2)
          lines (mat2[, 1], mat2[, 2], lty=3)
        }
      }
    }
    else {
      for(jharm in 1:length(harm)) {
#      for(iharm in harm) {
        if(jharm==2){
          op <- par(ask=TRUE)
          on.exit(par(op)) 
        }
        iharm <- harm[jharm]
#        
        fac <- {
          if (expand == 0) 2*sqrt(pcafd$values[iharm]) 
          else expand
        }
        
        meanmat <- drop(meanmat)
        matharm <- fdmat[, iharm, ]
        nvar <- dim(matharm)[2]
        for (jvar in 1:nvar) {
          pcmat <- cbind(meanmat[, jvar] + fac * matharm[, jvar],
                         meanmat[, jvar] - fac * matharm[, jvar])
          if (pointplot) plottype <- "p" else plottype <- "l"
          percentvar <- round(100 * pcafd$varprop[iharm], 1)
          plot(x, meanmat[,jvar], type=plottype,
               ylab=paste("Harmonic", iharm),
               sub = paste("PCA function", iharm,
                 "(Percentage of variability", 
                 percentvar,")"),
               main = dimnames(fdmat)[[3]][jvar],
               ...)
          if (pointplot) {
            points(x, pcmat[,1], pch="+")
            points(x, pcmat[,2], pch="-")
          }
          else {
            lines (x, pcmat[,1], lty=2)
            lines (x, pcmat[,2], lty=3)
          }
        }
      }
    }
  }
  invisible(NULL)
}
plot.pda.fd = function(x, whichdim=1,npts=501,...)
{
  # Last modified 16 January 2020
  
  # This basically plots the elements of bwtlist, allowing the user
  # to specify how the functions are collected.
  
  if (!inherits(x, "pda.fd")) stop("First argument is not of class pda.fd.")
  
  rangval = x$resfdlist[[1]]$basis$rangeval

  m = length(x$resfdlist)
  tfine = seq(rangval[1],rangval[2],length.out=npts)

  whichdim=unique(sort(whichdim))

  bwtlist = x$bwtlist

  # Firstly the one-variable case, do we plot all the functions
  # on one plot or not?
  if(m == 1){
    d = length(bwtlist)

    if(whichdim == 3){
      par(mfrow=c(d,1))
      for(i in 1:d){
        titlestr = paste('Coefficient for Derivative',i-1)
        plot(bwtlist[[i]]$fd,main=titlestr,...)
        }
    }
    else{
      betamat = matrix(0,npts,d)
      legendstr = c()

      for(i in 1:d){
        betamat[,i] = eval.fd(tfine,bwtlist[[i]]$fd)
        legendstr = c(legendstr,paste('Deriv',i))
      }
      xlabstr = names(bwtlist[[1]]$fd$fdnames)[[1]]
      ylabstr = names(bwtlist[[1]]$fd$fdnames)[[3]]

      matplot(tfine,betamat,type='l',lty=c(1:d),xlab=xlabstr,ylab=ylabstr,...)
      legend(x='topleft',legend=legendstr,lty=c(1:d),...)
    }
  }

  # Otherwise, we can plot by any combination of variables,
  # equations and derivatives.

  else{
    d = length(bwtlist[[1]][[1]])

    xlabstr = names(bwtlist[[1]][[1]][[1]]$fd$fdnames)[[1]]
    ylabstr = names(bwtlist[[1]][[1]][[1]]$fd$fdnames)[[3]]

    betamat = array(0,c(npts,m,m,d))
    legendstr = array('',c(m,m,d))

    for(i in 1:m){
      for(j in 1:m){
        for(k in 1:d){
                betamat[,i,j,k] = eval.fd(tfine,bwtlist[[i]][[j]][[k]]$fd)
                legendstr[i,j,k] = paste('var',i,'eq',j,'deriv',k)
        }
      }
    }

    if(length(whichdim)==1){
      if(whichdim==1){
        par(mfrow=c(m,1))
        for(i in 1:m){
          tbetamat = matrix(betamat[,i,,],npts,m*d,byrow=FALSE)
          tlegendstr = as.vector(legendstr[i,,])
          matplot(tfine,tbetamat,type='l',lty=c(1:(d*m)),col=c(1:(d*m)),xlab=xlabstr,ylab=ylabstr,...)
          legend(x='topleft',legend=tlegendstr,lty=c(1:(d*m)),col=c(1:(d*m)),...)
        }
      }
      if(whichdim==2){
        par(mfrow=c(m,1))
        for(j in 1:m){
          tbetamat = matrix(betamat[,,j,],npts,m*d,byrow=FALSE)
          tlegendstr = as.vector(legendstr[,j,])
          matplot(tfine,tbetamat,type='l',lty=c(1:(d*m)),col=c(1:(d*m)),xlab=xlabstr,ylab=ylabstr,...)
          legend(x='topleft',legend=tlegendstr,lty=c(1:(d*m)),col=c(1:(d*m)),...)
        }
      }
      if(whichdim==3){
        par(mfrow=c(d,1))
        for(k in 1:d){
          tbetamat = matrix(betamat[,,,k],npts,m*m,byrow=FALSE)
          tlegendstr = as.vector(legendstr[,,k])
          matplot(tfine,tbetamat,type='l',lty=c(1:(m*m)),col=c(1:(m*m)),xlab=xlabstr,ylab=ylabstr,...)
          legend(x='topleft',legend=tlegendstr,lty=c(1:(m*m)),col=c(1:(m*m)),...)
        }
      }
    }
    else if(length(whichdim)==2){
      if(whichdim[1]==1){
        if(whichdim[2]==2){
          par(mfrow=c(m,m))
          for(i in 1:m){
            for(j in 1:m){
              matplot(tfine,betamat[,i,j,],type='l',lty=c(1:d),col=c(1:d),xlab=xlabstr,ylab=ylabstr,...)
              legend(x='topleft',legend=legendstr[i,j,],lty=c(1:d),col=c(1:d),...)
            }
          }
        }
        if(whichdim[2]==3){
          par(mfrow=c(m,d))
          for(i in 1:m){
            for(k in 1:d){
              matplot(tfine,betamat[,i,,k],type='l',lty=c(1:m),col=c(1:m),xlab=xlabstr,ylab=ylabstr,...)
              legend(x='topleft',legend=legendstr[i,,k],lty=c(1:m),col=c(1:m),...)
            }
          }
        }
      }
      else{
        par(mfrow=c(m,d))
        for(j in 1:m){
          for(k in 1:d){
            matplot(tfine,betamat[,,j,k],type='l',lty=c(1:m),col=c(1:m),xlab=xlabstr,ylab=ylabstr,...)
            legend(x='topleft',legend=legendstr[,j,k],lty=c(1:m),col=c(1:m),...)
          }
        }
      }
    }
    else{
      for(j in 1:m){
        #X11()
        dev.new()
        par(mfrow=c(m,d))
        for(i in 1:m){
          for(k in 1:d){
            plot(bwtlist[[i]][[j]][[k]]$fd,main=legendstr[i,j,k],...)
          }
        }
      }
    }
  }
}
plotscores <- function(pcafd, scores = c(1, 2), xlab = NULL, ylab = NULL,
                       loc = 1, matplt2 = FALSE, ...)
{
#
#   Plot a scatter plot of the pca scores from a pca.fd object
#   If loc >0, you can then click on the plot in loc places and you'll get
#    plots of the functions with these values of the principal component
#    coefficients.
#
#  The present implementation doesn't work for multivariate functional data
#
#    pcafd      a pca.fd object
#    scores     a two dimensional vector giving the indices of the two
#                  scores to be plotted; if scores is a single number then
#                  that score will be plotted against component 1; the default
#                  is to print the first two components.
#    xlab, ylab   labels for the principal components scores scatterplot
#    loc   if an integer, the number of sample functions to be plotted.
#                  This number of clicks on the first plot are needed.
#      if a list with components x and y, the coordinates of the
#         functions to be plotted (the output from a previous call
#                  of plotscores, for instance).  No prompting will be done.
#               if 0 or NULL nothing is plotted beyond the scatterplot.
#
#    matplt2    the matplt value for the plot of the sample functions;
#               if matplt=TRUE, the curves are plotted on the same plot;
#               if matplt=FALSE, they are plotted separately.
#
#   ...         arguments to be passed to the pc scores scatterplot
#
#  RETURNS:   a list containing the PC scores of the plotted functions

#  Last modified 26 October 2005

   if (!(inherits(pcafd, "pca.fd"))) stop('Argument PCAFD is not a pca.fd object.')

   if(length(scores) == 1) scores <- c(1, scores)
   if(length(scores) != 2)
      scores <- c(1, 2)
   if(max(scores) > dim(pcafd$harmonics$coefs)[2]) {
      stop(paste("The pca.fd object does not contain ", max(scores),
         "components"))
   }
   if(is.null(xlab))
      xlab <- paste("PCA score ", scores[1])
   if(is.null(ylab))
      ylab <- paste("PCA score ", scores[2])
   plot(pcafd$scores[, scores], xlab = xlab, ylab = ylab, ...)
   if(is.list(loc))
      zz <- loc
   else {
      if(is.na(loc) || is.null(loc) || loc == 0)
         return()
      zz <- locator(loc)
   }
   zzmat <- rbind(zz$x, zz$y)
   coefs <- pcafd$meanfd$coefs %*% rep(1, dim(zzmat)[2]) + pcafd$harmonics$
      coefs[, scores] %*% zzmat
   fdnames <- pcafd$meanfd$fdnames
   fdnames[[2]] <- paste("Score", scores[1], "=", signif(zz$x, 2),
      "; Score", scores[2], "=", signif(zz$y, 2))
   names(fdnames)[2] <- "Sample function"
   names(fdnames)[3] <- "Function value"
#   fd <- create.fd(coefs, pcafd$meanfd$basis, fdnames)
   fd <- fd(coefs, pcafd$meanfd$basis, fdnames)
   plot(fd, matplt = matplt2)
   return(zz)
}
polintmat <- function(xa, ya, x) {
#  Polynomial extrapolation for a converging sequence
#  YA is an 3-D array with 1st D same as XA
  n     <- length(xa)
  dimya <- dim(as.array(ya))
  if (length(dimya) == 1) ya <- array(ya,c(dimya[1],1,1))
  if (length(dimya) == 2) ya <- array(ya,c(dimya[1],dimya[2],1))
  if (dimya[1] != n)      stop('First dimension of YA must match XA')
  difx <- xa - x
  absxmxa <- abs(difx)
  ns <- min((1:n)[absxmxa == min(absxmxa)])
  cs <- ds <- ya
  y  <- ya[ns,,]
  ns <- ns - 1
  for (m in 1:(n-1)) {
    for (i in 1:(n-m)) {
      ho      <- difx[i]
      hp      <- difx[i+m]
      w       <- (cs[i+1,,] - ds[i,,])/(ho - hp)
      ds[i,,] <- hp*w
      cs[i,,] <- ho*w
    }
    if (2*ns < n-m) {
      dy <- cs[ns+1,,]
    } else {
      dy <- ds[ns,,]
      ns <- ns - 1
    }
    y <- y + dy
  }
  return( list(y, dy) )
}
polygpen <- function(basisobj, Lfdobj=int2Lfd(1))
{

#  Computes the polygonal penalty matrix.
#  Arguments:
#  BASISOBJ ... a basis object of type "polyg"
#  LFDOBJ   ... either the order of derivative or a
#               linear differential operator to be penalized.
#          The highest derivative must be either 0 or 1.
#  Returns the penalty matrix.

#  Last modified 3 January 2008

if (!(inherits(basisobj, "basisfd"))) stop(
    "First argument is not a basis object.")

Lfdobj <- int2Lfd(Lfdobj)

type <- basisobj$type
if (type != "polyg" && type != "polygonal") 
stop("BASISOBJ not of type polyg or polygonal")

nderiv <- Lfdobj$nderiv

if (nderiv > 1) stop(
    "Derivative greater than 1 cannot be taken for polygonal basis.")

bwtlist <- Lfdobj$bwtlist
isintLfd <- TRUE
if (nderiv > 0) {
  for (ideriv in 1:nderiv) {
    fdj <- bwtlist[[ideriv]]
    if (!is.null(fdj)) {
      if (any(fdj$coefs != 0)) {
        isintLfd <- FALSE
        break
      }
    }
  }
}

if (isintLfd) {
    args    <- basisobj$params
    n       <- length(args)
    argdiff <- diff(args)
    penaltymat <- diag(rep(1,n))
    if (nderiv == 0) {
        penaltymat[1,1] = argdiff[  1]/3
        penaltymat[n,n] = argdiff[n-1]/3
        indx = 2:(n-1)
        for (i in indx)
            penaltymat[i,i] = (argdiff[i]+argdiff[i-1])/3
        indx = 2:n
        for (i in indx) {
            penaltymat[i,i-1] = argdiff[i-1]/6
            penaltymat[i-1,i] = argdiff[i-1]/6
        }
    } else {
        argdiff = 1/argdiff
        penaltymat[1,1] = argdiff[  1]
        penaltymat[n,n] = argdiff[n-1]
        indx = 2:(n-1)
        for (i in indx)
            penaltymat[i,i] = argdiff[i]+argdiff[i-1]
        indx = 2:n
        for (i in indx) {
            penaltymat[i,i-1] = -argdiff[i-1]
            penaltymat[i-1,i] = -argdiff[i-1]
        }
    }
    penaltymatrix <- penaltymat
} else {
    penaltymatrix <- inprod(basisobj, basisobj, Lfdobj, Lfdobj)
}

return( penaltymatrix )
}
polyg <- function(x, argvals, nderiv=0)
{
#  Evaluates the basis for a linear interpolant or its first derivative.
#  It calls function spline.des.
#  Arguments are as follows:
#  X      ... array of values at which the spline functions are to
#             evaluated
#  ARGVAL ... a STRICTLY INCREASING sequence of argument values.
#  NDERIV ... Either 0 or 1.  0 means only function values
#             are returned; 1 means derivative values are returned
#  Return is a matrix with length(X) rows and number of columns equal to
#             number of argument values

#  last modified 8 June 1999

  x <- as.vector(x)
  n <- length(x)

  if (!is.array(argvals)) argvals <- as.array(argvals)
  if (length(dim(argvals)) != 1) stop(
     'ARGVALS is not a vector or 1-dim. array.')
  if ( (max(x) > max(argvals)) || (min(x) < min(argvals)) ) stop(
     'ARGVALS do not span the values of X.')

  nargvals <- length(argvals)
  if (min(diff(argvals)) <= 0 ) stop(
     'Break-points are not strictly increasing')

  if (!(nderiv == 0 | nderiv == 1)) stop(
     'NDERIV is neither 0 nor 1.')
  derivs    <- rep(nderiv,n)
  nbasis <- length(argvals)

  knots <- c(argvals[1], argvals, argvals[nbasis])
  basismat <- spline.des(knots, x, 2, derivs)$design

  return (basismat)
}
polynompen <- function(basisobj, Lfdobj=2)
{

#  Computes the polynomial penalty matrix for polynomials of the form
#      (x-ctr)^l
#  Arguments:
#  BASISOBJ ... a basis object of type "polynom"
#  LFDOBJ   ... either the order of derivative or a  nonhomogeneous 
#               linear differential operator to be penalized.
#  Returns the penalty matrix.

#  Last modified 17 January 2006

if (!(inherits(basisobj, "basis"))) stop(
    "First argument is not a basis.fd object.")

Lfdobj <- int2Lfd(Lfdobj)

type <- basisobj$type
  if (type != "polynom") stop("BASISOBJ not of type polynom")

#  Find the highest order derivative in LFD

if (inherits(Lfdobj, "Lfd")) {
    nderiv <- Lfdobj$nderiv
} else {
    stop("Second argument must be an integer or a functional data object")
}

#  Compute penalty matrix

bwtlist <- Lfdobj$bwtlist
isintLfd <- TRUE
if (nderiv > 0) {
	for (ideriv in 1:nderiv) {
		fdj <- bwtlist[[ideriv]]
		if (!is.null(fdj)) {
			if (any(fdj$coefs != 0)) {
				isintLfd <- FALSE
				break
			}
		}
	}
}

if (isintLfd) {
    nbasis   <- basisobj$nbasis
    rangex   <- basisobj$rangeval
    ctr      <- basisobj$params[1]
    basismat <- getbasismatrix(rangex, basisobj, nderiv)
    penmatl  <- outer(basismat[1,],basismat[1,])*(rangex[1] - ctr)
    penmatu  <- outer(basismat[2,],basismat[2,])*(rangex[2] - ctr)
    penaltymatrix <- matrix(0,nbasis,nbasis)
    for (i in (nderiv+1):nbasis) for (j in (nderiv+1):i) {
      	penaltymatrix[i,j] <- (penmatu[i,j] - penmatl[i,j])/(i + j - 2*nderiv - 1)
      	penaltymatrix[j,i] <- penaltymatrix[i,j]
    }
} else {
    penaltymatrix <- inprod(basisobj, basisobj, Lfdobj, Lfdobj)
}

  return( penaltymatrix )
}
polynom <- function (x, norder=1, ctr=midrange, nderiv)
{
#  This computes values of the polynomials,
#        P_l(x) = (x-ctr)^l, l=0,...,NORDER-1
#  or their derivatives.
#  The degree of the highest order polynomial is one less than NORDER.
#  The default is the constant function.

#  Arguments are as follows:
#  X      ... array of values at which the polynomials are to
#             evaluated
#  NORDER ... the polynomial basis object
#  NDERIV ... highest order derivative.  0 means only function values
#             are returned.
#  CTR    ... a constant to be subtracted from a value prior to taking
#             its power
#  Return is a matrix with length(X) rows and NORDER columns containing
#  the values of the polynomials

#  last modified 17 January 2006

  x        <- as.vector(x)
  n        <- length(x)
  ndegree  <- norder - 1
  if (nderiv > ndegree) stop('NDERIV exceeds highest degree of polynomial.')
  rangex   <- range(x)
  midrange <- mean(rangex)
  lfac <- 1
  if (nderiv > 1) for (l in 2:nderiv) lfac <- lfac*l
  polyval <- matrix(0,n,norder)
  polyval[,nderiv+1] <- lfac
  if (norder > nderiv+1)
    for (l in (nderiv+2):norder)
       polyval[,l] <- polyval[,l-1]*(x-ctr)*(l-1)/(l-nderiv-1)
  return (polyval)
}
#  --------------------------------------------------------------

polyprod <- function(Coeff1, Coeff2){
# POLYCONV computes products of polynomials defined by columns of 
#   coefficient matrices Coeff1 and Coeff2

#  Last modified 6 June 2005

polyorder1 <- dim(Coeff1)[1] 
norder1    <- dim(Coeff1)[2] 
polyorder2 <- dim(Coeff2)[1] 
norder2    <- dim(Coeff2)[2] 
ndegree1 <- polyorder1 - 1 
ndegree2 <- polyorder2 - 1 

#  if the degrees are not equal, pad out the smaller matrix with 0s

if(ndegree1 != ndegree2){
    if(ndegree1 > ndegree2)  Coeff2 <- rbind(Coeff2, matrix(0,ndegree1-ndegree2,norder2))
    else                     Coeff1 <- rbind(Coeff1, matrix(0,ndegree2-ndegree1,norder1))
}

#  find order of the product
D <- max(c(ndegree1,ndegree2))   # maximum degree
N <- 2*D+1                       # order of product

#  compute the coefficients for the products
convmat <- array(0,c(norder1,norder2,N)) 
for (i in 0:(D-1)){
    ind <- c(0:i) + 1 
    if (length(ind) == 1) {
        convmat[,,i+1] <-     outer(Coeff1[ind,    ],Coeff2[i-ind+2,]) 
        convmat[,,N-i] <-     outer(Coeff1[D-ind+2,],Coeff2[D-i+ind,])	
    } else {
        convmat[,,i+1] <- crossprod(Coeff1[ind,    ],Coeff2[i-ind+2,]) 
        convmat[,,N-i] <- crossprod(Coeff1[D-ind+2,],Coeff2[D-i+ind,])
    }
}
ind <- c(0:D) + 1 
convmat[,,D+1] <-     crossprod(Coeff1[ind,    ],Coeff2[D-ind+2,])

if (ndegree1 != ndegree2) {
	convmat <- convmat[,,1:(ndegree1+ndegree2+1)] 
	convmat <- array(convmat,c(norder1,norder2,ndegree1+ndegree2+1))
}

return(convmat)
}

powerbasis <- function(x, exponents, nderiv=0) {
#POWERBASIS computes values of monomials, or their derivatives.
#  The powers of X are the NBASIS nonnegative integers in EXPONENTS.
#  The default is 1, meaning X itself.
#  Arguments are as follows:
#  X         ... vector of values at which the polynomials are to
#                evaluated
#  EXPONENTS ... vector of exponents
#  NDERIV    ... order of derivative to be returned.
#  Return is:
#  A matrix with length(X) rows and NBASIS columns containing
#    the values of the monomials or their derivatives

#  last modified 13 December 2002

	x <- as.vector(x)
	n <- length(x)

	nbasis <- length(exponents)

	powermat <- matrix(0,n,nbasis)
	if (nderiv == 0) {
    	for (ibasis in 1:nbasis)
        	powermat[,ibasis] <- x^exponents[ibasis]
	} else {
    	if (any(exponents - nderiv < 0) && any(x == 0)) {
        	stop("A negative exponent is needed and an argument value is 0.")
    	} else {
        	for (ibasis in 1:nbasis) {
            	degree <- exponents[ibasis]
            	if (nderiv <= degree) {
                	fac <- degree
                	for (ideriv in 2:nderiv) {
                    	fac <- fac*(degree-ideriv+1)
                	}
                	powermat[,ibasis] <- fac*x^(degree-nderiv)
            	}
        	}
    	}
	}
	return(powermat)
}
powerpen <- function(basisobj, Lfdobj=int2Lfd(2)) {
#  POWERPEN  Computes the power basis penalty matrix.
#  Arguments:
#  BASISFD  ... a power basis object
#  Lfd      ... either the order of derivative or a
#               linear differential operator to be penalized.
#  Returns a list the first element of which is the basis matrix
#   and the second element of which is the diagonal of the penalty matrix.

#  Last modified:  2008.10.17 by Spencer Graves
#  Previously modified:  3 January 2008 by Jim Ramsay

#  check BASISOBJ

if (!inherits(basisobj, "basisfd")) stop(
    	"First argument is not a basis object.")

if (basisobj$type != "power") stop("BASISOBJ not of type POWER.")

rangeval  <- basisobj$rangeval
exponents <- basisobj$params

#  check LFDOBJ

Lfdobj <- int2Lfd(Lfdobj)

if (is.integerLfd(Lfdobj)) {

    #  case where LFDOBJ is integer

    nderiv <- Lfdobj$nderiv

    if (any(exponents - nderiv < 0) && rangeval[1] <= 0)
        	stop(paste("A negative exponent is needed and",
                    "an argument value can be nonpositive."))
    nbasis     <- basisobj$nbasis
    penaltymat <- matrix(0,nbasis,nbasis)
    xrange     <- basisobj$rangeval
    for (ibasis in 1:nbasis) {
        ideg <- exponents[ibasis]
        if (nderiv == 0) {
            ifac <- 1
        } else {
            ifac <- ideg
            if(nderiv>1) for (k in 2:nderiv) {
                if (ideg == k-1) ifac = 0
                else             ifac <- ifac*(ideg - k + 1)
            }
        }
        if (ibasis > 1) {
            for (jbasis in 1:(ibasis-1)) {
                jdeg <- exponents[jbasis]
        	    if (nderiv == 0) {
                    jfac <- 1
                } else {
                    jfac <- jdeg
                    if(nderiv>1) for (k in 2:nderiv) {
                        if (jdeg == k-1) jfac = 0
                        else             jfac <- jfac*(jdeg - k + 1)
                    }
                }
                if (ifac*jfac == 0) {
                    penaltymat[ibasis,jbasis] = 0
                } else {
                    penaltymat[ibasis,jbasis] <- ifac*jfac*
	                      (xrange[2]^(ideg+jdeg-2*nderiv+1) -
	                       xrange[1]^(ideg+jdeg-2*nderiv+1))/
                            (ideg + jdeg - 2*nderiv + 1)
                }
	          penaltymat[jbasis,ibasis] <- penaltymat[ibasis,jbasis]
            }
        }
        if (ifac == 0)
            penaltymat[ibasis,ibasis] <- 0
        if (2*ideg - 2*nderiv + 1 == 0)
            penaltymat[ibasis,ibasis] <- ifac^2*
                    (log(xrange[2]) - log(xrange[1]))
        if (ifac*(2*ideg - 2*nderiv + 1) != 0)
                penaltymat[ibasis,ibasis] <- ifac^2*
	              (xrange[2]^(2*ideg-2*nderiv+1) -
	               xrange[1]^(2*ideg-2*nderiv+1))/
                    (2*ideg - 2*nderiv + 1)
    }
} else {
    	penaltymat <- inprod(basisobj, basisobj, Lfdobj, Lfdobj)
}

penaltymat

}

ppBspline <- function(t)
{
#PPBSPngapINE computes the coefficients of the polynomials 
# of the piecewise polynomial B-spline of order norder=length(t)-1
# corresponding to the knot sequence T 
# (i.e. B(k,T)(x) = (T(1+k)-T(1))[T(1),T(2),...,T(k+1)](.-x)_{+}^{k-1}),
# the coefficients of the polynomials each defines on 
# distinct gapinvals of T. 
# Say there are ngap distinct gapinvals in 
# T [T(knots[1]),T(knots[2])), [T(knots[2]),T(knots[3])),...,
# T(knots[ngap]),T(knots[ngap+1])], 
# then the coefficients are returned in the matrix COEFF as row vectors, 
# the i-th row corresponding to the coefficients of the polynomial on the
# gapinval [T(knots[i]),T(knots[i+1])), 
# such that, for x in [T(knots[i]),T(knots[i+1])), 
# B(k,T(i))(x) = 
#   COEFF(i,1)*(x-T(i))^{k-1} + ... + COEFF(i,k-1)*(x-T(i)) + COEFF(i,k). 
# Note that we assume T(1) < T(k+1), i.e. T is not a
# sequence of the same knot. 
# The function returns the ngap times k matrix COEFF and the vector
# INDEX indicating where in the sequence T we find the knots. 
# This code uses part of the code from the function bspline.m.

#  Argument:  t ... a row vector of NORDER + 1 knot values,
#  where NORDER is the order of the spline.

# Last modified 15 May 2005

norder <- length(t) - 1
ncoef  <- 2*(norder-1)
if (norder > 1) {
   adds  <- rep(1,norder-1)
   tt    <- c(adds*t[1], t, adds*t[norder+1])
   gapin <- (1:(length(tt)-1))[diff(tt) > 0]
   ngap  <- length(gapin)
   iseq  <- (2-norder):(norder-1)
   ind   <- outer(rep(1,ngap),iseq) + outer(gapin,rep(1,ncoef))
   tx    <- matrix(tt[c(ind)],ngap,ncoef)
   ty    <- tx - outer(tt[gapin],rep(1,ncoef))
   b     <- outer(rep(1,ngap),(1-norder):0)  + outer(gapin,rep(1,norder))
   a     <- c(adds*0, 1, adds*0)   
   d     <- matrix(a[c(b)],ngap,norder)
   for (j in 1:(norder-1)) {
      for (i in 1:(norder-j)) {
	      ind1 <- i + norder - 1;
	      ind2 <- i + j      - 1;
         d[,i] <-(ty[,ind1]*d[,i] - ty[,ind2]*d[,i+1])/  
                 (ty[,ind1]       - ty[,ind2])
      }
   }
   Coeff <- d
   for (j in 2:norder) {
      factor <- (norder-j+1)/(j-1)
      ind <- seq(norder,j,-1)
      for (i in ind) 
         Coeff[,i] <- factor*(Coeff[,i] - Coeff[,i-1])/ty[,i+norder-j]
   }
   ind   <- seq(norder,1,-1)
   if (ngap > 1) Coeff <- Coeff[,ind] else Coeff <- matrix(Coeff[,ind],1,norder)
   index <- gapin - (norder-1)
} else {
   Coeff <- as.matrix(1)
   index <- as.matrix(1)
}

list(Coeff,index)

}
#  ---------------------------------------------------------------------

ppderiv <- function(Coeff, Deriv=0){
#PPDERIV computes the DERIV-th derivatives of the polynomials 
# with coefficients COEFF such that the i-th polynomial is
# COEFF(i,1)*x^(k-1) + COEFF(i,2)*x^(k-2) + ... + COEFF(i,k)
# It returns a matrix COEFFD with the same number of rows as COEFF, 
# but with k-DERIV columns such that the DERIV-th derivative 
# of the i-th polynomial is expressed as
# COEFFD(i,1)*x^(k-1-DERIV) + COEFFD(i,k-DERIV-1)*x + COEFFD(i,k-DERIV),
# Note that if k-DERIV < 1, then COEFFD is the zero vector,
# and if DERIV < 1 we are not differentiating.

m <- dim(Coeff)[1]  # k is the order of the polynomials.
k <- dim(Coeff)[2]  

# If DERIV is not a positive integer, we are not differentiating.
if (Deriv < 1){
    CoeffD <- as.matrix(Coeff)  
    return(CoeffD)
}

# Compute the coefficient of the DERIV-th derivative of the function

if((k-Deriv) < 1){
    CoeffD <- matrix(0,m,1)  # The derivative is zero everywhere
    return(CoeffD)
    }
else{
    # initialize COEFFD with the coefficients from COEFF we will need
    CoeffD <- Coeff[,1:(k-Deriv)]
    if (!is.matrix(CoeffD)) CoeffD <- t(as.matrix(CoeffD))
    for (j in 1:(k-2)){
        bound1 <- max(1,j-Deriv+1) 
        bound2 <- min(j,k-Deriv) 
        CoeffD[,bound1:bound2] <- (k-j)*CoeffD[,bound1:bound2] 
    }
    return(CoeffD)
}    
}


predict.fRegress <- function (object, newdata = NULL, se.fit = FALSE, 
          interval = c("none", "confidence", "prediction"), level = 0.95, ...) 
{
  
  #  Last modfied by Jim Ramsay 10 August 2020
  
  #  compute predicted values 
  
  yhatfd <- object$yhatfdobj
  if (is.null(newdata)) {
    pred <- yhatfd
  } else {
    betaestlist <- object$betaestlist
    p <- length(betaestlist)
    for (j in 1:p) {
      if (inherits(betaestlist[[j]], "fdPar")) 
        betaestlist[[j]] <- betaestlist[[j]]$fd
    }
    Nnew <- dim(newdata[[1]]$coefs)[2]
    if (inherits(yhatfd, "fd") || inherits(yhatfd, "fdpar")) {
      for (j in 1:p) {
        xi <- newdata[[j]]
        bi <- betaestlist[[j]]
        if (j == 1) {
          pred <- bi * xi
        }
        else {
          pred <- pred + bi * xi
        }
      }
    }
    else {
      for (j in 1:p) {
        xi <- newdata[[j]]
        bi <- betaestlist[[j]]
        if (j == 1) {
          pred <- inprod(xi, bi)
        }
        else {
          pred <- pred + inprod(xi, bi)
        }
      }
    }
  }
  
  #  check that standard errors of predicted values are required
  
  int <- match.arg(interval)
  need.se <- (se.fit || (int != "none"))
  if (!need.se) {
    return(pred)
  }
  else {
    
    #  compute variance-covariance matrix over plotting grid
    
    Bvar = object$Bvar
    if (is.null(Bvar)) 
      stop(paste("Standard error for predict object cannot be computed", 
                 " without preliminary use of function fRegress.stderr()."))
    ncoef <- 0
    for (j in 1:p) {
      betafdj <- betaestlist[[j]]
      ncoefj <- betafdj$basis$nbasis
      ncoef <- ncoef + ncoefj
    }
    
    if (inherits(yhatfd, "fdPar") || inherits(yhatfd, "fd")) {

      #  functional dependent variable case
      
      nplot <- 101
      rangeval <- yhatfd$basis$rangeval
      tplot <- seq(rangeval[1], rangeval[2], len = nplot)
      YhatStderr <- matrix(0, nplot, N)
      B2YhatList <- vector("list", p)
      for (iplot in 1:nplot) {
        YhatVari <- matrix(0, N, N)
        tval <- tplot[iplot]
        for (j in 1:p) {
          Zmat <- eval.fd(tval, newdata[[j]])
          betabasisj <- betaestlist[[j]]$basis
          PsiMatj <- eval.basis(tval, betabasisj)
          B2YhatMapij <- t(Zmat) %*% PsiMatj
          B2YhatList[[j]] <- B2YhatMapij
        }
        m2j <- 0
        for (j in 1:p) {
          m1j <- m2j + 1
          m2j <- m2j + betaestlist[[j]]$basis$nbasis
          B2YhatMapij <- B2YhatList[[j]]
          m2k <- 0
          for (k in 1:p) {
            m1k <- m2k + 1
            m2k <- m2k + betaestlist[[k]]$basis$nbasis
            B2YhatMapik <- B2YhatList[[k]]
            YhatVari <- YhatVari + B2YhatMapij %*% Bvar[m1j:m2j, 
                                                        m1k:m2k] %*% t(B2YhatMapik)
          }
        }
        YhatStderr[iplot, ] <- matrix(sqrt(diag(YhatVari)), 
                                      1, N)
      }
    }
    else {
      
      #  scale dependent variable case
      
      ymat <- as.matrix(yhatfd)
      N <- dim(ymat)[1]
      B2YhatList <- vector("list", p)
      YhatVari <- matrix(0, N, N)
      for (j in 1:p) {
        betabasisj <- betaestlist[[j]]$basis
        Xfdj <- newdata[[j]]
        B2YhatMapij <- inprod(Xfdj, betabasisj)
        B2YhatList[[j]] <- B2YhatMapij
      }
      m2j <- 0
      for (j in 1:p) {
        m1j <- m2j + 1
        m2j <- m2j + betaestlist[[j]]$basis$nbasis
        B2YhatMapij <- B2YhatList[[j]]
        m2k <- 0
        for (k in 1:p) {
          m1k <- m2k + 1
          m2k <- m2k + betaestlist[[k]]$basis$nbasis
          B2YhatMapik <- B2YhatList[[k]]
          YhatVari <- YhatVari + B2YhatMapij %*% Bvar[m1j:m2j, 
                                                      m1k:m2k] %*% t(B2YhatMapik)
        }
      }
      YhatStderr <- matrix(sqrt(diag(YhatVari)), N, 1)
    }
    return(list(pred = pred, YhatStderr = YhatStderr))
  }
}
project.basis <- function(y, argvals, basisobj, penalize=FALSE)
{
  #  Arguments for this function:
  #
  #  Y        ... an array containing values of curves
  #               If the array is a matrix, rows must correspond to argument
  #               values and columns to replications, and it will be assumed
  #               that there is only one variable per observation.
  #               If Y is a three-dimensional array, the first dimension
  #               corresponds to argument values, the second to replications,
  #               and the third to variables within replications.
  #               If Y is a vector, only one replicate and variable are assumed.
  #  ARGVALS  ... A vector of argument values.  This must be of length
  #    length(Y) if Y is a vector or dim(Y)[1] otherwise.
  #  BASISOBJ ... A basis.fd object
  #  PENALIZE ... If TRUE, a penalty term is used to deal with a singular
  #               basis matrix.  But this is not normally needed.
  #
  #  Returns a coefficient vector or array. The first dimension is the number
  #     of basis functions and the other dimensions (if any) match
  #  the other dimensions of Y.
  #
  
  #  Last modified 16 January 2020  by Jim Ramsay
  
  #  Check BASISOBJ
  
  if (!inherits(basisobj, "basisfd")) stop(
    "Third argument BASISOBJ is not a basis object.")
  
  #
  #  Calculate the basis and penalty matrices, using the default
  #   for the number of derivatives in the penalty.
  #
  basismat <- getbasismatrix(argvals, basisobj, 0)
  Bmat     <- crossprod(basismat)
  if (penalize) {
    penmat <- eval.penalty(basisobj)
    #
    #  Add a very small multiple of the identity to penmat
    #   and find a regularization parameter
    #
    penmat <- penmat + 1e-10 * max(penmat) * diag(dim(penmat)[1])
    lambda <- (0.0001 * sum(diag(Bmat)))/sum(diag(penmat))
    Cmat <- Bmat + lambda * penmat
  } else {
    Cmat <- Bmat
  }
  #
  #  Do the fitting by a simple solution of the
  #    equations taking into account smoothing
  #
  if (is.array(y) == FALSE) y <- as.array(y)
  if(length(dim(y)) <= 2) {
    Dmat <- crossprod(basismat, y)
    coef <- symsolve(Cmat, Dmat)
  } else {
    nvar <- dim(y)[3]
    coef <- array(0, c(basisobj$nbasis, dim(y)[2], nvar))
    for(ivar in 1:nvar) {
      Dmat <- crossprod(basismat, y[,  , ivar])
      coef[,  , ivar] <- symsolve(Cmat, Dmat)
    }
  }
  coef
}
quadset <- function(nquad=5, basisobj=NULL, breaks){

# last modified 8 May 2012 by Jim Ramsay

##
## 1.  Check nquad
##
  {
    if(nquad<5){
      warning("nquad must be at least 5;  increase to this minimum.")
      nquad <- 5
    }
    else {
      if((nquad%%2)!=1){
        warning("nquad must be an odd integer;  increased to enforce this.")
        nquad <- 1+2*floor(nquad/2)
      }
    }
  }
##
## 2.  check basisobj
##
  if(!is.null(basisobj) && !is.basis(basisobj))
    stop('basisobj is not a basis object.')
##
## 3.  check breaks
##
  if(missing(breaks) || length(breaks) == 0) {
    if(is.null(basisobj) || !is.basis(basisobj))
      stop("Either 'breaks' or 'basisobj' must be provided.")
#
    type <- basisobj$type
    if(type != 'bspline')
      stop(
        "'breaks' not supplied and 'basisobj' is not a spline basis.")
#
    rangeval <- basisobj$rangeval
    params   <- basisobj$params
    knots    <- c(rangeval[1], params, rangeval[2])
    breaks   <- unique(knots)
  }
##
## 4.  quadpts and quadwts
##
  nbreaks = length(breaks);
#
  db <- diff(breaks)
  nquad1 <- nquad-1
  nbreaks1 <- nbreaks-1
# 4.1.  First create quadpts as a matrix
  quadpts. <- array(NA, dim=c(nbreaks1, nquad) )
  quadpts.[, 1] <- breaks[-nbreaks]
  db. <- db/nquad1
  for(i in 2:nquad)
    quadpts.[, i] <- (quadpts.[, i-1]+db.)
# 4.2.  Now convert quadpts matrix to the desired vector
  quadpts <- as.vector(t(quadpts.))
# 4.3.  Similarly first create quadwts as a matrix
  quadwts. <- outer(c(1, 4, rep(c(2, 4), (nquad1-2)/2), 1),
                   db/(nquad1*3) )
# 4.4.  Now convert quadwts matrix to the desired vector
  quadwts <- as.vector(quadwts.)
  quadvals <- cbind(quadpts=quadpts, quadwts=quadwts)

  if(is.null(basisobj)) return(quadvals)
#
  basisobj$quadvals <- quadvals
  values <- vector("list", 2)
  for( ivalue in 1:2){
    values[[ivalue]] <- eval.basis(quadpts, basisobj, ivalue-1)
  }
  basisobj$values <- values
  basisobj
}

rangechk <- function(rangeval)
{
#  check a range vector argument

#  last modified 29 September 2008 by Jim Ramsay

  nrangeval = length(rangeval)
  OK <- TRUE
  if (!is.numeric(rangeval))          OK <- FALSE
  if (!is.vector(rangeval))           OK <- FALSE
  if (nrangeval < 1 || nrangeval > 2) OK <- FALSE
  if (rangeval[1] >= rangeval[2])     OK <- FALSE
  return(OK)
}
reconsCurves <- function(data, PC){
  #Reconstruct data curves using functional principal components
  #
  # Arguments:
  #
  # DATA ...... an set of values of curves at discrete sampling points or 
  #             argument values. If the set is supplied as a matrix object, 
  #             the rows must correspond to argument values and columns to 
  #             replications, and it will be assumed that there is only one 
  #             variable per observation. If data is a three-dimensional array, 
  #             the first dimension corresponds to argument values, the second 
  #             to replications, and the third to variables within replications.
  # PC ......   an object of class "pca.fd"
  #
  # Returns a functional data object (i.e., having class "fd")
  ndim = length(dim(data))
  
  if(ndim == 3){
    nvar = dim(data)[3]
    nbasish = PC$harmonics$basis$nbasis
    datarecon = 0
    data.coefs = array(NA, dim = c(nbasish,ncol(data),nvar))
    for(i in 1:ncol(PC$harmonics$coefs)){
      for(l in 1:nvar){
        data.coefs[,,l] = t(replicate(nbasish,PC$scores[,i,l]))*PC$harmonics$coefs[,i,l]
      }
      datarecon = datarecon + fd(data.coefs,PC$harmonics$basis)
    }
    
    
    reconlist = list()
    for(p in 1:nvar){
      meanfdaux = PC$meanfd
      meanfdaux$coefs = replicate(ncol(data),PC$meanfd$coefs[,,p])
      reconaux = datarecon
      reconaux$coefs = as.matrix(datarecon$coefs[,,p])
      reconlist[[p]] = meanfdaux + reconaux
    }
    
    finalcoef = array(NA, dim = c(nrow(reconlist[[1]]$coefs), ncol(data),nvar))
    for(n in 1:nvar){
      finalcoef[,,n] = reconlist[[n]]$coefs
    }
    
    data.recons = reconlist[[1]]
    data.recons$coefs = finalcoef
    data.recons$fdnames$values = data.recons$fdnames$reps
    
  }else{
    datarecon = 0
    for(i in 1:ncol(PC$harmonics$coefs)){
      coefs = t(replicate(PC$harmonics$basis$nbasis,PC$scores[,i]))*PC$harmonics$coefs[,i]
      if(dim(data)[2]==1) 
      coefs = replicate(PC$harmonics$basis$nbasis,PC$scores[,i])*PC$harmonics$coefs[,i]
      datarecon = datarecon + fd(coefs,PC$harmonics$basis)
    }
    PC$meanfd$coefs = replicate(ncol(datarecon$coefs),as.vector(PC$meanfd$coefs))
    data.recons = PC$meanfd+datarecon
    data.recons$fdnames$values = "values"
  }
  
  
  return(data.recons)
}
register.fd <- function(y0fd=NULL, yfd=NULL, WfdParobj=NULL,
                    conv=1e-4, iterlim=20, dbglev=1, periodic=FALSE, crit=2)
{
#REGISTERFD registers a set of curves YFD to a target function Y0FD.
#  Arguments are:
#  Y0FD      ... Functional data object for target function.  It may be either
#                a single curve, or have the same dimensions as YFD.
#  YFD       ... Functional data object for functions to be registered
#  WFDPAROBJ ... Functional parameter object for function W defining warping
#                functions. The basis that is defined in WFDPAROBJ must be a
#                B-spline basis, and the number of basis functions must be at
#                least 2.
#                The coefficients, which supplied either in a functional
#                data object or as defaults to zero if a basis object is
#                supplied in the definition of WFDPAROBJ are used as the
#                initial values in the iterative computation of the final
#                warping functions.
#                NB:  The value of the first coefficient is NOT used.
#                This is because a warping function is normalized, and when
#                this happens, the impact of the first coefficient, if used,
#                would be eliminated.  This first position is used, however, to
#                contain the shift parameter in case the data are to be
#                treated as periodic.  At the end of the calculations,
#                the shift parameter is returned separately.
#                If WFDPAROBJ is not supplied, it defaults to a bspline
#                basis of order 2 with 2 basis functions.  This is equivalent
#                to using a linear function for W.
#  CONV    ... Convergence criterion
#  ITERLIM ... iteration limit for scoring iterations
#  DBGLEV  ... Level of output of computation history
#  PERIODIC... If one, curves are periodic and a shift parameter is fit.
#              Initial value for shift parameter is taken to be 0.
#              The periodic option should ONLY be used with a Fourier
#              basis for the target function Y0FD, the functions to be
#              registered, YFD.
#  CRIT    ... If 1 least squares, if 2 log eigenvalue ratio.  Default is 1.
#  Returns:

#  REGLIST ...  A list with fields:
#    REGLIST$REGFD  ... A functional data object for the registered curves
#    REGLIST$WARPFD ... A Functional data object for warping functions h
#    REGLIST$WFD    ... A Functional data object for functions W defining
#                         warping fns
#    REGLIST$SHIFT  ... Shift parameter value if curves are periodic
#    REGLIST$Y0FD   ... Argument Y0FD
#    REGLIST$YFD    ... Argument YFD

#  Last modified 16 November 2021 by Jim Ramsay

##
## 1.  Check y0fd and yfd
##

#  if YFD is not supplied, and therefore defaults to NULL,  it is
#  assumed that the first argument is to be taken as YFD, and that
#  the target function Y0FD defaults to NULL.  In this event,  Y0FD
#  is set up as the mean of the functions in YFD.

  if(is.null(yfd)){
    yfd  <- y0fd
    y0fd <- NULL
  }

#  Check that YFD is a functional data object

  if (!(inherits(yfd, "fd")))
      stop("'yfd' must be a functional data object.  ",
           "Instead, class(yfd) = ", class(yfd))

#  If target Y0FD is not supplied, and therefore defaults to NULL, replace
#  it be the mean of the functions in YFD

  if(is.null(y0fd)) {
    y0fd <- mean.fd(yfd)
  } else {
    if (!(inherits(y0fd, "fd")))
        stop("First argument is not a functional data object.",
             "Instead, class(y0fd) = ", class(y0fd))
  }

#  get dimensions of functions in YFD to be registered

  ycoefs <- yfd$coefs
  if (is.vector(ycoefs)) ycoefs = as.matrix(ycoefs)
  ydim   <- dim(ycoefs)
  ncurve <- ydim[2]
  ndimy  <- length(ydim)
  if (ndimy == 3) {
      nvar <- ydim[3]
  } else {
      nvar <- 1
  }
  if (ndimy > 3) stop("'yfd' is more than 3-dimensional.")

#  Extract basis information from YFD

  ybasis  <- yfd$basis
  ynbasis <- ybasis$nbasis
  yrange  <- ybasis$rangeval
  if (periodic && !(ybasis$type == "fourier"))
    stop("'periodic' is TRUE but 'type' is not 'fourier'; ",
         "periodic B-splines are not currently part of 'fda'")

#  Get dimensions of target function object Y0FD

  y0coefs0 <- y0fd$coefs
  if (is.vector(y0coefs0)) y0coefs0 = as.matrix(y0coefs0)
  y0dim0   <- dim(y0coefs0)
  ndimy00  <- length(y0dim0)
  if (ndimy00 > ndimy) stop("Y0FD has more dimensions than YFD")
  #  Determine whether the target function is full or not
  if (y0dim0[2] == 1) {
      fulltarg <- FALSE
  } else {
      if (y0dim0[2] == ydim[2]) {
          fulltarg <- TRUE
      } else {
          stop(paste("Second dimension of coefficient matrix for Y0FD",
                     "is neither 1 nor equal to the number of functions",
                     "to be registered."))
      }
  }
  if (ndimy00 == 3 && ydim[3] != y0dim0[3]) stop(
      "Third dimension of YOFD does not match that of YFD.")

#  Extract basis information from Y0FD

  y0basis  <- y0fd$basis
  y0nbasis <- y0basis$nbasis
  y0range  <- y0basis$rangeval
  #  check that target range matches function range
  if (!all(y0range == yrange)) stop(
      "Range for Y0FD does not match range for YFD.")

##
## 2.  Check WfdParobj
##

  if (is.null(WfdParobj)) {
      #  default WfdParobj to a B-spline basis of order 2 with 2 basis functions
      wbasis    <- create.bspline.basis(yrange,2,2)
      Wfd0      <- fd(matrix(0,2,ncurve),wbasis)
      WfdParobj <- fdPar(Wfd0)
  }
  WfdParobj <- fdParcheck(WfdParobj, ncurve)
  Wfd0      <- WfdParobj$fd
  wcoef     <- Wfd0$coefs
  if (is.vector(wcoef)) wcoef <- as.matrix(wcoef)
  wbasis <- Wfd0$basis
  wtype  <- wbasis$type
  if (wtype != "bspline") stop("Basis for Wfd is not a B-spline basis.")
  wnbasis <- wbasis$nbasis
  norder  <- wnbasis - length(wbasis$params)
  if (wnbasis < 2) stop(
      "At least two basis functions for W are required.")
  if (norder < 2) stop(
      "The order of the basis functions for W must be at least 2.")
  wtype  <- wbasis$type
  rangex <- wbasis$rangeval
  wdim   <- dim(wcoef)
  if (length(wdim) > 2) stop("WFDPAROBJ contains a multivariate function.")
  if (wdim[2] == 1) {
      wcoef      <- wcoef %*% matrix(1,1,ncurve)
      Wfd0$coefs <- wcoef
  } else {
      if (wdim[2] != ncurve) stop(
          "WFDPAROBJ and YFD containing differing numbers of functions.")
  }


##
## 3.  Do the work
##

#  set up a fine mesh of argument values

NFINEMIN <- 201
nfine <- 10*ynbasis + 1
if (nfine < NFINEMIN) nfine <- NFINEMIN
xlo   <- rangex[1]
xhi   <- rangex[2]
width <- xhi - xlo
xfine <- seq(xlo, xhi, len=nfine)

#  set up indices of coefficients that will be modified in ACTIVE

wcoef1   <- wcoef[1,]
if (periodic) {
   active   <- 1:wnbasis
   wcoef[1] <- 0
   shift    <- 0
} else {
   active <- 2:wnbasis
}

#  initialize matrix Kmat defining penalty term

lambda <- WfdParobj$lambda
if (lambda > 0) {
   Lfdobj <- WfdParobj$Lfd
   Kmat <- getbasispenalty(wbasis, Lfdobj)
   ind  <- 2:wnbasis
   Kmat <- lambda*Kmat[ind,ind]
} else {
   Kmat <- NULL
}

#  set up limits on coefficient sizes

climit <- 50*c(-rep(1,wnbasis), rep(1,wnbasis))

#  set up cell for storing basis function values

JMAX <- 15
basislist <- vector("list", JMAX)

yregcoef <- yfd$coefs

#  loop through the curves

wcoefnew <- wcoef
if (dbglev == 0 && ncurve > 1) cat("Progress:  Each dot is a curve\n")

for (icurve in 1:ncurve) {
  if (dbglev == 0 && ncurve > 1) cat(".")
  if (dbglev >= 1 && ncurve > 1)
      cat(paste("\n\n-------  Curve ",icurve,"  --------\n"))
  if (ncurve == 1) {
    yfdi  <- yfd
    y0fdi <- y0fd
    Wfdi  <- Wfd0
    cvec  <- wcoef
  } else {
    Wfdi <- Wfd0[icurve]
    cvec <- wcoef[,icurve]
    if (nvar == 1) {
      yfdi <- yfd[icurve]
    } else {
      yfdi <- yfd[icurve]
      yfdi$coef <- array(yfdi$coef,c(ynbasis,1,nvar))
    }
    if (fulltarg) {
      if (nvar == 1) {
        y0fdi <- y0fd[icurve]
      } else {
        y0fdi <- y0fd[icurve,]
      }
    } else {
      y0fdi <- y0fd
    }
  }

  #  evaluate curve to be registered at fine mesh

  yfine  <- matrix(eval.fd(xfine, yfdi, 0),nfine,nvar)

  #  evaluate target curve at fine mesh

  y0fine <- matrix(eval.fd(xfine, y0fdi, 0),nfine,nvar)

  #  evaluate objective function for starting coefficients

  #  first evaluate warping function and its derivative at fine mesh

  ffine  <-   monfn(xfine, Wfdi, basislist)
  Dffine <- mongrad(xfine, Wfdi, basislist)
  fmax   <- ffine[nfine]
  Dfmax  <- Dffine[nfine,]
  hfine  <- xlo + width*ffine/fmax
  Dhfine <- width*(fmax*Dffine - outer(ffine,Dfmax))/fmax^2
  hfine[1]     <- xlo
  hfine[nfine] <- xhi

  #  register curves given current Wfdi

  yregfdi <- regyfn(xfine, yfine, hfine, yfdi, Wfdi, periodic)

  #  compute initial criterion value and gradient

  Flist <- regfngrad(xfine, y0fine, Dhfine, yregfdi, Wfdi,
                     Kmat, periodic, crit)

  #  compute the initial expected Hessian

  if (crit == 2) {
     D2hwrtc <- monhess(xfine, Wfdi, basislist)
     D2fmax  <- D2hwrtc[nfine,]
     fmax2 <- fmax*fmax
     fmax3 <- fmax*fmax2
     m <- 1
     if (wnbasis > 1) {
        for (j in 2:wnbasis) {
           m <- m + 1
           for (k in 2:j) {
              m <- m + 1
              D2hwrtc[,m] <- width*(2*ffine*Dfmax[j]*Dfmax[k]
                   - fmax*(Dffine[,j]*Dfmax[k] + Dffine[,k]*Dfmax[j])
                   + fmax2*D2hwrtc[,m] - ffine*fmax*D2fmax[m])/fmax3
           }
        }
     }
  } else {
     D2hwrtc <- NULL
  }

  hessmat <- reghess(xfine, y0fine, Dhfine, D2hwrtc, yregfdi,
                     Kmat, periodic, crit)

  #  evaluate the initial update vector for correcting the initial cvec

  result   <- linesearch(Flist, hessmat, dbglev)
  deltac   <- result[[1]]
  cosangle <- result[[2]]
  #  initialize iteration status arrays

  iternum <- 0
  status <- c(iternum, Flist$f, Flist$norm)
  if (dbglev >= 1) {
        cat("\nIter.    Criterion   Grad Length")
        cat("\n")
        cat(iternum)
        cat("        ")
        cat(round(status[2],4))
        cat("      ")
        cat(round(status[3],4))
  }
  iterhist <- matrix(0,iterlim+1,length(status))
  iterhist[1,]  <- status
  if (iterlim == 0) break

  #  -------  Begin main iterations  -----------

  MAXSTEPITER <- 5
  MAXSTEP <- 100
  trial   <- 1
  reset   <- 0
  linemat <- matrix(0,3,5)
  cvecold <- cvec
  Foldlist <- Flist
  dbgwrd  <- dbglev >= 2
  #  ---------------  beginning of optimization loop  -----------
  for (iter in 1:iterlim) {
      iternum <- iternum + 1
      #  set logical parameters
      dblwrd <- rep(FALSE,2)
      limwrd <- rep(FALSE,2)
      ind <- 0
      ips <- 0
      #  compute slope
      linemat[2,1] <- sum(deltac*Foldlist$grad)
      #  normalize search direction vector
      sdg          <- sqrt(sum(deltac^2))
      deltac       <- deltac/sdg
      linemat[2,1] <- linemat[2,1]/sdg
      # initialize line search vectors
      linemat[,1:4] <- outer(c(0, linemat[2,1], Flist$f),rep(1,4))
      stepiter  <- 0
      if (dbglev >= 2) {
          cat("\n")
          cat(paste("                 ", stepiter, "  "))
          cat(format(round(t(linemat[,1]),4)))
      }
      #  return with stop condition if initial slope is nonnegative
      if (linemat[2,1] >= 0) {
        if (dbglev >= 2) cat("\nInitial slope nonnegative.")
        ind <- 3
        break
      }
      #  return successfully if initial slope is very small
      if (linemat[2,1] >= -min(c(1e-3,conv))) {
        if (dbglev >= 2) cat("\nInitial slope too small")
        ind <- 0
        break
      }
      #  first step set to trial
      linemat[1,5]  <- trial
      #  ------------  begin line search iteration loop  ----------
      cvecnew <- cvec
      Wfdnewi <- Wfdi
      for (stepiter in 1:MAXSTEPITER) {
        #  check the step size and modify if limits exceeded
        result <- stepchk(linemat[1,5], cvec, deltac, limwrd, ind,
                   climit, active, dbgwrd)
        linemat[1,5] <- result[[1]]
        ind          <- result[[2]]
        limwrd       <- result[[3]]
        if (ind == 1) break    # break of limit hit twice in a row
        if (linemat[1,5] <= 1e-7) {
           #  Current step size too small  terminate
           if (dbglev >= 2)
             cat("\nStepsize too small: ", round(linemat[1,5],4))
           break
        }
        #  update parameter vector
        cvecnew <- cvec + linemat[1,5]*deltac
        #  compute new function value and gradient
        Wfdnewi[[1]] <- cvecnew
        #  first evaluate warping function and its derivative at fine mesh
        cvectmp <- cvecnew
        cvectmp[1] <- 0
        Wfdtmpi <- Wfdnewi
        Wfdtmpi[[1]] <- cvectmp
        ffine  <-   monfn(xfine, Wfdtmpi, basislist)
        Dffine <- mongrad(xfine, Wfdtmpi, basislist)
        fmax   <- ffine[nfine]
        Dfmax  <- Dffine[nfine,]
        hfine  <- xlo + width*ffine/fmax
        Dhfine <- width*(fmax*Dffine - outer(ffine,Dfmax))/fmax^2
        hfine[1]     <- xlo
        hfine[nfine] <- xhi
        #  register curves given current Wfdi
        yregfdi <- regyfn(xfine, yfine, hfine, yfdi, Wfdnewi, periodic)
        Flist    <- regfngrad(xfine, y0fine, Dhfine, yregfdi, Wfdnewi,
                             Kmat, periodic, crit)
        linemat[3,5] <- Flist$f
        #  compute new directional derivative
        linemat[2,5] <- sum(deltac*Flist$grad)
        if (dbglev >= 2) {
          cat("\n")
          cat(paste("                 ", stepiter, "  "))
          cat(format(round(t(linemat[,5]),4)))
        }
        #  compute next line search step, also testing for convergence
        result  <- stepit(linemat, ips, dblwrd, MAXSTEP)
        linemat <- result[[1]]
        ips     <- result[[2]]
        ind     <- result[[3]]
        dblwrd  <- result[[4]]
        trial   <- linemat[1,5]
        #  ind == 0 implies convergence
        if (ind == 0 || ind == 5) break
     }
     #  ------------  end line search iteration loop  ----------
     cvec   <- cvecnew
     Wfdi   <- Wfdnewi
     #  test for function value made worse
     if (Flist$f > Foldlist$f) {
        #  Function value worse  warn and terminate
        ier <- 1
        if (dbglev >= 2) {
          cat("Criterion increased, terminating iterations.\n")
          cat(paste("\n",round(c(Foldlist$f, Flist$f),4)))
        }
        #  reset parameters and fit
        cvec   <- cvecold
        Wfdi[[1]] <- cvecold
        Flist   <- Foldlist
        deltac <- -Flist$grad
        if (dbglev > 2) {
          for (i in 1:wnbasis) cat(cvec[i])
          cat("\n")
        }
        if (reset == 1) {
           #  This is the second time in a row that this
           #     has happened   quit
           if (dbglev >= 2) cat("Reset twice, terminating.\n")
            break
        } else {
           reset <- 1
        }
     } else {
        #  function value has not increased,  check for convergence
        if (abs(Foldlist$f-Flist$f) < conv) {
           wcoef[,icurve]    <- cvec
           status <- c(iternum, Flist$f, Flist$norm)
           iterhist[iter+1,] <- status
           if (dbglev >= 1) {
              cat("\n")
              cat(iternum)
              cat("        ")
              cat(round(status[2],4))
              cat("      ")
              cat(round(status[3],4))
	        }
           break
        }
        #  update old parameter vectors and fit list
        cvecold <- cvec
        Foldlist <- Flist
        #  update the expected Hessian
        if (crit == 2) {
           cvectmp <- cvec
           cvectmp[1] <- 0
           Wfdtmpi[[1]] <- cvectmp
           D2hwrtc <- monhess(xfine, Wfdtmpi, basislist)
           D2fmax  <- D2hwrtc[nfine,]
           #  normalize 2nd derivative
           fmax2 <- fmax*fmax
           fmax3 <- fmax*fmax2
           m <- 1
           if (wnbasis > 1) {
              for (j in 2:wnbasis) {
                 m <- m + 1
                 for (k in 2:j) {
                    m <- m + 1
                    D2hwrtc[,m] <- width*(2*ffine*Dfmax[j]*Dfmax[k]
                   - fmax*(Dffine[,j]*Dfmax[k] + Dffine[,k]*Dfmax[j])
                   + fmax2*D2hwrtc[,m] - ffine*fmax*D2fmax[m])/fmax3
                 }
              }
           }
        } else {
           D2hwrtc <- NULL
        }
        hessmat <- reghess(xfine, y0fine, Dhfine, D2hwrtc, yregfdi,
                           Kmat, periodic, crit)
        #  update the line search direction vector
        result   <- linesearch(Flist, hessmat, dbglev)
        deltac   <- result[[1]]
        cosangle <- result[[2]]
        reset <- 0
     }
     status <- c(iternum, Flist$f, Flist$norm)
     iterhist[iter+1,] <- status
     if (dbglev >= 1) {
        cat("\n")
        cat(iternum)
        cat("        ")
        cat(round(status[2],4))
        cat("      ")
        cat(round(status[3],4))
     }
   }
  #  ---------------  end of optimization loop  -----------
  wcoef[,icurve] <- cvec
  if (nvar == 1) {
     yregcoef[,icurve]  <- yregfdi$coefs
  } else {
     yregcoef[,icurve,] <- yregfdi$coefs
  }
}

cat("\n")

#  --------------------   end of variable loop  -----------

#  create functional data objects for the registered curves

regfdnames <- yfd$fdnames
regfdnames[[3]] <- paste("Registered ",regfdnames[[3]])
ybasis  <- yfd$basis
regfd   <- fd(yregcoef, ybasis, regfdnames)

#  set up vector of time shifts

if (periodic) {
  shift <- c(wcoef[1,])
  wcoef[1,] <- wcoef1
} else {
  shift <- rep(0,ncurve)
}

#  functional data object for functions W(t)

Wfd <- fd(wcoef, wbasis)

#  functional data object for warping functions

warpmat = eval.monfd(xfine, Wfd)
warpmat = rangex[1] + (rangex[2]-rangex[1])*
           warpmat/outer(rep(1,nfine),warpmat[nfine,]) +
           outer(rep(1,nfine),shift)
if (wnbasis > 1) {
   warpfdobj  = smooth.basis(xfine, warpmat, wbasis)$fd
} else {
   wbasis    = create.monomial.basis(rangex, 2)
   warpfdobj = smooth.basis(xfine, warpmat, wbasis)$fd
}
warpfdnames       <- yfd$fdnames
warpfdnames[[3]]  <- paste("Warped",warpfdnames[[1]])
warpfdobj$fdnames <- warpfdnames

reglist <- list("regfd"=regfd, "warpfd"=warpfdobj, "Wfd"=Wfd,
                "shift"=shift, "y0fd"  =y0fd,      "yfd"=yfd)

return(reglist)
}

#  ----------------------------------------------------------------

regfngrad <- function(xfine, y0fine, Dhwrtc, yregfd, Wfd,
                      Kmat, periodic, crit=FALSE)
{
  y0dim <- dim(y0fine)
  if (length(y0dim) == 3) nvar <- y0dim[3] else nvar <- 1
  nfine <- length(xfine)
  cvec  <- Wfd$coefs
  ncvec <- length(cvec)
  onecoef <- matrix(1,1,ncvec)

  if (periodic) {
     Dhwrtc[,1] <- 1
  } else {
     Dhwrtc[,1] <- 0
  }
  yregmat  <- eval.fd(xfine, yregfd, 0)
  Dyregmat <- eval.fd(xfine, yregfd, 1)

  #  loop through variables computing function and gradient values

  Fval <- 0
  gvec <- matrix(0,ncvec,1)
  for (ivar in 1:nvar) {
    y0ivar  <-   y0fine[,ivar]
    ywrthi  <-  yregmat[,ivar]
    Dywrthi <- Dyregmat[,ivar]
    aa      <- mean(y0ivar^2)
    bb      <- mean(y0ivar*ywrthi)
    cc      <- mean(ywrthi^2)
    Dywrtc  <- (Dywrthi %*% onecoef)*Dhwrtc
    if (crit == 1) {
      res  <- y0ivar - ywrthi
      Fval <- Fval + aa - 2*bb + cc
      gvec <- gvec - 2*crossprod(Dywrtc, res)/nfine
    } else {
      ee   <- aa + cc
      ff   <- aa - cc
      dd   <- sqrt(ff^2 + 4*bb^2)
      Fval <- Fval + ee - dd
      Dbb  <- crossprod(Dywrtc, y0ivar)/nfine
      Dcc  <- 2.0 * crossprod(Dywrtc, ywrthi)/nfine
      Ddd  <- (4*bb*Dbb - ff*Dcc)/dd
      gvec <- gvec + (Dcc - Ddd)
    }
  }
  if (!is.null(Kmat)) {
     if (ncvec > 1) {
        ind   <- 2:ncvec
        ctemp <- cvec[ind,1]
        Kctmp <- Kmat%*%ctemp
        Fval  <- Fval + t(ctemp)%*%Kctmp
        gvec[ind] <- gvec[ind] + 2*Kctmp
     }
  }

#  set up FLIST list containing function value and gradient

  Flist      <- list(f=0, grad=rep(0,ncvec), norm=0)
  Flist$f    <- Fval
  Flist$grad <- gvec
  #  do not modify initial coefficient for B-spline and Fourier bases
  if (!periodic)  Flist$grad[1] <- 0
  Flist$norm <- sqrt(sum(Flist$grad^2))
  return(Flist)
}

#  ---------------------------------------------------------------

reghess <- function(xfine, y0fine, Dhfine, D2hwrtc, yregfd,
                    Kmat, periodic, crit=FALSE)
{
	#cat("\nreghess")
  y0dim <- dim(y0fine)
  if (length(y0dim) == 3) nvar <- y0dim[3] else nvar <- 1
  nfine   <- length(xfine)
  wnbasis   <- dim(Dhfine)[2]
  onecoef <- matrix(1,1,wnbasis)
  npair   <- wnbasis*(wnbasis+1)/2

  if (periodic) {
     Dhfine[,1] <- 1
  } else {
     Dhfine[,1] <- 0
  }
  yregmat  <- eval.fd(yregfd, xfine, 0)
  Dyregmat <- eval.fd(yregfd, xfine, 1)
  if (nvar > 1) {
	   y0fine   <- y0fine[,1,]
	   yregmat  <- yregmat[,1,]
	   Dyregmat <- Dyregmat[,1,]
  }

  if (crit == 2) {
     D2yregmat <- eval.fd(yregfd, xfine, 2)
     if (nvar > 1) D2yregmat <- D2yregmat[,1,]
     if (periodic) {
        D2hwrtc[,1] <- 0
        if (wnbasis > 1) {
           for (j in 2:wnbasis) {
              m <- j*(j-1)/2 + 1
              D2hwrtc[,m] <- Dhfine[,j]
           }
        }
     } else {
        D2hwrtc[,1] <- 1
        if (wnbasis > 1) {
           for (j in 2:wnbasis) {
              m <- j*(j-1)/2 + 1
              D2hwrtc[,m] <- 0
           }
        }
     }
  }

  hessvec <- matrix(0,npair,1)
  for (ivar in 1:nvar) {
    y0i        <-   y0fine[,ivar]
    yregmati   <-  yregmat[,ivar]
    Dyregmati  <- Dyregmat[,ivar]
    Dywrtc <- ((Dyregmati %*% onecoef)*Dhfine)
    if (crit == 1) {
      hessmat <-  2*crossprod(Dywrtc, Dywrtc)/nfine
      m <- 0
       for (j in 1:wnbasis) {
          for (k in 1:j) {
             m <- m + 1
             hessvec[m] <- hessvec[m] + hessmat[j,k]
          }
      }
    } else {
      D2yregmati <- D2yregmat[,ivar]
      aa     <- mean(y0i^2)
      bb     <- mean(y0i*yregmati)
      cc     <- mean(    yregmati^2)
      Dbb    <- crossprod(Dywrtc, y0i)/nfine
      Dcc    <- 2.0 * crossprod(Dywrtc, yregmati)/nfine
      D2bb   <- matrix(0,npair,1)
      D2cc   <- matrix(0,npair,1)
      crossprodmat <- matrix(0,nfine,npair)
      DyD2hmat     <- matrix(0,nfine,npair)
      m <- 0
      for (j in 1:wnbasis) {
        for (k in 1:j) {
          m <- m + 1
          crossprodmat[,m] <- Dhfine[,j]*Dhfine[,k]*D2yregmati
          DyD2hmat[,m] <- Dyregmati*D2hwrtc[,m]
          temp <- crossprodmat[,m] + DyD2hmat[,m]
          D2bb[m] <- mean(y0i*temp)
          D2cc[m] <- 2*mean(yregmati*temp +
                     Dyregmati^2*Dhfine[,j]*Dhfine[,k])
        }
      }
      ee     <- aa + cc
      ff     <- aa - cc
      ffsq   <- ff*ff
      dd     <- sqrt(ffsq + 4*bb*bb)
      ddsq   <- dd*dd
      ddcu   <- ddsq*dd
      m <- 0
      for (j in 1:wnbasis) {
        for (k in 1:j) {
          m <- m + 1
          hessvec[m] <- hessvec[m] + D2cc[m] -
            (4*Dbb[j]*Dbb[k] + 4*bb*D2bb[m] + Dcc[j]*Dcc[k] -
                   ff* D2cc[m])/dd +
            (4*bb*Dbb[j] - ff*Dcc[j])*(4*bb*Dbb[k] - ff*Dcc[k])/ddcu
        }
      }
    }
  }
  hessmat <- matrix(0,wnbasis,wnbasis)
  m <- 0
  for (j in 1:wnbasis) {
    for (k in 1:j) {
      m <- m + 1
      hessmat[j,k] <- hessvec[m]
      hessmat[k,j] <- hessvec[m]
    }
  }
  if (!is.null(Kmat)) {
     if (wnbasis > 1) {
        ind <- 2:wnbasis
        hessmat[ind,ind] <- hessmat[ind,ind] + 2*Kmat
     }
  }
  if (!periodic) {
     hessmat[1,]  <- 0
     hessmat[,1]  <- 0
     hessmat[1,1] <- 1
  }
  return(hessmat)
}

#  ----------------------------------------------------------------

regyfn <- function(xfine, yfine, hfine, yfd, Wfd, periodic)
{
	#cat("\nregyfn")
coef  <- Wfd$coefs
shift <- coef[1]
coef[1] <- 0
Wfd[[1]] <- coef

if (all(coef == 0)) {
   if (periodic) {
      if (shift == 0) {
         yregfd <- yfd
         return(yregfd)
      }
   } else {
      yregfd <- yfd
      return(yregfd)
   }
}

#  Estimate inverse of warping function at fine mesh of values
#  28 dec 000
#  It makes no real difference which
#     interpolation method is used here.
#  Linear is faster and sure to be monotone.
#  Using WARPSMTH added nothing useful, and was abandoned.
nfine       <- length(xfine)
hinv        <- approx(hfine, xfine, xfine)$y
hinv[1]     <- xfine[1]
hinv[nfine] <- xfine[nfine]

#  carry out shift if period and shift != 0
basis  <- yfd$basis
rangex <- basis$rangeval
ydim <- dim(yfine)
#if (length(ydim) == 3) yfine <- yfine[,1,]
if (periodic & shift != 0) yfine <- shifty(xfine, yfine, shift)
#  make FD object out of Y
ycoef  <- project.basis(yfine, hinv, basis, 1)
yregfd <- fd(ycoef, basis)
return(yregfd)
}

#  ----------------------------------------------------------------

linesearch <- function(Flist, hessmat, dbglev)
{
deltac   <- -solve(hessmat,Flist$grad)
cosangle <- -sum(Flist$grad*deltac)/sqrt(sum(Flist$grad^2)*sum(deltac^2))
if (dbglev >= 2) cat(paste("\nCos(angle) = ",round(cosangle,2)))
if (cosangle < 1e-7) {
   if (dbglev >=2) cat("\nangle negative")
   deltac <- -Flist$grad
}
return(list(deltac, cosangle))
}

#  ---------------------------------------------------------------------

shifty <- function(x, y, shift)
{
#SHIFTY estimates value of Y for periodic data for
#       X shifted by amount SHIFT.
#  It is assumed that X spans interval over which functionis periodic.
#  Last modified 6 February 2001

ydim <- dim(y)
if (is.null(ydim)) ydim <- 1
if (length(ydim) > 3) stop("Y has more than three dimensions")

if (shift == 0) {
   yshift <- y
   return(yshift)
}

n   <- ydim[1]
xlo <- min(x)
xhi <- max(x)
wid <- xhi - xlo
if (shift > 0) {
   while (shift > xhi)  shift <- shift - wid
   ind <- 2:n
   x2  <- c(x, x[ind]+wid)
   xshift <- x + shift
   if (length(ydim) == 1) {
	  y2 <- c(y, y[ind])
      yshift <- approx(x2, y2, xshift)$y
   }
   if (length(ydim) == 2) {
	   nvar <- ydim[2]
	   yshift <- matrix(0,n,nvar)
      for (ivar in 1:nvar) {
         y2 <- c(y[,ivar], y[ind,ivar])
         yshift[,ivar] <- approx(x2, y2, xshift)$y
      }
   }
   if (length(ydim) == 3) {
	   nrep <- ydim[2]
	   nvar <- ydim[3]
      yshift <- array(0,c(n,nrep,nvar))
      for (irep in 1:nrep) for (ivar in 1:nvar) {
         y2 <- c(y[,irep,ivar], y[ind,irep,ivar])
         yshift[,irep,ivar] <- approx(x2, y2, xshift)$y
      }
   }
} else {
   while (shift < xlo - wid) shift <- shift + wid
   ind <- 1:(n-1)
   x2 <- c(x[ind]-wid, x)
   xshift <- x + shift
   if (length(ydim) == 1) {
      y2 <- c(y[ind], y)
      yshift <- approx(x2, y2, xshift)$y
   }
   if (length(ydim) == 2) {
	   nvar <- ydim[2]
	   yshift <- matrix(0,n,nvar)
	   for (ivar in 1:nvar) {
		   y2 <- c(y[ind,ivar],y[,ivar])
		   yshift[,ivar] <- approx(x2, y2, xshift)$y
	   }
   }
   if (length(ydim) == 3) {
	   nrep <- ydim[2]
	   nvar <- ydim[3]
      yshift <- array(0, c(n,nrep,nvar))
      for (irep in 1:nrep) for (ivar in 1:nvar) {
         y2 <- c(y[ind,irep,ivar], y[,irep,ivar])
         yshift[,irep,ivar] <- approx(x2, y2, xshift)$y
      }
   }
}
return(yshift)
}

register.newfd <- function(yfd, Wfd,type=c('direct','monotone','periodic'))
{
coef  <- Wfd$coefs
shift <- coef[1]

if (all(coef == 0)) {
   if (type=='periodic') {
      if (shift == 0) {
         yregfd <- yfd
         return(yregfd)
      }
   } else {
      yregfd <- yfd
      return(yregfd)
   }
}

# Now evaluate on a fine grid:

Wrange = Wfd$basis$range
ybasis = yfd$basis
yrange = ybasis$range

if( type=='periodic' & any(Wrange != yrange) )
  stop('Registration functions and functions to be registered must have the same range')


neval    <- max(10*ybasis$nbasis + 1,101)
tfine  <- seq(yrange[1],yrange[2], length=neval)  

# Shift registration is easy

if( type=='periodic' ){ 
  yfine = eval.fd(tfine,yfd)
  yfine <- shifty(tfine,yfine,shift)
  ycoef  <- project.basis(yfine, tfine, ybasis, 1)
  yregfd <- fd(ycoef, ybasis)
  return(yregfd)
}

# On the other hand, if we have warping functions:

if( type=='direct') xfine = eval.fd(tfine,Wfd)

if( type=='monotone'){ 
  xfine = eval.monfd(tfine,Wfd)
  xfine = xfine%*%diag(1/xfine[neval,])*(Wrange[2]-Wrange[1])+Wrange[1]
}
xfine = xfine*( xfine>yrange[1] & xfine< yrange[2]) + yrange[2]*(xfine>=yrange[2]) + yrange[1]*(xfine<=yrange[1])
yfine = eval.fd(tfine,yfd)

xdim = dim(xfine)
ydim = dim(yfine)

# Check that we have the right dimensions, we can register multiple y dimensions
# to one warping function, but we must have as many warping functions as there
# are y replicates

if( xdim[2] != ydim[2] ) stop('There must be as many warping function replicates as y replicates')
if( length(ydim) == 3 & length(xdim)==2 ) xfine = array(xfine,ydim)


# Now do the registration

ycoef = 0*yfd$coef
cdim = dim(ycoef)

if(length(cdim)==1) {
  yfine = eval.fd(xfine,yfd)
  ycoef = project.basis(yfine,tfine,ybasis)
}

if(length(cdim)==2){
  for(i in 1:cdim[2]){
    yfine = eval.fd(xfine[,i],yfd[i])
    ycoef[,i] = project.basis(yfine,tfine,ybasis)
  }
}
    
if(length(cdim)==3){
  for(j in 1:cdim[3]){
    for(i in 1:cdim[2]){
       yfine = eval.fd(xfine[,i,j],yfd[i,j])  
       ycoef[,i,j] = project.basis(yfine,tfine,ybasis)
    }
  }
}

yregfd <- fd(ycoef, ybasis)
return(yregfd)
}

scoresPACE <- function(data, time, covestimate, PC){
  if(is.list(data)){
    datamat = sparse.mat(data)
    datalist = sparse.list(datamat, time)
  }else{
    datamat = data
    datalist = sparse.list(datamat, time)
  }
  

  indexes = lapply(datalist, function(x) which(time %in% x[,1])) #sampling points for each subject
  coef = PC$values
  phi = lapply(indexes, function(x) eval.fd(x,PC$harmonics))
  
  
  if(inherits(covestimate$cov.estimate, "bifd")) {
    nvar = 1
    mean.point = matrix(eval.fd(time, covestimate$meanfd),nrow = nrow(eval.fd(time, covestimate$meanfd)),ncol = nvar)
    data.var.mat = (apply(datamat,2, function(x) x - mean.point))^2 #data given as matrix
    mu = lapply(indexes,function(x) as.matrix(eval.fd(x,covestimate$meanfd),nrow = length(x),ncol = nvar))
    cent = lapply(1:length(datalist), function(x) datalist[[x]][,-1]-mu[[x]])
    
    covdiag = diag(eval.bifd(time,time,covestimate$cov.estimate))
    varest.mat = as.matrix(apply(data.var.mat,2, function(x) x-covdiag))
    varest = mean(as.vector(varest.mat)[!is.na(as.vector(varest.mat))])
    diagon = lapply(1:length(datalist), function(x) diag(varest, nrow = length(indexes[[x]]), ncol = length(indexes[[x]])))
    sigmai = Map('+', lapply(indexes, function(x) eval.bifd(x,x,covestimate$cov.estimate)), diagon)

    scr = t(do.call(cbind,lapply(1:length(datalist), function(x) coef*(t(phi[[x]])%*%solve(sigmai[[x]])%*%cent[[x]]) )))
    
  }else{
    k=0 #nvar
    l = length(covestimate$cov.estimate)
    while(l>0){
      k = k + 1
      l = l - k
    }
    nvar = k 
    mean.point = matrix(eval.fd(time, covestimate$meanfd),nrow = nrow(eval.fd(time, covestimate$meanfd)),ncol = nvar)
    data.var.mat = (apply(datamat,2, function(x) x - mean.point))^2 #data given as matrix
    mu = lapply(indexes,function(x) matrix(eval.fd(x,covestimate$meanfd),nrow = length(x),ncol = nvar))
    cent = lapply(1:length(datalist), function(x) datalist[[x]][,-1]-mu[[x]])
    
    ind = cumsum(c(1,k - 0:(k-2)))
    covdiag = as.vector(do.call(cbind, lapply(covestimate$cov.estimate[ind], function(x) diag(eval.bifd(time,time,x)))))
    varest.mat = as.matrix(apply(data.var.mat,2, function(x) x-covdiag))
    ntime = dim(datamat)[1]
    varest = unlist(lapply(1:nvar, function(x) mean(as.vector(varest.mat[(ntime*x-(ntime-1)):(ntime*x),])[!is.na(as.vector(varest.mat[(ntime*x-(ntime-1)):(ntime*x),]))])))
    
    diagon = list()
    sigmai = list()
    scr = array(NA, dim = c(ncol(datamat),ncol(coef),nvar))
    for(l in 1:nvar){
      diagon[[l]] = lapply(1:ncol(datamat), function(x) diag(varest[l], nrow = length(indexes[[x]]), ncol = length(indexes[[x]])))
      sigmai[[l]] = Map('+', lapply(indexes, function(x) eval.bifd(x,x,covestimate$cov.estimate[[ind[l]]])), diagon[[l]])
      scr[,,l] = t(do.call(cbind,lapply(1:ncol(datamat), function(x) coef[l,]*(t(phi[[x]][,,l])%*%solve(sigmai[[l]][[x]])%*%cent[[x]][,l]) )))
    }
  }
  
  
  return(scr)
}
smooth.basis1 <- function (argvals=1:n, y, fdParobj,
                           wtvec=NULL,   fdnames=NULL, covariates=NULL,
                           method="chol", dfscale=1, returnMatrix=FALSE)
{
  #  Arguments:
  # ARGVALS  A set of N argument values, set by default to equally spaced
  #             on the unit interval (0,1).
  # Y        an array containing values of curves
  #             If the array is a matrix, rows must correspond to argument
  #             values and columns to replications, and it will be assumed
  #             that there is only one variable per observation.
  #             If Y is a three-dimensional array, the first dimension
  #             corresponds to argument values, the second to replications,
  #             and the third to variables within replications.
  #             If Y is a vector, only one replicate and variable are assumed.
  # FDPAROBJ A functional parameter or fdPar object.  This object
  #             contains the specifications for the functional data
  #             object to be estimated by smoothing the data.  See
  #             comment lines in function fdPar for details.
  #             This argument may also be either a FD object, or a
  #             BASIS object.  In this case, the smoothing parameter
  #             LAMBDA is set to 0.
  # WEIGHT   A vector of N weights, set to one by default, that can
  #             be used to differentially weight observations, or
  #             a symmetric positive definite matrix of order N
  # FDNAMES  A cell of length 3 with names for
  #             1. argument domain, such as "Time"
  #             2. replications or cases
  #             3. the function.
  # COVARIATES  A N by Q matrix Z of covariate values used to augment
  #             the smoothing function, where N is the number of
  #             data values to be smoothed and Q is the number of
  #             covariates.  The process of augmenting a smoothing
  #             function in this way is often called "semi-parametric
  #             regression".  The default is the null object NULL.
  # METHOD      The method for computing coefficients.  The usual method
  #             computes cross-product matrices of the basis value matrix,
  #             adds the roughness penalty, and uses the Choleski decomposition
  #             of this to compute coefficients, analogous to using the
  #             normal equations in least squares fitting.  But this approach,
  #             while fast, contributes unnecessary rounding error, and the qr
  #             decomposition of the augmented basis matrix is prefererable.
  #             But nothing comes for free, and the computational overhead of
  #             the qr approach can be a serious problem for large problems
  #             (n of 1000 or more).  For this reason, the default is
  #             "method" = "chol", but if 'method' == 'qr', the qr
  #             decomposition is used.
  # DFFACTOR A multiplier of df in GCV, set to one by default
  #
  # Returns a list containing:
  #   FDOBJ   an object of class fd containing coefficients.
  #   DF      a degrees of freedom measure.
  #   GCV     a measure of lack of fit discounted for df.
  #              If the function is univariate, GCV is a vector
  #              containing the error  sum of squares for each
  #              function, and if the function is multivariate,
  #              GCV is a NVAR by NCURVES matrix.
  #   COEF    the coefficient matrix for the basis function
  #                expansion of the smoothing function
  #   SSE     the error sums of squares.
  #              SSE is a vector or matrix of the same size as
  #              GCV.
  #   PENMAT  the penalty matrix.
  #   Y2CMAP  the matrix mapping the data to the coefficients.
  #
  # last modified 16 November 2021 by Jim Ramsay
  
  #  This version of smooth.basis, introduced in March 2011, permits ARGVALS
  #  to be a matrix, with the same dimensions as the first two dimensions of Y
  #  This allows the sampling points to vary from one record to another.
  #  This first section of code selects the version of smooth.basis to use
  #  depending on whether ARGVALS is a vector (case 1) or a matrix (case 2)
  #  The earlier version of smooth.basis is found at the end of the file where
  #  it is names smooth.basis1.
  
  #  ---------------------------------------------------------------------
  #                      Check argments
  #  ---------------------------------------------------------------------
  
  #  check Y  and set nrep, nvar and ndim
  
  #  set up matrix or array for coefficients of basis expansion,
  #  as well as names for replications and, if needed, variables 
  
  if (is.vector(y))y <- matrix(y,length(y),1)
  dimy <- dim(y)
  n    <- dimy[1]
  
  ycheck <- ycheck(y, n)
  y      <- ycheck$y
  y0     <- y  #  preserve a copy of Y
  nrep   <- ycheck$ncurve
  nvar   <- ycheck$nvar
  ndim   <- ycheck$ndim
  ydim   <- dim(y)
  
  #  check ARGVALS
  
  if (!is.numeric(argvals)) stop("'argvals' is not numeric.")
  argvals <- as.vector(argvals)
  
  #  check fdParobj
  
  fdParobj <- fdParcheck(fdParobj, nrep)
  
  fdobj    <- fdParobj$fd
  lambda   <- fdParobj$lambda
  Lfdobj   <- fdParobj$Lfd
  penmat   <- fdParobj$penmat
  
  #  check LAMBDA
  
  if (lambda < 0) {
    warning ("Value of 'lambda' was negative  0 used instead.")
    lambda <- 0
  }
  
  #  check WTVEC
  
  wtlist <- wtcheck(n, wtvec)
  wtvec  <- wtlist$wtvec
  onewt  <- wtlist$onewt
  matwt  <- wtlist$matwt
  
  # if (matwt) wtmat <- wtvec #  else wtmat <- diag(as.vector(wtvec))
  
  #  set up names for first dimension of y
  
  tnames <- dimnames(y)[[1]]
  if (is.null(tnames)) tnames <- 1:n
  
  #  extract information from fdParobj
  
  nderiv   <- Lfdobj$nderiv
  basisobj <- fdobj$basis
  dropind  <- basisobj$dropind
  ndropind <- length(dropind)
  nbasis   <- basisobj$nbasis - ndropind 
  
  #  get names for basis functions
  
  names <- basisobj$names
  if (ndropind > 0) {
    names <- names[-dropind]
  }
  
  if (ndim == 2)  {
    coef   <- matrix(0,nbasis,nrep)
    ynames <- dimnames(y)[[2]]
    vnames <- "value"
    dimnames(coef) <- list(names, ynames)
  }
  
  if (ndim == 3)  {
    coef <- array(0,c(nbasis,nrep,nvar))
    ynames <- dimnames(y)[[2]]
    vnames <- dimnames(y)[[3]]
    dimnames(coef) <- list(names, ynames, vnames)
  }
  
  #  check COVARIATES and set value for q, the number of covariates
  
  if (!is.null(covariates)) {
    if (!is.numeric(covariates)) {
      stop(paste("smooth_basis_LS:covariates",
                 "Optional argument COVARIATES is not numeric."))
    }
    if (dim(covariates)[1] != n) {
      stop(paste("smooth_basis_LS:covariates",
                 "Optional argument COVARIATES has incorrect number of rows."))
    }
    q <- dim(covariates)[2]
  } else {
    q <- 0
    beta. <- NULL
  }
  
  #  set up names for first dimension of y
  
  tnames <- dimnames(y)[[1]]
  if (is.null(tnames)) tnames <- 1:n
  
  #  ----------------------------------------------------------------
  #                set up the linear equations for smoothing
  #  ----------------------------------------------------------------
  
  #  set up matrix of basis function values
  
  basismat <- eval.basis(as.vector(argvals), basisobj, 0, returnMatrix)
  
  if (method == "chol") {
    
    #  -----------------------------------------------------------------
    #  use the default choleski decomposition of the crossproduct of the
    #  basis value matrix plus the roughness penalty
    #  -----------------------------------------------------------------
    
    if (n > nbasis + q || lambda > 0) {
      
      #  augment BASISMAT0 and BASISMAT by the covariate matrix
      #  if it is supplied
      
      if (!is.null(covariates)) {
        ind1 <- 1:n
        ind2 <- (nbasis+1):(nbasis+q)
        basismat  <- as.matrix(basismat)
        basismat  <- cbind(basismat,  matrix(0,dim(basismat) [1],q))
        basismat[ind1,ind2]  <- covariates
      }
      
      #  Compute the product of the basis and weight matrix
      
      if (matwt) {
        wtfac   <- chol(wtvec)
        basisw  <- wtvec %*% basismat
      } else {
        rtwtvec <- sqrt(wtvec)
        rtwtmat <- matrix(rtwtvec,n,nrep)
        basisw  <- (wtvec %*% matrix(1,1,nbasis+q))*basismat
      }
      
      #  the weighted crossproduct of the basis matrix
      Bmat  <- t(basisw) %*% basismat
      Bmat0 <- Bmat
      
      #  set up right side of normal equations
      
      if (ndim < 3) {
        Dmat <- t(basisw) %*% y
      } else {
        Dmat <- array(0, c(nbasis+q, nrep, nvar))
        for (ivar in 1:nvar) {
          Dmat[,,ivar] <- crossprod(basisw,y[,,ivar])
        }
      }
      
      if (lambda > 0) {
        #  smoothing required, add the contribution of the penalty term
        if (is.null(penmat)) penmat <- eval.penalty(basisobj, Lfdobj)
        Bnorm   <- sqrt(sum(diag(t(Bmat0) %*% Bmat0)))
        pennorm <- sqrt(sum(penmat*penmat))
        condno  <- pennorm/Bnorm
        if (lambda*condno > 1e12) {
          lambda <- 1e12/condno
          warning(paste("lambda reduced to",lambda,
                        "to prevent overflow"))
        }
        if (!is.null(covariates)) {
          penmat <- rbind(cbind(penmat, matrix(0,nbasis,q)),
                          cbind(matrix(0,q,nbasis), matrix(0,q,q)))
        }
        Bmat   <- Bmat0 + lambda*penmat
      } else {
        penmat <- NULL
        Bmat   <- Bmat0
      }
      
      #  compute inverse of Bmat
      
      Bmat    <- (Bmat+t(Bmat))/2
      Lmat    <- try(chol(Bmat), silent=TRUE)
      if (inherits(Lmat, "try-error")) {
        Beig  <- eigen(Bmat, symmetric=TRUE)
        BgoodEig <- (Beig$values>0)
        Brank <- sum(BgoodEig)
        if (Brank<dim(Bmat)[1])
          warning("Matrix of basis function values has rank ",
                  Brank, " < dim(fdobj$basis)[2] = ",
                  length(BgoodEig), "  ignoring null space")
        goodVec <- Beig$vectors[, BgoodEig]
        Bmatinv <- (goodVec %*% (Beig$values[BgoodEig] * t(goodVec)))
      } else {
        Lmatinv <- solve(Lmat)
        Bmatinv <- Lmatinv %*% t(Lmatinv)
      }
      
      #  compute coefficient matrix by solving normal equations
      
      if (ndim < 3) {
        coef <- Bmatinv %*% Dmat
        if (!is.null(covariates)) {
          beta. <- as.matrix(coef[(nbasis+1):(nbasis+q),])
          coef  <- as.matrix(coef[1:nbasis,])
        } else {
          beta. <- NULL
        }
      } else {
        coef <- array(0, c(nbasis, nrep, nvar))
        if (!is.null(covariates)) {
          beta. <- array(0, c(q, nrep, nvar))
        } else {
          beta. <- NULL
        }
        for (ivar in 1:nvar) {
          coefi <- Bmatinv %*% Dmat[,,ivar]
          if (!is.null(covariates)) {
            beta.[,,ivar] <- coefi[(nbasis+1):(nbasis+q),]
            coef[,,ivar] <- coefi[1:nbasis,]
          } else {
            coef[,,ivar] <- coefi
          }
        }
      }
      
    } else {
      
      if (n == nbasis + q) {
        
        #  code for n == nbasis, q == 0, and lambda == 0
        if (ndim==2) {
          coef <- solve(basismat, y)
        } else {
          for (ivar in 1:var)
            coef[1:n, , ivar] <- solve(basismat, y[,,ivar])
        }
        penmat  <- NULL
      } else {
        
        #  n < nbasis+q:  this is treated as an error
        
        stop("The number of basis functions = ", nbasis+q, " exceeds ",
             n, " = the number of points to be smoothed.")
      }
    }
    
  } else {
    
    #  -------------------------------------------------------------
    #  computation of coefficients using the qr decomposition of the
    #  augmented basis value matrix
    #  -------------------------------------------------------------
    
    if (n > nbasis || lambda > 0) {
      
      #  Multiply the basis matrix and the data pointwise by the square root
      #  of the weight vector if the weight vector is not all ones.
      #  If the weights are in a matrix, multiply the basis matrix by its
      #  Choleski factor.
      
      if (!onewt) {
        if (matwt) {
          wtfac <- chol(wtvec)
          basismat.aug <- wtfac %*% basismat
          if (ndim < 3) {
            y <- wtfac %*% y
          } else {
            for (ivar in 1:nvar) {
              y[,,ivar] <- wtfac %*% y[,,ivar]
            }
          }
        } else {
          rtwtvec  <- sqrt(wtvec)
          basismat.aug <- matrix(rtwtvec,n,nbasis) * basismat
          if (ndim < 3) {
            y <- matrix(rtwtvec,n,nrep) * y
          } else {
            for (ivar in 1:nvar) {
              y[,,ivar] <- matrix(rtwtvec,n,nrep) * y[,,ivar]
            }
          }
        }
      } else {
        basismat.aug <- basismat
      }
      
      #  set up additional rows of the least squares problem for the
      #  penalty term.
      
      if (lambda > 0) {
        if (is.null(penmat)) penmat <- eval.penalty(basisobj, Lfdobj)
        eiglist <- eigen(penmat)
        Dvec    <- eiglist$values
        Vmat    <- eiglist$vectors
        #  Check that the lowest eigenvalue in the series that is to be
        #  kept is positive.
        neiglow <- nbasis - nderiv
        naug    <- n + neiglow
        if (Dvec[neiglow] <= 0) {
          stop(paste("smooth_basis:eig",
                     "Eigenvalue(NBASIS-NDERIV) of penalty matrix ",
                     "is not positive check penalty matrix."))
        }
        #  Compute the square root of the penalty matrix in the subspace
        #  spanned by the first N - NDERIV eigenvectors
        indeig <- 1:neiglow
        penfac <- Vmat[,indeig] %*% diag(sqrt(Dvec[indeig]))
        #  Augment basismat by sqrt(lambda)*t(penfac)
        basismat.aug <- rbind(basismat.aug, sqrt(lambda)*t(penfac))
        #  Augment data vector by n - nderiv 0's
        if (ndim < 3) {
          y <- rbind(y, matrix(0,nbasis-nderiv,nrep))
        } else {
          y <- array(0,c(ydim[1]+nbasis-nderiv,ydim[2],ydim[3]))
          y[1:ydim[1],,] <- y0
          ind1 <- (1:(nbasis-nderiv)) + ydim[1]
          for (ivar in 1:nvar) {
            y[ind1,,ivar] <- matrix(0,nbasis-nderiv,nrep)
          }
        }
      } else {
        penmat <- NULL
      }
      
      #  augment BASISMAT0 and BASISMAT by the covariate matrix
      #  if it is supplied
      
      if (!is.null(covariates)) {
        ind1 <- 1:n
        ind2 <- (nbasis+1):(nbasis+q)
        basismat.aug  <- cbind(basismat.aug,  matrix(0,naug,q))
        if (!onewt) {
          if (matwt) {
            basismat.aug[ind1,ind2]  <- wtfac %*% covariates
          } else {
            wtfac <- matrix(rtwtvec,n,q)
            basismat.aug[ind1,ind2]  <- wtfac*covariates
          }
        } else {
          basismat.aug[ind1,ind2]  <- covariates
        }
        penmat <- rbind(cbind(penmat,             matrix(0,nbasis,q)),
                        cbind(matrix(0,q,nbasis), matrix(0,q,q)))
      }
      
      #  solve the least squares problem using the QR decomposition with
      #  one iteration to improve accuracy
      
      qr <- qr(basismat.aug)
      if (ndim < 3) {
        coef <- qr.coef(qr,y)
        if (!is.null(covariates)) {
          beta. <- coef[ind2,]
          coef  <- coef[1:nbasis,]
        } else {
          beta. <- NULL
        }
      } else {
        coef <- array(0, c(nbasis, nrep, nvar))
        if (!is.null(covariates)) {
          beta. <- array(0, c(q,nrep,nvar))
        } else {
          beta. <- NULL
        }
        for (ivar in 1:nvar) {
          coefi <- qr.coef(qr,y[,,ivar])
          if (!is.null(covariates)) {
            beta.[,,ivar] <- coefi[ind2,]
            coef[,,ivar] <- coefi[1:nbasis,]
          } else {
            coef[,,ivar] <- coefi
          }
        }
      }
      
    } else {
      
      if (n == nbasis + q) {
        
        #  code for n == nbasis and lambda == 0
        if (ndim==2){
          coef <- solve(basismat, y)
        } else {
          for (ivar in 1:var)
            coef[,,ivar] <- solve(basismat, y[,,ivar])
        }
        penmat <- NULL
        
      } else {
        
        stop(paste("The number of basis functions = ", nbasis, " exceeds ",
                   n, " = the number of points to be smoothed.  "))
      }
    }
    
  }
  #  ----------------------------------------------------------------
  #            compute SSE, yhat, GCV and other fit summaries
  #  ----------------------------------------------------------------
  
  #  compute map from y to c
  
  if (onewt) {
    temp   <- t(basismat) %*% basismat
    if (lambda > 0) {
      temp <- temp + lambda*penmat
    }
    L      <- chol(temp)
    MapFac <- solve(t(L),t(basismat))
    y2cMap <- solve(L,MapFac)
  } else {
    if(matwt){
      temp <- t(basismat) %*% wtvec %*% basismat
    } else {
      temp <- t(basismat) %*% (as.vector(wtvec)*basismat)
    }
    
    if  (lambda > 0) {
      temp <- temp + lambda*penmat
    }
    L      <- chol((temp+t(temp))/2)
    MapFac <- solve(t(L),t(basismat))
    if(matwt){
      y2cMap <- solve(L, MapFac%*%wtvec)
    } else {
      y2cMap <- solve(L,MapFac*rep(as.vector(wtvec), e=nrow(MapFac)))
    }
  }
  
  #  compute degrees of freedom of smooth
  
  df. <- sum(diag(y2cMap %*% basismat))
  
  #  compute error sum of squares
  
  if (ndim < 3) {
    yhat <- basismat[,1:nbasis] %*% coef
    SSE  <- sum((y[1:n,] - yhat)^2)
    if (is.null(ynames)) ynames <- dimnames(yhat)[[2]]
  } else {
    SSE <- 0
    yhat <- array(0,c(n, nrep, nvar))
    dimnames(yhat) <- list(dimnames(basismat)[[1]],
                           dimnames(coef)[[2]],
                           dimnames(coef)[[3]])
    for (ivar in 1:nvar) {
      yhat[,,ivar] <- basismat[,1:nbasis] %*% coef[,,ivar]
      SSE <- SSE + sum((y[1:n,,ivar] - yhat[,,ivar])^2)
    }
    if (is.null(ynames))ynames <- dimnames(yhat)[[2]]
    if (is.null(vnames))vnames <- dimnames(yhat)[[2]]
  }
  if (is.null(ynames)) ynames <- paste("rep", 1:nrep, sep="")
  if (is.null(vnames)) vnames <- paste("value", 1:nvar, sep="")
  
  #  compute  GCV index
  
  if (!is.null(df.) && df. < n) {
    if (ndim < 3) {
      gcv <- rep(0,nrep)
      for (i in 1:nrep) {
        SSEi <- sum((y[1:n,i] - yhat[,i])^2)
        gcv[i] <- (SSEi/n)/((n - df.)/n)^2
      }
      if (ndim > 1) names(gcv) <- ynames
    } else {
      gcv <- matrix(0,nrep,nvar)
      for (ivar in 1:nvar) {
        for (i in 1:nrep) {
          SSEi <- sum((y[1:n,i,ivar] - yhat[,i,ivar])^2)
          gcv[i,ivar] <- (SSEi/n)/((n - df.)/n)^2
        }
      }
      dimnames(gcv) <- list(ynames, vnames)
    }
  } else {
    gcv <- NULL
  }
  
  #------------------------------------------------------------------
  #       Set up the functional data objects for the smooths
  #  ------------------------------------------------------------------
  
  #  set up default fdnames
  
  if (is.null(fdnames)) {
    fdnames <- list(time=tnames, reps=ynames, values=vnames)
  }
  
  #  set up the functional data object
  
  if (ndim < 3) {
    coef  <- as.matrix(coef)
    fdobj <- fd(coef[1:nbasis,],  basisobj, fdnames)
  } else {
    fdobj <- fd(coef[1:nbasis,,], basisobj, fdnames)
  }
  
  #  return penalty matrix to original state if there were covariates
  
  if (!is.null(penmat) && !is.null(covariates))
    penmat <- penmat[1:nbasis,1:nbasis]
  
  #  assemble the fdSmooth object returned by the function
  
  smoothlist <- list(fd=fdobj, df=df., gcv=gcv, beta=beta.,
                     SSE=SSE, penmat=penmat, y2cMap=y2cMap,
                     argvals=argvals, y=y0)
  
  class(smoothlist) <- "fdSmooth"
  
  return(smoothlist)
  
}

smooth.basis.glm <- function(argvals, y, fdParobj, wtvec=NULL, fdnames=NULL, 
                             covariates=NULL, family="binomial", dfscale=1) {
  #SMOOTH.GLM  Smooths discrete curve represented by basis function
  #  expansions fit by penalized least squares.
  #
  #  Required arguments for this function are
  #
  #  ARGVALS   A set of argument values, set by default to equally spaced
  #               on the unit interval (0,1).
  #  Y         If the family is not binomial, y is a matrix or an array 
  #               containing values of curves.
  #               If y is a matrix, rows must correspond to argument
  #               values and columns to replications, and it will be assumed
  #               that there is only one variable per observation.
  #               If Y is a three-dimensional array, the first dimension
  #               corresponds to argument values, the second to replications,
  #               and the third to variables within replications.
  #               If Y is a vector, only one replicate and variable are 
  #               assumed.
  #            If the family is binomial local sample sizes M.i,
  #               Y is a list vector of length 2, the first of which cantains
  #               the matrix or array as above containing observed frequencies,
  #               and the second of which contains the corresponding local
  #               sample sizes.
  #  FDPAROBJ  A functional parameter or fdPar object.  This object 
  #               contains the specifications for the functional data
  #               object to be estimated by smoothing the data.  See
  #               comment lines in function fdPar for details.
  #               This argument may also be either a FD object, or a 
  #               BASIS object.  If this argument is a basis object, the 
  #               smoothing parameter LAMBDA is set to 0.
  #
  #  Optional arguments are input in pairs  the first element of the pair
  #     is a string specifying the property that the argument value defines,
  #     and the second element is the value of the argument
  #
  #     Valid property/value pairs include
  #
  #     Property        Value
  #     ----------------------------------------------------------------
  #     weight          vector of the same length as the data vector to be
  #                     smoothed, containing nonnegative weights to be 
  #                     applied to the data values
  #     fdnames         A cell array of length 3 with names for
  #                       1. argument domain, such as "Time"
  #                       2. replications or cases
  #                       3. the function.
  #     covariates      A N by Q matrix Z of covariate values used to augment
  #                     the smoothing function, where N is the number of
  #                     data values to be smoothed and Q is the number of
  #                     covariates.  The process of augmenting a smoothing 
  #                     function in this way is often called "semi-parametric 
  #                     regression".  The default is the empty object NULL.
  #     dfscale         A scalar value multiplying the degrees of freedom
  #                     in the definition of the generalized 
  #                     cross-validated or GCV criterion for selecting the
  #                     bandwidth parameter LAMBDA.  It was recomm}ed by
  #                     Chong Gu that this be a number slightly larger than
  #                     1.0, such as 1.2, to prevent under-smoothing,
  #                     The default is 1.0.
  #     family          a character string containing one of
  #                       "normal"  
  #                       "binomial"
  #                       "poisson"
  #                       "gamma"
  #                       "inverse gaussian"
  #                     the value determines which of the link functions in
  #                     the generalized linear model (GLM) family is to be
  #                     used.  The default is "normal".
  #      control        a struct object controlling iterations with members
  #                       epsilon  convergence criterion (default 1e-8)
  #                       maxit    max. iterations       (default 25)
  #                       trace    output iteration info (0)
  #      start          a vector containing starting values for coefficients
  #                      
  #
  #  Returned objects are
  #
  #  FDOBJ    an object of class fd containing coefficients.
  #  DF       a degrees of freedom measure.
  #  GCV      a measure of lack of fit discounted for df.
  #                 If the function is univariate, GCV is a vector 
  #                 containing the stop  sum of squares for each 
  #                 function, and if the function is multivariate, 
  #                 GCV is a NVAR by NCURVES matrix.
  #  SSE      the stop sums of squares.  
  #                 SSE is a vector or matrix of the same size as 
  #                 GCV.
  #  PENMAT   the penalty matrix, if computed, otherwise NULL.
  #  Y2CMAP   the matrix mapping the data to the coefficients.
  #  ARGVALS  the input set of argument values.
  #  Y        the input array containing values of curves
  
  #  Last modified 15 May 2018 by Jim Ramsay
  
  n <- length(argvals)
  
  #  check ARGVALS
  
  if (!is.numeric(argvals)) stop("ARGVALS is not numeric.")
  argvals <- as.vector(argvals)
  if (length(argvals) < 2)  stop("ARGVALS does not contain at least two values.")
  
  #  check Y
  
  if (is.vector(y)) y <- as.matrix(y)
  c  
  #  check FDPAROBJ and get FDOBJ and LAMBDA
  
  if (!inherits(fdParobj, "fdPar")) {
    if (inherits(fdParobj, "fd") || inherits(fdParobj, "basisfd")) {
      fdParobj <- fdPar(fdParobj)
    } else
      stop(paste("'fdParobj' is not a functional parameter object,",
                 "not a functional data object, and",
                 "not a basis object."))
  }
  fdobj    <- fdParobj$fd
  lambda   <- fdParobj$lambda
  Lfdobj   <- fdParobj$Lfd
  
  #  check LAMBDA
  
  if (lambda < 0) { 
    lambda <- 0  
  }
  
  #  get BASIS and NBASIS
  
  basisobj <- fdobj$basis
  nbasis   <- basisobj$nbasis - length(basisobj$dropind)
  
  #  check WTVEC
  
  # wtList <- wtcheck(n, wtvec)
  # wtvec  <- wtList[[1]]
  # onewt  <- wtList[[2]]
  # 
  # if (onewt) {
  #   wtvec <- matrix(1,n,1)
  # }
  
  if (is.null(wtvec)) wtvec <- matrix(1,n,1)
  
  #  check FDNAMES
  
  if (!is.null(fdnames) && !is.list(fdnames)) {
    stop("Optional argument FDNAMES is not a list object.")
  }
  
  if (is.list(fdnames) && length(fdnames) != 3) {
    stop("Optional argument FDNAMES is not of length 3.")
  }
  
  #  check COVARIATES
  
  q <- 0
  if (!is.null(covariates)) {
    if (!is.numeric(covariates)) {
      stop("Optional argument COVARIATES is not numeric.")
    }
    if (dim(covariates)[1] != n) {
      stop("Optional argument COVARIATES has incorrect number of rows.")
    }
    q <- dim(covariates)[2]
  }
  
  #  ------------------------------------------------------------------
  #                set up the linear equations for smoothing
  #  ------------------------------------------------------------------
  
  #  set up matrix of basis function values
  
  basismat <- eval.basis(argvals, basisobj)
  
  if (n >= nbasis || lambda > 0) {
    
    #  The following code is for the coefficients completely determined
    
    #  set up additional rows of the least squares problem for the
    #  penalty term.
    
    basismat0 <- basismat
    y0        <- y
    
    if (lambda > 0) {
      penmat  <- eval.penalty(basisobj, Lfdobj)
      lamRmat <- lambda*penmat
    } else {
      lamRmat <- NULL
    }
    
    #  augment BASISMAT0 and BASISMAT by the covariate matrix 
    #  if (it is supplied
    
    if (!is.null(covariates)) {
      basismat0 <- matrix(cbind(basismat0, covariates))
      basismat  <- matrix(cbind(basismat,  covariates))
      if (!is.null(lamRmat)) {
        lamRmat <- rbind(cbind(lamRmat,            matrix(0,nbasis,q)),
                         cbind(matrix(0,q,nbasis), matrix(0,q)       ))
      }
    }
    
    #  ------------------------------------------------------------------
    #               compute solution using Matlab function glmfit
    #  ------------------------------------------------------------------
    
    dimy   <- dim(y)
    nbasis <- dimy[1]
    ncurve <- dimy[2]
    ndim   <- length(dimy)
    if (ndim < 3) {
      coef  <- matrix(0,nbasis,ncurve)
      dev   <- matrix(0,ncurve,1)
      glmList <- glm.fda(basismat, y, family, lamRmat, wtvec)
      coef <- glmList[[1]]
      dev  <- glmList[[2]]
    } else {
      nvar  <- dimy[3]
      coef  <- array(0,c(nbasis,ncurve,nvar))
      dev   <- matrix(0,ncurve,nvar)
      for (ivar in 1:nvar) {
        yi <- as.matrix(y[,,ivar])
        glmList <- glm.fda(basismat, yi, family, lamRmat, wtvec)
        coefi  <- glmList[[1]]
        devi   <- glmList[[2]]
        statsi <- glmList[[3]]
        coef[,,ivar]  <- coefi
        dev[,ivar]    <- devi
        stats[[ivar]] <- statsi
      }
    }
    
    #  compute basismat*R^{-1}
    
    if (is.null(lamRmat)) {
      M <- crossprod(basismat)
    } else {
      M <- crossprod(basismat) + lamRmat
    }
    
    #  compute map from y to c
    
    y2cMap <- solve(M,t(basismat))
    
    #  compute degrees of freedom of smooth
    
    df <- sum(diag(basismat %*% y2cMap))
    
  } else {
    stop(paste("The number of basis functions exceeds the number of ", 
               "points to be smoothed."))    
  }
  
  #  ------------------------------------------------------------------
  #            compute SSE, yhat, GCV and other fit summaries
  #  ------------------------------------------------------------------
  
  # #  compute stop sum of squares
  # 
  # if (ndim < 3) {
  #   yhat <- basismat0 %*% coef
  #   SSE  <- sum((y0 - yhat)^2)
  # } else {
  #   SSE <- matrix(0,nvar,ncurve)
  #   for (ivar in 1:nvar) {
  #     coefi <- coef[,,ivar]
  #     yhati <- basismat %*% coefi
  #     yi    <- y[,,ivar]
  #     SSE[ivar,] <- sum((yi - yhati)^2)
  #   }
  # }
  # 
  # #  compute  GCV index
  # 
  # if (df < n) {
  #   gcv <- (SSE/n)/((n - dfscale*df)/n)^2
  # } else {
  #   gcv <- NA
  # }
  
  #  set up the functional data object
  
  if (ndim < 3) {
    fdobj <- fd(coef[1:nbasis,],  basisobj, fdnames)
  } else {
    fdobj <- fd(coef[1:nbasis,,], basisobj, fdnames)
  }
  
  #  set up the regression coefficient matrix beta
  
  if (q > 0) {
    ind <- (nbasis+1):(nbasis+q)
    if (ndim < 3) {
      beta <- coef[ind,]
    } else {
      beta <- coef[ind,,]
    }
  } else {
    beta <- NULL
  }
  
  return(list(fdobj=fdobj, beta=beta))
         
}



smooth.basisPar <- function(argvals, y, fdobj=NULL, Lfdobj=NULL,
      lambda=0, estimate=TRUE, penmat=NULL,
      wtvec=NULL, fdnames=NULL, covariates=NULL,
                         method="chol", dfscale=1 ) {
#  This function acts as a wrapper for those who don't want to take the
#  step of setting up a functional parameter object before invoking function
#  smooth.basis.  It simply does this setup for the user.   See the help
#  file for smooth.basis for further details.
#
#  However, smooth.basisPar also sets up a default basis in the event that
#  argument fdobj is either NULL
#  (order 4 b-spline basis with breaks = argvals) or
#  a positive integer (same as the NULL case, but with order = integer).

##
## 1.  check fdobj
##

    if (is.null(fdobj)) {
      #  if fdobj is NULL, create an order 4 bspline basis with breaks equal to
      #  argvals (see help file for create.bspline.basis(argvals) for further
      #  details)
      fdobj <- create.bspline.basis(argvals)
    } else {
      if (is.numeric(fdobj)) {
        # if fdobj is a positive integer, use this integer as the order
        # of a bspline basis with breaks equal to argvals
        if (length(fdobj)==1) {
          if (round(fdobj) != fdobj)
            stop("'fdobj' is numeric but not an integer")
          if (fdobj <= 0)
            stop("'fdobj' is not a positive integer")
          fdobj <- create.bspline.basis(argvals, norder=fdobj)
        } else {
          #  fdobj is neither NULL nor numeric, so use whatever it is as an
          #  argument for function fd, which in turn will do further checking
          fdobj <- fd(fdobj)
        }
      }
    }
##
## 2.  fdPar:  set up the functional parameter object from arguments
##
  fdP <- fdPar(fdobj, Lfdobj=Lfdobj, lambda=lambda,
               estimate=estimate, penmat=penmat)
  
##
## 3.  smooth.basis:  carry out smoothing by a call to smooth.basis and
##     return the smoothList object that this function returns
##
  
  fdSmooth <- smooth.basis(argvals, y, fdP, wtvec=wtvec, fdnames=fdnames,
                           covariates=covariates, method="chol", dfscale=dfscale)
                           
  return(fdSmooth)
}
smooth.basis <- function(argvals=1:n, y, fdParobj,
                         wtvec=NULL,   fdnames=NULL, covariates=NULL,
                         method="chol", dfscale=1, returnMatrix=FALSE) {
#  Arguments:
# ARGVALS  A set of N argument values, set by default to equally spaced
#             on the unit interval (0,1).
# Y        an array containing values of curves
#             If the array is a matrix, rows must correspond to argument
#             values and columns to replications, and it will be assumed
#             that there is only one variable per observation.
#             If Y is a three-dimensional array, the first dimension
#             corresponds to argument values, the second to replications,
#             and the third to variables within replications.
#             If Y is a vector, only one replicate and variable are assumed.
# FDPAROBJ A functional parameter or fdPar object.  This object
#             contains the specifications for the functional data
#             object to be estimated by smoothing the data.  See
#             comment lines in function fdPar for details.
#             This argument may also be either a FD object, or a
#             BASIS object.  In this case, the smoothing parameter
#             LAMBDA is set to 0.
# WEIGHT   A vector of N weights, set to one by default, that can
#             be used to differentially weight observations, or
#             a symmetric positive definite matrix of order N
# FDNAMES  A cell of length 3 with names for
#             1. argument domain, such as "Time"
#             2. replications or cases
#             3. the function.
# COVARIATES  A N by Q matrix Z of covariate values used to augment
#             the smoothing function, where N is the number of
#             data values to be smoothed and Q is the number of
#             covariates.  The process of augmenting a smoothing
#             function in this way is often called "semi-parametric
#             regression".  The default is the null object NULL.
# METHOD      The method for computing coefficients.  The usual method
#             computes cross-product matrices of the basis value matrix,
#             adds the roughness penalty, and uses the Choleski
#             decomposition of this to compute coefficients, analogous
#             to using the normal equations in least squares fitting.
#             But this approach, while fast, contributes unnecessary
#             rounding error, and the qr decomposition of the augmented
#             basis matrix is prefererable.  But nothing comes for free,
#             and the computational overhead of the qr approach can be a
#             serious problem for large problems (n of 1000 or more).
#             For this reason, the default is "method" = "chol", but if
#             'method' == 'qr', the qr decomposition is used.
# DFFACTOR A multiplier of df in GCV, set to one by default
# RETURNMATRIX ... If False, a matrix in sparse storage model can be returned
#               from a call to function BsplineS.  See this function for
#               enabling this option.
#
# Returns a list containing:
#   FDOBJ   an object of class fd containing coefficients.
#   DF      a degrees of freedom measure.
#   GCV     a measure of lack of fit discounted for df.
#              If the function is univariate, GCV is a vector
#              containing the error  sum of squares for each
#              function, and if the function is multivariate,
#              GCV is a NVAR by NCURVES matrix.
#   COEF    the coefficient matrix for the basis function
#                expansion of the smoothing function
#   SSE     the error sums of squares.
#              SSE is a vector or matrix of the same size as
#              GCV.
#   PENMAT  the penalty matrix.
#   Y2CMAP  the matrix mapping the data to the coefficients.

#  This version of smooth.basis, introduced in March 2011, permits ARGVALS
#  to be a matrix, with the same dimensions as the first two dimensions of Y
#  This allows the sampling points to vary from one record to another.
#  This first section of code selects the version of smooth.basis to use
#  depending on whether ARGVALS is a vector (case 1) or a matrix (case 2)
#  The earlier version of smooth.basis is found at the end of the file where
#  it is named smooth.basis1.
#  RETURNMATRIX ... If False, a matrix in sparse storage model can be returned
#               from a call to function BsplineS.  See this function for
#               enabling this option.

# last modified 28 December 2012

##
##  check y
##
  if (!is.numeric(y)) stop("'y' is not numeric.")
  if (is.vector(y)) y <- as.matrix(y)
  dimy <- dim(y)
  ndy  <- length(dimy)
  n    <- dimy[1]
##
##  check argvals
##
  if (is.null(argvals)) stop('argvals required;  is NULL.')
#
  if (is.numeric(argvals)) {
    if(is.vector(argvals))argvals <- as.matrix(argvals)
    Argvals <- argvals
  } else {
    Argvals <- argvals
#     stop("'argvals' is not numeric.")
# turn off warnings in checking if argvals can be converted to numeric.
    op <- options(warn=-1)
    argvals <- as.matrix(as.numeric(Argvals))
    options(op)
    nNA <- sum(is.na(argvals))
    if(nNA>0)
      stop('as.numeric(argvals) contains ', nNA,
           ' NA', c('', 's')[1+(nNA>1)],
           ';  class(argvals) = ', class(argvals))
  }
#
  dima <- dim(argvals)
  nda  <- length(dima)
  if (ndy < nda) stop("argvals has ", nda, " dimensions  y has only ", ndy)
  
#  check that the first dimensions of argvals and y match
  
  if (dima[1] != dim(y)[1]) 
     stop("Lengths of first dimensions of argvals and y do not match.")
  
##
##  select which version of smooth.basis to use, according to dim. of argvals
##  are all dimensions of argvals equal to the first nda of those of y?
##
  if (nda < 3 ) {
    #  argvals is a matrix
    if (dima[2] == 1) {
      #  argvals is a matrix with a single column, the usual case
      #  the base version smooth.basis1 is called directly
      #  see separate file smooth.basis1.R for this function
      #  ---------------------------------------------------
      sb2 <- smooth.basis1(argvals, y, fdParobj,
                           wtvec=wtvec,   fdnames=fdnames,
                           covariates=covariates,
                           method=method, dfscale=dfscale,
                           returnMatrix=returnMatrix)
      #  ---------------------------------------------------
      sb2$argvals <- Argvals
    } else {
      # With class(argvals) == Date or POSIXct,
      # argvals can NOT be a matrix or 3-d array.
      #  smooth.basis2 is called, which in turn calls smooth.basis1 in a loop
      #  see below for smooth.basis2
      #  ---------------------------------------------------
      sb2 <- smooth.basis2(argvals, y=y, fdParobj=fdParobj,
                           wtvec=wtvec,   fdnames=fdnames,
                           covariates=covariates,
                           method=method, dfscale=dfscale,
                           returnMatrix=returnMatrix)
      #  ---------------------------------------------------
    }
    return(sb2)
  }
  # end if(nda<3)

  if (nda < 4) {
      #  argvals is an array, call smooth.basis3 which calls smooth.basis2
      #  inside a loop.  see below for smooth.basis3
      return(
               #  ---------------------------------------------------
               smooth.basis3(argvals, y=y, fdParobj=fdParobj,
                       wtvec=wtvec,   fdnames=fdnames,
                       covariates=covariates,
                       method=method, dfscale=dfscale,
                       returnMatrix=returnMatrix) 
               #  ---------------------------------------------------
             )
  } else {
      #  dimensions of argval inconsistent with those of y, throw error
      cat("dim(argvals) =", paste(dima, collapse=", "), "\n")
      cat("dim(y)      = ", paste(dimy, collapse=", "), "\n")
      stop("Dimensions of argvals do not match those of y")
      return()
  }

}

################################################################################

smooth.basis2 <- function(argvals=matrix(1:n,n,N), y, fdParobj,
                          wtvec=NULL,   fdnames=NULL, covariates=NULL,
                          method="chol", dfscale=1, returnMatrix=FALSE) {
##
## 1.  number of  dimensions of y = 2 or 3?
##
dimy     <- dim(y)
ndy      <- length(dimy)
n        <- dimy[1]
N        <- dimy[2]
ynames   <- dimnames(y)
argNames <- dimnames(argvals)
##
## 2.  ndy == 2
##
if (ndy < 3) {
    #  2.1.  Start by smoothing first record using argvals[, 1]

    sb1 <- smooth.basis1(argvals[, 1], y=y[, 1], fdParobj=fdParobj,
                         wtvec=wtvec,   fdnames=fdnames,
                         covariates=covariates,
                         method=method, dfscale=dfscale,
                         returnMatrix=returnMatrix)

    #  2.2.  set up output object
    dimc1   <- dim(sb1$fd$coefs)
    dimc    <- c(dimc1[1], dimy[-1])
    coefs   <- array(NA, dim=dimc)
    c1names <- dimnames(sb1$fd$coefs)
    cNames  <- vector("list", 2)
    if (!is.null(c1names[[1]])) cNames[[1]] <- c1names[[1]]
    if (!is.null(ynames[[2]]))  cNames[[2]] <- ynames[[2]]
    dimnames(coefs) <- cNames
    coefs[, 1] <- sb1$fd$coefs
    if (!is.null(covariates)) {
      q <- dim(covariates)[2]
      beta. <- matrix(0,q,dimy[2])
      beta.[,1] <- sb1$beta
    } else {
      beta. <- NULL
    }
    #   now loop through remaining records, smoothing each in term,
    #   using argvals[,1]
    for (i in seq(2, length=dimy[2]-1)) {

      sbi <- smooth.basis1(argvals[, i], y=y[, i], fdParobj=fdParobj,
                           wtvec=wtvec,   fdnames=fdnames,
                           covariates=covariates,
                           method=method, dfscale=dfscale,
                           returnMatrix=returnMatrix)

      coefs[, i] <- sbi$fd$coefs
      if (!is.null(covariates)) {
        beta.[,i] <- sbi$beta
      }
    }
    if (is.null(fdnames)) {
      fdnames <- sb1$fdnames
      if (is.null(fdnames))
        fdnames <- list(time=NULL, reps=NULL, values="value")
      valueChk <- ((length(fdnames$values)==1)
                   && (fdnames$values=="value")
                   && (length(fdnames$reps)==1)
                   && (!is.null(ynames[[2]])) )
      if (valueChk)fdnames$values <- fdnames$reps
      if (!is.null(ynames[[2]]))
        fdnames[[2]] <- ynames[[2]]
    }
} else {
##
## 3.  ndy == 3
##
    #  3.1.  argvals[, 1]
    sb1 <- smooth.basis1(argvals[, 1], y=y[, 1, ], fdParobj=fdParobj,
                         wtvec=wtvec,   fdnames=fdnames,
                         covariates=covariates,
                         method=method, dfscale=dfscale,
                         returnMatrix=returnMatrix)
    #  3.2.  set up output object
    coef1 <- sb1$fd$coefs
    dimc1 <- dim(coef1)
    dimc <- c(dimc1[1], dimy[-1])
    coefs <- array(NA, dim=dimc)
    yNames <- dimnames(y)
    c1Names <- dimnames(coef1)
    cNames <- vector("list", 3)
    if (!is.null(c1Names[[1]]))  cNames[[1]] <- c1Names[[1]]
    if (!is.null(yNames[[2]]))   cNames[[2]] <- yNames[[2]]
    if (is.null(c1Names[[2]])) {
      if (!is.null(yNames[[3]])) cNames[[3]] <- yNames[[3]]
    } else {
      cNames[[3]] <- c1Names[[2]]
    }
    dimnames(coefs) <- cNames
    coefs[, 1, ] <- coef1
    if (!is.null(covariates)) {
      q <- dim(covariates)[2]
      beta. <- array(0,c(q,dimy[2],dimy[3]))
      beta.[,,1] <- sb1$beta
    } else {
      beta. <- NULL
    }
    #
    for (i in seq(2, length=dimy[2]-1)) {
      sbi <- smooth.basis1(argvals[, i], y=y[, i, ], fdParobj=fdParobj,
                           wtvec=wtvec,   fdnames=fdnames,
                           covariates=covariates,
                           method=method, dfscale=dfscale)
      coefs[, i, ] <- sbi$fd$coefs
      if (!is.null(covariates)) {
        beta.[,,i] <- sbi$beta
      } else {
        beta. <- NULL
      }
    }
    if (is.null(fdnames)) {
      fdnames <- sb1$fdnames
      if (is.null(fdnames)) {
        fdnames <- list(time=NULL, reps=NULL, values=NULL)
        if (!is.null(argNames[[1]])) {
          fdnames[[1]] <- argNames[[1]]
        } else {
          fdnames[[1]] <- ynames[[1]]
        }
        if (!is.null(ynames[[2]]))fdnames[[2]] <- ynames[[2]]
        if (!is.null(ynames[[3]]))fdnames[[3]] <- ynames[[3]]
      }
    }
}
##
## 4.  done
##
sb <- sb1
sb$beta       <- beta.
sb$fd$coefs   <- coefs
sb$fd$fdnames <- fdnames
sb

}

############################################################################

smooth.basis3 <- function(argvals=array(1:n,c(n,N,M)), y, fdParobj,
                          wtvec=NULL,   fdnames=NULL, covariates=NULL,
                          method="chol", dfscale=1, returnMatrix=FALSE)
{
##
## 1.  check dimensions of argval and y
##

dimy <- dim(y)
ndy <- length(dimy)
n   <- dimy[1]
N   <- dimy[2]
M   <- dimy[3]
if (ndy < 3)stop("length(dim(y)) must be 3  is ", ndy)
if (any(dima != dimy)) {
    stop("dim(argvals) = ", paste(dima, collapse=", "),
         " != dim(y) = ", paste(dimy, collapse=", "))
}

dima <- dim(argvals)
nda  <- length(dima)
if (nda < 3) stop("length(dim(argvals)) must be 3  is ", nda)

##
## 2.  Call smooth.basis2 repeatedly
##
#  2.1.  argvals[, , 1]
sb1 <- smooth.basis2(argvals[, , 1], y=y[, , 1], fdParobj=fdParobj,
                     wtvec=wtvec,   fdnames=fdnames,
                     covariates=covariates,
                     method=method, dfscale=dfscale,
                     returnMatrix=returnMatrix)
#  2.2.  set up output object
coef1 <- sb1$fd$coefs
dimc1 <- dim(coef1)
dimc  <- c(dimc1[1], dimy[-1])
coefs <- array(NA, dim=dimc)
argNames <- dimnames(argvals)
yNames   <- dimnames(y)
c1Names  <- dimnames(coef1)
cNames   <- vector("list", 3)
if (!is.null(c1Names[[1]])) cNames[[1]] <- c1Names[[1]]
if (!is.null(yNames[[2]]))  cNames[[2]] <- yNames[[2]]
if (!is.null(yNames[[3]]))  cNames[[3]] <- yNames[[3]]
dimnames(coefs) <- cNames
if (!is.null(covariates)) {
  q <- dim(covariates)[2]
  beta. <- array(0,c(q,dimy[2],dimy[3]))
  beta.[,,1] <- sb1$beta
} else {
  beta. <- NULL
}
#
for (i in seq(2, length=dimy[3]-1)) {
    sbi <- smooth.basis2(argvals[,,i], y=y[,,i], fdParobj=fdParobj,
                         wtvec=wtvec,   fdnames=fdnames,
                         covariates=covariates,
                         method=method, dfscale=dfscale,
                         returnMatrix=returnMatrix)
    coefs[,,i] <- sbi$fd$coefs
    if (!is.null(covariates)) {
      beta.[,,i] <- sbi$beta
    }
}
if (is.null(fdnames)) {
    fdnames <- list(time=NULL, reps=NULL, values=NULL)
    if (!is.null(yNames[[1]])) {
        fdnames[[1]] <- yNames[[1]]
    } else {
      if (!is.null(argNames[[1]]))
        fdnames[[1]] <- argNames[[1]]
    }
    if (!is.null(yNames[[2]])) {
        fdnames[[2]] <- yNames[[2]]
    } else {
      if (!is.null(argNames[[2]]))
        fdnames[[2]] <- argNames[[2]]
    }
    if (!is.null(yNames[[3]])) {
        fdnames[[3]] <- yNames[[3]]
    } else {
      if (!is.null(argNames[[3]]))
        fdnames[[3]] <- argNames[[3]]
    }
}
##
## 3.  done
##
sb <- sb1
sb$fd$coefs   <- coefs
sb$fd$fdnames <- fdnames
sb$beta       <- beta.
sb

}


smooth.basis.sparse <- function(argvals, y, fdParobj, fdnames=NULL, covariates=NULL, 
                                method="chol", dfscale=1 ){
  
  #  Arguments:
  # ARGVALS  A set of N argument values, set by default to equally spaced
  #             on the unit interval (0,1).
  # Y        an array containing values of curves
  #             If the array is a matrix, rows must correspond to argument
  #             values and columns to replications, and it will be assumed
  #             that there is only one variable per observation.
  #             If Y is a three-dimensional array, the first dimension
  #             corresponds to argument values, the second to replications,
  #             and the third to variables within replications.
  #             If Y is a vector, only one replicate and variable are assumed.
  # FDPAROBJ A functional parameter or fdPar object.  This object
  #             contains the specifications for the functional data
  #             object to be estimated by smoothing the data.  See
  #             comment lines in function fdPar for details.
  #             This argument may also be either a FD object, or a
  #             BASIS object.  In this case, the smoothing parameter
  #             LAMBDA is set to 0.
  # FDNAMES  A cell of length 3 with names for
  #             1. argument domain, such as "Time"
  #             2. replications or cases
  #             3. the function.
  # COVARIATES  A N by Q matrix Z of covariate values used to augment
  #             the smoothing function, where N is the number of
  #             data values to be smoothed and Q is the number of
  #             covariates.  The process of augmenting a smoothing
  #             function in this way is often called "semi-parametric
  #             regression".  The default is the null object NULL.
  # METHOD      The method for computing coefficients.  The usual method
  #             computes cross-product matrices of the basis value matrix,
  #             adds the roughness penalty, and uses the Choleski
  #             decomposition of this to compute coefficients, analogous
  #             to using the normal equations in least squares fitting.
  #             But this approach, while fast, contributes unnecessary
  #             rounding error, and the qr decomposition of the augmented
  #             basis matrix is prefererable.  But nothing comes for free,
  #             and the computational overhead of the qr approach can be a
  #             serious problem for large problems (n of 1000 or more).
  #             For this reason, the default is "method" = "chol", but if
  #             'method' == 'qr', the qr decomposition is used.
  # DFFACTOR A multiplier of df in GCV, set to one by default
  #
  # Returns a list containing:
  #   FDOBJ   an object of class fd containing coefficients.
  #   DF      a degrees of freedom measure.
  #   GCV     a measure of lack of fit discounted for df.
  #              If the function is univariate, GCV is a vector
  #              containing the error  sum of squares for each
  #              function, and if the function is multivariate,
  #              GCV is a NVAR by NCURVES matrix.
  #   COEF    the coefficient matrix for the basis function
  #                expansion of the smoothing function
  #   SSE     the error sums of squares.
  #              SSE is a vector or matrix of the same size as
  #              GCV.
  #   PENMAT  the penalty matrix.
  #   Y2CMAP  the matrix mapping the data to the coefficients.
  
  if (is.fdPar(fdParobj)) {
    basisobj = fdParobj$fd$basis
  } else {
    if (is.fd(fdParobj)) {
      basisobj = fdParobj$basis
    } else {
      if (is.basis(fdParobj)) {
        basisobj = fdParobj
      } else {
        stop("fdParobj is not a fdPar, fd, or a basis object.")
      }
    }
  }
  if(length(dim(y)) == 2){
	coefs = matrix(0, nrow = basisobj$nbasis, ncol = dim(y)[2])
	for(i in 1:dim(y)[2]){
		curve = y[,i]
		curve.smooth = smooth.basis(argvals[!is.na(curve)],curve[!is.na(curve)],
                                basisobj, covariates, method)
		coefs[,i] = curve.smooth$fd$coefs
	}
  } else if(length(dim(y)) == 3){
 	coefs = array(0, c(basisobj$nbasis,dim(y)[2:3]))
	for(i in 1:dim(y)[2]){
		for(j in 1:dim(y)[3]){
			curve = y[,i,j]
			curve.smooth = smooth.basis(argvals[!is.na(curve)],curve[!is.na(curve)],
                                basisobj, covariates, method)
			coefs[,i,j] = curve.smooth$fd$coefs
		}
	} 
  }
  datafd = fd(coefs,basisobj, fdnames)
  return(datafd)
}

smooth.bibasis <- function (sarg, targ, y, fdPars, fdPart, fdnames=NULL,
                            returnMatrix=FALSE)
{
#  SMOOTH_BIBASIS  Smooths discrete surface values over a rectangular
#  lattice to estimate a smoothing function f(s,t)
#   using penalized basis functions.
#
#  Arguments for this function:
#
#  sarg     ... A set of argument values for the row dimension s.
#  targ     ... A set of argument values for the col dimension t.
#  Y        ... an array containing surface values.  If two-dimensional,
#               a single surface is assumed.  If three-dimensional,
#               the third dimension corresponds to replications.
#  FDPARS   ... A functional parameter or fdPar object for
#               variation along the row dimension s.
#  FDPART   ... A functional parameter or fdPar object for
#               variation along the col dimension t.
#  FDNAMES  ... A cell of length 3 with names for
#               1. argument s
#               2. argument t
#               3. the function f(s,t).
#  RETURNMATRIX ... If False, a matrix in sparse storage model can be returned
#               from a call to function BsplineS.  See this function for
#               enabling this option.

# Returns a list containing:
#   FDOBJ ...  an object of class fd containing coefficients.
#   DF    ...  a degrees of freedom measure.
#   GCV   ...  a measure of lack of fit discounted for df.
#              If the function is univariate, GCV is a vector
#              containing the error  sum of squares for each
#              function, and if the function is multivariate,
#              GCV is a NVAR by NCURVES matrix.
#   COEF  ...  the coefficient matrix for the basis function
#                expansion of the smoothing function
#   SSE   ...  the error sums of squares.
#              SSE is a vector or matrix of the same size as
#              GCV.
#   PENMAT...  the penalty matrix.
#   Y2CMAP...  the matrix mapping the data to the coefficients.

# last modified 16 Novembeer 2021 by Jim Ramsay

#  ---------------------------------------------------------------------
#                      Check argments
#  ---------------------------------------------------------------------

#  check argument values

sarg = argcheck(sarg)
targ = argcheck(targ)
ns   = length(sarg)
nt   = length(targ)

#  check Y

if(!inherits(y, "matrix") && !inherits(y, "array"))
    stop("'y' is not of class matrix or class array.")

ydim = dim(y)

if (ydim[1] != ns) stop(
    "Number of rows of Y is not the same length as SARG.")
if (ydim[2] != nt) stop(
    "Number of columns of Y is not the same length as TARG.")
if (length(ydim) == 2) {
    nsurf = 1
    ymat = matrix(y, ns*nt, 1)
} else {
    nsurf = ydim(3)
    ymat = matrix(0, ns*nt, nsurf)
    for (isurf in 1:nsurf)
        ymat[,isurf] = matrix(y[,,isurf], ns*nt, 1)
}

  #  check FDPARS, FDPART and BASES, LBFDOBJ"S and LAMBDA"S

fdPars  = fdParcheck(fdPars,nsurf)
fdobjs  = fdPars$fd
sbasis  = fdobjs$basis
snbasis = sbasis$nbasis - length(sbasis$dropind)
lambdas = fdPars$lambda
Lfds    = fdPars$Lfd

fdPart  = fdParcheck(fdPart,nsurf)
fdobjt  = fdPart$fd
tbasis  = fdobjt$basis
tnbasis = tbasis$nbasis - length(tbasis$dropind)
lambdat = fdPart$lambda
Lfdt    = fdPart$Lfd

#  check LAMBDA

if (lambdas < 0) {
    warning ("Value of lambdas was negative, 0 used instead.")
    lambdas = 0
}
if (lambdat < 0) {
    warning ("Value of lambdat was negative, 0 used instead.")
    lambdat = 0
}

#  set default argument values

if (is.null(fdnames)) {
    fdnames      = vector("list", 3)
    fdnames[[1]] = "argument s"
    fdnames[[2]] = "argument s"
    fdnames[[3]] = "function"
}

#  ----------------------------------------------------------------
#                set up the linear equations for smoothing
#  ----------------------------------------------------------------

sbasismat = eval.basis(sarg, sbasis, 0, returnMatrix)
tbasismat = eval.basis(targ, tbasis, 0, returnMatrix)
basismat  = kronecker(tbasismat,sbasismat)

if (ns*nt > snbasis*tnbasis || lambdas > 0 || lambdat > 0) {

    #  The following code is for the coefficients completely determined

    Bmat  = crossprod(basismat,basismat)

    #  set up right side of equations

    Dmat = crossprod(basismat,ymat)

    #  set up regularized cross-product matrix BMAT

    if (lambdas > 0) {
      penmats  = eval.penalty(sbasis, Lfds)
      Bnorm   = sqrt(sum(c(Bmat)^2))
      pennorm = sqrt(sum(c(penmats)^2))
      condno  = pennorm/Bnorm
      if (lambdas*condno > 1e12) {
        lambdas = 1e12/condno
        warning(paste("lambdas reduced to",lambdas,
                      "to prevent overflow"))
      }
      Imat = diag(rep(nt,1))
      Bmat = Bmat + lambdas*kronecker(Imat,penmats)
    }

    if (lambdat > 0) {
      penmatt  = eval.penalty(tbasis, Lfdt)
      Bnorm   = sqrt(sum(c(Bmat)^2))
      pennorm = sqrt(sum(c(penmatt)^2))
      condno  = pennorm/Bnorm
      if (lambdat*condno > 1e12) {
        lambdat = 1e12/condno
        warning(paste("lambdat reduced to",lambdat,
                      "to prevent overflow"))
      }
      Imat = diag(rep(ns,1))
      Bmat = Bmat + lambdat*kronecker(penmatt,Imat)
    }

    #  compute inverse of Bmat

    Bmat = (Bmat+t(Bmat))/2
    Lmat = chol(Bmat)
    # Lmat = try(chol(Bmat), silent=TRUE) {
    #   if (class(Lmat)=="try-error") {
    #     Beig = eigen(Bmat, symmetric=TRUE)
    #     BgoodEig = (Beig$values>0)
    #     Brank = sum(BgoodEig)
    #     if (Brank < dim(Bmat)[1])
    #       warning("Matrix of basis function values has rank ",
    #               Brank, " < dim(fdobj$basis)[2] = ",
    #               length(BgoodEig), "  ignoring null space")
    #     goodVec = Beig$vectors[, BgoodEig]
    #     Bmatinv = (goodVec %*% (Beig$values[BgoodEig] * t(goodVec)))
    #   } else {
        Lmatinv = solve(Lmat)
        Bmatinv = Lmatinv %*% t(Lmatinv)
    #   }
    # }

    #  ----------------------------------------------------------------
    #       Compute the coefficients defining the smooth and
    #            summary properties of the smooth
    #  ----------------------------------------------------------------

    #  compute map from y to c

    y2cMap = Bmatinv %*% t(basismat)

    #  compute degrees of freedom of smooth

    BiB0 = Bmatinv %*% Bmat

    df = sum(diag(BiB0))

    #  solve normal equations for each observation

    coef = solve(Bmat, Dmat)
    if (nsurf == 1) {
        coefmat = matrix(coef, snbasis, tnbasis)
    } else {
        coefmat = array(0, c(snbasis, tnbasis, nsurf))
        for (isurf in 1:nsurf)
            coefmat[,,isurf] = matrix(coef[,isurf], snbasis, tnbasis)
    }

} else {
      stop(paste("The number of basis functions exceeds the number of ",
                 "points to be smoothed."))

}

#  ----------------------------------------------------------------
#            compute SSE, yhat, GCV and other fit summaries
#  ----------------------------------------------------------------

#  compute error sum of squares

yhat = basismat %*% coef
SSE  = sum((ymat - yhat)^2)

#  compute  GCV index

N = ns*nt*nsurf
if (df < N) {
    gcv = (SSE/N)/((N - df)/N)^2
} else {
    gcv = NA
}

#  ------------------------------------------------------------------
#          Set up the functional data objects for the smooths
#  ------------------------------------------------------------------

bifdobj = bifd(coefmat, sbasis, tbasis, fdnames)

smoothlist = list(bifdobj=bifdobj,  df=df,           gcv=gcv,
                  SSE=SSE,          penmats=penmats, penmatt = penmatt,
                  y2cMap=y2cMap,    sarg=sarg,       targ=targ,
                  y=y,              coef = coefmat)

#  class(smoothlist) = "bifdSmooth"
return(smoothlist)

}
smooth.fdPar <- function(fdobj, Lfdobj=NULL,
         lambda=1e-4, estimate=TRUE, penmat=NULL){
##
## 1.  fdPar
##
  fdP <- fdPar(fdobj, Lfdobj=Lfdobj, lambda=lambda,
               estimate=estimate, penmat=penmat)
##
## 2.  smooth.fd
##
  return(smooth.fd(fdobj, fdP))
}
smooth.fd <- function(fdobj, fdParobj){
#SMOOTH_FD smooths a functional data object.
#
#  Arguments for this function:
#
#  FDOBJ    ... A functional data object.
#  FDPAROBJ ... A functional parameter object.
#
#  Returns a functional data object containing a smoothed version
#    of the input functional data object
#
#  Last modified: 20081003;  previously modified 26 October 2005
#

#  check fdParobj

if (!inherits(fdParobj,"fdPar")) stop(
		"FDPAROBJ is not a functional parameter object.")

#  check LFD

Lfdobj <- fdParobj$Lfd
Lfdobj <- int2Lfd(Lfdobj)
nderiv <- Lfdobj$nderiv

#  set up FDOBJ

newfdobj <- fdParobj$fd

#  set up basis

basisobj <- newfdobj$basis

#
#  Main smoothing step
#

coef  <- fdobj$coefs
coefd <- dim(coef)
ndim  <- length(coefd)
if (ndim == 3)  nvar <- coefd[3] else nvar <- 1

Bmat  <- inprod(basisobj, basisobj)

#
#  set up coefficient matrix for normal equations
#

lambda <- fdParobj$lambda
penmat <- eval.penalty(basisobj, Lfdobj)

penmat <- eval.penalty(basisobj, Lfdobj)
Cmat   <- Bmat + lambda * penmat

#
#  solve normal equations for each observation
#
if (ndim < 3){
     Dmat <- inprod(basisobj, fdobj)
     coef <- solve(Cmat, Dmat)
} else {
	coef <- array(0,coefd)
    for(ivar in 1:nvar){
        Dmat <- inprod(basisobj, fdobj[,ivar])
        coef[,,ivar] <- solve(Cmat, Dmat)
    }
}

#  set up the smoothed functional data object

fdnames <- fdobj$fdnames
smthfd  <- fd(coef, basisobj, fdnames)

return(smthfd)
}
smooth.monotone <- function(argvals, y, WfdParobj, wtvec=rep(1,n),
                            zmat=NULL, conv=.0001, iterlim=50,
                            active=rep(TRUE,nbasis), dbglev=1)
{
#  Smooths the relationship of Y to ARGVALS using weights in WTVEC by
#  fitting a monotone function of the form
#                   f(x) = b_0 + b_1 D^{-1} exp W(x)
#     where  W  is a function defined over the same range as ARGVALS,
#                 W + ln b_1 = log Df and w = D W = D^2f/Df.
#  The constant term b_0 in turn can be a linear combinations of
#  covariates:
#                         b_0 = zmat * c.
#  The fitting criterion is penalized mean squared error:
#    PENSSE(lambda) = \sum w_i[y_i - f(x_i)]^2 +
#                     \lambda * \int [L W(x)]^2 dx
#  where L is a linear differential operator defined in argument Lfdobj,
#  and w_i is a positive weight applied to the observation.
#  The function W(x) is expanded by the basis in functional data object
#  Wfdobj.   The coefficients of this expansion are called
#  "coefficients" in the comments, while the b's are called "regression
#  coefficients"

#  Arguments:
#  ARGVALS ...  Argument value array of length N, where N is the number
#               of observed curve values for each curve.  It is assumed
#               that these argument values are common to all observed
#               curves.  If this is not the case, you will need to
#               run this function inside one or more loops, smoothing
#               each curve separately.
#  y       ...  Function value array (the values to be fit).
#               If the functional data are univariate, this array will
#               be an N by NCURVE matrix, where N is the number of
#               observed curve values for each curve and NCURVE is the
#               number of curves observed.
#               If the functional data are mulivariate, this array will
#               be an N by NCURVE by NVAR matrix, where NVAR the number
#               of functions observed per case.  For example, for the
#               gait data, NVAR = 2, since we observe knee and hip
#               angles.
#  WFDPAROBJ... A functional parameter or fdPar object.  This object
#               contains the specifications for the functional data
#               object to be estimated by smoothing the data.  See
#               comment lines in function fdPar for details.
#               The functional data object WFD in WFDPAROBJ is used
#               to initialize the optimization process.
#               Its coefficient array contains the starting values for
#               the iterative minimization of mean squared error.
#  ZMAT    ...  An N by NCOV matrix of covariate values for the constant
#               term.  It defaults to NULL, in this case the constant
#               term is the value of BETA[1] for all values of a given
#               curve.
#  WTVEC   ...  A vector of weights, a vector of N one's by default.
#  CONV    ...  Convergence criterion, 0.0001 by default
#  ITERLIM ...  maximum number of iterations, 50 by default.
#  ACTIVE  ...  indices among 1:NBASIS of parameters to optimize.
#               Defaults to 1:NBASIS.
#  DBGLEV  ...  Controls the level of output on each iteration.  If 0,
#               no output, if 1, output at each iteration, if higher,
#               output at each line search iteration. 1 by default.

#  Returns are:
#  WFD     ...  Functional data object for W.
#               Its coefficient matrix an N by NCURVE (by NVAR) matrix
#               (or array), depending on whether the functional
#               observations are univariate or multivariate.
#  BETA    ...  The regression coefficients b_0 and b_1 for each
#               smoothed curve.
#               If the curves are univariate and
#                  ... ZMAT is NULL,       BETA is 2   by NCURVE.
#                  ... ZMAT has P columns, BETA is P+1 by NCURVE.
#               If the curves are multivariate and
#                  ... ZMAT is NULL,       BETA is 2   by NCURVE by NVAR.
#                  ... ZMAT has P columns, BETA is P+1 by NCURVE by NVAR.
#  YHATFD ...   A functional data object for the monotone curves that
#               smooth the data
#  FLIST  ...   A list object or a vector of list objects, one for
#               each curve (and each variable if functions are
#               multivariate).
#               Each list object has slots:
#                 f    ... The sum of squared errors
#                 grad ... The gradient
#                 norm ... The norm of the gradient
#  Y2CMAP ...   For each estimated curve (and variable if functions are
#               multivariate, this is an N by NBASIS matrix containing
#               a linear mappping from data to coefficients that can be
#               used for computing point-wise confidence intervals.
#               If NCURVE = NVAR = 1, a matrix is returned.  Otherwise
#               an NCURVE by NVAR list is returned, with each
#               slot containing this mapping.
#  When multiple curves and variables are analyzed, the lists containing
#  FLIST and Y2CMAP objects are indexed linear with curves varying
#  inside variables.

# last modified 16 November 2021 by Jim Ramsay

  #  check ARGVALS
  
  if (!is.numeric(argvals)) stop("ARGVALS is not numeric.")
  argvals <- as.vector(argvals)
  if (length(argvals) < 2) stop("ARGVALS does not contain at least two values.")
  n       <- length(argvals)
  onesobs <- matrix(1,n,1)
  
#  at least three points are necessary for monotone smoothing

if (n < 3) stop('ARGVALS does not contain at least three values.')

#  check Y

ychk   <- ycheck(y, n)
y      <- ychk$y
ncurve <- ychk$ncurve
nvar   <- ychk$nvar
ndim   <- ychk$ndim

#  check WfdParobj and get LAMBDA

WfdParobj <- fdParcheck(WfdParobj,ncurve)
lambda    <- WfdParobj$lambda

#  the starting values for the coefficients are in FD object WFDOBJ

Wfdobj   <- WfdParobj$fd
Lfdobj   <- WfdParobj$Lfd
basisobj <- Wfdobj$basis     #  basis for W(argvals)
nbasis   <- basisobj$nbasis  #  number of basis functions

#  set up initial coefficient array

coef0    <- Wfdobj$coefs
if( length(dim(coef0)) == 2 & nvar != 1 ){
	coef0 = array(0,c(nbasis,ncurve,nvar))
}
if( length(dim(coef0)) == 2 & ncol(coef0) != ncurve ){
	coef0 = matrix(0,nbasis,ncurve)	
}
if( length(dim(coef0)) == 3 & (all.equal(dim(coef0)[2:3],c(ncurve,nvar))!=TRUE) ){
	coef0 = array(0,c(nbasis,ncurve,nvar))	
}
# Note that we could be more carefull about this and try to adapt coefficients
# if they have something like the right shape, but I'm not sure we can do
# so in any reasonable way. 
	
#  check WTVEC

wtvec <- wtcheck(n, wtvec)$wtvec

#  check ZMAT

if (!is.null(zmat)) {
  zdim <- dim(zmat)
  if (zdim[1] != n) stop("First dimension of ZMAT not correct.")
  ncov   <- zdim[2]   #  number of covariates
} else {
  ncov <- 1
}

#  set up some variables

ncovp1 <- ncov + 1  #  index for regression coef. for monotone fn.
wtroot <- sqrt(wtvec)
wtrtmt <- wtroot %*% matrix(1,1,ncovp1)
yroot  <- y*as.numeric(wtroot)
climit <- c(-100*rep(1,nbasis), 100*rep(1,nbasis))
inact  <- !active   #  indices of inactive coefficients

#  set up list for storing basis function values

JMAX <- 15
basislist <- vector("list", JMAX)

#  initialize matrix Kmat defining penalty term

if (lambda > 0) {
  Kmat <- lambda*eval.penalty(basisobj, Lfdobj)
} else {
  Kmat <- matrix(0,nbasis,nbasis)
}

#  --------------------------------------------------------------------
#              loop through variables and curves
#  --------------------------------------------------------------------

#  set up arrays and lists to contain returned information

if (ndim == 2) {
    coef <- matrix(0,nbasis,ncurve)
    beta <- matrix(0,ncovp1,ncurve)
} else {
    coef <- array(0,c(nbasis,ncurve,nvar))
    beta <- array(0,c(ncovp1,ncurve,nvar))
}

if (ncurve > 1 || nvar > 1 ) {
    Flist <- vector("list",ncurve*nvar)
} else {
    Flist <- NULL
}

if (ncurve > 1 || nvar > 1)  {
    y2cMap <- vector("list",ncurve*nvar)
} else {
    y2cMap <- NULL
}

if (dbglev == 0 && ncurve > 1) cat("Progress:  Each dot is a curve\n")

for (ivar in 1:nvar) {
  for (icurve in 1:ncurve) {
    if (ndim == 2) {
        yi    <- y[,icurve]
        cveci <- coef0[,icurve]
    } else {
        yi    <- y[,icurve,ivar]
        cveci <- coef0[,icurve,ivar]
    }

  #  Compute initial function and gradient values

  result <- fngrad.smooth.monotone(yi, argvals, zmat, wtvec, cveci, lambda,
                                   basisobj, Kmat, inact, basislist)
  Flisti <- result[[1]]
  betai  <- result[[2]]
  Dyhat  <- result[[3]]

  #  compute the initial expected Hessian

  hessmat <- hesscal.smooth.monotone(betai, Dyhat, wtroot,
                                     lambda, Kmat, inact)

  #  evaluate the initial update vector for correcting the initial cveci

  result   <- linesearch.smooth.monotone(Flisti, hessmat, dbglev)
  deltac   <- result[[1]]
  cosangle <- result[[2]]

  #  initialize iteration status arrays

  iternum <- 0
  status  <- c(iternum, Flisti$f, Flisti$norm, betai)
  if (dbglev >= 1) {
    if (ncurve > 1 || nvar > 1) {
          if (ncurve > 1 && nvar > 1) {
            cat("\n")
            curvetitle <- paste('Results for curve',icurve,'and variable',ivar)
          }
          if (ncurve > 1 && nvar == 1) {
            cat("\n")
            curvetitle <- paste('Results for curve',icurve)
          }
          if (ncurve == 1 && nvar > 1) {
            cat("\n")
            curvetitle <- paste('Results for variable',ivar)
          }
    }
    else curvetitle <- 'Results'

    cat("\n",curvetitle,"\n")
    cat("\nIter.   PENSSE   Grad Length Intercept   Slope\n")
    cat(iternum)
    cat("        ")
    cat(round(status[2],4))
    cat("      ")
    cat(round(status[3],4))
    cat("      ")
    cat(round(betai[1],4))
    cat("      ")
    cat(round(betai[ncovp1],4))
  } else {
    cat(".")
  }

#  -------  Begin iterations  -----------

  MAXSTEPITER <- 10
  MAXSTEP     <- 100
  trial       <- 1
  reset       <- FALSE
  linemat     <- matrix(0,3,5)
  betaold     <- betai
  cvecold     <- cveci
  Foldlist    <- Flisti
  dbgwrd      <- dbglev >= 2

  if (iterlim > 0) {
  for (iter in 1:iterlim) {
      iternum <- iternum + 1
      #  initialize logical variables controlling line search
      dblwrd <- rep(FALSE,2)
      limwrd <- rep(FALSE,2)
      stpwrd <- FALSE
      ind    <- 0
      ips    <- 0
      #  compute slope at 0 for line search
      linemat[2,1] <- sum(deltac*Flisti$grad)
      #  normalize search direction vector
      sdg     <- sqrt(sum(deltac^2))
      deltac  <- deltac/sdg
      dgsum   <- sum(deltac)
      linemat[2,1] <- linemat[2,1]/sdg
      # initialize line search vectors
      linemat[,1:4] <- outer(c(0, linemat[2,1], Flisti$f),rep(1,4))
      stepiter <- 0
      if (dbglev >= 2) {
          cat("\n")
          cat(paste("                 ", stepiter, "  "))
          cat(format(round(t(linemat[,1]),6)))
      }
      #  break with error condition if initial slope is nonnegative
      if (linemat[2,1] >= 0) {
        if (dbgwrd >= 2) print("Initial slope nonnegative.")
        ind <- 3
        break
      }
      #  return successfully if initial slope is very small
      if (linemat[2,1] >= -1e-7) {
        if (dbglev >= 2) print("Initial slope too small")
        ind <- 0
        break
      }
      #  first step set to trial
      linemat[1,5]  <- trial
      #  Main iteration loop for linesearch
      for (stepiter in 1:MAXSTEPITER) {
        #  ensure that step does not go beyond limits on parameters
        limflg  <- FALSE
        #  check the step size
        result <-
              stepchk(linemat[1,5], cveci, deltac, limwrd, ind,
                      climit, active, dbgwrd)
        linemat[1,5] <- result[[1]]
        ind          <- result[[2]]
        limwrd       <- result[[3]]
        if (linemat[1,5] <= 1e-7)
        {
          #  Current step size too small ... terminate
          Flisti  <- Foldlist
          cvecnew <- cveci
          gvecnew <- Flisti$grad
          if (dbglev >= 2) {
            print("Stepsize too small")
            print(linemat[1,5])
          }
          if (limflg) ind <- 1 else ind <- 4
          break
        }
        #  compute new function value and gradient
        cvecnew <- cveci + linemat[1,5]*deltac
        result  <- fngrad.smooth.monotone(yi, argvals, zmat, wtvec, cvecnew, lambda,
                                          basisobj, Kmat, inact, basislist)
        Flisti  <- result[[1]]
        betai   <- result[[2]]
        Dyhat   <- result[[3]]
        linemat[3,5] <- Flisti$f
        #  compute new directional derivative
        linemat[2,5] <- sum(deltac*Flisti$grad)
        if (dbglev >= 2) {
          cat("\n")
          cat(paste("                 ", stepiter, "  "))
          cat(format(round(t(linemat[,5]),6)))
        }
        #  compute next line search step, also test for convergence
        result  <- stepit(linemat, ips, dblwrd, MAXSTEP)
        linemat <- result[[1]]
        ips     <- result[[2]]
        ind     <- result[[3]]
        dblwrd  <- result[[4]]
        trial   <- linemat[1,5]
        #  ind == 0  mean convergence
        if (ind == 0 | ind == 5) break
        #  end of line search loop
     }
     cveci  <- cvecnew
     #  check that function value has not increased
     if (Flisti$f > Foldlist$f) {
        # if it has, terminate iterations with a message
        if (dbglev >= 2) {
          cat("Criterion increased: ")
          cat(format(round(c(Foldlist$f, Flisti$f),4)))
          cat("\n")
        }
        #  reset parameters and fit
        betai        <- betaold
        cveci        <- cvecold
        Wfdobj$coefs <- cveci
        Flisti       <- Foldlist
        deltac       <- -Flisti$grad
        if (reset) {
          # This is the second time in a row that
          #  this has happened ... quit
          if (dbglev >= 2) cat("Reset twice, terminating.\n")
          break
        } else {
          reset <- TRUE
        }
     } else {
       if (abs(Foldlist$f - Flisti$f) < conv) {
	       if (dbglev >= 1) cat("\n")
	       break
       }
       cvecold  <- cveci
       betaold  <- betai
       Foldlist <- Flisti
       hessmat  <- hesscal.smooth.monotone(betai, Dyhat, wtroot,
                                           lambda, Kmat, inact)
       #  update the line search direction
       result   <- linesearch.smooth.monotone(Flisti, hessmat, dbglev)
       deltac   <- result[[1]]
       cosangle <- result[[2]]
       reset    <- FALSE
     }
     #  display iteration status
     status <- c(iternum, Flisti$f, Flisti$norm, betai)
     if (dbglev >= 1) {
        cat("\n")
        cat(iternum)
        cat("        ")
        cat(round(status[2],4))
        cat("      ")
        cat(round(status[3],4))
        cat("      ")
        cat(round(betai[1],4))
        cat("      ")
        cat(round(betai[ncovp1],4))
     }
    }
    }

    #  save coefficients in arrays COEF and BETA

    if (ndim == 2) {
        coef[,icurve] <- cveci
        beta[,icurve] <- betai
    } else {
        coef[,icurve,ivar] <- cveci
        beta[,icurve,ivar] <- betai
    }

    #  save Flisti if required in a list,
    #      indexed with curves varying inside variables.

    if (ncurve == 1 && nvar == 1) {
        Flist <- Flisti
    } else {
        Flist[[(ivar-1)*ncurve+icurve]] <- Flisti
    }

    #  save y2cMap if required in a list,
    #      indexed with curves varying inside variables.

    y2cMapij <- solve(crossprod(Dyhat) + lambda*Kmat) %*%
                    t(Dyhat)/sqrt(n)
    if (ncurve == 1 && nvar == 1) {
        y2cMap <- y2cMapij
    } else {
        y2cMap[[(ivar-1)*ncurve+icurve]] <- y2cMapij
    }
  }
}

Wfdobj <- fd(coef, basisobj)

#  Set up yhatfd, a functional data object for the monotone curves
#  fitting the data.
#  This can only be done if the covariate matrix ZMAT is NULL, meaning that
#  the same constant term is used for all curve values.

if (is.null(zmat)) {

  rangeval <- basisobj$rangeval
  narg     <- 10*nbasis+1
  evalarg  <- seq(rangeval[1], rangeval[2], len=narg)
  hmat     <- eval.monfd(evalarg, Wfdobj, 0)
  if (ndim == 2) {
    yhatmat <- matrix(0,narg,ncurve)
    for (icurve in 1:ncurve) {
      yhatmat[,icurve] <- beta[1,icurve] +
                          beta[2,icurve]*hmat[,icurve]
    }
    yhatcoef <- project.basis(yhatmat, evalarg, basisobj)
    yhatfd   <- fd(yhatcoef, basisobj)
  } else {
    yhatcoef <- array(0,c(nbasis,ncurve,nvar))
    yhatmati <- matrix(0,narg,ncurve)
    for (ivar in 1:nvar) {
      for (icurve in 1:ncurve) {
        yhatmati[,icurve] <- beta[1,icurve,ivar] +
                             beta[2,icurve,ivar]*hmat[,icurve,ivar]
      }
      yhatcoef[,,ivar] <- project.basis(yhatmati, evalarg, basisobj)
    }
    yhatfd <- fd(yhatcoef, basisobj)
  }
} else {
  yhatfd <- NULL
}

monFd <- list( "Wfdobj"  = Wfdobj,  "beta"   = beta, "yhatfd" = yhatfd,
               "Flist"   = Flist,   "y2cMap" = y2cMap,
               "argvals" = argvals, "y"      = y )
class(monFd) <- 'monfd'
monFd
}

#  ----------------------------------------------------------------

linesearch.smooth.monotone <- function(Flisti, hessmat, dbglev)
{
  deltac   <- -symsolve(hessmat,Flisti$grad)
  cosangle <- -sum(Flisti$grad*deltac)/sqrt(sum(Flisti$grad^2)*sum(deltac^2))
  if (dbglev >= 2) {
    cat(paste("\nCos(angle) =",format(round(cosangle,4))))
    if (cosangle < 1e-7) {
      if (dbglev >=2)  cat("\nCosine of angle too small\n")
      deltac <- -Flisti$grad
    }
  }
  return(list(deltac, cosangle))
}

#  ----------------------------------------------------------------

fngrad.smooth.monotone <- function(yi, argvals, zmat, wtvec, cveci, lambda,
                                   basisobj, Kmat, inact, basislist)
{
  if (!is.null(zmat)) {
    ncov   <- ncol(zmat)
    ncovp1 <- ncov + 1
  } else {
    ncov   <- 1
    ncovp1 <- 2
  }
  n      <- length(argvals)
  nbasis <- basisobj$nbasis
  Wfdobj <- fd(cveci, basisobj)
  h      <- monfn(argvals, Wfdobj, basislist)
  Dyhat  <- mongrad(argvals, Wfdobj, basislist)
  if (!is.null(zmat)) {
    xmat <- cbind(zmat,h)
  } else {
    xmat <- cbind(matrix(1,n,1),h)
  }
  Dxmat  <- array(0,c(n,ncovp1,nbasis))
  Dxmat[,ncovp1,] <- Dyhat
  wtroot <- sqrt(wtvec)
  wtrtmt <- wtroot %*% matrix(1,1,ncovp1)
  yroot  <- yi*as.numeric(wtroot)
  xroot  <- xmat*wtrtmt
  #  compute regression coefs.
  betai  <- lsfit(xmat, yi, wt=as.vector(wtvec), intercept=FALSE)$coef
  #  update fitted values
  yhat   <- xmat %*% betai
  #  update residuals and function values
  res    <- yi - yhat
  f      <- mean(res^2*wtvec)
  grad   <- matrix(0,nbasis,1)
  #print(betai)
  for (j in 1:nbasis) {
    Dxroot <- Dxmat[,,j]*wtrtmt
    yDx <- crossprod(yroot,Dxroot) %*% betai
    xDx <- crossprod(xroot,Dxroot)
    #print(crossprod(betai,(xDx+t(xDx))))
    #print(2*yDx)
    grad[j] <- crossprod(betai,(xDx+t(xDx))) %*% betai - 2*yDx
  }
  grad <- grad/n
  if (lambda > 0) {
    grad <- grad +          2 * Kmat %*% cveci
    f    <- f    + t(cveci) %*% Kmat %*% cveci
  }
  if (any(inact)) grad[inact] <- 0
  norm <- sqrt(sum(grad^2)) #  gradient norm
  Flisti <- list("f"=f,"grad"=grad,"norm"=norm)
  return(list(Flisti, betai, Dyhat))
}

#  ----------------------------------------------------------------

hesscal.smooth.monotone <- function(betai, Dyhat, wtroot, lambda,
                                    Kmat, inact)
{
  nbet    <- length(betai)
  Dydim   <- dim(Dyhat)
  n       <- Dydim[1]
  nbasis  <- Dydim[2]
  temp    <- betai[nbet]*Dyhat
  temp    <- temp*(wtroot %*% matrix(1,1,nbasis))
  hessmat <- 2*crossprod(temp)/n
  #  adjust for penalty
  if (lambda > 0) hessmat <- hessmat + 2*Kmat
  #  adjust for inactive coefficients
  if (any(inact)) {
    eyemat               <- diag(rep(1,nbasis))
    hessmat[inact,     ] <- 0
    hessmat[     ,inact] <- 0
    hessmat[inact,inact] <- eyemat[inact,inact]
  }
  return(hessmat)
}
smooth.morph <- function(x, y, ylim, WfdPar,   
                         conv=1e-4, iterlim=20, dbglev=0) {
  #  SMOOTH_MORPH smooths the relationship of Y to X 
  #  by fitting a monotone fn.  f(x) <- b_0 + b_1 D^{-1} exp W(t)
  #     where  W  is a function defined over the same range as X,
  #  W + ln b_1 <- log Df and w <- D W <- D^2f/Df.
  #  b_0 and b_1 are chosen so that values of f
  #  are within the interval [ylim[1],ylim[2]].
  #  The fitting criterion is penalized mean squared stop:
  #    PENSSE(lambda) <- \sum [y_i - f(t_i)]^2 +
  #                     \lambda * \int [L W]^2 
  #  W(x) is expanded by the basis in functional data object Wfdobj.
  #
  #  Arguments are 
  #  X         argument value array
  #  Y         data array containing the the values to be fit
  #  YLIM      Ordinate value limits. The values of the estimated function
  #               range between these limits.  The abscissa range is 
  #               defined in WfdPar.
  #  WFDPAR   A functional parameter or fdPar object.  This object 
  #               contains the specifications for the functional data
  #               object to be estimated by smoothing the data.  See
  #               comment lines in function fdPar for details.
  #               The functional data object WFD in FDPAROBJ is used
  #               to initialize the optimization process.
  #               It's coefficient array has a single column, and these 
  #               are the starting values for the iterative minimization 
  #               of mean squared stop.
  #               This argument may also be either a FD object, or a 
  #               BASIS object.  In this case, the smoothing parameter 
  #               LAMBDA is set to 0.
  #  WT        a vector of weights
  #  CONV      convergence criterion, 0.0001 by default
  #  ITERLIM   maximum number of iterations, 20 by default
  #  DBGLEV    Controls the level of output on each iteration.  If 0,
  #               no output, if 1, output at each iteration, if higher, 
  #               output at each line search iteration. 1 by default.
  #
  #  Returns are:
  #  WFD       Functional data object for W.  It's coefficient vector
  #               contains the optimized coefficients.
  
  #  last modified 17 May 2023 by Jim Ramsay
  
  #  number of observations and weights on x-values
  
  nobs <- length(x)        
  wt   <- matrix(1,nobs,1)
  wt[nobs] <- 10
  
  #  -----------------------------------------------------
  #                  Check arguments
  #  -----------------------------------------------------
  
  # check consistency of x and y and convert to column matrices
  
  if (length(y) != nobs) {
    stop('Arguments X and Y are not of same length')
  }
  
  if (!is.matrix(x)) x <- matrix(x,nobs,1)
  if (!is.matrix(y)) y <- matrix(y,nobs,1)
  
  #  check ylim for not being numeric
  
  if (!is.numeric(ylim)) {
    print("The third argument ylim is not numeric.  This argument
          should be a vector of length 2 containing the boundaries of
          the target interval.")
    stop("This argument has been added to allow morphs between two unequal invervals.")
  }
  
  #  check ylim for not having two strictly increasing numbers
  
  if (length(ylim) != 2 || ylim[1] >= ylim[2])
    stop("Argument ylim does not containing two strictly increasing numbers.")
  
  #  check WfdPar
  
  if (!is.fdPar(WfdPar)) {
    if (is.fd(WfdPar)) {
      WfdPar <- fdPar(WfdPar)
    } else {
      stop(paste("WFDPAR is not a functional parameter object,", 
                 "and not a functional data object."))
    }
  }
  
  #  -----------------------------------------------------
  #                  Initialize optimization
  #  -----------------------------------------------------
  
  #      extract information from WfdPar
  
  Wfdobj   <- WfdPar$fd
  cvec     <- Wfdobj$coef   #  initial coefficients
  Wbasis   <- Wfdobj$basis      #  basis for Wfdobj
  Wnbasis  <- Wbasis$nbasis      #  no. basis functions
  Wrange   <- Wbasis$rangeval
  Wtype    <- Wbasis$type
  xlim     <- Wbasis$rangeval
  WLfdobj  <- int2Lfd(WfdPar$Lfd)
  Wlambda  <- WfdPar$lambda
  if (any(wt < 0)) { 
    stop("One or more weights are negative.") 
  }
  
  #  check that values in x are within limits in xlim
  
  if (abs(x[1] - xlim[1]) > 1e-7 || abs(x[nobs] - xlim[2]) > 1e-7) {
    print("Argument vector x is out of range:")
    stop("Values are out of bounds by more than 1e-7")
  }  else {
    x[1   ] <- xlim[1]
    x[nobs] <- xlim[2]
  }
  
  #  transform coefficients to zero column sum 
  
  Zmat <- fda::zerobasis(length(cvec))
  bvec <- t(Zmat) %*% cvec
  cvec <- Zmat %*% bvec
  
  #  initialize matrix Kmat defining penalty term
  
  if (Wlambda > 0) {
    Kmat <- Wlambda*eval.penalty(Wbasis, WLfdobj)
  } else {
    Kmat  <- matrix(0,Wnbasis,Wnbasis)
  }
  
  #  load objects into morphList, used in fngrad_morph
  
  morphList <- list(x=x, y=y, xlim=xlim, ylim=ylim, wt=wt, Kmat=Kmat, 
                    Wlambda=Wlambda, Wbasis=Wbasis)
  
  #  -----------------------------------------------------
  #       Compute initial function and gradient values
  #  -----------------------------------------------------
  
  fnList <- fngrad_morph(bvec, morphList, Zmat)   
  f    <- fnList$f
  grad <- fnList$grad 
  hmat <- fnList$hmat
  norm <- fnList$norm
  
  #  compute initial badness of fit measures
  
  fold    <- f
  cvecold <- cvec
  
  #  evaluate the initial update vector 
  
  pvec <- -solve(hmat,grad)
  
  #  -----------------------------------------------------
  #  initialize iteration status arrays
  #  -----------------------------------------------------
  
  iternum <- 0
  status <- c(iternum, fold, norm)
  if (dbglev >= 1) {
    cat("\nIter.   PENSSE   Grad Length")
    cat("\n")
    cat(iternum)
    cat("        ")
    cat(round(status[2],4))
    cat("      ")
    cat(round(status[3],4))
  }
  iterhist <- matrix(0,iterlim+1,length(status))
  iterhist[1,]  <- status
  
  STEPMAX <- 10
  itermax <- 20
  TOLX    <- 1e-10
  fold    <- f
  
  #  -----------------------------------------------------
  #  --------  beginning of optimization loop  -----------
  #  -----------------------------------------------------
  
  for (iter in 1:iterlim) {
    iternum <- iternum + 1
    #  line search
    bvecold <- t(Zmat) %*% cvecold
    lnsrchList <- lnsrch_morph(bvecold, fold, grad, pvec, fngrad_morph, 
                                morphList, Zmat, STEPMAX, itermax, TOLX, 
                                dbglev)
    bvec   <- lnsrchList$x
    check  <- lnsrchList$check
    fnList <- fngrad_morph(bvec, morphList, Zmat)   
    f    <- fnList$f
    grad <- fnList$grad 
    hmat <- fnList$hmat
    norm <- fnList$norm
    cvec <- Zmat %*% bvec
    status <- c(iternum, f, norm)
    iterhist[iter+1,] <- status
    if (dbglev >= 1) {
      cat("\n")
      cat(iternum)
      cat("        ")
      cat(round(status[2],4))
      cat("      ")
      cat(round(status[3],4))
    }
    if (abs(f-fold) < conv) {
      break
    }
    if (f >= fold) { 
      warning("Current function value does not decrease fit.")
      cat("\n")
      break 
    }
    #  evaluate the update vector
    pvec <- -solve(hmat,grad)
    cosangle <- -t(grad) %*% pvec/sqrt(sum(grad^2)*sum(pvec^2))
    if (cosangle < 0) {
      if (dbglev > 1) {
        print("cos(angle) negative") 
      }
      pvec <- -grad
    }
    fold <- f
    cvecold <- cvec
  }
  
  #  -----------------------------------------------------
  #    construct output objects and return in a list
  #  -----------------------------------------------------
  
  Wfdobj  <- fd(cvec, Wbasis)
  
  xfine   <- as.matrix(seq(xlim[1],xlim[2],len=101))
  ywidth  <- ylim[2] - ylim[1]
  hfine   <- matrix(monfn(xfine, Wfdobj), 101, 1)
  hmax    <- monfn(xlim[2], Wfdobj)
  hmin    <- monfn(xlim[1], Wfdobj)
  hwidth  <- hmax - hmin
  hfine   <- (ylim[1] - hmin) + hfine*(ywidth/hwidth)
  
  return(list(Wfdobj=Wfdobj, f=f, grad=grad, hmat=hmat, 
              norm=norm, hfine=hfine, 
              iternum=iternum, iterhist=iterhist))
  
}

#  ------------------------------------------------------------------------

fngrad_morph <- function(bvec, morphList, Zmat) {
  #  -----------------------------------------------------
  #  extract data from morphList
  #  -----------------------------------------------------
  cvec     <- Zmat %*% bvec
  x        <- morphList$x
  y        <- morphList$y
  xlim     <- morphList$xlim
  ylim     <- morphList$ylim
  wt       <- morphList$wt
  Kmat     <- morphList$Kmat
  Wlambda  <- morphList$Wlambda
  Wbasis   <- morphList$Wbasis
  Wnbasis  <- Wbasis$nbasis
  Wfdobj   <- fd(cvec, Wbasis)
  #  -----------------------------------------------------
  #             compute fitting criterion f
  #  -----------------------------------------------------
  #  compute unnormalized monotone function values hraw
  
  nobs  <- length(x)
  hraw  <- matrix(monfn(  x, Wfdobj),nobs,1)
  #  adjust functions and derivatives for normalization
  hmax    <- monfn(  xlim[2], Wfdobj) 
  hmin    <- monfn(  xlim[1], Wfdobj) 
  #  width of hraw
  hwidth  <- hmax - hmin
  #  width of target interval
  ywidth  <- ylim[2] - ylim[1]
  #  normalized h varying horizontally over base interval and 
  #  vertically over target interval
  h   <- (ylim[1] - hmin) + hraw*(ywidth/hwidth)
  #  compute least squares fitting criterion
  res <- y - h
  f   <- mean(res^2*wt)
  #  -----------------------------------------------------
  #             compute fitting gradient grad
  #  -----------------------------------------------------
  #  un-normalized partial derivative of un-normalized h Dh
  Dhraw   <- matrix(mongrad(x, Wfdobj),nobs,Wnbasis)
  #  range of un-normalized gradient
  Dhmax   <- matrix(mongrad(xlim[2], Wfdobj), 1, Wnbasis)
  Dhmin   <- matrix(mongrad(xlim[1], Wfdobj), 1, Wnbasis)
  Dhwidth <- Dhmax - Dhmin
  #  normalized gradient
  Dh    <- ywidth*(Dhraw*hwidth - hraw %*% Dhwidth)/hwidth^2
  #  gradient of fitting function is computed
  temp  <- Dh*(wt %*% matrix(1,1,Wnbasis))
  grad  <- -2*t(temp) %*% res/nobs
  #  apply regularization if needed
  if (Wlambda > 0) {
    grad <- grad +           2 * Kmat  %*%  cvec
    f    <- f    + t(cvec)  %*%  Kmat  %*%  cvec
  }
  #  map parameter space into fitting space
  grad <- t(Zmat) %*% grad
  norm <- sqrt(sum(grad^2))   #  gradient norm
  #  -----------------------------------------------------
  #        compute fitting Hessian hmat
  #  -----------------------------------------------------
  wtroot  <- sqrt(wt)
  temp <- Dh * (wtroot %*% matrix(1,1,Wnbasis))
  hmat <- 2*t(temp) %*% temp/nobs
  #  apply regularization if needed
  if (Wlambda > 0) { 
    hmat <- hmat + 2*Kmat 
  }
  #  map parameter hessian into fitting hessian
  hmat <- t(Zmat) %*% hmat %*% Zmat
  
  return(list(f=f, grad=grad, hmat=hmat, norm=norm, h=h, Dh=Dh))
  
}

#  ------------------------------------------------------------------------

lnsrch_morph <- function(xold, fold, g, p, func, dataList, 
                          Zmat, stpmax, itermax=20, TOLX=1e-10, dbglev=0) {
  #  LNSRCH computes an approximately optimal parameter vector X given
  #  an initial parameter vector XOLD, an initial function value FOLD
  #  and an initial gradient vector G.  
  
  #  Arguments:
  #  XOLD      Initial parameter value
  #  FOLD      Initial function value
  #  G         Initial gradient value
  #  P         Search direction vector
  #  FUNC      Function object computing a function value and gradient
  #              vector
  #  DATALIST  List object used in function object FUNC
  #  STPMAX    Maximum step size
  #  ITERMAX   Maximum number of iterations
  #  TOLX      Tolerance for stop
  #  DBGLEV    Debugging output value:  none if 0, function value if 1,
  #              if greater than one, current step value, slope and
  #              function value.
  
  #  Last modified 12 February 2022
  
  #  set initial constants
  n <- length(xold)
  check <- FALSE
  f2    <- 0
  alam2 <- 0
  ALF   <- 1e-4
  psum  <- sqrt(sum(p^2))
  #  scale if attempted step is too big
  if (psum > stpmax) {
    p <- p*(stpmax/psum)
  }
  #  compute slope
  slope <- sum(g*p)
  # if (dbglev > 1) {
  #   status <- c(0, 0, slope, fold)
  #   cat("#10.f #10.4f #10.4f #10.4f\n", status)
  # }
  # check that initial slope is negative
  if (slope >= 0) {
    stop("Initial slope not negative.")
  }
  # compute lambdamin
  test <- 0
  for (i in 1:n) {
    temp <- abs(p[i])/max(abs(xold[i]),1)
    if (temp > test) {
      test <- temp
    }
  }
  alamin <- TOLX/test
  #  always try full Newton step first with step size 1
  alam   <- 1
  #  start of iteration loop
  iter <- 0
  while (iter <= itermax) {
    iter <- iter + 1
    x <- xold + alam*p
    #  -------------  function evaluation  -----------
    funcList <- func(x, dataList, Zmat)
    f    <- funcList$f
    gtmp <- funcList$grad
    slp <- sum(gtmp*p)
    # if (dbglev > 1) {
    #   status <- [iter, alam, slp, f]
    #   cat("#10.f #10.4f #10.4f #10.4f\n", status)
    # }
    #  -----------------------------------------------
    #  convergence on x change.
    if (alam < alamin) {
      x <- xold
      check <- TRUE
      return(list(x=x, check=check))
    } else {
      #  sufficient function decrease
      if (f <= fold + ALF*alam*slope) {
        return(list(x=x, check=check))
      }
      #  backtrack
      if (alam == 1) {
        #  first time
        tmplam <- -slope/(2*(f-fold-slope))
      } else {
        #  subsequent backtracks
        rhs1 <- f  - fold - alam *slope
        rhs2 <- f2 - fold - alam2*slope
        a <- (rhs1/alam^2 - rhs2/alam2^2)/(alam-alam2)
        b <- (-alam2*rhs1/alam^2 + alam*rhs2/(alam*alam2))/(alam-alam2)
        if (a == 0) {
          tmplam <- -slope/(2*b)
        } else {
          disc <- b^2 - 3*a*slope
          if (disc < 0) {
            tmplam <- 0.5*alam
          } else {
            if (b <= 0) {
              tmplam <- (-b+sqrt(disc))/(3*a)
            } else {
              tmplam <- -slope/(b+sqrt(disc))
            }
          }
          if (tmplam > 0.5*alam) {
            # lambda <= 0.5 lambda1
            tmplam <- 0.5*alam
          }
        }
      }
      alam2 <- alam
      f2    <- f
      #  lambda > 0.1 lambda1
      alam <- max(tmplam, 0.1*alam)
    }
    #  try again
    
  }
  
  return(list(x=x, check=check))
  
}


smooth.pos <- function(argvals, y, WfdParobj, wtvec=rep(1,n), conv=1e-4,
                       iterlim=50, dbglev=1) {
  #  Smooths the relationship of Y to ARGVALS using weights in WTVEC by fitting a
  #     positive function of the form
  #                      f(x) = exp W(x)
  #     where  W  is a function defined over the same range as ARGVALS,
  #                         W = log Df.
  #  The fitting criterion is penalized mean squared error:
  #    PENSSE(lambda) <- \sum w_i[y_i - f(x_i)]^2 +
  #                     \lambda * \int [L W(x)]^2 dx
  #  where L is a linear differential operator defined in argument Lfdobj,
  #  and w_i is a positive weight applied to the observation.
  #  The function W(x) is expanded by the basis in functional data object
  #    Wfdobj.
  
  #  Arguments:
  #  ARGVALS ...  Argument value array of length N, where N is the number of
  #               observed curve values for each curve.  It is assumed that
  #               that these argument values are common to all observed
  #               curves.  If this is not the case, you will need to
  #               run this function inside one or more loops, smoothing
  #               each curve separately.
  #  Y       ...  Function value array (the values to be fit).
  #               If the functional data are univariate, this array will be
  #               an N by NCURVE matrix, where N is the number of observed
  #               curve values for each curve and NCURVE is the number of
  #               curves observed.
  #               If the functional data are multivariate, this array will be
  #               an N by NCURVE by NVAR matrix, where NVAR the number of
  #               functions observed per case.  For example, for the gait
  #               data, NVAR = 2, since we observe knee and hip angles.
  #  WFDPAROBJ... An fd or an fdPar object.  This object
  #               contains the specifications for the functional data
  #               object to be estimated by smoothing the data.  See
  #               comment lines in function fdPar for details.
  #               The functional data object WFD in WFDPAROBJ is used
  #               to initialize the optimization process.
  #               Its coefficient array contains the starting values for
  #               the iterative minimization of mean squared error, and
  #               this coefficient array must be either a K by NCURVE
  #               matrix or a K by NUCRVE by NVAR array,  where K
  #               is the number of basis functions.
  #               If WFDPAROBJ is NULL, it will be initialized to
  #               a matrix or array of zeros.
  #  WTVEC   ...  a vector of weights, a vector of N one's by default.
  #  CONV    ...  convergence criterion, 0.0001 by default
  #  ITERLIM ...  maximum number of iterations, 50 by default.
  #  DBGLEV  ...  Controls the level of output on each iteration.  If 0,
  #               no output, if 1, output at each iteration, if higher,
  #               output at each line search iteration. 1 by default.
  #               enabling this option.
  
  #  Returns are:
  #  WFD     ...  Functional data object for W.
  #               Its coefficient matrix an N by NCURVE (by NVAR) matrix
  #               (or array), depending on whether the functional
  #               observations are univariate or multivariate.
  #  FLIST ... A list object or a vector of list objects, one for
  #            each curve (and each variable if functions are multivariate).
  #            Each list object has slots:
  #                 f    ... The sum of squared errors
  #                 grad ... The gradient
  #                 norm ... The norm of the gradient
  #  When multiple curves and variables are analyzed, the lists containing
  #  FLIST objects are indexed linear with curves varying inside
  #  variables.
  
  #  Last modified 17 January 2023 by Jim Ramsay
  
  #  check ARGVALS
  
  if (!is.numeric(argvals)) stop("ARGVALS is not numeric.")
  argvals <- as.vector(argvals)
  if (length(argvals) < 2) stop("ARGVALS does not contain at least two values.")
  n       <- length(argvals)
  onesobs <- matrix(1,n,1)
  
  #  at least two points are necessary for monotone smoothing
  
  if (n < 2) stop('ARGVALS does not contain at least two values.')
  
  #  check Y
  
  y      <- as.matrix(y)
  ychk   <- ycheck(y, n)
  y      <- ychk$y
  ncurve <- ychk$ncurve
  nvar   <- ychk$nvar
  ndim   <- ychk$ndim
  
  #  check WfdParobj and get LAMBDA
  
  if (inherits(WfdParobj, "fdPar") || inherits(WfdParobj, "fd")) {
    if (inherits(WfdParobj, "fd")) WfdParobj <- fdPar(WfdParobj)
  } else {
    stop(paste("Argument WFDPAROBJ is neither an fdPar object",
               "or an fd object."))
  }
  Wfdobj   <- WfdParobj$fd
  Lfdobj   <- WfdParobj$Lfd
  basisobj <- Wfdobj$basis     
  nbasis   <- basisobj$nbasis  
  coef0    <- Wfdobj$coefs
  lambda   <- WfdParobj$lambda
  
  #  check WTVEC
  
  wtvec <- wtcheck(n, wtvec)$wtvec
  
  #  set up some arrays
  
  climit  <- c(rep(-400,nbasis),rep(400,nbasis))
  coef0   <- Wfdobj$coefs
  active  <- 1:nbasis
  
  #  initialize matrix Kmat defining penalty term
  
  if (lambda > 0) Kmat <- lambda*eval.penalty(basisobj, Lfdobj)
  else            Kmat <- matrix(0,nbasis,nbasis)
  
  #  --------------------------------------------------------------------
  #              loop through variables and curves
  #  --------------------------------------------------------------------
  
  #  set up arrays and lists to contain returned information
  
  if (ndim == 2) {
    coef <- matrix(0,nbasis,ncurve)
  } else {
    coef <- array(0,c(nbasis,ncurve,nvar))
  }
  
  if (ncurve > 1 || nvar > 1 ) Flist <- vector("list",ncurve*nvar)
  else                         Flist <- NULL
  
  
  for (ivar in 1:nvar) {
    if (ndim == 2) {
      sclfac <- mean(c(y)^2)
    } else {
      sclfac <- mean(c(y[,,ivar])^2)
    }
    for (icurve in 1:ncurve) {
      if (ndim == 2) {
        yi    <- y[,icurve]
        cveci <- coef0[,icurve]
      } else {
        yi    <- y[,icurve,ivar]
        cveci <- coef0[,icurve,ivar]
      }
      
      #  evaluate log likelihood
      #    and its derivatives with respect to these coefficients
      
      result <- PENSSEfun(argvals, yi, basisobj, cveci, Kmat, wtvec)
      PENSSE   <- result[[1]]
      DPENSSE  <- result[[2]]
      
      #  compute initial badness of fit measures
      
      f0    <- PENSSE
      gvec0 <- DPENSSE
      Foldlist <- list(f = f0, grad = gvec0, norm = sqrt(mean(gvec0^2)))
      
      #  compute the initial expected Hessian
      
      hmat0 <- PENSSEhess(argvals, yi, basisobj, cveci, Kmat, wtvec)
      
      #  evaluate the initial update vector for correcting the initial bmat
      
      deltac   <- -solve(hmat0,gvec0)
      cosangle <- -sum(gvec0*deltac)/sqrt(sum(gvec0^2)*sum(deltac^2))
      
      #  initialize iteration status arrays
      
      iternum <- 0
      status <- c(iternum, Foldlist$f, -PENSSE, Foldlist$norm)
      if (dbglev >= 1) {
        if (ncurve > 1 || nvar > 1) {
          if (ncurve > 1 && nvar > 1) {
            cat("\n")
            curvetitle <- paste('Results for curve',icurve,'and variable',ivar)
          }
          if (ncurve > 1 && nvar == 1) {
            cat("\n")
            curvetitle <- paste('Results for curve',icurve)
          }
          if (ncurve == 1 && nvar > 1) {
            cat("\n")
            curvetitle <- paste('Results for variable',ivar)
          }
          cat("\n")
          cat(curvetitle)
        }
        cat("\n")
        cat("\nIter.   PENSSE   Grad Length")
        cat("\n")
        cat(iternum)
        cat("        ")
        cat(round(status[2],4))
        cat("      ")
        cat(round(status[3],4))
      }
      #  -------  Begin iterations  -----------
      
      MAXSTEPITER <- 10
      MAXSTEP     <- 400
      trial       <- 1
      linemat     <- matrix(0,3,5)
      Flisti      <- Foldlist
      gvec        <- gvec0
      dbgwrd      <- dbglev > 1
      
      if (iterlim == 0) {
        cat("\n")
      } else {
        for (iter in 1:iterlim) {
          iternum <- iternum + 1
          #  take optimal stepsize
          dblwrd <- rep(FALSE,2)
          limwrd <- rep(FALSE,2)
          stpwrd <- FALSE
          ind    <- 0
          ips    <- 0
          #  compute slope at 0 for line search
          linemat[2,1] <- sum(deltac*Flisti$grad)
          #  normalize search direction vector
          sdg     <- sqrt(sum(deltac^2))
          deltac  <- deltac/sdg
          dgsum   <- sum(deltac)
          linemat[2,1] <- linemat[2,1]/sdg
          # initialize line search vectors
          linemat[,1:4] <- outer(c(0, linemat[2,1], Flisti$f),rep(1,4))
          stepiter <- 0
          if (dbglev >= 2) {
            cat("\n")
            cat(paste("                 ", stepiter, "  "))
            cat(format(round(t(linemat[,1]),6)))
          }
          #  break with error condition if (initial slope is nonnegative
          if (linemat[2,1] >= 0) {
            print("Initial slope nonnegative.")
            ind <- 3
            break
          }
          #  return successfully if (initial slope is very small
          if (linemat[2,1] >= -1e-5) {
            if (dbglev > 1) print("Initial slope too small")
            break
          }
          #  first step set to trial
          linemat[1,5]  <- trial
          #  Main iteration loop for linesrch
          for (stepiter in 1:MAXSTEPITER) {
            #  ensure that step does not go beyond limits on parameters
            limflg  <- 0
            #  check the step size
            result <- stepchk(linemat[1,5], cveci, deltac, limwrd, ind,
                              climit, active, dbgwrd)
            linemat[1,5] <- result[[1]]
            ind          <- result[[2]]
            limwrd       <- result[[3]]
            if (linemat[1,5] <= 1e-7) {
              #  Current step size too small  terminate
              Flisti  <- Foldlist
              cvecnew <- cveci
              gvecnew <- gvec
              if (dbglev >= 2) {
                print("Stepsize too small")
                print(linemat[1,5])
              }
              if (limflg) ind <- 1 else ind <- 4
              break
            }
            #  compute new function value and gradient
            cvecnew  <- cveci + linemat[1,5]*deltac
            result   <- PENSSEfun(argvals, yi, basisobj, cvecnew, Kmat, wtvec)
            PENSSE   <- result[[1]]
            DPENSSE  <- result[[2]]
            Flisti$f <- PENSSE
            gvecnew  <- DPENSSE
            Flisti$grad <- gvecnew
            Flisti$norm <- sqrt(mean(gvecnew^2))
            linemat[3,5] <- Flisti$f
            #  compute new directional derivative
            linemat[2,5] <- sum(deltac*gvecnew)
            if (dbglev >= 2) {
              cat("\n")
              cat(paste("                 ", stepiter, "  "))
              cat(format(round(t(linemat[,5]),6)))
            }
            #  compute next step
            result  <- stepit(linemat, ips, dblwrd, MAXSTEP)
            linemat <- result[[1]]
            ips     <- result[[2]]
            ind     <- result[[3]]
            dblwrd  <- result[[4]]
            trial   <- linemat[1,5]
            #  ind == 0 implies convergence
            if (ind == 0 | ind == 5) break
            #  end of line search loop
          }
          cveci <- cvecnew
          gvec  <- gvecnew
          #  test for convergence
          if (abs(Flisti$f - Foldlist$f) < sclfac*conv) {
            cat("\n")
            break
          }
          if (Flisti$f >= Foldlist$f) break
          #  compute the Hessian
          hmat <- PENSSEhess(argvals, yi, basisobj, cveci, Kmat, wtvec)
          #  evaluate the update vector
          deltac <- -solve(hmat,gvec)
          cosangle  <- -sum(gvec*deltac)/sqrt(sum(gvec^2)*sum(deltac^2))
          if (cosangle < 0) {
            if (dbglev > 1) print("cos(angle) negative")
            deltac <- -gvec
          }
          Foldlist <- Flisti
          #  display iteration status
          status <- c(iternum, Flisti$f, Flisti$norm)
          if (dbglev >= 1) {
            cat("\n")
            cat(iternum)
            cat("        ")
            cat(round(status[2],4))
            cat("      ")
            cat(round(status[3],4))
          }
          #  end of iteration loop
        }
      }
      
      #  save coefficients in arrays COEF and BETA
      
      if (ndim == 2) {
        coef[,icurve] <- cveci
      } else {
        coef[,icurve,ivar] <- cveci
      }
      
      #  save Flisti
      
      if (ncurve == 1 && nvar == 1) {
        Flist <- Flisti
      } else {
        Flist[[(ivar-1)*ncurve+icurve]] <- Flisti
      }
      
    }
  }
  
  Wfdobj <- fd(coef, basisobj)
  
  posFd <- list("Wfdobj"=Wfdobj, "Flist"=Flist,
                "argvals"=argvals, "y"=y)
  class(posFd) <- 'posfd'
  
  return(posFd)
}

#  ---------------------------------------------------------------

PENSSEfun <- function(argvals, yi, basisobj, cveci, Kmat, wtvec) {
  #  Computes the log likelihood and its derivative with
  #    respect to the coefficients in CVEC
  n       <- length(argvals)
  nbasis  <- basisobj$nbasis
  phimat  <- getbasismatrix(argvals, basisobj, 0)
  Wvec    <- phimat %*% cveci
  EWvec   <- exp(Wvec)
  res     <- yi - EWvec
  PENSSE  <- mean(wtvec*res^2) + t(cveci) %*% Kmat %*% cveci
  DPENSSE <- -2*crossprod(phimat,wtvec*res*EWvec)/n + 2*Kmat %*% cveci
  return( list(PENSSE, DPENSSE) )
}

#  ---------------------------------------------------------------

PENSSEhess <- function(argvals, yi, basisobj, cveci, Kmat, wtvec) {
  #  Computes the expected Hessian
  n        <- length(argvals)
  nbasis   <- basisobj$nbasis
  phimat   <- getbasismatrix(argvals, basisobj, 0)
  Wvec     <- phimat %*% cveci
  EWvec    <- exp(Wvec)
  D2PENSSE <- 2*t(phimat) %*% diag(as.numeric(wtvec*EWvec^2)) %*% phimat/n + 2*Kmat
  return(D2PENSSE)
}


smooth.sparse.mean <- function(data, time ,rng=c(0, 1), type = "" , nbasis = NULL, knots = NULL, norder = NULL, lambda = NULL){
  
  #   Arguments:
  #   DATA        a matrix object or list -- If the set is supplied as a matrix object, 
  #               the rows must correspond to argument values and columns to replications, 
  #               and it will be assumed that there is only one variable per observation.  
  #               If y is a three-dimensional array, the first dimension corresponds to  
  #               argument values, the second to replications, and the third to variables 
  #               within replications. -- If it is a list, each element must be a matrix
  #               object, the rows correspond to argument values per individual. First 
  #               column corresponds to time points and followin columns to argument values 
  #               per variable.
  #   TIME        Array with time points where data was taken. length(time) == ncol(data)
  #   RNG         an array of length 2 containing the lower and upper
  #               boundaries for the rangeval of argument values
  #   TYPE        Type of basisfd for smoothing the mean estimate function
  #   NBASIS      An integer variable specifying the number of basis functions
  #   KNOTS       a vector specifying the break points if type = "bspline"
  #   NORDER      an integer specifying the order of b-splines if type = "bspline"
  #   LAMBDA      a nonnegative real number specifying the amount of smoothing to be applied 
  #               to the estimated functional parameter
  
  if(type == "bspline"){
    nbasis = length(knots) + norder - 2
    basis = create.bspline.basis(rng,nbasis,norder)
  }else if (type == "fourier"){
    basis = create.fourier.basis(rng,nbasis)
  }else if (type == "exp"){
    basis = create.exponential.basis(rng,nbasis)
  }else if (type == "const"){
    basis = create.constant.basis(rng)
  }else if (type == "mon"){
    basis = create.monomial.basis(rng,nbasis)
  }
  
  if(is.list(data)){
    data.list = data
  }else{
    data.list = sparse.list(data, time)
  }
  data.mat = do.call(rbind,data.list)
  
  if(!is.null(lambda)){
    curv.Lfd = int2Lfd(2)
    curv.fdPar = fdPar(basis,curv.Lfd,lambda)
    smooth = smooth.basis(data.mat[,1],data.mat[,-1],curv.fdPar)
  }else{
    smooth = smooth.basis(data.mat[,1],data.mat[,-1],basis)
  }
  
  if(ncol(smooth$fd$coefs)>1){
    smooth$fd$coefs = array(smooth$fd$coefs,dim=c(nrow(smooth$fd$coefs),1,ncol(smooth$fd$coefs)))
  }
  
  return(smooth$fd)
}

smooth.surp <- function(argvals, y, Bmat0, WfdPar, wtvec=NULL, conv=1e-4,
                       iterlim=50, dbglev=0) {
    #  Smooths the relationship of Y to ARGVALS using weights in WTVEC by fitting 
  #     surprisal functions to a set of surprisal transforms of choice 
  #     probabilities, where the surprisal transformation of each probability is 
  #                      W(p_m) = -log_M (p_m), m=1, ..., M,
  #     where  W  is a function defined over the same range as ARGVALS.
  #  The fitting criterion is penalized mean squared error:
  #    PENSSE(Wlambda) <- \sum w_i[y_i - f(x_i)]^2 +
  #                     \Wlambda * \int [L W(x)]^2 dx
  #  where L is a linear differential operator defined in argument Lfd,
  #  and w_i is a positive weight applied to the observation.
  #  The function W(x) is expanded by the basis in functional data object
  #    Wfd.
  #
  #  This version uses Numerical Recipes function lnsrch
  #
  #  Arguments:
  #  ARGVALS ...  Argument value array of length NBIN, the number of 
  #               surprisal values for each curve.  It is assumed that
  #               that these argument values are common to all observed
  #               curves.  
  #  Y       ...  A matrix containing the values to be fit.
  #               This will be an NBIN by M matrix, where NBIN is the 
  #               number of bins containing choice probabilities and M is
  #               the number of options in a specific question or rating
  #               scale.
  #  BMAT0   ...  An initial K by M-1 matrix defining the surprisal curves
  #               via spline functions.  K is the number of basis functions
  #               in the spline expansions, and M is the number of choices
  #               for a particular question in a test or rating scale.
  #  WFDPAR  ...  A functional parameter or fdPar object.  This object
  #               contains the specifications for the functional data
  #               object to be estimated by smoothing the data.  
  #               Note:  WFDPAR is only a container for its 
  #               functional basis object WBASIS, the penalty matrix WPEN, 
  #               and the smoothing parameter Wlambda.  A coefficient
  #               matrix in WFDPAR defined by using a function data object
  #               is discarded, and overwritten by argument BMAT0.
  #  WTVEC   ...  a vector of weights, a vector of N one's by default.
  #  CONV    ...  convergence criterion, 0.0001 by default
  #  ITERLIM ...  maximum number of iterations, 50 by default.
  #  DBGLEV  ...  Controls the level of output on each iteration.  If 0,
  #               no output, if 1, output at each iteration, if higher,
  #               output at each line search iteration. 1 by default.
  #               enabling this option.
  
  #  Returns are:
  #  WFD     ...  Functional data object for W.
  #               Its coefficient matrix an N by NCURVE (by NVAR) matrix
  #               (or array), depending on whether the functional
  #               observations are univariate or multivariate.
  #  FLIST ... A list object or a vector of list objects, one for
  #            each curve (and each variable if functions are multivariate).
  #            Each list object has slots:
  #                 f    ... The sum of squared errors
  #                 grad ... The gradient
  #                 norm ... The norm of the gradient
  #  When multiple curves and variables are analyzed, the lists containing
  #  FLIST objects are indexed linear with curves varying inside
  #  variables.
  
  #  Last modified 23 March 2022 by Jim Ramsay
  
  #  check ARGVALS, a vector of length n
  
  if (!is.numeric(argvals)) stop("ARGVALS is not numeric.")
  argvals <- as.vector(argvals)
  if (length(argvals) < 2) stop("ARGVALS does not contain at least two values.")
  n       <- length(argvals)
  onesobs <- matrix(1,n,1)
  
  #  Check y, an n by M-1 matrix of surprisal values.  
  #  It may not contain negative values.
  
  y    <- as.matrix(y)
  ydim <- dim(y)
  M    <- ydim[2]
  if (ydim[1] != n) 
      stop("The length of ARGVALS  and the number of rows of Y differ.")
  
  #  Check WfdPar and extract WBASIS, WNBASIS, Wlambda and WPENALTY.  
  #  Note that the coefficient matrix is not used.
  
  WfdPar   <- fdParcheck(WfdPar,M)
  Wbasis   <- WfdPar$fd$basis
  
  Wnbasis  <- Wbasis$nbasis
  Wlambda  <- WfdPar$lambda
  Wpenalty <- eval.penalty(Wbasis, WfdPar$Lfd)
  
  #  Check BMAT0, the WNBASIS by M-1 coefficient matrix
  
  if (is.null(Bmat0)) stop("BMAT0 is  NULL.")
  
  Bmatdim <- dim(Bmat0)
  if (Bmatdim[1] != Wnbasis) 
    stop("The first dimension of BMAT0 is not equal to WNBASIS.")
  if (Bmatdim[2] != M-1) 
    stop("The second dimension of BMAT0 is not equal to M - 1.")
  
  #  convert Bmat0 to a vector and NPAR to its length
  
  cvec <- as.vector(Bmat0)
  npar <- length(cvec)
  
  #  Set up the transformation from dimension M-1 to M
  #  where M-vectors sum to zero
  
  if (M == 2) {
    root2 <- sqrt(2)
    Zmat <- matrix(1/c(root2,-root2),2,1)
  } else {
    Zmat <- zerobasis(M)
  }
  
  #  Set up the matrix of basis function values 
  
  Phimat <- fda::eval.basis(argvals, Wbasis)
  
  #  check WTVEC
  
  if (is.null(wtvec)) wtvec<-rep(1,n)
  wtvec <- fda::wtcheck(n, wtvec)$wtvec
  
  #  initialize matrix Kmat defining penalty term
  
  if (Wlambda > 0) 
  {
    Kmat <- Wlambda*Wpenalty
  } else {
    Kmat <- matrix(0,Wnbasis,Wnbasis)
  }
  
  #  Set up list object for data required by PENSSEfun
  
  surpList <- list(argvals=argvals, y=y, wtvec=wtvec, Kmat=Kmat,
                   Zmat=Zmat, Phimat=Phimat, M=M)
  #  --------------------------------------------------------------------
  #              loop through variables and curves
  #  --------------------------------------------------------------------
  
  #  evaluate log likelihood
  #    and its derivatives with respect to these coefficients
  
  xold <- matrix(Bmat0, Wnbasis*(M-1),1)
  result    <- surp.fit(xold, surpList)
  PENSSE    <- result[[1]]
  DPENSSE   <- result[[2]]
  D2PENSSE  <- result[[3]]
  
  #  compute initial badness of fit measures
  
  fold <- PENSSE
  gvec <- DPENSSE
  hmat <- D2PENSSE
  
  Flist <- list(f = fold, grad = gvec, norm = sqrt(mean(gvec^2)))
  
  Foldlist <- Flist
  
  #  evaluate the initial update vector for correcting the initial bmat
  
  pvec   <- -solve(hmat,gvec)
  cosangle <- -sum(gvec*pvec)/sqrt(sum(gvec^2)*sum(pvec^2))
  
  #  initialize iteration status arrays
  
  iternum <- 0
  status <- c(iternum, Foldlist$f, Foldlist$norm)
  if (dbglev >= 1) {
    cat("\n")
    cat("\nIter.   PENSSE   Grad Length")
    cat("\n")
    cat(iternum)
    cat("        ")
    cat(round(status[2],4))
    cat("      ")
    cat(round(status[3],4))
  }
  
  #  -------  Begin iterations  -----------
  
  STEPMAX <- 10
  
  iternum <- 0
  for (iter in 1:iterlim) {
    iternum <- iternum + 1
    #  take optimal stepsize
    lnsrch_result <- 
      lnsrch(xold, fold, gvec, pvec, surp.fit, surpList, STEPMAX)
    x     <- lnsrch_result$x
    check <- lnsrch_result$check
    if (check) stop("lnsrch failure")
    #  set up new Bmat and evaluate function, gradient and hessian
    Bmatnew <- matrix(x,Wnbasis,M-1)
    func_result <- surp.fit(Bmatnew, surpList)
    f     <- func_result[[1]]
    gvec  <- func_result[[2]]
    hmat  <- func_result[[3]]
    SSE   <- func_result[[4]]
    DSSE  <- func_result[[5]]
    D2SSE <- func_result[[6]]
    #  set up list object for current fit
    Flist$f    <- f
    Flist$grad <- gvec
    Flist$norm <- sqrt(mean(gvec^2))
    xold <- x
    fold <- f
    #  display results at this iteration if dbglev > 0
    status <- c(iternum, Flist$f, Flist$norm)
    if (dbglev > 0) {
      cat("\n")
      cat(iternum)
      cat("        ")
      cat(round(status[2],4))
      cat("      ")
      cat(round(status[3],4))
    }
    #  test for convergence
    if (abs(Flist$f - Foldlist$f) < conv) {
      break
    }
    #  also terminate iterations if new fit is worse than old
    if (Flist$f >= Foldlist$f) break
    #  set up objects for new iteration
    #  evaluate the update vector using Newton Raphson direction
    pvec <- -solve(hmat,gvec)
    cosangle  <- -sum(gvec*pvec)/sqrt(sum(gvec^2)*sum(pvec^2))
    if (cosangle < 0) {
      if (dbglev > 1) print("cos(angle) negative")
      pvec <- -gvec
    }
    Foldlist <- Flist
  }
  
  #  end of iteration loop, output results
  
  Bmat <- matrix(xold, Wnbasis, M-1)
  Wfd  <- fda::fd(Bmat, Wbasis)
  surpResult <- surp.fit(Bmat, surpList)
  
  PENSSE   <- surpResult$PENSSE
  DPENSSE  <- surpResult$DPENSSE 
  D2PENSSE <- surpResult$D2PENSSE
  SSE      <- surpResult$SSE
  DSSE     <- surpResult$DSSE
  D2SSE    <- surpResult$D2SSE
  DvecSmatDvecB <- surpResult$DvecSmatDvecB

  surpFd <- list(Wfd=Wfd, Bmat=Bmat, f=f, gvec=gvec, hmat=hmat,
                 PENSSE=PENSSE, DPENSSE=DPENSSE, D2PENSSE=D2PENSSE,
                 SSE=SSE, DSSE=DSSE, D2SSE=D2SSE,
                 DvecSmatDvecB=DvecSmatDvecB)
  class(surpFd) <- 'surpfd'
  
  return(surpFd)
}

# ------------------------------------------------------------------

surp.fit <- function(x, surpList) {
  
  #  This function is called within smooth.surp() to
  #  evaluate fit at each iteration
  
  #  extract objects from surpList
  
  argvals <- surpList$argvals 
  y       <- surpList$y 
  wtvec   <- surpList$wtvec 
  Kmat    <- surpList$Kmat
  Zmat    <- surpList$Zmat 
  Phimat  <- surpList$Phimat 
  
  # set up dimensions and Bmat
  
  n       <- length(argvals)
  M       <- surpList$M
  K       <- dim(Phimat)[2]
  Bmat    <- matrix(x, K, M-1)
  
  #  compute fit, gradient and hessian
  
  logM     <- log(M)
  onewrd   <- all(wtvec == 1)
  Xmat     <- Phimat %*% Bmat %*% t(Zmat)
  expXmat  <- M^Xmat
  sumexpXmat <- as.matrix(apply(expXmat,1,sum))
  Pmat     <- expXmat/(sumexpXmat %*% matrix(1,1,M))
  Smat     <- -Xmat + (log(sumexpXmat) %*% matrix(1,1,M))/logM
  Rmat     <- y - Smat
  vecBmat  <- matrix(Bmat,K*(M-1),1,byrow=TRUE)
  vecRmat  <- matrix(Rmat,n*M,    1,byrow=TRUE)
  vecKmat  <- kronecker(diag(rep(1,M-1)),Kmat)
  fitscale <- 1
  #  compute raw fit and its penalized version
  if (!onewrd) {
    vecwtmat <- diag(rep(wtvec,M))
    SSE <- t(vecRmat) %*% vecwtmat %*% vecRmat
  } else {
    SSE <- crossprod(vecRmat)
  }
  PENSSE   <- SSE/fitscale + t(vecBmat) %*% vecKmat %*% vecBmat
  #  compute raw gradient and its penalized version
  DvecXmatDvecB <- kronecker(Zmat,Phimat)
  DvecSmatDvecX <- matrix(0,n*M,n*M)
  m2 <- 0
  for (m in 1:M) {
    m1 <- m2 + 1
    m2 <- m2 + n
    m4 <- 0
    for (l in 1:M) {
      m3 <- m4 + 1
      m4 <- m4 + n
      diagPl <- diag(Pmat[,l])
      DvecSmatDvecX[m1:m2,m3:m4] <- diagPl
    }
  }
  DvecSmatDvecX <- DvecSmatDvecX - diag(rep(1,n*M))
  DvecSmatDvecB <- DvecSmatDvecX %*% DvecXmatDvecB
  if (!onewrd) {
    DSSE <- -2*t(DvecSmatDvecB) %*% vecwtmat %*% vecRmat
  } else {
    DSSE <- -2*t(DvecSmatDvecB) %*% vecRmat
  }
  DPENSSE  <- DSSE/fitscale + 2*vecKmat %*% vecBmat
  #  compute raw hessian and its penalized version
  if (!onewrd) {
    D2SSE <- 2*t(DvecSmatDvecB) %*% vecwtmat %*% DvecSmatDvecB
  } else {
    D2SSE <- 2*crossprod(DvecSmatDvecB)
  }
  D2PENSSE <- D2SSE/fitscale + 2*vecKmat
  
  #  return list object containing raw and penalized fit data
  
  return(list(
    PENSSE   = PENSSE, 
    DPENSSE  = DPENSSE, 
    D2PENSSE = D2PENSSE,
    SSE      = SSE,
    DSSE     = DSSE,
    D2SSE    = D2SSE,
    DvecSmatDvecB = DvecSmatDvecB)
  )
  
}

# ------------------------------------------------------------------

ycheck <- function(y, n) {
  
  #  check Y
  
  if (is.vector(y)) y <- as.matrix(y)
  
  if (!inherits(y, "matrix") && !inherits(y, "array"))
    stop("Y is not of class matrix or class array.")
  
  ydim <- dim(y)
  
  if (ydim[1] != n) stop("Y is not the same length as ARGVALS.")
  
  #  set number of curves and number of variables
  
  ndim  <- length(ydim)
  if (ndim == 2) {
    ncurve <- ydim[2]
    nvar   <- 1
  }
  if (ndim == 3) {
    ncurve <- ydim[2]
    nvar   <- ydim[3]
  }
  if (ndim > 3) stop("Second argument must not have more than 3 dimensions")
  
  
  return(list(y=y, ncurve=ncurve, nvar=nvar, ndim=ndim))
  
}
sparse.list <- function(data,time){
  #Create a list with sparse data from a matrix that has NA
  #
  # Arguments:
  #
  # DATA .... Sparse data -- If the set is supplied as a matrix object, the rows must correspond to argument values and 
  #           columns to replications, and it will be assumed that there is only one variable per observation. If y is a 
  #           three-dimensional array, the first dimension corresponds to argument values, the second to replications, 
  #           and the third to variables within replications. 
  # TIME .... Time points where the observations where taken. length(time) == nrow(data)
  
  ndim = length(dim(data))
  if(ndim ==3){
    ind = apply(data[,,1],2, function(x) which(!(is.na(x))))
    y = lapply(1:dim(data)[2], function(x) cbind(time[ind[[x]]],data[ind[[x]],x,]))
  }else{
    ind = apply(data,2, function(x) which(!(is.na(x))))
    y = lapply(1:dim(data)[2], function(x) cbind(time[ind[[x]]],data[ind[[x]],x]))
  }
  
  
  return(y)
}
sparse.mat <- function(datalist){
  #
  #Create a matrix of sparse data with NAs out of a list of sparse data
  #
  # Arguments:
  #
  # DATALIST .... A list object. Each element of the list is a matrix with ncol > 1. 
  #               The first column of each element corresponds to the point index per 
  #               observation. The following columns are the observations per variable.
  
  time = sort(unique(unlist(lapply(datalist, function(x) x[,1]))))
  data = matrix(NA, nrow = length(time), ncol = length(datalist))
  
  nvar = ncol(datalist[[1]]) - 1
  
  if(nvar == 1){
    data = matrix(NA, nrow = length(time), ncol = length(datalist))
    for(i in 1:length(datalist)){
      data[which((time %in% datalist[[i]][,1])),i] = datalist[[i]][,2]
    }
  }else{
    data = array(NA, dim = c(length(time),length(datalist),nvar))
    for(j in 1:nvar){
      for(i in 1:length(datalist)){
        data[which((time %in% datalist[[i]][,1])),i,j] = datalist[[i]][,2]
      }
    }
  }
  return(data)
}
stddev.fd <- function(fdobj)
{
  #  Compute the standard deviation functions for functional observations
  #  Argument:
  #  fdobj    ... a functional data object
  #  Return:
  #  STDFD ... a functional data object for the standard deviation functions

  #  Last modified 16 January 2020

  if (!(is.fd(fdobj) || is.fdPar(fdobj)))  stop(
    "First argument is neither a functional data or a functional parameter object.")
  if (is.fdPar(fdobj)) fdobj <- fdobj$fd
  
  coef     <- fdobj$coefs
  coefd    <- dim(coef)
  ndim     <- length(coefd)
  if (coefd[1] == 1) stop("Only one replication found.")
  nrep     <- coefd[2]
  ones     <- rep(1,nrep)

  basisobj <- fdobj$basis
  fdnames  <- fdobj$fdnames
  nbasis   <- basisobj$nbasis
  rangeval <- basisobj$rangeval

  neval   <- max(c(201,10*nbasis + 1))
  evalarg <- seq(rangeval[1], rangeval[2], length=neval)
  fdarray <- eval.fd(evalarg, fdobj)

  if (ndim == 2) {
    mnvec  <- (fdarray %*% ones)/nrep
    resmat <- fdarray - (c(mnvec) %o% ones)
    varvec <- (resmat^2 %*% ones)/(nrep-1)
    stdmat <- sqrt(varvec)
  } else {
    nvar <- coefd[3]
    stdmat <- matrix(0, neval, nvar)
    for (j in 1:nvar) {
      mnvecj  <- (fdarray[,,j] %*% ones)/nrep
      resmatj <- fdarray[,,j] - outer(c(mnvecj), ones)
      varvecj <- (resmatj^2 %*% ones)/(nrep-1)
      stdmatj <- sqrt(varvecj)
      stdmat[,j] <- stdmatj
    }
  }
  stdcoef <- project.basis(stdmat, evalarg, basisobj)
  names(fdnames)[2] <- "Std. Dev."
  names(fdnames)[3] <- paste("Std. Dev.",names(fdnames)[3])
  stdfd <- fd(stdcoef, basisobj, fdnames)
  return(stdfd)
}
std.fd <- function(fdobj)
{
  #  Compute the standard deviation functions for functional observations
  #  Argument:
  #  fdobj    ... a functional data object
  #  Return:
  #  STDFD ... a functional data for the standard deviation functions

  #  Last modified 2007.11.28 by Spencer Graves
  #  Previously modified 26 February 2007

  if (!(is.fd(fdobj) || is.fdPar(fdobj)))  stop(
    "First argument is neither a functional data or a functional parameter object.")
  if (is.fdPar(fdobj)) fdobj <- fdobj$fd
  
  coef     <- fdobj$coefs
  coefd    <- dim(coef)
  ndim     <- length(coefd)
  if (coefd[1] == 1) stop("Only one replication found.")
  
  nrep     <- coefd[2]
  ones     <- rep(1,nrep)

  basisobj <- fdobj$basis
  fdnames  <- fdobj$fdnames
  nbasis   <- basisobj$nbasis
  rangeval <- basisobj$rangeval

  neval   <- max(c(201,10*nbasis + 1))
  evalarg <- seq(rangeval[1], rangeval[2], length=neval)
  fdarray <- eval.fd(evalarg, fdobj)

  if (ndim == 2) {
    mnvec  <- (fdarray %*% ones)/nrep
    resmat <- fdarray - (c(mnvec) %o% ones)
    varvec <- (resmat^2 %*% ones)/(nrep-1)
    stdmat <- sqrt(varvec)
  } else {
    nvar <- coefd[3]
    stdmat <- matrix(0, neval, nvar)
    for (j in 1:nvar) {
      mnvecj  <- (fdarray[,,j] %*% ones)/nrep
      resmatj <- fdarray[,,j] - outer(c(mnvecj), ones)
      varvecj <- (resmatj^2 %*% ones)/(nrep-1)
      stdmatj <- sqrt(varvecj)
      stdmat[,j] <- stdmatj
    }
  }
  stdcoef <- project.basis(stdmat, evalarg, basisobj)
  names(fdnames)[2] <- "Std. Dev."
  names(fdnames)[3] <- paste("Std. Dev.",names(fdnames)[3])
  stdfd <- fd(stdcoef, basisobj, fdnames)
  return(stdfd)
}

sd.fd <- function(fdobj)std.fd(fdobj)
stdev.fd <- function(fdobj)std.fd(fdobj)
stddev.fd <- function(fdobj)std.fd(fdobj)
stepchk <- function(oldstep, cvec, deltac, limwrd, ind,
                    climit=50*c(-rep(1,ncvec), rep(1,ncvec)),
                    active=1:ncvec, dbgwrd) {
  #  stepcheck checks the step size to keep parameters within boundaries
  
  # Last changed 2018 by Jim Ramsay
  
  # define vectors containing lower and upper limits
  
  ncvec   <- length(deltac)
  bot     <- climit[1:ncvec]
  top     <- climit[ncvec+(1:ncvec)]
  
  newstep <- oldstep
  
  #  ensure that step does not go beyond lower limit on parameters
  #  limwrd[2] flags that the lower limit has been hit once
  
  stepi   <- oldstep*deltac
  stepmin <- min(stepi)
  index   <- stepi[active] == stepmin
  if (any(stepi[index] < bot[index]-cvec[index]) &
      any(deltac[index] != 0) )  {
    stepnew <- min((bot[index]-cvec[index])/deltac[index])
    if (dbgwrd) {
      print("Lower limit reached ... new step:")
      cat(c(stepi, round(c(oldstep, stepnew),4)),"\n")
      cat(round(cvec + stepnew*deltac,4),"\n")
    }
    newstep <- stepnew
    if (limwrd[2]) ind <- 1 else limwrd[2] <- TRUE
  } else {
    limwrd[2] <- FALSE
  }
  
  #  check whether upper limit has been reached twice in a row
  
  #  ensure that step does not go beyond upper limit on parameters
  #  limwrd[1] flags that the upper limit has been hit once
  
  stepi   <- oldstep*deltac
  stepmax <- max(stepi)
  index   <- stepi[active] == stepmax
  if (any(stepi[index] > top[index]-cvec[index]) &
      any(deltac[index] != 0) ) {
    stepnew <- min((top[index]-cvec[index])/deltac[index])
    if (dbgwrd) {
      print("Upper limit reached ... new step:")
      cat(c(stepi, round(c(oldstep, stepnew),4)),"\n")
    }
    newstep <- stepnew
    if (limwrd[1]) ind <- 1 else limwrd[1] <- TRUE
  } else {
    limwrd[1] <- FALSE
  }
  
  return(list(newstep, ind, limwrd))
}
stepit <- function(linemat, ips, dblwrd, MAXSTEP) {
#STEPIT computes next step size in line search algorithm
  
#  Arguments:
#  LINEMAT:  Row 1 contains step values
#            Row 2 contains slope values
#            Row 3 contains function values
#  IPS:      If 1, previous slope was positive
#  DBLWRD:   Vector of length 2:  dblwrd[1] TRUE means step halved
#                                 dblwrd[2] TRUE means step doubled
#  MAXSTEP:  maximum size of step

#  Last modified 29 June 2018 by Jim Ramsay
  
#  Wolfe condition 1
test1.1 = linemat[3,5] <= linemat[3,1] + linemat[1,5]*linemat[2,1]/20
#  Wolfe condition 2
test1.2 = abs(linemat[2,5]) <= abs(linemat[2,1])/10 
# disp([test1.1, test1.2])
test1 = test1.1 && test1.2
# test1 = test1.2
test2 = linemat[3,5] > linemat[3,1]
test3 = linemat[2,5] > 0
if ((test1 || !test3) && test2) {
   #  ************************************************************
   #  function is worse and either slope is satisfory or negative
   ips = 0        #  step is halved
   if (dblwrd[2]) {
      ind = 5
      return(list(linemat = linemat, ips = ips, ind = ind, dblwrd = dblwrd))
   }
   linemat[1,5] = min(c(linemat[1,5]/2, MAXSTEP))
   linemat[,2] = linemat[,1]
   linemat[,3] = linemat[,1]
   dblwrd[1] = TRUE
   ind = 2
   return(list(linemat = linemat, ips = ips, ind = ind, dblwrd = dblwrd))
}
#  *********************************************************
if (test1) {
   #  test1 means successful convergence
   ind = 0
   return(list(linemat = linemat, ips = ips, ind = ind, dblwrd = dblwrd))
}
#  **********************************************************
if (test3) {
   #  Current slope is positive
   ips = 1
   linemat[,4] = linemat[,5]
   deltaf = linemat[3,3] - linemat[3,5]
   z = (3/(linemat[1,5] - linemat[1,3]))*deltaf + linemat[2,3] + linemat[2,5]
   w = z * z - linemat[2,3] * linemat[2,5]
   if (abs(linemat[2,3] + linemat[2,5] + 2 * z) >= 1e-05 && w > 0) {
     w = sqrt(w)
     linemat[1,5] = linemat[1,3] + (1 - ((linemat[2,5] + w - z)/ 
        (linemat[2,5] - linemat[2,3] + 2 * w))) * (linemat[1,5] - linemat[1,3])
   } else {
           #  linear interpolation necessary
           aerror = linemat[1,3]
           if (linemat[1,5] > linemat[1,3]) {
             aerror = linemat[1,5]
           }
           linemat[1,5] = linemat[1,3] - linemat[2,3] * 
                        ((linemat[1,5] - linemat[1,3])/ 
                         (linemat[2,5] - linemat[2,3]))
           if (linemat[1,5] > 2 * aerror) {
             linemat[1,5] = 2 * aerror
           }
   }
   linemat[1,5] = min(c(linemat[1,5], MAXSTEP))
   dblwrd = rep(FALSE,2)
   ind = 2
   return(list(linemat = linemat, ips = ips, ind = ind, dblwrd = dblwrd))
}
#  *************************************************************
#  Current slope is negative or zero
linemat[,2] = linemat[,3]
linemat[,3] = linemat[,5]
if (ips == 1) {
   #  *****************************************************
   #  previous slope is positive
   deltaf = linemat[3,5] - linemat[3,4]
   z = c(3/(linemat[1,4] - linemat[1,5])) * deltaf + 
            linemat[2,5] + linemat[2,4]
   w = z * z - linemat[2,5] * linemat[2,4]
   if (abs(linemat[2,5] + linemat[2,4] + 2 * z) >= 1e-05 && w > 0) {
     w = sqrt(w)
     linemat[1,5] = linemat[1,5] + (1 - ((linemat[2,4] + w - z)/ 
                   (linemat[2,4] - linemat[2,5] + 
                    2 * w))) * (linemat[1,4] - linemat[1,5])
   } else {
           aerror = linemat[1,5]
           if (linemat[1,4] > linemat[1,5]) {
                   aerror = linemat[1,4]
           }
           linemat[1,5] = linemat[1,5] - linemat[2,5] * 
                        ((linemat[1,4] - linemat[1,5])/ 
                         (linemat[2,4] - linemat[2,5]))
           if (linemat[1,5] > 2 * aerror) {
                   linemat[1,5] = 2 * aerror
           }
   }
   linemat[1,5] = min(c(linemat[1,5], MAXSTEP))
   dblwrd = rep(FALSE,2)
   ind = 2
   return(list(linemat = linemat, ips = ips, ind = ind, dblwrd = dblwrd))
}
#  ******************************************************
if ((linemat[2,3] - linemat[2,2]) * (linemat[1,3] - linemat[1,2]) > 0) {
   #  previous slope is negative
   z = c(3/(linemat[1,3] - linemat[1,2])) * (linemat[3,2] - linemat[3,3]) + 
           linemat[2,2] + linemat[2,3]
   w = z * z - linemat[2,2] * linemat[2,3]
   if (abs(linemat[2,2] + linemat[2,3] + 2 * z) >= 1e-05 && w > 0) {
           w = sqrt(w)
           linemat[1,5] = linemat[1,2] + 
               (1 - ((linemat[2,3] + w - z)/(linemat[2,3] - linemat[2,2] + 
                2 * w))) * (linemat[1,3] - linemat[1,2])
   } else {
     linemat[1,5] = linemat[1,2] - linemat[2,2] * 
                  ((linemat[1,3] - linemat[1,2])/ 
                   (linemat[2,3] - linemat[2,2]))
   }
   linemat[1,5] = min(c(linemat[1,5], MAXSTEP))
   dblwrd = rep(FALSE,2)
   ind = 2
   return(list(linemat = linemat, ips = ips, ind = ind, dblwrd = dblwrd))
} else {
   #  previous slope also negative but not as much
   if (dblwrd[1]) {
           ind = 5
           return(
             list(linemat = linemat, ips = ips, ind = ind, dblwrd = dblwrd))
   } else {
           linemat[1,5] = 2 * linemat[1,5]
           linemat[1,5] = min(c(linemat[1,5], MAXSTEP))
           dblwrd[2] = TRUE
           ind = 2
           return(
             list(linemat = linemat, ips = ips, ind = ind, dblwrd = dblwrd))
   }
}
ind = 2

}
                          "[.fd" <- function(fdobj, i=TRUE, j=TRUE, drop=TRUE) {
  
  #  Last moddified 16 January 2020
  
  if (!(is.fd(fdobj) || is.fdPar(fdobj)))  stop(
    "First argument is neither a functional data or a functional parameter object.")
  if (is.fdPar(fdobj)) fdobj <- fdobj$fd
  
  coef    <- as.array(fdobj$coefs)
  fdnames <- fdobj$fdnames
  coefdim <- dim(coef)
  ndim    <- length(coefdim)

  if(ndim == 2) {
    if (coefdim[2] == 1) {
      coefselect <- coef
    } else {
      coefselect <- coef[, i, drop=FALSE]
    }
    if (length(fdnames[[2]])>1) {
	    fdnames[[2]] = fdnames[[2]][i]
    }
  } else {
    if (coefdim[2] == 1) {
      coefselect <- coef
    } else {
      coefselect <- coef[, i, j,drop=drop]
    }
    if(length(fdnames[[2]])>1){
	    fdnames[[2]] = fdnames[[2]][i]
    }
    if(length(fdnames[[3]])>1){
	    fdnames[[3]] = fdnames[[3]][j]
    }
  }
  fd(coefselect, fdobj$basis, fdnames) 
}
symsolve <- function(Asym, Bmat)
{
  #  solves the system ASYM X = BMAT for X where ASYM is symmetric
  #  returns X
  n <- ncol(Asym)
  if (any(is.na(Asym))) stop("Asym has NA values.")
  if (max(abs(Asym-t(Asym)))/max(abs(Asym)) > 1e-10) {
    stop('Argument not symmetric.')
  } else {
    Asym <- (Asym + t(Asym))/2
  }
  Lmat <- chol(Asym)
  temp <- solve(t(Lmat),Bmat)
  Xmat <- backsolve(Lmat,temp)
  return(Xmat)
}
tperm.fd <- function(x1fd,x2fd,nperm=200,q=0.05,argvals=NULL,plotres=TRUE,...) # first and second 
{                                                                          # groups of data,
    if( !is.fd(x1fd) | !is.fd(x2fd) ){                                     # number permuts
        stop("x1fd and x2fd must both be functional data objects")         # quantile
    }                                                                      # where to evaluate
                                                                           # do I plot
    rangeobs = x1fd$basis$range
    rangehat = x2fd$basis$range


    if( !prod(rangeobs == rangehat) ){
        stop("x1fd and x2fd do not have the same range.")
    }

    if(is.null(argvals)){
        argvals = seq(rangeobs[1],rangeobs[2],length.out=101)
    }

    q = 1-q

    x1mat = eval.fd(argvals,x1fd)
    x2mat = eval.fd(argvals,x2fd)

    n1 = ncol(x1mat)
    n2 = ncol(x2mat)

    Xmat = cbind(x1mat,x2mat)

    Tnull = rep(0,nperm)

    Tnullvals = matrix(0,length(argvals),nperm)

    for(i in 1:nperm){
        tXmat = Xmat[,sample(n1+n2)]

        tmean1 = apply(tXmat[,1:n1],1,mean)
        tmean2 = apply(tXmat[,n1+(1:n2)],1,mean)

        tvar1 = apply(tXmat[,1:n1],1,var)/n1
        tvar2 = apply(tXmat[,n1+(1:n2)],1,var)/n2

        Tnullvals[,i] = abs(tmean1-tmean2)/sqrt(tvar1+tvar2)
        Tnull[i] = max(Tnullvals[,i])
    }

    mean1 = apply(Xmat[,1:n1],1,mean)
    mean2 = apply(Xmat[,n1+(1:n2)],1,mean)

    var1 = apply(Xmat[,1:n1],1,var)/n1
    var2 = apply(Xmat[,n1+(1:n2)],1,var)/n2

    Tvals = abs(mean1-mean2)/sqrt(var1+var2)
    Tobs = max(Tvals)

    pval = mean( Tobs < Tnull )
    qval = quantile(Tnull,q)

    pvals.pts = apply(Tvals<Tnullvals,1,mean)
    qvals.pts = apply(Tnullvals,1,quantile,q)

    if(plotres){

	  if( is.null(names(x1fd$fdnames)) | is.null(names(x2fd$fdnames)) ){
		xlab='argvals'
	  }	
	  else if( prod(names(x1fd$fdnames)[1] == names(x2fd$fdnames)[1]) ){
		xlab = names(x1fd$fdnames)[1]
	  }
	  else{ xlab = 'argvals' }

        ylims = c( min(Tvals,qvals.pts),max(Tobs,qval))

        plot(argvals,Tvals,type='l',col=2,ylim=ylims,lwd=2,
		xlab=xlab,ylab='t-statistic',...)
        lines(argvals,qvals.pts,lty=3,col=4,lwd=2)
        abline(h=qval,lty=2,col=4,lwd=2)
	
        legendstr = c('Observed Statistic',
			    paste('pointwise',1-q,'critical value'),
			    paste('maximum',1-q,'critical value'))

	  legend(argvals[1],ylims[2],legend=legendstr,col=c(2,4,4),
		lty=c(1,3,2),lwd=c(2,2,2))
    }


    return( list(pval=pval,qval=qval,Tobs=Tobs,Tnull=Tnull,
        Tvals=Tvals,Tnullvals=Tnullvals,qvals.pts=qvals.pts,
        pvals.pts=pvals.pts,argvals=argvals) )
}
trapzmat <- function(X,Y,delta=1,wt=rep(1,n)) {
#TRAPZMAT integrates the products of two matrices of values
#   using the trapezoidal rule, assuming equal spacing
#  X is the first  matrix of values
#  Y is the second matrix of values
#  DELTA is the spacing between argument values (one by default)
#  WT is a vector of weights (ones by default)
#
#  XtWY is a matrix of integral estimates, number of rows equal to
#  number of col of X, number of cols equal to number of cols of Y

	X <- as.matrix(X)
	Y <- as.matrix(Y)
	
	n <- dim(X)[1]

	if (dim(Y)[1] != n) {
    	stop("X and Y do not have same number of rows.")
	}

	if (length(wt) != n) {
    	stop("X and WT do not have same number of rows.")
	}

	if (delta <= 0) {
    	stop("DELTA is not a positive value.")
	}

	wt[c(1,n)] <- wt[c(1,n)]/2
	wt <- wt*delta

	X <- X*outer(wt,rep(1,dim(X)[2]))
	XtWY <- crossprod(X,Y)
	return(XtWY)
}

triplot <- function(p, t) {
  #  p ... NP by 2 matrix of point coordinates
  #  t ... NT by 3 (or 4) matrix of indices of triangle vertices in P
  
  #  Last modified 12 June 2015 by Jim Ramsay
  
  nt = dim(t)[1]
  np = dim(p)[1]
  plot(p, lwd=2, xlab="w", ylab="s")
  for (i in 1:nt) {
    lines(c(p[t[i,1],1],p[t[i,2],1]), c(p[t[i,1],2],p[t[i,2],2]), lwd=2)
    lines(c(p[t[i,2],1],p[t[i,3],1]), c(p[t[i,2],2],p[t[i,3],2]), lwd=2)   
    lines(c(p[t[i,3],1],p[t[i,1],1]), c(p[t[i,3],2],p[t[i,1],2]), lwd=2)    
  } 
}
var.fd <- function(fdobj1, fdobj2=fdobj1)
{
  #  compute the variance and covariance functions for functional observations

  #  Last modified 16 January 2010

  if (!(is.fd(fdobj1) || is.fdPar(fdobj1)))  stop(
    "First argument is neither a functional data or a functional parameter object.")
  if (is.fdPar(fdobj1)) fdobj1 <- fdobj1$fd
  
  if (!(is.fd(fdobj2) || is.fdPar(fdobj2)))  stop(
    "Second argument is neither a functional data or a functional parameter object.")
  if (is.fdPar(fdobj2)) fdobj2 <- fdobj2$fd
  
  coefx   <- fdobj1$coefs
  coefy   <- fdobj2$coefs
  coefdobj1  <- dim(coefx)
  coefdobj2  <- dim(coefy)
  basisx  <- fdobj1$basis
  basisy  <- fdobj2$basis
  nbasisx <- basisx$nbasis
  nbasisy <- basisy$nbasis

  if (coefdobj1[2] != coefdobj2[2]) stop(
    	"Number of replications are not equal.")
  if (length(coefdobj1) == 2) {
    	if(length(coefdobj2) == 2) {
      		coefvar   <- var(t(coefx),t(coefy))
      		coefnames <- list(dimnames(coefx)[[1]], dimnames(coefy)[[1]])
      		varbifd   <- bifd(coefvar, basisx, basisy, coefnames)
    	} else stop("Both arguments must be univariate.")
  } else {
    	nvar    <- coefdobj1[3]
    	npair   <- nvar*(nvar+1)/2
    	coefvar <- array(0,c(nbasisx,nbasisx,1,npair))
       varnames <- fdobj1$fdnames[[3]]
    	m <- 0
       bivarnames <- vector("list",npair)
    	for (i in 1:nvar) for (j in 1:i) {
      		m <- m + 1
      		coefvar[,,1,m] <- var(t(coefx[,,i]),t(coefx[,,j]))
          bivarnames[m] <- paste(varnames[i],"vs",varnames[j])
    	}
       bifdnames = fdobj1$fdnames
       bifdnames[[3]] <- bivarnames
    	varbifd <- bifd(coefvar, basisx, basisx, bifdnames)
  }
  return(varbifd)
}
varmx.cca.fd <- function(ccafd, nx=201)
{
  #  VARMX_CCA  Apply varimax rotation to CCA weight functions
  #             and scores
  #  Arguments:
  #  CCAFD ... An object of the CCA.FD class produced by a call to
  #            function CCA.FD.
  #  CCAWTFD2 ... A functional parameter object for the canonical weight
  #                functions for the second set of functions.
  #  CCAVAR1  ... Canonical variate scores for first  set of functions.
  #  CCAVAR2  ... Canonical variate scores for second set of functions.
  #  Return:  Arguments after rotation.
  
  #  last modified 16 January 2020
  
  if (!inherits(ccafd, "cca.fd")) stop("First argument is not a class cca.fd object.")

  ccawtfd1 <- ccafd[[1]]
  ccawtfd2 <- ccafd[[2]]
  ccavar1  <- ccafd[[4]]
  ccavar2  <- ccafd[[5]]

  wtcoef1 <- ccawtfd1$coefs
  wtcoef2 <- ccawtfd2$coefs

  basisobj <- ccawtfd1$basis
  rangex   <- basisobj$rangeval
  x        <- seq(rangex[1], rangex[2], len=nx)
  ccawtmat1 <- eval.fd(x, ccawtfd1)
  ccawtmat2 <- eval.fd(x, ccawtfd2)
  #  If fdmat is a 3-D array, stack into a matrix
  ccawtmat  <- rbind(ccawtmat1, ccawtmat2)
  #  compute rotation matrix for varimax rotation of ccawtmat
  rotmat <- varmx(ccawtmat)
  #  rotate coefficients and scores
  wtcoef1 <- wtcoef1 %*% rotmat
  wtcoef2 <- wtcoef2 %*% rotmat
  #  rotate ccawt objects
  ccawtfd1$coefs <- wtcoef1
  ccawtfd2$coefs <- wtcoef2
  #  rotate cca scores
  ccavar1 <- ccavar1 %*% rotmat
  ccavar2 <- ccavar2 %*% rotmat
  ccavard <- dim(ccavar1)
  canvarvalues      <- array(0, c(ccavard[1], ccavard[2], 2))
  canvarvalues[,,1] <- ccavar1
  canvarvalues[,,2] <- ccavar2

  ccavarmxlist <- list(weight1  = ccawtfd1, 
                       weight2  = ccawtfd2, 
                       variates = canvarvalues, 
                       rotation = rotmat)
                       
  class(ccavarmxlist) <- "cca.fd"
  
  return(ccavarmxlist)

}
varmx.pca.fd <- function(pcafd, nharm=scoresd[2], nx=501)
{
  #
  #  Apply varimax to the first NHARM components of a pca.fd object.
  #  Evaluates the harmonics at NX equally spaced points.
  #
  #  Returns:
  #  An object of class pcafd
  
  #  Note that pcafd is an oldClass type object
  
  #  Last modified 16 January 2020 by Jim Ramsay
  
  if (!(inherits(pcafd, "pca.fd"))) stop(
    "Argument PCAFD is not a pca.fd object.")
  
  harmfd   <- pcafd$harmonics
  harmcoef <- harmfd$coefs
  coefd    <- dim(harmcoef)
  ndim     <- length(coefd)
  
  scoresd  <- dim(pcafd$scores)
  if (nharm > scoresd[2]) nharm <- scoresd[2]
  
  basisobj <- harmfd$basis
  rangex   <- basisobj$rangeval
  x        <- seq(rangex[1], rangex[2], length = nx)
  delta    <- x[2]-x[1]
  harmmat  <- eval.fd(x, harmfd)
  
  #  If fdmat is a 3-D array, stack into a matrix
  
  if (ndim == 3) {
    harmmatd <- dim(harmmat)
    dimnames(harmmat) <- NULL
    harmmat  <- aperm(harmmat, c(1, 3, 2))
    dim(harmmat) <- c(harmmatd[1] * harmmatd[3], harmmatd[2])
  }
  
  #  compute rotation matrix for varimax rotation of harmmat
  
  rotm <- varmx(harmmat[,1:nharm])
  
  #  rotate coefficients and scores
  
  rotharmcoef <- harmcoef
  if (ndim == 2)
    rotharmcoef[,1:nharm] <- harmcoef[,1:nharm] %*% rotm
  else
    for(j in 1:coefd[3])
      rotharmcoef[,1:nharm,j] <- harmcoef[,1:nharm,j] %*% rotm
  
  #  rotate principal component scores
  
  rotharmscrs <- pcafd$scores		
  if (ndim == 2)
    rotharmscrs[,1:nharm] <- rotharmscrs[,1:nharm] %*% rotm
  else
    for (j in 1:coefd[3])
      rotharmscrs[,1:nharm,j] <- rotharmscrs[,1:nharm,j] %*% rotm
  
  
  #  compute proportions of variance
  
  rotharmvar <- apply(rotharmscrs^2,2,mean)
  varsum  <- sum(rotharmvar)
  propvar <- sum(pcafd$varprop)*rotharmvar/varsum
  
  #  modify pcafd object
  
  rotharmfd <- harmfd
  rotharmfd$coefs <- rotharmcoef
   
  rotpcafd           <- pcafd
  rotpcafd$harmonics <- rotharmfd
  rotpcafd$varprop   <- propvar
  rotpcafd$scores    <- rotharmscrs
  rotpcafd$rotmat    <- rotm
  rotpcafd$values   <- rotharmvar
  
  return(rotpcafd)
}

varmx <- function(amat, normalize=FALSE) {

  #  Does a VARIMAX rotation of a principal components solution

  #  Arguments:
  #  AMAT      ...  N by K matrix of harmonic values
  #  NORMALIZE ... either TRUE or FALSE.  If TRUE, the columns of AMAT
  #                are normalized prior to computing the rotation 
  #                matrix.  However, this is seldom needed for 
  #                functional data.

  #  Returns:
  #  ROTM  ...  Rotation matrixed loadings

  #  Last modified 22 October by Jim Ramsay

  n    <- nrow(amat)
  k    <- ncol(amat)
  rotm <- diag(k)
  onek <- matrix(1,1,k)

  if (k == 1) return(rotm)

  #  normalize loadings matrix

  if (normalize) {
      hvec <- as.matrix(apply(amat^2, 1, var))
      amat <- amat/(sqrt(hvec) %*% onek)
  }

  eps  <- 0.0011
  ccns <- 0.7071068

  varold <- 0
  varnow <- sum(apply(amat^2, 2, var))

  iter <- 0
  while (abs(varnow - varold) > 1e-7 && iter <= 50) {
    iter  <- iter + 1
    for (j in 1:(k-1)) for (l in (j+1):k) {
      avecj  <- amat[,j]
      avecl  <- amat[,l]
      uvec   <- avecj^2 - avecl^2
      tvec   <- 2*avecj*avecl
      aa <- sum(uvec)
      bb <- sum(tvec)
      cc <- sum(uvec^2 - tvec^2)
      dd <- 2*sum(uvec*tvec)
      tval <- dd - 2*aa*bb/n
      bval <- cc - (aa^2 - bb^2)/n

      if (tval == bval) {
        sin4t <- ccns
        cos4t <- ccns
      }

      if (tval < bval) {
        tan4t <- abs(tval/bval)
        if (tan4t >= eps) {
          cos4t <- 1/sqrt(1+tan4t^2)
          sin4t <- tan4t*cos4t
        } else {
          if (bval < 0) {
            sin4t <- ccns
            cos4t <- ccns
          } else {
            sin4t <- 0
            cos4t <- 1
          }
        }
      }

      if (tval > bval) {
        ctn4t <- abs(tval/bval)
        if (ctn4t >= eps) {
          sin4t <- 1/sqrt(1+ctn4t^2)
          cos4t <- ctn4t*sin4t
        } else {
          sin4t <- 1
          cos4t <- 0
        }
      }

      cos2t <- sqrt((1+cos4t)/2)
      sin2t <- sin4t/(2*cos2t)
      cost  <- sqrt((1+cos2t)/2)
      sint  <- sin2t/(2*cost)
      if (bval > 0) {
        cosp <- cost
        sinp <- sint
      } else {
        cosp <- ccns*(cost + sint)
        sinp <- ccns*abs(cost - sint)
      }
      if (tval <= 0) sinp <- -sinp

      amat[,j] <-  avecj*cosp + avecl*sinp
      amat[,l] <- -avecj*sinp + avecl*cosp
      rvecj    <- rotm[,j]
      rvecl    <- rotm[,l]
      rotm[,j] <-  rvecj * cosp + rvecl * sinp
      rotm[,l] <- -rvecj * sinp + rvecl * cosp

    }
    varold <- varnow
    varnow <- sum(apply(amat^2,2,var))
  }

  return( rotm )
}
vec2Lfd <- function(bwtvec, rangeval=c(0,1))
{
#VEC2LFD converts a vector of length m to a linear differential
#  operator object of order m.  The range of the
#  functional data object in any cell is set to RANGEVAL.
#  In the event that BWTVEC is already a linear differential operator
#  object, it returns the object.

# Last modified 13 June 2013 by Spencer Graves
# Previously modified 10 December 2005

#  return BWTVEC if it is of class LFD

if (inherits(bwtvec, "Lfd")) {
    Lfdobj <- bwtvec
    return(Lfdobj)
}

#  check BWTVEC
if(!is.numeric(bwtvec))
    stop('first argument is not a linear differential operator ',
         'and is not numeric.')

isVec <- is.vector(bwtvec)
if(!isVec){
    bdim <- dim(bwtvec)
    if(is.null(bdim))
        stop('first argument is neither a linear differential operator ',
             'nor a vector (but is numeric)')
    if(length(bdim)>2)
        stop('first argument is neither a linear differential operator ',
             'nore a vector (but is an array of more than 2 dimensions).')
    if(!(1 %in% bdim))
        stop('first argument is neither a linear differential operator ',
             'nor a vector (but is a numeric matrix)')
    bwtvec <- as.numeric(bwtvec)
}

#if (!(is.vector(bwtvec) ||
#    stop("Argument not a vector and not a linear differential operator.")

m <- length(bwtvec)

#  set up the list object for the homogeneous part

if (m==0) {
    #  if derivative is zero, BWTLIST is empty
    bwtlist <- NULL
} else {
    conbasis <- create.constant.basis(rangeval)
    bwtlist  <- vector("list", m)
    for (j in 1:m) bwtlist[[j]] <- fd(bwtvec[j], conbasis)
}

#  define the Lfd object

Lfdobj <- Lfd(m, bwtlist)

return(Lfdobj)

}
wtcheck = function(n, wtvec=NULL) {
#  WTVEC is either a vector or a matrix, and WTCHECK applies
#  a number of tests to confirm that will served as a weight
#  vector or matrix for smoothing data.  n is the required length
#  or order of WTVEC.  The functions returns WTVEC, ONEWT indicating
#  whether WTVEC contains only one's, and MATWT indicating that
#  WTVEC is in fact a square positive definite matrix.

#  Last modified 9 July 2011 by Jim Ramsay

#  check n

if (n != round(n)) stop("n is not an integer.")
if (n < 1)         stop("n is less than 1.")

#  check wtvec

if (!is.null(wtvec)) {
    dimw = dim(as.matrix(wtvec))
    if (any(is.na(as.vector(wtvec)))) stop("WTVEC has NA values.")
    if (all(dimw == n)) {
        #  WTVEC is a matrix of order n
        onewt = FALSE
        matwt = TRUE
        #  check weight matrix for being positive definite
        wteig = eigen(wtvec)$values
        if (any(is.complex(wteig))) stop("Weight matrix has complex eigenvalues.")
        if (min(wteig) <= 0)        stop("Weight matrix is not positive definite.")
    } else {
        #  WTVEC is treated as a vector
        if ((length(dimw) > 1 && dimw[1] > 1 && dimw[2] > 1) || length(dimw) > 2) {
            stop ("WTVEC is neither a vector nor a matrix of order n.")
        }
        wtvec = as.matrix(wtvec)
        if (length(wtvec) == 1) {
            wtvec = wtvec*matrix(1,n,1)
        }
        if (length(wtvec) != n) {
            stop("WTVEC of wrong length")
        }
        if (min(wtvec) <= 0) stop("Values in WTVEC are not positive.")
        onewt = FALSE	
        matwt = FALSE
    }
} else {
    wtvec = matrix(1,n,1)
    onewt = TRUE
    matwt = FALSE
}

return(list(wtvec=wtvec, onewt=onewt, matwt=matwt))

}

ycheck = function(y, n) {

#  check Y

if (is.vector(y)) y <- as.matrix(y)

if (!inherits(y, "matrix") && !inherits(y, "array"))
    stop("Y is not of class matrix or class array.")

ydim = dim(y)

if (ydim[1] != n) stop("Y is not the same length as ARGVALS.")

#  set number of curves and number of variables

ndim  = length(ydim)
if (ndim == 2) {
        ncurve = ydim[2]
        nvar   = 1
}
if (ndim == 3) {
        ncurve = ydim[2]
        nvar   = ydim[3]
}
if (ndim > 3) stop("Second argument must not have more than 3 dimensions")


return(list(y=y, ncurve=ncurve, nvar=nvar, ndim=ndim))

}


yListCheck = function(yList, nvar) {
  
  #  Last modified 3 August 2017
  
  if (!is.list(yList)) {
    stop("YLIST is not a list.")
  }
  errwrd = FALSE
  
  nvec    = rep(0,nvar)
  dataWrd = rep(FALSE,nvar)
  ydim    = rep(0,nvar)
  nobs    = 0
  #  find number of replications for (first non-empty cell
  for (ivar in 1:nvar) {
    if (is.list(yList[[ivar]])) {
      yListi = yList[[ivar]]
      nrep = dim(as.matrix(yListi$y))[2]
      break
    }
  }
  #  loop through variables
  for (ivar in 1:nvar) {
    if (is.list(yList[[ivar]]) && !is.null(yList[[ivar]]$y)) {
      dataWrd[ivar] = TRUE
      yListi = yList[[ivar]]
      if (is.null(yListi$argvals)) {
        warning(paste("ARGVALS is not a member for (YLIST[[", ivar,"]]."))
        errwrd = TRUE
      }
      ni = length(yListi$argvals)
      nvec[ivar] = ni
      ydimi = dim(as.matrix(yListi$y))
      if (length(ydimi) > 2) {
        warning(paste("More than two dimensions for (y in YLIST[[",
                      ivar,"]]."))
        errwrd = TRUE
      } else {
        ydim[ivar] = ydimi[1]
      }
      #  set up and check NREP
      nrepi = ydimi[2]
      if (nrepi != nrep) {
        warning("Second dimensions of YList.y are not equal.")
        errwrd = TRUE
      }
      nobs = nobs + 1
      if (ni != ydimi[1]) {
        print(c(ni,ydimi[1]))
        warning(paste("Length of ARGVALS and first dimension of Y",
                      "are not equal."))
        errwrd = TRUE
      }
    } else {
      dataWrd[ivar] = FALSE
    }
  }
  
  if (nobs == 0) {
    warning("No variables have observations.")
    errwrd = TRUE
  }
  
  if (errwrd) {
    stop("One or more terminal stop encountered in YLIST.")
  }
  
  return(list(nrep = nrep, nvec = nvec, dataWrd = dataWrd))
  
}
zerobasis <- function(k) {
# ZEROBASIS constructes a K by K-1 matrix that maps an unrestricted matrix B with K - 1 rows by 
#  the linear transformation ZEROBASIS %*% B = C into the subspace of matrices with K rows having #  column sums equal to zero.  
#  The matrix has orthonormal columns, so that crossprod(ZEROBASIS) is the identity matrix
#  of order K - 1.

  tk <- 0:(k-1) + 0.5
  fbasis     <- create.fourier.basis(k,k)
  fbasmat    <- eval.basis(tk, fbasis)
  fbasmat    <- fbasmat[,2:k]
  fbasnorm   <- sqrt(apply(fbasmat^2,2,sum))
  zerobasmat <- fbasmat/outer(rep(1,k),fbasnorm)
  return(zerobasmat)
}

zerofind <- function(fmat) {
  if (!is.numeric(fmat)) stop("Argument is not numeric.")
  pairNums <- c(min(fmat),max(fmat))
  if (length(pairNums) != 2) stop("Argument is not of length 2.")
  if (min(pairNums) <= 0 && max(pairNums) >= 0) return(TRUE) else return(FALSE)
}
