#######################################################################
## Testing psicologico (PSP6075525)
### Modelli e metodi statistici per la misurazione in psicologia
## A.A. 2023/2024
## prof. Antonio Calcagnì (antonio.calcagni@unipd.it)
#######################################################################


## CONTENUTO DEL CODICE ###############################################
# (A) Analisi esplorative vs. confermative
# (B) Validazione incrociata 1: step-by-step
# (C) Validazione incrociata 2: using the function kFold_validation()
# (D) Clustering gerarchico
#######################################################################


# Inizializzazione ambiente di lavoro -------------------------------------
rm(list=ls()); graphics.off()
setwd("~/MEGA/Lavoro_sync/Didattica/2023_2024/testing_psicologico/laboratorio/") #change it according to your local path!
library(lavaan); library(semPlot)
source("utilities.R")


# (A) Analisi esplorative vs. confermative --------------------------------
srs = read.csv(file = "data/srs.csv",header = TRUE)[,2:21] #non consideriamo la prima colonna
srs = scale(srs)                                           #standardizzazione
apply(srs,2,function(x)c(mean(x),var(x)))                  #controllo finale

out = split_dataset(data = srs,prop = 0.4,seedx = 112)     #divisione a metà del dataset (primo sottoinsieme: 40% dei dati originali)
srs_A = out$A       #dataset per le analisi esplorative
srs_B = out$B       #dataset per le analisi confermative

## Analisi esplorative
library(corrplot); 
x11();corrplot.mixed(corr = cor(srs_A), upper = "circle")
cols = colorRampPalette(c("blue", "white", "red"))(20)
x11();stats::heatmap(x = cor(srs_A), col = cols, symm = TRUE)

model1 = paste0("eta=~",paste0(colnames(srs),collapse = "+"))
model2 = "eta1=~V11+V17+V14+V13+V4+V3+V18+V1+V2+V8 \n eta2=~V12+V15+V9+V5+V7+V16+V6+V20+V10+V19"

## Analisi confermative
model1_fit = cfa(model = model1,data = srs_B)
summary(model1_fit,standardized=TRUE)
fitmeasures(object = model1_fit,fit.measures = c("rmsea","cfi","npar","srmr"))

model2_fit = cfa(model = model2,data = srs_B)
summary(model2_fit,standardized=TRUE)
fitmeasures(object = model2_fit,fit.measures = c("rmsea","cfi","npar","srmr"))

anova(model1_fit,model2_fit,test="Chisq")
fitMeasures_models(c(model1_fit,model2_fit))

# (B) Validazione incrociata 1 ---------------------------------------------
## Nota: questa sezione mostra step-by-step come funziona ed è implementata la funzione 
## kFold_validation() presente nel file utilities.R (Può essere saltata se non si è interessati a comprenderne il funzionamento).
## L'obiettivo qui è descrivere come la funzione opera per il calcolo dell'errore previsionale prodotto da due modelli CFA utilizzando
## la logica della validazione incrociata (cross-validation).

n = NROW(srs)                                         #numero di unità stats del dataset
B = 100                                               #numero di repliche Monte Carlo (per il calcolo dell'errore di previsione)
nfold = 5                                             #numero di partizioni del dataset originario
Output1 = matrix(NA, nrow = nfold, ncol = B)           
Output2 = matrix(NA, nrow = nfold, ncol = B)           
folds = cut(seq(1,n), breaks=nfold, labels=FALSE)
print(cbind(1:n,folds))

for(h in 1:nfold){
  iid = which(folds==h)         #indici per la h-esima partizione (sottoinsieme corrente)
  not_iid = setdiff(1:n,iid)
  test = srs[iid,]              #dataset di training
  train = srs[not_iid,]         #dataset di test
  
  model1_train = cfa(model = model1,data = train)   #adattamento del modello 1 al dataset di training
  model2_train = cfa(model = model2,data = train)   #adattamento del modello 2 al dataset di training
  
  out = lavInspect(object = model1_train,what = "est")    #calcolo della matrice di cov riprodotta dal modello 1
  Lambda = out$lambda; Phi = out$psi; Thetad = out$theta
  Sigma_hat1 = Lambda%*%Phi%*%t(Lambda) + Thetad
  
  out = lavInspect(object = model2_train,what = "est")    #calcolo della matrice di cov riprodotta dal modello 2
  Lambda = out$lambda; Phi = out$psi; Thetad = out$theta
  Sigma_hat2 = Lambda%*%Phi%*%t(Lambda) + Thetad
  
  err = rep(NA,B)                                        #calcolo della distanza tra cov del dataset di test e cov del b-esimo dataset generato da modello vero
  for(b in 1:B){                                         #errore di previsione
    data_predicted = mvtnorm::rmvnorm(n,sigma = Sigma_hat1)
    Output1[h,b] =  norm(cov(data_predicted)-cov(test))
    
    data_predicted = mvtnorm::rmvnorm(n,sigma = Sigma_hat2)
    Output2[h,b] =  norm(cov(data_predicted)-cov(test))  
  }
  
}
err1 = apply(Output1,1,mean) #errore di previsione medio su B campioni simulati - modello 1
err2 = apply(Output2,1,mean) #errore di previsione medio su B campioni simulati - modello 2
Err = cbind(err1,err2)
print(Err)
summary(Err)

apply(Err,2,mean)
apply(Err,2,sd)/apply(Err,2,mean)

# Scelta di nfold: non deve essere troppo alto il numero di partizioni se n è piccolo!
n/nfold
# In questo caso ogni partizione avrà un dataset di 100 unità su cui fare il training del modello.
# Per una discussione sul tema, si veda: 
# https://stats.stackexchange.com/questions/61783/bias-and-variance-in-leave-one-out-vs-k-fold-cross-validation/357749#357749



# (C) Validazione incrociata 2 --------------------------------------------
nfold = 5; B = 100;
err1 = kFold_validation(model_definition = model1,data = srs_B,error = "montecarlo",B = B,nfold = nfold)
err2 = kFold_validation(model_definition = model2,data = srs_B,error = "montecarlo",B = B,nfold = nfold)

Err = cbind(err1,err2)
print(Err)
summary(Err)

apply(Err,2,mean)
apply(Err,2,sd)/apply(Err,2,mean)



# (D) Clustering gerarchico -----------------------------------------------
# Source: https://uc-r.github.io/hc_clustering
# Look also at: http://www.econ.upf.edu/~michael/stanford/maeb7.pdf
library(factoextra)

## Dati
Corx = psych::Tucker 
# Tucker, Ledyard (1958) An inter-battery method of factor analysis, Psychometrika, 23, 111-136
# Maggiori informazioni: ?psych::Tucker

class(Corx)                       #i dati sono in forma di dataframe
Corx = as.matrix(Corx)            #trasformazione da dataframe in matrice numerica

x11(); corrplot::corrplot(corr = Corx,method = "color") #grafico della matrice di correlazione
x11(); corrplot::corrplot(corr = Corx,method = "color",hclust.method = "ward.D2",order = "hclust") #..ordinando le vars con hclust

Distx = factoextra::get_dist(x = Corx,method = "euclidean") #equivalente a: dist(Corx)
x11();factoextra::fviz_dist(dist.obj = Distx)


## Agglomerative clustering:
# It’s also known as AGNES (Agglomerative Nesting). It works in a bottom-up manner. 
# That is, each object is initially considered as a single-element cluster (leaf). At each step of the algorithm, 
# the two clusters that are the most similar are combined into a new bigger cluster (nodes). 
# This procedure is iterated until all points are member of just one single big cluster (root). 
# The result is a tree which can be plotted as a dendrogram.

# How do we measure the dissimilarity between two clusters of observations?
# (1) Average linkage clustering: It computes all pairwise dissimilarities between the elements in 
# cluster 1 and the elements in cluster 2, and considers the average of these dissimilarities as the distance between the two clusters.
# (2) Ward's minimum variance method: It minimizes the total within-cluster variance. 
# At each step the pair of clusters with minimum between-cluster distance are merged.

hc_ward = cluster::agnes(x = Distx,metric = "euclidean",method = "ward")      #equivalente a: hclust(d = Distx,method = "ward.D2")
hc_avg = cluster::agnes(x = Distx,metric = "euclidean",method = "average")    #equivalente a: hclust(d = Distx,method = "ward.D2")

x11(); par(mfrow=c(1,2));  #dendrogrammi
cluster::pltree(hc_ward,main="ward"); cluster::pltree(hc_avg,main="average")
x11();factoextra::fviz_dend(hc_ward,k = 2) #un altro modo per fare un dendrogramma

gps_ward = cutree(tree = hc_ward,k = 2) #estrazione dei 2 gruppi
gps_avg = cutree(tree = hc_avg,k = 2)

x11();fviz_cluster(list(data = Corx, cluster = gps_ward)) #altra posibile rappresentazione grafica
x11();fviz_cluster(list(data = Corx, cluster = gps_avg))

## Calcolo della qualità del raggruppamento in clusters: Silhouette index (Rousseeuw).
# The silhouette value is a measure of how similar an object is to its own cluster (cohesion) compared to other clusters (separation). 
# The silhouette ranges from −1 to +1, where a high value indicates that the object is well matched to its own cluster and poorly matched to neighboring clusters. 
# If most objects have a high value, then the clustering configuration is appropriate. If many points have a low or negative value, 
# then the clustering configuration may have too many or too few clusters. Source: https://en.wikipedia.org/wiki/Silhouette_(clustering)

compute_silhouette(Dist = Distx,hc = hc_ward,n_clusters = c(2,3,4)) #caso del raggruppamento secondo ward
# La soluzione con k=2 è quella che presenta più alta silhouette
# In alternativa: clusterSim::index.S(d = Distx,cl = gps_ward)

compute_silhouette(Dist = Distx,hc = hc_avg,n_clusters = c(2,3,4)) #caso del raggruppamento secondo avg
# La soluzione con k=2 è quella che presenta più alta silhouette


## Back to CFA
# Una volta trovato il raggruppamento gerarchico ottimale (se esiste!) è possibile utilizzare quest'ultimo per definire un modello confermativo appropriato.
# Per far ciò utilizziamo la funzione apposita presente in utilities.R
mod_ward = hclust2lavaan(tree = hc_ward,ngroups = 2)
print(mod_ward)
# La variabile 'mod_ward' può così essere utilizzata nella funzione cfa(..) di lavaan.




