#######################################################################
## Testing psicologico (PSP6075525)
### Modelli e metodi statistici per la misurazione in psicologia
## A.A. 2023/2024
## prof. Antonio Calcagnì (antonio.calcagni@unipd.it)
#######################################################################


## CONTENUTO DEL CODICE ###############################################
# (A) Clustering gerarchico
# (B) Analisi in componenti principali (PCA)
#######################################################################


# Inizializzazione ambiente di lavoro -------------------------------------
rm(list=ls()); graphics.off()
setwd("~/MEGA/Lavoro_sync/Didattica/2023_2024/testing_psicologico/laboratorio/") #change it according to your local path!
library(lavaan); library(factoextra)
source("utilities.R")


# (A) Clustering gerarchico -----------------------------------------------
# Source: https://uc-r.github.io/hc_clustering
# Look also at: http://www.econ.upf.edu/~michael/stanford/maeb7.pdf

## Dati
Corx = psych::Tucker 
# Tucker, Ledyard (1958) An inter-battery method of factor analysis, Psychometrika, 23, 111-136
# Maggiori informazioni: ?psych::Tucker

class(Corx)                       #i dati sono in forma di dataframe
Corx = as.matrix(Corx)            #trasformazione da dataframe in matrice numerica

x11(); corrplot::corrplot(corr = Corx,method = "color") #grafico della matrice di correlazione
x11(); corrplot::corrplot(corr = Corx,method = "color",hclust.method = "ward.D2",order = "hclust") #..ordinando le vars con hclust

Distx = factoextra::get_dist(x = Corx,method = "euclidean") #equivalente a: dist(Corx)
x11();factoextra::fviz_dist(dist.obj = Distx)


## Agglomerative clustering:
# It’s also known as AGNES (Agglomerative Nesting). It works in a bottom-up manner. 
# That is, each object is initially considered as a single-element cluster (leaf). At each step of the algorithm, 
# the two clusters that are the most similar are combined into a new bigger cluster (nodes). 
# This procedure is iterated until all points are member of just one single big cluster (root). 
# The result is a tree which can be plotted as a dendrogram.

# How do we measure the dissimilarity between two clusters of observations?
# (1) Average linkage clustering: It computes all pairwise dissimilarities between the elements in 
# cluster 1 and the elements in cluster 2, and considers the average of these dissimilarities as the distance between the two clusters.
# (2) Ward's minimum variance method: It minimizes the total within-cluster variance. 
# At each step the pair of clusters with minimum between-cluster distance are merged.

hc_ward = cluster::agnes(x = Distx,metric = "euclidean",method = "ward")      #equivalente a: hclust(d = Distx,method = "ward.D2")
hc_avg = cluster::agnes(x = Distx,metric = "euclidean",method = "average")    #equivalente a: hclust(d = Distx,method = "ward.D2")

x11(); par(mfrow=c(1,2));  #dendrogrammi
cluster::pltree(hc_ward,main="ward"); cluster::pltree(hc_avg,main="average")
x11();factoextra::fviz_dend(hc_ward,k = 2) #un altro modo per fare un dendrogramma

gps_ward = cutree(tree = hc_ward,k = 2) #estrazione dei 2 gruppi
gps_avg = cutree(tree = hc_avg,k = 2)

x11();fviz_cluster(list(data = Corx, cluster = gps_ward)) #altra posibile rappresentazione grafica
x11();fviz_cluster(list(data = Corx, cluster = gps_avg))

## Calcolo della qualità del raggruppamento in clusters: Silhouette index (Rousseeuw).
# The silhouette value is a measure of how similar an object is to its own cluster (cohesion) compared to other clusters (separation). 
# The silhouette ranges from −1 to +1, where a high value indicates that the object is well matched to its own cluster and poorly matched to neighboring clusters. 
# If most objects have a high value, then the clustering configuration is appropriate. If many points have a low or negative value, 
# then the clustering configuration may have too many or too few clusters. Source: https://en.wikipedia.org/wiki/Silhouette_(clustering)

sil=list(); kk=c(2,3,4)         #calcolo della qualità dei raggruppamenti (mediante silhouette) per k={2,3,4} possibili gruppi
for(k in 1:length(kk)){
  sil[[k]] = cluster::silhouette(cutree(hc_ward,kk[k]),Distx) #caso del raggruppamento secondo Ward
}
lapply(sil,average_silhouette) #estrazione della silhouette media (nota: sil è una lista)
# La soluzione con k=2 è quella che presenta più alta silhouette
# In alternativa: clusterSim::index.S(d = Distx,cl = gps_ward)

for(k in 1:length(kk)){
  sil[[k]] = cluster::silhouette(cutree(hc_avg,kk[k]),Distx) #caso del raggruppamento secondo avg
}
lapply(sil,average_silhouette) #estrazione della silhouette media (nota: sil è una lista)
# La soluzione con k=2 è quella che presenta più alta silhouette

x11();par(mfrow=c(1,3))     #rappresentazione grafica delle silhouette
for(k in 1:length(kk)){
  plot(sil[[k]])
}

## Back to CFA
# Una volta trovato il raggruppamento gerarchico ottimale (se esiste!) è possibile utilizzare quest'ultimo per definire un modello confermativo appropriato.
# Per far ciò utilizziamo la funzione apposita presente in utilities.R
mod_ward = hclust2lavaan(tree = hc_ward,ngroups = 2)
print(mod_ward)
# La variabile 'mod_ward' può così essere utilizzata nella funzione cfa(..) di lavaan.



# (C) Analisi in componenti principali (PCA) ------------------------------
# Source: https://ocw.mit.edu/courses/18-650-statistics-for-applications-fall-2016/d85e1a9d113142ade8ce5e4f5ef0b4e8_MIT18_650F16_PCA.pdf
# Look also at: 
#     https://stats.stackexchange.com/questions/2691/making-sense-of-principal-component-analysis-eigenvectors-eigenvalues
#     https://uc-r.github.io/pca


## Dati
X = factoextra::decathlon2[1:23,1:10]
# Dettagli su: ?factoextra::decathlon2

# Rappresentazione grafica di alcune triple di variabili in X
# In practice, p is large. If p > 3, it becomes impossible to represent the cloud on a picture.
x11(); par(mfrow=c(1,3))
scatterplot3d::scatterplot3d(X[,1],X[,2],X[,3],color = "darkblue",pch = 20,box = FALSE)
scatterplot3d::scatterplot3d(X[,1],X[,2],X[,4],color = "darkblue",pch = 20,box = FALSE)
scatterplot3d::scatterplot3d(X[,1],X[,8],X[,9],color = "darkblue",pch = 20,box = FALSE)

# Is it possible to project the cloud onto a linear subspace of dimension p' < p by keeping as much information as possible ?
# PCA does this by keeping as much covariance structure as possible by keeping orthogonal directions that discriminate well the points of the cloud.
X = scale(X)
out = eigen(x = cov(X),symmetric = TRUE)  #spectral decomposition of cov(X)
h = out$values                            #eigenvalues 
D = out$vectors                           #eigenvectors

D%*%diag(h)%*%t(D)                        #decomposizione della matrice cov(X)
# D: matrice nxp delle componenti principali
# h: varianze spiegate da ciascuna colonna di D (da componente principale)

# How to choose p' ? 
# Experimental rule: Take p' where there is an inflection point in the sequence h1,..,hp (scree plot).
exv = cumsum(h)/sum(h)
plot(1:length(exv),exv,bty="n",type="b")   #consideriamo le prime due componenti che spiegano quasi il 60% della varianza
abline(v = 2,lty=2)

PC = -D[,1:2]    #componenti principali (matrice dei loadings)
Projs = X%*%PC   #proiezioni delle unità statistiche nel nuovo sottospazio a p' dimensioni
x11();plot(Projs[,1],Projs[,2],bty="n",xlab="PC1",ylab="PC2",xlim=c(-7,7),ylim=c(-4,4),pch=20,cex=2,col="lightblue")
abline(h = 0,v = 0,lty=2)
text(x = Projs[,1],y = Projs[,2],labels = rownames(X),pos = 1,cex = 0.5)

# Migliorare l'interpretazione della PCA mediante rotazione della matrice dei loadings
# Take a look at: https://en.wikipedia.org/wiki/Varimax_rotation
out = GPArotation::Varimax(L = PC)  #calcolo della matrice di rotazione con algoritmo iterativo Varimax
Th = out$Th                         #matrice di rotazione
PC_th = PC%*%solve(t(Th))           #matrice delle componenti principali ruotata

Projs = X%*%PC_th   
points(Projs[,1],Projs[,2],pch=20,cex=2,col="lightgreen") #unità statistiche ruotate

## Back to CFA
# Una volta trovato il numero ottimale di componenti principali che sintetizzano l'informazione della matrice iniziale dei dati,
# la matrice dei loadings della PCA può essere utilizzata per definire un modello da sottoporre ad analisi confermativa.
# Consideriamo la matrice dei loadings ruotata
colnames(PC_th) = c("PC1","PC2"); rownames(PC_th) = colnames(X);
print(PC_th)
# La matrice finale PC_th contiene loadings/coefficienti vicini allo zero (le variabili ad essi collegate contribuiscono per niente alla definizione delle
# componenti principali) e potremmo decidere di toglierle. Ad esempio, utilizziamo il criterio standard secondo cui loadings minori di 0.35 in valore assoluto 
# possono essere trascurati. In questo caso:
PC_th[abs(PC_th)<0.35]=NA
print(PC_th)
# Il modello confermativo sarà definito da due fattori (PC1,PC2) legati alle variabili i cui coeffs sono diversi da NA.
# In maniera automatizzata, possiamo utilizzare la funzione prcomp2lavaan(..) presente nel file utilities.R come segue:

out_pca = prcomp(x = X)                                         #modo veloce per fare una PCA
x11(); par(mfrow=c(1,2)); plot(out_pca); biplot(out_pca)
mod_pca = prcomp2lavaan(prcomp_output = out_pca,numPC = 2,thr = 0.35,rotate = "varimax")   
print(mod_pca)
# La variabile 'mod_pca' può così essere utilizzata nella funzione cfa(..) di lavaan.


