# Compute final PFS vector:----
final.PFS <- function(res.red, all.0s, all.1s, N)
{
  if (length(c(all.0s, all.1s)) > 0)
  {
    res <- rep(NA,N)
    res[(1:N)[-c(all.0s, all.1s)]] <- res.red
  } else
  {
    res <- res.red
  }
  res
}

# Process the PFS scores from vector to data frame:----
res.process <- function(matrix, N, res)
{
  res <- data.frame(PFscores=round(res,4))
  if (is.null(row.names(matrix)))
  {
    row.names(res) <- rep(paste0("Resp.", 1:N))
  } else
  {
    row.names(res) <- row.names(matrix)
  }
  res
}

# Export results (nonparametric):----
export.res.NP <- function(matrix, N, res, PFStatistic, part.res, Ncat, NAs, 
                          IRT.PModel, IP, Ability.PModel, Ability, IP.NA, Ability.NA, NAs.imp)
{
  res <- list(PFscores  = res.process(matrix, N, res), PFStatistic = PFStatistic, 
              PerfVects = part.res$perfect.vectors, ID.all0s = part.res$all.0s, ID.all1s = part.res$all.1s, 
              Matrix = matrix, Ncat=Ncat, 
              IRT.PModel = if(IP.NA & (NAs == "PModel")) {IRT.PModel} else {NULL}, 
              IP = IP, 
              Ability.PModel = if(Ability.NA & (NAs == "PModel")) {Ability.PModel} else {NULL}, 
              Ability = Ability, 
              NAs.method = if(NAs.imp) {NAs} else {NULL})
  class(res) <- "PerFit"
  res
}

# Export results (parametric):----
export.res.P <- function(matrix, N, res, PFStatistic, part.res, Ncat, NAs, 
                         IRT.PModel, IP, Ability.PModel, Ability, IP.NA, Ability.NA, NAs.imp)
{
  res <- list(PFscores  = res.process(matrix, N, res), PFStatistic = PFStatistic, 
              PerfVects = part.res$perfect.vectors, ID.all0s = part.res$all0s, ID.all1s = part.res$all.1s, 
              Matrix = matrix, Ncat=Ncat, 
              IRT.PModel = if(IP.NA) {IRT.PModel} else {NULL}, 
              IP = IP, 
              Ability.PModel = if(Ability.NA) {Ability.PModel} else {NULL}, 
              Ability = Ability, 
              NAs.method = if(NAs.imp) {NAs} else {NULL})
  class(res) <- "PerFit"
  res
}


# Estimate item parameters if not provided (using 'irtoys'):----
estIP <- function(matrix, ip, model)
{
  I <- dim(matrix)[2]
  if (is.null(ip)) 
  {
    # Sanity check - IP model:
    Sanity.IPm(model)
    ip <- est(matrix, model, engine="ltm", rasch=TRUE, nqp=20)$est
  } else 
  {
    ip <- as.matrix(ip)
    # Sanity check - IP matrix adequacy:
    Sanity.IPa(ip, I)
  }
  ip
}

# Estimate item parameters if not provided (polytomous):----
estIP.poly <- function(matrix, Ncat, ip, model)
{
  I       <- dim(matrix)[2]
  matrix2 <- data.frame(apply(matrix, 2, as.factor)) # eliminates item levels with no answers
  if (is.null(ip)) 
  {
    # Sanity check - IP model (polytomous:
    Sanity.IPm.poly(model)
    ip <- switch(model,
                 PCM  =gpcm(matrix2, constraint="rasch", IRT.param=TRUE),
                 GPCM =gpcm(matrix2, constraint="gpcm" , IRT.param=TRUE),
                 GRM  =grm (matrix2, constrained=FALSE , IRT.param=TRUE))
    ip.coef <- coef(ip)
  } else 
  {
    ip <- as.matrix(ip)
    # Sanity check - IP matrix adequacy (polytomous):
    Sanity.IPa.poly(ip, I, Ncat)
    ip.coef <- ip
  }
  # In case NOT all answer categories of all items were used:
  if (is.list(ip.coef)) 
  {
    abs.freqs <- apply(matrix,2,table)
    abs.freqs <- lapply(abs.freqs,function(vect) as.numeric(names(vect)))
    tmp       <- matrix(NA,nrow=I,ncol=Ncat)
    for (i in 1:I) 
    {
      tmp[i,abs.freqs[[i]][-length(abs.freqs[[i]])]+1] <- ip.coef[[i]][-length(abs.freqs[[i]])]
      tmp[i,Ncat] <- ip.coef[[i]][length(ip.coef[[i]])]
    }
    ip.coef <- tmp
  }
  # 
  list(ip.coef, ip)
}

# Estimate ability parameters if not provided (using 'ltm'):----
estAb <- function(matrix, ip, ability, method, mu, sigma)
{
  N <- dim(matrix)[1]
  if (is.null(ability))
  {
    # Sanity check - Ability method:
    Sanity.Abm(method)
    ability <- switch(method,
                      ML = mlebme(matrix, ip, mu, sigma, method="ML")[,1],
                      BM = mlebme(matrix, ip, mu, sigma, method="BM")[,1],
                      WL = wle(matrix, ip)[,1])
  } else
  {
    ability <- as.vector(ability)
    # Sanity check - Ability matrix adequacy:
    Sanity.ABa(ability, N)
  }
  ability
}

# Estimate ability parameters if not provided (using 'ltm') (polytomous):----
estAb.poly <- function(matrix, ip.ltm, ability, method)
{
  N       <- dim(matrix)[1]
  matrix2 <- data.frame(apply(matrix,2,as.factor)) # eliminates item levels with no answers
  if (is.null(ability)) 
  {
    # Sanity check - Ability method:
    Sanity.Abm.poly(method)
    ability <- ltm::factor.scores(ip.ltm, resp.patterns=matrix2, method=method)
    ability <- ability$score.dat[, ncol(ability$score.dat)-1]
  } else
  {
    ability <- as.vector(ability)
    # Sanity check - Ability matrix adequacy:
    Sanity.ABa(ability, N)
  }
  ability
}

# Compute P.CRF (polytomous):----
estP.CRF <- function(I, Ncat, model, ip.coef, ability)
{
  N <- length(ability)
  M <- Ncat - 1
  #  Based on GRM:
  if (model == "GRM") 
  {
    # P.ISRF is N x I*M:
    P.ISRF <- t(
      sapply(ability, function(x)
      {
        as.vector(t(1/(1+exp(-ip.coef[,ncol(ip.coef)]*(x-ip.coef[,-ncol(ip.coef)])))))
      })
    )
    # Fix for datasets with non-chosen answer options (the NA entries):
    #   1s for NAs in the first item steps
    #   0s for NAs in the last item steps
    #   entry (x+1) for NAs in entry x
    if (sum(is.na(ip.coef)) > 0) 
    {
      first.cols  <- (which(is.na(ip.coef[,1]))-1)*M+1; P.ISRF[,first.cols][is.na(P.ISRF[,first.cols])] <- 1
      last.cols   <- which(is.na(ip.coef[,M]))*M; P.ISRF[,last.cols][is.na(P.ISRF[,last.cols])] <- 0
      middle.cols <- sort(which(is.na(t(cbind(rep(0,I),ip.coef[,-c(1,M,Ncat)],rep(0,I))))), decreasing=TRUE)
      for (i in 1:length(middle.cols)){P.ISRF[,middle.cols] <- P.ISRF[,middle.cols+1]}
    }
    P.CRF <- matrix(, nrow=N, ncol=I*Ncat)
    for (i in 1:I) 
    {
      P.ISRF.item <- -cbind(rep(1,N), P.ISRF[,((i-1)*M+1):(i*M)], rep(0,N))
      P.CRF[,((i-1)*Ncat+1):(i*Ncat)] <- P.ISRF.item[,(2:(M+2))] - P.ISRF.item[,(1:(M+1))]
    }
  }
  
  # Based on PCM or GPCM:
  if (model == "PCM" | model == "GPCM") 
  {
    lin.it <- t(sapply(ability,function(x){as.vector(t(ip.coef[,ncol(ip.coef)]*(x-ip.coef[,-ncol(ip.coef)])))}))
    lin.it[,is.na(as.vector(t(ip.coef[,-ncol(ip.coef)])))] <- 0 # NAs -> 0 to eliminate these terms from the sums
    P.CRF  <- matrix(, nrow=N, ncol=I*Ncat)
    tri    <- matrix(1, nrow=M+1, ncol=M+1) * upper.tri(tri, diag=TRUE)
    for (i in 1:I) 
    {
      num <- exp(cbind(rep(0,N), lin.it[,((i-1)*M+1):(i*M)]) %*% tri)
      P.CRF[, ((i-1)*Ncat+1):(i*Ncat)] <- num / rowSums(num)
    }  
  }
  
  P.CRF
}
########################################################################################
########################################################################################
# A, D, E (Kane & Brennan, 1980)
########################################################################################
########################################################################################
A.KB <- function(matrix, 
                 NA.method="NPModel", Save.MatImp=FALSE, 
                 IP=NULL, IRT.PModel="2PL", Ability=NULL, Ability.PModel="ML", mu=0, sigma=1)
{
  matrix      <- as.matrix(matrix)
  N           <- dim(matrix)[1]; I <- dim(matrix)[2]
  IP.NA       <- is.null(IP); Ability.NA  <- is.null(Ability)
  # Sanity check - Data matrix adequacy:
  Sanity.dma(matrix, N, I)
  # Dealing with missing values:
  res.NA <- MissingValues(matrix, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma)
  matrix <- res.NA[[1]]
  # Sanity check - Perfect response vectors:
  part.res  <- Sanity.prv(matrix, N, I)
  NC        <- part.res$NC
  all.0s    <- part.res$all.0s
  all.1s    <- part.res$all.1s
  matrix.sv <- matrix
  matrix    <- part.res$matrix.red
  # Compute PFS:
  pi        <- colMeans(matrix)
  res.red   <- as.vector(matrix %*% pi)
  # Compute final PFS vector:
  res <- final.PFS(res.red, all.0s, all.1s, N)
  # Export results:
  export.res.NP(matrix.sv, N, res, "A.KB", part.res, Ncat=2, NA.method, 
                IRT.PModel, res.NA[[2]], Ability.PModel, res.NA[[3]], IP.NA, Ability.NA, res.NA[[4]])
}

D.KB <- function(matrix, 
                 NA.method="NPModel", Save.MatImp=FALSE, 
                 IP=NULL, IRT.PModel="2PL", Ability=NULL, Ability.PModel="ML", mu=0, sigma=1)
{
  matrix      <- as.matrix(matrix)
  N           <- dim(matrix)[1]; I <- dim(matrix)[2]
  IP.NA       <- is.null(IP); Ability.NA  <- is.null(Ability)
  # Sanity check - Data matrix adequacy:
  Sanity.dma(matrix, N, I)
  # Dealing with missing values:
  res.NA <- MissingValues(matrix, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma)
  matrix <- res.NA[[1]]
  # Sanity check - Perfect response vectors:
  part.res  <- Sanity.prv(matrix, N, I)
  NC        <- part.res$NC
  all.0s    <- part.res$all.0s
  all.1s    <- part.res$all.1s
  matrix.sv <- matrix
  matrix    <- part.res$matrix.red
  # Compute PFS:
  pi      <- colMeans(matrix)
  pi.ord  <- sort(pi, decreasing=TRUE)
  a       <- matrix %*% pi
  a.max   <- cumsum(pi.ord)[NC]
  res.red <- as.vector(a.max - a)
  # Compute final PFS vector:
  res <- final.PFS(res.red, all.0s, all.1s, N)
  # Export results:
  export.res.NP(matrix.sv, N, res, "D.KB", part.res, Ncat=2, NA.method, 
                IRT.PModel, res.NA[[2]], Ability.PModel, res.NA[[3]], IP.NA, Ability.NA, res.NA[[4]])
}

E.KB <- function(matrix, 
                 NA.method="NPModel", Save.MatImp=FALSE, 
                 IP=NULL, IRT.PModel="2PL", Ability=NULL, Ability.PModel="ML", mu=0, sigma=1)
{
  matrix      <- as.matrix(matrix)
  N           <- dim(matrix)[1]; I <- dim(matrix)[2]
  IP.NA       <- is.null(IP); Ability.NA  <- is.null(Ability)
  # Sanity check - Data matrix adequacy:
  Sanity.dma(matrix, N, I)
  # Dealing with missing values:
  res.NA <- MissingValues(matrix, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma)
  matrix <- res.NA[[1]]
  # Sanity check - Perfect response vectors:
  part.res  <- Sanity.prv(matrix, N, I)
  NC        <- part.res$NC
  all.0s    <- part.res$all.0s
  all.1s    <- part.res$all.1s
  matrix.sv <- matrix
  matrix    <- part.res$matrix.red
  # Compute PFS:
  pi      <- colMeans(matrix)
  pi.ord  <- sort(pi,decreasing=TRUE)
  a       <- matrix %*% pi
  a.max   <- cumsum(pi.ord)[NC]
  res.red <- as.vector(a / a.max)
  # Compute final PFS vector:
  res <- final.PFS(res.red, all.0s, all.1s, N)
  # Export results:
  export.res.NP(matrix.sv, N, res, "E.KB", part.res, Ncat=2, NA.method, 
                IRT.PModel, res.NA[[2]], Ability.PModel, res.NA[[3]], IP.NA, Ability.NA, res.NA[[4]])
}
########################################################################################
########################################################################################
# C (Sato, 1975):
########################################################################################
########################################################################################
C.Sato <- function(matrix, 
                   NA.method="NPModel", Save.MatImp=FALSE, 
                   IP=NULL, IRT.PModel="2PL", Ability=NULL, Ability.PModel="ML", mu=0, sigma=1)
{
  matrix      <- as.matrix(matrix)
  N           <- dim(matrix)[1]; I <- dim(matrix)[2]
  IP.NA       <- is.null(IP); Ability.NA  <- is.null(Ability)
  # Sanity check - Data matrix adequacy:
  Sanity.dma(matrix, N, I)
  # Dealing with missing values:
  res.NA <- MissingValues(matrix, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma)
  matrix <- res.NA[[1]]
  # Sanity check - Perfect response vectors:
  part.res  <- Sanity.prv(matrix, N, I)
  NC        <- part.res$NC
  all.0s    <- part.res$all.0s
  all.1s    <- part.res$all.1s
  matrix.sv <- matrix
  matrix    <- part.res$matrix.red
  # Compute PFS:
  pi             <- colMeans(matrix.sv)
  pi.ord         <- sort(pi, decreasing=TRUE)
  matrix.ord     <- matrix[, order(pi, decreasing=TRUE)]
  num            <- cov(t(matrix.ord), pi.ord)
  matrix.easiest <- (col(matrix) <= NC) * 1
  den            <- cov(t(matrix.easiest), pi.ord)
  res.red        <- as.vector(1 - num / den)
  # Compute final PFS vector:
  res <- final.PFS(res.red, all.0s, all.1s, N)
  # Export results:
  export.res.NP(matrix.sv, N, res, "C.Sato", part.res, Ncat=2, NA.method, 
                IRT.PModel, res.NA[[2]], Ability.PModel, res.NA[[3]], IP.NA, Ability.NA, res.NA[[4]])
}
########################################################################################
########################################################################################
# C* (Harnisch & Linn, 1981):
########################################################################################
########################################################################################
Cstar <- function(matrix, 
                  NA.method="NPModel", Save.MatImp=FALSE, 
                  IP=NULL, IRT.PModel="2PL", Ability=NULL, Ability.PModel="ML", mu=0, sigma=1)
{
  matrix      <- as.matrix(matrix)
  N           <- dim(matrix)[1]; I <- dim(matrix)[2]
  IP.NA       <- is.null(IP); Ability.NA  <- is.null(Ability)
  # Sanity check - Data matrix adequacy:
  Sanity.dma(matrix, N, I)
  # Dealing with missing values:
  res.NA <- MissingValues(matrix, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma)
  matrix <- res.NA[[1]]
  # Sanity check - Perfect response vectors:
  part.res  <- Sanity.prv(matrix, N, I)
  NC        <- part.res$NC
  all.0s    <- part.res$all.0s
  all.1s    <- part.res$all.1s
  matrix.sv <- matrix
  matrix    <- part.res$matrix.red
  # Compute PFS:
  pi          <- colMeans(matrix.sv)
  pi.ord      <- sort(pi, decreasing=TRUE)
  sum.firstpi <- cumsum(pi.ord)[NC]
  pi.ordrev   <- sort(pi, decreasing=FALSE)
  sum.lastpi  <- cumsum(pi.ordrev)[NC]
  res.red     <- as.vector((sum.firstpi - as.vector(matrix %*% pi)) / (sum.firstpi - sum.lastpi))
  # Compute final PFS vector:
  res <- final.PFS(res.red, all.0s, all.1s, N)
  # Export results:
  export.res.NP(matrix.sv, N, res, "Cstar", part.res, Ncat=2, NA.method, 
                IRT.PModel, res.NA[[2]], Ability.PModel, res.NA[[3]], IP.NA, Ability.NA, res.NA[[4]])
}
# Determine the cutoff for a PFS based on model-fitting item response vectors:
cutoff <- function(x, #x = an object from 'PerFit' class
                   ModelFit="NonParametric", Nreps=1000, 
                   IP=x$IP, IRT.PModel=x$IRT.PModel, Ability=x$Ability, Ability.PModel=x$Ability.PModel, mu=0, sigma=1, 
                   Blvl = 0.05, Breps = 1000, CIlvl = 0.95, 
                   UDlvl = NA)
{
  # Sanity check - Class PerFit:
  Sanity.cls(x)  
  # 
  N        <- dim(x$Matrix)[1]; I <- dim(x$Matrix)[2]; Ncat <- x$Ncat
  upp.PFS  <- c("Cstar", "C.Sato", "U3", "ZU3", "G", "Gnormed", "Gpoly", "Gnormed.poly", "U3poly", "D.KB")
  low.PFS  <- c("r.pbis", "NCI", "Ht", "A.KB", "E.KB", "lz", "lzstar", "lzpoly")
  dico.PFS <- c("Cstar", "C.Sato", "U3", "ZU3", "G", "Gnormed", "D.KB", "r.pbis", "NCI", "Ht", "A.KB", "E.KB", "lz", "lzstar")
  poly.PFS <- c("Gpoly", "Gnormed.poly", "U3poly", "lzpoly")
  perfectAllowed.PFS <- c("G", "Gnormed", "Gpoly", "Gnormed.poly", "U3poly", "NCI", "lz", "lzstar", "lzpoly")
  perfectNotAllowed.PFS <- c("Cstar", "C.Sato", "U3", "ZU3", "A.KB", "D.KB", "E.KB", "r.pbis", "Ht")
  # 
  pf.scores       <- x$PFscores[[1]]
  PFS.NA          <- is.na(pf.scores)
  pf.scores.noNAs <- pf.scores[!PFS.NA]
  
  if (ModelFit == "Parametric" & any(x$PFStatistic == dico.PFS))
  {
    # Generate model-fitting item score vectors based on a parametric model (1PL, 2PL, or 3PL).
    # 
    # Estimate item parameters if not provided:
    if (is.null(IRT.PModel)) 
    {
      IRT.PModel <- "2PL"
    } else
    {
      Sanity.IPm(IRT.PModel)
    }
    IP <- estIP(x$Matrix, IP, IRT.PModel)
    # Estimate ability parameters if not provided:
    if (is.null(Ability.PModel)) 
    {
      Ability.PModel <- "ML"
    } else
    {
      Sanity.Abm(Ability.PModel)
    }
    Ability <- estAb(x$Matrix, IP, Ability, Ability.PModel, mu, sigma)
    #
    Ability.gen <- sample(Ability, size=Nreps, replace=TRUE)
    #
    A   <- IP[,1]; B <- IP[,2]; C <- IP[,3]
    P                   <- do.call(cbind, lapply(1:I,function (x) {C[x]+(1-C[x]) / (1+exp(-A[x]*(Ability.gen - B[x])))}))
    matrix.modelfitting <- matrix(rbinom(length(P), 1, P), ncol=I)
    # Compute PFS values on the model-fitting item response vectors:
    if (any(x$PFStatistic == perfectNotAllowed.PFS))
    {
      nonperfectvects.mf <- (rowSums(matrix.modelfitting) %% I) != 0
      PFS.modelfitting <- eval(parse(text = x[[2]]))(matrix.modelfitting[nonperfectvects.mf,])$PFscores[[1]]
    } else
    {
      PFS.modelfitting <- eval(parse(text = x[[2]]))(matrix.modelfitting)$PFscores[[1]]
    }
  }
  
  if (ModelFit == "Parametric" & any(x$PFStatistic == poly.PFS))
  {
    # Generate model-fitting item score vectors based on a parametric model (PCM, GPCM, or GRM).
    # 
    # Estimate item parameters if not provided:
    if (is.null(IRT.PModel)) 
    {
      IRT.PModel <- "GRM"
    } else
    {
      Sanity.IPm.poly(IRT.PModel)
    }
    IP.res <- estIP.poly(x$Matrix, Ncat, IP, IRT.PModel)
    IP     <- IP.res[[1]]
    IP.ltm <- IP.res[[2]]
    # Estimate ability parameters if not provided:
    if (is.null(Ability.PModel)) 
    {
      Ability.PModel <- "EAP"
    } else
    {
      Sanity.Abm.poly(Ability.PModel)
    }
    Ability <- estAb.poly(x$Matrix, IP.ltm, Ability, Ability.PModel)
    #
    Ability.gen <- sample(Ability, size=Nreps, replace=TRUE)
    #
    P.CRF               <- estP.CRF(I, Ncat, IRT.PModel, IP, Ability.gen)
    P.CRF.ind           <- matrix(1:(Ncat*I), nrow=Ncat)
    matrix.modelfitting <- matrix(,nrow=Nreps, ncol=I)
    for (i in 1:I)
    {
      matrix.modelfitting[,i] <- rMultinom(P.CRF[,((i-1)*Ncat + 1) : (i*Ncat)],1) - 1
    }
    # Compute PFS values on the model-fitting item response vectors:
    if (any(x$PFStatistic == perfectNotAllowed.PFS))
    {
      nonperfectvects.mf <- (rowSums(matrix.modelfitting) %% I) != 0
      PFS.modelfitting <- eval(parse(text = x[[2]]))(matrix.modelfitting[nonperfectvects.mf,], Ncat)$PFscores[[1]]
    } else
    {
      PFS.modelfitting <- eval(parse(text = x[[2]]))(matrix.modelfitting, Ncat)$PFscores[[1]]
    }
  }
  
  # 
  if (ModelFit == "NonParametric" & any(x$PFStatistic == dico.PFS))
  {
    NC <- rowSums(x$Matrix)
    # 
    NC.gen <- sample(NC[!PFS.NA], size=Nreps, replace=TRUE)
    #
    uniqueNC            <- sort(unique(NC.gen))
    matrix.modelfitting <- matrix(, nrow=Nreps, ncol=I)
    # 
    for (i in 1:length(uniqueNC))
    {
      NC.i <- which(NC.gen == uniqueNC[i])
      pi.i <- colMeans(x$Matrix[NC == uniqueNC[i],,drop=FALSE])
      matrix.modelfitting[NC.i, ] <- rbinom(length(NC.i)*I, 1, rep(pi.i, each=length(NC.i)))
    }
    # Compute PFS values on the model-fitting item response vectors:
    if (any(x$PFStatistic == perfectNotAllowed.PFS))
    {
      nonperfectvects.mf <- (rowSums(matrix.modelfitting) %% I) != 0
      PFS.modelfitting <- eval(parse(text = x[[2]]))(matrix.modelfitting[nonperfectvects.mf,])$PFscores[[1]]
    } else
    {
      PFS.modelfitting <- eval(parse(text = x[[2]]))(matrix.modelfitting)$PFscores[[1]]
    }
  }
  
  # 
  if (ModelFit == "NonParametric" & any(x$PFStatistic == poly.PFS))
  {
    NC <- rowSums(x$Matrix)
    # 
    NC.gen <- sample(NC[!PFS.NA], size=Nreps, replace=TRUE)
    #
    uniqueNC            <- sort(unique(NC.gen))
    matrix.modelfitting <- matrix(, nrow=Nreps, ncol=I)
    # 
    for (i in 1:length(uniqueNC))
    {
      NC.i       <- which(NC.gen == uniqueNC[i])
      freq.abs.i <- apply(x$Matrix[NC == uniqueNC[i], , drop=FALSE], 2,
                          function(vec) {table(factor(vec, levels=0:(Ncat - 1)))})
      matrix.modelfitting[NC.i, ] <- 
        apply(freq.abs.i, 2, function(vect) 
        {
          which(rmultinom(length(NC.i),1,vect) == 1, arr.ind=TRUE)[,1] - 1
        })
    }
    # Compute PFS values on the model-fitting item response vectors:
    if (any(x$PFStatistic == perfectNotAllowed.PFS))
    {
      nonperfectvects.mf <- (rowSums(matrix.modelfitting) %% I) != 0
      PFS.modelfitting <- eval(parse(text = x[[2]]))(matrix.modelfitting[nonperfectvects.mf,], Ncat)$PFscores[[1]]
    } else
    {
      PFS.modelfitting <- eval(parse(text = x[[2]]))(matrix.modelfitting, Ncat)$PFscores[[1]]
    }
  }
  
  # Compute cutoff: 
  if (is.na(UDlvl)) # By means of bootstrap
  {
    if (any(x$PFStatistic == upp.PFS))
    {
      tail <- "upper"
      Blvl.use <- 1-Blvl
    }
    if (any(x$PFStatistic == low.PFS))
    {
      tail <- "lower"
      Blvl.use <- Blvl
    }   
    Bvec <- c()
    for (i in 1:Breps)
    {
      Bvec <- c(Bvec, quantile(sample(PFS.modelfitting, size=length(PFS.modelfitting), replace=TRUE), probs=Blvl.use))
    }
    cutoff.use <- round(median(Bvec), 4)
    cutoff.SE  <- round(sd(Bvec), 4)
    cutoff.CI  <- round(quantile(Bvec, probs=c((1-CIlvl)/2, (1+CIlvl)/2)), 4)
  } else
  { 
    if (any(x$PFStatistic == upp.PFS)) {tail <- "upper"}
    if (any(x$PFStatistic == low.PFS)) {tail <- "lower"}
    cutoff.use <- UDlvl
    cutoff.SE  <- NA
    cutoff.CI  <- NA
  }
  
  # Determine the proportion of flagged subjects:
  if (any(x$PFStatistic == upp.PFS))
  {
    prop.flagged <- round(sum(pf.scores.noNAs >= cutoff.use) / N, 4)
  }
  if (any(x$PFStatistic == low.PFS))
  {
    prop.flagged <- round(sum(pf.scores.noNAs <= cutoff.use) / N, 4)
  }
  #
  res        <- list(Cutoff=as.numeric(cutoff.use), Cutoff.SE=cutoff.SE, Prop.flagged=prop.flagged, Tail=tail, 
                     Cutoff.CI=cutoff.CI)
  class(res) <- "PerFit.cutoff"
  res
}
flagged.resp <- function(x, #x = an object from 'PerFit' class
                         cutoff.obj=NULL, #cutoff.obj = an object from 'PerFit.cutoff' class
                         scores=TRUE, ord=TRUE,
                         ModelFit="NonParametric", Nreps=1000, 
                         IP=x$IP, IRT.PModel=x$IRT.PModel, Ability=x$Ability, Ability.PModel=x$Ability.PModel, mu=0, sigma=1, 
                         Blvl = 0.05, Breps = 1000, CIlvl = 0.95, 
                         UDlvl=NA)
{
  # Sanity check - Class PerFit:
  Sanity.cls(x)  
  # 
  upp.PFS  <- c("Cstar", "C.Sato", "U3", "ZU3", "G", "Gnormed", "Gpoly", "Gnormed.poly", "U3poly", "D.KB")
  low.PFS  <- c("r.pbis", "NCI", "Ht", "A.KB", "E.KB", "lz", "lzstar", "lzpoly")
  # 
  if (is.null(cutoff.obj))
  {
    cutoff.res <- cutoff(x, ModelFit, Nreps, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma, Blvl, Breps, CIlvl, UDlvl)
  } else
  {
    Sanity.clsPO(cutoff.obj)
    cutoff.res <- cutoff.obj
  }
  #   
  if (any(x$PFStatistic == upp.PFS)) 
  {
    flagged.subs <- which(x$PFscores[,1] >= cutoff.res$Cutoff)
  }
  if (any(x$PFStatistic == low.PFS)) 
  {
    flagged.subs <- which(x$PFscores[,1] <= cutoff.res$Cutoff)
  }
  Ps <- round(colMeans(x$Matrix),3)
  # Not ordered by pvalue:
  if (ord == FALSE)
  {
    flagged.scores           <- x$Matrix[flagged.subs, ]
    colnames(flagged.scores) <- paste("It", 1:dim(x$Matrix)[2], sep="")
  }
  # Ordered by pvalue:
  if (ord == TRUE)
  {
    matrix.ord               <- x$Matrix[, order(Ps,decreasing=TRUE)] # ordered from easy to difficult
    flagged.scores           <- matrix.ord[flagged.subs,]
    colnames(flagged.scores) <- paste("It", order(Ps,decreasing=TRUE), sep="")
    Ps                       <- sort(Ps,decreasing=TRUE)
  }
  flagged.scores           <- as.matrix(flagged.scores)
  rownames(flagged.scores) <- NULL
  res                      <- if (scores == FALSE)
  {
    list(PFSscores=cbind(FlaggedID=flagged.subs, PFscores=x$PFscores[flagged.subs,1]), Cutoff.lst=cutoff.res, PFS=x[[2]])
  } else
  {
    list(Scores=cbind(FlaggedID=flagged.subs, flagged.scores, PFscores=x$PFscores[flagged.subs,1]), MeanItemValue=Ps, Cutoff.lst=cutoff.res, 
         PFS=x[[2]])
  }
  res
}########################################################################################
########################################################################################
# Polytomous items: Number of Guttman errors
# Gpoly reduces to G for 0-1 data (Ncat=2) (currently Gpoly needs G to be loaded)
# Gnormed.poly reduces to Gnormed for 0-1 data (Ncat=2) (currently Gnormed.poly needs G to be loaded)
########################################################################################
########################################################################################

Gnormed.poly <- function(matrix, Ncat,
                          NA.method="NPModel", Save.MatImp=FALSE, 
                          IP=NULL, IRT.PModel="GRM", Ability=NULL, Ability.PModel="EAP")
{
  matrix      <- as.matrix(matrix)
  N           <- dim(matrix)[1]; I <- dim(matrix)[2]; M <- Ncat-1
  IP.NA       <- is.null(IP); Ability.NA  <- is.null(Ability)
  # Sanity check - Data matrix adequacy:
  Sanity.dma.poly(matrix, N, I, M)
  # Dealing with missing values:
  res.NA <- MissingValues.poly(matrix, Ncat, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel)
  matrix <- res.NA[[1]]
  # Perfect response vectors allowed (albeit uninformative).
  # Compute PFS:
  # Numerator:
  probs.ISD      <- matrix(NA, nrow=I, ncol=M)
  for (m in 1:M) {probs.ISD[,m] <- colMeans(matrix >= m)}
  f.scoresISD    <- function (x) {c(rep(1,x), rep(0,M-x))}
  matrix.ISD     <- matrix(unlist(lapply(t(matrix), f.scoresISD)), byrow=TRUE, nrow=N)
  probs.ISD.vect <- as.vector(t(probs.ISD))
  matrix.ISD.ord <- matrix.ISD[, order(probs.ISD.vect, decreasing=TRUE)]
  num            <- G(matrix.ISD.ord)$PFscores[,1]
  # Denominator: 
  NC        <- rowSums(matrix)
  ranks.ISD <- matrix(rank(I*M - probs.ISD.vect, ties.method="first"), nrow=I, byrow=TRUE)
  if (Ncat>2) {cumranks.ISD <- cbind(rep(0,I), t(apply(ranks.ISD, 1, cumsum)))}
  if (Ncat==2) {cumranks.ISD <- cbind(rep(0,I), ranks.ISD)}
  V        <- matrix(rep(cumranks.ISD[1,], Ncat), ncol=Ncat, byrow=FALSE)
  add.term <- matrix(rep(cumranks.ISD[2,], nrow(V)), ncol=Ncat, byrow=TRUE)
  V        <- V + add.term
  T        <- sapply(2:sum(dim(V)), function(x) {max(V[col(V)+row(V) == x])})
  for (i in 3:I) 
  {
    V        <- matrix(rep(T, Ncat), ncol=Ncat, byrow=FALSE)
    add.term <- matrix(rep(cumranks.ISD[i,], nrow(V)), ncol=Ncat, byrow=TRUE)
    V        <- V + add.term
    T        <- sapply(2:sum(dim(V)), function(x) {max(V[col(V)+row(V) == x])})
  }
  maxG       <- T - sapply(0:(I*M), function(x) {.5*x*(x+1)})
  maxG[c(1,length(maxG))] <- 1 # so that vectors (0,0,...,0) and (M,M,...,M), which have 0 Guttman errors, are divided by 1 instead of 0
  # 
  res <- num / maxG[NC+1]
  # Export results:
  export.res.NP(matrix, N, res, "Gnormed.poly", vector("list", 5) , Ncat=Ncat, NA.method, 
               IRT.PModel, res.NA[[2]], Ability.PModel, res.NA[[3]], IP.NA, Ability.NA, res.NA[[4]])

}########################################################################################
########################################################################################
# Gnormed (van der Flier, 1977; Meijer, 1994):
# This statistic is perfectly linearly related to NCI (Tatsuoka & Tatsuoaka, 1982, 1983)
# NCI = 1-2Gnormed
########################################################################################
########################################################################################
Gnormed <- function(matrix, 
                    NA.method="NPModel", Save.MatImp=FALSE, 
                    IP=NULL, IRT.PModel="2PL", Ability=NULL, Ability.PModel="ML", mu=0, sigma=1)
{
  matrix      <- as.matrix(matrix)
  N           <- dim(matrix)[1]; I <- dim(matrix)[2]
  IP.NA       <- is.null(IP); Ability.NA  <- is.null(Ability)
  # Sanity check - Data matrix adequacy:
  Sanity.dma(matrix, N, I)
  # Dealing with missing values:
  res.NA <- MissingValues(matrix, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma)
  matrix <- res.NA[[1]]
  # Perfect response vectors allowed (albeit uninformative).
  # Compute PFS:
  NC         <- rowSums(matrix)
  uniqueNC   <- sort(unique(NC))
  pi         <- colMeans(matrix)
  matrix.ord <- matrix[,order(pi,decreasing=TRUE)]
  per.row <- function(vect)
  {
    ind.0     <- which(vect == 0)
    ind.1     <- which(vect == 1)
    all.cases <- expand.grid(ind.0,ind.1)
    sum((all.cases[,2] - all.cases[,1]) > 0)
  }
  res <- apply(matrix.ord,1,per.row)
  res <- res / (NC*(I-NC))
  res[is.nan(res)] <- 0 # all-0s or all-1s vector -> Gnormed = 0
  # Export results:
  export.res.NP(matrix, N, res, "Gnormed", vector("list", 5), Ncat=2, NA.method, 
                IRT.PModel, res.NA[[2]], Ability.PModel, res.NA[[3]], IP.NA, Ability.NA, res.NA[[4]])
}
########################################################################################
########################################################################################
# Polytomous items: Number of Guttman errors
# Gpoly reduces to G for 0-1 data (Ncat=2) (currently Gpoly needs G to be loaded)
# Gnormed.poly reduces to Gnormed for 0-1 data (Ncat=2) (currently Gnormed.poly needs G to be loaded)
########################################################################################
########################################################################################

Gpoly <- function(matrix, Ncat,
                  NA.method="NPModel", Save.MatImp=FALSE, 
                  IP=NULL, IRT.PModel="GRM", Ability=NULL, Ability.PModel="EAP")
{
  matrix      <- as.matrix(matrix)
  N           <- dim(matrix)[1]; I <- dim(matrix)[2]; M <- Ncat-1
  IP.NA       <- is.null(IP); Ability.NA  <- is.null(Ability)
  # Sanity check - Data matrix adequacy:
  Sanity.dma.poly(matrix, N, I, M)
  # Dealing with missing values:
  res.NA <- MissingValues.poly(matrix, Ncat, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel)
  matrix <- res.NA[[1]]
  # Perfect response vectors allowed (albeit uninformative).
  # Compute PFS:
  probs.ISD       <- matrix(NA, nrow=I, ncol=M)
  for (m in 1:M) {probs.ISD[,m] <- colMeans(matrix >= m)}
  f.scoresISD     <- function (x){c(rep(1,x), rep(0,M-x))}
  matrix.ISD      <- matrix(unlist(lapply(t(matrix), f.scoresISD)), byrow=TRUE, nrow=N)
  probs.ISD.vect  <- as.vector(t(probs.ISD))
  matrix.ISD.ord  <- matrix.ISD[,order(probs.ISD.vect, decreasing=TRUE)]
  res             <- G(matrix.ISD.ord)$PFscores[,1]
  # Export results:
  export.res.NP(matrix, N, res, "Gpoly", vector("list", 5) , Ncat=Ncat, NA.method, 
               IRT.PModel, res.NA[[2]], Ability.PModel, res.NA[[3]], IP.NA, Ability.NA, res.NA[[4]])
}
########################################################################################
########################################################################################
# G (van der Flier, 1977; Meijer, 1994):
########################################################################################
########################################################################################
G <- function(matrix, 
              NA.method="NPModel", Save.MatImp=FALSE, 
              IP=NULL, IRT.PModel="2PL", Ability=NULL, Ability.PModel="ML", mu=0, sigma=1)
{
  matrix      <- as.matrix(matrix)
  N           <- dim(matrix)[1]; I <- dim(matrix)[2]
  IP.NA       <- is.null(IP); Ability.NA  <- is.null(Ability)
  # Sanity check - Data matrix adequacy:
  Sanity.dma(matrix, N, I)
  # Dealing with missing values:
  res.NA <- MissingValues(matrix, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma)
  matrix <- res.NA[[1]]
  # Perfect response vectors allowed (albeit uninformative).
  # Compute PFS:
  NC         <- rowSums(matrix)
  uniqueNC   <- sort(unique(NC))
  pi         <- colMeans(matrix)
  matrix.ord <- matrix[,order(pi,decreasing=TRUE)]
  per.row    <- function(vect)
  {
    ind.0     <- which(vect == 0)
    ind.1     <- which(vect == 1)
    all.cases <- expand.grid(ind.0,ind.1)
    sum((all.cases[,2] - all.cases[,1]) > 0)
  }
  res <- apply(matrix.ord,1,per.row)
  # Export results:
  export.res.NP(matrix, N, res, "G", vector("list", 5), Ncat=2, NA.method, 
                IRT.PModel, res.NA[[2]], Ability.PModel, res.NA[[3]], IP.NA, Ability.NA, res.NA[[4]])
}
# Hotdeck imputation (dichotomous, polytomous):
HD.imputation <- function(matrix, save.matImp, ip, ability)
{
  N <- dim(matrix)[1]; I <- dim(matrix)[2]
  matrix.imp <- matrix
  position.NA <- is.na(matrix)
  recipients <- which(rowSums(position.NA) > 0)
  N.recipients <- length(recipients)
  donors <- (1:N)[-recipients]
  N.donors <- length(donors)
  donors.hlp.mat <- t(matrix[donors,])
  # 
  for (i in 1:N.recipients)
  {
    rcp <- recipients[i]
    distance <- rowSums(abs(t(matrix[rcp,] - donors.hlp.mat)), na.rm=TRUE)
    min.distance <- which(distance == min(distance))
    closest.donor <- if (length(min.distance) == 1) {min.distance} else {sample(min.distance, size=1)}
    matrix.imp[rcp, position.NA[rcp,]] <- matrix[donors[closest.donor], position.NA[rcp,]]
  }
  # 
  if (save.matImp == TRUE)
  {
    write.matrix(matrix.imp, file="Datamatrix_imputted.txt", sep=" ")
  }
  list(matrix.imp, ip, ability, 1)
}########################################################################################
########################################################################################
# Ht (Sijtsma, 1986)
########################################################################################
########################################################################################
Ht <- function(matrix, 
               NA.method="NPModel", Save.MatImp=FALSE, 
               IP=NULL, IRT.PModel="2PL", Ability=NULL, Ability.PModel="ML", mu=0, sigma=1)
{
  matrix      <- as.matrix(matrix)
  N           <- dim(matrix)[1]; I <- dim(matrix)[2]
  IP.NA       <- is.null(IP); Ability.NA  <- is.null(Ability)
  # Sanity check - Data matrix adequacy:
  Sanity.dma(matrix, N, I)
  # Dealing with missing values:
  res.NA <- MissingValues(matrix, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma)
  matrix <- res.NA[[1]]
  # Sanity check - Perfect response vectors:
  part.res  <- Sanity.prv(matrix, N, I)
  NC        <- part.res$NC
  all.0s    <- part.res$all.0s
  all.1s    <- part.res$all.1s
  matrix.sv <- matrix
  matrix    <- part.res$matrix.red
  # Compute PFS: 
  singlePs  <- rowMeans(matrix)
  tot.score <- colSums(matrix.sv)
  N.red     <- dim(matrix)[1]
  num       <- apply(matrix,1,function(vect){cov(vect, tot.score - vect)}) * (I-1)/I
  df        <- data.frame(1:N.red, singlePs, num)
  df.ord    <- df[order(df[,2]),]
  singlePs.ord <- df.ord[[2]]
  pos       <- which(diff(singlePs.ord, lag=1)>0) 
  less      <- sapply(c(pos,N.red), function(x)
  {
    sum(singlePs.ord[1:(x-1)])
  })
  if (pos[1]==1){less[1] <- 0}
  less      <- rep(less, c(pos[1], diff(c(pos, N.red),lag=1))) * (1-singlePs.ord)
  more      <- sapply(pos,function(x){sum(1-singlePs.ord[(x+1):N.red])}); more <- c(more,0)
  more      <- rep(more, c(pos[1],diff(c(pos, N.red), lag=1))) * singlePs.ord
  den       <- less + more
  df.ord    <- data.frame(df.ord,den)
  df        <- df.ord[order(df.ord[, 1]),]
  res.red   <- df$num / df$den
  # Compute final PFS vector:
  res <- final.PFS(res.red, all.0s, all.1s, N)
  # Export results:
  export.res.NP(matrix.sv, N, res, "Ht", part.res, Ncat=2, NA.method, 
                IRT.PModel, res.NA[[2]], Ability.PModel, res.NA[[3]], IP.NA, Ability.NA, res.NA[[4]])
}########################################################################################
########################################################################################
# lzp
########################################################################################
########################################################################################
lzpoly <- function(matrix, Ncat,
                   NA.method="NPModel", Save.MatImp=FALSE, 
                   IP=NULL, IRT.PModel="GRM", Ability=NULL, Ability.PModel="EAP")
{
  if (!is.null(IP) & is.null(Ability))
  {
    stop('If "IP" is provided then "Ability" must also be provided (for estimation purposes).
         Aborted.')
  }
  #
  matrix <- as.matrix(matrix)
  N      <- dim(matrix)[1]; I <- dim(matrix)[2]; M <- Ncat-1
  IP.NA       <- is.null(IP); Ability.NA  <- is.null(Ability)
  # Sanity check - Data matrix adequacy:
  Sanity.dma.poly(matrix, N, I, M)
  # Estimate item parameters if not provided (polytomous):
  IP.res <- estIP.poly(matrix, Ncat, IP, IRT.PModel)
  IP     <- IP.res[[1]]
  IP.ltm <- IP.res[[2]] 
  # Estimate ability parameters if not provided (using 'ltm'):
  Ability <- estAb.poly(matrix, IP.ltm, Ability, Ability.PModel)
  # Dealing with missing values:
  res.NA <- MissingValues.poly(matrix, Ncat, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel)
  matrix <- res.NA[[1]]
  # Perfect response vectors allowed.
  # Compute PFS:
  P.CRF   <- estP.CRF(I, Ncat, IRT.PModel, IP, Ability)
  # 
  idty      <- diag(Ncat)
  f.scores  <- function (x) {idty[x+1,]}
  matrix.01 <- matrix(unlist(lapply(t(matrix), f.scores)), byrow=TRUE, nrow=N)
  # If there are answer options not chosen by any respondent then some entries in 'P.CRF' might be 0.
  # Below all corresponding logs are set from Inf to 0.
  # (Reason: They carry no information regarding aberrant response behavior).
  log.P.CRF <- log(P.CRF)
  log.P.CRF[is.infinite(log.P.CRF)] <- 0           
  #
  l0p  <- rowSums(matrix.01 * log.P.CRF)
  El0p <- rowSums(P.CRF * log.P.CRF)
  # Variance (two equivalent options up to time efficiency):
  if (I*Ncat < 300)
  {
    ones.block <- vector("list", I)
    ones.block[1:I] <- list(matrix(rep(1, Ncat^2), nrow=Ncat))
    to.sum <- as.matrix(do.call(bdiag, ones.block))
    V.row  <- function(vect) {
      log.vect <- log.P.CRF[vect[1],]
      vect2 <- vect[2:(dim(P.CRF)[2]+1)]
      tmp.part1 <- (vect2 %*% t(vect2)) * to.sum
      tmp.part2 <- matrix(rep(log.vect, dim(P.CRF)[2]), nrow=dim(P.CRF)[2]) * to.sum
      # 
      sum(tmp.part1 * tmp.part2 * (tmp.part2 - t(tmp.part2)))
    }
    Vl0p <- apply(cbind(1:N, P.CRF), 1, V.row)
  } else
  {
    V.row <- function(vect) {
      tot <- 0;
      for (i in 1:I) {
        log.vect  <- log.P.CRF[vect[1], ((i-1)*Ncat+1):(Ncat*i)]
        vect2     <- vect[((i-1)*Ncat+1):(Ncat*i)+1]
        tmp.part1 <- vect2 %*% t(vect2)
        tmp.part2 <- matrix(rep(log.vect, Ncat), nrow=Ncat)
        # 
        tot       <- tot + sum(tmp.part1 * tmp.part2 * (tmp.part2 - t(tmp.part2)));
      }
      tot
    }
    Vl0p <- apply(cbind(1:N,P.CRF),1,V.row)
  }
  res <- (l0p - El0p) / sqrt(Vl0p)
  # Export results:
  export.res.P(matrix, N, res, "lzpoly", vector("list", 5) , Ncat=Ncat, NA.method, 
               IRT.PModel, res.NA[[2]], Ability.PModel, res.NA[[3]], IP.NA, Ability.NA, res.NA[[4]])
}########################################################################################
########################################################################################
# lz
########################################################################################
########################################################################################
lz <- function(matrix, 
                NA.method="NPModel", Save.MatImp=FALSE, 
                IP=NULL, IRT.PModel="2PL", Ability=NULL, Ability.PModel="ML", mu=0, sigma=1)
{
  matrix      <- as.matrix(matrix)
  N           <- dim(matrix)[1]; I <- dim(matrix)[2]
  IP.NA       <- is.null(IP); Ability.NA  <- is.null(Ability)
  # Sanity check - Data matrix adequacy:
  Sanity.dma(matrix, N, I)
  # Estimate item parameters if not provided (using 'irtoys'):
  IP <- estIP(matrix, IP, IRT.PModel)
  # Estimate ability parameters if not provided (using 'irtoys'):
  Ability <- estAb(matrix, IP, Ability, Ability.PModel, mu, sigma)
  # Dealing with missing values:
  res.NA <- MissingValues(matrix, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma)
  matrix <- res.NA[[1]]
  # Perfect response vectors allowed.
  # Compute PFS: 
  A   <- IP[,1]; B <- IP[,2]; C <- IP[,3]
  P   <- do.call(cbind, lapply(1:I, function (x) {C[x]+(1-C[x]) / (1+exp(-A[x]*(Ability - B[x])))})); Q <- 1-P
  l0  <- rowSums(matrix*log(P) + (1-matrix)*log(Q))
  El0 <- rowSums(P*log(P) + Q*log(Q))
  Vl0 <- rowSums(P*Q*(log(P/Q))^2)
  res <- as.vector(round((l0 - El0) / sqrt(Vl0),4))
  # Export results:
  export.res.P(matrix, N, res, "lz", vector("list", 5) , Ncat=2, NA.method, 
                IRT.PModel, res.NA[[2]], Ability.PModel, res.NA[[3]], IP.NA, Ability.NA, res.NA[[4]])
}########################################################################################
########################################################################################
# lzstar
########################################################################################
########################################################################################
lzstar <- function(matrix, 
                    NA.method="NPModel", Save.MatImp=FALSE, 
                    IP=NULL, IRT.PModel="2PL", Ability=NULL, Ability.PModel="ML", mu=0, sigma=1)
{
  matrix      <- as.matrix(matrix)
  N           <- dim(matrix)[1]; I <- dim(matrix)[2]
  IP.NA       <- is.null(IP); Ability.NA  <- is.null(Ability)
  # Sanity check - Data matrix adequacy:
  Sanity.dma(matrix, N, I)
  # Estimate item parameters if not provided (using 'irtoys'):
  IP <- estIP(matrix, IP, IRT.PModel)
  # Estimate ability parameters if not provided (using 'irtoys'):
  Ability <- estAb(matrix, IP, Ability, Ability.PModel, mu, sigma)
  # Dealing with missing values:
  res.NA <- MissingValues(matrix, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma)
  matrix <- res.NA[[1]]
  # Perfect response vectors allowed.
  # Compute PFS: 
  A   <- IP[,1]; B <- IP[,2]; C <- IP[,3]
  P   <- do.call(cbind, lapply(1:I, function (x){C[x]+(1-C[x]) / (1+exp(-A[x]*(Ability - B[x])))}))
  Q   <- 1-P
  d1P <- do.call(cbind, lapply(1:I,function (x){(1-C[x])*A[x]*exp(A[x]*(Ability - B[x])) / (1+exp(A[x]*(Ability - B[x])))^2}))
  d2P <- do.call(cbind, lapply(1:I,function (x){
    (1-C[x])*(A[x]^2)*exp(A[x]*(Ability - B[x]))*(1-exp(A[x]*(Ability - B[x]))) / (1+exp(A[x]*(Ability - B[x])))^3}))
  ri  <- d1P/(P*Q)
  r0  <- switch(Ability.PModel,
               ML = 0,
               BM = (mu-Ability) / (sigma^2),
               WL = rowSums((d1P*d2P)/(P*Q)) / (2*rowSums((d1P^2)/(P*Q))))
  wi       <- log(P/Q)
  Wn       <- rowSums((matrix - P)*wi)
  sigma2n  <- rowSums((wi^2)*P*Q) / I
  cn       <- rowSums(d1P*wi) / rowSums(d1P*ri)
  wi.tilde <- wi - matrix(rep(cn,I), nrow=N) * ri
  tau2n    <- rowSums((wi.tilde^2)*P*Q) / I
  EWn      <- -cn * r0
  VWn      <- I * tau2n
  res      <- as.vector(round((Wn - EWn) / sqrt(VWn), 4))
  # Export results:
  export.res.P(matrix, N, res, "lzstar", vector("list", 5) , Ncat=2, NA.method, 
               IRT.PModel, res.NA[[2]], Ability.PModel, res.NA[[3]], IP.NA, Ability.NA, res.NA[[4]])
}# Dealing with missing values:
MissingValues <- function(matrix, NAs, Save.MatImp, IP, ParModel, Ability, Method, mu, sigma)
{
  N <- dim(matrix)[1]; I <- dim(matrix)[2]
  if (sum(is.na(matrix)) > 0)
  {
    lst <- switch(
      NAs,
      Hotdeck = HD.imputation(matrix, Save.MatImp, IP, Ability),
      NPModel = NPModel.imputation(matrix, Save.MatImp, IP, Ability),
      PModel  = {
        # Sanity check - IP model:
        Sanity.IPm(ParModel)
        # Sanity check - Ability method:
        Sanity.Abm(Method)
        # 
        PModel.imputation(matrix, Save.MatImp, 
                                      IP, ParModel, Ability, Method, mu, sigma)
      }
    )
  } else
  {
    if (Save.MatImp == TRUE)
    {
      write.matrix(matrix, file="Datamatrix_original.txt", sep=" ")
    }
    lst <- list(matrix, IP, Ability, 0)
  }
  lst
}

# Dealing with missing values (polytomous):
MissingValues.poly <- function(matrix, Ncat, NAs, Save.MatImp, IP, ParModel, Ability, Method)
{
  N <- dim(matrix)[1]; I <- dim(matrix)[2]
  if (sum(is.na(matrix)) > 0)
  {
    lst <- switch(
      NAs,
      Hotdeck = HD.imputation(matrix, Save.MatImp, IP, Ability),
      NPModel = NPModel.imputation.poly(matrix, Ncat, Save.MatImp, IP, Ability),
      PModel  = {
        PModel.imputation.poly(matrix, Ncat, Save.MatImp,IP, ParModel, Ability, Method)
      }
    )
  } else
  {
    if (Save.MatImp == TRUE)
    {
      write.matrix(matrix, file="Datamatrix_original.txt", sep=" ")
    }
    lst <- list(matrix, IP, Ability, 0)
  }
  lst
}
########################################################################################
########################################################################################
# NCI (Tatsuoka & Tatsuoaka, 1982, 1983):
# This statistic is perfectly linearly related to Gnormed (van der Flier, 1977; Meijer, 1994)
# NCI = 1-2Gnormed
########################################################################################
########################################################################################
NCI <- function(matrix, 
                NA.method="NPModel", Save.MatImp=FALSE, 
                IP=NULL, IRT.PModel="2PL", Ability=NULL, Ability.PModel="ML", mu=0, sigma=1)
{
  matrix      <- as.matrix(matrix)
  N           <- dim(matrix)[1]; I <- dim(matrix)[2]
  IP.NA       <- is.null(IP); Ability.NA  <- is.null(Ability)
  # Sanity check - Data matrix adequacy:
  Sanity.dma(matrix, N, I)
  # Dealing with missing values:
  res.NA <- MissingValues(matrix, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma)
  matrix <- res.NA[[1]]
  # Perfect response vectors allowed (albeit uninformative).
  # Compute PFS:
  NC         <- rowSums(matrix)
  uniqueNC   <- sort(unique(NC))
  pi         <- colMeans(matrix)
  matrix.ord <- matrix[,order(pi,decreasing=TRUE)]
  per.row <- function(vect)
  {
    ind.0     <- which(vect == 0)
    ind.1     <- which(vect == 1)
    all.cases <- expand.grid(ind.0,ind.1)
    sum((all.cases[,2] - all.cases[,1]) > 0)
  }
  res <- apply(matrix.ord,1,per.row)
  res <- res / (NC*(I-NC))
  res <- 1 - 2 * res
  res[is.nan(res)] <- 0 # all-0s or all-1s vector -> NCI = 0
  # Export results:
  export.res.NP(matrix, N, res, "NCI", vector("list", 5), Ncat=2, NA.method, 
                IRT.PModel, res.NA[[2]], Ability.PModel, res.NA[[3]], IP.NA, Ability.NA, res.NA[[4]])
}
# Nonparametric model imputation (polytomous)
# Similar to the hotdeck imputation, but item scores are generated from multinomial distributions, 
#    with probabilities defined by donors with similar total score than the recipient (based on all items except the NAs):
NPModel.imputation.poly <- function(matrix, Ncat, save.matImp, ip, ability)
{
  N <- dim(matrix)[1]; I <- dim(matrix)[2]
  M <- Ncat - 1
  matrix.imp <- matrix
  position.NA <- is.na(matrix)
  recipients <- which(rowSums(position.NA) > 0)
  N.recipients <- length(recipients)
  donors <- (1:N)[-recipients]
  N.donors <- length(donors)
  # 
  vect.NC <- rowSums(matrix, na.rm=TRUE)
  for (i in 1:N.recipients)
  {
    rcp <- recipients[i]
    rcp.noNA <- (1:I)[!position.NA[rcp, ]]
    rcp.NC <- vect.NC[rcp]
    donors.NC <- rowSums(matrix[donors, rcp.noNA])
    mar <- 0
    ctrl <- 0
    while (ctrl == 0)
    {
      closest.donors <- (abs(donors.NC - rcp.NC) <= mar)
      if (sum(closest.donors) > 0)
      {
        ctrl <- 1
      } else
      {
        mar <- mar+1
      }
    }
    freq.abs <- apply(matrix[donors[closest.donors], position.NA[rcp,], drop=FALSE],2,
                      function(vec) {table(factor(vec, levels=0:M))})
    matrix.imp[rcp, position.NA[rcp,]] <- 
      which(apply(freq.abs,2, function(vect) {rmultinom(1,1,vect)}) == 1, arr.ind=TRUE)[,1] - 1
    
  }
  # 
  if (save.matImp == TRUE)
  {
    write.matrix(matrix.imp, file="Datamatrix_imputted.txt", sep=" ")
  }
  list(matrix.imp, ip, ability, 1)
}
# Nonparametric model imputation (dichotomous)
# Similar to the hotdeck imputation, but item scores are generated from Bernoulli distributions, 
#    with probabilities defined by donors with similar total score than the recipient (based on all items except the NAs):
NPModel.imputation <- function(matrix, save.matImp, ip, ability)
{
  N <- dim(matrix)[1]; I <- dim(matrix)[2]
  matrix.imp <- matrix
  position.NA <- is.na(matrix)
  recipients <- which(rowSums(position.NA) > 0)
  N.recipients <- length(recipients)
  donors <- (1:N)[-recipients]
  N.donors <- length(donors)
  # 
  vect.NC <- rowSums(matrix, na.rm=TRUE)
  # 
  for (i in 1:N.recipients)
  {
    rcp <- recipients[i]
    rcp.noNA <- (1:I)[!position.NA[rcp, ]]
    rcp.NC <- vect.NC[rcp]
    donors.NC <- rowSums(matrix[donors, rcp.noNA])
    mar <- 0
    ctrl <- 0
    while (ctrl == 0)
    {
      closest.donors <- (abs(donors.NC - rcp.NC) <= mar)
      if (sum(closest.donors) > 0)
      {
        ctrl <- 1
      } else
      {
        mar <- mar+1
      }
    }
    matrix.imp[rcp, position.NA[rcp,]] <- rbinom(sum(position.NA[rcp,]), 1, colMeans(matrix[donors[closest.donors], position.NA[rcp,], drop=FALSE]))
  }
  # 
  if (save.matImp == TRUE)
  {
    write.matrix(matrix.imp, file="Datamatrix_imputted.txt", sep=" ")
  }
  list(matrix.imp, ip, ability, 1)
}

PerFit.PFS <- function(matrix, method=NULL, simplified=TRUE, 
                   NA.method="NPModel", Save.MatImp=FALSE, 
                   IP=NULL, IRT.PModel=NULL, Ability=NULL, Ability.PModel=NULL, mu=0, sigma=1)
{
  matrix      <- as.matrix(matrix)
  N        <- dim(matrix)[1]; I <- dim(matrix)[2];
  dico.PFS <- c("Cstar", "C.Sato", "U3", "ZU3", "G", "Gnormed", "D.KB", "r.pbis", "NCI", "Ht", "A.KB", "E.KB", "lz", "lzstar")
  poly.PFS <- c("Gpoly", "Gnormed.poly", "U3poly", "lzpoly")
  
  # Sanity check - Are data dichotomous or polytomous?
  data.type <- NA
  if (class(try(Sanity.dma(matrix, N, I), silent=TRUE)) != "try-error")
  {
    data.type <- "dico"
    Ncat      <- 2
  } else
  {
    Ncat <- max(matrix, na.rm = TRUE) + 1
    M    <- Ncat - 1
    if (class(try(Sanity.dma.poly(matrix, N, I, M), silent=TRUE)) != "try-error")
    {
      data.type <- "poly"
    } else
    {
      stop('The data matrix is not dichotomous (0/1 scores) nor 
           polytomous (scores in {0, 1, ..., Ncat-1}, including 0 and (Ncat - 1)). Aborted.')
    }
  }
  
  # Sanity check - Were any PFS methods added?
  if (length(method) == 0)
  {
    stop('Please add your PFSs of choice to vector "method" before proceeding. Aborted.')
  }
  
  # Sanity check - Are the methods in accordance to the type of data?
  if ((data.type == "dico") & !all(method %in% dico.PFS))
  {
    stop('One or more PFSs declared in parameter "method" are not suitable to dichotomous data. Aborted.')
  } else
  {
    if (is.null(IRT.PModel)) {IRT.PModel <- "2PL"}
    if (is.null(Ability.PModel)) {Ability.PModel <- "ML"}
  }
  # 
  if ((data.type == "poly") & !all(method %in% poly.PFS))
  {
    stop('One or more PFSs declared in parameter "method" are not suitable to polyotomous data. Aborted.')
  } else
  {
    if (is.null(IRT.PModel)) {IRT.PModel <- "GRM"}
    if (is.null(Ability.PModel)) {Ability.PModel <- "EAP"}
  }
  
  # Compute PFSs:
  res <- vector("list", length(method))
  if (data.type == "dico")
  {
    for (i in 1:length(method))
    {
      res[[i]] <- eval(parse(text = method[i]))(matrix, 
                                                NA.method, Save.MatImp, 
                                                IP, IRT.PModel, Ability, Ability.PModel, mu, sigma)
    }
  }
  # 
  if (data.type == "poly")
  {
    for (i in 1:length(method))
    {
      res[[i]] <- eval(parse(text = method[i]))(matrix, Ncat, 
                                                NA.method, Save.MatImp, 
                                                IP, IRT.PModel, Ability, Ability.PModel)
    }
  }
  
  if (simplified == TRUE)
  {
    rownames.bckp <- rownames(res[[1]]$PFscores)
    res           <- data.frame(matrix(unlist(lapply(res, function(lst) {lst[[1]]})), nrow=N))
    colnames(res) <- method
    rownames(res) <- rownames.bckp
  }
  
  res
}




# Jackknife procedure to estimate SE for PFSs:
PerFit.SE <- function(x)
{
  dico.PFS <- c("Cstar", "C.Sato", "U3", "ZU3", "G", "Gnormed", "D.KB", "r.pbis", "NCI", "Ht", "A.KB", "E.KB", "lz", "lzstar")
  poly.PFS <- c("Gpoly", "Gnormed.poly", "U3poly", "lzpoly")
  # 
  matrix <- x$Matrix      # 'matrix' is NA-free
  N      <- dim(matrix)[1]; I <- dim(matrix)[2] 
  PFS    <- x$PFStatistic
  JK.mat <- matrix(NA, nrow=N, ncol=I)
  if (PFS == "lzpoly")
  {
    for (it in 1:I)
    {
      JK.mat[,it] <- do.call(PFS,c( list(matrix[,-it]), list(Ncat=x$Ncat, IP=NULL, Ability=NULL) ))$PFscores[,1]
    }
  } else
  {
    if (PFS %in% dico.PFS)
    {
      for (it in 1:I)
      {
        JK.mat[,it] <- do.call(PFS,c( list(matrix[,-it]), list(IP=x$IP[-it,], Ability=NULL) ))$PFscores[,1]
      }
    }
    if (PFS %in% poly.PFS)
    {
      for (it in 1:I)
      {
        JK.mat[,it] <- do.call(PFS,c( list(matrix[,-it]), list(Ncat=x$Ncat, IP=x$IP[-it,], Ability=NULL) ))$PFscores[,1]
      }
    }
  }
  SE <- sqrt( ((I-1)/I) * rowSums((JK.mat - rowMeans(JK.mat, na.rm=TRUE))^2, na.rm=TRUE) )
  PFS.NA    <- c(x$ID.all0s, x$ID.all1s)
  if (!is.null(PFS.NA))
  {
    SE[PFS.NA] <- NA
  }
  cbind(PFscores = x$PFscores[,1], PFscores.SE = round(SE, 4))
}
# Define plot() function for class "PerFit":
plot.PerFit <- function (x, #x = an object from 'PerFit' class
                         cutoff.obj=NULL, #cutoff.obj = an object from 'PerFit.cutoff' class
                         ModelFit="NonParametric", Nreps=1000, 
                         IP=x$IP, IRT.PModel=x$IRT.PModel, Ability=x$Ability, Ability.PModel=x$Ability.PModel, mu=0, sigma=1, 
                         Blvl = 0.05, Breps = 1000, CIlvl = 0.95, 
                         UDlvl = NA, 
                         # 
                         Type="Density", Both.scale=TRUE, Cutoff=TRUE, Cutoff.int=TRUE, Flagged.ticks = TRUE, 
                         Xlabel=NA, Xcex=1.5, title=NA, Tcex=1.5,
                         col.area="lightpink", col.hist="lightblue", col.int="darkgreen", col.ticks="red", ...)
{  
  # Sanity check - Class PerFit:
  Sanity.cls(x)  
  # 
  upp.PFS  <- c("Cstar", "C.Sato", "U3", "ZU3", "G", "Gnormed", "Gpoly", "Gnormed.poly", "U3poly", "D.KB")
  low.PFS  <- c("r.pbis", "NCI", "Ht", "A.KB", "E.KB", "lz", "lzstar", "lzpoly")
  # 
  pf.scores       <- x$PFscores[[1]]
  PFS.NA          <- is.na(pf.scores)
  pf.scores.noNAs <- pf.scores[!PFS.NA]
  # 
  if (is.null(cutoff.obj))
  {
    cutoff.res <- cutoff(x, ModelFit, Nreps, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma, Blvl, Breps, CIlvl, UDlvl)
  } else
  {
    Sanity.clsPO(cutoff.obj)
    cutoff.res <- cutoff.obj
  }
  # 
  x.line       <- cutoff.res$Cutoff
  perc.flagged <- round(100 * cutoff.res$Prop.flagged, 2)
  direction    <- paste(", ",cutoff.res$Tail," ",sep="")
  # 
  PFS.flagged  <- flagged.resp(x, cutoff.res, scores=FALSE)[[1]][,2]
  # Find correct scale for y-axis:
  ymax.hist    <- max(hist(pf.scores.noNAs, plot=FALSE)$density)
  ymax.dens    <- max(density(pf.scores.noNAs)$y)
  ymax         <- switch(Type,
                         Density   = ymax.dens,
                         Histogram = ymax.hist,
                         Both      = if (Both.scale == TRUE) {max(ymax.dens,ymax.hist)} else {min(ymax.dens,ymax.hist)})
  par(mar=c(4,3.5,2,1)+.1,las=1)
  hist(pf.scores.noNAs,freq=FALSE,border="white",ann=FALSE,ylim=c(0,ymax))
  #
  if (Cutoff == TRUE)
  {
    if (any(x$PFStatistic == upp.PFS))
    {
      rect(x.line,0,par("usr")[2], par("usr")[4],col=col.area,border=NA)
    }
    if (any(x$PFStatistic == low.PFS))
    {
      rect(par("usr")[1],0, x.line,par("usr")[4],col=col.area,border=NA)
    }
  }
  #
  if (Type == "Histogram")
  {
    par(new=TRUE)
    hist(pf.scores.noNAs,freq=FALSE,col=col.hist,ann=FALSE,ylim=c(0,ymax))
  }
  #
  if (Type == "Density")
  {
    points(density(pf.scores.noNAs),type="l",lwd=2,ann=FALSE,ylim=c(0,ymax))
  }
  #
  if (Type == "Both")
  {
    par(new=TRUE)
    hist(pf.scores.noNAs,freq=FALSE,col=col.hist,ann=FALSE,ylim=c(0,ymax))
    points(density(pf.scores.noNAs),type="l",lwd=2,ann=FALSE,ylim=c(0,ymax))
  }
  box(col="black")
  #
  if (Cutoff == FALSE)
  {
    tmp <- if (is.na(Xlabel)) {x$PFStatistic} else {Xlabel}
    mtext(side=1,text=tmp,line=2.5,col="black",cex=1.5,font=1)
  }
  # Add flagged respondents:
  if (Flagged.ticks == TRUE)
  {
    axis(3, at=PFS.flagged, labels=FALSE, tick=TRUE, lwd.ticks=2, col.ticks=col.ticks)
  }
  # Add bootstrap CIlvl% CI to x-axis:
  if (Cutoff.int == TRUE)
  {
    segments(x0=cutoff.res$Cutoff.CI[1], y0=par("usr")[3], 
             x1=cutoff.res$Cutoff.CI[2], y1=par("usr")[3], lwd=6, col=col.int, xpd=TRUE)
  }
  # 
  if (Cutoff == TRUE)
  {
    abline(v=x.line,lwd=2)
    tmp <- if (is.na(Xlabel))
    {
      paste(x$PFStatistic," (cutoff=",round(x.line,3),direction,perc.flagged,"%)",sep="")
    } else 
    {
      Xlabel
    }
    mtext(side=1,text=tmp,line=2.5,col="black",cex=Xcex,font=1) 
  }
  #
  tmp <- if (is.na(title))
  {
    "Distribution"
  } else 
  {
    title
  }
  mtext(side=3,text=tmp,line=.5,col="black",cex=Tcex,font=2)
}
# Parametric model imputation (polytomous)
# Item scores are generated from multinomial distributions, with probabilities estimated by means of parametric IRT models
#     (PCM, GPCM, GRM):
PModel.imputation.poly <- function(matrix, Ncat, save.matImp, ip, model, ability, method)
{
  N <- dim(matrix)[1]; I <- dim(matrix)[2]
  # 
  matrix2 <- data.frame(apply(matrix,2,as.factor)) # eliminates item levels with no answers
  
  # Estimate item parameters if not provided (polytomous):
  ip.res <- estIP.poly(matrix, Ncat, ip, model)
  ip     <- ip.res[[1]]
  ip.ltm <- ip.res[[2]] 
  
  # Estimate ability parameters if not provided (using 'ltm'):
  ability <- estAb.poly(matrix, ip.ltm, ability, method)
  
  # Compute P.CRF:
  P.CRF <- estP.CRF(I, Ncat, model, ip, ability)
  
  # Fill in NAs:
  matrix.imp      <- matrix
  position.NA.mat <- is.na(matrix)
  resp.NA         <- which(rowSums(position.NA.mat) > 0)
  for (i in 1:length(resp.NA))
  {
    resp <- resp.NA[i]
    position.NA <- (1:I)[position.NA.mat[resp, ]]
    P.CRF.ind   <- sapply(position.NA,function(x){((x-1)*Ncat+1):(x*Ncat)})
    P.CRF.NA    <- matrix(P.CRF[resp, P.CRF.ind], ncol=length(position.NA))
    matrix.imp[resp, position.NA] <- 
      which(apply(P.CRF.NA,2, function(vect) {rmultinom(1,1,vect)}) == 1, arr.ind=TRUE)[,1] - 1
  }
  # 
  if (save.matImp == TRUE)
  {
    write.matrix(matrix.imp, file="Datamatrix_imputted.txt", sep=" ")
  }
  list(matrix.imp, ip, ability, 1)
}
# Parametric model imputation
# Item scores are generated from Bernoulli distributions, with probabilities estimated by means of parametric IRT models
#     (1PLM, 2PLM, 3PLM):
PModel.imputation <- function(matrix, save.matImp, ip, model, ability, method, mu, sigma)
{
  N <- dim(matrix)[1]; I <- dim(matrix)[2]
  
  # Estimate item parameters if not provided (using 'irtoys'):
  ip <- estIP(matrix, ip, model)
  
  # Estimate ability parameters if not provided (using 'irtoys'):
  ability <- estAb(matrix, ip, ability, method, mu, sigma)
  
  A   <- ip[,1]; B <- ip[,2]; C <- ip[,3]
  P   <- do.call(cbind, lapply(1:I,function (x) {C[x]+(1-C[x]) / (1+exp(-A[x]*(ability - B[x])))}))
  # 
  matrix.imp <- matrix
  position.NA <- which(is.na(matrix) == 1, arr.ind=TRUE)
  P.NA <- P[position.NA]
  matrix.imp[position.NA] <- rbinom(length(P.NA), 1, P.NA)
  # 
  if (save.matImp == TRUE)
  {
    write.matrix(matrix.imp, file="Datamatrix_imputted.txt", sep=" ")
  }
  list(matrix.imp, ip, ability, 1)
}
# Compute PRFs for all respondents simultaneously.
# Also, compute a functional data object with all the PRFs.
# See: Ramsay, Hooker, & Graves (2009). Functional data analysis with R and Matlab.
PRF <- function(matrix, h=.09, N.FPts=101)
{
  Diffs       <- 1 - colMeans(matrix)
  focal.pts   <- seq(0, 1, length.out=N.FPts)
  GaussKernel <- function(x) {dnorm(x, mean=0, sd=1)}
  KernArg     <- expand.grid(focal.pts, Diffs)
  KernArg     <- matrix(KernArg[,1] - KernArg[,2], nrow=length(focal.pts), byrow=F) / h
  weights     <- GaussKernel(KernArg) / rowSums(GaussKernel(KernArg))
  PRFest      <- weights %*% t(matrix)
  # 
  # Specify a B-spline basis system (Chapter 3).
  # 'basis.bspline' below is a functional basis object of class 'basisfd'.
  # It is based on B-splines (piecewise polinomials, all of the same degree/order [order = degree + 1]).
  # Here we focus of degree three / order four splines (i.e., cubic polinomial segments),
  #   with one knot per break point.
  # This allows any two consecutive splines (piecewise polinomials), sp1 and sp2, with common break point BP,
  #   verifying sp1(BP) = sp2(BP), sp1'(BP) = sp2'(BP), and sp1''(BP) = sp2''(BP).
  # At 0 and 1 (extremes of the x-range), four (= order) knots are used.
  basis.bspline <- create.bspline.basis(rangeval = c(0, 1), norder = 4, nbasis = (4 + 9))
  # 
  # Specify coefficients c for the B-spline basis system computed above and then create functional data objects.
  # Based on smoothing using regression analysis (Section 4.3 in Ramsay et al., 2009).
  x.values     <- focal.pts
  basis.values <- eval.basis(evalarg=x.values, basisobj=basis.bspline)
  y.values     <- PRFest
  basis.coefs  <- solve(crossprod(basis.values), crossprod(basis.values, y.values))
  # Observe that 'basis.values %*% basis.coefs' is the B-spline approximation of PRFest.
  fd.obj       <- fd(basis.coefs, basis.bspline, list("Item difficulty", "Subject", "Probability correct answer"))
  list(PRFdiffs=focal.pts, PRFest=PRFest, FDO=fd.obj)
}

PRF.VarBands <- function (matrix, h=.09, N.FPts=101, alpha=.05)
{
  focal.pts <- seq(0, 1, length.out=N.FPts)
  N         <- dim(matrix)[1]; I <- dim(matrix)[2]
  PRFscores <- PRF(matrix, h, N.FPts)$PRFest
  # Jackknife estimate of the SE:
  PRF.SEarray <- array(NA, c(length(focal.pts), I, N))
  for (it in 1:I)
  {
    matrix.jack         <- matrix[, -it]
    PRF.SEarray[, it, ] <- PRF(matrix.jack, h, N.FPts)$PRFest
  }
  PRF.SE <- apply(PRF.SEarray, 3, function(mat)
  {
    sqrt( ((I-1)/I) * rowSums((mat - rowMeans(mat))^2) )
  })
  crit.val         <- qnorm(1-alpha, mean=0, sd=1)
  PRF.VarBandsLow  <- PRFscores-crit.val*PRF.SE
  PRF.VarBandsHigh <- PRFscores+crit.val*PRF.SE
  # 
  # Specify a B-spline basis system.
  basis.bspline <- create.bspline.basis(rangeval = c(0,1), norder = 4, nbasis = (4 + 9))
  # Specify coefficients c for the B-spline basis system computed above and then create functional data objects.
  x.values      <- focal.pts
  basis.values  <- eval.basis(evalarg=x.values, basisobj=basis.bspline)
  # 
  y.values        <- PRF.VarBandsLow
  basis.coefs.Low <- solve(crossprod(basis.values), crossprod(basis.values, y.values))
  fd.obj.Low      <- fd(basis.coefs.Low, basis.bspline, list("Item difficulty", "Subject", "Probability correct answer"))
  # 
  y.values         <- PRF.VarBandsHigh
  basis.coefs.High <- solve(crossprod(basis.values), crossprod(basis.values, y.values))
  fd.obj.High      <- fd(basis.coefs.High, basis.bspline, list("Item difficulty", "Subject", "Probability correct answer"))
  #
  list(PRF.VarBandsLow=PRF.VarBandsLow, PRF.VarBandsHigh=PRF.VarBandsHigh, 
       FDO.VarBandsLow=fd.obj.Low, FDO.VarBandsHigh=fd.obj.High)
}

PRFplot <- function (matrix, respID, h=.09, N.FPts=101, 
                     VarBands=FALSE, VarBands.area=FALSE, alpha=.05,
                     Xlabel=NA, Xcex=1.5, Ylabel=NA, Ycex=1.5, title=NA, Tcex=1.5, 
                     NA.method="NPModel", Save.MatImp=FALSE, 
                     IP=NULL, IRT.PModel="2PL", Ability=NULL, Ability.PModel="ML", mu=0, sigma=1, 
                     message=TRUE)
{
  matrix      <- as.matrix(matrix)
  N <- dim(matrix)[1]; I <- dim(matrix)[2]
  # Sanity check - Dichotomous data only:
  Sanity.dma(matrix, N, I)
  # 
  # Dealing with missing values:
  res.NA <- MissingValues(matrix, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma)
  matrix <- res.NA[[1]]
  # 
  res1 <- PRF(matrix, h, N.FPts)
  res2 <- PRF.VarBands(matrix, h, N.FPts, alpha)
  # 
  basis.bspline    <- create.bspline.basis(rangeval = c(0,1), norder = 4, nbasis = (4 + 9))
  x.values         <- seq(0,1,length.out=N.FPts)
  basis.values     <- eval.basis(evalarg=x.values, basisobj=basis.bspline)
  PRF.VarBandsLow  <- basis.values %*% res2$FDO.VarBandsLow$coefs
  PRF.VarBandsHigh <- basis.values %*% res2$FDO.VarBandsHigh$coefs
  for (i in 1:length(respID))
  {
    if (message)
    {
      readline(prompt=paste0("Respondent ", respID[i], ": Press ENTER."))
    }
    par(mar=c(4,4,2,1)+.1, las=1)
    plot(1, type="n", axes=FALSE, ann=FALSE, frame.plot=TRUE, xlim=c(0,1), ylim=c(0,1))
    tmpx <- if (is.na(Xlabel)) {"Item difficulty"} else {Xlabel}
    axis(1,at=seq(0, 1, by=.2)); mtext(side=1, text=tmpx, line=2.5, col="black", cex=Xcex, font=1)
    tmpy <- if (is.na(Ylabel)) {"Probability correct answer"} else {Ylabel}
    axis(2,at=seq(0, 1, by=.2)); mtext(side=2, text=tmpy, line=2.8, col="black", cex=Ycex, font=1, las=3)
    if (VarBands.area)
    {
      polygon(c(x.values,rev(x.values)),
              c(PRF.VarBandsHigh[,respID[i]], rev(PRF.VarBandsLow[,respID[i]])),col = "lightpink1",border=NA)
      par(new=TRUE); plot(res2$FDO.VarBandsLow[respID[i]],ann=FALSE, xlim=c(0,1),ylim=c(0,1), lty=2,lwd=1.5, href=FALSE, axes=FALSE)
      par(new=TRUE); plot(res2$FDO.VarBandsHigh[respID[i]],ann=FALSE,xlim=c(0,1),ylim=c(0,1), lty=2,lwd=1.5, href=FALSE, axes=FALSE)
      
    }
    if (!VarBands.area & VarBands)
    {
      par(new=TRUE); plot(res2$FDO.VarBandsLow[respID[i]],ann=FALSE, xlim=c(0,1),ylim=c(0,1), lty=2,lwd=1.5, href=FALSE, axes=FALSE)
      par(new=TRUE); plot(res2$FDO.VarBandsHigh[respID[i]],ann=FALSE,xlim=c(0,1),ylim=c(0,1), lty=2,lwd=1.5, href=FALSE, axes=FALSE)
    }
    par(new=TRUE); plot(res1$FDO[respID[i]],lwd=2,axes=FALSE,ann=FALSE,frame.plot=T,xlim=c(0,1),ylim=c(0,1), href=FALSE)
    tmp <- if (is.na(title)) {paste("PRF (respID # ",respID[i]
                                    ,")",sep="")} else {title}
    mtext(side=3, text=tmp, line=.5, col="black", cex=Tcex, font=2)
  }
  list(PRF.FDO = res1$FDO, VarBandsLow.FDO = res2$FDO.VarBandsLow,  VarBandsHigh.FDO = res2$FDO.VarBandsHigh)
}

# Define print() function for class "PerFit":
print.PerFit <- function(x, #x = an object from 'PerFit' class
                         cutoff.obj=NULL, #cutoff.obj = an object from 'PerFit.cutoff' class
                         ModelFit="NonParametric", Nreps=1000, 
                         IP=x$IP, IRT.PModel=x$IRT.PModel, Ability=x$Ability, Ability.PModel=x$Ability.PModel, mu=0, sigma=1, 
                         Blvl = 0.05, Breps = 1000, CIlvl = 0.95, 
                         UDlvl = NA, ...)
{
  N <- dim(x$Matrix)[1]; I <- dim(x$Matrix)[2]
  # Sanity check - Class PerFit:
  Sanity.cls(x)  
  # 
  dico.PFS <- c("Cstar", "C.Sato", "U3", "ZU3", "G", "Gnormed", "D.KB", "r.pbis", "NCI", "Ht", "A.KB", "E.KB", "lz", "lzstar")
  poly.PFS <- c("Gpoly", "Gnormed.poly", "U3poly", "lzpoly")
  
  # Compute cutoff:
  if (is.null(cutoff.obj))
  {
    cutoff.res <- cutoff(x, ModelFit, Nreps, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma, Blvl, Breps, CIlvl, UDlvl)
  } else
  {
    Sanity.clsPO(cutoff.obj)
    cutoff.res <- cutoff.obj
  }
  
  # Compute flagged:
  flagged.res <- flagged.resp(x, cutoff.res, scores=FALSE)[[1]][,1]
  
  # Summarize results:
  flagged.bin <- rep("", N)
  flagged.bin[flagged.res] <- "*"
  all.PFS <- data.frame(PerFit.SE(x), Flagged=flagged.bin)
  print(all.PFS)
  # 
  cat(paste0("\nPFS = ", x$PFStatistic, "\n"))
  cat(paste0("Cutoff = ", cutoff.res$Cutoff, " (SE = ", cutoff.res$Cutoff.SE, ").\n"))
  cat(paste0("Tail = ", cutoff.res$Tail, ".\n"))
  cat(paste0("Proportion of flagged respondents = ", cutoff.res$Prop.flagged, ".\n"))
  cat("(N.B.: The cutoff varies each time cutoff() is run due to bootstrapping.)\n\n")
  # 
  cat(paste0("Identified respondents - ", length(flagged.res), " in total:\n"))
  cat("   ", flagged.res, "\n\n")
}
########################################################################################
########################################################################################
# r.pbis, personal point-biserial correlation (Brennan, 1980, cited in Harhisch & Linn, 1981):
########################################################################################
########################################################################################
r.pbis <- function(matrix, 
                   NA.method="NPModel", Save.MatImp=FALSE, 
                   IP=NULL, IRT.PModel="2PL", Ability=NULL, Ability.PModel="ML", mu=0, sigma=1)
{
  matrix      <- as.matrix(matrix)
  N           <- dim(matrix)[1]; I <- dim(matrix)[2]
  IP.NA       <- is.null(IP); Ability.NA  <- is.null(Ability)
  # Sanity check - Data matrix adequacy:
  Sanity.dma(matrix, N, I)
  # Dealing with missing values:
  res.NA <- MissingValues(matrix, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma)
  matrix <- res.NA[[1]]
  # Sanity check - Perfect response vectors:
  part.res  <- Sanity.prv(matrix, N, I)
  NC        <- part.res$NC
  all.0s    <- part.res$all.0s
  all.1s    <- part.res$all.1s
  matrix.sv <- matrix
  matrix    <- part.res$matrix.red
  # Compute PFS:
  pi      <- colMeans(matrix.sv)
  res.red <- as.vector(cor(t(matrix), pi))
  # Compute final PFS vector:
  res <- final.PFS(res.red, all.0s, all.1s, N)
  # Export results:
  export.res.NP(matrix.sv, N, res, "r.pbis", part.res, Ncat=2, NA.method, 
                IRT.PModel, res.NA[[2]], Ability.PModel, res.NA[[3]], IP.NA, Ability.NA, res.NA[[4]])
}
# Sanity check - Data matrix adequacy: ----
Sanity.dma <- function(matrix, N, I)
{
  if (!is.numeric(matrix) | (sum(matrix == 0 | matrix == 1, na.rm=TRUE) != (N*I - sum(is.na(matrix)))))
  {
    stop('The data matrix is not numeric with 0/1 entries only. Aborted.')
  }
}

# Sanity check - Data matrix adequacy (polytomous): ----
Sanity.dma.poly <- function(matrix, N, I, M)
{
  if (!is.numeric(matrix) | 
        (sum(matrix %in% (0:M), na.rm=TRUE) != (N*I - sum(is.na(matrix)))) | 
        sum(c(min(matrix, na.rm=TRUE), max(matrix, na.rm=TRUE)) != c(0, M), na.rm=TRUE))
  {
    stop('The data matrix is not numeric with entries {0, 1, ..., Ncat-1} only. Aborted.')
  }
}

# Sanity check - Perfect response vectors: ----
Sanity.prv <- function(matrix, N, I)
{
  NC       <- rowSums(matrix)
  uniqueNC <- sort(unique(NC))
  all.0s   <- vector(mode="numeric", length=0)
  all.1s   <- vector(mode="numeric", length=0)
  perfect.vectors <- NULL
  if (min(uniqueNC)==0 | max(uniqueNC)==I)
  {
    perfect.vectors <- noquote("Not all item response vectors were included in the analysis (all-0s and/or all-1s patterns removed).")
    all.0s  <- which(NC == 0)
    all.1s  <- which(NC == I)
    NC      <- NC[-c(all.0s, all.1s)]
    if (length(all.0s) > 0) {uniqueNC <- uniqueNC[-1]}
    if (length(all.1s) > 0) {uniqueNC <- uniqueNC[-length(uniqueNC)]}
  } else
  {
    perfect.vectors <- noquote("All item response vectors were included in the analysis.")
  }
  # Data matrix without perfect vectors:
  if (length(c(all.0s, all.1s)) == 0)
  {
    matrix.red <- matrix
  } else
  {
    matrix.red <- matrix[(1:N)[-c(all.0s, all.1s)],]
  }
  list(perfect.vectors=perfect.vectors, all.0s=all.0s, all.1s=all.1s, NC=NC, matrix.red=matrix.red)
}

# Sanity check - IP matrix adequacy:----
Sanity.IPa <- function(ip, I)
{
  if (!is.numeric(ip) | dim(ip)[1] != I | dim(ip)[2] != 3)
  {
    stop('The item parameters matrix "IP" is not numeric with dimension I x 3 
           (I = number of items; columns = discrimination, difficulty, pseudo-guessing). 
           Aborted.')
  }
}

# Sanity check - IP matrix adequacy (polytomous):----
Sanity.IPa.poly <- function(IP, I, Ncat)
{
  if (!is.numeric(IP) | dim(IP)[1] != I | dim(IP)[2] != Ncat)
  {
    stop('The item parameters matrix "IP" is not numeric with dimension I x Ncat 
           (I = number of items; 
            first (Ncat-1) columns = thresholds [GRM] or difficulties [PCM, GPCM]; 
            column Ncat-th = slopes). 
           Aborted.')
  }
}

# Sanity check - Ability matrix adequacy:----
Sanity.ABa <- function(Ability, N)
{
  if (!is.numeric(Ability) | length(Ability) != N)
  {
    stop('The person parameters vector "Ability" is not numeric with length N 
           (N = number of respondents). 
           Aborted.')
  }
}

# Sanity check - IP model:----
Sanity.IPm <- function(model)
{
  if (!(model %in% c("1PL", "2PL", "3PL")))
  {
    stop('Parameter "model" can only be "1PL", "2PL", or "3PL". Aborted.')
  }
}

# Sanity check - IP model (polytomous):----
Sanity.IPm.poly <- function(model)
{
  if (!(model %in% c("PCM", "GPCM", "GRM")))
  {
    stop('Parameter "model" can only be "PCM", "GPCM", or "GRM". Aborted.')
  }
}

# Sanity check - Ability method:----
Sanity.Abm <- function(method)
{
  if (!(method %in% c("ML", "BM", "WL")))
  {
    stop('Parameter "method" can only be "ML", "BM", or "WL". Aborted.')
  }
}

# Sanity check - Ability method (polytomous):----
Sanity.Abm.poly <- function(method)
{
  if (!(method %in% c("EB", "EAP", "MI")))
  {
    stop('Parameter "method" can only be "EB", "EAP", or "MI". Aborted.')
  }
}

# Sanity check - Class PerFit:----
Sanity.cls <- function(x)
{
  if (class(x) != "PerFit")
  {
    stop('Object "x" is not of class PerFit. Aborted.')
  }
}

# Sanity check - Class PerFit.object:----
Sanity.clsPO <- function(x)
{
  if (class(x) != "PerFit.cutoff")
  {
    stop('Object "cutoff.obj" is not of class PerFit.cutoff. Aborted.')
  }
}# Define summary() function for class "PerFit":
summary.PerFit <- function(object, #object = an object from 'PerFit' class
                           cutoff.obj=NULL, #cutoff.obj = an object from 'PerFit.cutoff' class
                           ModelFit="NonParametric", Nreps=1000, 
                           IP=object$IP, IRT.PModel=object$IRT.PModel, Ability=object$Ability, Ability.PModel=object$Ability.PModel,
                           mu=0, sigma=1, 
                           Blvl = 0.05, Breps = 1000, CIlvl = 0.95, 
                           UDlvl = NA, ...)
{
  x <- object
  # Sanity check - Class PerFit:
  Sanity.cls(x)  
  
  # Compute cutoff:
  if (is.null(cutoff.obj))
  {
    cutoff.res <- cutoff(x, ModelFit, Nreps, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma, Blvl, Breps, CIlvl, UDlvl)
  } else
  {
    Sanity.clsPO(cutoff.obj)
    cutoff.res <- cutoff.obj
  }
  
  # Compute flagged:
  flagged.res <- flagged.resp(x, cutoff.res, scores=FALSE)$PFSscores[,1]
  
  # Summarize results:
  cat(paste0("\nPFS = ", x$PFStatistic, "\n"))
  cat(paste0("Cutoff = ", cutoff.res$Cutoff, " (SE = ", cutoff.res$Cutoff.SE, ").\n"))
  cat(paste0("Tail = ", cutoff.res$Tail, ".\n"))
  cat(paste0("Proportion of flagged respondents = ", cutoff.res$Prop.flagged, ".\n"))
  cat("(N.B.: The cutoff varies each time cutoff() is run due to bootstrapping.)\n\n")
  # 
  cat(paste0("Identified respondents - ", length(flagged.res), " in total:\n"))
  cat("   ", flagged.res, "\n\n")
}
########################################################################################
########################################################################################
# Polytomous items: U3p
# U3poly reduces to U3 for 0-1 data (Ncat=2)
########################################################################################
########################################################################################
U3poly <- function(matrix, Ncat,
                   NA.method="NPModel", Save.MatImp=FALSE, 
                   IP=NULL, IRT.PModel="GRM", Ability=NULL, Ability.PModel="EAP")
{
  matrix      <- as.matrix(matrix)
  N           <- dim(matrix)[1]; I <- dim(matrix)[2]; M <- Ncat-1
  IP.NA       <- is.null(IP); Ability.NA  <- is.null(Ability)
  # Sanity check - Data matrix adequacy:
  Sanity.dma.poly(matrix, N, I, M)
  # Dealing with missing values:
  res.NA <- MissingValues.poly(matrix, Ncat, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel)
  matrix <- res.NA[[1]]
  # Perfect response vectors allowed (albeit uninformative).
  # Compute PFS:
  NC        <- rowSums(matrix)
  probs.ISD <- matrix(NA,nrow=I,ncol=M)
  for (m in 1:M)
  {
    probs.ISD[,m] <- colMeans(matrix >= m)
  }
  f.scoresISD    <- function (x) {c(rep(1,x), rep(0,M-x))}
  matrix.ISD     <- matrix(unlist(lapply(t(matrix), f.scoresISD)), byrow=TRUE, nrow=N)
  probs.ISD.vect <- as.vector(t(probs.ISD))
  # If there are answer options not chosen by any respondent then some entries in 'probs.ISD.vect' are 0 and others are 1.
  # Below all corresponding logs are set from Inf to 0.
  # (Reason: They carry no information regarding aberrant response behavior).
  logits.ISD <- log(probs.ISD.vect/(1-probs.ISD.vect))
  logits.ISD[is.infinite(logits.ISD)] <- 0 # a vector
  W <- as.vector(matrix.ISD %*% logits.ISD)
  # 
  logits.ISD <- matrix(logits.ISD, nrow=I, byrow=TRUE) # a matrix
  if (Ncat>2)  {cumlogits.ISD <- cbind(rep(0,I), t(apply(logits.ISD, 1, cumsum)))}
  if (Ncat==2) {cumlogits.ISD <- cbind(rep(0,I), logits.ISD)}
  V        <- matrix(rep(cumlogits.ISD[1,], Ncat), ncol=Ncat, byrow=FALSE)
  add.term <- matrix(rep(cumlogits.ISD[2,], nrow(V)), ncol=Ncat, byrow=TRUE)
  V        <- V + add.term
  T        <- sapply(2:sum(dim(V)),function(x) {min(V[col(V)+row(V) == x])})
  for (i in 3:I) {
    V        <- matrix(rep(T,Ncat), ncol=Ncat, byrow=FALSE)
    add.term <- matrix(rep(cumlogits.ISD[i,], nrow(V)), ncol=Ncat, byrow=TRUE)
    V        <- V + add.term
    T        <- sapply(2:sum(dim(V)),function(x) {min(V[col(V)+row(V) == x])})
  }
  maxW <- sapply(0:(I*M), function(x) {sum(sort(as.vector(logits.ISD),decreasing=TRUE)[0:x])})
  minW <- T
  den  <- maxW - minW
  den[c(1,length(den))] <- 1 # so that vectors (0,0,...,0) and (M,M,...,M), which have 0 Guttman errors, are divided by 1 instead of 0
  #   
  res <- (maxW[NC+1] - W) / den[NC+1]
  # Export results:
  export.res.NP(matrix, N, res, "U3poly", vector("list", 5) , Ncat=Ncat, NA.method, 
               IRT.PModel, res.NA[[2]], Ability.PModel, res.NA[[3]], IP.NA, Ability.NA, res.NA[[4]])
}
########################################################################################
########################################################################################
# U3 (van der Flier, 1980, 1982):
########################################################################################
########################################################################################
U3 <- function(matrix, 
               NA.method="NPModel", Save.MatImp=FALSE, 
               IP=NULL, IRT.PModel="2PL", Ability=NULL, Ability.PModel="ML", mu=0, sigma=1)
{
  matrix      <- as.matrix(matrix)
  N           <- dim(matrix)[1]; I <- dim(matrix)[2]
  IP.NA       <- is.null(IP); Ability.NA  <- is.null(Ability)
  # Sanity check - Data matrix adequacy:
  Sanity.dma(matrix, N, I)
  # Dealing with missing values:
  res.NA <- MissingValues(matrix, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma)
  matrix <- res.NA[[1]]
  # Sanity check - Perfect response vectors:
  part.res  <- Sanity.prv(matrix, N, I)
  NC        <- part.res$NC
  all.0s    <- part.res$all.0s
  all.1s    <- part.res$all.1s
  matrix.sv <- matrix
  matrix    <- part.res$matrix.red
  # Compute PFS:
  pi <- colMeans(matrix.sv); qi <- 1-pi
  # If there are answer options not chosen by any respondent then some entries in pi are 0 or 1.
  # Below all corresponding logs are set from Inf to 0.
  # (Reason: They carry no individual information regarding aberrant response behavior.):
  log.odds     <- log(pi/qi)
  log.odds[is.infinite(log.odds)] <- 0
  log.odds.ord <- sort(log.odds,decreasing=TRUE)
  # 
  sum.first.logodds <- cumsum(log.odds.ord)[NC]
  logodds.ordrev    <- sort(log.odds, decreasing=FALSE)
  sum.last.logodds  <- cumsum(logodds.ordrev)[NC]
  res.red           <- as.vector((sum.first.logodds - as.vector(matrix %*% log.odds)) / (sum.first.logodds - sum.last.logodds))
  # Compute final PFS vector:
  res <- final.PFS(res.red, all.0s, all.1s, N)
  # Export results:
  export.res.NP(matrix.sv, N, res, "U3", part.res, Ncat=2, NA.method, 
                IRT.PModel, res.NA[[2]], Ability.PModel, res.NA[[3]], IP.NA, Ability.NA, res.NA[[4]])
}
########################################################################################
########################################################################################
# ZU3 (van der Flier, 1980, 1982):
########################################################################################
########################################################################################
ZU3 <- function(matrix, 
                NA.method="NPModel", Save.MatImp=FALSE, 
                IP=NULL, IRT.PModel="2PL", Ability=NULL, Ability.PModel="ML", mu=0, sigma=1)
{
  matrix      <- as.matrix(matrix)
  N           <- dim(matrix)[1]; I <- dim(matrix)[2]
  IP.NA       <- is.null(IP); Ability.NA  <- is.null(Ability)
  # Sanity check - Data matrix adequacy:
  Sanity.dma(matrix, N, I)
  # Dealing with missing values:
  res.NA <- MissingValues(matrix, NA.method, Save.MatImp, IP, IRT.PModel, Ability, Ability.PModel, mu, sigma)
  matrix <- res.NA[[1]]
  # Sanity check - Perfect response vectors:
  part.res  <- Sanity.prv(matrix, N, I)
  NC        <- part.res$NC
  all.0s    <- part.res$all.0s
  all.1s    <- part.res$all.1s
  matrix.sv <- matrix
  matrix    <- part.res$matrix.red
  # Compute PFS:
  pi <- colMeans(matrix.sv); qi <- 1-pi
  # If there are answer options not chosen by any respondent then some entries in pi are 0 or 1.
  # Below all corresponding logs are set from Inf to 0.
  # (Reason: They carry no individual information regarding aberrant response behavior.):
  log.odds     <- log(pi/qi)
  log.odds[is.infinite(log.odds)] <- 0
  log.odds.ord <- sort(log.odds,decreasing=TRUE)
  # 
  alpha             <- sum(pi*log.odds) + sum(pi*qi*log.odds)*(NC-sum(pi)) / sum(pi*qi)
  beta              <- sum(pi*qi*(log.odds)^2) - ((sum(pi*qi*log.odds))^2) / sum(pi*qi)
  sum.first.logodds <- cumsum(log.odds.ord)[NC]
  logodds.ordrev    <- sort(log.odds, decreasing=FALSE)
  sum.last.logodds  <- cumsum(logodds.ordrev)[NC]
  exp.val           <- (sum.first.logodds - alpha) / (sum.first.logodds - sum.last.logodds)
  var.val           <- beta / ((sum.first.logodds - sum.last.logodds)^2)
  # 
  U3.nz   <- as.vector((sum.first.logodds - as.vector(matrix %*% log.odds)) / (sum.first.logodds - sum.last.logodds))
  res.red <- as.vector((U3.nz - exp.val) / sqrt(var.val))
  # Compute final PFS vector:
  res <- final.PFS(res.red, all.0s, all.1s, N)
  # Export results:
  export.res.NP(matrix.sv, N, res, "ZU3", part.res, Ncat=2, NA.method, 
                IRT.PModel, res.NA[[2]], Ability.PModel, res.NA[[3]], IP.NA, Ability.NA, res.NA[[4]])
}
